"""
Author: Vasiliy Zdanovskiy
email: vasilyvz@gmail.com

Context builder for registration with proxy server.
"""

from __future__ import annotations

from typing import Any, Dict, Optional

from .models import RegistrationContext
from .resolvers import _resolve_registration_credentials
from .config_parsers import (
    parse_simple_config_format,
    convert_model_to_dict_config,
)
from .url_builders import (
    build_proxy_url_from_client_config,
    build_proxy_url_from_registration_config,
)
from .metadata_builders import build_advertised_url, build_server_metadata


def prepare_registration_context(
    config: Dict[str, Any], logger: Any
) -> Optional[RegistrationContext]:
    """Build registration context from configuration.

    Uses registration section from SimpleConfigModel (independent from server section).
    Supports backward compatibility with proxy_client and proxy_registration formats.

    Returns ``None`` when registration should not be performed.
    """
    # Fallback to old formats for backward compatibility
    proxy_client_config = dict(config.get("proxy_client") or {})
    proxy_registration_config = dict(config.get("proxy_registration") or {})
    # Also check for "registration" key in config dict (SimpleConfig format)
    if not proxy_registration_config and "registration" in config:
        proxy_registration_config = dict(config.get("registration") or {})

    # Try to parse SimpleConfig format
    registration_config_from_model = parse_simple_config_format(
        proxy_registration_config, logger
    )

    use_proxy_client = bool(proxy_client_config)

    # Initialize registration_config and registration_enabled
    registration_config = None
    registration_enabled = False

    # Use registration from SimpleConfigModel if available
    if registration_config_from_model:
        proxy_registration_config, _ = convert_model_to_dict_config(
            registration_config_from_model, logger
        )
        registration_enabled = (
            registration_config_from_model.enabled
            and registration_config_from_model.auto_on_startup
        )
        registration_config = {}
    elif proxy_registration_config and "heartbeat" in proxy_registration_config:
        # proxy_registration_config already has heartbeat - use it directly (SimpleConfig format)
        heartbeat_config = proxy_registration_config.get("heartbeat", {})
        if isinstance(heartbeat_config, dict) and heartbeat_config.get("url"):
            # heartbeat.url exists - this is SimpleConfig format
            registration_enabled = proxy_registration_config.get(
                "enabled", False
            ) and proxy_registration_config.get("auto_on_startup", True)
            registration_config = proxy_registration_config
            logger.debug(
                f"Using SimpleConfig format: enabled={registration_enabled}, "
                f"heartbeat.url={heartbeat_config.get('url')}"
            )
        else:
            # heartbeat exists but url is missing - error
            error_msg = (
                "registration.heartbeat.url is required when registration.enabled=true. "
                "It must be a full URL (e.g., http://localhost:3005/proxy/heartbeat)"
            )
            logger.error(f"❌ Configuration error: {error_msg}")
            raise ValueError(error_msg)
    elif use_proxy_client:
        registration_config = proxy_client_config.get("registration") or {}
        registration_enabled = registration_config.get("auto_on_startup", False)
    else:
        registration_config = proxy_registration_config
        # Old format proxy_registration: check both enabled and auto_register_on_startup
        registration_enabled = proxy_registration_config.get(
            "enabled", False
        ) and proxy_registration_config.get("auto_register_on_startup", False)

    if not registration_enabled:
        logger.info(
            "Proxy registration disabled (auto_on_startup=false or enabled=false)"
        )
        return None

    # Build proxy URL and register endpoint
    if use_proxy_client:
        proxy_url, register_endpoint = build_proxy_url_from_client_config(
            proxy_client_config, registration_config, logger
        )
    else:
        result = build_proxy_url_from_registration_config(
            proxy_registration_config, logger
        )
        if result[0] is None:
            return None
        proxy_url, register_endpoint, _, _, _, _ = result

    # Build server metadata
    server_config = dict(config.get("server") or {})
    advertised_url = build_advertised_url(server_config, logger)
    server_name, capabilities, metadata = build_server_metadata(
        config,
        proxy_client_config,
        proxy_registration_config,
        use_proxy_client,
        server_config,
    )

    # Resolve credentials
    credentials = _resolve_registration_credentials(
        use_proxy_client, proxy_client_config, proxy_registration_config
    )

    return RegistrationContext(
        server_name=server_name,
        advertised_url=advertised_url,
        proxy_url=proxy_url,
        register_endpoint=register_endpoint,
        capabilities=capabilities,
        metadata=metadata,
        use_proxy_client=use_proxy_client,
        proxy_client_config=proxy_client_config,
        proxy_registration_config=proxy_registration_config,
        credentials=credentials,
    )
