"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const kms = require("@aws-cdk/aws-kms");
const lambda = require("@aws-cdk/aws-lambda");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test deployment with new Lambda function
// --------------------------------------------------------------
test('Test deployment with new Lambda function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSns(stack, 'lambda-to-sns-stack', {
        deployLambda: true,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'deployed-function'
            }
        }
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                LAMBDA_NAME: 'deployed-function'
            }
        }
    });
    // Assertion 3
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Test deployment with existing Lambda function
// --------------------------------------------------------------
test('Test deployment with existing Lambda function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSns(stack, 'lambda-to-sns-stack', {
        deployLambda: true,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'override-function'
            }
        }
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                LAMBDA_NAME: 'override-function'
            }
        }
    });
    // Assertion 3
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Test deployment with imported encryption key
// --------------------------------------------------------------
test('Test deployment with imported encryption key', () => {
    // Stack
    const stack = new core_1.Stack();
    // Setup
    const kmsKey = new kms.Key(stack, 'imported-key', {
        enableKeyRotation: false
    });
    // Helper declaration
    new lib_1.LambdaToSns(stack, 'lambda-to-sns-stack', {
        deployLambda: true,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'deployed-function-no-enc'
            }
        },
        enableEncryption: true,
        encryptionKey: kmsKey
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                LAMBDA_NAME: 'deployed-function-no-enc'
            }
        }
    });
    // Assertion 3
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
});
// --------------------------------------------------------------
// Test the getter methods
// --------------------------------------------------------------
test('Test the getter methods', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const pattern = new lib_1.LambdaToSns(stack, 'lambda-to-sns-stack', {
        deployLambda: true,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`)
        }
    });
    // Assertion 1
    const func = pattern.lambdaFunction();
    expect(func).toBeDefined();
    // Assertion 2
    const topic = pattern.snsTopic();
    expect(topic).toBeDefined();
});
//# sourceMappingURL=data:application/json;base64,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