# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2013-2019 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

from openquake.hazardlib.gsim.base import CoeffsTable

COEFFS_FORELAND_60Bars  = CoeffsTable(sa_damping=5, table="""\
IMT                     a1                            a2                        a3                  a4                    a5                   a6                    a7                    a8                   a9                   a10                  a11                   a12                 a13                  a14                   a15                   a16                 a17                   a18                   a19                   a20                 a21                   a22                   a23                   tau    mean_phi_ss sigma_tot phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv                     -6.68764117047698             5.17836168608062          -2.72357965923663   0.981772779887652     -0.18794348818416    0.0174845924015038    -0.000627568495085281 0.788457477603706    -0.220273537315267   0.00656134713848716  0.000254627802704765  -2.00824221704951   -0.733915029041026   0.223805612148217     -0.0123276023822747   0.920864248165468   0.576321803891029     -0.161286779113245    0.00936374316464263   -0.16656693325605   -0.113955629075892    0.0320828679553869    -0.00195427475302179  0.5010 0.4600      0.68015   0.00000 0.00000 0.00000 5   7   16   36
pga                     -4.87344448904996             5.72551392383395          -3.23511717883835   1.17530129227868      -0.225779044194617   0.0211047107082395    -0.000762337284651901 0.747546958399791    -0.285870266803578   0.0285222649722448   -0.00147459266470677  -1.83889798037392   -0.778285956887197   0.228987224612386     -0.0126059406885344   0.784942282308357   0.632772794901676     -0.173989445744905    0.0103259717600289    -0.152057234501119  -0.122671039034162    0.0345457666103071    -0.00216747273515938  0.3532 0.4600      0.57998   0.58000 0.47000 0.35000 5   7   16   36
0.01                    -4.24464528701162             4.8375251725165           -2.7229788021309    1.02382946492236      -0.201504058458784   0.0191039108682566    -0.000695981815561547 0.676102162286005    -0.197438373890842   0.00683293074400265  3.14380736582837e-06  -1.67180936015094   -0.938907492152067   0.265562004945089     -0.0150244133016282   0.662201098362309   0.73425714009627      -0.196341719637515    0.0117916311952206    -0.125961932984455  -0.142795750920038    0.0389158522966839    -0.00245353561056639  0.3529 0.4600      0.57975   0.58000 0.47000 0.35000 5   7   16   36
0.02                    -4.81531340249482             5.83807098342726          -3.3163640277755    1.20433893515254      -0.231062485564841   0.0215786965244617    -0.000779241623520298 1.33035870782831     -0.418504095360988   0.0308782068817182   -0.000860755253438791 -2.18609434033764   -0.807019480663948   0.259184150432205     -0.0153296962646344   0.724440912488775   0.738019892545604     -0.204538048512532    0.012648971445691     -0.112364107245305  -0.151466653629851    0.0416404194034116    -0.00268457610254328  0.3674 0.4570      0.58639   0.56796 0.46097 0.37408 5   7   16   36
0.03                    -3.6050636356179              4.84893704647401          -2.81956042365627   1.06166675529961      -0.208650564924716   0.0197849744180356    -0.000721803529688156 -0.213105033590212   -0.118272005905898   0.0397195622106262   -0.00374139010241344  -0.303738306192     -1.06584969145193    0.227472727395779     -0.0108413452457289   -0.149657278890344  0.801411934907682     -0.178576889059417    0.00996897612210427   0.0181449535198095  -0.150360878054255    0.035400860421632     -0.00214099911738658  0.3749 0.4552      0.58975   0.56092 0.45569 0.38817 5   7   16   36
0.04                    -3.51964328277704             5.18433961053195          -3.05123291874821   1.13119671193126      -0.219912395037152   0.0207387892660009    -0.00075453354046627  -1.04143248912517    -0.0412128788904239  0.0653048511379604   -0.00659230050641493  0.311342397844077   -0.9460468201515     0.165410346268866     -0.00595380958134403  -0.207527000772036  0.637255635000036     -0.133669344047297    0.00687633787479656   -0.0122545453616859 -0.107729167851367    0.0258446228012686    -0.00152325896133532  0.3801 0.4540      0.59211   0.55592 0.45194 0.39816 5   7   16   36
0.05                    -3.58335693729946             5.09491628138349          -2.95003273441774   1.09571928562478      -0.213880900790288   0.0202307404764504    -0.000737524379342396 -0.184965334998059   -0.42825229884086    0.122217510514879    -0.00934314299541468  -1.09746942381067   -0.307057656633139   0.0680288471747899    -0.000918500844591686 0.676687773547805   0.257540278109172     -0.0758599608260421   0.00380651567992179   -0.187959512569028  -0.0369184827783211   0.01519693286208      -0.0009529008019825   0.3855 0.4530      0.59483   0.55204 0.44903 0.40592 5   7   16   36
0.1                     -4.92501100228238             5.4174958036084           -2.86530729586551   1.04472969588324      -0.203679540041406   0.0192723236132154    -0.000702730673695176 2.48445683728691     -0.888653246056202   0.0940406929527083   -0.00360619811324155  -3.66114762194063   -0.172521512629352   0.16331030012352      -0.0103888018007496   1.68490651694268    0.385072135162914     -0.150858550116926    0.0096580272999716    -0.302768410963105  -0.0938812333786166   0.0329141017991156    -0.00216645013397452  0.3864 0.4500      0.59312   0.54000 0.44000 0.43000 5   7   16   36
0.15                    -5.22633474501933             5.2721817885936           -2.69864999054627   0.988673729169688     -0.194367725119206   0.0184960366218397    -0.000676888996835389 2.33940669959614     -0.78969442730106    0.0761999374428704   -0.00257515037478002  -3.35700210416741   -0.265038343864823   0.16623245917025      -0.010006139879047    1.48204304305231    0.428691666445927     -0.1469143432092      0.00896588299070917   -0.252964021005159  -0.101192033119752    0.0313663711615242    -0.00196899237581671  0.3841 0.4675      0.60507   0.58095 0.47510 0.40075 5   7   16   36
0.2                     -5.57535598932319             5.38984119454482          -2.74367383056819   1.00579597213145      -0.197843243494085   0.0188273855860197    -0.000688835659473087 2.11885003522438     -0.731275256894937   0.0731115069054352   -0.00266371436754708  -3.10826688088209   -0.254513520751207   0.148027891589439     -0.00835441217649434  1.36357415629254    0.402720052433297     -0.132296555799967    0.00775007250185203   -0.2292052636306    -0.0942314187522584   0.0281598882463669    -0.00171117547126275  0.3690 0.4800      0.60546   0.61000 0.50000 0.38000 5   7   16   36
0.25                    -6.36478987659494             6.29437453964111          -3.2946442092011    1.18071586096165      -0.227177234322681   0.0213041863099761    -0.000771947278564599 1.72961886475168     -0.539237050433386   0.0407769766826306   -0.000841057054273248 -2.58846242437667   -0.478864245690973   0.183435559073212     -0.0102992047244847   1.08506472435449    0.52082123440524      -0.151092176750531    0.00879413999192502   -0.175979090700042  -0.115849313570858    0.0316596883067854    -0.00190871939612346  0.3445 0.4800      0.59082   0.62651 0.50000 0.37450 5   7   16   36
0.3                     -6.07840771097283             5.64358469351264          -2.93528279673249   1.08352843339215      -0.212635079970058   0.0201470703911316    -0.000733788019997517 1.8090298443913      -0.677159433339779   0.0764898459808062   -0.00333299600807248  -2.83269101570429   -0.179369373288581   0.11089177776691      -0.00535878344649455  1.24793633549501    0.339279132535668     -0.10798697935765     0.00586962038868909   -0.20689574922488   -0.081204617797326    0.0235168637633149    -0.0013570003956442   0.3377 0.4800      0.58688   0.64000 0.50000 0.37000 5   7   16   36
0.35                    -6.3055619525167              5.87481699593989          -3.11804887545767   1.14825643838948      -0.223652386322297   0.0210469194089291    -0.000762249101532504 1.49889698516272     -0.554135556798787   0.0601403674354462   -0.00260608345562364  -2.55181311061808   -0.25064706107356    0.114700276848634     -0.00524836601217806  1.1339612290722     0.358355431305682     -0.106815615060306    0.00562731902644393   -0.188758470612616  -0.0830239200046972   0.0230655183175435    -0.00130048896910242  0.3482 0.4740      0.58812   0.62793 0.49396 0.37000 5   7   16   36
0.4                     -7.41747785232085             7.01208621102711          -3.72664469621988   1.32732344732958      -0.252284803761802   0.0233713445906773    -0.00083718092621546  3.1077754520607      -1.52771752374073    0.241597138271361    -0.0132724631175423   -4.70910152899192   1.09023325728712     -0.139655990175536    0.00991206757988129   2.23675077626282    -0.334044134714414    0.0256350244089717    -0.00233047480002398  -0.379156019406782  0.0377355097259135    -0.000148898514408928 0.000101356888616509  0.3552 0.4687      0.58811   0.61747 0.48874 0.37000 5   7   16   36
0.45                    -7.82779489530386             7.62608389285792          -4.13989644832304   1.45652510325165      -0.272723465345259   0.0249716183396561    -0.000886627926379577 2.69662763831394     -1.35997134158091    0.21891039093953     -0.0122562630822623   -4.36666674763937   0.981910923828589    -0.129457902160427    0.00967713118509161   2.10585628439286    -0.299682346214073    0.023746028961832     -0.00238982287976358  -0.359739812273479  0.0334212533254844    -4.84051559530649e-05 0.000122916970468237  0.3495 0.4641      0.58097   0.60825 0.48413 0.37000 5   7   16   36
0.5                     -8.2265366500011              8.24540453242853          -4.55038456632292   1.58198362099072      -0.292101843584821   0.0264520496364808    -0.000931220136631121 2.28970262789775     -1.19271165287317    0.196060897158034    -0.0112193188429192   -4.04441683892171   0.87610422984418     -0.118702549348006    0.00936860786183337   1.98821018645651    -0.267178344626396    0.0215874706186362    -0.00240948188645301  -0.343037989289351  0.029511850309713     8.62871304091623e-05  0.000138349530357847  0.3548 0.4600      0.58095   0.60000 0.48000 0.37000 5   7   16   36
0.55                    -8.64317736278674             8.89880072736536          -4.97259289447859   1.70805912607717      -0.311172387001311   0.0278799617717292    -0.000973362345375043 1.90357136233138     -1.03217559312196    0.173807137380107    -0.0101912484267852   -3.75040793918096   0.774298566268998    -0.10728371203741     0.00896521075834516   1.88391235512684    -0.235204221931502    0.0187271386357347    -0.00235554003672769  -0.328563920095524  0.025519987532463     0.000379963713397183  0.000138532490621126  0.3639 0.4586      0.58544   0.59175 0.47587 0.37413 5   7   16   36
0.6                     -8.98359215815951             9.46063349568456          -5.33793661418211   1.81532349375695      -0.326945129489013   0.0290188284929106    -0.00100553300878085  1.46135994504029     -0.834681681241097   0.144329641993094    -0.00872563754622641  -3.38798101148068   0.622428223252598    -0.0856446305032103   0.0079285201297841    1.74646300448103    -0.178177684494891    0.0107099472007446    -0.00198119350972301  -0.308642941115824  0.0173028892552261    0.00154343187526878   8.4766847597516e-05   0.3757 0.4574      0.59187   0.58422 0.47211 0.37789 5   7   16   36
0.65                    -9.3407751330966              10.0219421261663          -5.68953487694033   1.91576985682585      -0.34134167300814    0.0300294507173742    -0.00103312699769988  1.11372183497756     -0.682323388727345   0.121937172074277    -0.00762556767402954  -3.12486936143796   0.513063355856014    -0.0700455073274795   0.00717505091375391   1.65247109091039    -0.138647276722649    0.00502541304708094   -0.00170833765185468  -0.295545777540947  0.011707744853447     0.00237162286548871   4.47828451616023e-05  0.3831 0.4562      0.59573   0.57729 0.46864 0.38136 5   7   16   36
0.7                     -9.64134367434335             10.4913159274342          -5.97951401423115   1.99606108759853      -0.352310792452397   0.0307482997310921    -0.00105088464380326  0.782518454191035    -0.530313874679248   0.0985526687032792   -0.00642654917600223  -2.86712363277193   0.393004307698089    -0.0509700540410993   0.00616130590449853   1.55753612943817    -0.0916942584082049   -0.00275618735872998  -0.00128542620199308  -0.282012568896759  0.00469718955730389   0.00357850177537767   -2.17671366144663e-05 0.3886 0.4551      0.59846   0.57087 0.46544 0.38456 5   7   16   36
0.75                    -9.89617727714873             10.8798931397945          -6.21442197614757   2.05844499524952      -0.36027285369283    0.031214736657753     -0.00106024857594995  0.478194203338849    -0.384169065977961   0.0751294250769081   -0.00518226934742213  -2.62332165147751   0.26724447333045     -0.0293283566549528   0.00493995674399626   1.46473081175757    -0.0392645294200378   -0.0122685942102544   -0.000733875221478945 -0.268426747317785  -0.00346000292810114  0.00511239039255462   -0.000111852343212231 0.3852 0.4542      0.59554   0.56490 0.46245 0.38755 5   7   16   36
0.8                     -10.1017571008476             11.1836890997379          -6.39342066905857   2.10309049818896      -0.365311907933718   0.0314396645426143    -0.00106169619765271  0.186073574673437    -0.234525649499643   0.0498371554274166   -0.00378046322419485  -2.37133499270396   0.122371209008467    -0.00257018544010322  0.00335760036188044   1.36221142005538    0.0257130096833896    -0.0248451284868128   2.58603376101028e-05  -0.252672993923546  -0.0140160733071675   0.00720873462594542   -0.000239454643187089 0.3850 0.4532      0.59468   0.55932 0.45966 0.39034 5   7   16   36
0.85                    -10.3207866961798             11.4720445473654          -6.55188615101902   2.14010809173968      -0.369050183113345   0.0315578059521413    -0.00105970672855725  0.00966352342646416  -0.142890968386861   0.0341288091472209   -0.00289894896550942  -2.24461228307757   0.0397180262631316   0.0138791593500665    0.00233963276533668   1.31750035824528    0.0615375867819503    -0.0325056596431506   0.000512136772389228  -0.246426102909941  -0.0197002762848421   0.00846919559252451   -0.000319932522954643 0.3811 0.4523      0.59150   0.55407 0.45703 0.39297 5   7   16   36
0.9                     -10.4803726002543             11.6620173481884          -6.64830183208708   2.15808473274019      -0.369726637458965   0.0314274653760033    -0.00104967788234136  -0.151698394476941   -0.0517564599832401  0.0175842000509591   -0.00193290204025563  -2.11888455167896   -0.0525516070707646  0.0331883918302299    0.00111362909783973   1.26940984341102    0.103932542542325     -0.0418557945429539   0.00111514130427375   -0.239279737505159  -0.026632538658316    0.0100323615469151    -0.000420841554962483 0.3799 0.4515      0.59006   0.54912 0.45456 0.39544 5   7   16   36
0.95                    -10.5790651713865             11.7579671354421          -6.68769524275165   2.15896976140082      -0.367702694750998   0.0310815551264473    -0.00103279593603841  -0.328541995938085   0.0569747107125359   -0.00314192332373603 -0.000686612548280872 -1.95503095882317   -0.177567903271775   0.0596449249747172    -0.000572105504591064 1.1985827986512     0.164338737905988     -0.0550205860725819   0.0019598032224686    -0.227964141225727  -0.0367416553289978   0.0122567727751332    -0.000563260161148672 0.3733 0.4507      0.58525   0.54444 0.45222 0.39778 5   7   16   36
1                       -10.6632488299448             11.8188355412008          -6.70222414016975   2.15207571057838      -0.364454424386432   0.0306403212094396    -0.00101298729757431  -0.474031863599619   0.154452184632651    -0.0225790615197156  0.000513591657832871  -1.81362882574581   -0.297494274989463   0.0860149423523365    -0.00228383453306543  1.13503452284048    0.223968017891565     -0.0684078990121189   0.00283105351894359   -0.217526431609695  -0.0468727067370018   0.0145410309222391    -0.000711329521275267 0.3716 0.4500      0.58362   0.54000 0.45000 0.40000 5   7   16   36
1.05                    -10.6864272259535             11.7697988446318          -6.64694038551815   2.12403063318927      -0.357860295477306   0.0299313908800189    -0.000984603851080203 -0.550277151027723   0.216212806053926    -0.0359635945395228  0.00137816924114376   -1.74889915732484   -0.376497892045277   0.105181550791928     -0.00358189252625364  1.10812560752886    0.263197264561872     -0.0780801922980944   0.003483499475334     -0.213293223389353  -0.0534402793267444   0.0161610438074483    -0.000819661402463557 0.3691 0.4482      0.58063   0.53956 0.44778 0.40000 5   7   16   36
1.1                     -10.7084186738144             11.7089548299897          -6.58233514165462   2.0932419962201       -0.35089392495294    0.0291997293001803    -0.000955754520832503 -0.602465956842806   0.270657901329922    -0.048735542338945   0.00223364330388752   -1.70106656297324   -0.452793164710268   0.124585311975511     -0.00491882556116113  1.08687341818146    0.302067463715155     -0.0879790081365837   0.00415863649540398   -0.209807121062128  -0.0599994626661499   0.0178200667883553    -0.000931505314362753 0.3650 0.4465      0.57674   0.53913 0.44566 0.40000 5   7   16   36
1.15                    -10.7378970596601             11.6526586111527          -6.51876736528738   2.06291325651283      -0.344082826311371   0.0284896883718251    -0.000927934925489131 -0.638781231361668   0.31949003667451     -0.0608711352387828  0.0030666311173054    -1.66185833117365   -0.527591689949986   0.144072782754653     -0.00627355668245168  1.06763271463385    0.341057171010031     -0.0980304016051084   0.00484758855895092   -0.206472899772965  -0.06664025419144     0.0195112443403748    -0.0010459308370601   0.3627 0.4449      0.57402   0.53873 0.44364 0.40000 5   7   16   36
1.2                     -10.6485474077977             11.4178964478041          -6.35026757445492   2.00160901781522      -0.332383416982351   0.0273854824935968    -0.000887347936391156 -0.669393995006016   0.368900204858294    -0.0735210806369239  0.00394391941246783   -1.62480197475838   -0.604438481736086   0.164222919703656     -0.00767492198375143  1.04921081741059    0.380567015899541     -0.10824138088585     0.00554676285055426   -0.203327207215862  -0.0732670138760803   0.0212015768000185    -0.00116012678440446  0.3625 0.4434      0.57267   0.53834 0.44170 0.40000 5   7   16   36
1.25                    -10.6342386806523             11.2783683346981          -6.23415548593192   1.95598881304907      -0.323295512312007   0.0265064411421418    -0.00085456317871432  -0.638065343434247   0.385176955309118    -0.0804104805870561  0.00449600770872786   -1.65572596141422   -0.643606413296353   0.17761573900355      -0.00868475144347223  1.06256734280871    0.401873143313493     -0.115111858427177    0.00604917275196951   -0.205444696284061  -0.0768081211095708   0.0223167803464812    -0.00124009051202923  0.3631 0.4419      0.57195   0.53797 0.43984 0.40000 5   7   16   36
1.3                     -10.5106751027452             10.9888296503646          -6.03331443016316   1.88577416366246      -0.310352313261487   0.0253173912878445    -0.000811753012142016 -0.653175500412612   0.429670363930098    -0.0925659393032545  0.00535798250161791   -1.62567249804454   -0.719119388670399   0.197721950967834     -0.0100875586708982   1.04558928324628    0.441024363720082     -0.125265381042037    0.00674387570714024   -0.202415515379781  -0.0833565740447192   0.023984781300999     -0.00135252035677446  0.3610 0.4404      0.56949   0.53761 0.43806 0.40000 5   7   16   36
1.35                    -10.4023775042233             10.7172996445178          -5.84234528475302   1.81878616239293      -0.298011381611198   0.0241859898342138    -0.000771120368548868 -0.650985589126667   0.466304570700293    -0.103503325562083   0.0061563900035421    -1.61139860041899   -0.786971657645752   0.216543081609308     -0.0114181452193095   1.03548433270184    0.476489278669039     -0.134754905030858    0.00739954553003989   -0.200495196153968  -0.0892657088531227   0.0255324905547926    -0.00145771954569332  0.3605 0.4391      0.56808   0.53727 0.43634 0.40000 5   7   16   36
1.4                     -10.2759210519915             10.4079933592757          -5.62734389050832   1.74489998870857      -0.28466465342806    0.0229815754901649    -0.000728385235355707 -0.595834502556166   0.47361896333134     -0.109225913348809   0.00665588557272298   -1.66006456680375   -0.818194182072409   0.228600560750611     -0.0123501071308372   1.05629182220606    0.493349959128287     -0.14073682339353     0.00784573241909052   -0.203857987630116  -0.0919294371457712   0.026460797872737     -0.0015256604136042   0.3616 0.4377      0.56779   0.53694 0.43469 0.40000 5   7   16   36
1.45                    -10.156036029824              10.1159501592853          -5.42463684615669   1.67520943357872      -0.272074161897451   0.0218462075920898    -0.000688159011290661 -0.571348850468349   0.499233299832337    -0.118324344466072   0.00735232101371294   -1.66552346346637   -0.874734934359166   0.245292356662439     -0.013551388060113    1.05461483673205    0.523273821854311     -0.14911021617667     0.00843128548000303   -0.203277294149513  -0.0968787897505517   0.0278084277949002    -0.00161812306451991  0.3604 0.4365      0.56604   0.53662 0.43309 0.40000 5   7   16   36
1.5                     -10.0144704662283             9.78886606473377          -5.20134434811119   1.59978787567828      -0.258659225924946   0.0206514393950344    -0.000646233773272289 -0.524346151183701   0.512023853992513    -0.125093849200077   0.00791315923226436   -1.69814161291855   -0.914438030562006   0.258750090014976     -0.0145571880503014   1.066711562323      0.544242683317982     -0.155722303606865    0.00890909528372142   -0.205118886648975  -0.100214419428679    0.0288351029969503    -0.00169086575212821  0.3559 0.4352      0.56223   0.53631 0.43155 0.40000 5   7   16   36
1.55                    -9.86835876760519             9.45435339905193          -4.9740888202583    1.52358118116212      -0.245202753899851   0.0194605785314858    -0.00060466947660808  -0.465669832629459   0.518172000561061    -0.13064606589701    0.0084023612476149    -1.74382986434231   -0.945583030765901   0.270500257539422     -0.0154569689171443   1.08523231497023    0.56066484827085      -0.161398523783871    0.00932803129511443   -0.208072018360202  -0.102732510231496    0.0296913195624727    -0.00175284873736426  0.3507 0.4340      0.55804   0.53601 0.43005 0.40000 5   7   16   36
1.6                     -9.73235832030792             9.1327618367854           -4.7544990321619    1.45004017125013      -0.232261345516345   0.0183197816873411    -0.000565004800297924 -0.390677021354925   0.514746039869276    -0.134423549907194   0.00878665489036748   -1.80722285722217   -0.965389499652716   0.280036979855361     -0.0162215579981184   1.11209884628712    0.571467925560139     -0.165955423826398    0.00967802672376641   -0.212415596325985  -0.104290609018866    0.0303547538622392    -0.00180294262818112  0.3479 0.4329      0.55537   0.53572 0.42861 0.40000 5   7   16   36
1.65                    -9.54384764727695             8.73982117912969          -4.49512588587826   1.36515446169164      -0.217575342112888   0.0170412097480802    -0.000520954335961618 -0.328261024931304   0.517645291229309    -0.139218486237017   0.00922393319014234   -1.85741091275704   -0.990681571844011   0.290282671452443     -0.0170146168803342   1.13346808787579    0.584127829107317     -0.170680599210584    0.0100306327043648    -0.215954739982974  -0.106065160568747    0.0310262929039938    -0.00185221668803517  0.3486 0.4318      0.55495   0.53544 0.42721 0.40000 5   7   16   36
1.7                     -9.4031416780783              8.41764938715692          -4.27789512704847   1.29327257237208      -0.205065956510234   0.0159494217724004    -0.000483328574282362 -0.258010958308059   0.5158680026258      -0.143142145339325   0.00960986185874754   -1.91380860026851   -1.01164289651939    0.299670993464193     -0.0177560694655856   1.15700536506834    0.595053805152754     -0.17505755898615     0.0103627699766186    -0.219763574599053  -0.107597485545982    0.0316496726761104    -0.00189879889126539  0.3485 0.4307      0.55401   0.53517 0.42585 0.40000 5   7   16   36
1.75                    -9.21926624062798             8.03812926808273          -4.02918861966929   1.21254580916728      -0.191217067024882   0.0147535705774073    -0.000442447495785175 -0.202291200112505   0.521506328683017    -0.148290890762167   0.0100613494696683    -1.95311224955435   -1.0406306778835     0.310297030884572     -0.0185600318997759   1.17258849732354    0.609527847989483     -0.179962558732396    0.0107209785797088    -0.22228186697563   -0.10968952950764     0.03235504343256      -0.0019494180183862   0.3470 0.4296      0.55226   0.53491 0.42453 0.40000 5   7   16   36
1.8                     -9.07941980618793             7.71253001267345          -3.80941460443016   1.14035020368755      -0.178770413015702   0.0136767918020204    -0.000405615405290621 -0.0927302433027334  0.495053611629173    -0.147473687777828   0.0101614024936357    -2.05844842016881   -1.02893440266319    0.313185719792174     -0.0189005414440444   1.22074744106531    0.603562056088139     -0.180935777319421    0.0108417730679608    -0.230351757744028  -0.108261520729816    0.0323786273556912    -0.00195865796531679  0.3469 0.4286      0.55137   0.53465 0.42325 0.40000 5   7   16   36
1.85                    -8.90745302600357             7.35508421517056          -3.57558679499108   1.06479599630073      -0.165880720804385   0.0125701681364911    -0.000367997071209402 -0.0343963911824126  0.497556782536739    -0.151855573519104   0.0105607145191384    -2.10101120043668   -1.05270054641554    0.322477058774622     -0.0196126267231813   1.23821776405035    0.615008756694718     -0.185081400227193    0.0111486280873142    -0.233232342218877  -0.109795305362813    0.0329463889592278    -0.00200010392044313  0.3448 0.4276      0.54931   0.53440 0.42200 0.40000 5   7   16   36
1.9                     -8.74852401607075             7.01027739953641          -3.34796638512552   0.991257772076282     -0.153376818865349   0.0115010470595978    -0.000331795252800941 0.0642647376794128   0.475391378709248    -0.151571964071455   0.0106813532735227    -2.19404986557001   -1.04443116754132    0.325562420859224     -0.0199476099091317   1.28087942038909    0.610176843589152     -0.186041087340815    0.011260590386774     -0.240439316684184  -0.108507154740994    0.0329588774200993    -0.00200746242386994  0.3435 0.4266      0.54775   0.53416 0.42079 0.40000 5   7   16   36
1.95                    -8.58366251187044             6.66077532944809          -3.1189799849662    0.917698530631054     -0.140929169793606   0.0104410867509014    -0.00029603072704287  0.1566439171145      0.455800628898829    -0.151645113647603   0.0108190400714783    -2.28011239347679   -1.03841789260024    0.32887541113044      -0.0202900672495859   1.32048770259932    0.606153762097559     -0.187055913011251    0.0113731852228449    -0.247175272954931  -0.107325978611173    0.0329757230714415    -0.00201470349527113  0.3422 0.4257      0.54621   0.53392 0.41961 0.40000 5   7   16   36
2                       -8.41384826630551             6.31080964851136          -2.89174801742057   0.845108081164135     -0.128698415353477   0.00940349573373194   -0.000261141439674421 0.23221538279298     0.444486056481392    -0.153049110657203   0.0110265427292396    -2.34564522850514   -1.04227630353664    0.333760280205569     -0.0207147963787868   1.34996493813988    0.607031524109279     -0.188866993858667    0.0115285646337258    -0.252174008324681  -0.106999575130578    0.0331349381962944    -0.00202980102718356  0.3412 0.4248      0.54482   0.53369 0.41845 0.40000 5   7   16   36
2.5                     -7.33875584576641             3.99945106313111          -1.39649564658178   0.369623305916747     -0.0492392261950248  0.00274786541967731   -4.10231329082882e-05 0.287267001300632    0.682940849372422    -0.225310774278005   0.0162668416081971    -2.03199306012078   -1.57874575541526    0.468242335446203     -0.0298200209053085   1.12819138208581    0.89006213526231      -0.255979816503358    0.015963185591397     -0.209099890240917  -0.154569638540663    0.0440584911012801    -0.00274184195179852  0.3321 0.4166      0.53278   0.53166 0.40830 0.40000 5   7   16   36
3                       -5.96607204985882             1.32866498765663          0.278372657209518   -0.143943159114856    0.0335057211242823   -0.00396191985939997  0.000174950674382639  1.15646492267805     0.287461590708416    -0.162956830466154   0.0129262369557008    -3.02354302406731   -1.10345091357549    0.387339229686134     -0.025138332223711    1.64256019993785    0.609014229778007     -0.203479839488433    0.0127504280989301    -0.30212267009638   -0.0989629436044139   0.0331772013781195    -0.00206126177771737  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
3.5                     -5.60340327096894             0.134048528294328         1.07687297540947    -0.38782549539731     0.071494067954708    -0.00688449529534853  0.000262921798463959  2.5593346759438      -0.404396544362553   -0.0500795255791565  0.00687339559497594   -4.6944400930347    -0.237348222081177   0.239199921063253     -0.0168623849401243   2.4841816044135     0.14125292354889      -0.119574722406021    0.00791158773074936   -0.448998282415852  -0.0131911154207513   0.0173572870057223    -0.00113399965257021  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
4                       -4.70510452665419             -1.12115495352804         1.73955929260267    -0.571443666596866    0.0987152452965654   -0.008896109184819    0.000320564131133259  1.17459185937244     0.421160109390596    -0.202707610613517   0.015806595439945     -2.79744871894971   -1.36864987642412    0.450167186769499     -0.0293552719436915   1.5129349661054     0.711488890805856     -0.225977066882261    0.0142603567792119    -0.279973127194294  -0.111315711352884    0.0357061368876945    -0.00223687352192037  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
5                       -4.93359935006484             -1.15384208654528         1.74732884214205    -0.555835810677992    0.0915414471075853   -0.00782061095683337  0.000267381399324212  1.72290980837958     0.0326826310575246   -0.12292471599255    0.0107931413224658    -3.53926625483883   -0.781148254767582   0.324383031742072     -0.0213034569494526   1.93828342930707    0.351291075303955     -0.147931353380552    0.00927498436238294   -0.360757646468109  -0.0408667391625592   0.0204182593818387    -0.00126430134874644  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
6                       -5.75357393589501             -0.0960764165598715       1.07432530476948    -0.332551249933858    0.0510277426245839   -0.00404668069242632  0.000127438054563065  1.72105399658165     -0.0633926484074114  -0.095106309138637   0.00880994327540693   -3.47605786313683   -0.664307297156046   0.2889022645604       -0.0188257446784704   1.89871914172644    0.282866369541474     -0.128239663756182    0.00797892122003591   -0.352816751754128  -0.0284682745739752   0.0169734828221692    -0.00104785184540477  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
7                       -6.82016543943683             0.796999512362286         0.636802106125093   -0.196185265986084    0.0257892256093526   -0.00162120534989146  3.56695225114416e-05  4.14435789406844     -1.51391080746542    0.172470222584905    -0.00677479220521582  -6.49503696804838   1.20522843481753     -0.0619715820175337   0.00180930877683981   3.39826518797646    -0.667851876758662    0.0517032838191878    -0.00262932197636807  -0.60920961239682   0.136313173575137     -0.0143497646493773   0.000800630103629894  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
8                       -7.46551808662386             1.77659416161843          -0.0050209768780784 0.0131777626807384    -0.0108479460460024  0.0016505639718868    -8.07174567872759e-05 3.7271709135763      -1.38435735388036    0.160436690204671    -0.00648063772670446  -5.96526358923897   1.04903779995354     -0.046802865677253    0.00130063190034589   3.13148146971291    -0.590741408046655    0.043272815634031     -0.00223865360099427  -0.563083067585731  0.122669342083253     -0.0126625877706851   0.000708154080833036  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
10                      -8.20589620987515             3.26646849300338          -1.06278006635633   0.361098375861763     -0.0708012385106065  0.00688723487859206   -0.000262861786210615 1.5572334116826      -0.356898306386432   -0.00235683502439138 0.00208572021777938   -3.24271763091998   -0.255660621829228   0.166775624900545     -0.0104126803285788   1.76772257815016    0.0734893969252243    -0.0691078568544794   0.00415823683566276   -0.328631379970024  0.00636655422281645   0.00751331395567397   -0.000467751663634615 0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
""")

COEFFS_FORELAND_10Bars  = CoeffsTable(sa_damping=5, table="""\
IMT                     a1                            a2                        a3                  a4                    a5                   a6                    a7                    a8                   a9                   a10                  a11                   a12                 a13                  a14                   a15                   a16                 a17                   a18                   a19                   a20                 a21                   a22                   a23                   tau    mean_phi_ss sigma_tot phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv                     -5.26703737213063             3.13626055595851          -1.28721490767671   0.436079788498271     -0.0829694150747114  0.00771941667433342   -0.000277067746761317 0.74508294283781     -0.177550060398838   -0.00692405597192397 0.00140980488088061   -2.73428576422863   -0.241021771853078   0.134037341082296     -0.00744900618819384  1.4320858073911     0.21696583939213      -0.091577923509563    0.00530347186733211   -0.274542030927066  -0.037119151249474    0.0168338402506107    -0.0010443484939524   0.5010 0.4600      0.68015   0.00000 0.00000 0.00000 5   7   16   36
pga                     -3.32670943088517             3.5556355694027           -1.66796867190138   0.555994862354272     -0.102788396912392   0.00936561238918964   -0.00033180439271463  0.555130811558793    -0.144572670521034   -0.00528485560008013 0.000986829030292697  -2.38483573457844   -0.408692340930479   0.165923710215004     -0.00952080426240121  1.22824738940427    0.319867067519261     -0.114640382921819    0.00698469487064828   -0.255141025194734  -0.0488239178206482   0.0199848093149447    -0.00130916077711155  0.3532 0.4600      0.57998   0.58000 0.47000 0.35000 5   7   16   36
0.01                    -2.90032331039001             3.00771118106566          -1.36659418225595   0.468847119039609     -0.0889325952933405  0.00822483663401041   -0.000293939270332192 0.383096253859452    -0.00193278859418574 -0.0362606847465143  0.00297673302173677   -2.08072931884389   -0.647225078834056   0.216404527580963     -0.0127312142626954   1.0301472296529     0.465869481767691     -0.145174391240488    0.00892731673695207   -0.215256868895925  -0.0772802216626953   0.0259131745125165    -0.00168735572718088  0.3529 0.4600      0.57975   0.58000 0.47000 0.35000 5   7   16   36
0.02                    -2.86793967770404             3.11957545481108          -1.44394481169995   0.496381301993663     -0.0939362738875977  0.00866805840520567   -0.000309296794946307 1.43770523860379     -0.486268870539032   0.0393716434308876   -0.00102796260401371  -3.03121437516251   -0.225744475432394   0.152496705058198     -0.00948599225546078  1.26387254222847    0.354984656368731     -0.130167860870933    0.00832524039826581   -0.225856183555568  -0.0696423596932119   0.0252653293038128    -0.00170173556334653  0.3674 0.4570      0.58639   0.56796 0.46097 0.37408 5   7   16   36
0.03                    -2.16875768931814             2.93489934024924          -1.43459105894108   0.499837073222821     -0.0947306683586153  0.00874974186094818   -0.000312699115798526 -0.223342010322253   -0.0925155441903644  0.0288732519421651   -0.00271047074485224  -1.00987562162679   -0.596222025919401   0.142721849006004     -0.0062721191533488   0.361946503484306   0.446270225755573     -0.109282919043174    0.00589453184562083   -0.100910987898319  -0.066350546126415    0.0184784203750211    -0.00111125182821801  0.3749 0.4552      0.58975   0.56092 0.45569 0.38817 5   7   16   36
0.04                    -1.85991645548955             3.04087166125883          -1.54871816828247   0.533719070446058     -0.0998123573575956  0.0091491951187514    -0.000325621137388423 -1.3844518823419     0.145571111665104    0.0296119548631258   -0.00433027265268335  -0.217804963695111  -0.519674944939597   0.0799761142062009    -0.000933654160594287 0.347945152396632   0.215174961893972     -0.0464542083514342   0.00149304679086812   -0.158745761246146  0.00120068287671574   0.00318803752712602   -0.000106003696259673 0.3801 0.4540      0.59211   0.55592 0.45194 0.39816 5   7   16   36
0.05                    -2.51948530496056             3.62718909115511          -1.78489685374243   0.590563172686545     -0.10794741346688    0.00977470104363337   -0.000345233105117686 -0.0868652495442327  -0.46513978917484    0.122309026775045    -0.008924784566403    -2.03276077380964   0.306886649833306    -0.0426682561296838   0.00509964101810955   1.40977959551608    -0.243939302659869    0.0206090733931124    -0.00181969897618993  -0.361891011724064  0.0839891905189254    -0.00865112427436276  0.000475147411294994  0.3855 0.4530      0.59483   0.55204 0.44903 0.40592 5   7   16   36
0.1                     -5.77358387436002             6.52393668216616          -3.08296751765327   0.925059070897834     -0.156651010101127   0.0134923596560024    -0.000461134700977491 2.3862235049158      -0.848133168473764   0.0836930367787418   -0.00270024127595521  -4.26403703369255   0.276965871262683    0.0798537575155584    -0.00587238018077545  2.1858848735903     0.0152281112227743    -0.0786249157740781   0.00547415982840304   -0.413804520844781  -0.0120653598525187   0.0166316255884569    -0.0012012483427712   0.3864 0.4500      0.59312   0.54000 0.44000 0.43000 5   7   16   36
0.15                    -5.30583820332985             4.57147814476841          -1.64150862856109   0.455751355572553     -0.0782503812028876  0.00690612627534351   -0.000240411259580944 3.67862894376596     -1.55141960737129    0.209288264313549    -0.00992297401782265  -5.63946698153338   1.13195830309055     -0.0904590201274529   0.0047278337908568    2.76864385048372    -0.386621615439569    0.00803289408848597   -0.000183266622885591 -0.488872667002482  0.0513114139181355    0.00179433168854613   -0.000193166871774228 0.3841 0.4675      0.60507   0.58095 0.47510 0.40075 5   7   16   36
0.2                     -4.8760227543236              3.62635317760238          -1.06326877843336   0.285774505056033     -0.0518247964889933  0.00481897370009935   -0.000174526648118189 2.16939742026467     -0.750670467177896   0.0715639824125344   -0.0022223936864509   -3.64194575390159   0.100982769432395    0.0847177612331092    -0.00501031995001263  1.73499232352242    0.143711641756067     -0.0822271115591027   0.00485245500125762   -0.302085472443669  -0.0422192617485912   0.0177224430055716    -0.00108467627956935  0.3690 0.4800      0.60546   0.61000 0.50000 0.38000 5   7   16   36
0.25                    -4.0659366071034              1.9750102876965           -0.066202325287251  -0.000908561427696893 -0.008299049777254   0.00144262018770231   -6.88796756959308e-05 2.28979362626723     -0.899566736265911   0.107506354822729    -0.00464500942414325  -3.83840333371008   0.363295444084535    0.02047404780428      -0.000625691843811995 1.84584631903045    -0.00485414653192391  -0.045887559616537    0.00236264420411267   -0.320012315733279  -0.0148538641279796   0.0110331791002791    -0.000625501971802456 0.3445 0.4800      0.59082   0.62651 0.50000 0.37450 5   7   16   36
0.3                     -3.4446227347508              0.776505142311062         0.610052504303671   -0.182686351800709    0.0175445872052436   -0.00043916828432286  -1.34397144244561e-05 2.28488743598187     -0.971180365168955   0.128840128651905    -0.00619968796872504  -3.86990348989904   0.504836185463384    -0.0186367752494092   0.00216153187464643   1.86588448023042    -0.0813555401218749   -0.0250879232560473   0.000882867744422645  -0.321750129601174  -0.00159303630903208  0.00744640618968298   -0.000370178240421374 0.3377 0.4800      0.58688   0.64000 0.50000 0.37000 5   7   16   36
0.35                    -4.20919464550076             1.84319736005503          -0.109066744901458  0.0617065300400491    -0.02586458339047    0.00341874557794479   -0.000148964300792415 1.84454489235892     -0.757716727457258   0.0941441650251749   -0.00432433325143163  -3.357412359353     0.270116271536361    0.0188621800062666    0.00013464329125139   1.61129813064348    0.0380953997774251    -0.0449442688467146   0.00199359454517875   -0.276216897227806  -0.0231082592445331   0.0111647367941777    -0.000584439554062137 0.3482 0.4740      0.58812   0.62793 0.49396 0.37000 5   7   16   36
0.4                     -3.01191202355174             0.0249171168324805        0.872636948417757   -0.206468588728205    0.0140690492357776   0.000324295909637535  -5.12302500704934e-05 1.5790521617184      -0.664250427373821   0.0836896711374899   -0.00396248193637705  -3.1345998612459    0.224852114895893    0.0185369595949898    0.000464181507149376  1.51969064858795    0.0521325175992905    -0.0436884028575706   0.00178199082555719   -0.260982053304277  -0.0252854310940837   0.0109933669116105    -0.000553785745262452 0.3552 0.4687      0.58811   0.61747 0.48874 0.37000 5   7   16   36
0.45                    -2.39536337589362             -0.859038752807281        1.29147300857695    -0.305361607852727    0.0267840057240139   -0.000532693351624234 -2.73924494700274e-05 1.41044233602955     -0.646659675612702   0.0898234095395644   -0.004690837229584    -3.13196045324413   0.333295804336893    -0.01397839046987     0.00287264489562082   1.56830728050447    -0.0295160494486703   -0.0225242791306657   0.000286222830128434  -0.273667363398667  -0.00868538490301403  0.00691870682065016   -0.000271118831944569 0.3495 0.4641      0.58097   0.60825 0.48413 0.37000 5   7   16   36
0.5                     -2.7187794073925              -0.285337273971392        0.844599198887477   -0.146652543600387    -0.00153508331395166 0.001957488445184     -0.000113468259847804 1.00260658409836     -0.455083133121522   0.0596006722515259   -0.00309971999287024  -2.75535890374622   0.16217732478686     0.013230739126438     0.00139941102337739   1.39988701185671    0.0518373824727645    -0.0364171393912544   0.00108459221145721   -0.245061228335271  -0.0232029487453185   0.00955494967008869   -0.000429099427426789 0.3548 0.4600      0.58095   0.60000 0.48000 0.37000 5   7   16   36
0.55                    -2.73061108259444             -0.16598276897927         0.674133012288854   -0.0738339080253308   -0.0155390441754578  0.00322975701888526   -0.000158011666485762 0.748243903819447    -0.372535828313165   0.0520157499432002   -0.00295078896229932  -2.6266004512416    0.162179246994893    0.00498894050742213   0.00219592277915087   1.37151317016543    0.0367457173910277    -0.0302219800566656   0.000588777775954979  -0.242513396325005  -0.0198128684406513   0.00844198322565587   -0.000344364942003761 0.3639 0.4586      0.58544   0.59175 0.47587 0.37413 5   7   16   36
0.6                     -3.07469473835377             0.438943504470161         0.229826391358228   0.0770819440282695    -0.0415401356588965  0.00544960575558232   -0.000232769751372524 0.55450683709777     -0.314840630325713   0.0477516778500959   -0.00294689161949614  -2.57238610266481   0.189540270119185    -0.00640517327444354  0.00310303118217674   1.37864884519723    0.00923608765424011   -0.0226932937891917   5.21607161925125e-05  -0.245882807538553  -0.0144303079125226   0.00712199021018569   -0.000253724747275147 0.3757 0.4574      0.59187   0.58422 0.47211 0.37789 5   7   16   36
0.65                    -4.0041642470341              1.86492925880952          -0.680091596650323  0.365005051858577     -0.0894959489883151  0.00948183193832565   -0.000367768721140511 0.782908746906793    -0.523579636900172   0.09506977629739     -0.00607730520970717  -3.1362136618738    0.614075909655312    -0.096099124256471    0.00884534389720285   1.72486248895825    -0.239084655164742    0.0288779138486362    -0.00323044667116876  -0.311439142189987  0.0317813931741247    -0.00240397596980073  0.000351627096970609  0.3831 0.4562      0.59573   0.57729 0.46864 0.38136 5   7   16   36
0.7                     -3.22304101806481             0.752737911632899         -0.0833993939806737 0.196152343074749     -0.0628124335911086  0.00726655951778206   -0.000292793893707899 0.694706676027004    -0.503932746636367   0.0947124661510137   -0.00616894433161526  -3.16594120703429   0.654760104946267    -0.10532404540998     0.00942331430972474   1.75814178224429    -0.263397853277       0.0334406133507601    -0.00348774484721859  -0.317562598713776  0.0355377351280951    -0.00301775787762244  0.00038260019224898   0.3886 0.4551      0.59846   0.57087 0.46544 0.38456 5   7   16   36
0.75                    -4.59943411687309             2.93062393975585          -1.44413490548783   0.615897583001915     -0.131221516769946   0.0129238207960591    -0.000479983764980231 0.410936068027649    -0.375157290420169   0.0755529318691398   -0.00523085178526517  -3.02163054282876   0.604179659124215    -0.10015668818843     0.00929590390501408   1.73212007594885    -0.260440557578446    0.0345564239422639    -0.00363233307430121  -0.317232668793219  0.0370901650331006    -0.00354844436550272  0.00042682791310466   0.3852 0.4542      0.59554   0.56490 0.46245 0.38755 5   7   16   36
0.8                     -4.16189273505846             2.38737930853645          -1.18590169670584   0.545528322900921     -0.119769599256334   0.0119166349764161    -0.00044379698024331  -0.0774489285196579  -0.082779436138788   0.0204170037270989   -0.00192498839845936  -2.4770886608049    0.249133411796762    -0.0290544040804481   0.00484327715041968   1.45802371671973    -0.0724443637764314   -0.00416104342912593  -0.0011662836224135   -0.268356024480471  0.00278748018780977   0.00361491421230814   -3.30045225655419e-05 0.3850 0.4532      0.59468   0.55932 0.45966 0.39034 5   7   16   36
0.85                    -4.83600688873843             3.34953299949953          -1.76923806017654   0.722114908948742     -0.147989162971985   0.0141967545513086    -0.000517224944832575 0.302458126829892    -0.356295822249276   0.0769500887048721   -0.00549328246037467  -3.14461440053889   0.699038984857333    -0.119604145454608    0.0104963558976722    1.83103693229198    -0.319179461121739    0.0453238308242295    -0.00426252575800953  -0.335904652927138  0.0472676514045454    -0.00530997112419382  0.000527246157053706  0.3811 0.4523      0.59150   0.55407 0.45703 0.39297 5   7   16   36
0.9                     -5.42572617892849             4.01827495701163          -2.14176191123694   0.830953142754875     -0.164926473108621   0.0155261793032089    -0.000558616627790827 1.24223321230333     -0.878996912296595   0.169271397205819    -0.0107229890277651   -4.48789067920962   1.45601339038        -0.255383391748188    0.0183122991490182    2.5340412855497     -0.719002294106609    0.117906563188442     -0.00849382854428405  -0.459391962530967  0.11813473270719      -0.0182932245831081   0.00129119232537173   0.3799 0.4515      0.59006   0.54912 0.45456 0.39544 5   7   16   36
0.95                    -5.78701442329129             4.88309469142426          -2.75269524436434   1.0224221051428       -0.195414306259177   0.017957963264861     -0.000636123284289745 -0.264182530465534   -0.00204761602833165 0.00863176550507075  -0.00134568276675083  -2.60129488312117   0.310902904710321    -0.0386860107282356   0.00533166759089956   1.57638855645317    -0.123577328624291    0.00327995505330274   -0.00153857382733556  -0.292022841525564  0.0126206523542751    0.00223149444602246   3.61725501147959e-05  0.3733 0.4507      0.58525   0.54444 0.45222 0.39778 5   7   16   36
1                       -6.28981993122377             5.3228412759602           -2.95453694853352   1.07453560836465      -0.202751764611578   0.0184736689221306    -0.00065000419437941  1.09000360059332     -0.783843900759027   0.150860115655482    -0.00960116802537146  -4.47152177862127   1.4061586332752      -0.241165595967891    0.0172717129213225    2.54767911431575    -0.699005327292589    0.111022491674918     -0.00796985484769539  -0.462444550360028  0.114599575703224     -0.0170435402966915   0.0011967216676158    0.3716 0.4500      0.58362   0.54000 0.45000 0.40000 5   7   16   36
1.05                    -6.71827434264888             6.21071586041932          -3.55151029771988   1.25736106057995      -0.23146522619895    0.0207412298672082    -0.000721670323275789 -0.177040596606083   -0.0453598512071796  0.015592254295175    -0.00171026024706868  -2.84595046832016   0.414795190716366    -0.0531962449802778   0.00600535552648473   1.71002139661082    -0.175849704337179    0.0101613028156019    -0.00184919811710077  -0.314632694663449  0.0210755332205777    0.00116195609775574   8.40330204442184e-05  0.3691 0.4482      0.58063   0.53956 0.44778 0.40000 5   7   16   36
1.1                     -6.65135088947864             6.05237970550709          -3.45551546913388   1.22455171863581      -0.225079419782231   0.0201059291459138    -0.000696868784053706 0.00595111955996341  -0.1466957880942     0.0336076184832762   -0.00274417037137979  -3.18315446324461   0.598949658663457    -0.0860714309409154   0.007916685167105     1.90299159034293    -0.282846596828002    0.0297355540233861    -0.00301691242289253  -0.350390205425597  0.0413358623172355    -0.00262257547658329  0.000314116845058944  0.3650 0.4465      0.57674   0.53913 0.44566 0.40000 5   7   16   36
1.15                    -7.51187505615059             7.05544301700349          -3.99121702112986   1.3753065438137       -0.248037122808307   0.0218935397517645    -0.00075265058441949  1.02989019910956     -0.698285747976506   0.129043529205733    -0.00807570356046503  -4.55220714181947   1.34686460594587     -0.21769467344469     0.0154008649286052    2.59916019046442    -0.668183919126842    0.0985835495501888    -0.00699067392379508  -0.470689789403292  0.108707973257365     -0.0147984849459745   0.00102448734089152   0.3627 0.4449      0.57402   0.53873 0.44364 0.40000 5   7   16   36
1.2                     -7.87128890600199             7.54273362800571          -4.27463082080956   1.45522482300006      -0.259688755401952   0.0227416566010974    -0.000777062529864843 1.02593407331821     -0.671667262905619   0.121280210528805    -0.00750669021017292  -4.57798305960614   1.31855239372288     -0.207475850864438    0.0146079545051539    2.61199553534731    -0.652354367616709    0.0931202212044926    -0.00657741521689512  -0.472446203249117  0.105757864810089     -0.0138332841189208   0.00095330897681451   0.3625 0.4434      0.57267   0.53834 0.44170 0.40000 5   7   16   36
1.25                    -8.20723192706778             7.98915711832606          -4.53087877081253   1.52642508931548      -0.269884810451332   0.0234678665913826    -0.000797424724363086 1.02052325642682     -0.641213328118898   0.112482218296826    -0.00686466865672107  -4.59291283260952   1.28088426215766     -0.195221324448003    0.0136843828049016    2.61790048199874    -0.631352507486342    0.0865911945687255    -0.00609754796395417  -0.472919173177719  0.101909888807378     -0.0126884108595635   0.000871049665696424  0.3631 0.4419      0.57195   0.53797 0.43984 0.40000 5   7   16   36
1.3                     -8.53338759226144             8.40108317868254          -4.76054132969195   1.58883215729656      -0.278617426223151   0.0240728543662286    -0.000813798695838148 1.07441454955372     -0.642228081527005   0.10910108592185     -0.00652559530815929  -4.67247332995782   1.27850930983986     -0.189214606462594    0.0131190868133279    2.65406019968063    -0.627424698738592    0.0831644652077682    -0.00579918641946631  -0.478420720162457  0.100971625891051     -0.0120824854439955   0.000820765514775956  0.3610 0.4404      0.56949   0.53761 0.43806 0.40000 5   7   16   36
1.35                    -8.82756817317637             8.77366592313133          -4.96775279115425   1.64443385629268      -0.286238509072199   0.0245858642629619    -0.000827152559820241 1.0771198738053      -0.612674340465896   0.10006025983079     -0.00585468398773951  -4.68273911641768   1.2348918661798      -0.175570325434041    0.0121048505774742    2.65530829733187    -0.602836460881826    0.0759137177059345    -0.00527550132209425  -0.477961569946219  0.0965159437716492    -0.0108252605201069   0.000731994615544572  0.3605 0.4391      0.56808   0.53727 0.43634 0.40000 5   7   16   36
1.4                     -9.07838463910802             9.06011166764328          -5.11813901206369   1.68233094877543      -0.290983075093712   0.0248624828152051    -0.000832731213273195 1.15060316313405     -0.620657510899213   0.0974525047819544   -0.00554135671768854  -4.77376089309954   1.23544045255531     -0.169716879725268    0.0115351473085041    2.69535209647887    -0.600108254283054    0.0726141419917565    -0.00498184750631767  -0.484057055605317  0.0958360575083216    -0.0102627353776029   0.000684247596982338  0.3616 0.4377      0.56779   0.53694 0.43469 0.40000 5   7   16   36
1.45                    -9.32874069024729             9.34478010941254          -5.26629078953152   1.71939096345155      -0.295593753334268   0.0251296897687515    -0.000838082166900054 1.21973704485852     -0.626147484258446   0.0944234133798243   -0.00520590092351642  -4.85601837599259   1.23165212747621     -0.163203559297103    0.0109335437065675    2.730832806751      -0.595375708263892    0.0690398931212394    -0.00467605361062684  -0.489384246392762  0.0948464860549967    -0.00966121970383426  0.000634877250328207  0.3604 0.4365      0.56604   0.53662 0.43309 0.40000 5   7   16   36
1.5                     -9.51806739530675             9.55061912125549          -5.37028670579332   1.74343691899193      -0.298120678211407   0.0252260622496632    -0.000837825696850119 1.22550663596519     -0.593752383432419   0.0843605359819737   -0.00445705965162281  -4.8540296883936    1.17743561870549     -0.14730530634109     0.00977884685378998   2.72388659316217    -0.565271109366462    0.0607392384382607    -0.00409130016938934  -0.487452248619582  0.0895167767401694    -0.0082510402600284   0.00053776618093449   0.3559 0.4352      0.56223   0.53631 0.43155 0.40000 5   7   16   36
1.55                    -9.67748296253128             9.69692801466036          -5.43506283673076   1.75538220625915      -0.298710561262577   0.0251659259512811    -0.000832508114325558 1.29010021201015     -0.595116975753867   0.0804334698354      -0.00406483571966318  -4.92269833447959   1.16533461838759     -0.139283446382673    0.00909215522565971   2.75193146553883    -0.556585805524515    0.0565149932360248    -0.00375170038121697  -0.491536293059258  0.0879310529055529    -0.00756073355470927  0.000484226991767932  0.3507 0.4340      0.55804   0.53601 0.43005 0.40000 5   7   16   36
1.6                     -9.88374753376927             9.91262203963992          -5.53997532989788   1.77925348160401      -0.301228313653104   0.0252667156682387    -0.000832615708764995 1.34486515277684     -0.59034963039728    0.0753533623157668   -0.0036045520089709   -4.97469272202634   1.14343689085613     -0.129473344670133    0.00830138756655687   2.77104048975821    -0.542796280088445    0.051367032938886     -0.00335823763583564  -0.494051890472452  0.0854605074995414    -0.00671052143778972  0.000421308437952812  0.3479 0.4329      0.55537   0.53572 0.42861 0.40000 5   7   16   36
1.65                    -10.0773596863429             10.0945345077145          -5.62103104630827   1.79568990877141      -0.302570763856558   0.0252746218779301    -0.000829795694171467 1.46069813353868     -0.620935773461294   0.0767271445837206   -0.00352052113803297  -5.10090607883753   1.16603114956087     -0.128005699946137    0.00800735449327573   2.82694076295741    -0.551599135210873    0.0505271036262652    -0.00322449137939249  -0.502879540284296  0.0869275554459288    -0.00661957175138667  0.00040453828247766   0.3486 0.4318      0.55495   0.53544 0.42721 0.40000 5   7   16   36
1.7                     -10.1362664181032             10.0984447883904          -5.60256689474218   1.78294333773516      -0.299249154344056   0.0248991537624412    -0.000814309248878937 1.48225448635607     -0.596818825713523   0.0681272625907554   -0.00285678417449599  -5.11008883422233   1.12040040466296     -0.114030933201795    0.00698227655648889   2.82580187006214    -0.527036295008853    0.0435405820820607    -0.00272958908953991  -0.502128003876641  0.0827701617309949    -0.00548771519430278  0.000326313317663301  0.3485 0.4307      0.55401   0.53517 0.42585 0.40000 5   7   16   36
1.75                    -10.2295715995199             10.1466003575197          -5.60826490555134   1.77737267699596      -0.297118356402037   0.0246264339749973    -0.000802394682996599 1.53023088002827     -0.588383747264569   0.0624313954750566   -0.00236358491343972  -5.14766949021333   1.09227462043347     -0.103367085739084    0.00615392243374283   2.83772347285334    -0.510740854936643    0.0381356756255752    -0.00232903562523979  -0.503508006222609  0.079984462702008     -0.00462007973093286  0.000263880370725398  0.3470 0.4296      0.55226   0.53491 0.42453 0.40000 5   7   16   36
1.8                     -10.3422887144858             10.212159957605           -5.6208763912044    1.7735461434229       -0.295268937066378   0.0243786026843957    -0.000791368480060055 1.6262680198153      -0.608871902023914   0.0621657406544559   -0.00219367238188685  -5.24620221489038   1.1022851892707      -0.100073237505036    0.00577439932501019   2.88078525478757    -0.514372687209383    0.0366402743818239    -0.00216920273737101  -0.510348309337974  0.0807385101628484    -0.00445291520913117  0.000244838430336911  0.3469 0.4286      0.55137   0.53465 0.42325 0.40000 5   7   16   36
1.85                    -10.409223931235              10.2204553737294          -5.60255945071256   1.76091757331718      -0.29205547184858    0.0240223803986091    -0.000776892964121042 1.67684579518646     -0.602733760169828   0.0570128494996963   -0.00173828477612535  -5.28565949308683   1.07756020732157     -0.0903570718882531   0.00501647924066341   2.89416614996833    -0.500467008507929    0.0318765603079016    -0.00181511650534767  -0.512082025621588  0.0784572929795042    -0.00371599588514746  0.000191684148172938  0.3448 0.4276      0.54931   0.53440 0.42200 0.40000 5   7   16   36
1.9                     -10.468417738647              10.2135738310116          -5.57419591823112   1.74527195984186      -0.288382684469521   0.023631376440431     -0.000761371929811586 1.74628273318211     -0.608660298448307   0.0541991464848477   -0.00142469445210626  -5.3489575107089    1.06889509562231     -0.0838539618183151   0.00445758166462359   2.91994347417249    -0.495115861794134    0.0288458454298997    -0.00156920905819056  -0.516027337203693  0.0777181856262201    -0.00329320742895359  0.000158202758022021  0.3435 0.4266      0.54775   0.53416 0.42079 0.40000 5   7   16   36
1.95                    -10.5084504481979             10.180296231089           -5.53079742424569   1.72530690239271      -0.284049954938685   0.0231891595052711    -0.000744261836172083 1.8091062161697      -0.611260193607292   0.0508532868063176   -0.00108361196871344  -5.40408392933899   1.05660344416972     -0.0768395687292616   0.00387546529010937   2.94210237660972    -0.488345522974357    0.025642248767997     -0.00131677391024255  -0.519423184334577  0.0767875232138236    -0.00285112748391805  0.000124219336015851  0.3422 0.4257      0.54621   0.53392 0.41961 0.40000 5   7   16   36
2                       -10.5694580357158             10.1775937786133          -5.50510629800079   1.71070905014594      -0.280609790600401   0.0228236390398249    -0.000729804801109328 1.87037999559173     -0.613037376655987   0.0473539000396686   -0.000732843321271107 -5.4525398645741    1.04054742268773     -0.0691074307216372   0.0032484549304439    2.95964663459482    -0.478900370653217    0.0219148697071122    -0.00103089834705024  -0.521875068669127  0.0752947910457592    -0.002296638991388    8.29509250680297e-05  0.3412 0.4248      0.54482   0.53369 0.41845 0.40000 5   7   16   36
2.5                     -10.2602556261818             8.87638196150135          -4.5242408333601    1.3584472470141       -0.215009319667911   0.0167706855663364    -0.000511428718707286 2.44184686768443     -0.664896004725838   0.0257728877466416   0.00169340576622647   -5.93964112524486   0.977583881233024    -0.0214603690188623   -0.000801714368981106 3.16626060088194    -0.455644011280593    0.00402879982569178   0.000439582369099429  -0.55579549879936   0.0752745892096795    -0.000596136096474022 -6.10134876875475e-05 0.3321 0.4166      0.53278   0.53166 0.40830 0.40000 5   7   16   36
3                       -9.45623258696517             7.01317203648444          -3.26289083904284   0.939140910213432     -0.141665851955771   0.0103360960257266    -0.000288550365398866 2.85102896975277     -0.716317703198826   0.0151430819258944   0.00304380340913452   -6.27834890447373   0.965695999055653    0.00122862307239267   -0.00279844461926157  3.32409785933278    -0.473582487988736    0.00118370920244588   0.000761313212304736  -0.584310983452188  0.083684300958259     -0.00163636686985252  -4.07269064600723e-06 0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
3.5                     -8.53196014508671             5.12852696332062          -2.03468181243755   0.543041707441337     -0.0742387600489656  0.00456259085233062   -9.28519994813575e-05 3.13007497528117     -0.769174609104747   0.0135160983186569   0.00353423439873387   -6.51560386860705   0.999973798212628    0.00410858468737169   -0.00322657613898817  3.44742970319369    -0.521574946720883    0.00891382241814374   0.000295311570621531  -0.608228652858042  0.0973734136990582    -0.00440415355190766  0.000177253087246185  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
4                       -7.04407963381076             3.00228282271771          -0.812585029085377  0.169842989822711     -0.0123948718696135  -0.000640419857701629 8.0796602200645e-05   1.34417375915254     0.284179792083092    -0.179181719865672   0.014703074497891     -4.20163000410498   -0.35901874286434    0.252650365880472     -0.017656397429983    2.30709920692371    0.134000508590595     -0.109691170722201    0.00714634014660406   -0.415813047924639  -0.0113027262228926   0.015102211351161     -0.000946277753963273 0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
5                       -5.44541155330971             0.153988002508143         0.942261783030322   -0.371159239830323    0.07583733604444     -0.00788389777687502  0.000316370726828705  1.31359245258169     0.307280267568705    -0.182860475511287   0.0148599492478095    -4.09254320501498   -0.368730538307707   0.246745144764532     -0.0169721718851667   2.28062230823202    0.0861058143547161    -0.0932657633517613   0.00588842737009974   -0.416271040771956  0.00418323984262617   0.0105401456010585    -0.000616045394548169 0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
6                       -4.45975309758623             -1.64262710404888         2.03560945785358    -0.699970779925666    0.127588011057734    -0.0119505583775958   0.000442217943673367  1.57158959986427     0.0639612143682541   -0.127055271281243   0.011149989106092     -4.44903871755647   0.00972395151643453  0.157484854148027     -0.0109848016192904   2.50190140858951    -0.154690677112432    -0.0366764797308073   0.00212206788693821   -0.460102277652494  0.0517564593607709    -0.00055252398416054  0.000116734705954121  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
7                       -4.14323654884943             -2.26579530621173         2.38823737108443    -0.797186145143462    0.140929653423117    -0.0127867725606678   0.000459663944298618  1.44727887898932     0.0374573811057174   -0.111117633124041   0.00980381258510162   -4.31040069865685   0.0863626827552974   0.126909306333092     -0.00862457518071756  2.46139619644971    -0.225116044695943    -0.0145097864802452   0.000532802900031743  -0.457009639107258  0.0674806842676655    -0.00508071111283919  0.000430131135194113  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
8                       -3.92072601500178             -2.65162634232736         2.5768640495621     -0.841209365490053    0.145397582002376    -0.0128884178980405   0.000453225640814314  1.13904880338633     0.122523751207596    -0.116448731986259   0.00974099211112658   -3.94041266194955   0.00989712703718198  0.127127141054346     -0.00818841672354807  2.29741042856605    -0.20748957690224     -0.0106722188786461   0.000109102800775492  -0.431553276544808  0.066547014196589     -0.00608771654680961  0.000518295539478288  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
10                      -5.22076099376969             -1.23900957805002         1.7752748976784     -0.58631759135787     0.0995277529016549   -0.0086016006724535   0.000293483860485304  2.70286309864854     -0.952586915768218   0.0974738878185804   -0.00333130916422102  -5.94603095036781   1.43751971445192     -0.160491201877887    0.00947392815453608   3.32198656195457    -0.950282566950248    0.139425152495734     -0.00909394342526459  -0.610236415803407  0.196754717126982     -0.0323835824944728   0.00212760384684684   0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
""")

COEFFS_FORELAND_20Bars  = CoeffsTable(sa_damping=5, table="""\
IMT                     a1                            a2                        a3                  a4                    a5                   a6                    a7                    a8                   a9                   a10                  a11                   a12                 a13                  a14                   a15                   a16                 a17                   a18                   a19                   a20                 a21                   a22                   a23                   tau    mean_phi_ss sigma_tot phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv                     -5.98828131155829             4.22469803134838          -2.04120191203769   0.711900097679621     -0.134707909947453   0.0124643591916466    -0.000446133011460594 0.630842790775833    -0.0928130260632013  -0.0234695312288233  0.00237307994631626   -2.25270036839014   -0.586289153227759   0.202879758774249     -0.0116107156904395   1.12087240865561    0.443071405843821     -0.137998604186644    0.00818930883574762   -0.211714156943433  -0.0829839987851407   0.026367887527244     -0.00164429058370831  0.5010 0.4600      0.68015   0.00000 0.00000 0.00000 5   7   16   36
pga                     -3.57341468559359             3.89406666605519          -1.9910815291063    0.713864870869433     -0.137633907435241   0.0128860596514615    -0.000465319011030701 0.733374338472405    -0.2677335171402     0.0215002442160946   -0.000820469906954345 -2.38580467000541   -0.410061286858664   0.162211794259492     -0.00898095055760459  1.19263829294972    0.34958518594612      -0.119460706718737    0.00716101400712219   -0.242827667755347  -0.0587703943125401   0.0219147348475217    -0.00141343332876649  0.3532 0.4600      0.57998   0.58000 0.47000 0.35000 5   7   16   36
0.01                    -3.66743153936838             4.0833366945383           -2.11281958305273   0.751891855924717     -0.143853828238472   0.013400738903455     -0.000482400908626494 0.700472243381621    -0.210633524471801   0.00677312909063763  0.000196126792663533  -2.2771807700428    -0.524807253924      0.188629454411503     -0.0107224250740284   1.10158713459205    0.426553333105461     -0.136382477398065    0.00826297813046203   -0.222612248279306  -0.0743960964551353   0.0252900732897098    -0.00163279108560039  0.3529 0.4600      0.57975   0.58000 0.47000 0.35000 5   7   16   36
0.02                    -3.21721322756304             3.6646823135325           -1.90305837020615   0.695760572681955     -0.135367246950739   0.0127216417047358    -0.000460351773686727 0.994903323785383    -0.184224404389253   -0.0198431165991221  0.00253479602614211   -2.17036101349235   -0.823753543345508   0.271226690744581     -0.0166998341243554   0.779685365326613   0.697568003950682     -0.19984802045031     0.0126484320428385    -0.134015426371943  -0.135332171605082    0.0388312763299031    -0.00255479067195552  0.3674 0.4570      0.58639   0.56796 0.46097 0.37408 5   7   16   36
0.03                    -2.92162816383717             3.92740274999306          -2.11285847523795   0.759937032622705     -0.145818034420719   0.0136104320285313    -0.000490952093098338 -0.0465062951426165  -0.185177721199031   0.046032478581522    -0.00377984698227908  -0.913573079089911  -0.699614486293161   0.164936070354633     -0.00759789820087996  0.240901235442149   0.551475105946652     -0.13235084216017     0.00736075351745219   -0.0674038086917286 -0.0936210352789223   0.0244589841863351    -0.00149727616225175  0.3749 0.4552      0.58975   0.56092 0.45569 0.38817 5   7   16   36
0.04                    -2.25135518696001             3.39855473521377          -1.83255084185381   0.674594132094176     -0.131821812415327   0.0124524357507197    -0.000452719561937173 -0.820882091448008   -0.159028461190703   0.0836107502988822   -0.00745308647359514  -0.524734107262933  -0.406695418877261   0.0653016884293525    -0.000242511278972384 0.382745341949978   0.241136878784618     -0.0565932729133551   0.00228054314610945   -0.145244852633547  -0.0168099777905328   0.0077793058471597    -0.000425004583348209 0.3801 0.4540      0.59211   0.55592 0.45194 0.39816 5   7   16   36
0.05                    -3.08846069144692             4.51102140688791          -2.44732498313647   0.852332388983161     -0.159755320154826   0.0147156846829954    -0.0005267948218755   -0.556782829108367   -0.158922237736494   0.0637131572670605   -0.00545330213481491  -1.06593387962841   -0.346054697853019   0.0855000585522369    -0.00264740167472607  0.808182864483781   0.170287856126069     -0.0625906992234232   0.00331047720435551   -0.238904927804963  -0.00156580602859809  0.00873528026200072   -0.000607879132769956 0.3855 0.4530      0.59483   0.55204 0.44903 0.40592 5   7   16   36
0.1                     -5.77381522064529             6.65996416583251          -3.35756903821643   1.08316010690227      -0.193303877788584   0.0172815476550876    -0.000607076181010974 2.12601290048317     -0.662175838027661   0.047130186318862    -0.000523491774349856 -3.64017330942559   -0.166726253495373   0.168034457325214     -0.0111979650724834   1.79350253795703    0.297345734002743     -0.13593526005622     0.00900698071502286   -0.336433231811581  -0.0680465436829255   0.0281400134403636    -0.00191851316128736  0.3864 0.4500      0.59312   0.54000 0.44000 0.43000 5   7   16   36
0.15                    -4.5288509087307              3.99833772308925          -1.62619089206225   0.544283495187664     -0.104883880945589   0.00991093079846204   -0.000361094064431474 2.10720468151892     -0.633985518429212   0.0426658009613842   -0.000319923878237958 -3.46100017499033   -0.188523730477986   0.155800711057944     -0.00975061417404652  1.63840797633576    0.314128369824754     -0.125411149912236    0.00779761052958059   -0.291044767951622  -0.0731346991660461   0.0258260112840425    -0.00164709035797563  0.3841 0.4675      0.60507   0.58095 0.47510 0.40075 5   7   16   36
0.2                     -4.99984005680657             4.11052397450312          -1.61080152090012   0.538229619548805     -0.104869254431855   0.0100140968340209    -0.000367897423150802 2.22132219342131     -0.788742529576432   0.0811655023213542   -0.00295235824144577  -3.59152533519282   0.0661536136258476   0.0888576270385245    -0.00504557886232003  1.6947955165186     0.174945917275703     -0.0878904291780187   0.00512684233567758   -0.295062472580705  -0.0481256996538392   0.0189253746018569    -0.00115170776666187  0.3690 0.4800      0.60546   0.61000 0.50000 0.38000 5   7   16   36
0.25                    -5.37904869703009             3.74992517919784          -1.26461712960997   0.434722766437382     -0.0901994893707853  0.00899198673205055   -0.000339362452234875 4.40283817117515     -2.07079732535419    0.315576651098064    -0.016543606294086    -6.29863507051178   1.73150466708218     -0.224346344582829    0.0134981759840357    3.02432804942507    -0.66190457271849     0.0716479605427892    -0.00442035473425382  -0.517523069423763  0.0956137778023706    -0.00872107332528764  0.000514078573144588  0.3445 0.4800      0.59082   0.62651 0.50000 0.37450 5   7   16   36
0.3                     -4.85235731757604             3.30385301243137          -1.1754045451144    0.44029552386053      -0.0943141515955869  0.00951134724654604   -0.000361191652743864 1.87000677934739     -0.696372924899498   0.0756549175343424   -0.00301785919423031  -3.13324484743424   0.00257566976030655  0.081279933576891     -0.00395874489381901  1.44444134878591    0.211723273660018     -0.0848991622718174   0.00462648668558162   -0.244139824805915  -0.056233585337054    0.0187836077632297    -0.00108977326989945  0.3377 0.4800      0.58688   0.64000 0.50000 0.37000 5   7   16   36
0.35                    -3.70942019447747             1.50445491965538          -0.190742000059618  0.168871113254748     -0.0535630142690284  0.00632897704874315   -0.000259974290867697 1.63049830494119     -0.608501437317624   0.0649225903156644   -0.00258163683612026  -2.92071808442561   -0.0374467721684619  0.0804296025290412    -0.00362291902211861  1.35578823929439    0.221152846245511     -0.0826488777257522   0.00435273248388937   -0.229115879071968  -0.0572451067264152   0.0183288294863526    -0.00103991179123528  0.3482 0.4740      0.58812   0.62793 0.49396 0.37000 5   7   16   36
0.4                     -4.58812652459041             2.73243135706965          -0.980598202817508  0.424834763237566     -0.0971853833949476  0.0100682923424894    -0.000387158284203658 1.65676062082992     -0.721855566949741   0.0973215302467968   -0.00494169070550364  -3.15058241977165   0.239065459986138    0.0129012800324908    0.00101560227234715   1.5212104988611     0.0521185544905542    -0.0429466268398732   0.00166086910231919   -0.262017076164754  -0.0250961663416329   0.0109234165685042    -0.000540788883086446 0.3552 0.4687      0.58811   0.61747 0.48874 0.37000 5   7   16   36
0.45                    -4.43630114309848             2.55232050761433          -0.968719720680485  0.441910974661534     -0.10198346400274    0.0105605486585054    -0.00040519487426458  1.22710006784167     -0.519733639167352   0.065368753745662    -0.00325726234720819  -2.74483186688519   0.0611233634356745   0.0402041214636742    -0.000413618066212469 1.34214767158186    0.132545289416859     -0.0558022414815011   0.00235935347359534   -0.232158867730963  -0.0388152172814604   0.013230561437997     -0.000670999125682101 0.3495 0.4641      0.58097   0.60825 0.48413 0.37000 5   7   16   36
0.5                     -4.50586385222257             2.67755773757453          -1.11776122896163   0.502844538374866     -0.113331891496302   0.0115594473467057    -0.000439042642223728 1.16037153268301     -0.558282069643029   0.0813487889318559   -0.004540093200322    -2.88377827050308   0.242769710713886    -0.00456373193646096  0.00265718549807202   1.46258557294865    0.0145383193771725    -0.0286066837426709   0.000540164581600701  -0.257178175007891  -0.0161180828005224   0.00814721321433626   -0.000334381802846034 0.3548 0.4600      0.58095   0.60000 0.48000 0.37000 5   7   16   36
0.55                    -5.15438207637486             3.36151575443468          -1.51471408047366   0.627477535041698     -0.13417938576453    0.0133071926126174    -0.000496789953601184 2.40011474536106     -1.31485579890472    0.223219337799739    -0.0129185637131193   -4.65755250803145   1.33719320093522     -0.211846525247273    0.0150170004003977    2.3910883182983     -0.559909644439457    0.0807459876193047    -0.0060203699667461   -0.419678286548415  0.0848377363825648    -0.011132196467462    0.000827053808710764  0.3639 0.4586      0.58544   0.59175 0.47587 0.37413 5   7   16   36
0.6                     -4.83340257889472             3.26109869434841          -1.59512969959635   0.669473246952341     -0.141735464955265   0.0139150826166185    -0.000515372571596543 0.941350317580372    -0.559799921212877   0.0963261688764287   -0.00598388459734347  -3.03635153513865   0.482525844854539    -0.0653271043228924   0.00686039549318956   1.62964517303555    -0.14945520888011     0.00907097821170556   -0.00197238765810458  -0.293166858157963  0.015609691899728     0.00112773264857076   0.000127431817609132  0.3757 0.4574      0.59187   0.58422 0.47211 0.37789 5   7   16   36
0.65                    -5.66832412857763             4.63481101051284          -2.47969305674159   0.943519962963147     -0.18598267590204    0.0175184033334463    -0.000632565013734294 0.291530313449288    -0.196251035128097   0.0308036781157718   -0.00217451076710301  -2.30850469855929   0.0477761696433289   0.0175708973463853    0.00181795679645024   1.26816896863939    0.0780599565509183    -0.0358415257327472   0.000823043090021284  -0.229372284336283  -0.0257198575961361   0.00945536545540426   -0.000397469984560453 0.3831 0.4562      0.59573   0.57729 0.46864 0.38136 5   7   16   36
0.7                     -6.17705594075872             5.16379397817055          -2.77127526606286   1.02929710337578      -0.199392993586426   0.018567508292261     -0.000664802119541237 1.45765527048981     -0.916800211965164   0.167405211610659    -0.0103201846818747   -4.0552331368284    1.13268105752493     -0.189986039106353    0.0143234793312391    2.20731786980262    -0.507387074214674    0.0769646112444735    -0.00602863150254277  -0.397083749704622  0.0793393543061661    -0.0108943837962966   0.00084563487335472   0.3886 0.4551      0.59846   0.57087 0.46544 0.38456 5   7   16   36
0.75                    -6.34713907506838             5.70920288767914          -3.18940004863404   1.16278992698438      -0.220528114232266   0.0202232709494643    -0.00071627078824952  0.224563607971506    -0.25235933466936    0.0521685161389337   -0.00386214080421185  -2.63156964156843   0.334118396097342    -0.0466229908169065   0.00605525425926096   1.51758638957525    -0.108762478815573    0.00383612236553557   -0.00174382188473398  -0.279957250505553  0.0104064581521428    0.00192674958366209   8.72286921021733e-05  0.3852 0.4542      0.59554   0.56490 0.46245 0.38755 5   7   16   36
0.8                     -6.43050444458713             5.9140894627738           -3.35512099373672   1.21453696991243      -0.228088756580686   0.020742250263521     -0.000729637481771929 -0.231106432897431   0.0190719074824216   0.001111625880063    -0.000804633247724106 -2.14133363030437   0.0133558558224274   0.0177698004356313    0.002016445592287     1.27705112542298    0.0578042026024075    -0.0306286796757161   0.000456658377886314  -0.237676794661635  -0.01964986489366     0.00824116144867916   -0.000319300075538908 0.3850 0.4532      0.59468   0.55932 0.45966 0.39034 5   7   16   36
0.85                    -7.37317383708032             7.07865273698934          -3.99937347746666   1.39939872451042      -0.256590786084458   0.0229816694153078    -0.000800009625038322 0.823934426979039    -0.616197332692308   0.119798876855388    -0.0078176807446046   -3.68553895766062   0.947958227247019    -0.158611112336332    0.0125610154752374    2.099694488503      -0.44315756353929     0.0648090977614544    -0.00530622999650408  -0.383992841894283  0.070063517934674     -0.00897221274132163  0.000727559653461543  0.3811 0.4523      0.59150   0.55407 0.45703 0.39297 5   7   16   36
0.9                     -7.77412906233071             7.68112558896073          -4.36898886287413   1.5070234560027       -0.272703659156599   0.0241860707053853    -0.000835693696168616 0.683106122094917    -0.540356499650147   0.106468030176341    -0.00705625467691949  -3.61083530191382   0.893420792321567    -0.14707266880954     0.0118199075779246    2.07861535744459    -0.420631783272988    0.0594913260995943    -0.00495139418712411  -0.381333708510264  0.0664169392869299    -0.00806748149312851  0.000666690633903782  0.3799 0.4515      0.59006   0.54912 0.45456 0.39544 5   7   16   36
0.95                    -8.12257754199402             8.18875745119941          -4.67556293614459   1.59442472104043      -0.28542671958348    0.025104319969879     -0.00086176854008865  0.569999363880391    -0.47256885682069    0.0936783931598852   -0.00628976598115535  -3.55479268674596   0.84021476999346     -0.134724168665651    0.0109914941533314    2.06357733604019    -0.397478301017078    0.0536153599734384    -0.00454750111839654  -0.379466907515038  0.0626030090811841    -0.00706803368667834  0.00059795439846149   0.3733 0.4507      0.58525   0.54444 0.45222 0.39778 5   7   16   36
1                       -8.49153478335097             8.70965852253233          -4.98280764427285   1.68075252599477      -0.297861746029965   0.0259938107015725    -0.000886812244808537 0.485474232433898    -0.415128130394581   0.0820706258669508   -0.00556475393395194  -3.51667537101372   0.790058043218917    -0.122136564189013    0.0101197776391369    2.05352418903608    -0.374040567219721    0.0473545146420795    -0.00410928883034758  -0.378117406097671  0.0586097717547312    -0.00598807027149671  0.000522878207578795  0.3716 0.4500      0.58362   0.54000 0.45000 0.40000 5   7   16   36
1.05                    -8.78316468731945             9.10516170169037          -5.21183193753122   1.7429457539015       -0.306356555411532   0.0265570448516982    -0.000901063319945489 0.412674195723398    -0.358219707275981   0.0698480736790464   -0.00477671225422826  -3.47965079351904   0.733401547154029    -0.107579710513579    0.00910400510652071   2.04125264257374    -0.346272492743311    0.0399735767833273    -0.00359334790024148  -0.376161840107934  0.0538062331827069    -0.00471141585249521  0.000434444149481581  0.3691 0.4482      0.58063   0.53956 0.44778 0.40000 5   7   16   36
1.1                     -9.08252989246715             9.49574001661641          -5.43186265870531   1.80151356245484      -0.314202983330648   0.0270653716819638    -0.000913514075021275 0.373706056449137    -0.316281005963086   0.0598219089204955   -0.00409540042147623  -3.47074404342541   0.688015363883033    -0.0945263489248494   0.00815475468266944   2.04011206132678    -0.322963412429111    0.0331935606116428    -0.00310445843260428  -0.37588591129597   0.0496993584745717    -0.00353380269120583  0.000350741533407965  0.3650 0.4465      0.57674   0.53913 0.44566 0.40000 5   7   16   36
1.15                    -9.35146884448661             9.83594591800505          -5.61954369378866   1.84995580778359      -0.32039220802898    0.0274372700003986    -0.000921503045447979 0.33091455723867     -0.26821815270269    0.0482755224665843   -0.00331162524540111  -3.44724030940539   0.630211721323597    -0.0788536474232902   0.00704278764241049   2.03023868272936    -0.293156605915985    0.0251229163298372    -0.00253835305450681  -0.374055634277293  0.0445139689857076    -0.00215103842145836  0.000255139981589808  0.3627 0.4449      0.57402   0.53873 0.44364 0.40000 5   7   16   36
1.2                     -9.58452120954388             10.0979304707973          -5.75405269384833   1.88169049700263      -0.323868137781988   0.0275871540396382    -0.000922231360830156 0.364747118424208    -0.259474716732061   0.0432935689853954   -0.00288526726831277  -3.50573053149538   0.614972227478628    -0.0703817192571735   0.00632823196507034   2.05782993548398    -0.283208051954941    0.0204689061435873    -0.00216341546846674  -0.378309621852216  0.0426087931602139    -0.00134029730652552  0.00019186919161676   0.3625 0.4434      0.57267   0.53834 0.44170 0.40000 5   7   16   36
1.25                    -9.77297814727847             10.3070168814434          -5.85956329178316   1.90489943894447      -0.325982101495773   0.0276262620820698    -0.000919375783516587 0.3307826409734      -0.210823633075817   0.0310228307044625   -0.00203673283179342  -3.47704651554911   0.548860550133368    -0.0526891924696789   0.00508294454695411   2.0430735256879     -0.248857983756995    0.0114340557637169    -0.00153816272212647  -0.375555174622301  0.0367074764222706    0.000181396623609115  8.82711995818705e-05  0.3631 0.4419      0.57195   0.53797 0.43984 0.40000 5   7   16   36
1.3                     -9.95025772586989             10.4809474411121          -5.9395746773739    1.92013067402004      -0.326841207034534   0.0275672882746528    -0.000913478479578194 0.357063373870765    -0.193016974321303   0.0238973573105562   -0.00146822709727571  -3.51233551350598   0.51606297291964     -0.0406344820880282   0.00414784267146591   2.05734411505391    -0.229959293852338    0.00504845885478314   -0.00105956403266052  -0.377469313965511  0.0333273469916988    0.00126808532325598   8.74089342647141e-06  0.3610 0.4404      0.56949   0.53761 0.43806 0.40000 5   7   16   36
1.35                    -10.090496498566              10.5957488424415          -5.98296542697641   1.92419459982193      -0.325904202809691   0.0273619660832034    -0.000902816266233147 0.3852667870961      -0.174531842436071   0.0164626021184861   -0.000875059544983714 -3.54360512583889   0.479511391774768    -0.0277745322485265   0.00316240935335707   2.06866824321813    -0.208994673134279    -0.00173373306109116  -0.00055764619607057  -0.378830966840392  0.0296051286604122    0.00241583085287285   -7.42193286973906e-05 0.3605 0.4391      0.56808   0.53727 0.43634 0.40000 5   7   16   36
1.4                     -10.2192639486074             10.6912413419697          -6.01424232200576   1.92472774821567      -0.324446223132162   0.027118930833056     -0.00089109451225632  0.41956227291611     -0.158673366635739   0.00944197190295878  -0.000305295497654029 -3.57964777380876   0.445812425998312    -0.0155278067015799   0.00222010093514361   2.08260841396783    -0.190082252347382    -0.008030383627695    -9.06728123168111e-05 -0.380749407866562  0.0263575415898934    0.00344946086587065   -0.000148988292413501 0.3616 0.4377      0.56779   0.53694 0.43469 0.40000 5   7   16   36
1.45                    -10.3167359043056             10.7365420554558          -6.0147821149609    1.91605813664255      -0.321539025110038   0.026760367419381     -0.000875691932328589 0.460084387425757    -0.145050026855405   0.00268172706934094  0.000254943895805085  -3.61747545800657   0.412209281550261    -0.00321265261005921  0.00127067993961457   2.09638737103585    -0.170899616952164    -0.014390470901407    0.000380745392336723  -0.382560306689028  0.0230465842650334    0.00449447873975618   -0.000224510757424676 0.3604 0.4365      0.56604   0.53662 0.43309 0.40000 5   7   16   36
1.5                     -10.3921041608629             10.7461930370297          -5.99348523372892   1.9009172654215       -0.317630174469185   0.0263236565653413    -0.000857858642622973 0.509056512884372    -0.135659931671282   -0.00336221343306838 0.000774846914773066  -3.66312384930888   0.38331693088807     0.00818450597766859   0.000378422396586375  2.11396667874219    -0.154359721091385    -0.0202009999808215   0.000816949963845799  -0.385050008139115  0.0202406542156117    0.00543192823538711   -0.000293097499432613 0.3559 0.4352      0.56223   0.53631 0.43155 0.40000 5   7   16   36
1.55                    -10.4643609380134             10.7448700178753          -5.96402382956744   1.88330849690722      -0.313360685234909   0.0258616070828598    -0.000839341648535762 0.577988711865138    -0.136882594561928   -0.00757616147446222 0.0011924610285917    -3.72720456251677   0.364669682439795    0.0177570457386058    -0.000409634010521279 2.13914443513153    -0.142245780494706    -0.0252059155182544   0.00120683573371888   -0.388670013567344  0.0181137821864061    0.00624460181709662   -0.000354538081461795 0.3507 0.4340      0.55804   0.53601 0.43005 0.40000 5   7   16   36
1.6                     -10.4610061361604             10.6413471592789          -5.8770747171044    1.84901576555509      -0.306475389076543   0.0251893681921052    -0.000814040763959727 0.6144572293268      -0.120372347766604   -0.01486032126295    0.00178065037352955   -3.75497455415318   0.327271562845515    0.0304094849533634    -0.00136045592981352  2.14872089382268    -0.122581944111327    -0.0313717513627732   0.00165394176370231   -0.389994552350295  0.0149557185887115    0.00720197830101026   -0.000422324104492301 0.3479 0.4329      0.55537   0.53572 0.42861 0.40000 5   7   16   36
1.65                    -10.4884969000741             10.5787740811896          -5.81309010620071   1.82166713713009      -0.300755085158244   0.0246183472765385    -0.000792276484630689 0.665792378512815    -0.111373036258438   -0.0209000568682745  0.00230128535734005   -3.79450157874777   0.295750239186237    0.0421145267931655    -0.00226180681119303  2.16217892641166    -0.104794504743796    -0.037262148686982    0.00208871353878844   -0.391734707983499  0.0119738105680258    0.00814121837467315   -0.000489908404498394 0.3486 0.4318      0.55495   0.53544 0.42721 0.40000 5   7   16   36
1.7                     -10.4912303422108             10.4758335079193          -5.7243913808557    1.78698590918265      -0.293892953905138   0.0239571515374317    -0.000767654557879517 0.737683675471489    -0.115312327874788   -0.0244384685609477  0.00267026719545262   -3.86125953231969   0.282594124199552    0.050111444073619     -0.00293158328766064  2.19039706923696    -0.0972838786122354   -0.0410409409414894   0.0023901235050721    -0.39619795746017   0.0109149408514596    0.0086818750526648    -0.000532164920066027 0.3485 0.4307      0.55401   0.53517 0.42585 0.40000 5   7   16   36
1.75                    -10.4535175711189             10.3219088826826          -5.60823031689318   1.74460588513617      -0.285863012994452   0.0232052838850381    -0.000740211367623528 0.777581894880418    -0.101243941142804   -0.0311810011515321  0.00322143087082014   -3.88697045875607   0.246497999701477    0.0621810683242123    -0.00383281377748426  2.19818723175256    -0.0783634973538746   -0.0468541310042651   0.00280780466898239   -0.397167248862962  0.00789703746297095   0.00957402371794798   -0.000594645549484384 0.3470 0.4296      0.55226   0.53491 0.42453 0.40000 5   7   16   36
1.8                     -10.3935834321793             10.130413211576           -5.46892785586262   1.69536587149895      -0.276767121502411   0.0223694064267076    -0.000710110538210373 0.838153401985076    -0.0990842283933315  -0.0357637068247589  0.00364800610836208   -3.93727187065978   0.225452233238592    0.0714182448832924    -0.00456699573804303  2.21829618413555    -0.0672384311870301   -0.0511785200077873   0.00313726166868171   -0.400280380641791  0.00625804372674076   0.0102013856640705    -0.000641452031146785 0.3469 0.4286      0.55137   0.53465 0.42325 0.40000 5   7   16   36
1.85                    -10.40611385189               10.0390213758623          -5.38659153423757   1.6631202334281       -0.270441976285288   0.0217662513915031    -0.000687869409613521 0.931642447012803    -0.116843901699174   -0.0366009416918302  0.00385175865529685   -4.02544016128506   0.22819741786373     0.0760783136714186    -0.00502458155490882  2.25642109031463    -0.0676912428604111   -0.0532507582778242   0.00332975851457215   -0.406392866438496  0.00657256653163818   0.010446390809342     -0.000664928263216738 0.3448 0.4276      0.54931   0.53440 0.42200 0.40000 5   7   16   36
1.9                     -10.3658945578495             9.87991831168562          -5.26742032113573   1.62042417082743      -0.262506595493475   0.0210356616932901    -0.000661575151573827 0.99157627814893     -0.116129784681468   -0.0406820551171594  0.00423864109893077   -4.07403521004895   0.209534432686834    0.0844555490414789    -0.00569046014962139  2.27595194405844    -0.0581669741090384   -0.057051786772005    0.00361923141052139   -0.409458991380417  0.00525482597891737   0.0109748652694976    -0.000704373417149337 0.3435 0.4266      0.54775   0.53416 0.42079 0.40000 5   7   16   36
1.95                    -10.3030863930942             9.67843161550122          -5.1214105933905    1.56971709059232      -0.253325186618029   0.0202068332249728    -0.000632167173395444 1.09274999976238     -0.139492107253532   -0.0403512352396002  0.00436820456487682   -4.17283086338522   0.221229240649827    0.0871324878346404    -0.00601798726117064  2.32029354938669    -0.0639590268291324   -0.0579392104093056   0.00373422921495046   -0.416772391463613  0.00659239436946594   0.0109940063610383    -0.000713142301321088 0.3422 0.4257      0.54621   0.53392 0.41961 0.40000 5   7   16   36
2                       -10.2348905209933             9.48733406259205          -4.98590625024409   1.52271635484037      -0.244785826910561   0.0194335037415287    -0.000604676200204132 1.12531793923266     -0.125408055809236   -0.0465501933678284  0.00486456231458812   -4.1881574220467    0.187118050968868    0.0978520096892313    -0.00680035821586471  2.32415483817185    -0.0473928903443825   -0.0627833815436359   0.00407469561473149   -0.417282163718103  0.00414591347491212   0.011687753275843     -0.000760605398127038 0.3412 0.4248      0.54482   0.53369 0.41845 0.40000 5   7   16   36
2.5                     -9.22692088448393             7.11999437553273          -3.38132070694569   0.990120019109023     -0.151783924091792   0.0112845746832992    -0.000322650735472411 1.74912644440964     -0.20756826709422    -0.0619928107278932  0.00688924858597445   -4.72821762665886   0.156360558354821    0.138557109161023     -0.0103769480402776   2.56542085252764    -0.0463387166018929   -0.0760414735837069   0.00524511865681681   -0.458389294002624  0.00880694251613388   0.0124389359572783    -0.000845185769316043 0.3321 0.4166      0.53278   0.53166 0.40830 0.40000 5   7   16   36
3                       -7.5005875983168              4.17908633954675          -1.57147564700283   0.422997028132706     -0.0568472115813722  0.00324425434764633   -5.22972799332388e-05 1.0698736653467      0.325898879907676    -0.172992634844557   0.0137862387030204    -3.70737676718078   -0.602089924638517   0.291953969647134     -0.0197291864730914   2.05619514889096    0.307073807312054     -0.14486226559542     0.00933272792955062   -0.373998206300486  -0.0462745005520869   0.0228020965031206    -0.00144596176657555  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
3.5                     -7.70898329089191             4.04744636756378          -1.41395844698286   0.359984153119025     -0.0452840629422931  0.00226924691451404   -2.19027674913545e-05 1.22097751274672     0.339354976355412    -0.185683506708435   0.014879397713419     -3.75266802387415   -0.676301033991508   0.314714091883612     -0.0213398743084218   2.07212361310344    0.323364598135072     -0.149661208866544    0.0096529807386254    -0.377623465080185  -0.0452054267360198   0.022584406033567     -0.00142920984081353  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
4                       -6.05332887522093             1.26373067336533          0.277751270740351   -0.156681512341692    0.0387177191275031   -0.00465114738978626  0.000205190418123404  1.18857307903963     0.418656355267906    -0.206491150735324   0.0163186816898281    -3.61316102748666   -0.804470923297884   0.342748123988649     -0.0231354395082388   2.00379726334836    0.365124021004332     -0.157377554892551    0.0101108198538167    -0.367299455807829  -0.0487882271964525   0.0229962302196334    -0.00144707489343457  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
5                       -4.65761679240962             -1.24469966597133         1.79539616509668    -0.611673382157254    0.110399407721484    -0.0103100208712071   0.000381600798868659  1.42104637224398     0.247773501713367    -0.169945401667784   0.0139445216488875    -3.8771845445307    -0.542960626999063   0.281154354237546     -0.0189729924305477   2.17283760878041    0.177127692505916     -0.112606624468992    0.00710583981661319   -0.402073191380701  -0.00894858369928854  0.0135560761352213    -0.000818892801735358 0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
6                       -4.60153290918582             -1.57441957797141         1.9823159427026     -0.657240479293405    0.114726758915294    -0.0103294396705445   0.000369371710619985  1.56354327021525     0.0677017836366456   -0.12566820593371    0.0109204772549014    -4.08971252047343   -0.248223604296966   0.208335225706943     -0.0140301092371045   2.3188164941194     -0.0170540122436434   -0.065752398489327    0.00398685308671943   -0.432511046744246  0.0300129406457788    0.00431889446898919   -0.000212019892704149 0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
7                       -5.02836177547903             -1.07501452812197         1.65818459690792    -0.546028498091288    0.093546737856253    -0.00824972967391894  0.000288431699659694  1.55870850591557     -0.04442244916612    -0.0923868754318035  0.00850177851219552   -4.10590053987738   -0.0622737360768474  0.156016726928822     -0.0103463704428281   2.35040759673527    -0.137387407499564    -0.0339460774379828   0.00182733397472708   -0.440915608410137  0.0535828587020419    -0.00169121566067482  0.000187118454853491  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
8                       -5.24234503506386             -0.854591761090886        1.492067237134      -0.48132288306305     0.0799702537628983   -0.00682691221474729  0.000230782703377148  1.47832007077731     -0.0884997080076282  -0.0752806901654426  0.00720708442764337   -3.93704477463443   -0.0345937640948584  0.140397665445395     -0.00917803420867819  2.24950169067876    -0.145693838134182    -0.0278051357757488   0.00141025165219786   -0.421330401978076  0.0536510876627531    -0.00235856378303534  0.00022572476317753   0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
10                      -5.83527455595751             0.0315464379234589        0.863133908189103   -0.26286289262433     0.0394612819186106   -0.00301217529196601  8.86282614046283e-05  0.846352357953211    0.125633253870989    -0.0992631150260203  0.00810480158349865   -3.12581664737427   -0.301683046315996   0.172320142929305     -0.0106273085763232   1.84569215436564    -0.0146718168019165   -0.0448478537605027   0.00232338769077233   -0.352419559663795  0.0307216734684378    0.000902995519959268  3.29773028069473e-05  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
""")

COEFFS_FORELAND_30Bars  = CoeffsTable(sa_damping=5, table="""\
IMT                     a1                            a2                        a3                  a4                    a5                   a6                    a7                    a8                   a9                   a10                  a11                   a12                 a13                  a14                   a15                   a16                 a17                   a18                   a19                   a20                 a21                   a22                   a23                   tau    mean_phi_ss sigma_tot phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv                     -6.60836235126132             4.84694009472205          -2.39084226280178   0.835493674863462     -0.158208778581342   0.0146540216411737    -0.000524856695766496 1.92813086913898     -0.846893255125564   0.115051313509965    -0.00574471498317203  -3.78047016610218   0.307370266479717    0.0358406913156406    -0.00162299576607241  1.87440812486284    -0.000858521915720337 -0.0544302370699523   0.00315378427875909   -0.339619026381192  -0.0072494828549807   0.0120355556800168    -0.000775824833908316 0.5010 0.4600      0.68015   0.00000 0.00000 0.00000 5   7   16   36
pga                     -4.26773876283286             4.66062276191853          -2.44992803379178   0.878819709238118     -0.169159826546462   0.0158421562150212    -0.000572599707089873 1.73983347947757     -0.836947798376102   0.124208975427573    -0.00676353729165467  -3.45900810523565   0.186976080204287    0.053881438747656     -0.00262167989008315  1.68253011351636    0.0805181136773349    -0.0709825650070436   0.00431840314963412   -0.319781003851133  -0.0172479353549835   0.0145245685626244    -0.000982657689383878 0.3532 0.4600      0.57998   0.58000 0.47000 0.35000 5   7   16   36
0.01                    -3.42070352607803             3.72252521509834          -1.97621534000435   0.745915839639278     -0.148138514568302   0.0141088037411728    -0.00051503166038575  0.45344413682068     -0.0432738734277859  -0.0262017353478419  0.002196384003409     -1.69505023049468   -0.926113313239115   0.268368523870818     -0.0155857254530764   0.740535249114651   0.678893701475896     -0.187446445965455    0.0114269967214264    -0.150567063514161  -0.125134607057212    0.0356665384302973    -0.00228176973543997  0.3529 0.4600      0.57975   0.58000 0.47000 0.35000 5   7   16   36
0.02                    -4.04430634548212             4.76336075706821          -2.58320776312728   0.929092361280602     -0.177989042622611   0.0165968462698239    -0.000598294837091144 1.4361312431513      -0.47301351659638    0.038039547911637    -0.00109001759513276  -2.59531346940146   -0.547096722296105   0.213910504049344     -0.0129650645650404   0.975409295833596   0.571879850390896     -0.173908170006465    0.0109495664422388    -0.164246698579761  -0.116265951600694    0.0349231140683876    -0.00229809247603578  0.3674 0.4570      0.58639   0.56796 0.46097 0.37408 5   7   16   36
0.030                   -3.45579209384007             4.62113254346883          -2.56185669553144   0.925050295651142     -0.177542567452163   0.0165926057803452    -0.000599488765045219 0.251513206869825    -0.393683348444572   0.0893740702058072   -0.00655522483850171  -1.16509479843494   -0.521690154545483   0.1260141358101       -0.00497042369653844  0.343865037003839   0.479990864175187     -0.116752259744696    0.00629682659428839   -0.0797845084899927 -0.085258766818236    0.0226259531339782    -0.00136931779508163  0.3749 0.4552      0.58975   0.56092 0.45569 0.38817 5   7   16   36
0.040                   -2.79688227544362             4.15311017889385          -2.33230914133473   0.857341993016611     -0.166623869292573   0.0157034092533232    -0.000570638206875094 -0.745781648202932   -0.220715318837517   0.0978623877609361   -0.00843477901275025  -0.430397342147539  -0.460628838841228   0.0730371557588036    -0.000530097088160579 0.279401240556468   0.309269173565666     -0.0692480283818452   0.00298894280113199   -0.117187257576869  -0.0359658865972596   0.0115272614356396    -0.000647537609264551 0.3801 0.4540      0.59211   0.55592 0.45194 0.39816 5   7   16   36
0.05                    -3.1229028474549              4.43803384398834          -2.44405019954542   0.883554064524002     -0.17023046428679    0.0159694765936261    -0.000578724611542772 0.0503975020621722   -0.559470758296552   0.1438069359582      -0.0104394689957902   -1.7708732386175    0.121632372774587    -0.0106458858147063   0.00352462472866558   1.14259554381025    -0.0507260946017048   -0.0168912152321279   0.00034631017967593   -0.290800766757071  0.0324621226736301    0.00166276996541916   -0.000144437272854676 0.3855 0.4530      0.59483   0.55204 0.44903 0.40592 5   7   16   36
0.10                    -5.09026583839153             5.64401304136817          -2.8419427553752    0.967177755293738     -0.180467463223072   0.0166210369826503    -0.000595495959262195 2.25092615890413     -0.740586376486769   0.0631646396896498   -0.00156497651651888  -3.6284946449303    -0.183754604774273   0.169952112760539     -0.0111696024215742   1.74227658392801    0.338600053669888     -0.143691098987928    0.0094167848907914    -0.322127248627667  -0.0791382332839997   0.0303118003795075    -0.00204175551883442  0.3864 0.4500      0.59312   0.54000 0.44000 0.43000 5   7   16   36
0.15                    -5.10624254886643             5.0282958034255           -2.37455109068042   0.816389897961101     -0.155467358220098   0.0145352278522892    -0.000526226834354929 1.76463724422637     -0.419944613561654   0.00245257247381534  0.00204282386310635   -2.84031655860942   -0.594259928549407   0.234256523965203     -0.0144561172128913   1.27696596754512    0.555881308218966     -0.173242666690805    0.0107203156880061    -0.223370515590068  -0.119000806751869    0.0350290841946772    -0.00221599076398908  0.3841 0.4675      0.60507   0.58095 0.47510 0.40075 5   7   16   36
0.20                    -5.08720778756587             4.45862860644353          -1.97423032742201   0.697218552783197     -0.13727927929937    0.0131287732765187    -0.00048238810105414  1.91801721534927     -0.603922172190824   0.047067419860385    -0.000978597819908711 -3.06850900535069   -0.269240983928317   0.15276307529471      -0.00883019202267885  1.39897402616893    0.369683314454044     -0.125971261865704    0.00742989378698663   -0.240951798056177  -0.0842955806076228   0.0261150874111664    -0.00159240660112848  0.3690 0.4800      0.60546   0.61000 0.50000 0.38000 5   7   16   36
0.25                    -4.86731107442626             3.69336125210304          -1.50138072948894   0.565275703336483     -0.118002270398934   0.0116835486791691    -0.000438277899750615 2.21696854573907     -0.853618617322861   0.100702442223089    -0.00438699804029047  -3.43089260828157   0.0814062049984911   0.074450237735473     -0.00375361531256659  1.5656168474533     0.194956064562119     -0.0863303987383769   0.00483719642388855   -0.264594890069468  -0.0550317861758321   0.0193997994137722    -0.00115087976416837  0.3445 0.4800      0.59082   0.62651 0.50000 0.37450 5   7   16   36
0.30                    -5.44901412946409             4.36050855128079          -1.93700471690231   0.712965132191721     -0.144117548185888   0.0139837803271423    -0.000518128968518246 2.03051647656925     -0.809654267732007   0.0997791673222152   -0.00460060972417005  -3.28260179967881   0.10910804099991     0.0573431921351985    -0.00229915253006865  1.51394973381002    0.163206644697938     -0.0740797732215464   0.00386992020361337   -0.256454676015537  -0.047810783883058    0.0169442494704327    -0.000962112565525938 0.3377 0.4800      0.58688   0.64000 0.50000 0.37000 5   7   16   36
0.35                    -4.99194353185349             3.58163762887113          -1.54552621736997   0.615284558023766     -0.130667133995223   0.0129996807904448    -0.000488177019044619 1.77279165894083     -0.717050080026239   0.0889429180313871   -0.00419072260454404  -3.0437775645586    0.0603883161194736   0.0572805775369601    -0.00197680262011821  1.41086857735487    0.177942225860411     -0.0724189202408046   0.00361508871731196   -0.238778013934565  -0.0498607776908692   0.0166154157568662    -0.000916964331281583 0.3482 0.4740      0.58812   0.62793 0.49396 0.37000 5   7   16   36
0.40                    -5.0985176479026              3.69129664856851          -1.68062829728217   0.672804751954135     -0.141646737184185   0.0139828458876481    -0.000521964999170582 1.51827911382411     -0.629102564532826   0.0793758413218718   -0.00387375568359453  -2.84556956901668   0.029491475454992    0.0544252641651964    -0.0015068895007478   1.33576716428933    0.181703240713848     -0.0692142376537209   0.00328687262464097   -0.226886109044282  -0.0498954700405014   0.0160230128506772    -0.000860256131785113 0.3552 0.4687      0.58811   0.61747 0.48874 0.37000 5   7   16   36
0.45                    -5.51846872135268             4.34136786712787          -2.14219151210517   0.826611269423493     -0.167704648382054   0.0161731324426429    -0.000594692407524179 1.15902107947723     -0.481776944680289   0.0592473719365184   -0.00295663577997941  -2.55650702119695   -0.0602623054112216  0.0628438538947199    -0.00171237191082391  1.2249347018735     0.211193560857471     -0.0711770208465336   0.00327873356527528   -0.20997584316314   -0.0540131919262198   0.0162776811604225    -0.000857213327415395 0.3495 0.4641      0.58097   0.60825 0.48413 0.37000 5   7   16   36
0.50                    -5.79500189369318             4.73563621429326          -2.42791108774761   0.922435569635811     -0.183808913046578   0.017501798134528     -0.000637645194744098 1.1875844561392      -0.587711060819976   0.0890220931219639   -0.00510454449808858  -2.81911434648936   0.209123050249793    -1.14711013947692e-05 0.00250115054905443   1.40953552855593    0.0474375384959427    -0.0344658682232184   0.000852995447245808  -0.246366277243755  -0.0231750964554065   0.00948902005999452   -0.000411136811150888 0.3548 0.4600      0.58095   0.60000 0.48000 0.37000 5   7   16   36
0.55                    -5.49018384646958             4.36713251335888          -2.28453911420208   0.889994999975749     -0.178896782107233   0.0170593373349386    -0.000620764521489708 0.766733180534971    -0.38968761726755    0.0578548225574492   -0.00347479423668842  -2.48108890406126   0.0622742548175077   0.0219560409334741    0.00139145833718543   1.28066786329754    0.103103682172159     -0.0428068406625999   0.00127228700447783   -0.227270646257817  -0.0314481395812822   0.0107645761830146    -0.0004761995966876   0.3639 0.4586      0.58544   0.59175 0.47587 0.37413 5   7   16   36
0.60                    -6.86486686877694             6.40374539675197          -3.51807303138578   1.26246471114077      -0.238510941103281   0.0219063544794379    -0.000778516727247488 0.736843015674579    -0.436576817159394   0.0736968926962447   -0.00467613071901061  -2.61161889650394   0.206571508449378    -0.0117880783265457   0.00363206068702543   1.3752924949923     0.0201320186167743    -0.0246297354513705   9.76852732843798e-05  -0.245141928833445  -0.016846898974431    0.00766593882017229   -0.000278556396054143 0.3757 0.4574      0.59187   0.58422 0.47211 0.37789 5   7   16   36
0.65                    -7.14348239992714             6.64087835699209          -3.6506566687056    1.3006525891116       -0.24397481623675    0.022270450478434     -0.000787142098066904 1.52519093365672     -0.919590987091526   0.164332190694729    -0.0100246467719383   -3.84807483678143   0.964295689134757    -0.154818279949248    0.0121443957156372    2.04878678683764    -0.392541988807676    0.0536878925446913    -0.00459820573869577  -0.365861658672247  0.0573719571156885    -0.00647742068381063  0.000574074822542216  0.3831 0.4562      0.59573   0.57729 0.46864 0.38136 5   7   16   36
0.70                    -7.52390462987363             7.26854648440312          -4.06024838242216   1.42448215088748      -0.263046325328198   0.0237321362466038    -0.000831515860217264 1.22682490170462     -0.787442775688431   0.144781061634178    -0.00906114698906555  -3.64906669826984   0.882850906958418    -0.143471479235736    0.0116135894590407    1.98515020813745    -0.366502167964456    0.0501029291998233    -0.00443764727262246  -0.357755547424784  0.0540094271531302    -0.00600204192848836  0.00055323017276381   0.3886 0.4551      0.59846   0.57087 0.46544 0.38456 5   7   16   36
0.75                    -7.90787012382226             7.88485609567565          -4.45307641042689   1.54098411816552      -0.280666608966963   0.0250579507487853    -0.000870995790547959 0.959356754562034    -0.663251518616941   0.125488890395747    -0.00806374308652843  -3.46780866197708   0.797400231455424    -0.129696278683241    0.0108710074025608    1.92461753822973    -0.335135021857539    0.0446946388447311    -0.00413473494223543  -0.349616427034543  0.0494146724266362    -0.00515832067760371  0.000504683296336726  0.3852 0.4542      0.59554   0.56490 0.46245 0.38755 5   7   16   36
0.80                    -8.28113182388701             8.46638059503981          -4.8159007770586    1.64654639931031      -0.296317024106034   0.0262100808621943    -0.000904476650697939 0.730610615703567    -0.552371437096148   0.107578664842056    -0.00710628233707827  -3.31549798935853   0.716599544178357    -0.115434057159477    0.0100501560580742    1.87383947958296    -0.303734770960366    0.038671788677562     -0.00377378438960554  -0.342745126151845  0.0446351178938329    -0.00418680792056682  0.000445316201414629  0.3850 0.4532      0.59468   0.55932 0.45966 0.39034 5   7   16   36
0.85                    -8.61808928503216             8.97621500558033          -5.12814368465179   1.73533795518597      -0.309103785775831   0.0271178977955594    -0.000929702329159017 0.530903262423651    -0.449237587445784   0.0900168782015946   -0.0061266195806213   -3.17813340099571   0.632576444304402    -0.0992221630960013   0.00906204995124766   1.82538290849637    -0.268113264140605    0.0312468887009229    -0.00330615331134244  -0.335811294946851  0.0389197140095357    -0.00294471680202779  0.000366204658157708  0.3811 0.4523      0.59150   0.55407 0.45703 0.39297 5   7   16   36
0.90                    -8.92794097064509             9.42339373627003          -5.39481835272624   1.80902687885344      -0.31933334226606    0.0278096732952168    -0.000947693681698626 0.382734928994583    -0.367317245195976   0.0753329104083108   -0.00527580121142784  -3.08681413937282   0.564993982931201    -0.0849047631552565   0.00814360838089087   1.79590636672959    -0.239062665262591    0.0245685011439217    -0.00286624548176443  -0.331822580251755  0.0342452070663314    -0.00182962527878716  0.000292438812909529  0.3799 0.4515      0.59006   0.54912 0.45456 0.39544 5   7   16   36
0.95                    -9.23422423267119             9.85146591571727          -5.64416511879198   1.87648471409083      -0.328476732862658   0.0284094202909421    -0.000962640116287345 0.253604612338906    -0.289500889045409   0.0606203139484274   -0.00439358436151917  -3.00151090544799   0.492359005656747    -0.0687347270077421   0.00708231329850278   1.76543482802063    -0.205596712603424    0.0166691228473816    -0.0023394806076842   -0.327308894878682  0.028663505716307     -0.000484806520220512 0.000202810180992498  0.3733 0.4507      0.58525   0.54444 0.45222 0.39778 5   7   16   36
1.00                    -9.49986351244937             10.2035139244456          -5.84294716953662   1.92795553860822      -0.334993857553761   0.028792189146421     -0.000970451375334239 0.154100691766031    -0.222003167440382   0.0470059756013201   -0.00354481071431478  -2.93667923409993   0.423765527198296    -0.0524513408592765   0.00598284360002449   1.74170547864393    -0.172727209099351    0.00853193051941362   -0.00178641804374483  -0.323678178891616  0.0231024125534669    0.000904893088852973  0.000108876175138007  0.3716 0.4500      0.58362   0.54000 0.45000 0.40000 5   7   16   36
1.05                    -9.74589450732915             10.5090615612529          -6.00846581130985   1.96869002161435      -0.33974531411955    0.0290300698055477    -0.000973544164289439 0.0957527217236463   -0.172290983045782   0.0359652243230536   -0.00282058620400309  -2.90908808035977   0.370268637071948    -0.0382595548916123   0.00498221491225841   1.7336923434035     -0.146402458646667    0.00134587670336006   -0.00128102851560046  -0.322526515618657  0.0186245822240616    0.00212684252655738   2.38728423291898e-05  0.3691 0.4482      0.58063   0.53956 0.44778 0.40000 5   7   16   36
1.10                    -9.95863377624418             10.7572168147988          -6.13696417718203   1.99793530125602      -0.342639876531187   0.0291170622701221    -0.000971761007003459 0.0414863995576718   -0.120036379556478   0.0239096200537577   -0.00201617893015631  -2.87259256930142   0.30604654060779     -0.0215085103933377   0.00380999057184256   1.71836540698407    -0.113480600462706    -0.00729775330720406  -0.00068107772165324  -0.319882691182269  0.0129281877843849    0.00360875205475542   -7.77538446215981e-05 0.3650 0.4465      0.57674   0.53913 0.44566 0.40000 5   7   16   36
1.15                    -10.1000348474245             10.8935791817363          -6.19769477525699   2.00681240954494      -0.342289175856386   0.0289413298530825    -0.000961465746845629 -0.0033618467009693  -0.0691583837915427  0.0116647008514114   -0.00118490938857119  -2.83888983426881   0.23940479347946     -0.00391684776037577  0.00257597713999464   1.70300572356621    -0.0790564417102299   -0.0163523925840257   -5.32303663220932e-05 -0.31715436728736   0.00699034599290728   0.00515079007622993   -0.000183290361202126 0.3627 0.4449      0.57402   0.53873 0.44364 0.40000 5   7   16   36
1.20                    -10.2369649784008             11.0115644195108          -6.24396659425346   2.01101458614832      -0.341195091085208   0.0287076723827817    -0.000949398419767965 -0.0200638889575388  -0.0305188433861261  0.00120248414739615  -0.000440233229878615 -2.83076426798021   0.183780349416793    0.0120416406597646    0.00142400781368829   1.69873140249394    -0.0497629403398019   -0.0245824745154229   0.000529973171103837  -0.316223416579226  0.00194641983667431   0.00653905830671277   -0.00028005786317202  0.3625 0.4434      0.57267   0.53834 0.44170 0.40000 5   7   16   36
1.25                    -10.3220701618782             11.0510491943836          -6.24323902447304   2.00121707400921      -0.337891272726468   0.0282969566477529    -0.000931657262101359 -0.0399018346940094  0.0118396918059071   -0.0101226856718546  0.00036069362102084   -2.81234198984      0.120764647699886    0.0294510123682269    0.00018542734844602   1.68851726454653    -0.0167344679056101   -0.0335008499932545   0.00115290111860549   -0.31424761713075   -0.0037078481155554   0.00803527152930638   -0.000382939982847197 0.3631 0.4419      0.57195   0.53797 0.43984 0.40000 5   7   16   36
1.30                    -10.4284725298049             11.1053502574564          -6.24691458366462   1.99245369083313      -0.334786547739222   0.0279079681641679    -0.000914834789674199 -0.00771366813535051 0.0281158448394389   -0.017151799200439   0.000929344409204083  -2.84722158540453   0.0848788201674838   0.0423071832094931    -0.000801961897139159 1.70216436971323    0.0037920149980136    -0.0402683780273485   0.00165462419382921   -0.316114422937405  -0.00729795951877591  0.00916878250472298   -0.000465029421358804 0.3610 0.4404      0.56949   0.53761 0.43806 0.40000 5   7   16   36
1.35                    -10.4852001660847             11.0879953249075          -6.20879315331082   1.97141911508874      -0.329762730372023   0.0273662807388621    -0.000893149038717948 0.0137966250292528   0.0514369611053336   -0.0255227436389967  0.00157673741588576   -2.86656035792565   0.0399639418232716   0.0567415782721269    -0.00187604379753094  1.70842419592699    0.0282084044560765    -0.0476671207316542   0.00218896317369984   -0.31681124909941   -0.01145598787398     0.0103878575508838    -0.000551254029964727 0.3605 0.4391      0.56808   0.53727 0.43634 0.40000 5   7   16   36
1.40                    -10.4821735883939             10.9833822198454          -6.11992723050257   1.93555242520422      -0.322425799336723   0.026640666944562     -0.000865598162309335 0.0268671697883593   0.0811965322148886   -0.0352353626000423  0.00230838074619814   -2.87099259395495   -0.0147319677027719  0.0730722157610122    -0.00306440215174347  1.70687029688542    0.0574245965886399    -0.0559689401756622   0.00277702357963987   -0.316176147986892  -0.0164035071940701   0.0117531763410546    -0.000646151698561402 0.3616 0.4377      0.56779   0.53694 0.43469 0.40000 5   7   16   36
1.45                    -10.5301013727802             10.941735478407           -6.06468941786934   1.90952694822742      -0.3167021672299     0.0260526379944531    -0.000842774420285702 0.0863241307704126   0.0848614037952647   -0.0402447201988807  0.00276739792963481   -2.92752632283955   -0.038733616575853   0.0836704729512184    -0.0039113066822355   1.73002552877984    0.0715484143301979    -0.0613860198393701   0.00319057610756205   -0.319673616788347  -0.0187628013380748   0.0126159072249717    -0.000710337933616884 0.3604 0.4365      0.56604   0.53662 0.43309 0.40000 5   7   16   36
1.50                    -10.5311876215732             10.8235666251612          -5.96299852166464   1.86972343062895      -0.308820097998435   0.0252925798037149    -0.000814444905640075 0.17419044478292     0.0734770542534873   -0.0426682782165802  0.00308222593546174   -3.01585419228248   -0.0448734442383345  0.0910561898313306    -0.00457264204461108  1.7684388315813     0.0767242207102687    -0.0651519835538848   0.00350724132188637   -0.325738571231432  -0.0195796073223857   0.0131904304911343    -0.000757513348590218 0.3559 0.4352      0.56223   0.53631 0.43155 0.40000 5   7   16   36
1.55                    -10.5131645935139             10.6949016748526          -5.85989199077318   1.8303399893566       -0.301112664096342   0.0245553121633493    -0.000787154283257759 0.195375979403532    0.0995094756502602   -0.0517347907042114  0.00377494895718706   -3.02057411148093   -0.0978562705691237  0.106771416181438     -0.00570941691613567  1.76613896846624    0.104651984725314     -0.0729692751606945   0.00405593474477167   -0.324997464550661  -0.024194721916239    0.0144420850738555    -0.000843571575002732 0.3507 0.4340      0.55804   0.53601 0.43005 0.40000 5   7   16   36
1.60                    -10.4558769208426             10.5024254800278          -5.71795056996344   1.77935867057804      -0.291569310610593   0.0236697634858606    -0.000755044173664368 0.240615193388661    0.112489486378923    -0.0585111278410957  0.00433730566811485   -3.05391317766829   -0.134199100909333   0.119412751297372     -0.00666432804599734  1.77802449084643    0.123988945883154     -0.0791620802959735   0.00450717295617042   -0.326691687771117  -0.0273025982179903   0.015405445975029     -0.000912235731086413 0.3479 0.4329      0.55537   0.53572 0.42861 0.40000 5   7   16   36
1.65                    -10.4001619394175             10.3092923525987          -5.57531470212418   1.72845213441824      -0.282113443530303   0.0227984977610769    -0.000723641168655326 0.302687977276948    0.115884539922291    -0.063546915236325   0.00479852318979597   -3.10556246677332   -0.159187785668401   0.129872851054439     -0.00748751434033163  1.79854715559364    0.137663539062564     -0.084237359691273    0.0048899723256349    -0.329825634701413  -0.0294355975113065   0.0161736993340619    -0.00096888142966963  0.3486 0.4318      0.55495   0.53544 0.42721 0.40000 5   7   16   36
1.70                    -10.3368670392901             10.1057637165544          -5.42698819012109   1.67612928746518      -0.272490885329221   0.0219189600538523    -0.000692142449513193 0.366698834175691    0.117543014045574    -0.0681809812371341  0.00523252516230109   -3.15895729645536   -0.181556431886473   0.139647917349548     -0.0082619978305718   1.82026264896852    0.149599867251745     -0.0888678193462165   0.00524167857753305   -0.333233029628462  -0.0312080930546841   0.0168521300824617    -0.00101934900826622  0.3485 0.4307      0.55401   0.53517 0.42585 0.40000 5   7   16   36
1.75                    -10.2481418936857             9.8665925487974           -5.25849265415412   1.61811275377925      -0.262016487486019   0.0209746802015459    -0.000658672771765666 0.429662671288272    0.118927623731572    -0.072640972435487   0.00565069147457238   -3.21029669627912   -0.203422395024484   0.149106271369937     -0.00900815993077935  1.84102127278058    0.161111250207243     -0.0932990958415375   0.00557672318921655   -0.336496547478282  -0.0328870694557376   0.0174923561344748    -0.00106674637850974  0.3470 0.4296      0.55226   0.53491 0.42453 0.40000 5   7   16   36
1.80                    -10.1661167165388             9.63741844962564          -5.09610558442555   1.56217165278077      -0.251931576577181   0.0200674398514704    -0.000626588087660865 0.498903525639833    0.115982514576719    -0.0762274461237908  0.00601467767050711   -3.26891872180479   -0.219504143325459   0.157328139653091     -0.00967549962470597  1.86553376455303    0.169492797038044     -0.0970554234683288   0.00586884403663378   -0.340427597570416  -0.0340027449787219   0.0180113841247801    -0.00110649066326803  0.3469 0.4286      0.55137   0.53465 0.42325 0.40000 5   7   16   36
1.85                    -10.0725569393033             9.39723087281102          -4.92899689745382   1.50523049160247      -0.241748006454255   0.0191570722979762    -0.000594560955605728 0.551536651203595    0.121470489782218    -0.081183091750249   0.00645042893536399   -3.3078263378291    -0.244984698986906   0.166990869092196     -0.0104143707319684   1.88089039141885    0.182056875047389     -0.101431536197781    0.00619088252479661   -0.34288129123549   -0.0357763038395907   0.0186261485580726    -0.00115080749782241  0.3448 0.4276      0.54931   0.53440 0.42200 0.40000 5   7   16   36
1.90                    -9.96912558026572             9.13544234980545          -4.74796422442572   1.44427633818459      -0.230977077675986   0.0182037039840514    -0.000561274857255124 0.639301361366003    0.106894875648423    -0.082540916505593   0.00668069673056517   -3.38808262711735   -0.246042361646363   0.1721802843381       -0.0108940203493066   1.91612823952543    0.182644876645653     -0.103590592515091    0.00638357287351835   -0.348663984187211  -0.0355206798076689   0.0188626435649038    -0.0011729455382111   0.3435 0.4266      0.54775   0.53416 0.42079 0.40000 5   7   16   36
1.95                    -9.86707679947                8.88205485992444          -4.57356643349473   1.38567467583765      -0.220638759127391   0.0172903164100142    -0.000529452306733539 0.709574401235324    0.100743224012811    -0.0852075620372429  0.0069768680890041    -3.44806565613044   -0.256335506250771   0.178729212585315     -0.0114385553977511   1.94202935223857    0.187350420301409     -0.106340845635051    0.00660397806090082   -0.352940686329593  -0.0359196126701974   0.0191925416055078    -0.00119948290720049  0.3422 0.4257      0.54621   0.53392 0.41961 0.40000 5   7   16   36
2.00                    -9.74261496562178             8.59860545293479          -4.38267466966616   1.32250140573443      -0.209623417771092   0.0163257326040817    -0.000496073892892529 0.77415807444278     0.0974469182632426   -0.0883503652696932  0.00729934231728307   -3.50033668854162   -0.270175771876376   0.185842692514114     -0.0120138438987978   1.96405944269353    0.193783379189885     -0.109368934870156    0.00684005139323185   -0.356555653082924  -0.0366139515874944   0.0195712600713491    -0.0012288775755408   0.3412 0.4248      0.54482   0.53369 0.41845 0.40000 5   7   16   36
2.50                    -7.73719854415598             4.9849313842394           -2.11303959891367   0.602967300628316     -0.0880361367604938  0.00593617866216222   -0.000143646549842113 0.37787280710646     0.598500769655239    -0.208849563904978   0.0153621401125253    -2.69727681441477   -1.08993298336372    0.372061990976785     -0.0241261340729448   1.52217667844745    0.602692480952722     -0.198993854970705    0.0125509139917249    -0.279075766689959  -0.103537099260945    0.033865157857116     -0.00212601206309804  0.3321 0.4166      0.53278   0.53166 0.40830 0.40000 5   7   16   36
3.00                    -7.05707982954326             3.29990829668506          -0.981584593603232  0.237866066777406     -0.0266868533877141  0.000786409446984046  2.68966333883864e-05  1.11483325866446     0.3061067519659      -0.168518778721164   0.0134345581517456    -3.49133653822368   -0.766241447789932   0.323478715212636     -0.0215193609961113   1.92758463084306    0.403629332272474     -0.163690121195826    0.0104270456822193    -0.352318819575228  -0.0626090583090034   0.0260256564706806    -0.00163604898099529  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
3.50                    -5.83101436252432             1.06172239143741          0.399822267324979   -0.186133461103695    0.0421466185841653   -0.00484698829200588  0.000209815492713065  1.22449076484502     0.344636959538116    -0.186101509558349   0.0148308828827666    -3.48427031684283   -0.872599072136962   0.35266648239147      -0.0235399304529161   1.91837372656436    0.435817616547965     -0.171819977621711    0.0109693423161649    -0.351785814214733  -0.06423090625883     0.0263904847244192    -0.00165921239343139  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
4.00                    -5.89088708619789             0.463779393960284         0.875833097457137   -0.34024925159362     0.0666718948451634   -0.00674067583895381  0.000266413432104583  2.97707303742802     -0.600018938364775   -0.0215347916141859  0.00555867106005148   -5.65210147221667   0.355760722480262    0.130106628931189     -0.0106106015064692   3.01861021841114    -0.21956051163024     -0.0492796944750036   0.00370318316670891   -0.543829751313916  0.0541012681728631    0.00385111294458164   -0.000307912001080335 0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
5.00                    -5.4825537290669              -0.744036463168919        1.64952179450124    -0.568988713543616    0.100743428092264    -0.00919263056381447  0.000333409465097575  4.02319379937909     -1.21259376101342    0.0913600206780031   -0.00105871873435923  -6.91011796523039   1.16387556292447     -0.0266608697706417   -0.00113685118489094  3.6697687258066     -0.673950750625199    0.041863984684184     -0.00188299792280918  -0.659478274671376  0.138764238788859     -0.013389081377356    0.00075366559643466   0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
6.00                    -4.67805980702777             -1.56781396625844         1.97320638748031    -0.637060940258363    0.107536343785729    -0.00937412839510184  0.000325687956617877  1.70775156634404     -0.0330714607884236  -0.104393466247931   0.00953764851507122   -4.00616403867702   -0.310769044321153   0.220252257473229     -0.0147061551285489   2.24980603204056    0.0371344400804896    -0.0774883485101018   0.00475233120438498   -0.419773375418587  0.0195728934183342    0.00672624046811781   -0.000377701548391354 0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
7.00                    -5.16218960494071             -0.948562649497839        1.56353269674354    -0.497243270613634    0.0814579527823799   -0.00687955675762913  0.000230993963153441  1.57016628203885     -0.0521006766687665  -0.0905744136925167  0.00837129234524203   -3.82259874898189   -0.267456684771477   0.198414916614194     -0.0130223528845607   2.1674444803942     -0.000850055394187969 -0.0635148762938281   0.00377589628322477   -0.406867315608547  0.0276494655373225    0.00409195685820659   -0.00020363215752606  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
8.00                    -5.98536991284128             0.23075492450654          0.81627427629354    -0.256018593253181    0.0392576350800975   -0.00309110725256368  9.50507077009833e-05  1.36865888172568     -0.0321731675617039  -0.084335561452936   0.00767209803241737   -3.53821884330828   -0.298200396624397   0.192378418892592     -0.0123616529581177   2.01915586775386    0.0142324443090848    -0.0611596092179959   0.00355655494888518   -0.380505431429783  0.0244940206130155    0.00392959132396138   -0.000190373279017243 0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
10.0                    -6.43166037718965             0.781715417336081         0.419067880454194   -0.111703048232771    0.0111834875674998   -0.000347547879850276 -9.58880388830187e-06 1.39969757465226     -0.244919016206169   -0.0251821119076448  0.00351030447501622   -3.63841700023342   0.0319436321583016   0.106164012425708     -0.00652157603960815  2.09161717274124    -0.169762357847494    -0.0153647877577974   0.000563014058030854  -0.396033090352291  0.0577917572882087    -0.00409638232458104  0.000322671994810415  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
""")

COEFFS_FORELAND_50Bars  = CoeffsTable(sa_damping=5, table="""\
IMT                     a1                            a2                        a3                  a4                    a5                   a6                    a7                    a8                   a9                   a10                  a11                   a12                 a13                  a14                   a15                   a16                 a17                   a18                   a19                   a20                 a21                   a22                   a23                   tau    mean_phi_ss sigma_tot phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv                     -7.07692104560858             5.53453557159042          -2.85638253778918   1.00455096336982      -0.189766212430622   0.0175339379429905    -0.000626946378448282 1.81360194718373     -0.789495658576955   0.10664113201381     -0.00537702655581446  -3.40075366709673   0.0695084444372301   0.0787711312860849    -0.00400441422151852  1.63540808826056    0.155336375482169     -0.0839712344789103   0.00486580061157626   -0.291743467951392  -0.0391738553207627   0.0182003147413057    -0.00113992749901153  0.5010 0.4600      0.68015   0.00000 0.00000 0.00000 5   7   16   36
pga                     -4.04537660224925             4.54792449211361          -2.52312602512793   0.947393883303524     -0.186633142737737   0.0177041231939612    -0.000645125482634016 0.567936855195552    -0.159096356116191   0.00189062729435474  0.000246448178818437  -1.67722490671008   -0.895375073520955   0.255212398467628     -0.0144070079256093   0.715702210915304   0.682852678952893     -0.185408762439109    0.0111290504165921    -0.142521524189518  -0.129561201230503    0.0361572032396897    -0.00228425426985864  0.3532 0.4600      0.57998   0.58000 0.47000 0.35000 5   7   16   36
0.01                    -4.33129785720388             4.90730267582611          -2.71118914472022   1.00149016177602      -0.195318394943918   0.0184259980704779    -0.000669228385949504 1.15323979809213     -0.507528571770804   0.0673111371051944   -0.00367009353555145  -2.43765935600041   -0.429688018238893   0.164979880183164     -0.00886274315523457  1.08990004397154    0.445894984892876     -0.138638952003058    0.00822009691799743   -0.205937834474777  -0.0883910754473618   0.0279396332818193    -0.00176974999415353  0.3529 0.4600      0.57975   0.58000 0.47000 0.35000 5   7   16   36
0.02                    -3.929015468659               4.5812092095625           -2.55942670675771   0.962816967078341     -0.189636103742108   0.0179804638488938    -0.000655141599622459 1.14586095542526     -0.286918953493489   0.00301111268208939  0.000951105727287412  -1.9950621947376    -0.946489805845478   0.290348333514757     -0.0174612593429473   0.628005110033105   0.80803873863489      -0.22021606881651     0.0137301852823831    -0.0965435354310332 -0.162913167459021    0.04421565176498      -0.00286384473060578  0.3674 0.4570      0.58639   0.56796 0.46097 0.37408 5   7   16   36
0.03                    -3.79576038754562             5.15446826921304          -2.97055465422299   1.09005263705476      -0.210549615312847   0.0197562018668775    -0.000716007481419421 -0.238421451356831   -0.084288168682659   0.0305841390835215   -0.00306598420797397  -0.315910333360088  -1.07820459749773    0.233684795655916     -0.0114266825130683   -0.136664242877729  0.801741640401787     -0.180375536919727    0.0101798355496652    0.013274417879303   -0.148444438015574    0.0352924653446786    -0.00215101710528786  0.3749 0.4552      0.58975   0.56092 0.45569 0.38817 5   7   16   36
0.04                    -3.16586851254491             4.67535694574959          -2.71977809646547   1.01475051554163      -0.198443716212214   0.0187780115666877    -0.000684479672715533 -0.942776444940338   -0.101535646087869   0.0763701886232662   -0.00722682473490678  0.0643329476126501  -0.784107405308255   0.134357717350496     -0.00411289513844539  -0.0441400938496711 0.527226159823204     -0.111955362291936    0.00555752453369718   -0.0475076823112305 -0.0836524713817856   0.0210278562497944    -0.00122736777827073  0.3801 0.4540      0.59211   0.55592 0.45194 0.39816 5   7   16   36
0.05                    -3.44275884917096             4.89496525788337          -2.79670549168367   1.03168433842938      -0.200741685173797   0.0189502981497045    -0.000689861496519861 -0.110434144867079   -0.471620175119103   0.129634924567262    -0.0097351488066636   -1.30267264415401   -0.174858658748331   0.0435803541341531    0.000470596868170658  0.818545595283387   0.16301463274311      -0.0577216870252026   0.00274048491972502   -0.219393543400959  -0.0156109888052529   0.0110357424780664    -0.000704367397276833 0.3855 0.4530      0.59483   0.55204 0.44903 0.40592 5   7   16   36
0.1                     -5.03742150457096             5.59244249320109          -2.92445588344025   1.04161877932927      -0.199988993021078   0.0187487746633633    -0.000679641005966615 2.34945203871884     -0.806973928279071   0.0780262685281908   -0.00260324327687869  -3.55942081553392   -0.231098782611624   0.175695624141248     -0.0112432434111646   1.65289628265373    0.401469093720247     -0.154549494265484    0.00994088772563011   -0.2997066563727    -0.0950197959599274   0.033220504947543     -0.00219599947759583  0.3864 0.4500      0.59312   0.54000 0.44000 0.43000 5   7   16   36
0.15                    -5.45039432928302             5.57831961324232          -2.82253547918325   1.00307488126673      -0.193379415459503   0.0181941444314495    -0.000661262063110059 2.2642446086671      -0.74665604458733    0.0680766456724801   -0.00207862495313837  -3.37922600595439   -0.23957115018774    0.160723455378262     -0.00967468653669371  1.53104080982961    0.389200768877299     -0.138534655675032    0.0084399720152028    -0.266110176696103  -0.0910391182111169   0.0292075168269655    -0.00183189075606569  0.3841 0.4675      0.60507   0.58095 0.47510 0.40075 5   7   16   36
0.2                     -6.06865299279911             5.52714096845149          -2.63283241811959   0.937448944613042     -0.183348793195218   0.0174431042504235    -0.000638433620346316 4.29898862352523     -1.95448246739811    0.290462109887979    -0.0150293434936831   -5.89832124006457   1.35230611121364     -0.143427964993093    0.00850644953877613   2.76507481977756    -0.418335899173056    0.0188082739034156    -0.00109585045523422  -0.470441286969558  0.0487742585754864    0.00157309831922625   -0.000141707076300871 0.3690 0.4800      0.60546   0.61000 0.50000 0.38000 5   7   16   36
0.25                    -4.80778491708833             3.84103038117896          -1.7681139842944    0.70112398438631      -0.147168142105362   0.0145378937107334    -0.000543627864073168 2.07437433135515     -0.760482130291877   0.0833252462938688   -0.00339300022139929  -3.11417347486767   -0.132679849599163   0.115566145803866     -0.00616741915791912  1.37402970629308    0.327392661971259     -0.112540799125661    0.00641617372518889   -0.22854795997288   -0.0802887533066189   0.0244923794565756    -0.00146257529706872  0.3445 0.4800      0.59082   0.62651 0.50000 0.37450 5   7   16   36
0.3                     -5.35215287757865             4.47744073901096          -2.18483406877739   0.839751563551757     -0.171051196233187   0.016585803984351     -0.000612935499576337 1.79423456138276     -0.661925494689424   0.0722888647993326   -0.00300815137915469  -2.82407783362113   -0.191527441631266   0.115200172185733     -0.00574156159130255  1.24090708674098    0.346600325910232     -0.110323316571018    0.00607401208960832   -0.204877947993228  -0.0829318595836682   0.023998642984773     -0.00139696572930031  0.3377 0.4800      0.58688   0.64000 0.50000 0.37000 5   7   16   36
0.35                    -5.64411963495102             4.74667890488446          -2.37902114294528   0.908815859998103     -0.183199298250136   0.0176197559143143    -0.000647080332806681 1.83397993907059     -0.762442563051826   0.0994582393731292   -0.00493301950794444  -3.03009121002071   0.0547693042833134   0.0560057634492846    -0.00172881750423306  1.38619949225407    0.19455579545511      -0.0748480736843195   0.00368789204898731   -0.233298618254309  -0.0537872698490851   0.0172990077887946    -0.000947768342273749 0.3482 0.4740      0.58812   0.62793 0.49396 0.37000 5   7   16   36
0.4                     -6.30192734848287             5.71729681547816          -3.02099423002918   1.1135353770378       -0.216900412367069   0.0203965882271151    -0.000737984798561643 1.24346749992289     -0.457569098646538   0.0473692318270585   -0.00201066323257468  -2.35888394188666   -0.288046986228968   0.115547341403566     -0.0051512569988218   1.05807239264794    0.367122597639997     -0.105767187138378    0.00550791171630795   -0.175829506118006  -0.0844559531091217   0.0229399786836759    -0.00128571362430565  0.3552 0.4687      0.58811   0.61747 0.48874 0.37000 5   7   16   36
0.45                    -6.80905684788946             6.43147442157787          -3.48887061097789   1.26089406461486      -0.24072406779131    0.0223148156347551    -0.000799099013250877 1.10759196661593     -0.452995304482968   0.0550169940972883   -0.00278640320787224  -2.39637144511911   -0.163679520248918   0.0812929366774717    -0.00269980999903821  1.12923836403836    0.275291428734765     -0.0831669779702803   0.00394920978955816   -0.19278161632126   -0.0657590521863379   0.018528403474415     -0.000985525801983303 0.3495 0.4641      0.58097   0.60825 0.48413 0.37000 5   7   16   36
0.5                     -7.56708971558161             7.20964743024481          -3.90873859173151   1.38292330183751      -0.259732454868458   0.023805718098589     -0.000845260127163041 2.37541209467368     -1.24358788807258    0.20518303733786     -0.01173159395664     -4.20331620814395   0.977165090023884    -0.137708894090556    0.0104763827395459    2.07781240204044    -0.325729998519654    0.0328802408839264    -0.0030799548396765   -0.359236506934742  0.0402565869168079    -0.00201743481118206  0.00026465944959567   0.3548 0.4600      0.58095   0.60000 0.48000 0.37000 5   7   16   36
0.55                    -7.97930883897475             7.87055578845981          -4.34522370479101   1.51636816190398      -0.280454552971377   0.0254034563903743    -0.000893979715333687 1.99769745160434     -1.08854590677194    0.184051836733162    -0.0107753502481621   -3.92466820259808   0.885929245113231    -0.12849130654718     0.0102148496941316    1.98185827684032    -0.29954952293841     0.0312233399025309    -0.00310332265673441  -0.346230145076708  0.0372859027804604    -0.00193480920757857  0.000278338506007972  0.3639 0.4586      0.58544   0.59175 0.47587 0.37413 5   7   16   36
0.6                     -8.34745623242581             8.47631655053791          -4.74290057000642   1.6357693177138       -0.29856502770764    0.0267625098408471    -0.000934192181766675 1.604199095762       -0.918557364158763   0.159534967519677    -0.00959678314997007  -3.62504515725032   0.770810393103703    -0.113639982941941    0.00957735671342835   1.87494720643706    -0.260587099843878    0.0265563615584799    -0.00292628190621118  -0.33139180048504   0.0320898442287352    -0.00133201378544407  0.000257575789716962  0.3757 0.4574      0.59187   0.58422 0.47211 0.37789 5   7   16   36
0.65                    -8.75839795739087             9.13191756939183          -5.15898016342101   1.75807597998265      -0.316840398443597   0.028117613151013     -0.000973866499080002 1.25816611965066     -0.766916378626255   0.137272734242784    -0.00850503977961973  -3.36923546864866   0.666297926961706    -0.0990128482031688   0.00888416614793525   1.78512906361753    -0.223845980864156    0.021415045558499     -0.00268595962075723  -0.319008506323281  0.0269654162732952    -0.000592180842595093 0.000222651786927107  0.3831 0.4562      0.59573   0.57729 0.46864 0.38136 5   7   16   36
0.7                     -9.05752267665599             9.61519647241148          -5.46614821822175   1.84592258740446      -0.329348830612732   0.0289847240530126    -0.000997095583821459 0.920755903179767    -0.611345408575504   0.113261289401128    -0.00727099965958466  -3.10845573535188   0.545280665788353    -0.0798578174278571   0.00786938033516457   1.68884574708675    -0.176517547787899    0.0135992125139647    -0.00226191414961451  -0.305190329255299  0.0198477458514961    0.000631585723273669  0.000155094265710587  0.3886 0.4551      0.59846   0.57087 0.46544 0.38456 5   7   16   36
0.75                    -9.36677836906082             10.0942756650936          -5.76099977276039   1.92816102055272      -0.34076634273895    0.0297527914843642    -0.00101687028462934  0.634874260344991    -0.476558908452868   0.0920149497205861   -0.00615845279491584  -2.89533099476721   0.439572914435365    -0.0621842088065537   0.00689297885449726   1.61250748419161    -0.135011499147803    0.00626139228350996   -0.00184509160627324  -0.294505346718839  0.0136270561603039    0.00177920183366749   8.89719782478146e-05  0.3852 0.4542      0.59554   0.56490 0.46245 0.38755 5   7   16   36
0.8                     -9.6867671347057              10.5768739601185          -6.05099714866898   2.00763731515338      -0.351627854730426   0.0304714833442554    -0.00103501277054008  0.358648410375198    -0.337294437878594   0.0687680535169971   -0.00488200128083159  -2.6682979311551    0.311402272482576    -0.0387169872251973   0.00551035525410293   1.52250767528677    -0.0783860001534654   -0.00469321016580678  -0.00118200245629658  -0.280821937934532  0.00444062200363444   0.00361406379899195   -2.33734992613794e-05 0.3850 0.4532      0.59468   0.55932 0.45966 0.39034 5   7   16   36
0.85                    -9.89172934675343             10.8662836292503          -6.21909290692388   2.04942194240068      -0.356339838586854   0.030681273474575     -0.00103631093412495  0.146009839453362    -0.225386900273327   0.0494499391439347   -0.00379416620500839  -2.50553068629651   0.209486775472561    -0.0189322401690125   0.00430525700159192   1.461562219754      -0.03398548152915     -0.0138322518682096   -0.000613070264228642 -0.271935011741096  -0.00263713046645337  0.00511498901456681   -0.000117270597092978 0.3811 0.4523      0.59150   0.55407 0.45703 0.39297 5   7   16   36
0.9                     -10.1093921682817             11.1497966785106          -6.37501312613971   2.08634642699547      -0.36020179479357    0.0308209153329088    -0.00103536958282368  -0.00556365375462292 -0.14042636144007    0.0340817975407157   -0.00289827653721081  -2.39520585829692   0.127509023010188    -0.00162287741971158  0.00319918990677804   1.42079809304369    0.00346011466963579   -0.0222262565904333   -6.69057792999991e-05 -0.265938174258916  -0.00879007606861844  0.00652898616211159   -0.000209406918167123 0.3799 0.4515      0.59006   0.54912 0.45456 0.39544 5   7   16   36
0.95                    -10.268440844437              11.3368566792387          -6.47068996267761   2.10490253744985      -0.361121885511587   0.0307224928222341    -0.00102674567336456  -0.144208824625293   -0.0546275467821441  0.01761220195247     -0.00190168944851007  -2.28612279075041   0.0364239616342387   0.0184240847803069    0.00189366908060363   1.37781466755011    0.0464981509098258    -0.0321028879657466   0.000582547554480535  -0.259365638691326  -0.0159360971533819   0.00819278805503678   -0.00031857987872196  0.3733 0.4507      0.58525   0.54444 0.45222 0.39778 5   7   16   36
1                       -10.4269365342967             11.5122929544077          -6.55550965066283   2.1197490326566       -0.36144423676334    0.0305780737261388    -0.00101676165005004  -0.26924019169941    0.0299742094454503   0.000630778348195605 -0.000848205662689818 -2.17625810743144   -0.0630002514411933  0.040815399106727     0.000422410485331815  1.330422862646      0.0955679666002615    -0.0434370954752976   0.00132947482852465   -0.251674136290978  -0.0242714417030785   0.0101280508225986    -0.000445506134704967 0.3716 0.4500      0.58362   0.54000 0.45000 0.40000 5   7   16   36
1.05                    -10.5081951444904             11.5483815556689          -6.55168854601308   2.10759867353027      -0.357463725798145   0.0300864559245307    -0.000995560471592616 -0.305841241931928   0.0680697468547556   -0.00837644642264592 -0.000240075883820284 -2.16650770727509   -0.10807766569282    0.0535739034459915    -0.000494482961692376 1.33129024107495    0.117449344707444     -0.0498038851539334   0.00178377281006219   -0.25219783380615   -0.0278456377711423   0.0111739642767367    -0.000519202209746055 0.3691 0.4482      0.58063   0.53956 0.44778 0.40000 5   7   16   36
1.1                     -10.6152121639921             11.6262846831677          -6.57285039760574   2.10307967630933      -0.354757305323607   0.029705242269293     -0.000978234509783025 -0.384780261073052   0.137190750261096    -0.0237732911660574  0.000767722434939828  -2.08792566631466   -0.201379166123601   0.0761003209437633    -0.00201823562345887  1.29438520076281    0.165249301225699     -0.0613910284220807   0.00256224940748961   -0.24593205138324   -0.0360317562221882   0.0131466915922095    -0.00065050228910034  0.3650 0.4465      0.57674   0.53913 0.44566 0.40000 5   7   16   36
1.15                    -10.5973341523723             11.5184322024427          -6.4849103082763    2.06631205271642      -0.346954184660507   0.0289126681225349    -0.000947579804685161 -0.465753027943387   0.210923870169889    -0.0403583102198379  0.00185653816130101   -2.00083075971361   -0.302577934451956   0.100305639055911     -0.00364599536507236  1.25272009487159    0.216697933367233     -0.0736918102299192   0.00338285573202512   -0.238869078927548  -0.0447658434004461   0.0152196089034094    -0.000787490725051071 0.3627 0.4449      0.57402   0.53873 0.44364 0.40000 5   7   16   36
1.2                     -10.638270905742              11.4779158103732          -6.43107113020769   2.03927369740099      -0.34073399896074    0.0282551320396997    -0.000921576732361709 -0.479136541077997   0.248861371647481    -0.0507623549001663  0.0025978269918231    -1.98917831926487   -0.362380134790737   0.117118191059591     -0.00484537084423903  1.24698750532815    0.247703934942249     -0.0822392322011831   0.00398122605580744   -0.237848391171036  -0.0499712751213087   0.0166319540607151    -0.000884804154304927 0.3625 0.4434      0.57267   0.53834 0.44170 0.40000 5   7   16   36
1.25                    -10.6073065728747             11.328543987397           -6.31330965890497   1.99357054447322      -0.331619499835797   0.0273692874049677    -0.000888360741792505 -0.491551488346704   0.289991413944439    -0.0621669781905567  0.00341340534677159   -1.96842189908202   -0.431303701129218   0.136016464101457     -0.00618250973783296  1.23448207548978    0.284199947629767     -0.091968973118216    0.00465600558012172   -0.235463458847873  -0.0561994714321983   0.0182592340027228    -0.000995926365058712 0.3631 0.4419      0.57195   0.53797 0.43984 0.40000 5   7   16   36
1.3                     -10.5851365050851             11.1830635694642          -6.19575392562962   1.94801439181721      -0.322597286663719   0.0264982983816367    -0.000855879614893014 -0.467921624314853   0.312187893628384    -0.0703201940635917  0.00404683121402033   -1.98691526207601   -0.478423379780618   0.150991819907835     -0.00729145718020662  1.24062697644997    0.309791253387977     -0.0996704973966644   0.00521015805862176   -0.236231052592027  -0.0605195744714621   0.0195239812425068    -0.00108520026352189  0.3610 0.4404      0.56949   0.53761 0.43806 0.40000 5   7   16   36
1.35                    -10.5302156295949             10.9897609392292          -6.0506152650109    1.89459390998917      -0.312392831689842   0.0255374435326607    -0.000820675995253299 -0.447333588821089   0.337775196693386    -0.0792373652762771  0.00472894871973454   -1.99806873425352   -0.530676922769689   0.166932808666612     -0.00845543602558545  1.24289459435442    0.337649793489692     -0.107754311798191    0.00578439342609586   -0.236354232237759  -0.0651727596135063   0.0208394009179299    -0.00117691411088635  0.3605 0.4391      0.56808   0.53727 0.43634 0.40000 5   7   16   36
1.4                     -10.4613017120537             10.7722793462702          -5.89077706530239   1.83706890549614      -0.301609670816631   0.0245365910672574    -0.000784400861827921 -0.411938502821214   0.355818217534956    -0.0868797927034352  0.00534021003453583   -2.02280209664312   -0.575256312046881   0.181463207373193     -0.00953586746610174  1.25115866488003    0.36175954890204      -0.11511154738458     0.00631442307729116   -0.237457347947625  -0.0691741771986184   0.0220250643271056    -0.00126065998289786  0.3616 0.4377      0.56779   0.53694 0.43469 0.40000 5   7   16   36
1.45                    -10.3243488185471             10.4605079590864          -5.67795758944976   1.76438846935846      -0.288496661103945   0.0233524229963407    -0.000742330469365145 -0.395656847725721   0.385901375624198    -0.0967933591144392  0.00608507812661606   -2.02304027753177   -0.634165569517615   0.19856159471484      -0.0107614255809717   1.24777102961047    0.3923319251217       -0.123585626247361    0.00690580817563003   -0.236667975375916  -0.0741853149231549   0.0233801086071843    -0.00135348521663832  0.3604 0.4365      0.56604   0.53662 0.43309 0.40000 5   7   16   36
1.5                     -10.289724954978              10.2792139223257          -5.5367089986428    1.71282446281066      -0.278827475197256   0.0224586117928259    -0.000710104811088829 -0.293882858857335   0.367756282393651    -0.0980642905572574  0.00633358638781152   -2.12283824791259   -0.635375254136526   0.205117773055284     -0.0113727863278387   1.2919236152429     0.394833775421224     -0.126872010327428    0.00719268557105589   -0.24379536356707   -0.0744660304759998   0.0238548270254161    -0.00139441888237278  0.3559 0.4352      0.56223   0.53631 0.43155 0.40000 5   7   16   36
1.55                    -10.1417497521387             9.9546907760946           -5.31784490108241   1.63908817051318      -0.265696438706023   0.0212862006743122    -0.000668849609226952 -0.279295216786027   0.397754077848377    -0.10778453798421    0.00705846773828365   -2.11753534271269   -0.69435563354845    0.22179496576978      -0.0125548589138384   1.28580498542592    0.424894619348215     -0.134990446732398    0.00775283216978809   -0.242599259389947  -0.0793065608190685   0.0251303092563794    -0.0014807881241231   0.3507 0.4340      0.55804   0.53601 0.43005 0.40000 5   7   16   36
1.6                     -10.0183523577858             9.65624490079733          -5.11242615843259   1.56953133277637      -0.253313530725372   0.0201829212176209    -0.000630127723744209 -0.219213122027415   0.403177671549731    -0.113236592374905   0.00754389064772161   -2.16455201821332   -0.724054021983178   0.233251704350515     -0.0134382971536625   1.30465361792394    0.440643292499024     -0.140519551554547    0.00816344448012511   -0.245571018726564  -0.081722675779682    0.0259632834698386    -0.00154146195398278  0.3479 0.4329      0.55537   0.53572 0.42861 0.40000 5   7   16   36
1.65                    -9.89574395668602             9.35614168014446          -4.90558933983889   1.49977806116429      -0.240962509859162   0.0190880119367496    -0.000591861402402895 -0.137245139846497   0.394620667066045    -0.115960305530192   0.00786246634372953   -2.23828132425611   -0.735500265671677   0.241003299772341     -0.014089159885528    1.33671603039552    0.447046558486709     -0.144122421122427    0.00845218201516863   -0.250790173860046  -0.0825198859253369   0.0264601219850494    -0.00158080466768257  0.3486 0.4318      0.55495   0.53544 0.42721 0.40000 5   7   16   36
1.7                     -9.7390162659992              9.01948454506651          -4.68156754178914   1.42572945683694      -0.228026341884592   0.0179525768432651    -0.000552492900412286 -0.101518719157665   0.4126910860018      -0.123506548895264   0.00845964123219868   -2.25599741607228   -0.778712491714127   0.254465796736696     -0.0150689080856176   1.34210800530869    0.468437670697921     -0.150414863454675    0.00889606152963997   -0.251622607610931  -0.0857706643963686   0.0273973315176694    -0.0016456163976706   0.3485 0.4307      0.55401   0.53517 0.42585 0.40000 5   7   16   36
1.75                    -9.61989949537707             8.72745690025784          -4.48121492111625   1.35870755829723      -0.216263462096657   0.0169186137622444    -0.000516638353219805 -0.0175114501637232  0.403008218782381    -0.126031161377211   0.00876716673183524   -2.3284477075234    -0.78977402594787    0.26204310158225      -0.0157071827496923   1.37291173452335    0.474743802400215     -0.153939276711482    0.00917956428707664   -0.256579847759512  -0.0865629288730774   0.0278838509410909    -0.00168440774590077  0.3470 0.4296      0.55226   0.53491 0.42453 0.40000 5   7   16   36
1.8                     -9.45968991593349             8.37980556775855          -4.24992343736656   1.28292324129017      -0.203167339135742   0.0157807189023837    -0.000477521828442466 0.059036345979013    0.395752935894846    -0.128752448928887   0.00907535198729539   -2.39459204143321   -0.801154059157472   0.269241362176003     -0.0163041126072295   1.4019647260692     0.480132326629376     -0.157044657319475    0.00942759399722036   -0.261430563368709  -0.087058390448071    0.0282693397718618    -0.00171524537882604  0.3469 0.4286      0.55137   0.53465 0.42325 0.40000 5   7   16   36
1.85                    -9.32295394908681             8.06616669095919          -4.03879786843135   1.21343811352967      -0.191144111855813   0.014736394139295     -0.000441664947213909 0.149166692906703    0.379941539690587    -0.129820270693947   0.00928317367399208   -2.47745150254353   -0.801265725567496   0.274187632694583     -0.0167618950801199   1.43920303888831    0.479843288750098     -0.159008205629135    0.00960519991674254   -0.267652418014518  -0.0865913260953089   0.0284614346180359    -0.00173422725359916  0.3448 0.4276      0.54931   0.53440 0.42200 0.40000 5   7   16   36
1.9                     -9.17179016767725             7.73623915716045          -3.81982281319171   1.14206891263936      -0.178889039633139   0.013678483158276     -0.000405526185122618 0.228205233032974    0.369631586876923    -0.131789219758691   0.00953974440141284   -2.54623871992442   -0.808004982824782   0.280189039040507     -0.0172762184413821   1.46956890888065    0.482737362236988     -0.161482191296019    0.00981078473559349   -0.272718153133702  -0.0866614923704362   0.0287413399685506    -0.00175815565141008  0.3435 0.4266      0.54775   0.53416 0.42079 0.40000 5   7   16   36
1.95                    -9.01321686238452             7.40002295551972          -3.59863423803624   1.07042132104382      -0.166647097083579   0.0126260619030655    -0.00036970070253227  0.297418478700577    0.363827427455038    -0.13443658500887    0.0098298471548197    -2.60335199854797   -0.819619086889047   0.286861080776263     -0.0178208771018727   1.4944732068996     0.48779938273381      -0.164242960080009    0.0100289689411941    -0.276895577449426  -0.08707536080827     0.0290659619711136    -0.0017840372536953   0.3422 0.4257      0.54621   0.53392 0.41961 0.40000 5   7   16   36
2                       -8.85985327757844             7.07139053841041          -3.38232019548061   1.0005391730865       -0.154751052226412   0.0116073201213195    -0.000335149515034632 0.379284176313856    0.349717075384934    -0.135448275381779   0.0100198887109414    -2.67666807249312   -0.819941511483856   0.291243852052256     -0.0182231960817252   1.52752152896855    0.487059914686142     -0.165821534031428    0.0101736802365387    -0.28246857868178   -0.0864878235191427   0.0291866771090303    -0.00179729080236882  0.3412 0.4248      0.54482   0.53369 0.41845 0.40000 5   7   16   36
2.5                     -7.05961561539708             3.69574140531695          -1.25507615218126   0.331774914205199     -0.0432749448048768  0.00222966702485759   -2.21123623220274e-05 0.385773704211287    0.610053734769837    -0.210812343599662   0.0153967151237219    -2.36802932006476   -1.33873181673086    0.420154618743315     -0.0268825930854354   1.33239531069933    0.744233727308208     -0.226444858531786    0.0141358873100126    -0.247289190331369  -0.127251651876551    0.0384765947399036    -0.00239304396797002  0.3321 0.4166      0.53278   0.53166 0.40830 0.40000 5   7   16   36
3                       -6.27482981179261             1.95689345114742          -0.137426352604704  -0.0172745727650173   0.0136703138970732   -0.00241327070143146  0.000127135946512074  0.79251323685838     0.52033176095423     -0.209058550714479   0.0157902108463318    -2.70194317570535   -1.30571385819058    0.428390726386543     -0.0277756854293226   1.49197469706148    0.703843470048549     -0.222837782087008    0.014002177849369     -0.276061650739905  -0.11546013351436     0.0365420251723034    -0.00227837857358265  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
3.5                     -5.52811237826864             0.443315148636669         0.798996765034952   -0.300389888338542    0.058422680720992    -0.00594597204866196  0.000236911767384378  0.98854720247722     0.504628370860628    -0.216502679501992   0.0165976938596572    -2.77876303512906   -1.36157063107164    0.448808910814347     -0.0293317446638242   1.51311285669874    0.71971038707325      -0.228715647900419    0.0144660271934382    -0.279025863300495  -0.115601907353786    0.0368331519262641    -0.00231042305846745  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
4                       -5.20401298231564             -0.351811601733439        1.29268791088154    -0.445367029164505    0.0800939548004394   -0.00751993243209109  0.000280515389819392  1.38940589864236     0.279884763705666    -0.175297580029304   0.014157104660722     -3.2577497168558    -1.05559984895836    0.388065165969816     -0.0255535600045145   1.77288189553871    0.532035549839226     -0.18968680659151     0.0119974481027899    -0.326989525585267  -0.0785432091561378   0.0289876636700703    -0.00181244851834683  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
5                       -5.38204035060699             -0.903487390367453        1.71377036064386    -0.565936651518239    0.0955998348695961   -0.00834092000431372  0.000290750651768098  3.59477882058341     -0.997893173040457   0.0575544888961179   0.000643409395422057  -6.01241877409122   0.630919065952545    0.0701281737771576    -0.00668485435087707  3.17790676695851    -0.369946593283064    -0.0157913841963699   0.00156054170214816   -0.572699644907836  0.0839531680361895    -0.00272019467710135  0.000100695046121841  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
6                       -5.36682201546016             -0.649513835387427        1.41452719495678    -0.445776301196695    0.071571610504577    -0.00594377949138499  0.000196826109101049  1.72931942034956     -0.0520291859429754  -0.0995864843794736  0.00918565192415031   -3.62239706981821   -0.580424939052927   0.274582106613961     -0.0180646430925865   1.98885298783235    0.226214625309543     -0.117417802698182    0.00733250127016922   -0.369156069479609  -0.0177618434654216   0.0148206048991321    -0.000913060979859072 0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
7                       -6.0800485577606              0.299808588494077         0.81108980125611    -0.247097694649689    0.0358843040423474   -0.00265629074493398  7.629484000109e-05    1.76088888177068     -0.190094944087353   -0.0616536509603404  0.00652087647620163   -3.66325071049536   -0.379713293137711   0.220943615309948     -0.0144063502386577   2.01609525289336    0.111866311339595     -0.0881420787936657   0.00541202959339174   -0.374580638310607  0.00303596905318986   0.00964869601150369   -0.000582921683758873 0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
8                       -6.29618436773334             0.593576235293328         0.579279030255573   -0.159949119182875    0.0187078666431546   -0.000966748084588227 1.15109217475502e-05  1.52680097661609     -0.141570970727533   -0.0621079232508557  0.00628022287785056   -3.36398830862652   -0.429344312146806   0.220076269805878     -0.0141304450734351   1.86789591416677    0.131624640989186     -0.0874949132717207   0.00533448839482066   -0.349446065958659  -0.000108743133007239 0.00961626900463011   -0.000583721048212146 0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
10                      -7.94617073706378             3.05358061930519          -0.96854271695036   0.328701833762793     -0.0641329222023237  0.00621123983538322   -0.000237030434091669 0.852119696321722    0.0960439733951024   -0.0905840253406567  0.00745876905697177   -2.46581768172236   -0.751883096438912   0.263733211281735     -0.0163560022914423   1.39627843374341    0.308820068352249     -0.114561775416081    0.00691668166539836   -0.265745759050759  -0.0332960394074053   0.0151080521904201    -0.000924908549758687 0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
""")

COEFFS_FORELAND_75Bars  = CoeffsTable(sa_damping=5, table="""\
IMT                     a1                            a2                        a3                  a4                    a5                   a6                    a7                    a8                   a9                   a10                  a11                   a12                 a13                  a14                   a15                   a16                 a17                   a18                   a19                   a20                 a21                   a22                   a23                   tau    mean_phi_ss sigma_tot phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv                     -6.57469117091242             4.99953275450203          -2.63630223447654   0.964945468210338     -0.186558236049026   0.0174520362677213    -0.000628362710479202 0.736128035784037    -0.18405076231684    -0.00042944959645579 0.00066348420610104   -1.8547283707141    -0.840201161811686   0.244402214776006     -0.0135418482823542   0.823870328769334   0.644330715358799     -0.174753020261893    0.0101735495940055    -0.147006060895269  -0.12774355277971     0.0348389980606951    -0.00212152529378195  0.5010 0.4600      0.68015   0.00000 0.00000 0.00000 5   7   16   36
pga                     -5.05332584690767             5.79901385618117          -3.26108447540141   1.19210621748764      -0.230359980632197   0.021619901601699     -0.000782956443637468 1.51940367002587     -0.720027011149298   0.106344661381523    -0.00594637999032869  -2.77260459099821   -0.24536021612361    0.131284092376549     -0.00685788526516533  1.24467365974551    0.367784426871266     -0.124915752970536    0.00740894059934724   -0.229179987928886  -0.0780429527375448   0.0262393564636032    -0.00167085432058654  0.3532 0.4600      0.57998   0.58000 0.47000 0.35000 5   7   16   36
0.01                    -5.03308929500615             5.95532718084826          -3.39343112087485   1.23604164503124      -0.237614520763719   0.0222194887601988    -0.000802863647712344 0.802943090522214    -0.279325682625856   0.0232976265234875   -0.00103591219783821  -1.74296656825196   -0.895132238261383   0.255974179083335     -0.0143534291337507   0.675157965888372   0.728362111455346     -0.194846828811973    0.011661105344992     -0.124483564194218  -0.144417088330239    0.0392059949105782    -0.00246546217489565  0.3529 0.4600      0.57975   0.58000 0.47000 0.35000 5   7   16   36
0.02                    -4.43206432247003             5.28465897257925          -3.02603332313367   1.13205445952515      -0.22151131795556    0.0209278683657994    -0.00076115858868952  1.0689804726595      -0.237409323769859   -0.00555438967927899 0.00139029735859209   -1.75340682380594   -1.10841218537062    0.319880426124991     -0.0190813179670835   0.483693068213756   0.907166010359534     -0.238980263703154    0.0147974549190806    -0.0670615841431125 -0.183468585372862    0.0482028784349703    -0.00309636273005866  0.3674 0.4570      0.58639   0.56796 0.46097 0.37408 5   7   16   36
0.03                    -4.04210060864728             5.44859211600948          -3.18872979204392   1.18464682025448      -0.230504590372225   0.0217318893401123    -0.000790160449559954 -0.158122957812843   -0.140927722712977   0.0429574977602812   -0.00390076427302286  -0.295123236733824  -1.089881063778      0.233564502447648     -0.011244988579028    -0.174364265387554  0.82803917127361      -0.184774855589626    0.0103761348039583    0.0264371358055296  -0.157819864758864    0.0370772355906604    -0.00225036694505071  0.3749 0.4552      0.58975   0.56092 0.45569 0.38817 5   7   16   36
0.04                    -3.50814623952043             5.12219807303376          -3.03194540860864   1.13853168852237      -0.223262246959654   0.021168012124203     -0.000772783590792309 -0.90636237595157    -0.131883357296504   0.0837730845605056   -0.00776230060156996  0.190320157066656   -0.864327590730317   0.147889509691303     -0.00478196050821506  -0.1641385850468    0.608606117573065     -0.127411000142113    0.00644367240586362   -0.0155828286986485 -0.105724296802532    0.0253639231320855    -0.00148562916772507  0.3801 0.4540      0.59211   0.55592 0.45194 0.39816 5   7   16   36
0.05                    -3.92777796171344             5.52887582632792          -3.21473751969697   1.18721309042827      -0.230768410639752   0.0217793517570532    -0.000792988769028437 -0.0493699819797845  -0.505714033133351   0.136800297811278    -0.0102375911940411   -1.09664279192669   -0.324104366588237   0.0721247191165782    -0.00115227455058761  0.616619347537972   0.306776306172964     -0.0861748128078448   0.00443903307374138   -0.168559593956199  -0.0515957775046445   0.0182246583515166    -0.00113939323443705  0.3855 0.4530      0.59483   0.55204 0.44903 0.40592 5   7   16   36
0.1                     -4.83765240743472             5.35861764446358          -2.88900894363929   1.07130139033827      -0.210647928335339   0.020025033377757     -0.000732352385687606 2.14999080541572     -0.662788770699336   0.0491817789641587   -0.000854606439040395 -3.10825602633171   -0.550978366103919   0.239184773325717     -0.0150785456886178   1.36444574533634    0.60655615875537      -0.195722444735205    0.012454596858992     -0.241851121407014  -0.136227895835483    0.0415449686551001    -0.00270719823935258  0.3864 0.4500      0.59312   0.54000 0.44000 0.43000 5   7   16   36
0.15                    -5.59159308439166             5.86851694460389          -3.1015168939449    1.12766380927885      -0.219334019428562   0.0207254047539055    -0.000755141763433999 2.34636496241716     -0.804453296366617   0.0808946180192766   -0.00295637570861426  -3.36881042457513   -0.244116192393059   0.159131098986757     -0.00939840263123012  1.49509456114707    0.413830301871979     -0.142447572057814    0.00859710341162531   -0.256533523128516  -0.0977965160105405   0.0304293407248427    -0.00189432799558631  0.3841 0.4675      0.60507   0.58095 0.47510 0.40075 5   7   16   36
0.2                     -5.15804111381105             4.9009034654897           -2.53139032218476   0.962935241138934     -0.193468608299341   0.0186169966448811    -0.000685440012586735 1.79680758635409     -0.524935754930739   0.0334854113401513   -0.000292736818778834 -2.67252479995888   -0.538132590755503   0.202865782329216     -0.0116463380535839   1.14056807177044    0.549631723375205     -0.160999810552775    0.00948569804962739   -0.190625218291043  -0.119854325486219    0.0332048334806357    -0.00201793894557636  0.3690 0.4800      0.60546   0.61000 0.50000 0.38000 5   7   16   36
0.25                    -5.78733651809744             5.43501490441464          -2.82333068273784   1.05473057822457      -0.209007122501049   0.0199348098234132    -0.000729408632453431 2.19831519120513     -0.854033056320698   0.104295209804105    -0.00482070359652399  -3.25583670850077   -0.0235676952036678  0.0897224947783986    -0.00431698046524624  1.45010189376995    0.269987112827089     -0.0990721687253469   0.00544802941506089   -0.243079458341577  -0.0695353099474771   0.0220142914879446    -0.00128569904407564  0.3445 0.4800      0.59082   0.62651 0.50000 0.37450 5   7   16   36
0.3                     -6.27897059802911             6.04565191156584          -3.23288917679747   1.18853797082238      -0.231310188498635   0.0217787802840481    -0.000789663428754964 1.58708244159589     -0.529468503768307   0.0474926581753491   -0.00157189973905197  -2.49946021720347   -0.404812726921511   0.155681571088359     -0.00810384719031903  1.06677216644686    0.463387885478564     -0.132979132727473    0.00741755302797299   -0.174235421779449  -0.103766574302704    0.0281043160280195    -0.00164334389724569  0.3377 0.4800      0.58688   0.64000 0.50000 0.37000 5   7   16   36
0.35                    -6.59028601404524             6.33107077997725          -3.42315249370846   1.25100627915301      -0.241463639059633   0.0225745327247493    -0.000813692635103873 1.58119464259429     -0.597651288222297   0.0678457783667113   -0.00305547420211465  -2.6095259441953    -0.226259790326846   0.110858683995915     -0.00502999801048073  1.15012534071121    0.354962860091219     -0.106749440469857    0.00563591937460068   -0.190167103654796  -0.0834056224726638   0.023264935255364     -0.00131578386258254  0.3482 0.4740      0.58812   0.62793 0.49396 0.37000 5   7   16   36
0.4                     -7.88957204437175             7.77506089547374          -4.21316842730352   1.48232702391876      -0.278064888660444   0.0255201179264213    -0.000908171729997802 3.04039087034841     -1.48769106541919    0.234515653628523    -0.0128829081139085   -4.59451234423807   1.01835716682758     -0.126469422918911    0.00916653298777281   2.17362897795164    -0.29334509131991     0.0179546746997061    -0.00188688760659144  -0.367871882873541  0.0303397174716343    0.0012721773441708    1.81406187665755e-05  0.3552 0.4687      0.58811   0.61747 0.48874 0.37000 5   7   16   36
0.45                    -8.31471338126794             8.40233899293099          -4.6265349690501    1.60886631042734      -0.297647470451922   0.027017906949431     -0.000953304427123181 2.61475971668654     -1.31090396956081    0.210030807568096    -0.0117531449834418   -4.23016279085783   0.895609279929402    -0.113286862860018    0.00873730282131017   2.03130888576359    -0.251350359416791    0.0144743441522631    -0.00184200227507064  -0.346479587809059  0.0246979192095176    0.00164970228239129   2.15435560468594e-05  0.3495 0.4641      0.58097   0.60825 0.48413 0.37000 5   7   16   36
0.5                     -8.71739614680091             9.02026174215506          -5.028805330014     1.72927358945475      -0.31580675448336    0.028367888584461     -0.000992721031231071 2.18871329096299     -1.13350920723757    0.185422120779421    -0.0106167519704263   -3.88324465995552   0.776038084817404    -0.10004602796433     0.00828371033807448   1.90176480883569    -0.212111424753618    0.0110913013162884    -0.00178969235619481  -0.327803857274107  0.0196627882347915    0.00198960833317486   2.4857080054713e-05   0.3548 0.4600      0.58095   0.60000 0.48000 0.37000 5   7   16   36
0.55                    -9.1090345536822              9.62777119300447          -5.41693264701414   1.84287741106876      -0.332532468272029   0.0295787379844092    -0.00102702495686182  1.78686872062342     -0.964064934337458   0.16153153478068     -0.00949185706484097  -3.56594348497949   0.660234831030135    -0.0859742452411095   0.00772109404087682   1.78586323503603    -0.17313211728367     0.00691413670372935   -0.0016577890970676   -0.311417390438495  0.0145247471610395    0.002495221435837     1.27501029815064e-05  0.3639 0.4586      0.58544   0.59175 0.47587 0.37413 5   7   16   36
0.6                     -9.48147861397032             10.2102623605741          -5.78286764686674   1.94760071001981      -0.347548223978046   0.0306321312628962    -0.00105573904085502  1.39365114257667     -0.793622066511025   0.136777285470618    -0.00829054951859361  -3.25361466337752   0.534993726024774    -0.0688296759931572   0.00692798404082214   1.67026474280751    -0.127453332380082    0.000771160433878605  -0.00138263779848476  -0.294882422055528  0.00803271801576256   0.00337930570742474   -2.65791461497796e-05 0.3757 0.4574      0.59187   0.58422 0.47211 0.37789 5   7   16   36
0.65                    -9.78581335068295             10.6930400255876          -6.08399903919114   2.03122401499529      -0.358967327433556   0.0313783430644502    -0.00107407298044597  0.994935830233409    -0.611392016642579   0.108880488638594    -0.00686732699065158  -2.92343587107696   0.385129753179967    -0.0456119211370618   0.00572083321979708   1.54308056110808    -0.0676473568719268   -0.00875022555677678  -0.000880486737380558 -0.276189263104723  -0.000999884052500456 0.00485942449518668   -0.00010556439430652  0.3831 0.4562      0.59573   0.57729 0.46864 0.38136 5   7   16   36
0.7                     -10.0561289528869             11.1109680206135          -6.33819533483385   2.09911925260819      -0.367704755924835   0.0318975646226958    -0.00108481772160336  0.644747264456221    -0.44873531416833    0.0835923368598391   -0.00555924914201909  -2.63922597923088   0.249282951798731    -0.0236152942428402   0.00453643425704878   1.43531641681854    -0.0129507888393493   -0.0179625880269624   -0.000374518016298915 -0.260567960279522  -0.00927173493946804  0.00629745421513692   -0.000185381788717144 0.3886 0.4551      0.59846   0.57087 0.46544 0.38456 5   7   16   36
0.75                    -10.3089499874849             11.4745963644912          -6.54938391029769   2.15247142826686      -0.373990574160005   0.0322125421025681    -0.00108883668909068  0.38416429498739     -0.327896550077805   0.0647762789030702   -0.00458229693357863  -2.44395614086054   0.151303054646962    -0.00703029250838659  0.0036097012068004    1.36574813649745    0.0260833587963028    -0.0250095033744405   3.18009731939449e-05  -0.250909442344851  -0.0151431136029815   0.00740608388071471   -0.000250177485805825 0.3852 0.4542      0.59554   0.56490 0.46245 0.38755 5   7   16   36
0.8                     -10.4704094013695             11.7029284262045          -6.67878296946358   2.18099501316293      -0.37628028613241    0.0322019700031159    -0.00108229161130072  0.0908418432198812   -0.176522785512197   0.0390430710927003   -0.00314987749251424  -2.18508609705534   0.000347068858683678 0.0210759652329983    0.0019397601881299    1.25922261071608    0.0944973702183993    -0.0383343366189369   0.000839693302440309  -0.23445916176271   -0.0262935398666115   0.00963100902421112   -0.000386034313441457 0.3850 0.4532      0.59468   0.55932 0.45966 0.39034 5   7   16   36
0.85                    -10.6076298018721             11.8725064682804          -6.76574454977833   2.19586146386419      -0.376343908377884   0.0320107523778093    -0.00106992939916817  -0.135997743672241   -0.0549618516629357  0.0177899199737384   -0.00194220319044381  -1.99021510030703   -0.123986939469712   0.045393939728497     0.000452945848305257  1.18031142395813    0.151453310977468     -0.0500072195500203   0.00156601770292673   -0.222391273781245  -0.0355914976149928   0.0115793798849289    -0.000507743015738908 0.3811 0.4523      0.59150   0.55407 0.45703 0.39297 5   7   16   36
0.9                     -10.7040896626874             11.9668497325521          -6.80351345635519   2.19558108003812      -0.373997657126394   0.0316272297345906    -0.00105146510604788  -0.326572743707335   0.0540222949160373   -0.00211153319951873 -0.000777302684848556 -1.82223559492636   -0.242877981572641   0.069792545269174     -0.00107749072969693  1.11071612020815    0.20749064748244      -0.0619715462391906   0.00232551650407807   -0.211579598226959  -0.0448534468103676   0.013588951429653     -0.000635359793859302 0.3799 0.4515      0.59006   0.54912 0.45456 0.39544 5   7   16   36
0.95                    -10.762442912401              11.990093014838           -6.79513057656353   2.18122955220502      -0.369434625081605   0.0310684577285243    -0.00102748862046162  -0.481265606654783   0.150248896727297    -0.0205741761656846  0.00033731704854191   -1.68176338949605   -0.355062659121701   0.0939357251728042    -0.00262778275794055  1.05110556501544    0.261695368455673     -0.0740034243248346   0.00310303281528074   -0.202171417955104  -0.0538957460467768   0.0156163090614935    -0.000766011467343463 0.3733 0.4507      0.58525   0.54444 0.45222 0.39778 5   7   16   36
1                       -10.7898701672028             11.9548561694662          -6.74888720352868   2.15545682582172      -0.363098441180419   0.0303718767319511    -0.000999257995354133 -0.603957394574669   0.234926666242054    -0.0377045432103142  0.00140362746832488   -1.56518367094431   -0.461177198342106   0.117780060210673     -0.00418949808877188  0.999847861113808   0.314338053830531     -0.0860801853672099   0.00389466163702001   -0.193887131440162  -0.062776730100311    0.0176615769107056    -0.000899345232198412 0.3716 0.4500      0.58362   0.54000 0.45000 0.40000 5   7   16   36
1.05                    -10.7655536206901             11.8377915439518          -6.65344945027733   2.1152620666571       -0.354542484473078   0.0295025872552859    -0.000965662289273046 -0.714731466379623   0.317993298269781    -0.0550987868156642  0.00250508408299654   -1.45310390069176   -0.570069988833987   0.142646136652613     -0.00582703916286299  0.949192533587664   0.368643101421464     -0.0986359995016166   0.00471915350307808   -0.185629714989546  -0.0719024266439628   0.0197710175489992    -0.00103686796998481  0.3691 0.4482      0.58063   0.53956 0.44778 0.40000 5   7   16   36
1.1                     -10.7366633636213             11.6946975547551          -6.53787909725924   2.0688200945665       -0.345039087007101   0.0285631024120288    -0.000930030067947484 -0.763666044997555   0.371110262458242    -0.0676714434381841  0.00335019495224369   -1.40497809013825   -0.647598794610241   0.162351453117936     -0.00718349727778809  0.927473059289029   0.408290490233633     -0.108708183856938    0.00540514507049521   -0.182069150542856  -0.0785845442477778   0.0214570172118914    -0.00115040670634276  0.3650 0.4465      0.57674   0.53913 0.44566 0.40000 5   7   16   36
1.15                    -10.6807796440647             11.5095951698984          -6.39772994967126   2.01552279377089      -0.334551765017222   0.0275536233984917    -0.000892452808249235 -0.808105303057822   0.425587915863082    -0.080916719315975   0.00424973195378059   -1.35364062597525   -0.730775480667422   0.183458529463428     -0.00863360844538882  0.902634715761335   0.451111542462516     -0.119487719179051    0.00613626751431916   -0.177876262063608  -0.0858081712222272   0.0232572852618487    -0.001271060739549    0.3627 0.4449      0.57402   0.53873 0.44364 0.40000 5   7   16   36
1.2                     -10.592442936357              11.2726633801436          -6.22656666015376   1.95328357963022      -0.322720044436246   0.0264423654417304    -0.00085180627473986  -0.842711950398256   0.477491815980171    -0.094010572022258   0.00515132359749743   -1.30626564621409   -0.814313832294765   0.204860645676213     -0.0101072749585021   0.878294950471139   0.49444717512071      -0.130416411325862    0.0068771126095739    -0.173665353305206  -0.0931206274509756   0.0250763023610529    -0.00139274572791187  0.3625 0.4434      0.57267   0.53834 0.44170 0.40000 5   7   16   36
1.25                    -10.4826064889811             10.9968661596178          -6.03162627248646   1.88431644251493      -0.309914843810843   0.0252610316416653    -0.000809169457163743 -0.844998966487206   0.513634781244534    -0.104535935213962   0.00591439607780644   -1.29444143551277   -0.8792412550534     0.223019797705816     -0.0113957266638159   0.871051819894272   0.528239921239472     -0.139607670792046    0.00751579496629527   -0.172369541609151  -0.098737026223817    0.0265785041855576    -0.00149552217381171  0.3631 0.4419      0.57195   0.53797 0.43984 0.40000 5   7   16   36
1.3                     -10.361394270017              10.7016826001673          -5.82528423845633   1.81237355836634      -0.296735202142476   0.0240583762797914    -0.000766139860819629 -0.836027462635102   0.545279964225944    -0.114442268634102   0.00664808545257733   -1.29017801413888   -0.94114238478786    0.24072665413854      -0.0126596731108477   0.866322000988505   0.560833419597327     -0.148586606998449    0.0081415525707671    -0.171403156572492  -0.104163762335143    0.0280429492731476    -0.00159588621944273  0.3610 0.4404      0.56949   0.53761 0.43806 0.40000 5   7   16   36
1.35                    -10.2358160695892             10.3958610404847          -5.61216763782127   1.73873064654636      -0.283370168307293   0.0228485175974454    -0.000723130387953491 -0.804846806952927   0.565166623915969    -0.12229220743977    0.00726426342910024   -1.31070426799641   -0.988469011500579   0.255672436168979     -0.0137563291114209   0.873751878929568   0.585736907550897     -0.1560487126379      0.0086732823533978    -0.172538509289327  -0.108204704625578    0.0292282408075301    -0.00167876348050509  0.3605 0.4391      0.56808   0.53727 0.43634 0.40000 5   7   16   36
1.4                     -10.0890780243664             10.0671069726672          -5.38874821706414   1.6629019421966       -0.269797986105992   0.0216334783098206    -0.000680336609741955 -0.801141670704085   0.602331641863039    -0.133448802641056   0.00807897023626327   -1.2924032200102    -1.05952974440436    0.275115653508377     -0.0151227515828736   0.861176142170851   0.622703024566196     -0.165796746174872    0.00934292468212436   -0.170207674668044  -0.114324812695646    0.030808573555334     -0.00178562125945291  0.3616 0.4377      0.56779   0.53694 0.43469 0.40000 5   7   16   36
1.45                    -9.92225061499878             9.69726360639057          -5.13857869665757   1.57903710341927      -0.254974620558777   0.0203195472210146    -0.00063438607962628  -0.745883585644447   0.610133013778183    -0.139260995372634   0.00858127488115051   -1.33656497940701   -1.09310628223239    0.287465956659834     -0.016062338826993    0.879493901619411   0.64051413565308      -0.171845917920227    0.00978705561769556   -0.173167410225669  -0.117098884786756    0.0317351139864302    -0.0018523395205879   0.3604 0.4365      0.56604   0.53662 0.43309 0.40000 5   7   16   36
1.5                     -9.74653750107373             9.31772101174546          -4.8846694852144    1.49481633968536      -0.240232402421232   0.0190240946738428    -0.000589433033774702 -0.691886786114053   0.618843227202975    -0.145245905729726   0.00909358418058402   -1.37715632439066   -1.12804747829025    0.299972358811808     -0.017007164161934    0.895852070225016   0.658864013703944     -0.177929789078237    0.010230989832686     -0.175788606019155  -0.119947958536678    0.0326637583596248    -0.00191883562828496  0.3559 0.4352      0.56223   0.53631 0.43155 0.40000 5   7   16   36
1.55                    -9.57545342989156             8.94569794713044          -4.63573279032717   1.41242645552978      -0.225851824766282   0.0177638938229047    -0.00054580798107735  -0.63547879035216    0.626418580087306    -0.151046306868958   0.00959599399626191   -1.41711933912093   -1.16287146137556    0.31239617852622      -0.0179451750196397   0.91103691793558    0.677440129406005     -0.184009092777417    0.0106735155494039    -0.178117671159514  -0.122871532746265    0.0335972966824135    -0.00198545395207845  0.3507 0.4340      0.55804   0.53601 0.43005 0.40000 5   7   16   36
1.6                     -9.40450849349473             8.57365925754491          -4.38744488610809   1.33069623830564      -0.211671083797028   0.0165281096031539    -0.000503240512030347 -0.564401165277125   0.624987117669376    -0.155120760501115   0.00999364195001947   -1.4767343853962    -1.18437177931216    0.322117687585871     -0.0187131991683221   0.936945615975073   0.688451500571687     -0.188527714445329    0.011017017107247     -0.182429576471744  -0.124374627201529    0.0342358135750944    -0.00203329338010365  0.3479 0.4329      0.55537   0.53572 0.42861 0.40000 5   7   16   36
1.65                    -9.23989947315138             8.21032514370398          -4.14480822916366   1.25110399954367      -0.197926215848586   0.0153360230048464    -0.000462361264761184 -0.47818513570452    0.614346698296586    -0.157459460812013   0.0102881288520006    -1.55326230998969   -1.19468297600199    0.329635454689768     -0.0193469410103073   0.970668670224119   0.694054898272548     -0.191967608243012    0.0112947251374178    -0.187997795203191  -0.124987180383376    0.0346962365133118    -0.00207032760351099  0.3486 0.4318      0.55495   0.53544 0.42721 0.40000 5   7   16   36
1.7                     -9.04504215624828             7.81260827408196          -3.88502303368164   1.16699068227493      -0.183530657227305   0.0140958125827572    -0.000420055499373807 -0.427799406261778   0.624112618087135    -0.163475467368066   0.0107946584270243    -1.58375612277833   -1.23099932977995    0.341846240082947     -0.0202529714987998   0.98131496086846    0.712714381361259     -0.197784052793526    0.0117118784403739    -0.189602725512311  -0.127854466210254    0.0355707415193188    -0.00213189927555032  0.3485 0.4307      0.55401   0.53517 0.42585 0.40000 5   7   16   36
1.75                    -8.873463294004               7.44300345419045          -3.64068496499072   1.08771207744518      -0.169983454078442   0.0129318415933343    -0.000380463173744925 -0.333395313033291   0.606651166536497    -0.164296495298531   0.0109888976865117    -1.67180525023175   -1.23019451821353    0.346770564304236     -0.0207117748026301   1.02167896166203    0.711796003342666     -0.199706748669753    0.0118878209804311    -0.196443061987582  -0.127232677329025    0.0357461740424977    -0.00214993724427189  0.3470 0.4296      0.55226   0.53491 0.42453 0.40000 5   7   16   36
1.8                     -8.70958996796447             7.08570208565688          -3.40426989412037   1.0111421445552       -0.1569352907806     0.0118141993170606    -0.000342565933644072 -0.236446834110557   0.587931296156158    -0.164928875701665   0.0111744410326854    -1.76023461346503   -1.22905304579793    0.351667868287452     -0.0211723978784789   1.06134488370844    0.711196485875189     -0.201717742197817    0.0120717480377294    -0.203054921353337  -0.126734235070267    0.0359518598132517    -0.00217040627307752  0.3469 0.4286      0.55137   0.53465 0.42325 0.40000 5   7   16   36
1.85                    -8.53101390180516             6.71683119618666          -3.16411444634603   0.934098985010004     -0.143896643767456   0.0107035481097054    -0.000305085197259423 -0.163676188338912   0.581003546529689    -0.167433866342953   0.0114572629782174    -1.82171245982442   -1.24023241555791    0.358407888572015     -0.0217237068366052   1.08878494798655    0.715959399275947     -0.204503951813632    0.0122928531325852    -0.207710907323731  -0.127074617491594    0.0362770238537212    -0.00219658064039592  0.3448 0.4276      0.54931   0.53440 0.42200 0.40000 5   7   16   36
1.9                     -8.36398611368848             6.36031219745502          -2.93041504121502   0.859141142310306     -0.131244236088828   0.00962929092030315   -0.000268947316959264 -0.0635340707932495  0.557860473938419    -0.16695272872323    0.0115657064384501    -1.91531933768094   -1.23155318891847    0.361399029879222     -0.0220524103617642   1.13153324802545    0.711056248221791     -0.205447288132911    0.0124038892721602    -0.214910397929409  -0.125790348953873    0.0362906847012666    -0.00220407761129394  0.3435 0.4266      0.54775   0.53416 0.42079 0.40000 5   7   16   36
1.95                    -8.20132816155781             6.01598161356663          -2.70561479519779   0.787258797583453     -0.119146281021413   0.00860515014242549   -0.000234599465724942 0.0249293051760943   0.539937906839713    -0.167241081861463   0.0117116336630314    -1.99576270516383   -1.2281777844204     0.365095601775633     -0.0224123500818263   1.1683154837273     0.708415955954535     -0.206678435087421    0.0125274398648024    -0.221158403825654  -0.124856432993674    0.0363485547977497    -0.00221355319187751  0.3422 0.4257      0.54621   0.53392 0.41961 0.40000 5   7   16   36
2                       -8.02617314014993             5.65224765817545          -2.46970828957372   0.712356445817606     -0.106622670526436   0.00755080022495615   -0.00019939391749323  0.129854531111831    0.511697000371092    -0.165560076676977   0.0117397045097917    -2.09718966506746   -1.21095372048899    0.366076208426127     -0.0226070653917292   1.21554205771706    0.698742587006017     -0.206520740689749    0.0125663032110657    -0.229182927693211  -0.122716907669134    0.0361679291349549    -0.00220848970676828  0.3412 0.4248      0.54482   0.53369 0.41845 0.40000 5   7   16   36
2.5                     -6.6684775484358              2.88841045119528          -0.707018616482422  0.159932501783845     -0.0155243980748953  1.362480875029e-06    4.83017487385205e-05  0.456651188644025    0.572693487623111    -0.202840369353726   0.0148304454855753    -2.16111187600759   -1.49589330653482    0.450034442565491     -0.0285567349689524   1.19635582797231    0.845455697205115     -0.246026204577665    0.0152651285365081    -0.222761621908021  -0.145471245760656    0.0420407753745444    -0.00260168453468329  0.3321 0.4166      0.53278   0.53166 0.40830 0.40000 5   7   16   36
3                       -5.84206952781517             1.13061183746446          0.393653354456592   -0.175132305362759    0.0377313988658422   -0.00422454138895925  0.000180308440500834  0.817481080977566    0.516217318965556    -0.20819931345847    0.0156905170048108    -2.42521622808283   -1.5141324239025     0.469497595360447     -0.0302036155043454   1.31553809993194    0.835506911022231     -0.249306650241187    0.0156065906754323    -0.243729135712875  -0.139655252022048    0.0414822218349477    -0.00258324121839611  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
3.5                     -5.02321585597621             -0.462768346894148        1.36012670909079    -0.461113489506829    0.0818991473450723   -0.00762774570652359  0.000283449400860214  1.1521312467498      0.386096047781037    -0.190457916387482   0.0148476366742991    -2.74695110509117   -1.37747477676915    0.448032828640553     -0.0290190264754752   1.4810803055293     0.740472497164148     -0.231269772974902    0.0145115208318137    -0.274126092811044  -0.118998502784115    0.0373099375992966    -0.00232486311670115  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
4                       -5.31445345891238             -0.672871956053453        1.57760535020792    -0.52726288895724     0.0904949013024448   -0.00807254418640061  0.000288179061064831  3.00215601160927     -0.640880417584972   -0.00806873758060579 0.00443486096485056   -5.04962683388133   -0.0390108142182571  0.202162465507245     -0.0146238637744609   2.64768538231671    0.0323960239140611    -0.0979453115209292   0.00658934230324738   -0.477183074026203  0.00790455563860447   0.0130655126351428    -0.00087285874113943  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
5                       -5.64064823244533             -0.610999809104297        1.53340279791329    -0.494537575923482    0.0802208503879397   -0.00671666827895325  0.00022512925542148   3.65851439077081     -1.0516534072216     0.0701190375130818   -0.000224413548267213 -5.79532906256482   0.487737277866894    0.0969641014528817    -0.00823098916377004  3.023661802332      -0.262382770970511    -0.0376115212038514   0.00292206205010909   -0.542722580146242  0.0624754434179575    0.00180362851961299   -0.000191456348864899 0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
6                       -5.93333056513431             0.098739964875573         0.961953477274722   -0.291446741115548    0.0426800866223301   -0.00320356179987194  9.45814373529845e-05  1.88909892860652     -0.162855201067552   -0.0767330810662324  0.00773317942126022   -3.53835746641383   -0.643795434890855   0.28706712823396      -0.0188001200576594   1.90080379019812    0.294637947181957     -0.132387108041446    0.00831840417761097   -0.350495712404151  -0.0326082401740599   0.0182025869382428    -0.00114372654537025  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
7                       -7.26436442256171             2.01151553497494          -0.227412899739449  0.0861155566320003    -0.0221295638244123  0.00249677094119649   -0.000105917185083749 1.83431965865454     -0.241676838352771   -0.0508566858083055  0.00582561056201292   -3.48684148658164   -0.508533262006088   0.247215689902775     -0.0160320184908442   1.88376258518633    0.212723629565566     -0.110171701016986    0.00686515225451765   -0.348566752568521  -0.0173206443214255   0.0142545806203346    -0.000895707226025834 0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
8                       -8.00971555487462             3.05569332576059          -0.8865844093771    0.298062768543727     -0.0589525086907512  0.00576668805611753   -0.000221572905263774 1.74006190178607     -0.287277680307656   -0.0325940714416031  0.00443370746028287   -3.34319851825824   -0.453294254530879   0.225596925537138     -0.014485444391766    1.80033549005251    0.189193675086616     -0.101092396847131    0.00627862446823875   -0.332679578504415  -0.0143447763457875   0.0130506968114372    -0.000826955576192081 0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
10                      -8.89616399409441             3.82295484619195          -1.29301720784567   0.426310708719824     -0.0821750892343912  0.0079218185692639    -0.00029977707904657  3.35953773826936     -1.35519982116262    0.173705309306849    -0.00788604346172544  -5.35714826823203   0.920153620376867    -0.0417912639242878   0.00147457931373009   2.78499439720122    -0.494130813960782    0.0316012388426109    -0.00157734304928458  -0.499307111174816  0.101940037398947     -0.00945449770327235  0.000498297268513102  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
""")

COEFFS_FORELAND_90Bars  = CoeffsTable(sa_damping=5, table="""\
IMT                     a1                            a2                        a3                  a4                    a5                   a6                    a7                    a8                   a9                   a10                  a11                   a12                 a13                  a14                   a15                   a16                 a17                   a18                   a19                   a20                 a21                   a22                   a23                   tau    mean_phi_ss sigma_tot phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv                     -7.26876674629289             6.08963385302017          -3.31507446158009   1.17940533292671      -0.222510936489639   0.0205038808145898    -0.000731672729411663 0.290911984537749    0.105040997412745    -0.0569331093714345  0.00410652179670896   -1.14906550623334   -1.30984203506267    0.337662259478034     -0.0192917452757793   0.430675205090011   0.909805698096816     -0.228187792240094    0.0135040442590207    -0.0739460222541126 -0.177497810946461    0.044931025347864     -0.00275442685227546  0.5010 0.4600      0.68015   0.00000 0.00000 0.00000 5   7   16   36
pga                     -5.00310682403581             5.92938822758371          -3.41666915736972   1.25743678305428      -0.243111417503381   0.0228089084146908    -0.000825891003915581 0.491736022411357    -0.120703192634808   -0.00293483626868646 0.000382318681752032  -1.33823717187248   -1.11143719437234    0.293153502096488     -0.0164126015129734   0.489502218123629   0.833005600456351     -0.213307702035917    0.0126953584339756    -0.093970501498063  -0.162486090383482    0.0424573711718931    -0.00264895393066593  0.3532 0.4600      0.57998   0.58000 0.47000 0.35000 5   7   16   36
0.01                    -5.4238471324466              6.53513434442895          -3.76130659889304   1.35907721228572      -0.259373877296683   0.0241457470546542    -0.000870114181211109 0.734131556851785    -0.243552010794904   0.0178454081688453   -0.00078448806517579  -1.60616151083379   -0.97519744908536    0.269247878178229     -0.0150182910584174   0.59789889480444    0.775539562740956     -0.20303093671649     0.0120898697235417    -0.109241497465197  -0.154027262110554    0.0409322401592339    -0.00255922721148283  0.3529 0.4600      0.57975   0.58000 0.47000 0.35000 5   7   16   36
0.02                    -4.79506448795343             5.76504241369016          -3.31913219152396   1.23125002998668      -0.239452502099443   0.0225477388713252    -0.000818528917526168 1.27948101395641     -0.383463991405525   0.0247215420274951   -0.000547288914679664 -2.0023838728356    -0.932844266433628   0.282141941379996     -0.0165790030611505   0.612439020572246   0.816416449565954     -0.21941523716        0.0134916693347547    -0.0886982261089623 -0.168229232001896    0.0449050658197046    -0.00287480649147595  0.3674 0.4570      0.58639   0.56796 0.46097 0.37408 5   7   16   36
0.03                    -4.17919826996848             5.64401628293089          -3.33047231542288   1.24102701162779      -0.241761277498651   0.022811692945998     -0.00082996496326037  -0.200480667543389   -0.122170082845492   0.0406245917069123   -0.00382480419477351  -0.20631428023938   -1.1385106006847     0.241062542276389     -0.0115859845162618   -0.224012082563616  0.857016231008287     -0.18962512009478     0.0106190802664785    0.0371112327992494  -0.16442215408695     0.0382590531002725    -0.00231427824310386  0.3749 0.4552      0.58975   0.56092 0.45569 0.38817 5   7   16   36
0.04                    -3.38360185625591             5.20856354139875          -3.17466245244146   1.20139696827402      -0.235710253067637   0.0223371877801448    -0.000815259393148012 -1.94069008379168    0.420282003912868    -0.0105030052465146  -0.00256343654099576  1.38803741454164    -1.50613886826123    0.25683331393128      -0.0107266841297219   -0.724813932964581  0.9104701385218       -0.178690798886865    0.00923462823839182   0.0781020137613923  -0.156684849415297    0.0340743247274302    -0.00196115623371166  0.3801 0.4540      0.59211   0.55592 0.45194 0.39816 5   7   16   36
0.05                    -3.81340057002067             5.42248939848951          -3.20006061217868   1.19952910044905      -0.23509243196188    0.0222920334707019    -0.000814109343799076 -0.316420770254613   -0.349269394421034   0.108378691978858    -0.00859489402816292  -0.761707936484261  -0.524690070459789   0.108326304668773     -0.003207991988089    0.446577128733269   0.411235269317964     -0.105286213013316    0.00553016014773913   -0.136856996063117  -0.0715828593148302   0.0219414035610664    -0.00135382825137296  0.3855 0.4530      0.59483   0.55204 0.44903 0.40592 5   7   16   36
0.1                     -4.63701022187681             5.07035746319802          -2.75699966646046   1.04828895441898      -0.209405303202315   0.0200890198314088    -0.000738775949162804 2.22026901264604     -0.713440297470808   0.0603050264702325   -0.00160469216429671  -3.18102800693122   -0.496430137791723   0.226016964446236     -0.0141162051809141   1.39890934136416    0.580661395175986     -0.189273831994839    0.0119674732428627    -0.247451570600156  -0.132006391865832    0.0404636124563658    -0.0026232493861877   0.3864 0.4500      0.59312   0.54000 0.44000 0.43000 5   7   16   36
0.15                    -5.53668432063211             5.81892209005249          -3.11984107553707   1.15125843792829      -0.225713184073138   0.0214210981383531    -0.000782495144865081 2.46045196094273     -0.878763253227194   0.0960537787258381   -0.00392730045029468  -3.51914619274179   -0.143883486868455   0.137769847090616     -0.00797281190581528  1.57705627552339    0.358888105014382     -0.130683283402509    0.00780566960490837   -0.271755974379292  -0.0875594082433318   0.0282373212418734    -0.00174655916543336  0.3841 0.4675      0.60507   0.58095 0.47510 0.40075 5   7   16   36
0.2                     -6.54030653692611             6.46167952375063          -3.36822894768684   1.217165710282        -0.236274430177786   0.0223010346141773    -0.000811451520422787 4.09031912987753     -1.8309924402297     0.268962226879965    -0.0138698965729362   -5.50655470330618   1.10520369679927     -0.098445584220794    0.00598769420009906   2.53777637067627    -0.270201453570274    -0.009173469455986    0.000522278725153944  -0.428327582758319  0.0208019426932235    0.00697756182607084   -0.00046039752183291  0.3690 0.4800      0.60546   0.61000 0.50000 0.38000 5   7   16   36
0.25                    -6.94016762066179             6.78017773235319          -3.56469518384124   1.28224377251126      -0.247254898226216   0.0232073336493704    -0.000840630103451092 3.89767585360609     -1.7850349858793     0.26744198865796     -0.0140228099465794   -5.31222772874495   1.12016832273772     -0.11368672893786     0.00731247367949889   2.44939913670776    -0.291421259164582    0.00162771206028279   -0.000354051065446286 -0.411321848254703  0.0256807412014473    0.00483395783468591   -0.000290556478954683 0.3445 0.4800      0.59082   0.62651 0.50000 0.37450 5   7   16   36
0.3                     -7.3657548421926              7.23364110002112          -3.85868912569494   1.37691408280655      -0.262722304777035   0.0244505529890997    -0.000879819048982258 3.63338226811133     -1.69600252027182    0.258032617268604    -0.0137233445085088   -5.05878794820509   1.08446004573644     -0.118207257745444    0.00796787361936549   2.34054244648187    -0.287928251539718    0.0067756742296881    -0.000864772246246058 -0.392572388588086  0.0262650130166058    0.00372666314206848   -0.000188961565523535 0.3377 0.4800      0.58688   0.64000 0.50000 0.37000 5   7   16   36
0.35                    -7.8321229975009              7.81100194842029          -4.2315995822339    1.4933143296693       -0.281201466398994   0.0259002521673611    -0.000924568878210312 3.32004728893296     -1.58034282780769    0.244073346405488    -0.0131753052985127   -4.76852132886688   1.01733389877126     -0.116096450855787    0.00820588101575572   2.21929368819584    -0.268736209616624    0.00828622137864834   -0.0011352737025115   -0.372741511382334  0.0240687442947637    0.00329737511508671   -0.000133081263258676 0.3482 0.4740      0.58812   0.62793 0.49396 0.37000 5   7   16   36
0.4                     -8.31153560312506             8.4706893881255           -4.65866313466491   1.62385284985214      -0.301450337869222   0.0274539839501077    -0.000971544297048998 2.91252480623161     -1.412325300405      0.220931643619393    -0.0121120450705883   -4.40631537031639   0.90265321457195     -0.105065339432975    0.00793036739625239   2.07452418640662    -0.230964971703352    0.00616675060846002   -0.00119577571296659  -0.350491739431516  0.0192402185374809    0.00339967914354873   -0.000107885582593197 0.3552 0.4687      0.58811   0.61747 0.48874 0.37000 5   7   16   36
0.45                    -8.7935489860088              9.16059609728275          -5.0977738439622    1.75502212754416      -0.321337351379218   0.0289453744101295    -0.00101557174182226  2.52304853089324     -1.25578559090482    0.19999225469052     -0.0111799877797499   -4.08199021925978   0.802023603217434    -0.0956938566648115   0.00770982608745787   1.95108666589277    -0.199459011906735    0.0045057608408122    -0.00125104984715073  -0.332247823010333  0.0153631315946746    0.00346765762940065   -8.72658942620342e-05 0.3495 0.4641      0.58097   0.60825 0.48413 0.37000 5   7   16   36
0.5                     -9.20484696345073             9.77375173753602          -5.48753667138701   1.86891851682036      -0.33805063926102    0.0301480880075816    -0.00104933177524782  2.109280456814       -1.08401356925224    0.176164720463071    -0.0100765340885702   -3.74208067537231   0.683741822328374    -0.0822332728168265   0.0072197591259965    1.8227884311984     -0.159352391152852    0.000702983318317025  -0.00116033941941982  -0.3135304921018    0.0100081783572218    0.00391477457197497   -9.28618898610131e-05 0.3548 0.4600      0.58095   0.60000 0.48000 0.37000 5   7   16   36
0.55                    -9.61366681224272             10.3982127641752          -5.87813294076351   1.98081388388102      -0.354136497030712   0.0312803425351592    -0.00108032098043958  1.69125421498569     -0.906358869719017   0.150916939700179    -0.0088789717582821   -3.4044952558263    0.557162969175265    -0.0663437700401634   0.00655983235789621   1.69803030820201    -0.115793961371103    -0.00420437517121261  -0.000992510800263548 -0.295844107963903  0.00423198053409408   0.00451279364181426   -0.000108817927874941 0.3639 0.4586      0.58544   0.59175 0.47587 0.37413 5   7   16   36
0.6                     -9.91138680752551             10.8695415904707          -6.17388949505261   2.06260554364605      -0.365154042595493   0.0319819970095568    -0.00109677709724861  1.25060699317232     -0.709219913478318   0.12137097665507     -0.00740136588470568  -3.03311008437456   0.397719936816734    -0.0429492495194096   0.00540205802680407   1.55433726491655    -0.0537233831631866   -0.0133471492477968   -0.000542434516766238 -0.274723871882224  -0.00493872947024284  0.00588566590330935   -0.000176534711247903 0.3757 0.4574      0.59187   0.58422 0.47211 0.37789 5   7   16   36
0.65                    -10.2309100282382             11.3510856281194          -6.46418654604934   2.14071174478741      -0.375410143379353   0.0326152265919051    -0.0011109332944915   0.897331266379162    -0.552866303263483   0.0981161745262448   -0.00624431078070124  -2.75225425403765   0.275378693609315    -0.0246115538325257   0.0044728598727986    1.45019865279694    -0.00676210183262785  -0.0205838129160695   -0.000170957204467553 -0.259845268708952  -0.0118377248281033   0.0069841513398272    -0.000233580589201619 0.3831 0.4562      0.59573   0.57729 0.46864 0.38136 5   7   16   36
0.7                     -10.4472957594478             11.680862669134           -6.66109318831952   2.1899168359996       -0.380932334793689   0.0328563781824548    -0.001112155296511    0.524201200384389    -0.375044982562459   0.0698012930152184   -0.00474828251620883  -2.42832839793005   0.112592359373464    0.00284245634218591   0.00294593199158753   1.32027214974973    0.0630517303395194    -0.0328683578152683   0.000528436820759406  -0.240202245126487  -0.0228578746352589   0.00898136464797907   -0.00034866149991253  0.3886 0.4551      0.59846   0.57087 0.46544 0.38456 5   7   16   36
0.75                    -10.6230224681074             11.9286032357761          -6.80085353052907   2.22076897178506      -0.383439072191467   0.0328501060363415    -0.00110531714591895  0.226282194388909    -0.23184435493818    0.046810273450799    -0.00352450610776368  -2.18235471128002   -0.0167754231896953  0.0253887158575671    0.00166355034994457   1.22558337287225    0.117779458764045     -0.0429021018684794   0.00111283755994381   -0.226331996396805  -0.0313696216607077   0.0105921710269411    -0.00044333712003522  0.3852 0.4542      0.59554   0.56490 0.46245 0.38755 5   7   16   36
0.8                     -10.7440868833868             12.0829653696055          -6.88019057944956   2.23283860238982      -0.382910089321892   0.0325973143407365    -0.00109051819458633  -0.0487131527529853  -0.0919310467140031  0.0232948715470917   -0.00222725259532586  -1.94286099659612   -0.155792627253606   0.0511530687317715    0.000139522933026893  1.128780959113      0.180110116312841     -0.055007016061127    0.00184318437757382   -0.211620125169886  -0.0414132526431405   0.0125914820104155    -0.000564668531129795 0.3850 0.4532      0.59468   0.55932 0.45966 0.39034 5   7   16   36
0.85                    -10.8510111720586             12.1894175149855          -6.92344728762357   2.23341036983255      -0.380556810696027   0.0322015868618402    -0.00107129586988383  -0.242411601007229   0.0126038395275461   0.00487696326192045  -0.00117317255476964  -1.78077197327617   -0.264332652828055   0.0729626793761461    -0.0012161255290757   1.06448242002049    0.230154621360575     -0.0655914456750352   0.00251276351440704   -0.201907232178844  -0.0495995323342138   0.0143639525209048    -0.000677140708841655 0.3811 0.4523      0.59150   0.55407 0.45703 0.39297 5   7   16   36
0.9                     -10.8791072785365             12.1796726489034          -6.89733941489499   2.21328373471289      -0.374931389412676   0.0315436100725358    -0.00104367841658598  -0.457673020338103   0.135957712974755    -0.0176327253731203  0.00014218310879346   -1.57629921104495   -0.405525934040152   0.101536058336981     -0.00299281859709528  0.976125086847795   0.297819763138868     -0.0797516289207924   0.00340276492096701   -0.187860275938585  -0.0608813452181693   0.0167570969150761    -0.000827684892851918 0.3799 0.4515      0.59006   0.54912 0.45456 0.39544 5   7   16   36
0.95                    -10.880657980651              12.1182268575302          -6.8379878782571    2.18338611916447      -0.367848813245637   0.0307783863260925    -0.0010129765507638   -0.635675540307339   0.246309091027479    -0.0387192500732915  0.00141048572280783   -1.40391145169185   -0.537598819469037   0.129411715267748     -0.00476235845452445  0.901083794572245   0.361670383348774     -0.0935870226696625   0.00428565216930702   -0.175939127823885  -0.0714910285325704   0.0190756244272813    -0.000975226912167714 0.3733 0.4507      0.58525   0.54444 0.45222 0.39778 5   7   16   36
1                       -10.8531907264178             11.9969036397369          -6.73893261816934   2.1414199309877       -0.358885097820306   0.029866186820035     -0.000977696837984325 -0.763338662579654   0.334245882764399    -0.0564557356645394  0.00251134016137253   -1.2779449029152    -0.65003897250439    0.154447010771213     -0.00639242439813519  0.845358436765674   0.417301192514048     -0.106215466811668    0.00510825442133577   -0.166975340431137  -0.0808207914331131   0.0212016829390394    -0.001112967903345    0.3716 0.4500      0.58362   0.54000 0.45000 0.40000 5   7   16   36
1.05                    -10.7912359739728             11.8140506696014          -6.60152378457718   2.08820743723444      -0.348217524967701   0.0288245407163467    -0.000938508323364729 -0.859399703555345   0.410071674169414    -0.0726526167800083  0.00354715571459765   -1.17872586365533   -0.753306781289458   0.178434271291274     -0.00798225740604791  0.800445753438851   0.469099867385049     -0.118367774637744    0.00590985797829759   -0.159675818539862  -0.0895149679622787   0.0232386449815038    -0.0012462227208394   0.3691 0.4482      0.58063   0.53956 0.44778 0.40000 5   7   16   36
1.1                     -10.698029818851              11.5827439304214          -6.43532963720154   2.02672725539662      -0.336313441594052   0.0276901167802927    -0.000896561100355532 -0.956136690895923   0.491096466702479    -0.0903051957135131  0.00468595642113313   -1.06906248416082   -0.867623682014884   0.20493708519368      -0.00973605020874077  0.748629560650153   0.526903742775032     -0.131817488437234    0.0067944873919958    -0.151069208453067  -0.0992574846053419   0.0254966491538675    -0.00139356346592707  0.3650 0.4465      0.57674   0.53913 0.44566 0.40000 5   7   16   36
1.15                    -10.5996668832191             11.3247861934893          -6.24976277129193   1.95967353941551      -0.323643064432248   0.0265058326039759    -0.000853402350463369 -0.976029620700646   0.532176706081641    -0.101175369880685   0.00544917893714376   -1.04381743185931   -0.9366817928041     0.223516352927627     -0.0110386334867617   0.736226656128734   0.562865341713745     -0.141342014614834    0.00745095998121414   -0.148984649585496  -0.105293740340106    0.0270756021059489    -0.00150086050913901  0.3627 0.4449      0.57402   0.53873 0.44364 0.40000 5   7   16   36
1.2                     -10.4711600014558             11.0236324533154          -6.03972808875224   1.88583490824599      -0.309989579467039   0.0252501379864513    -0.000808195058109007 -1.00236749305719    0.579883491754862    -0.113539054077305   0.0063085849740398    -1.00492136850197   -1.01610279324741    0.244190709790422     -0.0124684754346065   0.716384600873665   0.60389884704211      -0.151847588625952    0.00816576242498054   -0.145612642670144  -0.112157674686296    0.0288104702257377    -0.00161733815764739  0.3625 0.4434      0.57267   0.53834 0.44170 0.40000 5   7   16   36
1.25                    -10.3298969702254             10.6934970460453          -5.81088153913234   1.80666225936089      -0.295584106656915   0.0239427030637572    -0.000761606727489232 -0.98793712353069    0.606742084080895    -0.122393153469522   0.00697446106803453   -1.00848193788592   -1.07278330650371    0.260868545133466     -0.0136697044308321   0.715852817367569   0.634083797281335     -0.160382142201527    0.00876522259843967   -0.145388612390956  -0.117191506544068    0.0302052861241307    -0.00171363919346135  0.3631 0.4419      0.57195   0.53797 0.43984 0.40000 5   7   16   36
1.3                     -10.1406356650276             10.3051737652815          -5.55214218439155   1.71955535143072      -0.280045204862122   0.0225528906312108    -0.000712638957316819 -1.01243141877513    0.657360522029477    -0.135664735525047   0.00789839194668981   -0.964734204256032  -1.15718596613008    0.282555712687307     -0.0151574646096099   0.693151935963091   0.67685802449159      -0.171149180558079    0.0094909203614451    -0.141581738052065  -0.124215979832083    0.03194852497549      -0.00182951376829913  0.3610 0.4404      0.56949   0.53761 0.43806 0.40000 5   7   16   36
1.35                    -10.0044674108434             9.98232890634769          -5.32906884771148   1.6431747095429       -0.266308367857645   0.0213196852722128    -0.0006691258902935   -0.988346647102774   0.682628400902816    -0.144623015987189   0.00858339259073004   -0.970484036905003  -1.2152959284109     0.299712938207949     -0.0163923377483386   0.691958507160339   0.708003091351568     -0.179898176964676    0.010103861377306     -0.14110732036604   -0.129418461011104    0.0333745904791001    -0.00192770169283752  0.3605 0.4391      0.56808   0.53727 0.43634 0.40000 5   7   16   36
1.4                     -9.82541797813155             9.59375148390327          -5.06780492826745   1.55595279486593      -0.25094934642208    0.0199631617906109    -0.000621850141057481 -0.937671361816131   0.692229582245098    -0.150658105122114   0.00909446973560815   -1.0112231583015    -1.25081101733753    0.312384678341117     -0.0173483762891706   0.709473646007406   0.726502274773446     -0.186074894030046    0.0105545651704661    -0.144037881333165  -0.132267390037637    0.0343163747106773    -0.00199515102530114  0.3616 0.4377      0.56779   0.53694 0.43469 0.40000 5   7   16   36
1.45                    -9.62570392325247             9.18340573304939          -4.79681087892602   1.46659971207112      -0.235364217151906   0.018597124447418     -0.000574544776043547 -0.915365504478906   0.719094987500349    -0.159928088480499   0.00979696628834983   -1.01320233900192   -1.30935047924196    0.329337654494102     -0.0185579880501209   0.707132735201145   0.756675087376326     -0.194427845773429    0.0111352527427932    -0.143520737776439  -0.137138073341157    0.0356369042477522    -0.00208541616019695  0.3604 0.4365      0.56604   0.53662 0.43309 0.40000 5   7   16   36
1.5                     -9.4737787709752              8.83703190613569          -4.56221867322137   1.38834376392943      -0.221639486658975   0.0173915016812402    -0.000532778201395055 -0.861073830368127   0.72814100069366     -0.166003955729302   0.0103146994585301    -1.04960121070615   -1.34764943472082    0.342519870171324     -0.0195429847375102   0.720426327165207   0.777316175254414     -0.200970586280394    0.0116068090174777    -0.145510097971084  -0.140447893885788    0.0366580183183167    -0.00215752981022645  0.3559 0.4352      0.56223   0.53631 0.43155 0.40000 5   7   16   36
1.55                    -9.26153447813307             8.39605068100939          -4.27162039816119   1.29374855900347      -0.205382663095731   0.0159853711051186    -0.000484613591152749 -0.767913619318261   0.714346877666342    -0.167864520497544   0.0105849838915872    -1.13632095719484   -1.35453501821047    0.349661327979924     -0.0201630132372984   0.759691808552281   0.781208768085315     -0.204231913293525    0.0118778481871157    -0.152089196952073  -0.140743682215715    0.0370825010414879    -0.00219297170959519  0.3507 0.4340      0.55804   0.53601 0.43005 0.40000 5   7   16   36
1.6                     -9.05851072096611             7.97567312353             -3.9955227228332    1.20413553885586      -0.19002840562686    0.0146615982361225    -0.00043942479440381  -0.691066704629644   0.70921197047907     -0.171203606831826   0.0109371191592315    -1.20153645803317   -1.37240468779789    0.358644157159101     -0.0208843790736622   0.788160485580947   0.79055987352034      -0.208415739247714    0.0122005187797824    -0.156809288864391  -0.141984492071354    0.037669225241684     -0.00223767232063367  0.3479 0.4329      0.55537   0.53572 0.42861 0.40000 5   7   16   36
1.65                    -8.89844698622059             7.62195428117081          -3.75982165005415   1.1270352860673       -0.17676308265399    0.0135161253228218    -0.000400330078267673 -0.616703783207617   0.705038891315985    -0.174638880422691   0.0112907482272742    -1.2620393592016    -1.39172248968277    0.367719465814051     -0.0216034470788187   0.813894722382042   0.800713587287138     -0.212654442821511    0.0125227288473332    -0.161013531845013  -0.143382312704113    0.0382693484451196    -0.00228260386583231  0.3486 0.4318      0.55495   0.53544 0.42721 0.40000 5   7   16   36
1.7                     -8.67680668566223             7.18445616675929          -3.47759210228721   1.03670560591637      -0.16146682933655    0.0122104862781298    -0.000356145913776091 -0.559570196757657   0.710553828746694    -0.179814664753909   0.0117450585958479    -1.30211212623472   -1.42175287404378    0.378640196082355     -0.0224268415783093   0.830165516994718   0.81560717649938      -0.217692362929572    0.0128899093391268    -0.163702246727297  -0.145516640809886    0.0389921194981132    -0.00233444926006369  0.3485 0.4307      0.55401   0.53517 0.42585 0.40000 5   7   16   36
1.75                    -8.51141815544558             6.82302835621648          -3.23852582790994   0.959389707233223     -0.148322376473567   0.0110877819546169    -0.000318191979695957 -0.457501557070368   0.688116034785965    -0.17965006224647    0.0118783875802084    -1.39837795777262   -1.41558879806607    0.382492230293689     -0.0228187426283663   0.874376956683492   0.812193147800924     -0.219121894546048    0.0130355332570213    -0.171171056609032  -0.14449078020034     0.0390890042627171    -0.00234774756452608  0.3470 0.4296      0.55226   0.53491 0.42453 0.40000 5   7   16   36
1.8                     -8.34133920588152             6.45852549350323          -2.99919397982659   0.882444234358746     -0.135306784805842   0.0099809365922724    -0.000280918573697726 -0.364100913376475   0.670900056856481    -0.180474552796602   0.0120713588827116    -1.48134758814962   -1.4170423208057     0.387770037102468     -0.0232972224936267   0.911220545543225   0.812989596093029     -0.221351778116763    0.0132307776500888    -0.177283437756483  -0.144248457338668    0.039337240653521     -0.00237058003212761  0.3469 0.4286      0.55137   0.53465 0.42325 0.40000 5   7   16   36
1.85                    -8.14519651112743             6.06075805865252          -2.74224650855741   0.800756670467048     -0.121606552921551   0.00882347880876224   -0.000242138256366769 -0.275954514509124   0.655240422307202    -0.181396430241757   0.0122620452340837    -1.56058192197539   -1.41795843113       0.392611309355905     -0.0237358787400051   0.947240478368475   0.812723302375372     -0.223199130176998    0.0133958388030696    -0.183384719747888  -0.143733022220077    0.0395017391668329    -0.00238712713262943  0.3448 0.4276      0.54931   0.53440 0.42200 0.40000 5   7   16   36
1.9                     -7.97124758704099             5.69620992209811          -2.5052713146759    0.725341433122381     -0.108975162864618   0.00775902714075244   -0.000206586023119228 -0.178592545467325   0.633186934451514    -0.18104022757459    0.0123744716910127    -1.65031768796876   -1.41096530314658    0.395827351770464     -0.0240740784406199   0.988142563917008   0.808653687774904     -0.22426494802376     0.0135130311409373    -0.19027426575997   -0.142595139968713    0.0395390912937999    -0.00239597775737837  0.3435 0.4266      0.54775   0.53416 0.42079 0.40000 5   7   16   36
1.95                    -7.78097222592561             5.31698219980537          -2.26238483543988   0.64867608338019      -0.0962045474030312  0.00668725115054628   -0.000170908890014925 -0.105964397714876   0.623702942473287    -0.182765656531561   0.0125994468705966    -1.71245851706878   -1.41738127398815    0.401192560757398     -0.0245258565073087   1.01644394014385    0.810581256254734     -0.226283788503121    0.0136806055756208    -0.195138262582421  -0.142413936763764    0.0397286980637383    -0.00241294482906249  0.3422 0.4257      0.54621   0.53392 0.41961 0.40000 5   7   16   36
2                       -7.63733162098538             4.99801117088423          -2.05309309661384   0.58210468724519      -0.0851055993705734  0.00575793944695678   -0.000140092896180025 0.0126176476643789   0.586955326121367    -0.17945196159675    0.0125290269052532    -1.82927671576141   -1.39043192582513    0.400283772297668     -0.0246055019352268   1.0709055505018     0.796304076984311     -0.225231451194084    0.0136651233268865    -0.20435199361645   -0.139514796112484    0.0394007549034344    -0.00239896837871009  0.3412 0.4248      0.54482   0.53369 0.41845 0.40000 5   7   16   36
2.5                     -6.34076041424126             2.20706331054104          -0.256027680356774  0.0209878296399781    0.00668511113095843  -0.00178135283670364  0.000105155633996792  1.00468155277897     0.264371905419115    -0.147027557433571   0.0115850288231401    -2.80774301003265   -1.12961026763773    0.382106376134206     -0.024495525873893    1.53332977237135    0.650254502429722     -0.209153166736802    0.0130285047986366    -0.283470549005822  -0.109683078676995    0.035205154606004     -0.00218405840536602  0.3321 0.4166      0.53278   0.53166 0.40830 0.40000 5   7   16   36
3                       -5.19881810758103             0.0805611767873749        1.03380322437094    -0.365599090888046    0.0676057081271332   -0.00659513014879632  0.000255384341388546  0.978216852763983    0.408482090979854    -0.186042749675011   0.0142725167680487    -2.52896850900443   -1.44554136656756    0.454220011621278     -0.0291396501110674   1.36226298240441    0.804887284218975     -0.242422328245217    0.0151214859440765    -0.252253166895619  -0.134082517717846    0.0402504839869038    -0.00249770805147611  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
3.5                     -5.03755251912391             -0.533411321833903        1.4188789959246     -0.475461625413779    0.0830240302869239   -0.00759487336603594  0.000277873210707869  1.3221347488462      0.274561150485705    -0.16804049715716    0.0134426219462124    -2.83634792151053   -1.3227506954679     0.436527081484602     -0.0282490025778086   1.5101966816867     0.72452909153941      -0.22802852954819     0.0142951147714763    -0.278235196616685  -0.116985079537634    0.0369400537421851    -0.002301841747166    0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
4                       -4.33459588509111             -1.78157284204592         2.13617652163101    -0.678495613417229    0.113004156025965    -0.00978941999517922  0.000340394402687915  1.36784367984988     0.275602490149976    -0.170816045408284   0.013686990491395     -2.86617083343281   -1.30736230471252    0.433394296138174     -0.0280438209376325   1.54319459626424    0.685878253132472     -0.219025736558668    0.0137150030662258    -0.287001620092744  -0.105961802173789    0.0343952422233304    -0.00214012518703773  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
5                       -5.19006862735358             -0.90258810565653         1.60488132393033    -0.498527104622468    0.0787807195820308   -0.00644150854130464  0.000210961873353466  2.02588111491438     -0.177708011965829   -0.0798003664973344  0.00807190186958431   -3.63245878073079   -0.718196342934959   0.310628771048694     -0.0203710257763568   1.930928227644      0.360614004260065     -0.15028538305789     0.0094412242953802    -0.35442947461898   -0.0464666756636188   0.0217832846986561    -0.00136178631123472  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
6                       -6.08314902302583             0.30973723301837          0.824395463544861   -0.241184511501711    0.0328175840951564   -0.00223900749106477  5.7864460930144e-05   1.81955044542258     -0.124727134962408   -0.083294709408048   0.0080937386363158    -3.31405505636647   -0.790522038081264   0.315674587257284     -0.0205335400664175   1.76293371705925    0.388703014270462     -0.151593523643157    0.00953105858789937   -0.32450159783943   -0.0507479954478801   0.0220032949246621    -0.00138906335640091  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
7                       -7.61905903020313             2.51726511081928          -0.53525824897565   0.186581508722994     -0.0399364660623586  0.00410169272745143   -0.000163247487123048 1.77978537334872     -0.209258101629239   -0.0569517047007813  0.00619067980113411   -3.25289454285404   -0.66831233496147    0.279632857730468     -0.0180650819635908   1.72858737687259    0.322292426599986     -0.133246341388285    0.00835924218611923   -0.317959819513381  -0.0393006431025473   0.0189756890607149    -0.00120646035712587  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
8                       -8.36112404682273             3.55856048777613          -1.19165163746085   0.396994320950031     -0.0763445287065532  0.00731934629981129   -0.000276462589313594 1.71201536764756     -0.278052661459308   -0.0333659764999204  0.00443822238512465   -3.1691964632426    -0.565123534950947   0.247309866599232     -0.0157995156681267   1.68328734123097    0.269052720482077     -0.117584919033023    0.00733096314073971   -0.309694480512774  -0.03045665360681     0.0164743906214399    -0.00105052454446912  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
10                      -9.17989619745715             4.20221145773307          -1.51629856201014   0.49830440089873      -0.0947967424221091  0.00904074598062506   -0.00033883529314331  3.40367123595467     -1.38456497146438    0.179501143627895    -0.00824080808115213  -5.27536565336961   0.858449863392392    -0.0286624327581027   0.000631169325975492  2.71178387228714    -0.437897146604191    0.0190964707092874    -0.000740423263877386 -0.48351810742625   0.0896778501059212    -0.0066743156842739   0.000309164016148211  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
""")

COEFFS_FORELAND_120Bars = CoeffsTable(sa_damping=5, table="""\
IMT                     a1                            a2                        a3                  a4                    a5                   a6                    a7                    a8                   a9                   a10                  a11                   a12                 a13                  a14                   a15                   a16                 a17                   a18                   a19                   a20                 a21                   a22                   a23                   tau    mean_phi_ss sigma_tot phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv                     -7.82108412777637             6.58780944333771          -3.55529537223548   1.25607833984171      -0.236374820444362   0.0217596471124324    -0.000775905888199971 1.61006701452372     -0.672434476960728   0.0871178653427606   -0.00438283969087223  -2.78817933944484   -0.326044784410847   0.151150747929228     -0.00805202716636021  1.25478075912093    0.409068916159769     -0.132172954991963    0.00765609359435457   -0.215263916765026  -0.0909480113722333   0.0282036245109193    -0.00172801293183309  0.5010 0.4600      0.68015   0.00000 0.00000 0.00000 5   7   16   36
pga                     -5.46305601887405             6.37976847368286          -3.6655070637383    1.34567425471441      -0.260015842307388   0.0243971709163212    -0.000883488711581384 1.39604643825301     -0.645934131789525   0.0936279202259357   -0.00528140599504769  -2.43653162527586   -0.462962566585634   0.170774407790943     -0.00903957500733608  1.03378678364698    0.508302850930722     -0.151352791979848    0.00892016187454288   -0.185187685425041  -0.107855746132408    0.0319696806715751    -0.00200528749650398  0.3532 0.4600      0.57998   0.58000 0.47000 0.35000 5   7   16   36
0.01                    -5.61503158657718             6.80567205474103          -3.96595035143808   1.44251395846701      -0.276165918043518   0.0257591108397444    -0.000929563114638748 0.715496358040355    -0.22243756302465    0.0134112370467816   -0.000528003255001875 -1.50489421786965   -1.05368689988118    0.284053999143034     -0.0158306112140465   0.532515437056612   0.825698434870789     -0.212704100656028    0.0126338768166469    -0.0946353000612041 -0.165090961886225    0.0430989925165671    -0.00268376091200671  0.3529 0.4600      0.57975   0.58000 0.47000 0.35000 5   7   16   36
0.02                    -5.32759842002676             6.23015029987828          -3.5563947683976    1.31468260122619      -0.255671087699252   0.0240944364938827    -0.000875258945307426 2.46356805362691     -1.04897345873506    0.144027612262378    -0.00740858172473647  -3.42544192955981   -0.121205183694483   0.133464649556283     -0.00783717194015989  1.31395181419667    0.411828975262321     -0.144551316939171    0.00904643911540908   -0.206210492965111  -0.100142158293149    0.0322408042435184    -0.00211847977137574  0.3674 0.4570      0.58639   0.56796 0.46097 0.37408 5   7   16   36
0.03                    -4.23930559467295             5.69446601623706          -3.3934067053107    1.27998001368622      -0.251220416857527   0.023812613424614     -0.000868926642882698 -0.245803767050094   -0.0962067263711608  0.0364553559367688   -0.00362636830900183  -0.0474477385952595 -1.24084809878906    0.259664055564546     -0.0126168796421682   -0.331674621080146  0.928910576815865     -0.203404390787896    0.01142424933701      0.0619914815431394  -0.181367914001532    0.041593349087784     -0.00251423557391327  0.3749 0.4552      0.58975   0.56092 0.45569 0.38817 5   7   16   36
0.04                    -3.77172065575026             5.52354267497533          -3.34630390724349   1.26930169520085      -0.249890011222291   0.0237470478606811    -0.000868397805795522 -1.28794962549871    0.105554181267901    0.0392261978679991   -0.00514899659177825  0.879450555782266   -1.31049270339245    0.233175649938338     -0.00983855109982238  -0.579423776458943  0.883824788899321     -0.181170663684972    0.00968519929215823   0.0698187475598715  -0.163162089712883    0.0367176006507311    -0.00217639192011972  0.3801 0.4540      0.59211   0.55592 0.45194 0.39816 5   7   16   36
0.05                    -4.05971033631025             5.67728514766251          -3.36976849866913   1.27012959675255      -0.24982498402102    0.0237468055188371    -0.000868668547538897 -0.10392939657201    -0.457062987588088   0.126961169406587    -0.00966691624262743  -0.821169885746515  -0.526262717022407   0.111383346418213     -0.00343372158705763  0.400796456369386   0.463252154707907     -0.117126034689796    0.00627964009570364   -0.116566813841817  -0.0889998667717604   0.0256720896252912    -0.00158661585925375  0.3855 0.4530      0.59483   0.55204 0.44903 0.40592 5   7   16   36
0.1                     -5.23435642363691             6.00153532126982          -3.37389209332543   1.25954818293644      -0.24725967820829    0.0234690638684192    -0.000857587132615502 2.04245205083677     -0.58237112347319    0.0337733066272212   1.90478425080144e-05  -2.83926969193266   -0.742275430587546   0.275096796343917     -0.0170912472287072   1.19097619597299    0.730042416911766     -0.219288090692918    0.0137970493626134    -0.206723608684099  -0.161245723383809    0.0463584933643856    -0.00298365282286301  0.3864 0.4500      0.59312   0.54000 0.44000 0.43000 5   7   16   36
0.15                    -6.01109015259886             6.56775871666529          -3.62869099638965   1.32943608766994      -0.258045874431371   0.0243246237210552    -0.000884680458561567 2.42825994487785     -0.847621184389715   0.0893665455429173   -0.00351503199572914  -3.35116371160111   -0.270290045011733   0.163478074871225     -0.00954636683037515  1.45276360841991    0.450738854256739     -0.14958950959807     0.00898031251536733   -0.245442719457227  -0.106896064991742    0.0322476414034456    -0.00199785451324651  0.3841 0.4675      0.60507   0.58095 0.47510 0.40075 5   7   16   36
0.2                     -6.87046327153964             7.03647106124759          -3.78386806451413   1.36705131299011      -0.263723756247117   0.0247661060822132    -0.000897869929045684 4.05736914046069     -1.80585273267484    0.264279628068752    -0.0136172029348783   -5.41189074071448   1.03738273130729     -0.0857241058597748   0.00527957494998589   2.4770935808005     -0.226345365359992    -0.0177024931820339   0.00101476191349542   -0.416552668022757  0.0122221431019092    0.00868368691040801   -0.000561031106473563 0.3690 0.4800      0.60546   0.61000 0.50000 0.38000 5   7   16   36
0.25                    -7.36979086911063             7.52438705904224          -4.07938676466456   1.45895961614865      -0.27839713262359    0.0259227338540504    -0.000933648839966607 3.79427799897711     -1.71969480523779    0.255457557136828    -0.0133450357431809   -5.13798463942731   1.00657365994057     -0.0924708073701005   0.00609734653027997   2.35199344422253    -0.226379259164289    -0.0108701973333584   0.000378113759548766  -0.393613324703953  0.0136693562978294    0.00718789293139461   -0.000430674086466131 0.3445 0.4800      0.59082   0.62651 0.50000 0.37450 5   7   16   36
0.3                     -7.9383368824154              8.18437436359239          -4.48531337967515   1.58291272967652      -0.297838455012268   0.0274348064761593    -0.000979974768426847 3.52632875291349     -1.62888754572165    0.245725161883775    -0.0130255671630182   -4.88172652213434   0.969708130559451    -0.0967248524263416   0.00673113182175303   2.24256154289419    -0.222932565952585    -0.00572169515227991  -0.000130211652546499 -0.37489873492838   0.0143651987125353    0.00605704618727459   -0.000327932388159262 0.3377 0.4800      0.58688   0.64000 0.50000 0.37000 5   7   16   36
0.35                    -8.54589635962017             8.95616795501717          -4.95992047080242   1.72507671947761      -0.319696952605137   0.0291041725312415    -0.00103026838880868  3.22069395318168     -1.51654923745726    0.232183175784117    -0.0124924988358413   -4.60423045025364   0.90923589792216     -0.0956933861546076   0.00702651909920466   2.12924492514578    -0.208339126687349    -0.00336703141837205  -0.000451330654804195 -0.356646075245083  0.0131354653604596    0.00544021156581811   -0.000260288003396449 0.3482 0.4740      0.58812   0.62793 0.49396 0.37000 5   7   16   36
0.4                     -9.0260777351857              9.60350752186793          -5.36672609144529   1.84491947316252      -0.337485253374236   0.0303996993583984    -0.00106708809726301  2.79270322474243     -1.33766288587302    0.207092295033887    -0.0113138745097247   -4.21045097180715   0.775758125713675    -0.0809329743971403   0.00651090790523676   1.9675492543541     -0.160083239816044    -0.00762919682055734  -0.000370353230253016 -0.331384020604332  0.00640021546130578   0.00593814170566711   -0.000261572311803188 0.3552 0.4687      0.58811   0.61747 0.48874 0.37000 5   7   16   36
0.45                    -9.56518690565607             10.3417937669086          -5.81597191296802   1.97412368431706      -0.356346560738985   0.0317562195823614    -0.00110522828853533  2.46543387235537     -1.21675310200791    0.192557566959236    -0.0107496718961084   -3.95691235602084   0.715798634267908    -0.0789224729329133   0.00671779938529786   1.87821531493086    -0.148477922663003    -0.00561674119332846  -0.000643479984988459 -0.31891093381763   0.00594232965743154   0.00536343414275947   -0.000202137223991733 0.3495 0.4641      0.58097   0.60825 0.48413 0.37000 5   7   16   36
0.5                     -9.95864613458884             10.925236098833           -6.17877306702197   2.07618568591319      -0.370551948668022   0.0327081921126254    -0.00112946116461807  1.97840467254434     -1.00286020642738    0.161039249375757    -0.00919725015666258  -3.52258990282617   0.541562837748668    -0.0550297902702148   0.00560996788468249   1.70393711159237    -0.0808919001312076   -0.0145544060420494   -0.000248925261735316 -0.292558650917327  -0.00398439094868859  0.00666263706769449   -0.000257919456007378 0.3548 0.4600      0.58095   0.60000 0.48000 0.37000 5   7   16   36
0.55                    -10.3268873117465             11.4684064969983          -6.50855708079084   2.16613395459796      -0.382576810044645   0.033469644985735     -0.0011472082191168   1.55638565998801     -0.821828859004111   0.13496703811259     -0.00794007603007211  -3.16867102495641   0.402431122532542    -0.0363370785771906   0.00476002251771326   1.5684809817868     -0.0292121137173321   -0.021254271033327    3.97307511957116e-05  -0.272760238885077  -0.0113558380297979   0.00761146000108545   -0.000297530243739165 0.3639 0.4586      0.58544   0.59175 0.47587 0.37413 5   7   16   36
0.6                     -10.5386567074829             11.7925877667047          -6.70583445150174   2.21512619693009      -0.387856994573823   0.0336676377415193    -0.00114617011781618  1.11383540908813     -0.623693609885816   0.10522180620207     -0.00644959736653399  -2.78760900632387   0.236297271351819    -0.0115768755638276   0.00351723596935788   1.41949379260245    0.0365392310834388    -0.0311295291640029   0.000534026552248218  -0.250786815169219  -0.0211159722480468   0.00909797304277345   -0.000371842991688479 0.3757 0.4574      0.59187   0.58422 0.47211 0.37789 5   7   16   36
0.65                    -10.7977543816784             12.1761218634905          -6.93023540118539   2.2709033180456       -0.394153851934876   0.0339508029857737    -0.00114814744992083  0.713958151682896    -0.440093871203797   0.0769495211219503   -0.00499813244317358  -2.43978106108351   0.0728733655832096   0.0145483767468982    0.00212231489536717   1.28124702383215    0.104716620405462     -0.042412596998002    0.0011483704860428    -0.230082583882146  -0.0316600951780937   0.0108928109371452    -0.000470718740554439 0.3831 0.4562      0.59573   0.57729 0.46864 0.38136 5   7   16   36
0.7                     -10.9442578175081             12.3832477598744          -7.04489629147229   2.29297921999345      -0.394997526605595   0.0337874948990217    -0.00113552964221433  0.348077953073497    -0.266543970057777   0.049422005165832    -0.00354818837027291  -2.11800052207572   -0.0896613613214313  0.0420294809336559    0.000592622196390192  1.15212888356918    0.17457962909917      -0.0547238653946255   0.00184846349173975   -0.210649881941412  -0.042633189164997    0.012880071765047     -0.000584812124687301 0.3886 0.4551      0.59846   0.57087 0.46544 0.38456 5   7   16   36
0.75                    -11.0342393630186             12.4933868330157          -7.09675859214756   2.29565550577421      -0.392737410753768   0.0333751546957229    -0.00111497144102094  0.0208486331080156   -0.104997101389659   0.0229169353485063   -0.00211310166545088  -1.82371240503467   -0.25061135718535    0.0707668936542926    -0.00106744568294411  1.03161489428894    0.246356061963988     -0.0681136981417945   0.00263817674698665   -0.192243510623825  -0.0541505147920947   0.0150833122602371    -0.000715771875302009 0.3852 0.4542      0.59554   0.56490 0.46245 0.38755 5   7   16   36
0.8                     -11.0283609954174             12.4530783643401          -7.05638322892501   2.27042037567094      -0.386017136759923   0.032601400260595     -0.00108269947351097  -0.295528635260248   0.0610223844936844   -0.0056106226067576  -0.000514441945316484 -1.52188730500466   -0.430267180547148   0.104472388024696     -0.00307442177795325  0.902517004934203   0.329939439016261     -0.0843898363197723   0.00362304737361223   -0.171956514026006  -0.0678790978777529   0.0178074313568241    -0.000881426041591907 0.3850 0.4532      0.59468   0.55932 0.45966 0.39034 5   7   16   36
0.85                    -11.0530505024765             12.4351468210284          -7.0233337261424    2.24734815704746      -0.379789852226325   0.0318860361533625    -0.00105302315536741  -0.526545330203866   0.186923092762775    -0.0278517241444736  0.000757707010935218  -1.30686033335007   -0.570609393118648   0.13212977627573      -0.00476925650380081  0.811996788475724   0.395895545787195     -0.0979068354298833   0.00446265426576649   -0.157878413803706  -0.0787203207647906   0.0200683066520129    -0.00102212944516256  0.3811 0.4523      0.59150   0.55407 0.45703 0.39297 5   7   16   36
0.9                     -11.0209777077715             12.3179236961835          -6.92731463194769   2.20517007335961      -0.370534692467821   0.0309278737442096    -0.00101554577654248  -0.712943111958171   0.29480576063783     -0.0476477202540875  0.0019180279815029    -1.13124178175751   -0.696979139195334   0.158105224906996     -0.00639445505651455  0.737669273729338   0.45616379701478      -0.110723478469777    0.00527165942968862   -0.146305783376529  -0.0886513875247947   0.0222084096317507    -0.00115707143286822  0.3799 0.4515      0.59006   0.54912 0.45456 0.39544 5   7   16   36
0.95                    -10.9464267219844             12.1295018943973          -6.78786763002204   2.15037499609694      -0.359380220041359   0.0298265271819402    -0.000973793455196985 -0.88276986685462    0.402260510472055    -0.0683919088517012  0.0031727557237712    -0.957923468539714  -0.832906687909906   0.187012182392004     -0.00823501969564946  0.660335464205794   0.523066856389299     -0.125284385460648    0.00620253133457355   -0.133849046667039  -0.0998658895673857   0.0246662352751358    -0.0013137470778687   0.3733 0.4507      0.58525   0.54444 0.45222 0.39778 5   7   16   36
1                       -10.8214818792221             11.8581677002732          -6.59831865071614   2.08106849958286      -0.346037793687804   0.0285592027105915    -0.000927029038579491 -1.02227221441396    0.497957282862096    -0.0876031760629167  0.0043602824068438    -0.809920312562812  -0.960118570694354   0.214888891814331     -0.0100334304352021   0.593097869736397   0.586405084052865     -0.139391346950035    0.00711262715110156   -0.122928072451635  -0.110502072660902    0.0270422467425792    -0.00146627042817624  0.3716 0.4500      0.58362   0.54000 0.45000 0.40000 5   7   16   36
1.05                    -10.6918787977418             11.565665010944           -6.39322816661866   2.00722649235081      -0.33206095614274    0.0272496938697718    -0.000879208855247877 -1.11215395607872    0.571301149083291    -0.103417014691132   0.00537446309645581   -0.713104298058488  -1.06371053321777    0.239007380640763     -0.0116290981825821   0.549437049568829   0.638197883459556     -0.151547940110189    0.0079119246208733    -0.11593329506556   -0.119115677401       0.0290607147397219    -0.00159778210558673  0.3691 0.4482      0.58063   0.53956 0.44778 0.40000 5   7   16   36
1.1                     -10.5344631040505             11.2248933595441          -6.15915931968867   1.92532413437935      -0.316943811502127   0.025860725112098     -0.000829236276104719 -1.17535036056294    0.634373954575513    -0.117926649114448   0.00633420281886966   -0.637030161055716  -1.16057956182121    0.262460931213155     -0.0132070097068042   0.513196225051368   0.687811185180241     -0.163538145836646    0.00871070622530066   -0.109931460788875  -0.127461724213082    0.0310651256462376    -0.00172997332154377  0.3650 0.4465      0.57674   0.53913 0.44566 0.40000 5   7   16   36
1.15                    -10.3578554824267             10.8551243851647          -5.90875039411107   1.83914408577654      -0.301266653553519   0.0244368029276681    -0.000778465282611877 -1.23243767370573    0.697228710246215    -0.132718488484228   0.00732076992955173   -0.56321110755073   -1.25867217777541    0.286313271355808     -0.0148109262816413   0.477582123732723   0.737681746540201     -0.175588337625842    0.00951151028123307   -0.104056289926728  -0.135768907550905    0.0330563029715787    -0.00186084634450297  0.3627 0.4449      0.57402   0.53873 0.44364 0.40000 5   7   16   36
1.2                     -10.170413651794              10.4624996074814          -5.64423917827614   1.74926442617206      -0.285129416016798   0.0229873521360455    -0.000727251004156818 -1.25811461568467    0.745099361887236    -0.145115923264532   0.00818024159065323   -0.520844145495745  -1.34103941114386    0.307516825731536     -0.016267161759263    0.455994989932803   0.780108554325607     -0.186327446400958    0.0102369111488613    -0.100437978973704  -0.142825343692774    0.0348205427235649    -0.00197843337306827  0.3625 0.4434      0.57267   0.53834 0.44170 0.40000 5   7   16   36
1.25                    -9.9623580796963              10.0327809569743          -5.35746923924672   1.65326536790971      -0.268138546672601   0.0214791353955328    -0.000674455946498107 -1.25088340097017    0.776846430771102    -0.154885537333998   0.00889836462120288   -0.512683493052821  -1.40564190634308    0.32566848031728      -0.0175519605187645   0.44996524627418    0.813990639471148     -0.195545436577038    0.0108750099580636    -0.0993528252868118 -0.148436960061677    0.0363220688587211    -0.00208078876919746  0.3631 0.4419      0.57195   0.53797 0.43984 0.40000 5   7   16   36
1.3                     -9.75971758184275             9.61522495459481          -5.07968156824781   1.5606400886968       -0.251813732138701   0.0200360835252461    -0.000624144948921416 -1.25146358604996    0.814057878316227    -0.165715493930402   0.00967882781898262   -0.492068709379982  -1.47733938250753    0.34502235378036      -0.018900123183       0.437731317168167   0.850959973920948     -0.205230452960205    0.0115351383048813    -0.0972592293600082 -0.154498033165455    0.0378835259912743    -0.00218555482088805  0.3610 0.4404      0.56949   0.53761 0.43806 0.40000 5   7   16   36
1.35                    -9.549865089473               9.18220181208206          -4.79267502777029   1.46577185122033      -0.23524673442917    0.0185833972606713    -0.000573837569771203 -1.22288577282009    0.836253197806299    -0.174005108769602   0.0103187797097833    -0.503759198713624  -1.53117927359317    0.361186349426773     -0.0200645304131682   0.440727220911332   0.879070875422761     -0.213288323440831    0.012100116676226     -0.0976930200734648 -0.159043628245198    0.0391625725140865    -0.00227371419250707  0.3605 0.4391      0.56808   0.53727 0.43634 0.40000 5   7   16   36
1.4                     -9.3023574346843              8.69709622449341          -4.47679890493016   1.36290221099683      -0.217499416476063   0.0170419143824111    -0.000520846265196437 -1.19994043023079    0.862251119265334    -0.183021532039371   0.0110012872659895    -0.506237938653986  -1.58985817657808    0.378128897833751     -0.0212685756075622   0.439038564520211   0.909301368078459     -0.221646917198651    0.0126786612381074    -0.0973455492407947 -0.163907609368193    0.0404818930535322    -0.00236345779922509  0.3616 0.4377      0.56779   0.53694 0.43469 0.40000 5   7   16   36
1.45                    -9.12171876014685             8.29973419127236          -4.21074140647348   1.2752939905193       -0.202328084627779   0.0157240258126844    -0.000475620659474886 -1.11624854797047    0.853628025025388    -0.18578014694225    0.0113218313024322    -0.580123097854738  -1.60626663590022    0.3872293915266       -0.0220106400794604   0.471591855212932   0.918770092591146     -0.226102551239327    0.0130255661458575    -0.102714181649509  -0.165254253706554    0.0411350034793153    -0.0024135254041242   0.3604 0.4365      0.56604   0.53662 0.43309 0.40000 5   7   16   36
1.5                     -8.93387310412896             7.90099160101601          -3.94687294097301   1.18904375037112      -0.18747475179064    0.0144397904414648    -0.000431735369270989 -1.05916198418975    0.860147325066991    -0.191244776731606   0.0117967891117068    -0.620403200436249  -1.6410434535001     0.399534933133942     -0.0229330125485      0.48773301759656    0.937027629430708     -0.232083036763529    0.0134584334106012    -0.105325526231758  -0.168066097342408    0.0420427877923252    -0.00247805460708617  0.3559 0.4352      0.56223   0.53631 0.43155 0.40000 5   7   16   36
1.55                    -8.699667165782               7.44026255360113          -3.64926586732393   1.09345742392112      -0.171231850121539   0.0130494086109025    -0.000384580012935344 -1.01453786131876    0.873974047287983    -0.198040326970856   0.012348620068003     -0.644834764040408  -1.6841940322893     0.413262982463771     -0.0239336965008123   0.496230569322531   0.959061600148551     -0.238678488828967    0.013924289040812     -0.106676567973218  -0.171473452389383    0.0430444594781587    -0.00254751260787304  0.3507 0.4340      0.55804   0.53601 0.43005 0.40000 5   7   16   36
1.6                     -8.46800307371276             6.97408026542703          -3.34683307154962   0.996632953345753     -0.154867012777324   0.0116560660412046    -0.000337542026323212 -0.918542680836      0.857660724799135    -0.199289079792281   0.0125755989907706    -0.732496600929624  -1.68866493158909    0.419667277884114     -0.0244964917858624   0.535924431490741   0.961590539205556     -0.241534871518112    0.0141648224007309    -0.113335211720637  -0.171522692949486    0.0433982517691994    -0.00257778155266821  0.3479 0.4329      0.55537   0.53572 0.42861 0.40000 5   7   16   36
1.65                    -8.27681503998483             6.57415907403721          -3.08545565683227   0.912636722657945     -0.14065004696293    0.0104461837734904    -0.000296775180723221 -0.837171812135988   0.849204085108189    -0.201895192715655   0.0128786284979985    -0.800053937356205  -1.70357173014319    0.427854645350691     -0.0251595308526688   0.565188992794485   0.969494565844789     -0.245320451726165    0.014458698820588     -0.118165085329947  -0.172517936192796    0.0439178954981781    -0.00261775100773797  0.3486 0.4318      0.55495   0.53544 0.42721 0.40000 5   7   16   36
1.7                     -8.07264469739494             6.15586285583097          -2.81405142305268   0.826058859095587     -0.126095273889017   0.00921464485023245   -0.000255472612487971 -0.743196838882144   0.831995328612894    -0.202722477477169   0.0130703394058127    -0.885050506221859  -1.70540498626492    0.433285969819834     -0.0256464675069615   0.603789635061815   0.970222646877451     -0.247582319056052    0.0146550295746564    -0.12467420817612   -0.172213075442782    0.0441611933884593    -0.00264004737787467  0.3485 0.4307      0.55401   0.53517 0.42585 0.40000 5   7   16   36
1.75                    -7.8735062953086              5.74727634052983          -2.54963414387255   0.742062691670249     -0.112040089281601   0.00803088970408337   -0.000215949865749256 -0.644568406001319   0.811556936395885    -0.202902139860922   0.0132223479240853    -0.975229288998239  -1.70291829440658    0.437791862717422     -0.0260750276632222   0.644785810527101   0.968794793852484     -0.249379482643093    0.0148223318461392    -0.131572373903947  -0.171547152757583    0.0443271005245487    -0.00265757834956668  0.3470 0.4296      0.55226   0.53491 0.42453 0.40000 5   7   16   36
1.8                     -7.53298328450387             5.00531666888557          -2.05651788755436   0.586331843120737     -0.0863498985059123  0.00589128605859114   -0.000144818584696127 -0.0835270209526806  0.514895978190159    -0.151621782508216   0.0103254008192006    -1.57567425934735   -1.39367907754714    0.384899430246513     -0.0230879141006015   0.917349619062884   0.827343662055127     -0.224858828054411    0.0134164308703789    -0.175588556879552  -0.148300479853494    0.0402260519384291    -0.00241864322337134  0.3469 0.4286      0.55137   0.53465 0.42325 0.40000 5   7   16   36
1.85                    -7.50523206740926             4.98242502130605          -2.05484789715947   0.585411557112448     -0.0859477543120479  0.00584454503990258   -0.000143336223891129 -0.441002334528226   0.763775749750942    -0.201635327042626   0.0134178153302248    -1.16440398258644   -1.68669397364445    0.444078252499569     -0.0267488768651602   0.731447990384609   0.959794733048527     -0.251499911427441    0.0150591610454295    -0.146187271094099  -0.169134361548798    0.044402332287913     -0.00267583020541604  0.3448 0.4276      0.54931   0.53440 0.42200 0.40000 5   7   16   36
1.9                     -7.07441700074486             4.27366569971295          -1.63094412236263   0.456654590154273     -0.064961477045954   0.00410426206161063   -8.57784884237297e-05 -0.752428691903949   1.03395301264739     -0.261831004988108   0.0173630975984096    -0.555947851923846  -2.17693228419436    0.551181580011968     -0.0337367448193046   0.361644110694149   1.24788701462561      -0.313865888432716    0.0191213233764357    -0.0749817951125743 -0.223513094571942    0.0561142648800951    -0.00343814293209728  0.3435 0.4266      0.54775   0.53416 0.42079 0.40000 5   7   16   36
1.95                    -7.16522788846099             4.2705256029515           -1.595584789879     0.440862032349304     -0.0620447825378096  0.00385708690154718   -7.78428630115492e-05 -0.232121170186765   0.708099916477879    -0.198382891211514   0.0134767845408323    -1.36178771623279   -1.65744560426495    0.447086108870993     -0.0271987087320934   0.822596145337799   0.943750035913093     -0.251881478283204    0.0151793260521611    -0.161604291091119  -0.165490183811379    0.0441784969637336    -0.00267431672773818  0.3422 0.4257      0.54621   0.53392 0.41961 0.40000 5   7   16   36
2                       -7.0079931505112              3.93716867710999          -1.38070578013357   0.373493506391392     -0.0509630247886035  0.00294107859164479   -4.78422097545962e-05 -0.124478727078993   0.676733296016697    -0.195921223036338   0.0134502082195255    -1.46508679932605   -1.63730611141186    0.447272643980733     -0.0273354692929314   0.87050339661816    0.932863275844498     -0.251395903639591    0.0151948219409955    -0.169704550089488  -0.163182794867515    0.0439531793033656    -0.0026661809470883   0.3412 0.4248      0.54482   0.53369 0.41845 0.40000 5   7   16   36
2.5                     -5.82621479722566             1.38958820093316          0.246500931337168   -0.129371545070955    0.0302209239575327   -0.00362380074730232  0.000162040931653604  0.642481911474023    0.454023355049086    -0.177822572963111   0.0131650284827653    -2.09478323561626   -1.5515552808234     0.457791449427121     -0.0287513222165232   1.13671288806149    0.891444916433439     -0.253512559250878    0.015581293813405     -0.212376770332034  -0.153476798582158    0.0433688225357372    -0.00265991869328725  0.3321 0.4166      0.53278   0.53166 0.40830 0.40000 5   7   16   36
3                       -4.70255790535645             -0.808957857764994        1.59046897283591    -0.529398003904327    0.0925131297518338   -0.008483017091067    0.000311811184602821  1.29015821840389     0.204150875443061    -0.144726340275899   0.0116611184499397    -2.79342409102025   -1.2737230963841     0.417622631761675     -0.0266881455812356   1.49706332289238    0.716945027453688     -0.223600639925501    0.0138547620841807    -0.277733141086581  -0.117361240611394    0.0366939935037484    -0.00226008211565723  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
3.5                     -4.61120219296634             -1.24638384573882         1.84398279038206    -0.594217507284341    0.0999969074801344   -0.00878383601344104  0.000309925816876795  1.36451228527329     0.239682374402442    -0.159513697609013   0.0128254124925766    -2.71971173335204   -1.39694638715076    0.44869083286776      -0.0288246308169881   1.43584611546624    0.774240487247042     -0.236977324934418    0.0147781613897947    -0.265120254067172  -0.125986417993363    0.0386302431710204    -0.00239774614811543  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
4                       -5.1895125240869              -0.614808644856793        1.45827501799334    -0.466276177036       0.0763947904998414   -0.00653242009284598  0.000224285003482453  1.50067530497357     0.18737915835797     -0.152795763721773   0.0125422485157569    -2.8010363185561    -1.35806629286281    0.442966287085165     -0.0285751104463183   1.46856190428413    0.742512090224091     -0.230763331575983    0.0144505733108395    -0.270282611456638  -0.118715628469398    0.03712654209765      -0.00231724881326536  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
5                       -5.90519648397264             0.134720106022902         0.976464461004404   -0.296936748454183    0.0435870379627133   -0.00329650472952399  9.88173942869285e-05  1.89171704998266     -0.0958706181375064  -0.0949212225945394  0.00894992614739164   -3.24694207373974   -0.97631040086573    0.361331476267226     -0.02345191543908     1.69968100809684    0.520293211956097     -0.182804469434793    0.0114834704111128    -0.311374323544643  -0.0767409033654358   0.0280862328543025    -0.00176541603003237  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
6                       -6.98897061254476             1.5972556044291           0.0578641443793025  0.000902277732179408  -0.00884190739003891 0.00142993301055186   -7.10306864761604e-05 1.96343728277056     -0.232278838689973   -0.0604015690975674  0.00661547144885889   -3.32761122263503   -0.774918382312037   0.310968481206834     -0.0201500991929666   1.74369574530359    0.402674083775945     -0.154535032464378    0.0097179299179407    -0.319361538661958  -0.0549209763275891   0.022994697949069     -0.00145855015691884  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
7                       -7.92596286097706             2.3862848161355           -0.337861056141561  0.125325038353875     -0.0317608882861527  0.00360605050367611   -0.000152084187620671 4.00540462919935     -1.46438905612757    0.167427181884793    -0.00665539564210113  -5.82835543000224   0.786481853676889    0.018134978497695     -0.00299308144672029  2.96246724228088    -0.376157969498585    -0.00778771721514426  0.00113857878060546   -0.524523890990936  0.077926381201307     -0.00207348478278896  3.57902932672174e-06  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
8                       -8.03243030394125             3.03502289358228          -0.893423741715858  0.317369138988265     -0.0653018037420109  0.00654504617366308   -0.000254475681040024 1.60592187151642     -0.2080135325016     -0.0473590296364213  0.00530711721299164   -2.82006036141049   -0.809331623874063   0.297340057118074     -0.0189476893567513   1.46148072622812    0.428931943257202     -0.151528746369184    0.00953361564357632   -0.267110406632051  -0.0616692865715747   0.0232338869819482    -0.00149638108121167  0.3244 0.4100      0.52282   0.53000 0.40000 0.40000 5   7   16   36
10                      -9.55075544332661             4.68488735297049          -1.79819674847135   0.5896778543221       -0.110896618838083   0.0104697485755357    -0.000388568276808906 3.47288597906148     -1.42960378756977    0.188332387292502    -0.00878103114116427  -5.15288974629765   0.76527289132044     -0.00891343321622578  -0.000628797288646536 2.59902796847329    -0.351074818366381    -0.000150908980048818 0.000542117598010521  -0.458870382435115  0.0705262546171767    -0.00234717674570946  1.60431564598715e-05  0.3293 0.4100      0.52588   0.53000 0.40000 0.40000 5   7   16   36
""")
