# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2012-2019 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

"""
Module exports :class:`CauzziFaccioli2008SWISS01`
               :class:`CauzziFaccioli2008SWISS04`
               :class:`CauzziFaccioli2008SWISS08`.
"""
import numpy as np
# standard acceleration of gravity in m/s**2
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, SA
from openquake.hazardlib.gsim.base import CoeffsTable
from openquake.hazardlib.gsim.cauzzi_faccioli_2008 import CauzziFaccioli2008
from openquake.hazardlib.gsim.cauzzi_faccioli_2008_swiss_coeffs import (
    COEFFS_FS_ROCK_SWISS01,
    COEFFS_FS_ROCK_SWISS04,
    COEFFS_FS_ROCK_SWISS08)
from openquake.hazardlib.gsim.utils_swiss_gmpe import _apply_adjustments


class CauzziFaccioli2008SWISS01(CauzziFaccioli2008):
    """
    This class extends
    :class:`openquake.hazardlib.gsim.cauzzi_faccioli_2008.CauzziFaccioli2008`
    adjusted to be used for the Swiss Hazard Model [2014].
    This GMPE is valid for a fixed value of vs30=700m/s

    #. kappa value K-adjustments corresponding to model 01 - as prepared by
    Ben Edwards K-value for PGA were not provided but infered
    from SA[0.01s] the model considers a fixed value of vs30=700 to match the
    reference vs30=1100m/s

    #. small-magnitude correction

    #. single station sigma - inter-event magnitude/distance adjustment

    Disclaimer: these equations are modified to be used for the
    Swiss Seismic Hazard Model [2014].

    The use of these models is the soly responsability of the hazard modeler.

    Model implemented by laurentiu.danciu@gmail.com
    """

    #: Supported standard deviation type is only total
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = set([const.StdDev.TOTAL])

    #: Supported intensity measure types are spectral acceleration, peak
    #: ground acceleration and peak ground velocity.
    #: The original paper provides coefficients for PGA and PGV, while SA
    #: is obtained from displacement response spectrum values.
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = set([
        PGA,
        SA
    ])

    def get_mean_and_stddevs(self, sites, rup, dists, imt, stddev_types):
        """
        See :meth:`superclass method
        <.base.GroundShakingIntensityModel.get_mean_and_stddevs>`
        for spec of input and result values.
        """

        sites.vs30 = 700 * np.ones(len(sites.vs30))

        mean, stddevs = super().get_mean_and_stddevs(
            sites, rup, dists, imt, stddev_types)

        C = CauzziFaccioli2008SWISS01.COEFFS

        tau_ss = 'tau'
        log_phi_ss = np.log(10)
        mean, stddevs = _apply_adjustments(
            C, self.COEFFS_FS_ROCK[imt], tau_ss,
            mean, stddevs, sites, rup, dists.rhypo, imt, stddev_types,
            log_phi_ss)

        return mean, np.log(10 ** np.array(stddevs))

    COEFFS_FS_ROCK = COEFFS_FS_ROCK_SWISS01

    #: Coefficient table
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT       a1          a2         aN          aR          aS          a3          aB         aC         aD        phi      tau    sigma
    pga       -1.2960000  0.5560000  -0.0600000  0.0940000   -0.0130000  -1.5820000  0.2200000  0.3040000  0.3320000 0.2620   0.2230 0.3410000
    0.05      -2.8853436  0.5241770  -0.1002050  0.1323607   -0.0104223  -1.7131617  0.1590351  0.1913741  0.1662669 0.2715   0.2426 0.3569631
    0.10      -1.9083649  0.4879421  -0.0893446  0.1273364   -0.0141940  -1.7141846  0.2191124  0.2180647  0.1599583 0.2905   0.2382 0.3698820
    0.15      -1.9913830  0.5218353  -0.0733947  0.1189461   -0.0178601  -1.6011819  0.2516196  0.3149463  0.2929666 0.2898   0.2374 0.3701501
    0.20      -2.1855097  0.5575823  -0.0410869  0.1014822   -0.0242622  -1.5157781  0.2795653  0.4209394  0.3950845 0.2912   0.2372 0.3735782
    0.25      -2.2986747  0.5918757  -0.0505207  0.0853928   -0.0144371  -1.4772492  0.2468365  0.4352964  0.4414190 0.2936   0.2204 0.3658115
    0.30      -2.3197219  0.6141047  -0.0423759  0.0653264   -0.0096095  -1.4608963  0.2027381  0.4325227  0.4545156 0.2881   0.2121 0.3574604
    0.35      -2.3532043  0.6301540  -0.0290204  0.0569969   -0.0117476  -1.4526222  0.1948161  0.4572770  0.5023281 0.2910   0.2073 0.3576558
    0.40      -2.4625603  0.6520124  -0.0248118  0.0514677   -0.0113052  -1.4182980  0.1736606  0.4457811  0.5335752 0.2936   0.2024 0.3571774
    0.45      -2.4967480  0.6641638  -0.0098857  0.0366742   -0.0112005  -1.4072543  0.1769518  0.4470774  0.5839604 0.2949   0.1946 0.3542876
    0.50      -2.5409604  0.6736385  -0.0050996  0.0283795   -0.0095620  -1.3894703  0.1855959  0.4482634  0.6360733 0.2966   0.1992 0.3585923
    0.55      -2.5528057  0.6813095  -0.0013096  0.0256173   -0.0098734  -1.3833845  0.1834712  0.4509198  0.6792498 0.2979   0.1974 0.3587491
    0.60      -2.6105531  0.6942826  -0.0009457  0.0166886   -0.0062698  -1.3683149  0.1785156  0.4451175  0.6919865 0.2968   0.1949 0.3566331
    0.65      -2.5991259  0.7000182  0.0057530   0.0075781   -0.0050956  -1.3658287  0.1614781  0.4327439  0.6945308 0.2965   0.1925 0.3550096
    0.70      -2.6256918  0.7102002  0.0021694   0.0050372   -0.0026038  -1.3609069  0.1479847  0.4247098  0.6955686 0.2973   0.1912 0.3550060
    0.75      -2.6965075  0.7273575  0.0028068   0.0020510   -0.0016181  -1.3612589  0.1450193  0.4164749  0.7043074 0.2966   0.1882 0.3526839
    0.80      -2.7389002  0.7403566  0.0036660   -0.0040497  0.0005981   -1.3610415  0.1380784  0.4042236  0.7036039 0.2928   0.1886 0.3496514
    0.85      -2.7914083  0.7553379  0.0032023   -0.0059857  0.0016152   -1.3691674  0.1406749  0.4031042  0.7086573 0.2932   0.1894 0.3504153
    0.90      -2.7973391  0.7611568  0.0107582   -0.0055180  -0.0015391  -1.3755505  0.1451434  0.4082058  0.7090638 0.2966   0.1917 0.3545320
    0.95      -2.8305291  0.7686452  0.0163434   -0.0048616  -0.0039647  -1.3722514  0.1501412  0.4134206  0.7030495 0.2984   0.1891 0.3545101
    1.00      -2.8634011  0.7772102  0.0201847   -0.0026251  -0.0063816  -1.3692034  0.1464775  0.4066130  0.6863604 0.3000   0.1891 0.3558715
    1.05      -2.8680321  0.7801400  0.0181339   -0.0022563  -0.0057100  -1.3655567  0.1448087  0.4020379  0.6707605 0.2993   0.1900 0.3557457
    1.10      -2.8936464  0.7879813  0.0170920   -0.0027664  -0.0050829  -1.3658939  0.1389319  0.3901474  0.6569073 0.2988   0.1907 0.3558140
    1.15      -2.9355453  0.7992595  0.0171742   -0.0037400  -0.0046909  -1.3693257  0.1331251  0.3811657  0.6394404 0.2981   0.1891 0.3543397
    1.20      -2.9532117  0.8044803  0.0219944   -0.0055788  -0.0058288  -1.3657305  0.1279489  0.3723968  0.6233881 0.2962   0.1904 0.3533761
    1.25      -2.9782632  0.8090258  0.0250012   -0.0077949  -0.0061059  -1.3591759  0.1263850  0.3690539  0.6091518 0.2957   0.1919 0.3537186
    1.30      -3.0054272  0.8146208  0.0275714   -0.0091296  -0.0065852  -1.3550484  0.1235212  0.3666377  0.5942269 0.2949   0.1924 0.3533137
    1.35      -3.0314767  0.8208146  0.0284279   -0.0104723  -0.0063730  -1.3565895  0.1254993  0.3712946  0.5845589 0.2946   0.1922 0.3528877
    1.40      -3.0466187  0.8270614  0.0267230   -0.0095818  -0.0060518  -1.3607541  0.1224343  0.3655060  0.5695337 0.2948   0.1913 0.3525659
    1.45      -3.0654260  0.8330859  0.0250260   -0.0091609  -0.0055125  -1.3639165  0.1208328  0.3614019  0.5593142 0.2957   0.1884 0.3517607
    1.50      -3.0896821  0.8399120  0.0221389   -0.0092808  -0.0042695  -1.3666268  0.1165296  0.3568769  0.5511064 0.2960   0.1840 0.3496758
    1.55      -3.1171695  0.8476929  0.0174047   -0.0069108  -0.0033789  -1.3700564  0.1122110  0.3507933  0.5427573 0.2954   0.1802 0.3472015
    1.60      -3.1370747  0.8536100  0.0140424   -0.0050005  -0.0028569  -1.3707941  0.1080054  0.3419899  0.5315377 0.2947   0.1769 0.3449319
    1.65      -3.1433377  0.8570622  0.0092720   -0.0042635  -0.0013202  -1.3699352  0.1006163  0.3323793  0.5154721 0.2934   0.1743 0.3425114
    1.70      -3.1579402  0.8610034  0.0013194   -0.0032280  0.0012934   -1.3689420  0.0985499  0.3287989  0.5044447 0.2918   0.1718 0.3398077
    1.75      -3.1654876  0.8635322  -0.0079932  -0.0012780  0.0039846   -1.3692309  0.0977796  0.3266681  0.4968893 0.2911   0.1688 0.3376558
    1.80      -3.1627827  0.8650292  -0.0143883  0.0012671   0.0052926   -1.3701685  0.0950735  0.3202899  0.4885798 0.2902   0.1672 0.3359642
    1.85      -3.1615383  0.8667376  -0.0192120  0.0026373   0.0064626   -1.3707310  0.0931597  0.3135946  0.4817112 0.2898   0.1645 0.3342168
    1.90      -3.1522848  0.8676873  -0.0239616  0.0049096   0.0071912   -1.3712825  0.0870145  0.3043254  0.4719422 0.2900   0.1618 0.3329950
    1.95      -3.1431538  0.8680758  -0.0296569  0.0076970   0.0079890   -1.3715113  0.0829618  0.2976415  0.4651566 0.2902   0.1579 0.3310917
    2.00      -3.1421699  0.8685196  -0.0344938  0.0095700   0.0088897   -1.3693361  0.0834894  0.2950648  0.4603763 0.2901   0.1559 0.3298747
    2.05      -3.1474772  0.8700464  -0.0393413  0.0120142   0.0095059   -1.3678657  0.0844281  0.2941582  0.4593924 0.2908   0.1539 0.3293833
    2.10      -3.1614270  0.8732865  -0.0447820  0.0151246   0.0100114   -1.3671317  0.0841764  0.2931830  0.4587144 0.2912   0.1518 0.3285693
    2.15      -3.1760473  0.8759402  -0.0493498  0.0172018   0.0106142   -1.3650601  0.0852595  0.2939214  0.4593409 0.2917   0.1498 0.3278627
    2.20      -3.1872939  0.8778648  -0.0545726  0.0196852   0.0112416   -1.3620672  0.0857041  0.2932367  0.4571966 0.2914   0.1475 0.3263398
    2.25      -3.1929032  0.8782645  -0.0599400  0.0229501   0.0115343   -1.3562898  0.0843744  0.2908264  0.4528229 0.2909   0.1450 0.3244696
    2.30      -3.1964880  0.8785770  -0.0639389  0.0266334   0.0111443   -1.3495460  0.0795512  0.2845144  0.4457230 0.2902   0.1424 0.3223445
    2.35      -3.2061680  0.8796669  -0.0656158  0.0294290   0.0103948   -1.3421863  0.0763004  0.2788990  0.4404207 0.2892   0.1407 0.3206195
    2.40      -3.2124688  0.8800817  -0.0683920  0.0323025   0.0100710   -1.3351181  0.0754372  0.2733777  0.4367805 0.2879   0.1402 0.3189619
    2.45      -3.2173552  0.8802805  -0.0721574  0.0364129   0.0095688   -1.3273998  0.0721815  0.2667552  0.4308779 0.2868   0.1404 0.3177777
    2.50      -3.2208583  0.8799805  -0.0756910  0.0404754   0.0089382   -1.3195408  0.0695631  0.2620014  0.4254937 0.2858   0.1399 0.3163032
    2.55      -3.2244744  0.8798919  -0.0783250  0.0432722   0.0084324   -1.3134787  0.0692867  0.2610122  0.4237224 0.2848   0.1381 0.3143530
    2.60      -3.2355390  0.8810115  -0.0801680  0.0454326   0.0078271   -1.3074378  0.0701497  0.2596374  0.4210976 0.2841   0.1354 0.3123591
    2.65      -3.2599554  0.8841480  -0.0813756  0.0477987   0.0069265   -1.3014910  0.0719688  0.2594378  0.4205230 0.2835   0.1329 0.3105163
    2.70      -3.2834526  0.8875307  -0.0809548  0.0505024   0.0054163   -1.2962006  0.0727993  0.2577696  0.4195371 0.2827   0.1314 0.3091150
    2.75      -3.3028676  0.8902376  -0.0802845  0.0533437   0.0037658   -1.2914620  0.0739445  0.2565391  0.4206958 0.2823   0.1302 0.3081833
    2.80      -3.3164602  0.8923488  -0.0802537  0.0557539   0.0024698   -1.2879074  0.0743410  0.2546159  0.4215031 0.2821   0.1285 0.3071375
    2.85      -3.3328281  0.8947180  -0.0796644  0.0580173   0.0010041   -1.2839725  0.0757831  0.2532998  0.4205388 0.2817   0.1265 0.3058747
    2.90      -3.3508249  0.8973864  -0.0791587  0.0602144   -0.0003880  -1.2784672  0.0744210  0.2492444  0.4155354 0.2812   0.1250 0.3047832
    2.95      -3.3692426  0.8999885  -0.0781803  0.0620852   -0.0017386  -1.2727125  0.0735201  0.2458899  0.4118983 0.2809   0.1240 0.3039768
    3.00      -3.3883104  0.9025955  -0.0780050  0.0639634   -0.0027770  -1.2667687  0.0731364  0.2427131  0.4066463 0.2804   0.1234 0.3032581
    3.05      -3.4089144  0.9050712  -0.0777145  0.0652581   -0.0035732  -1.2604383  0.0736191  0.2410029  0.4021932 0.2800   0.1228 0.3026315
    3.10      -3.4302607  0.9080662  -0.0769700  0.0655633   -0.0040268  -1.2557977  0.0742839  0.2393353  0.3988910 0.2796   0.1221 0.3020134
    3.15      -3.4449030  0.9101579  -0.0762476  0.0656209   -0.0042581  -1.2520585  0.0755457  0.2377807  0.3968486 0.2787   0.1222 0.3013446
    3.20      -3.4566489  0.9116875  -0.0754699  0.0652424   -0.0043002  -1.2482556  0.0774894  0.2365001  0.3945458 0.2776   0.1222 0.3004223
    3.25      -3.4709224  0.9137012  -0.0750555  0.0646868   -0.0041063  -1.2445079  0.0789917  0.2354185  0.3926102 0.2761   0.1223 0.2991849
    3.30      -3.4861508  0.9163004  -0.0747234  0.0645185   -0.0040099  -1.2419588  0.0798637  0.2339357  0.3916166 0.2745   0.1228 0.2978846
    3.35      -3.5022077  0.9190528  -0.0728415  0.0636728   -0.0040702  -1.2394832  0.0810714  0.2330698  0.3913265 0.2725   0.1232 0.2964171
    3.40      -3.5203978  0.9217943  -0.0713307  0.0624610   -0.0038150  -1.2364777  0.0832514  0.2332679  0.3918926 0.2707   0.1239 0.2952740
    3.45      -3.5408118  0.9246496  -0.0705947  0.0611375   -0.0032988  -1.2327771  0.0853155  0.2337748  0.3915468 0.2692   0.1243 0.2941983
    3.50      -3.5601771  0.9273972  -0.0702652  0.0597769   -0.0026400  -1.2292914  0.0870118  0.2338685  0.3903120 0.2678   0.1247 0.2931537
    3.55      -3.5854744  0.9307797  -0.0701598  0.0587242   -0.0020068  -1.2249578  0.0889839  0.2347028  0.3893576 0.2665   0.1254 0.2924559
    3.60      -3.6138749  0.9347379  -0.0696739  0.0574896   -0.0013891  -1.2206922  0.0907968  0.2354381  0.3877713 0.2657   0.1266 0.2923746
    3.65      -3.6416871  0.9388613  -0.0692068  0.0565333   -0.0009180  -1.2172276  0.0919148  0.2355807  0.3855391 0.2651   0.1278 0.2924609
    3.70      -3.6667806  0.9426348  -0.0691106  0.0560806   -0.0005845  -1.2141418  0.0924525  0.2361611  0.3830481 0.2646   0.1288 0.2925229
    3.75      -3.6928280  0.9462948  -0.0696536  0.0557950   -0.0001683  -1.2107893  0.0938360  0.2378644  0.3816221 0.2642   0.1295 0.2925011
    3.80      -3.7192534  0.9500416  -0.0701051  0.0554917   0.0002090   -1.2073914  0.0949498  0.2392272  0.3789399 0.2637   0.1300 0.2923401
    3.85      -3.7461033  0.9539773  -0.0700016  0.0545840   0.0006554   -1.2041114  0.0956039  0.2397455  0.3759065 0.2631   0.1302 0.2919152
    3.90      -3.7715185  0.9577344  -0.0694757  0.0537659   0.0009270   -1.2004853  0.0951864  0.2388567  0.3723133 0.2625   0.1306 0.2916133
    3.95      -3.7952464  0.9613004  -0.0691349  0.0532504   0.0011510   -1.1969673  0.0942025  0.2372400  0.3688114 0.2619   0.1313 0.2914914
    4.00      -3.8152625  0.9646448  -0.0689546  0.0527977   0.0013845   -1.1946064  0.0928633  0.2348112  0.3656192 0.2612   0.1320 0.2911708
    4.05      -3.8340273  0.9678713  -0.0685971  0.0522826   0.0015856   -1.1923050  0.0911828  0.2316196  0.3621850 0.2605   0.1326 0.2908003
    4.10      -3.8534471  0.9710945  -0.0682666  0.0512167   0.0020157   -1.1895800  0.0897472  0.2282775  0.3595744 0.2597   0.1328 0.2902313
    4.15      -3.8717479  0.9740822  -0.0672992  0.0497144   0.0024451   -1.1867849  0.0882628  0.2250725  0.3574817 0.2589   0.1330 0.2897667
    4.20      -3.8885736  0.9768598  -0.0662365  0.0482856   0.0028051   -1.1838169  0.0862260  0.2215819  0.3542480 0.2583   0.1334 0.2894828
    4.25      -3.9059444  0.9798431  -0.0649702  0.0467729   0.0030837   -1.1807954  0.0830785  0.2173570  0.3501377 0.2577   0.1334 0.2890718
    4.30      -3.9254943  0.9828558  -0.0633793  0.0449200   0.0033809   -1.1773877  0.0811116  0.2146505  0.3474807 0.2571   0.1330 0.2884671
    4.35      -3.9449954  0.9856556  -0.0619490  0.0428895   0.0037976   -1.1740278  0.0803051  0.2133299  0.3459795 0.2567   0.1325 0.2880207
    4.40      -3.9627506  0.9879447  -0.0605603  0.0408345   0.0042235   -1.1700078  0.0796258  0.2124220  0.3448773 0.2564   0.1320 0.2876482
    4.45      -3.9817401  0.9904571  -0.0593444  0.0391547   0.0045623   -1.1659322  0.0789487  0.2114441  0.3438778 0.2561   0.1318 0.2874406
    4.50      -4.0002776  0.9929580  -0.0583964  0.0374740   0.0049782   -1.1621150  0.0781115  0.2103225  0.3429005 0.2557   0.1314 0.2869797
    4.55      -4.0173906  0.9953528  -0.0571914  0.0356600   0.0053930   -1.1591073  0.0781459  0.2098330  0.3421470 0.2552   0.1313 0.2865754
    4.60      -4.0350420  0.9978267  -0.0560046  0.0337316   0.0058546   -1.1562649  0.0785077  0.2094146  0.3414571 0.2546   0.1311 0.2860716
    4.65      -4.0520126  1.0001513  -0.0551285  0.0315962   0.0065061   -1.1535066  0.0791992  0.2093513  0.3412293 0.2538   0.1308 0.2853654
    4.70      -4.0710588  1.0025623  -0.0543276  0.0298517   0.0070201   -1.1496486  0.0799784  0.2095075  0.3406167 0.2531   0.1307 0.2847398
    4.75      -4.0922969  1.0052053  -0.0531177  0.0277591   0.0075337   -1.1454339  0.0809215  0.2100243  0.3406828 0.2524   0.1304 0.2841189
    4.80      -4.1113809  1.0075057  -0.0519228  0.0256020   0.0080660   -1.1412920  0.0816156  0.2106448  0.3408304 0.2518   0.1300 0.2835025
    4.85      -4.1264482  1.0094846  -0.0514031  0.0236120   0.0087351   -1.1377478  0.0810001  0.2098730  0.3394549 0.2513   0.1294 0.2828076
    4.90      -4.1435918  1.0117697  -0.0507501  0.0214252   0.0094379   -1.1339983  0.0802852  0.2089583  0.3373972 0.2508   0.1288 0.2821764
    4.95      -4.1614380  1.0141080  -0.0501963  0.0189470   0.0102701   -1.1303976  0.0801779  0.2087913  0.3355871 0.2505   0.1278 0.2814264
    5.00      -4.1802390  1.0165525  -0.0496561  0.0164721   0.0110960   -1.1270094  0.0805882  0.2091696  0.3342016 0.2502   0.1265 0.2806413
    5.05      -4.2005221  1.0192277  -0.0492642  0.0140553   0.0119272   -1.1237562  0.0812199  0.2098886  0.3330671 0.2500   0.1250 0.2798441
    5.10      -4.2219992  1.0220510  -0.0488820  0.0118079   0.0126951   -1.1200374  0.0814728  0.2101721  0.3311920 0.2498   0.1237 0.2791035
    5.15      -4.2432274  1.0249565  -0.0479768  0.0097534   0.0132088   -1.1164432  0.0813101  0.2097741  0.3288936 0.2496   0.1224 0.2784241
    5.20      -4.2632242  1.0278248  -0.0471567  0.0081614   0.0135422   -1.1130039  0.0802548  0.2086363  0.3263017 0.2495   0.1211 0.2776858
    5.25      -4.2814105  1.0302352  -0.0461208  0.0066267   0.0137904   -1.1093598  0.0795133  0.2081120  0.3244664 0.2493   0.1199 0.2770149
    5.30      -4.2977794  1.0323906  -0.0451060  0.0050515   0.0140579   -1.1059272  0.0785813  0.2075033  0.3224703 0.2491   0.1187 0.2763617
    5.35      -4.3144645  1.0345604  -0.0443053  0.0035612   0.0143609   -1.1024300  0.0775883  0.2069144  0.3203146 0.2489   0.1174 0.2756179
    5.40      -4.3288983  1.0364072  -0.0432779  0.0020134   0.0146096   -1.0990836  0.0765945  0.2063372  0.3178092 0.2486   0.1161 0.2748337
    5.45      -4.3410552  1.0379742  -0.0421163  0.0005491   0.0147858   -1.0963506  0.0760367  0.2059743  0.3156195 0.2484   0.1149 0.2740774
    5.50      -4.3526016  1.0394662  -0.0409054  -0.0007999  0.0149113   -1.0938383  0.0756334  0.2056329  0.3140303 0.2481   0.1140 0.2734820
    5.55      -4.3633752  1.0408525  -0.0393238  -0.0018985  0.0148243   -1.0914925  0.0755483  0.2055798  0.3135373 0.2479   0.1134 0.2730842
    5.60      -4.3757024  1.0425147  -0.0377116  -0.0027883  0.0146532   -1.0894445  0.0759617  0.2059492  0.3132083 0.2477   0.1131 0.2727795
    5.65      -4.3883910  1.0441946  -0.0360708  -0.0038406  0.0145426   -1.0870686  0.0764218  0.2060061  0.3127964 0.2474   0.1129 0.2724828
    5.70      -4.4014385  1.0459059  -0.0340765  -0.0053216  0.0144909   -1.0843215  0.0765775  0.2058447  0.3118029 0.2472   0.1124 0.2721769
    5.75      -4.4132258  1.0474551  -0.0320068  -0.0069405  0.0144671   -1.0815278  0.0762418  0.2054107  0.3106124 0.2470   0.1119 0.2717783
    5.80      -4.4258093  1.0490781  -0.0298502  -0.0084406  0.0143672   -1.0784047  0.0757301  0.2046521  0.3091301 0.2468   0.1114 0.2713594
    5.85      -4.4386967  1.0506897  -0.0278721  -0.0097120  0.0142349   -1.0750240  0.0752216  0.2037437  0.3079376 0.2465   0.1111 0.2710187
    5.90      -4.4499864  1.0521582  -0.0259821  -0.0110298  0.0141438   -1.0718734  0.0741986  0.2022672  0.3066715 0.2463   0.1106 0.2706087
    5.95      -4.4593583  1.0532148  -0.0241406  -0.0122160  0.0140131   -1.0685676  0.0734422  0.2011022  0.3058030 0.2462   0.1102 0.2703419
    6.00      -4.4679415  1.0540811  -0.0226778  -0.0135131  0.0140572   -1.0650398  0.0725164  0.1998525  0.3048156 0.2460   0.1100 0.2700997
    6.05      -4.4762829  1.0547947  -0.0209668  -0.0150543  0.0141252   -1.0611763  0.0716589  0.1985437  0.3035990 0.2458   0.1097 0.2697964
    6.10      -4.4855420  1.0555475  -0.0191786  -0.0167523  0.0142377   -1.0572422  0.0712689  0.1977796  0.3031324 0.2455   0.1096 0.2694271
    6.15      -4.4945677  1.0563378  -0.0176127  -0.0184349  0.0144140   -1.0537145  0.0710832  0.1971042  0.3026475 0.2451   0.1094 0.2689959
    6.20      -4.5046251  1.0572594  -0.0162132  -0.0202959  0.0147193   -1.0501644  0.0710965  0.1963313  0.3018990 0.2448   0.1093 0.2685647
    6.25      -4.5138228  1.0580605  -0.0149466  -0.0222689  0.0151132   -1.0467651  0.0712230  0.1956748  0.3012923 0.2444   0.1094 0.2682012
    6.30      -4.5225077  1.0588303  -0.0137210  -0.0244369  0.0155987   -1.0434924  0.0712748  0.1949439  0.3006938 0.2439   0.1095 0.2677405
    6.35      -4.5329188  1.0598922  -0.0126246  -0.0266841  0.0161572   -1.0402547  0.0713551  0.1942486  0.3002755 0.2435   0.1094 0.2672387
    6.40      -4.5433916  1.0610632  -0.0116600  -0.0290699  0.0168119   -1.0373328  0.0714603  0.1933902  0.2998049 0.2432   0.1092 0.2667638
    6.45      -4.5541225  1.0622103  -0.0105018  -0.0315090  0.0174218   -1.0342552  0.0718178  0.1926803  0.2995561 0.2429   0.1091 0.2663783
    6.50      -4.5652135  1.0634215  -0.0094558  -0.0339892  0.0180807   -1.0313311  0.0723379  0.1920862  0.2996129 0.2426   0.1092 0.2660720
    6.55      -4.5760896  1.0645421  -0.0081198  -0.0365132  0.0186584   -1.0285027  0.0732906  0.1919729  0.3003201 0.2424   0.1093 0.2658074
    6.60      -4.5867693  1.0656721  -0.0065512  -0.0390510  0.0191537   -1.0258608  0.0742173  0.1919672  0.3012089 0.2421   0.1095 0.2655464
    6.65      -4.5966524  1.0666166  -0.0047981  -0.0414500  0.0195223   -1.0230938  0.0750479  0.1921775  0.3019930 0.2418   0.1099 0.2652596
    6.70      -4.6058162  1.0674772  -0.0034439  -0.0436108  0.0199204   -1.0202807  0.0752768  0.1920060  0.3021423 0.2414   0.1104 0.2650292
    6.75      -4.6141898  1.0682248  -0.0022478  -0.0456946  0.0203361   -1.0174034  0.0750101  0.1915879  0.3020727 0.2411   0.1109 0.2648462
    6.80      -4.6220395  1.0688609  -0.0012937  -0.0475246  0.0207275   -1.0143606  0.0745378  0.1910157  0.3023318 0.2408   0.1114 0.2646944
    6.85      -4.6313512  1.0696737  -0.0004664  -0.0492917  0.0211334   -1.0111888  0.0743141  0.1906279  0.3026158 0.2405   0.1118 0.2644696
    6.90      -4.6399017  1.0703947  0.0005012   -0.0511524  0.0215240   -1.0081795  0.0742013  0.1903052  0.3027447 0.2402   0.1123 0.2642915
    6.95      -4.6481057  1.0711696  0.0011095   -0.0528478  0.0219705   -1.0053373  0.0734373  0.1894435  0.3022383 0.2400   0.1126 0.2641548
    7.00      -4.6564338  1.0719859  0.0016623   -0.0544566  0.0224005   -1.0025043  0.0724019  0.1884992  0.3015693 0.2398   0.1129 0.2639914
    7.05      -4.6645241  1.0727822  0.0021087   -0.0558420  0.0227790   -0.9998099  0.0713930  0.1876521  0.3009488 0.2396   0.1132 0.2638702
    7.10      -4.6718847  1.0734638  0.0027788   -0.0572113  0.0230700   -0.9972469  0.0704930  0.1870359  0.3006397 0.2393   0.1136 0.2636516
    7.15      -4.6786732  1.0740111  0.0036481   -0.0586894  0.0233297   -0.9945984  0.0696309  0.1864975  0.3002830 0.2390   0.1139 0.2633801
    7.20      -4.6850935  1.0743879  0.0048040   -0.0602061  0.0235027   -0.9917470  0.0690776  0.1862713  0.2999113 0.2385   0.1144 0.2630783
    7.25      -4.6916019  1.0747585  0.0059187   -0.0616558  0.0236606   -0.9890111  0.0688639  0.1863594  0.2997983 0.2381   0.1147 0.2627337
    7.30      -4.6974575  1.0750997  0.0068145   -0.0630654  0.0238810   -0.9866848  0.0687467  0.1866099  0.2996886 0.2377   0.1150 0.2623916
    7.35      -4.7030079  1.0754274  0.0075970   -0.0644417  0.0241300   -0.9846289  0.0687476  0.1870692  0.2997346 0.2374   0.1151 0.2620576
    7.40      -4.7080183  1.0756657  0.0082142   -0.0657937  0.0244227   -0.9827268  0.0689887  0.1878099  0.3002970 0.2370   0.1153 0.2616899
    7.45      -4.7132405  1.0759228  0.0088490   -0.0671594  0.0247123   -0.9809058  0.0695284  0.1886492  0.3011786 0.2366   0.1155 0.2612931
    7.50      -4.7184342  1.0762929  0.0093855   -0.0685422  0.0250370   -0.9794317  0.0700415  0.1894098  0.3018391 0.2362   0.1158 0.2609032
    7.55      -4.7235003  1.0766033  0.0098755   -0.0699241  0.0253762   -0.9779934  0.0707763  0.1904078  0.3025285 0.2358   0.1161 0.2605820
    7.60      -4.7284430  1.0768915  0.0105246   -0.0713429  0.0256743   -0.9766568  0.0715651  0.1915217  0.3032653 0.2354   0.1164 0.2602994
    7.65      -4.7333837  1.0772140  0.0113584   -0.0728298  0.0259375   -0.9753910  0.0722029  0.1925290  0.3042325 0.2352   0.1166 0.2600373
    7.70      -4.7373559  1.0774660  0.0121957   -0.0743514  0.0262123   -0.9743814  0.0726741  0.1933697  0.3051336 0.2350   0.1168 0.2597990
    7.75      -4.7416466  1.0777728  0.0128658   -0.0758130  0.0265183   -0.9732904  0.0728978  0.1940842  0.3059485 0.2347   0.1170 0.2595587
    7.80      -4.7457680  1.0780299  0.0135641   -0.0772443  0.0268006   -0.9719277  0.0727681  0.1944019  0.3063400 0.2344   0.1173 0.2592479
    7.85      -4.7497414  1.0782000  0.0141330   -0.0786335  0.0271126   -0.9704617  0.0727670  0.1948756  0.3070301 0.2341   0.1174 0.2588959
    7.90      -4.7534506  1.0783052  0.0148319   -0.0800742  0.0273947   -0.9689004  0.0727423  0.1952855  0.3075726 0.2337   0.1176 0.2584831
    7.95      -4.7581199  1.0785516  0.0155607   -0.0815090  0.0276671   -0.9672867  0.0727983  0.1955861  0.3082346 0.2333   0.1176 0.2580498
    8.00      -4.7626885  1.0788044  0.0162454   -0.0829210  0.0279467   -0.9656956  0.0726904  0.1956210  0.3086744 0.2330   0.1176 0.2575974
    8.05      -4.7665886  1.0789803  0.0169358   -0.0842348  0.0281855   -0.9642105  0.0726910  0.1955636  0.3093254 0.2327   0.1178 0.2572404
    8.10      -4.7700794  1.0791282  0.0176797   -0.0855551  0.0284043   -0.9628105  0.0726542  0.1953642  0.3099819 0.2324   0.1179 0.2568998
    8.15      -4.7732371  1.0792491  0.0183643   -0.0869682  0.0286743   -0.9615282  0.0726398  0.1952027  0.3105540 0.2320   0.1181 0.2565271
    8.20      -4.7763286  1.0793996  0.0189037   -0.0883768  0.0289897   -0.9603004  0.0724353  0.1947804  0.3106712 0.2317   0.1182 0.2561634
    8.25      -4.7796380  1.0796619  0.0192182   -0.0897405  0.0293670   -0.9593245  0.0722249  0.1943082  0.3105713 0.2314   0.1184 0.2557813
    8.30      -4.7832874  1.0799691  0.0194238   -0.0909772  0.0297372   -0.9584142  0.0721205  0.1940355  0.3105578 0.2311   0.1185 0.2554279
    8.35      -4.7865078  1.0802141  0.0195244   -0.0920933  0.0301034   -0.9575725  0.0719822  0.1937691  0.3106331 0.2308   0.1186 0.2550582
    8.40      -4.7892784  1.0803992  0.0194128   -0.0931738  0.0305313   -0.9567578  0.0717292  0.1933543  0.3104110 0.2305   0.1187 0.2547065
    8.45      -4.7916492  1.0805978  0.0191142   -0.0940896  0.0309689   -0.9562115  0.0713832  0.1927574  0.3101309 0.2302   0.1186 0.2543620
    8.50      -4.7941878  1.0808480  0.0187293   -0.0948903  0.0313963   -0.9558112  0.0711794  0.1922034  0.3099592 0.2300   0.1186 0.2540273
    8.55      -4.7972561  1.0811811  0.0184135   -0.0956798  0.0317989   -0.9554520  0.0710746  0.1917249  0.3098812 0.2298   0.1186 0.2537017
    8.60      -4.8003724  1.0815272  0.0181117   -0.0964288  0.0321830   -0.9551487  0.0709386  0.1913320  0.3100579 0.2296   0.1185 0.2534126
    8.65      -4.8032572  1.0819053  0.0177544   -0.0971231  0.0325700   -0.9550626  0.0707406  0.1908932  0.3102053 0.2294   0.1184 0.2531347
    8.70      -4.8061043  1.0823138  0.0173877   -0.0978012  0.0329590   -0.9549669  0.0702222  0.1901884  0.3101735 0.2292   0.1183 0.2528240
    8.75      -4.8090039  1.0827344  0.0170120   -0.0984665  0.0333452   -0.9549550  0.0698091  0.1895978  0.3101426 0.2290   0.1182 0.2525078
    8.80      -4.8117903  1.0831586  0.0165172   -0.0989960  0.0337240   -0.9550325  0.0694165  0.1890670  0.3102656 0.2288   0.1181 0.2521449
    8.85      -4.8144080  1.0835596  0.0160393   -0.0994567  0.0340667   -0.9550519  0.0689427  0.1884223  0.3102450 0.2285   0.1180 0.2517523
    8.90      -4.8166316  1.0839068  0.0156945   -0.0999493  0.0343739   -0.9550925  0.0684323  0.1877153  0.3101927 0.2282   0.1178 0.2513758
    8.95      -4.8188422  1.0842174  0.0153947   -0.1004674  0.0346745   -0.9550275  0.0679239  0.1869933  0.3101146 0.2279   0.1177 0.2509823
    9.00      -4.8215023  1.0845602  0.0151716   -0.1009873  0.0349482   -0.9549074  0.0676062  0.1864257  0.3101962 0.2276   0.1176 0.2505924
    9.05      -4.8243651  1.0849492  0.0150286   -0.1014167  0.0351559   -0.9547977  0.0672304  0.1858608  0.3100967 0.2273   0.1175 0.2502004
    9.10      -4.8273470  1.0853183  0.0149687   -0.1017354  0.0352933   -0.9545373  0.0668824  0.1853036  0.3099099 0.2270   0.1173 0.2497791
    9.15      -4.8302099  1.0857072  0.0148901   -0.1019413  0.0353945   -0.9544321  0.0665933  0.1848195  0.3097247 0.2266   0.1172 0.2494146
    9.20      -4.8327702  1.0860440  0.0147585   -0.1021489  0.0355138   -0.9543765  0.0664303  0.1844235  0.3095646 0.2264   0.1171 0.2490719
    9.25      -4.8349965  1.0863712  0.0147374   -0.1024022  0.0356107   -0.9544739  0.0662494  0.1839903  0.3093731 0.2261   0.1170 0.2487250
    9.30      -4.8368640  1.0866012  0.0147551   -0.1025448  0.0356534   -0.9544362  0.0660094  0.1836431  0.3090714 0.2258   0.1169 0.2483730
    9.35      -4.8385006  1.0867920  0.0148296   -0.1026749  0.0356689   -0.9544144  0.0658520  0.1833232  0.3087947 0.2255   0.1167 0.2480003
    9.40      -4.8396595  1.0869070  0.0150762   -0.1028747  0.0356507   -0.9544054  0.0656842  0.1830280  0.3086196 0.2252   0.1166 0.2476253
    9.45      -4.8402948  1.0869362  0.0153007   -0.1030859  0.0356418   -0.9543600  0.0654170  0.1826539  0.3083844 0.2249   0.1165 0.2472734
    9.50      -4.8406320  1.0869316  0.0154698   -0.1032534  0.0356362   -0.9543334  0.0650698  0.1822341  0.3080600 0.2246   0.1164 0.2469597
    9.55      -4.8409575  1.0869060  0.0156111   -0.1034828  0.0356629   -0.9543325  0.0648798  0.1819317  0.3078291 0.2243   0.1163 0.2466628
    9.60      -4.8414445  1.0868861  0.0157722   -0.1037108  0.0356846   -0.9543067  0.0647759  0.1816860  0.3077684 0.2241   0.1162 0.2463870
    9.65      -4.8418767  1.0868878  0.0159301   -0.1039059  0.0356985   -0.9543802  0.0646080  0.1814507  0.3075527 0.2239   0.1161 0.2461159
    9.70      -4.8432550  1.0870026  0.0161409   -0.1040324  0.0356713   -0.9543412  0.0645455  0.1812483  0.3072964 0.2237   0.1159 0.2458109
    9.75      -4.8448301  1.0871395  0.0162848   -0.1041190  0.0356551   -0.9542876  0.0645008  0.1810066  0.3069951 0.2235   0.1157 0.2455038
    9.80      -4.8458452  1.0871545  0.0164981   -0.1042243  0.0356149   -0.9541557  0.0645134  0.1807746  0.3066987 0.2233   0.1155 0.2452255
    9.85      -4.8472373  1.0872268  0.0167811   -0.1043532  0.0355605   -0.9539727  0.0644992  0.1805418  0.3063476 0.2232   0.1153 0.2449635
    9.90      -4.8486669  1.0872922  0.0171504   -0.1045048  0.0354867   -0.9537759  0.0645466  0.1804061  0.3059744 0.2230   0.1151 0.2447166
    9.95      -4.8497512  1.0873154  0.0176149   -0.1046294  0.0353730   -0.9536196  0.0645639  0.1802776  0.3055848 0.2229   0.1149 0.2445081
    10.00     -4.8508738  1.0873776  0.0182216   -0.1047668  0.0352207   -0.9536000  0.0646184  0.1801535  0.3052846 0.2228   0.1148 0.2443381
    10.05     -4.8520525  1.0874715  0.0188559   -0.1048495  0.0350378   -0.9536086  0.0645785  0.1799751  0.3050582 0.2227   0.1147 0.2441544
    10.10     -4.8531293  1.0875526  0.0194988   -0.1049785  0.0348672   -0.9536830  0.0646650  0.1798980  0.3050001 0.2225   0.1145 0.2439613
    10.15     -4.8542345  1.0876338  0.0200578   -0.1050455  0.0347021   -0.9536802  0.0646216  0.1797502  0.3049350 0.2224   0.1144 0.2437704
    10.20     -4.8548889  1.0876630  0.0205549   -0.1051107  0.0345569   -0.9536498  0.0643829  0.1793861  0.3048211 0.2222   0.1144 0.2435657
    10.25     -4.8553387  1.0877168  0.0210123   -0.1051962  0.0344334   -0.9536686  0.0638718  0.1787846  0.3044119 0.2220   0.1143 0.2433559
    10.30     -4.8557950  1.0877832  0.0214332   -0.1052755  0.0343224   -0.9537135  0.0633511  0.1781602  0.3039007 0.2218   0.1143 0.2431565
    10.35     -4.8560147  1.0878608  0.0218281   -0.1053753  0.0342318   -0.9538751  0.0627204  0.1774032  0.3032747 0.2217   0.1142 0.2429631
    10.40     -4.8558789  1.0878883  0.0221702   -0.1054319  0.0341415   -0.9540642  0.0621120  0.1766440  0.3026522 0.2215   0.1142 0.2427868
    10.45     -4.8558458  1.0879276  0.0224772   -0.1054086  0.0340367   -0.9542783  0.0615596  0.1759337  0.3021191 0.2213   0.1141 0.2426167
    10.50     -4.8559204  1.0880274  0.0228641   -0.1053427  0.0338880   -0.9546210  0.0610104  0.1752243  0.3016720 0.2212   0.1141 0.2424636
    10.55     -4.8559903  1.0881597  0.0231565   -0.1052096  0.0337478   -0.9550110  0.0603409  0.1743910  0.3010521 0.2210   0.1142 0.2423390
    10.60     -4.8559723  1.0882937  0.0233616   -0.1050659  0.0336326   -0.9553946  0.0595259  0.1734449  0.3003596 0.2208   0.1142 0.2422435
    10.65     -4.8551954  1.0883055  0.0236849   -0.1049635  0.0334815   -0.9557810  0.0587336  0.1725502  0.2996567 0.2207   0.1144 0.2421949
    10.70     -4.8541666  1.0882964  0.0239370   -0.1048805  0.0333649   -0.9561930  0.0578187  0.1715451  0.2989210 0.2206   0.1145 0.2421648
    10.75     -4.8529033  1.0882591  0.0242091   -0.1047844  0.0332377   -0.9566112  0.0568152  0.1704948  0.2980115 0.2205   0.1146 0.2421267
    10.80     -4.8511335  1.0881011  0.0244151   -0.1046786  0.0331285   -0.9569159  0.0558209  0.1695045  0.2971087 0.2203   0.1147 0.2420731
    10.85     -4.8494208  1.0879092  0.0246379   -0.1045914  0.0330209   -0.9571443  0.0549477  0.1686949  0.2963376 0.2202   0.1148 0.2420026
    10.90     -4.8479970  1.0877240  0.0248492   -0.1045372  0.0329293   -0.9572759  0.0541150  0.1679817  0.2955721 0.2201   0.1149 0.2419061
    10.95     -4.8466023  1.0875424  0.0249844   -0.1044733  0.0328624   -0.9574042  0.0532431  0.1672514  0.2948414 0.2199   0.1150 0.2418088
    11.00     -4.8455767  1.0874164  0.0251354   -0.1044587  0.0328044   -0.9575525  0.0525074  0.1666401  0.2942763 0.2197   0.1151 0.2416798
    11.05     -4.8451272  1.0873393  0.0252461   -0.1044601  0.0327632   -0.9576145  0.0519027  0.1662010  0.2938975 0.2195   0.1151 0.2415404
    11.10     -4.8448186  1.0872734  0.0253992   -0.1044616  0.0327064   -0.9576825  0.0513572  0.1658164  0.2935938 0.2193   0.1152 0.2413874
    11.15     -4.8444895  1.0871889  0.0255796   -0.1044614  0.0326416   -0.9577433  0.0508885  0.1655010  0.2934984 0.2192   0.1152 0.2412555
    11.20     -4.8441165  1.0870937  0.0257649   -0.1044777  0.0325803   -0.9578506  0.0505479  0.1653058  0.2935147 0.2190   0.1153 0.2411464
    11.25     -4.8433818  1.0869303  0.0259813   -0.1045180  0.0325148   -0.9579722  0.0503430  0.1651964  0.2935834 0.2189   0.1154 0.2410602
    11.30     -4.8428825  1.0867746  0.0261210   -0.1045130  0.0324638   -0.9580208  0.0501904  0.1651464  0.2936029 0.2188   0.1154 0.2409712
    11.35     -4.8424033  1.0866413  0.0262078   -0.1044773  0.0324204   -0.9581255  0.0500471  0.1650322  0.2936077 0.2186   0.1154 0.2408577
    11.40     -4.8419125  1.0864944  0.0263218   -0.1044056  0.0323527   -0.9582672  0.0500782  0.1650340  0.2936956 0.2185   0.1154 0.2407148
    11.45     -4.8409569  1.0862966  0.0264211   -0.1043143  0.0322854   -0.9584456  0.0499643  0.1648808  0.2936400 0.2183   0.1154 0.2405652
    11.50     -4.8401097  1.0861703  0.0264425   -0.1042113  0.0322431   -0.9587850  0.0498383  0.1646788  0.2934808 0.2182   0.1153 0.2403893
    11.55     -4.8392570  1.0860672  0.0264411   -0.1040844  0.0321994   -0.9592195  0.0497454  0.1644967  0.2932492 0.2180   0.1152 0.2402175
    11.60     -4.8386082  1.0859766  0.0264538   -0.1039743  0.0321592   -0.9596104  0.0497138  0.1643451  0.2930223 0.2178   0.1152 0.2400583
    11.65     -4.8381898  1.0859394  0.0265112   -0.1038288  0.0320917   -0.9600518  0.0497075  0.1642198  0.2927549 0.2177   0.1151 0.2399215
    11.70     -4.8377778  1.0859077  0.0265806   -0.1035993  0.0319886   -0.9605196  0.0497032  0.1641466  0.2925716 0.2176   0.1150 0.2398025
    11.75     -4.8378659  1.0859555  0.0266477   -0.1033411  0.0318757   -0.9610096  0.0497511  0.1641969  0.2924711 0.2175   0.1149 0.2396877
    11.80     -4.8380533  1.0860244  0.0266921   -0.1030636  0.0317647   -0.9615650  0.0498771  0.1642785  0.2924387 0.2174   0.1148 0.2395951
    11.85     -4.8378086  1.0860230  0.0267097   -0.1028034  0.0316687   -0.9621425  0.0500548  0.1643652  0.2923961 0.2173   0.1146 0.2394984
    11.90     -4.8374250  1.0860036  0.0267422   -0.1025730  0.0315801   -0.9627583  0.0502885  0.1645031  0.2923472 0.2173   0.1145 0.2394194
    11.95     -4.8370244  1.0859907  0.0266855   -0.1023682  0.0315326   -0.9634591  0.0505874  0.1647576  0.2923922 0.2173   0.1143 0.2393555
    12.00     -4.8366642  1.0859751  0.0265629   -0.1021471  0.0315028   -0.9641647  0.0509617  0.1650594  0.2925003 0.2172   0.1142 0.2392792
    12.05     -4.8364094  1.0859750  0.0263890   -0.1019359  0.0314970   -0.9648862  0.0513573  0.1653605  0.2926679 0.2172   0.1140 0.2391885
    12.10     -4.8361559  1.0859864  0.0262925   -0.1017421  0.0314666   -0.9656270  0.0517370  0.1656378  0.2928545 0.2171   0.1139 0.2390943
    12.15     -4.8355714  1.0859581  0.0261593   -0.1015788  0.0314579   -0.9663973  0.0520990  0.1659018  0.2929811 0.2170   0.1138 0.2389903
    12.20     -4.8345695  1.0858665  0.0261163   -0.1014431  0.0314254   -0.9671865  0.0524896  0.1661797  0.2931332 0.2170   0.1137 0.2389035
    12.25     -4.8333309  1.0857361  0.0261508   -0.1013411  0.0313743   -0.9680018  0.0529375  0.1664913  0.2933075 0.2169   0.1136 0.2387982
    12.30     -4.8321450  1.0856025  0.0261684   -0.1012592  0.0313395   -0.9688009  0.0534033  0.1668001  0.2935109 0.2168   0.1135 0.2386694
    12.35     -4.8311105  1.0854901  0.0262118   -0.1011641  0.0312918   -0.9696590  0.0539711  0.1672158  0.2938140 0.2167   0.1133 0.2385414
    12.40     -4.8297205  1.0853420  0.0262184   -0.1010431  0.0312490   -0.9706058  0.0545339  0.1676385  0.2941208 0.2166   0.1132 0.2383953
    12.45     -4.8281832  1.0851863  0.0262381   -0.1009592  0.0312150   -0.9715865  0.0550302  0.1680143  0.2944165 0.2165   0.1130 0.2382538
    12.50     -4.8267983  1.0850639  0.0262508   -0.1008653  0.0311767   -0.9725810  0.0555308  0.1683581  0.2947263 0.2164   0.1129 0.2381260
    12.55     -4.8256300  1.0849539  0.0262744   -0.1007723  0.0311344   -0.9735115  0.0560610  0.1687491  0.2950917 0.2163   0.1128 0.2379794
    12.60     -4.8245266  1.0848092  0.0262610   -0.1006666  0.0310993   -0.9743106  0.0566436  0.1692159  0.2954450 0.2161   0.1128 0.2378045
    12.65     -4.8232403  1.0846413  0.0262727   -0.1005607  0.0310574   -0.9750685  0.0570886  0.1695401  0.2956459 0.2160   0.1126 0.2376419
    12.70     -4.8221555  1.0845149  0.0262667   -0.1004764  0.0310271   -0.9758250  0.0575135  0.1698359  0.2958633 0.2158   0.1126 0.2374827
    12.75     -4.8211031  1.0844048  0.0262656   -0.1004236  0.0310088   -0.9766382  0.0579730  0.1701685  0.2961261 0.2157   0.1125 0.2373147
    12.80     -4.8200114  1.0842839  0.0262315   -0.1003799  0.0310041   -0.9774817  0.0584768  0.1705698  0.2964927 0.2155   0.1124 0.2371498
    12.85     -4.8188944  1.0841838  0.0261369   -0.1002920  0.0310018   -0.9783925  0.0589641  0.1709392  0.2968484 0.2154   0.1124 0.2369944
    12.90     -4.8178186  1.0841031  0.0260634   -0.1002234  0.0310002   -0.9793144  0.0593925  0.1712593  0.2971641 0.2152   0.1123 0.2368494
    12.95     -4.8166150  1.0839941  0.0260275   -0.1002048  0.0310007   -0.9802397  0.0598521  0.1716700  0.2975270 0.2150   0.1123 0.2366935
    13.00     -4.8153272  1.0838635  0.0260146   -0.1001833  0.0309926   -0.9811625  0.0603390  0.1721548  0.2979329 0.2149   0.1122 0.2365344
    13.05     -4.8137963  1.0837094  0.0259874   -0.1001172  0.0309742   -0.9821164  0.0608092  0.1725948  0.2982473 0.2147   0.1122 0.2363861
    13.10     -4.8122283  1.0835450  0.0259375   -0.1000166  0.0309507   -0.9830616  0.0613387  0.1730180  0.2985631 0.2146   0.1121 0.2362369
    13.15     -4.8107612  1.0833798  0.0258631   -0.0998796  0.0309218   -0.9839619  0.0618998  0.1734775  0.2989463 0.2144   0.1121 0.2360788
    13.20     -4.8095346  1.0832551  0.0257959   -0.0997334  0.0308852   -0.9848879  0.0625067  0.1739819  0.2993840 0.2143   0.1120 0.2359328
    13.25     -4.8083083  1.0831092  0.0257606   -0.0996305  0.0308534   -0.9857526  0.0631415  0.1745271  0.2997599 0.2141   0.1120 0.2357845
    13.30     -4.8071585  1.0829697  0.0257116   -0.0995501  0.0308328   -0.9866263  0.0638395  0.1751548  0.3001520 0.2140   0.1119 0.2356340
    13.35     -4.8061292  1.0828416  0.0256489   -0.0994825  0.0308232   -0.9874663  0.0645207  0.1757860  0.3005933 0.2138   0.1119 0.2354863
    13.40     -4.8051217  1.0826985  0.0255770   -0.0994458  0.0308275   -0.9882449  0.0652211  0.1764256  0.3011359 0.2136   0.1119 0.2353332
    13.45     -4.8038868  1.0825334  0.0254987   -0.0994133  0.0308368   -0.9890534  0.0658637  0.1770095  0.3016893 0.2135   0.1118 0.2351944
    13.50     -4.8024239  1.0823730  0.0253600   -0.0993621  0.0308609   -0.9899422  0.0663880  0.1774497  0.3021424 0.2133   0.1118 0.2350599
    13.55     -4.8008862  1.0822101  0.0251997   -0.0993044  0.0308914   -0.9908800  0.0669207  0.1778896  0.3025833 0.2132   0.1118 0.2349244
    13.60     -4.7993563  1.0820485  0.0250348   -0.0992297  0.0309193   -0.9917992  0.0674028  0.1782854  0.3029475 0.2131   0.1117 0.2347763
    13.65     -4.7976366  1.0818814  0.0248539   -0.0991312  0.0309456   -0.9927529  0.0677424  0.1785778  0.3031409 0.2129   0.1117 0.2346214
    13.70     -4.7960157  1.0817378  0.0246715   -0.0990215  0.0309671   -0.9936981  0.0679883  0.1788409  0.3033234 0.2128   0.1116 0.2344551
    13.75     -4.7944668  1.0816051  0.0244679   -0.0988908  0.0309910   -0.9946419  0.0681971  0.1790905  0.3035344 0.2126   0.1115 0.2342970
    13.80     -4.7929032  1.0814764  0.0242355   -0.0987557  0.0310232   -0.9956093  0.0683836  0.1793264  0.3037899 0.2125   0.1114 0.2341490
    13.85     -4.7912189  1.0813383  0.0240137   -0.0986289  0.0310541   -0.9966098  0.0685423  0.1795860  0.3041442 0.2124   0.1114 0.2340211
    13.90     -4.7892300  1.0811350  0.0239220   -0.0985304  0.0310486   -0.9975623  0.0687294  0.1798755  0.3044686 0.2123   0.1113 0.2338992
    13.95     -4.7869067  1.0808893  0.0239037   -0.0984477  0.0310209   -0.9985045  0.0688552  0.1800773  0.3047168 0.2121   0.1112 0.2337757
    14.00     -4.7841550  1.0805819  0.0238202   -0.0983405  0.0310078   -0.9994449  0.0689329  0.1802110  0.3048412 0.2120   0.1112 0.2336442
    14.05     -4.7813854  1.0802594  0.0237047   -0.0982132  0.0309981   -1.0003559  0.0690138  0.1803659  0.3050219 0.2119   0.1111 0.2335061
    14.10     -4.7785897  1.0799130  0.0235853   -0.0981191  0.0310004   -1.0012381  0.0691607  0.1806046  0.3053345 0.2117   0.1110 0.2333679
    14.15     -4.7759194  1.0795757  0.0234962   -0.0980332  0.0309947   -1.0021289  0.0694071  0.1809492  0.3057072 0.2116   0.1110 0.2332303
    14.20     -4.7732917  1.0792394  0.0234813   -0.0979732  0.0309728   -1.0030406  0.0697412  0.1813700  0.3061326 0.2115   0.1110 0.2330916
    14.25     -4.7710102  1.0789350  0.0234461   -0.0979221  0.0309621   -1.0039059  0.0701510  0.1818828  0.3065905 0.2113   0.1109 0.2329480
    14.30     -4.7688688  1.0786366  0.0233897   -0.0978736  0.0309613   -1.0047392  0.0706246  0.1824482  0.3070692 0.2112   0.1109 0.2328159
    14.35     -4.7667450  1.0783256  0.0233085   -0.0978452  0.0309725   -1.0055367  0.0711715  0.1830769  0.3075832 0.2110   0.1108 0.2326739
    14.40     -4.7646670  1.0780141  0.0232164   -0.0978083  0.0309804   -1.0063160  0.0717616  0.1837517  0.3081310 0.2109   0.1108 0.2325393
    14.45     -4.7626265  1.0777104  0.0231100   -0.0977951  0.0310032   -1.0071021  0.0723428  0.1844138  0.3086651 0.2107   0.1108 0.2324163
    14.50     -4.7603009  1.0773754  0.0229665   -0.0977716  0.0310335   -1.0079186  0.0728598  0.1850324  0.3091492 0.2106   0.1108 0.2322923
    14.55     -4.7575713  1.0769761  0.0228549   -0.0977583  0.0310556   -1.0087182  0.0733263  0.1856133  0.3095289 0.2104   0.1108 0.2321677
    14.60     -4.7548636  1.0765553  0.0227576   -0.0977356  0.0310718   -1.0094307  0.0738031  0.1862186  0.3098453 0.2103   0.1108 0.2320530
    14.65     -4.7522813  1.0761366  0.0226986   -0.0977178  0.0310757   -1.0100692  0.0742822  0.1867970  0.3101810 0.2102   0.1108 0.2319278
    14.70     -4.7497848  1.0757218  0.0226024   -0.0977033  0.0310929   -1.0106835  0.0747917  0.1873962  0.3105709 0.2100   0.1108 0.2318015
    14.75     -4.7472930  1.0752827  0.0224760   -0.0976875  0.0311206   -1.0112189  0.0753477  0.1880216  0.3109780 0.2099   0.1108 0.2316731
    14.80     -4.7447541  1.0748449  0.0223254   -0.0976597  0.0311542   -1.0117278  0.0757871  0.1885495  0.3112721 0.2097   0.1108 0.2315483
    14.85     -4.7422085  1.0744110  0.0221484   -0.0976276  0.0311951   -1.0122201  0.0761399  0.1890205  0.3115217 0.2096   0.1108 0.2314401
    14.90     -4.7397546  1.0739978  0.0219923   -0.0976228  0.0312391   -1.0127350  0.0765169  0.1894916  0.3118554 0.2095   0.1108 0.2313541
    14.95     -4.7374115  1.0736142  0.0218280   -0.0976309  0.0312930   -1.0132843  0.0768770  0.1899358  0.3121921 0.2094   0.1108 0.2312818
    15.00     -4.7351622  1.0732454  0.0216832   -0.0976572  0.0313461   -1.0138508  0.0772720  0.1904162  0.3125631 0.2094   0.1108 0.2312069
    15.05     -4.7327127  1.0728460  0.0215290   -0.0976787  0.0314001   -1.0144161  0.0776711  0.1908959  0.3129103 0.2093   0.1107 0.2311329
    15.10     -4.7302110  1.0724443  0.0213384   -0.0977080  0.0314715   -1.0149826  0.0780303  0.1913404  0.3132345 0.2092   0.1107 0.2310669
    15.15     -4.7276504  1.0720412  0.0211184   -0.0977149  0.0315470   -1.0155643  0.0783576  0.1917483  0.3135057 0.2092   0.1107 0.2310019
    15.20     -4.7251823  1.0716488  0.0208990   -0.0977142  0.0316200   -1.0161360  0.0786955  0.1921643  0.3138020 0.2091   0.1106 0.2309334
    15.25     -4.7227379  1.0712665  0.0206539   -0.0977143  0.0317006   -1.0167244  0.0790633  0.1925929  0.3141124 0.2091   0.1106 0.2308595
    15.30     -4.7202541  1.0708919  0.0203899   -0.0977086  0.0317855   -1.0173503  0.0794412  0.1930146  0.3144547 0.2090   0.1105 0.2307880
    15.35     -4.7178315  1.0705268  0.0201266   -0.0976864  0.0318661   -1.0179744  0.0798123  0.1934331  0.3147755 0.2090   0.1105 0.2307247
    15.40     -4.7155291  1.0701848  0.0198715   -0.0976747  0.0319473   -1.0186034  0.0801897  0.1938373  0.3150762 0.2089   0.1104 0.2306579
    15.45     -4.7133027  1.0698603  0.0196013   -0.0976576  0.0320337   -1.0192672  0.0805900  0.1942705  0.3154266 0.2089   0.1103 0.2305955
    15.50     -4.7110987  1.0695422  0.0193120   -0.0976455  0.0321301   -1.0199535  0.0810024  0.1947128  0.3158384 0.2089   0.1102 0.2305408
    15.55     -4.7089699  1.0692398  0.0190568   -0.0976373  0.0322137   -1.0206326  0.0814027  0.1951351  0.3162268 0.2089   0.1102 0.2305078
    15.60     -4.7070967  1.0689764  0.0188445   -0.0976334  0.0322831   -1.0212865  0.0817967  0.1955432  0.3167070 0.2088   0.1102 0.2304669
    15.65     -4.7054210  1.0687400  0.0186734   -0.0976362  0.0323421   -1.0219407  0.0822569  0.1959874  0.3172516 0.2088   0.1102 0.2304276
    15.70     -4.7038210  1.0685162  0.0185307   -0.0976224  0.0323821   -1.0225997  0.0827523  0.1964787  0.3178195 0.2087   0.1102 0.2304036
    15.75     -4.7021376  1.0682721  0.0184286   -0.0976039  0.0324047   -1.0232170  0.0832464  0.1969506  0.3183487 0.2087   0.1102 0.2303783
    15.80     -4.7004663  1.0680240  0.0183248   -0.0975736  0.0324249   -1.0238277  0.0837420  0.1974383  0.3188922 0.2086   0.1102 0.2303550
    15.85     -4.6987932  1.0677838  0.0182453   -0.0975396  0.0324358   -1.0244510  0.0842060  0.1979095  0.3193994 0.2086   0.1102 0.2303273
    15.90     -4.6970673  1.0675578  0.0181594   -0.0975183  0.0324523   -1.0250656  0.0845219  0.1982290  0.3197538 0.2085   0.1103 0.2302839
    15.95     -4.6953586  1.0673362  0.0180549   -0.0974703  0.0324666   -1.0256581  0.0847820  0.1984990  0.3200263 0.2085   0.1103 0.2302487
    16.00     -4.6937548  1.0671358  0.0179393   -0.0973911  0.0324746   -1.0262230  0.0849605  0.1986900  0.3202485 0.2084   0.1103 0.2302207
    16.05     -4.6923572  1.0669711  0.0178204   -0.0973114  0.0324849   -1.0267997  0.0851413  0.1988959  0.3204643 0.2084   0.1103 0.2301987
    16.10     -4.6910703  1.0668240  0.0176856   -0.0972430  0.0325048   -1.0273921  0.0853441  0.1991421  0.3206696 0.2083   0.1103 0.2301751
    16.15     -4.6900638  1.0667321  0.0175543   -0.0971864  0.0325278   -1.0280104  0.0855337  0.1994054  0.3208618 0.2083   0.1103 0.2301426
    16.20     -4.6892081  1.0666681  0.0174339   -0.0971086  0.0325380   -1.0286435  0.0857302  0.1997039  0.3210797 0.2083   0.1103 0.2301121
    16.25     -4.6882101  1.0665812  0.0173375   -0.0970294  0.0325410   -1.0292916  0.0859453  0.2000185  0.3213237 0.2083   0.1102 0.2300943
    16.30     -4.6871577  1.0664834  0.0172649   -0.0969327  0.0325297   -1.0299455  0.0861705  0.2003568  0.3215539 0.2082   0.1102 0.2300821
    16.35     -4.6859866  1.0663683  0.0172147   -0.0968240  0.0325070   -1.0306158  0.0863936  0.2007056  0.3217842 0.2082   0.1102 0.2300693
    16.40     -4.6844581  1.0662119  0.0171844   -0.0967004  0.0324707   -1.0313005  0.0865343  0.2009569  0.3219247 0.2082   0.1102 0.2300536
    16.45     -4.6829163  1.0660548  0.0171636   -0.0965646  0.0324270   -1.0319896  0.0866818  0.2011866  0.3220628 0.2081   0.1102 0.2300384
    16.50     -4.6813447  1.0658975  0.0171682   -0.0964409  0.0323796   -1.0326724  0.0867880  0.2013669  0.3221690 0.2081   0.1102 0.2300161
    16.55     -4.6798765  1.0657657  0.0171690   -0.0963393  0.0323403   -1.0333728  0.0868776  0.2015479  0.3222673 0.2081   0.1101 0.2299883
    16.60     -4.6785102  1.0656579  0.0171644   -0.0962463  0.0323064   -1.0341084  0.0869803  0.2017401  0.3224111 0.2080   0.1101 0.2299620
    16.65     -4.6771776  1.0655500  0.0171470   -0.0961616  0.0322802   -1.0348354  0.0870961  0.2019490  0.3226047 0.2080   0.1101 0.2299381
    16.70     -4.6757734  1.0654368  0.0171497   -0.0960839  0.0322504   -1.0355932  0.0872389  0.2021688  0.3227835 0.2080   0.1101 0.2299154
    16.75     -4.6743159  1.0653184  0.0171542   -0.0959886  0.0322147   -1.0363682  0.0873900  0.2023881  0.3229671 0.2080   0.1100 0.2298930
    16.80     -4.6728867  1.0652029  0.0171471   -0.0958938  0.0321832   -1.0371497  0.0875539  0.2026095  0.3231750 0.2079   0.1100 0.2298740
    16.85     -4.6715492  1.0651030  0.0171361   -0.0957863  0.0321485   -1.0379311  0.0876988  0.2028368  0.3233503 0.2079   0.1100 0.2298588
    16.90     -4.6702329  1.0650099  0.0171258   -0.0956624  0.0321093   -1.0387313  0.0878417  0.2030778  0.3235388 0.2079   0.1100 0.2298489
    16.95     -4.6689237  1.0649152  0.0171137   -0.0955281  0.0320670   -1.0395288  0.0879858  0.2033293  0.3237164 0.2079   0.1100 0.2298486
    17.00     -4.6675134  1.0647923  0.0171065   -0.0953903  0.0320193   -1.0402976  0.0881642  0.2036060  0.3239206 0.2079   0.1100 0.2298577
    17.05     -4.6660140  1.0646495  0.0170882   -0.0952570  0.0319786   -1.0410556  0.0883592  0.2038878  0.3241292 0.2079   0.1100 0.2298726
    17.10     -4.6646511  1.0645232  0.0170635   -0.0951328  0.0319428   -1.0418071  0.0885791  0.2041979  0.3243785 0.2079   0.1100 0.2298932
    17.15     -4.6633531  1.0644047  0.0170666   -0.0950193  0.0319014   -1.0425700  0.0888388  0.2045350  0.3246605 0.2079   0.1099 0.2299197
    17.20     -4.6619289  1.0642770  0.0170469   -0.0949010  0.0318679   -1.0433643  0.0890622  0.2048385  0.3248946 0.2079   0.1099 0.2299521
    17.25     -4.6604410  1.0641456  0.0170264   -0.0947814  0.0318340   -1.0441646  0.0892486  0.2051066  0.3250794 0.2079   0.1099 0.2299816
    17.30     -4.6586578  1.0639689  0.0170053   -0.0946914  0.0318109   -1.0449485  0.0893867  0.2053397  0.3252274 0.2080   0.1099 0.2300153
    17.35     -4.6563665  1.0637107  0.0169468   -0.0946099  0.0318011   -1.0457354  0.0895250  0.2055897  0.3253705 0.2080   0.1099 0.2300551
    17.40     -4.6540277  1.0634495  0.0168731   -0.0945033  0.0317891   -1.0465340  0.0896464  0.2058501  0.3254761 0.2080   0.1099 0.2300908
    17.45     -4.6517418  1.0631944  0.0167738   -0.0943912  0.0317839   -1.0473300  0.0897633  0.2061409  0.3255759 0.2080   0.1099 0.2301179
    17.50     -4.6494883  1.0629398  0.0166603   -0.0942646  0.0317774   -1.0481138  0.0898934  0.2064408  0.3257322 0.2081   0.1099 0.2301417
    17.55     -4.6470506  1.0626676  0.0165633   -0.0941541  0.0317718   -1.0489303  0.0900297  0.2067105  0.3258672 0.2081   0.1098 0.2301681
    17.60     -4.6445814  1.0623893  0.0165054   -0.0940256  0.0317453   -1.0497098  0.0901301  0.2069234  0.3259550 0.2081   0.1098 0.2302117
    17.65     -4.6421749  1.0621183  0.0164439   -0.0938623  0.0317051   -1.0504661  0.0902339  0.2071135  0.3260625 0.2081   0.1099 0.2302693
    17.70     -4.6398074  1.0618582  0.0163676   -0.0936873  0.0316665   -1.0512301  0.0903089  0.2072879  0.3261484 0.2081   0.1099 0.2303325
    17.75     -4.6373872  1.0615992  0.0162742   -0.0934887  0.0316252   -1.0519862  0.0902888  0.2073950  0.3261244 0.2082   0.1099 0.2303997
    17.80     -4.6350106  1.0613510  0.0161757   -0.0932913  0.0315870   -1.0527487  0.0902719  0.2075053  0.3260738 0.2082   0.1099 0.2304586
    17.85     -4.6326753  1.0611078  0.0160714   -0.0931094  0.0315565   -1.0535132  0.0902568  0.2076309  0.3260459 0.2082   0.1100 0.2305202
    17.90     -4.6303255  1.0608665  0.0159562   -0.0929230  0.0315285   -1.0543025  0.0902443  0.2077620  0.3260424 0.2083   0.1100 0.2305901
    17.95     -4.6279721  1.0606296  0.0158294   -0.0927453  0.0315081   -1.0551004  0.0902235  0.2078697  0.3260325 0.2083   0.1100 0.2306635
    18.00     -4.6256806  1.0604029  0.0157045   -0.0925735  0.0314897   -1.0559032  0.0902143  0.2079853  0.3260227 0.2083   0.1101 0.2307335
    18.05     -4.6235482  1.0601937  0.0155649   -0.0923856  0.0314696   -1.0566806  0.0902402  0.2081103  0.3259923 0.2083   0.1101 0.2307986
    18.10     -4.6215125  1.0599832  0.0154289   -0.0922064  0.0314529   -1.0574170  0.0902938  0.2082641  0.3260346 0.2084   0.1102 0.2308669
    18.15     -4.6195446  1.0597824  0.0152808   -0.0920299  0.0314411   -1.0581603  0.0903571  0.2084408  0.3261046 0.2084   0.1102 0.2309436
    18.20     -4.6175680  1.0595838  0.0151290   -0.0918681  0.0314366   -1.0588991  0.0903924  0.2085860  0.3261727 0.2084   0.1103 0.2310192
    18.25     -4.6155501  1.0593811  0.0149677   -0.0917003  0.0314337   -1.0596409  0.0904073  0.2087303  0.3262638 0.2085   0.1103 0.2311089
    18.30     -4.6134608  1.0591673  0.0147923   -0.0915262  0.0314338   -1.0603974  0.0904314  0.2088947  0.3264244 0.2085   0.1104 0.2312029
    18.35     -4.6113606  1.0589534  0.0146054   -0.0913432  0.0314342   -1.0611660  0.0904607  0.2090800  0.3265908 0.2086   0.1105 0.2312943
    18.40     -4.6091615  1.0587210  0.0144167   -0.0911767  0.0314414   -1.0619288  0.0904961  0.2092733  0.3267645 0.2086   0.1105 0.2313800
    18.45     -4.6069186  1.0584798  0.0142445   -0.0910011  0.0314396   -1.0626962  0.0905433  0.2094949  0.3269409 0.2086   0.1106 0.2314630
    18.50     -4.6045548  1.0582263  0.0141048   -0.0908264  0.0314256   -1.0634875  0.0905958  0.2097171  0.3271208 0.2087   0.1106 0.2315474
    18.55     -4.6022555  1.0579779  0.0140189   -0.0906770  0.0313998   -1.0642804  0.0907114  0.2099961  0.3273333 0.2087   0.1107 0.2316349
    18.60     -4.5999354  1.0577260  0.0139643   -0.0905460  0.0313709   -1.0650795  0.0908500  0.2102950  0.3275477 0.2088   0.1107 0.2317254
    18.65     -4.5975723  1.0574669  0.0139311   -0.0904195  0.0313357   -1.0658857  0.0909980  0.2106238  0.3277787 0.2088   0.1108 0.2318139
    18.70     -4.5951842  1.0572068  0.0138828   -0.0902919  0.0313065   -1.0667150  0.0911585  0.2109674  0.3280141 0.2088   0.1109 0.2319032
    18.75     -4.5928423  1.0569459  0.0138195   -0.0901605  0.0312806   -1.0675214  0.0913371  0.2113231  0.3282595 0.2089   0.1110 0.2319961
    18.80     -4.5905630  1.0566968  0.0137501   -0.0900262  0.0312559   -1.0683239  0.0915036  0.2116534  0.3284932 0.2089   0.1110 0.2320896
    18.85     -4.5882897  1.0564454  0.0136895   -0.0899062  0.0312338   -1.0691287  0.0916884  0.2120215  0.3287538 0.2090   0.1111 0.2321822
    18.90     -4.5859444  1.0561877  0.0136342   -0.0897936  0.0312124   -1.0699621  0.0918883  0.2124011  0.3290464 0.2090   0.1111 0.2322761
    18.95     -4.5835739  1.0559358  0.0135791   -0.0896816  0.0311921   -1.0708360  0.0921012  0.2127846  0.3293473 0.2091   0.1112 0.2323750
    19.00     -4.5811854  1.0556836  0.0135058   -0.0895712  0.0311799   -1.0717170  0.0923031  0.2131652  0.3296134 0.2092   0.1113 0.2324712
    19.05     -4.5785562  1.0553965  0.0134393   -0.0894466  0.0311563   -1.0725925  0.0924814  0.2135224  0.3298248 0.2092   0.1113 0.2325693
    19.10     -4.5759216  1.0551149  0.0133694   -0.0893173  0.0311335   -1.0734791  0.0926409  0.2138565  0.3300005 0.2093   0.1114 0.2326680
    19.15     -4.5733085  1.0548385  0.0133009   -0.0891829  0.0311088   -1.0743627  0.0927815  0.2141700  0.3301508 0.2093   0.1115 0.2327696
    19.20     -4.5707328  1.0545735  0.0132251   -0.0890475  0.0310864   -1.0752708  0.0929357  0.2144811  0.3303240 0.2094   0.1115 0.2328733
    19.25     -4.5681581  1.0543094  0.0131417   -0.0889108  0.0310661   -1.0761907  0.0931160  0.2148037  0.3305288 0.2094   0.1116 0.2329716
    19.30     -4.5655769  1.0540419  0.0130449   -0.0887642  0.0310480   -1.0771128  0.0933079  0.2151399  0.3307210 0.2095   0.1117 0.2330717
    19.35     -4.5629537  1.0537768  0.0129323   -0.0886151  0.0310358   -1.0780591  0.0934791  0.2154531  0.3308830 0.2096   0.1117 0.2331716
    19.40     -4.5603791  1.0535132  0.0128234   -0.0884418  0.0310120   -1.0789847  0.0936696  0.2157718  0.3310389 0.2096   0.1118 0.2332786
    19.45     -4.5579092  1.0532598  0.0127360   -0.0882808  0.0309862   -1.0798843  0.0938509  0.2160907  0.3312164 0.2097   0.1118 0.2333875
    19.50     -4.5554637  1.0530066  0.0126536   -0.0881226  0.0309604   -1.0807806  0.0940355  0.2164319  0.3314249 0.2098   0.1118 0.2334919
    19.55     -4.5530926  1.0527660  0.0125638   -0.0879671  0.0309387   -1.0816819  0.0942248  0.2167771  0.3316603 0.2098   0.1119 0.2335976
    19.60     -4.5507698  1.0525329  0.0124638   -0.0878078  0.0309185   -1.0825853  0.0944184  0.2171272  0.3319116 0.2099   0.1119 0.2337101
    19.65     -4.5484820  1.0523011  0.0123580   -0.0876287  0.0308943   -1.0834763  0.0946293  0.2174838  0.3321520 0.2100   0.1120 0.2338338
    19.70     -4.5461606  1.0520732  0.0122562   -0.0874555  0.0308706   -1.0843841  0.0948125  0.2178082  0.3323524 0.2101   0.1120 0.2339553
    19.75     -4.5438000  1.0518500  0.0121600   -0.0872900  0.0308500   -1.0853000  0.0949700  0.2181100  0.3325200 0.2102   0.1120 0.2340828
    19.80     -4.5413705  1.0516076  0.0120795   -0.0871289  0.0308198   -1.0862221  0.0951415  0.2184096  0.3326915 0.2103   0.1121 0.2342155
    19.85     -4.5390661  1.0513886  0.0119939   -0.0869706  0.0307932   -1.0871300  0.0953188  0.2187053  0.3328751 0.2104   0.1121 0.2343419
    19.90     -4.5370396  1.0512291  0.0118930   -0.0868176  0.0307737   -1.0880405  0.0954304  0.2189416  0.3330119 0.2105   0.1122 0.2344619
    19.95     -4.5350267  1.0510743  0.0117816   -0.0866590  0.0307559   -1.0889565  0.0955338  0.2191705  0.3331256 0.2106   0.1122 0.2345780
    20.00     -4.5330479  1.0509283  0.0116658   -0.0865082  0.0307421   -1.0898739  0.0956245  0.2193853  0.3332079 0.2107   0.1122 0.2346923
    """)


class CauzziFaccioli2008SWISS04(CauzziFaccioli2008SWISS01):

    """
    This class extends 
    :class:`openquake.hazardlib.gsim.cauzzi_faccioli_2008.CauzziFaccioli2008`, 
    following same strategy
    as for 
    :class:`openquake.hazardlib.gsim.cauzzi_faccioli_2008_swiss.CauzziFaccioli2008SWISS01`
    """
    COEFFS_FS_ROCK = COEFFS_FS_ROCK_SWISS04


class CauzziFaccioli2008SWISS08(CauzziFaccioli2008SWISS01):

    """
    This class extends 
    :class:`openquake.hazardlib.gsim.cauzzi_faccioli_2008.CauzziFaccioli2008`, 
    following same strategy
    as for :class:`openquake.hazardlib.gsim.cauzzi_faccioli_2008_swiss.CauzziFaccioli2008SWISS01`
    """
    COEFFS_FS_ROCK = COEFFS_FS_ROCK_SWISS08
