
--/ snowmobile note1 /---------------------------------------------------------
-- This is an example of a script-level wrap.
-- `snowmobile` will recognize this based on the facts that;
---- (a) It falls between the start and end pattern specified in `snowmobile.toml`
---- (b) Its second line begins with `__script__`
-- *Note: Point (a) above is the case for all tags mentioned below, so I'm going
--  to omit clarifying that in the other 'snowmobile notes' below.
-------------------------------------------------------------------------------
/*-
__script__
__name: Illustration of how `snowmobile.Script` parses a sql file and handles different
types of st / QA checks.
__author: Grant Murray
__authored-date: 2020-08-31
__description:
This script is an example .sql file built to:
1. Provide high-level context on ow `snowmobile` parses raw sql files and how it will generate statement
tags if user-defined tags don't exist
1. Demonstrate how to use single-line tags in order to run the simplest form of QA/verification that data
is being shaped how it's intended to be throughout a script/that new data has not entered a pipeline that
the original code did not account for and relationships have broken (i.e. verifying the uniqueness of a table
on a given field).
1. How multi-line statement tags can be used to generate documentation and even to pass `snowmobile` arguments
with which it will use to run validations on the results returned by the corresponding statement.
-*/


--/ snowmobile note2 /---------------------------------------------------------
-- This is an example of a statement-level wrap (as all except the script wrap are);
-- `snowmobile` will recognize this based on the facts that;
---- (a) There is a valid sql statement beginning on line 39.
---- (b) The first word-character directly preceding the start of the valid sql
----     matches the closing wrap as  specified in `snowmobile.toml`.
-------------------------------------------------------------------------------
/*-create-table~dummy_example1-*/
create or replace temp table snowmobile.dummy_table_1 as
select
  'dim1'
    as sample_dimension1
  ,'dim2'
    as sample_dimension2
  ,1
    as dummy_metric1
  ,null
    as omit_this_column_when_comparing
	,2
	  as dummy_metric2
  ,3
    as other_column_to_compare_with_a_different_pattern
	,current_timestamp()
		as also_omit_this_column_when_comparing;


--/ snowmobile note3 /---------------------------------------------------------
-- This is an example without a wrap
-- `snowmobile` will recognize that based on the facts that:
---- (a) It is a valid sql statement but does not match the criteria outlined above.

-- Additionally, `snowmobile` will recognize that:
---- (b) It is a 'select' statement
---- (c) That it is the 2nd statement in the script
-- As a result it will assign this a generic wrap of `select-data~2*`, with the
-- '*' indicating that the wrap was generated by `snowmobile` and not user-pr.
-------------------------------------------------------------------------------
select
	*
from snowmobile.dummy_table_1 a;

-- TODO: Figure out why this is causing an error
--/ snowmobile note3 /---------------------------------------------------------
-- This is an example of a multi-line `qa-diff` wrap
-- `snowmobile` will recognize that based on the facts that:
---- (a) The open/closing tags span multiple lines
---- (b) The wrap name once left-partitioned on '-' returns the value `qa-diff`

-- `snowmobile` will included all arguments within the wrap (partition on, end-index-at, etc)
-- and will default to what's specified in `snowmobile.toml` if there are required arguments
-- that aren't specified here.

-- If the combination of what's supplied within the in-script wrap and the
-- configuration file don't result in a valid set of arguments given the table
-- for `snowmobile` to run the validation with (partition-on column doesn't exist
-- within the table, etc), it will raise an error and the script will stop executing
-- at this point regardless of whether or not the data that's returned from the
-- subsequent sql would result in a valid outcome if the correct arguments were
-- passed.

-- This is the case to ensure that the checks put in place continue to run as your
-- script becomes deeper it get tougher to remember what level/in what form a given piece
-- of data is intended to be at; passing `skip_qa` or 'silence_qa' to
-- `Script.run()` will respectively result in the checks being skipped entirely or
-- the checks being run and their results piped to stdout.
--  :Note: In the case the latter, validations that fail will *not* break the script
--    and execution will continue; however, if the table that is provided to them cannot
--    be validated given the parameters, it *will* raise an exception and fail.
--    If you want to skip them entirely or only run certain types of st, see
--    the 'scopes` argument of `snowmobile.Script()`.

-- Statement types:
---- create
---- update
---- insert
---- select
---- qa-diff
---- qa-empty
---- generic

-------------------------------------------------------------------------------
/*-
__name: qa-diff~verify-snowmobile.dummy1-ties to itself
__partition-on: 'src_description'
__end-index-at: 'sample_dimension2'
__compare-patterns: 'dummy_metric_.*','.*to_compare_.*'
__ignore-patterns: '.*_omit.*'
__description:
This is an illustration of how to use tags within a script for `snowmobile` to parse
as arguments to the validation based on the fact that its type is `qa-diff`.

Quick explanation of each for clarity:
- **partition-on: 'src_description'** tells `snowmobile` what column to partition the data on before
running the comparison; this could be omitted since it defaults to `src_description` but including here
illustrative purposes.
- **end-index-at: 'sample_dimension2'** tells `snowmobile` to use all columns between the first column
and the `sample_dimension2` column when joining the two sets of partitioned data back together
- **compare-patterns: 'dummy_metric.*','.*to_compare_.*'** is just a comma separated list of regex patterns
to use for isolation of the numeric columns to run run the comparison on. The last is only included to clarify
how you would feed multiple patterns into either the 'compare' or 'exclude' patterns arguments.
- **ignore-patterns: '.*_omit.*'** tells snowmobile the columns to throw away when running validation; this is because
often times tables are very wide and include metrics that have been pulled when it's time to start running validation
across a very wide table, some of which have no implications on if the two sets of data are or are not the same.
-*/
with union_of_two_datasets_we_know_are_identical as (
	  select
	    'Data1'
	      as src_description
	    ,d.*
	  from snowmobile.dummy_table_1 d
  union all
	  select
	    'Data2'
	      as src_description
	    ,d.*
	  from snowmobile.dummy_table_1 d
)
	select
		a.*
	from union_of_two_datasets_we_know_are_identical a;

/*-qa-empty~verifying that something we know to be distinct is indistinct (causing a failure)-*/
with an_intentionally_indistinct_field as (
	  select
	    d.*
	  from snowmobile.dummy_table_1 d
  union all
    select
      d.*
    from snowmobile.dummy_table_1 d
    where
      d.dummy_metric1 = 1
)
	select
		a.dummy_metric1
		,count(*)
			as count
	from an_intentionally_indistinct_field a
	group by 1
	having count(*) <> 1;
-- select
-- 	count(*)
-- from snowmobile.dummy_table_1 d;

/*-this is a wrap that doesn't tie to anything-*/

/*-this is another with its sql commented out-*/
-- select 1;

/*-what about commented out sql with two semicolons in it-*/
--select;
-- ;

/*-create-table~testing-if-sqlparse-just-splits-on-semicolons-*/
select
	';' as text_field_with_a_semicolon
	,1 as dummy_numeric_field;


select 1 as testing_a_single_line_statement_without_a_tag;


create or replace transient table test_transient_create_without_a_tag as
select
	*
from a_dummy_table_that_doesnt_exist;

-- create or replace transient table test_transient_create_without_a_tag_and_with_cte as with
-- sample_cte as (
--   select 1
-- )
-- 	select
-- 		*
-- 	from sample_cte;

/*-select-data~testing a statement without a semicolon-*/
-- select
-- 	*
-- from snowmobile.dummy_table_1 d
