# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ruff_lsp']

package_data = \
{'': ['*']}

install_requires = \
['pygls>1.0.0a3', 'ruff>0.0.150', 'typing_extensions']

entry_points = \
{'console_scripts': ['ruff-lsp = ruff_lsp.__main__:main']}

setup_kwargs = {
    'name': 'ruff-lsp',
    'version': '0.0.5',
    'description': 'A Language Server Protocol implementation for Ruff.',
    'long_description': '# ruff-lsp\n\nA [Language Server Protocol](https://microsoft.github.io/language-server-protocol/) implementation for\n[Ruff](https://github.com/charliermarsh/ruff), an extremely fast Python linter and code transformation\ntool, written in Rust.\n\nEnables Ruff to be used in any editor that supports the LSP, including [Neovim](#example-neovim),\n[Sublime Text](#example-sublime-text), Emacs and more.\n\nFor Visual Studio Code, check out the [Ruff VS Code extension](https://github.com/charliermarsh/ruff-vscode).\n\n## Highlights\n\n### "Quick Fix" actions for auto-fixable violations (like unused imports)\n\n![](https://user-images.githubusercontent.com/1309177/205176932-44cfc03a-120f-4bad-b710-612bdd7765d6.gif)\n\n### "Fix all": automatically fix all auto-fixable violations\n\n![](https://user-images.githubusercontent.com/1309177/205175763-cf34871d-5c05-4abf-9916-440afc82dbf8.gif)\n\n### "Organize Imports": `isort`-compatible import sorting\n\n![](https://user-images.githubusercontent.com/1309177/205175987-82e23e21-14bb-467d-9ef0-027f24b75865.gif)\n\n## Installation and Usage\n\n`ruff-lsp` is available as [`ruff-lsp`](https://pypi.org/project/ruff-lsp/) on PyPI:\n\n```shell\npip install ruff-lsp\n```\n\nFrom there, `ruff-lsp` can be used with any editor that supports the Language Server Protocol,\nincluding Neovim, Emacs, Sublime Text, and more.\n\n### Example: Neovim\n\nFor example, to use `ruff-lsp` with Neovim, install `ruff-lsp` from PyPI along with\n[`nvim-lspconfig`](https://github.com/neovim/nvim-lspconfig). Then, add something like the following\nto your `init.lua`:\n\n```lua\n-- See: https://github.com/neovim/nvim-lspconfig/tree/54eb2a070a4f389b1be0f98070f81d23e2b1a715#suggested-configuration\nlocal opts = { noremap=true, silent=true }\nvim.keymap.set(\'n\', \'<space>e\', vim.diagnostic.open_float, opts)\nvim.keymap.set(\'n\', \'[d\', vim.diagnostic.goto_prev, opts)\nvim.keymap.set(\'n\', \']d\', vim.diagnostic.goto_next, opts)\nvim.keymap.set(\'n\', \'<space>q\', vim.diagnostic.setloclist, opts)\n\n-- Use an on_attach function to only map the following keys\n-- after the language server attaches to the current buffer\nlocal on_attach = function(client, bufnr)\n  -- Enable completion triggered by <c-x><c-o>\n  vim.api.nvim_buf_set_option(bufnr, \'omnifunc\', \'v:lua.vim.lsp.omnifunc\')\n\n  -- Mappings.\n  -- See `:help vim.lsp.*` for documentation on any of the below functions\n  local bufopts = { noremap=true, silent=true, buffer=bufnr }\n  vim.keymap.set(\'n\', \'gD\', vim.lsp.buf.declaration, bufopts)\n  vim.keymap.set(\'n\', \'gd\', vim.lsp.buf.definition, bufopts)\n  vim.keymap.set(\'n\', \'K\', vim.lsp.buf.hover, bufopts)\n  vim.keymap.set(\'n\', \'gi\', vim.lsp.buf.implementation, bufopts)\n  vim.keymap.set(\'n\', \'<C-k>\', vim.lsp.buf.signature_help, bufopts)\n  vim.keymap.set(\'n\', \'<space>wa\', vim.lsp.buf.add_workspace_folder, bufopts)\n  vim.keymap.set(\'n\', \'<space>wr\', vim.lsp.buf.remove_workspace_folder, bufopts)\n  vim.keymap.set(\'n\', \'<space>wl\', function()\n    print(vim.inspect(vim.lsp.buf.list_workspace_folders()))\n  end, bufopts)\n  vim.keymap.set(\'n\', \'<space>D\', vim.lsp.buf.type_definition, bufopts)\n  vim.keymap.set(\'n\', \'<space>rn\', vim.lsp.buf.rename, bufopts)\n  vim.keymap.set(\'n\', \'<space>ca\', vim.lsp.buf.code_action, bufopts)\n  vim.keymap.set(\'n\', \'gr\', vim.lsp.buf.references, bufopts)\n  vim.keymap.set(\'n\', \'<space>f\', function() vim.lsp.buf.format { async = true } end, bufopts)\nend\n\n-- Configure `ruff-lsp`.\nlocal configs = require \'lspconfig.configs\'\nif not configs.ruff_lsp then\n  configs.ruff_lsp = {\n    default_config = {\n    cmd = { "ruff-lsp" },\n    filetypes = {\'python\'},\n    root_dir = require(\'lspconfig\').util.find_git_ancestor,\n    settings = {\n      ruff_lsp = {\n        -- Any extra CLI arguments for `ruff` go here.\n        args = {}\n      }\n    }\n  }\n}\nend\n\nrequire(\'lspconfig\').ruff_lsp.setup {\n  on_attach = on_attach,\n}\n```\n\nUpon successful installation, you should see Ruff\'s diagnostics surfaced directly in your editor:\n\n![Code Actions available in Neovim](https://user-images.githubusercontent.com/1309177/208278707-25fa37e4-079d-4597-ad35-b95dba066960.png)\n\n### Example: Sublime Text\n\nTo use `ruff-lsp` with Sublime Text, install Sublime Text\'s [LSP](https://github.com/sublimelsp/LSP)\npackage, then add something like the following to `LSP.sublime-settings`:\n\n```json\n{\n  "clients": {\n    "python-lsp-server": {\n      "command": ["ruff-lsp"],\n      "enabled": true,\n      "selector": "source.python"\n    }\n  }\n}\n```\n\nUpon successful installation, you should see errors surfaced directly in your editor:\n\n![](https://user-images.githubusercontent.com/1309177/208266375-331ad8e5-8ac1-4735-bca8-07734eb38536.png)\n\n## Settings\n\nThe exact mechanism by which settings will be passed to `ruff-lsp` will vary by editor. However,\nthe following settings are supported:\n\n| Settings         | Default | Description                                                                            |\n|------------------|---------|----------------------------------------------------------------------------------------|\n| args             | `[]`    | Custom arguments passed to `ruff`. E.g `"args": ["--config=/path/to/pyproject.toml"]`. |\n| logLevel         | `error` | Sets the tracing level for the extension.                                              |\n| path             | `[]`    | Setting to provide custom `ruff` executable. E.g. `["~/global_env/ruff"]`.             |\n| interpreter      | `[]`    | Path to a Python interpreter to use to run the linter server.                          |\n| showNotification | `off`   | Setting to control when a notification is shown.                                       |\n\n## Development\n\n`ruff-lsp` uses Poetry for environment management and packaging. To get started, clone the\nrepository, install Poetry, and run `poetry install`.\n\nTo automatically format the codebase, run: `make format`.\n\nTo run lint and type checks, run: `make check`.\n\n## License\n\nMIT\n',
    'author': 'Charlie Marsh',
    'author_email': 'charlie.r.marsh@gmail.com',
    'maintainer': 'Charlie Marsh',
    'maintainer_email': 'charlie.r.marsh@gmail.com',
    'url': 'https://github.com/charliermarsh/ruff-lsp',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<3.12',
}


setup(**setup_kwargs)
