"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = exports.PlatformCapabilities = exports.LogDriver = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * The log driver to use for the container.
 */
var LogDriver;
(function (LogDriver) {
    /**
     * Specifies the Amazon CloudWatch Logs logging driver.
     */
    LogDriver["AWSLOGS"] = "awslogs";
    /**
     * Specifies the Fluentd logging driver.
     */
    LogDriver["FLUENTD"] = "fluentd";
    /**
     * Specifies the Graylog Extended Format (GELF) logging driver.
     */
    LogDriver["GELF"] = "gelf";
    /**
     * Specifies the journald logging driver.
     */
    LogDriver["JOURNALD"] = "journald";
    /**
     * Specifies the logentries logging driver.
     */
    LogDriver["LOGENTRIES"] = "logentries";
    /**
     * Specifies the JSON file logging driver.
     */
    LogDriver["JSON_FILE"] = "json-file";
    /**
     * Specifies the Splunk logging driver.
     */
    LogDriver["SPLUNK"] = "splunk";
    /**
     * Specifies the syslog logging driver.
     */
    LogDriver["SYSLOG"] = "syslog";
})(LogDriver = exports.LogDriver || (exports.LogDriver = {}));
/**
 * Platform capabilities
 */
var PlatformCapabilities;
(function (PlatformCapabilities) {
    /**
     * Specifies EC2 environment.
     */
    PlatformCapabilities["EC2"] = "EC2";
    /**
     * Specifies Fargate environment.
     */
    PlatformCapabilities["FARGATE"] = "FARGATE";
})(PlatformCapabilities = exports.PlatformCapabilities || (exports.PlatformCapabilities = {}));
/**
 * Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 */
class JobDefinition extends core_1.Resource {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_batch_JobDefinitionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.validateProps(props);
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const isFargate = !!((_b = props.platformCapabilities) === null || _b === void 0 ? void 0 : _b.includes(PlatformCapabilities.FARGATE));
        const jobDef = new batch_generated_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container, isFargate),
            type: 'container',
            nodeProperties: props.nodeProps
                ? {
                    mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps, isFargate),
                    numNodes: props.nodeProps.count,
                }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
            platformCapabilities: (_c = props.platformCapabilities) !== null && _c !== void 0 ? _c : [PlatformCapabilities.EC2],
        });
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * Imports an existing batch job definition by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobDefinitionArn
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = core_1.Stack.of(scope);
        const jobDefName = stack.splitArn(jobDefinitionArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports an existing batch job definition by its name.
     * If name is specified without a revision then the latest active revision is used.
     *
     * @param scope
     * @param id
     * @param jobDefinitionName
     */
    static fromJobDefinitionName(scope, id, jobDefinitionName) {
        const stack = core_1.Stack.of(scope);
        const jobDefArn = stack.formatArn({
            service: 'batch',
            resource: 'job-definition',
            arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: jobDefinitionName,
        });
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefArn;
                this.jobDefinitionName = jobDefinitionName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return undefined;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    /**
     * Validates the properties provided for a new job definition.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.executionRole === undefined) {
            throw new Error('Fargate job must have executionRole set');
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.gpuCount !== undefined) {
            throw new Error('Fargate job must not have gpuCount set');
        }
        if ((props.platformCapabilities === undefined || props.platformCapabilities.includes(PlatformCapabilities.EC2))
            && props.container.assignPublicIp !== undefined) {
            throw new Error('EC2 job must not have assignPublicIp set');
        }
    }
    buildJobContainer(container, isFargate) {
        if (container === undefined) {
            return undefined;
        }
        return {
            command: container.command,
            environment: this.deserializeEnvVariables(container.environment),
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            executionRoleArn: container.executionRole && container.executionRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            logConfiguration: container.logConfiguration ? {
                logDriver: container.logConfiguration.logDriver,
                options: container.logConfiguration.options,
                secretOptions: container.logConfiguration.secretOptions
                    ? this.buildLogConfigurationSecretOptions(container.logConfiguration.secretOptions)
                    : undefined,
            } : undefined,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            networkConfiguration: container.assignPublicIp ? {
                assignPublicIp: container.assignPublicIp ? 'ENABLED' : 'DISABLED',
            } : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            volumes: container.volumes,
            fargatePlatformConfiguration: container.platformVersion ? {
                platformVersion: container.platformVersion,
            } : undefined,
            resourceRequirements: [
                { type: 'VCPU', value: String(container.vcpus || (isFargate ? 0.25 : 1)) },
                { type: 'MEMORY', value: String(container.memoryLimitMiB || (isFargate ? 512 : 4)) },
            ].concat(container.gpuCount ? [{ type: 'GPU', value: String(container.gpuCount) }] : []),
        };
    }
    buildNodeRangeProps(multiNodeProps, isFargate) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container, isFargate),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
    buildLogConfigurationSecretOptions(secretOptions) {
        return secretOptions.map(secretOption => {
            return {
                name: secretOption.optionName,
                valueFrom: secretOption.secretArn,
            };
        });
    }
}
exports.JobDefinition = JobDefinition;
_a = JSII_RTTI_SYMBOL_1;
JobDefinition[_a] = { fqn: "@aws-cdk/aws-batch.JobDefinition", version: "1.156.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiam9iLWRlZmluaXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJqb2ItZGVmaW5pdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFHQSx3Q0FBZ0Y7QUFFaEYsdURBQXFEO0FBRXJELCtFQUF5RTtBQUV6RTs7R0FFRztBQUNILElBQVksU0F3Q1g7QUF4Q0QsV0FBWSxTQUFTO0lBQ25COztPQUVHO0lBQ0gsZ0NBQW1CLENBQUE7SUFFbkI7O09BRUc7SUFDSCxnQ0FBbUIsQ0FBQTtJQUVuQjs7T0FFRztJQUNILDBCQUFhLENBQUE7SUFFYjs7T0FFRztJQUNILGtDQUFxQixDQUFBO0lBRXJCOztPQUVHO0lBQ0gsc0NBQXlCLENBQUE7SUFFekI7O09BRUc7SUFDSCxvQ0FBdUIsQ0FBQTtJQUV2Qjs7T0FFRztJQUNILDhCQUFpQixDQUFBO0lBRWpCOztPQUVHO0lBQ0gsOEJBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQXhDVyxTQUFTLEdBQVQsaUJBQVMsS0FBVCxpQkFBUyxRQXdDcEI7QUFFRDs7R0FFRztBQUNILElBQVksb0JBVVg7QUFWRCxXQUFZLG9CQUFvQjtJQUM5Qjs7T0FFRztJQUNILG1DQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILDJDQUFtQixDQUFBO0FBQ3JCLENBQUMsRUFWVyxvQkFBb0IsR0FBcEIsNEJBQW9CLEtBQXBCLDRCQUFvQixRQVUvQjtBQThTRDs7OztHQUlHO0FBQ0gsTUFBYSxhQUFjLFNBQVEsZUFBUTtJQWlEekMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF5Qjs7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtTQUN0QyxDQUFDLENBQUM7Ozs7Ozs7Ozs7UUFFSCxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRTFCLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxzREFBd0IsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRXZFLE1BQU0sU0FBUyxHQUFHLENBQUMsUUFBQyxLQUFLLENBQUMsb0JBQW9CLDBDQUFFLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPLEVBQUMsQ0FBQztRQUV2RixNQUFNLE1BQU0sR0FBRyxJQUFJLGtDQUFnQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDcEQsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtZQUMxQyxtQkFBbUIsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUM7WUFDdkUsSUFBSSxFQUFFLFdBQVc7WUFDakIsY0FBYyxFQUFFLEtBQUssQ0FBQyxTQUFTO2dCQUM3QixDQUFDLENBQUM7b0JBQ0EsUUFBUSxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsUUFBUTtvQkFDbEMsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDO29CQUN6RSxRQUFRLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxLQUFLO2lCQUNoQztnQkFDRCxDQUFDLENBQUMsU0FBUztZQUNiLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixhQUFhLEVBQUU7Z0JBQ2IsUUFBUSxFQUFFLEtBQUssQ0FBQyxhQUFhLElBQUksQ0FBQzthQUNuQztZQUNELE9BQU8sRUFBRTtnQkFDUCxzQkFBc0IsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO2FBQzlFO1lBQ0Qsb0JBQW9CLFFBQUUsS0FBSyxDQUFDLG9CQUFvQixtQ0FBSSxDQUFDLG9CQUFvQixDQUFDLEdBQUcsQ0FBQztTQUMvRSxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDL0QsT0FBTyxFQUFFLE9BQU87WUFDaEIsUUFBUSxFQUFFLGdCQUFnQjtZQUMxQixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDaEMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7S0FDcEU7SUF0RkQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLG9CQUFvQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGdCQUF3QjtRQUN2RixNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLEVBQUUsZ0JBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLFlBQWEsQ0FBQztRQUVqRyxNQUFNLE1BQU8sU0FBUSxlQUFRO1lBQTdCOztnQkFDa0IscUJBQWdCLEdBQUcsZ0JBQWdCLENBQUM7Z0JBQ3BDLHNCQUFpQixHQUFHLFVBQVUsQ0FBQztZQUNqRCxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztLQUM5QjtJQUVEOzs7Ozs7O09BT0c7SUFDSSxNQUFNLENBQUMscUJBQXFCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsaUJBQXlCO1FBQ3pGLE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDOUIsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUNoQyxPQUFPLEVBQUUsT0FBTztZQUNoQixRQUFRLEVBQUUsZ0JBQWdCO1lBQzFCLFNBQVMsRUFBRSxnQkFBUyxDQUFDLG1CQUFtQjtZQUN4QyxZQUFZLEVBQUUsaUJBQWlCO1NBQ2hDLENBQUMsQ0FBQztRQUVILE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNrQixxQkFBZ0IsR0FBRyxTQUFTLENBQUM7Z0JBQzdCLHNCQUFpQixHQUFHLGlCQUFpQixDQUFDO1lBQ3hELENBQUM7U0FBQTtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQzlCO0lBOENPLHVCQUF1QixDQUFDLEdBQWdDO1FBQzlELE1BQU0sSUFBSSxHQUFHLElBQUksS0FBSyxFQUF3QyxDQUFDO1FBRS9ELElBQUksR0FBRyxLQUFLLFNBQVMsRUFBRTtZQUNyQixPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBWSxFQUFFLEVBQUU7WUFDcEMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUN4QyxDQUFDLENBQUMsQ0FBQztRQUVILE9BQU8sSUFBSSxDQUFDO0tBQ2I7SUFFRDs7T0FFRztJQUNLLGFBQWEsQ0FBQyxLQUF5QjtRQUM3QyxJQUFJLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDdkIsT0FBTztTQUNSO1FBRUQsSUFBSSxLQUFLLENBQUMsb0JBQW9CLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxRQUFRLENBQUMsb0JBQW9CLENBQUMsT0FBTyxDQUFDO2VBQzVHLEtBQUssQ0FBQyxTQUFTLENBQUMsYUFBYSxLQUFLLFNBQVMsRUFBRTtZQUNoRCxNQUFNLElBQUksS0FBSyxDQUFDLHlDQUF5QyxDQUFDLENBQUM7U0FDNUQ7UUFFRCxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLG9CQUFvQixDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPLENBQUM7ZUFDNUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxRQUFRLEtBQUssU0FBUyxFQUFFO1lBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsd0NBQXdDLENBQUMsQ0FBQztTQUMzRDtRQUVELElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxRQUFRLENBQUMsb0JBQW9CLENBQUMsR0FBRyxDQUFDLENBQUM7ZUFDMUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxjQUFjLEtBQUssU0FBUyxFQUFFO1lBQ2pELE1BQU0sSUFBSSxLQUFLLENBQUMsMENBQTBDLENBQUMsQ0FBQztTQUM3RDtLQUNGO0lBRU8saUJBQWlCLENBQUMsU0FBaUMsRUFBRSxTQUFrQjtRQUM3RSxJQUFJLFNBQVMsS0FBSyxTQUFTLEVBQUU7WUFDM0IsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFFRCxPQUFPO1lBQ0wsT0FBTyxFQUFFLFNBQVMsQ0FBQyxPQUFPO1lBQzFCLFdBQVcsRUFBRSxJQUFJLENBQUMsdUJBQXVCLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQztZQUNoRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxTQUFTO1lBQ2pDLFlBQVksRUFBRSxTQUFTLENBQUMsWUFBWSxJQUFJLFNBQVMsQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFO1lBQ3pFLFVBQVUsRUFBRSxTQUFTLENBQUMsT0FBTyxJQUFJLFNBQVMsQ0FBQyxPQUFPLENBQUMsT0FBTztZQUMxRCxnQkFBZ0IsRUFBRSxTQUFTLENBQUMsYUFBYSxJQUFJLFNBQVMsQ0FBQyxhQUFhLENBQUMsT0FBTztZQUM1RSxlQUFlLEVBQUUsU0FBUyxDQUFDLFdBQVc7Z0JBQ3BDLENBQUMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxTQUFTLENBQUMsV0FBVyxDQUFDLHFCQUFxQixFQUFFLENBQUMsT0FBTyxFQUFFO2dCQUNwRSxDQUFDLENBQUMsU0FBUztZQUNiLGdCQUFnQixFQUFFLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUM7Z0JBQzdDLFNBQVMsRUFBRSxTQUFTLENBQUMsZ0JBQWdCLENBQUMsU0FBUztnQkFDL0MsT0FBTyxFQUFFLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPO2dCQUMzQyxhQUFhLEVBQUUsU0FBUyxDQUFDLGdCQUFnQixDQUFDLGFBQWE7b0JBQ3JELENBQUMsQ0FBQyxJQUFJLENBQUMsa0NBQWtDLENBQUMsU0FBUyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQztvQkFDbkYsQ0FBQyxDQUFDLFNBQVM7YUFDZCxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2IsV0FBVyxFQUFFLFNBQVMsQ0FBQyxXQUFXO1lBQ2xDLFVBQVUsRUFBRSxTQUFTLENBQUMsVUFBVSxJQUFJLEtBQUs7WUFDekMsb0JBQW9CLEVBQUUsU0FBUyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUM7Z0JBQy9DLGNBQWMsRUFBRSxTQUFTLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFVBQVU7YUFDbEUsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNiLHNCQUFzQixFQUFFLFNBQVMsQ0FBQyxRQUFRLElBQUksS0FBSztZQUNuRCxPQUFPLEVBQUUsU0FBUyxDQUFDLE9BQU87WUFDMUIsSUFBSSxFQUFFLFNBQVMsQ0FBQyxJQUFJO1lBQ3BCLE9BQU8sRUFBRSxTQUFTLENBQUMsT0FBTztZQUMxQiw0QkFBNEIsRUFBRSxTQUFTLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQztnQkFDeEQsZUFBZSxFQUFFLFNBQVMsQ0FBQyxlQUFlO2FBQzNDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDYixvQkFBb0IsRUFBRTtnQkFDcEIsRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQUssSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUMxRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxTQUFTLENBQUMsY0FBYyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7YUFDckYsQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7U0FDekYsQ0FBQztLQUNIO0lBRU8sbUJBQW1CLENBQUMsY0FBK0IsRUFBRSxTQUFrQjtRQUM3RSxNQUFNLFVBQVUsR0FBRyxJQUFJLEtBQUssRUFBOEMsQ0FBQztRQUUzRSxLQUFLLE1BQU0sSUFBSSxJQUFJLGNBQWMsQ0FBQyxVQUFVLEVBQUU7WUFDNUMsVUFBVSxDQUFDLElBQUksQ0FBQztnQkFDZCxTQUFTLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDO2dCQUM1RCxXQUFXLEVBQUUsR0FBRyxJQUFJLENBQUMsYUFBYSxJQUFJLENBQUMsSUFBSSxJQUFJLENBQUMsV0FBVyxJQUFJLGNBQWMsQ0FBQyxLQUFLLEVBQUU7YUFDdEYsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxPQUFPLFVBQVUsQ0FBQztLQUNuQjtJQUVPLGtDQUFrQyxDQUFDLGFBQThCO1FBQ3ZFLE9BQU8sYUFBYSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsRUFBRTtZQUN0QyxPQUFPO2dCQUNMLElBQUksRUFBRSxZQUFZLENBQUMsVUFBVTtnQkFDN0IsU0FBUyxFQUFFLFlBQVksQ0FBQyxTQUFTO2FBQ2xDLENBQUM7UUFDSixDQUFDLENBQUMsQ0FBQztLQUNKOztBQTVMSCxzQ0E2TEMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBlY3MgZnJvbSAnQGF3cy1jZGsvYXdzLWVjcyc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBBcm5Gb3JtYXQsIER1cmF0aW9uLCBJUmVzb3VyY2UsIFJlc291cmNlLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5Kb2JEZWZpbml0aW9uIH0gZnJvbSAnLi9iYXRjaC5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgRXhwb3NlZFNlY3JldCB9IGZyb20gJy4vZXhwb3NlZC1zZWNyZXQnO1xuaW1wb3J0IHsgSm9iRGVmaW5pdGlvbkltYWdlQ29uZmlnIH0gZnJvbSAnLi9qb2ItZGVmaW5pdGlvbi1pbWFnZS1jb25maWcnO1xuXG4vKipcbiAqIFRoZSBsb2cgZHJpdmVyIHRvIHVzZSBmb3IgdGhlIGNvbnRhaW5lci5cbiAqL1xuZXhwb3J0IGVudW0gTG9nRHJpdmVyIHtcbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgQW1hem9uIENsb3VkV2F0Y2ggTG9ncyBsb2dnaW5nIGRyaXZlci5cbiAgICovXG4gIEFXU0xPR1MgPSAnYXdzbG9ncycsXG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgRmx1ZW50ZCBsb2dnaW5nIGRyaXZlci5cbiAgICovXG4gIEZMVUVOVEQgPSAnZmx1ZW50ZCcsXG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgR3JheWxvZyBFeHRlbmRlZCBGb3JtYXQgKEdFTEYpIGxvZ2dpbmcgZHJpdmVyLlxuICAgKi9cbiAgR0VMRiA9ICdnZWxmJyxcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBqb3VybmFsZCBsb2dnaW5nIGRyaXZlci5cbiAgICovXG4gIEpPVVJOQUxEID0gJ2pvdXJuYWxkJyxcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBsb2dlbnRyaWVzIGxvZ2dpbmcgZHJpdmVyLlxuICAgKi9cbiAgTE9HRU5UUklFUyA9ICdsb2dlbnRyaWVzJyxcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBKU09OIGZpbGUgbG9nZ2luZyBkcml2ZXIuXG4gICAqL1xuICBKU09OX0ZJTEUgPSAnanNvbi1maWxlJyxcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBTcGx1bmsgbG9nZ2luZyBkcml2ZXIuXG4gICAqL1xuICBTUExVTksgPSAnc3BsdW5rJyxcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBzeXNsb2cgbG9nZ2luZyBkcml2ZXIuXG4gICAqL1xuICBTWVNMT0cgPSAnc3lzbG9nJ1xufVxuXG4vKipcbiAqIFBsYXRmb3JtIGNhcGFiaWxpdGllc1xuICovXG5leHBvcnQgZW51bSBQbGF0Zm9ybUNhcGFiaWxpdGllcyB7XG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgRUMyIGVudmlyb25tZW50LlxuICAgKi9cbiAgRUMyID0gJ0VDMicsXG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyBGYXJnYXRlIGVudmlyb25tZW50LlxuICAgKi9cbiAgRkFSR0FURSA9ICdGQVJHQVRFJ1xufVxuXG4vKipcbiAqIExvZyBjb25maWd1cmF0aW9uIG9wdGlvbnMgdG8gc2VuZCB0byBhIGN1c3RvbSBsb2cgZHJpdmVyIGZvciB0aGUgY29udGFpbmVyLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIExvZ0NvbmZpZ3VyYXRpb24ge1xuICAvKipcbiAgICogVGhlIGxvZyBkcml2ZXIgdG8gdXNlIGZvciB0aGUgY29udGFpbmVyLlxuICAgKi9cbiAgcmVhZG9ubHkgbG9nRHJpdmVyOiBMb2dEcml2ZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBjb25maWd1cmF0aW9uIG9wdGlvbnMgdG8gc2VuZCB0byB0aGUgbG9nIGRyaXZlci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBjb25maWd1cmF0aW9uIG9wdGlvbnMgYXJlIHNlbnRcbiAgICovXG4gIHJlYWRvbmx5IG9wdGlvbnM/OiBhbnk7XG5cbiAgLyoqXG4gICAqIFRoZSBzZWNyZXRzIHRvIHBhc3MgdG8gdGhlIGxvZyBjb25maWd1cmF0aW9uIGFzIG9wdGlvbnMuXG4gICAqIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC91c2VyZ3VpZGUvc3BlY2lmeWluZy1zZW5zaXRpdmUtZGF0YS1zZWNyZXRzLmh0bWwjc2VjcmV0cy1sb2djb25maWdcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBzZWNyZXRzIGFyZSBwYXNzZWRcbiAgICovXG4gIHJlYWRvbmx5IHNlY3JldE9wdGlvbnM/OiBFeHBvc2VkU2VjcmV0W107XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBvZiBhIGpvYiBkZWZpbml0aW9uIGNvbnRhaW5lci5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBKb2JEZWZpbml0aW9uQ29udGFpbmVyIHtcbiAgLyoqXG4gICAqIFRoZSBjb21tYW5kIHRoYXQgaXMgcGFzc2VkIHRvIHRoZSBjb250YWluZXIuXG4gICAqXG4gICAqIElmIHlvdSBwcm92aWRlIGEgc2hlbGwgY29tbWFuZCBhcyBhIHNpbmdsZSBzdHJpbmcsIHlvdSBoYXZlIHRvIHF1b3RlIGNvbW1hbmQtbGluZSBhcmd1bWVudHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQ01EIHZhbHVlIGJ1aWx0IGludG8gY29udGFpbmVyIGltYWdlLlxuICAgKi9cbiAgcmVhZG9ubHkgY29tbWFuZD86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBUaGUgZW52aXJvbm1lbnQgdmFyaWFibGVzIHRvIHBhc3MgdG8gdGhlIGNvbnRhaW5lci5cbiAgICpcbiAgICogQGRlZmF1bHQgbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIC8qKlxuICAgKiBUaGUgaW1hZ2UgdXNlZCB0byBzdGFydCBhIGNvbnRhaW5lci5cbiAgICovXG4gIHJlYWRvbmx5IGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2U7XG5cbiAgLyoqXG4gICAqIFRoZSBpbnN0YW5jZSB0eXBlIHRvIHVzZSBmb3IgYSBtdWx0aS1ub2RlIHBhcmFsbGVsIGpvYi4gQ3VycmVudGx5IGFsbCBub2RlIGdyb3VwcyBpbiBhXG4gICAqIG11bHRpLW5vZGUgcGFyYWxsZWwgam9iIG11c3QgdXNlIHRoZSBzYW1lIGluc3RhbmNlIHR5cGUuIFRoaXMgcGFyYW1ldGVyIGlzIG5vdCB2YWxpZFxuICAgKiBmb3Igc2luZ2xlLW5vZGUgY29udGFpbmVyIGpvYnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlPzogZWMyLkluc3RhbmNlVHlwZTtcblxuICAvKipcbiAgICogVGhlIElBTSByb2xlIHRoYXQgdGhlIGNvbnRhaW5lciBjYW4gYXNzdW1lIGZvciBBV1MgcGVybWlzc2lvbnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQW4gSUFNIHJvbGUgd2lsbCBjcmVhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgam9iUm9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogTGludXgtc3BlY2lmaWMgbW9kaWZpY2F0aW9ucyB0aGF0IGFyZSBhcHBsaWVkIHRvIHRoZSBjb250YWluZXIsIHN1Y2ggYXMgZGV0YWlscyBmb3IgZGV2aWNlIG1hcHBpbmdzLlxuICAgKiBGb3Igbm93LCBvbmx5IHRoZSBgZGV2aWNlc2AgcHJvcGVydHkgaXMgc3VwcG9ydGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmUgd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgbGludXhQYXJhbXM/OiBlY3MuTGludXhQYXJhbWV0ZXJzO1xuXG4gIC8qKlxuICAgKiBUaGUgbG9nIGNvbmZpZ3VyYXRpb24gc3BlY2lmaWNhdGlvbiBmb3IgdGhlIGNvbnRhaW5lci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBjb250YWluZXJzIHVzZSB0aGUgc2FtZSBsb2dnaW5nIGRyaXZlciB0aGF0IHRoZSBEb2NrZXIgZGFlbW9uIHVzZXNcbiAgICovXG4gIHJlYWRvbmx5IGxvZ0NvbmZpZ3VyYXRpb24/OiBMb2dDb25maWd1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgaGFyZCBsaW1pdCAoaW4gTWlCKSBvZiBtZW1vcnkgdG8gcHJlc2VudCB0byB0aGUgY29udGFpbmVyLiBJZiB5b3VyIGNvbnRhaW5lciBhdHRlbXB0cyB0byBleGNlZWRcbiAgICogdGhlIG1lbW9yeSBzcGVjaWZpZWQgaGVyZSwgdGhlIGNvbnRhaW5lciBpcyBraWxsZWQuIFlvdSBtdXN0IHNwZWNpZnkgYXQgbGVhc3QgNCBNaUIgb2YgbWVtb3J5IGZvciBFQzIgYW5kIDUxMiBNaUIgZm9yIEZhcmdhdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gNCBmb3IgRUMyLCA1MTIgZm9yIEZhcmdhdGVcbiAgICovXG4gIHJlYWRvbmx5IG1lbW9yeUxpbWl0TWlCPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbW91bnQgcG9pbnRzIGZvciBkYXRhIHZvbHVtZXMgaW4geW91ciBjb250YWluZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gbW91bnQgcG9pbnRzIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IG1vdW50UG9pbnRzPzogZWNzLk1vdW50UG9pbnRbXTtcblxuICAvKipcbiAgICogV2hlbiB0aGlzIHBhcmFtZXRlciBpcyB0cnVlLCB0aGUgY29udGFpbmVyIGlzIGdpdmVuIGVsZXZhdGVkIHByaXZpbGVnZXMgb24gdGhlIGhvc3QgY29udGFpbmVyIGluc3RhbmNlIChzaW1pbGFyIHRvIHRoZSByb290IHVzZXIpLlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJpdmlsZWdlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFdoZW4gdGhpcyBwYXJhbWV0ZXIgaXMgdHJ1ZSwgdGhlIGNvbnRhaW5lciBpcyBnaXZlbiByZWFkLW9ubHkgYWNjZXNzIHRvIGl0cyByb290IGZpbGUgc3lzdGVtLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVhZE9ubHk/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIHBoeXNpY2FsIEdQVXMgdG8gcmVzZXJ2ZSBmb3IgdGhlIGNvbnRhaW5lci4gVGhlIG51bWJlciBvZiBHUFVzIHJlc2VydmVkIGZvciBhbGxcbiAgICogY29udGFpbmVycyBpbiBhIGpvYiBzaG91bGQgbm90IGV4Y2VlZCB0aGUgbnVtYmVyIG9mIGF2YWlsYWJsZSBHUFVzIG9uIHRoZSBjb21wdXRlIHJlc291cmNlIHRoYXQgdGhlIGpvYiBpcyBsYXVuY2hlZCBvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBHUFUgcmVzZXJ2YXRpb24uXG4gICAqL1xuICByZWFkb25seSBncHVDb3VudD86IG51bWJlcjtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIHVsaW1pdHMgdG8gc2V0IGluIHRoZSBjb250YWluZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gbGltaXRzLlxuICAgKi9cbiAgcmVhZG9ubHkgdWxpbWl0cz86IGVjcy5VbGltaXRbXTtcblxuICAvKipcbiAgICogVGhlIHVzZXIgbmFtZSB0byB1c2UgaW5zaWRlIHRoZSBjb250YWluZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZSB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSB1c2VyPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIHZDUFVzIHJlc2VydmVkIGZvciB0aGUgY29udGFpbmVyLiBFYWNoIHZDUFUgaXMgZXF1aXZhbGVudCB0b1xuICAgKiAxLDAyNCBDUFUgc2hhcmVzLiBZb3UgbXVzdCBzcGVjaWZ5IGF0IGxlYXN0IG9uZSB2Q1BVIGZvciBFQzIgYW5kIDAuMjUgZm9yIEZhcmdhdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gMSBmb3IgRUMyLCAwLjI1IGZvciBGYXJnYXRlXG4gICAqL1xuICByZWFkb25seSB2Y3B1cz86IG51bWJlcjtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIGRhdGEgdm9sdW1lcyB1c2VkIGluIGEgam9iLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGRhdGEgdm9sdW1lcyB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSB2b2x1bWVzPzogZWNzLlZvbHVtZVtdO1xuXG4gIC8qKlxuICAgKiBGYXJnYXRlIHBsYXRmb3JtIHZlcnNpb25cbiAgICpcbiAgICogQGRlZmF1bHQgLSBMQVRFU1QgcGxhdGZvcm0gdmVyc2lvbiB3aWxsIGJlIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IHBsYXRmb3JtVmVyc2lvbj86IGVjcy5GYXJnYXRlUGxhdGZvcm1WZXJzaW9uXG5cbiAgLyoqXG4gICAqIFRoZSBJQU0gcm9sZSB0aGF0IEFXUyBCYXRjaCBjYW4gYXNzdW1lLlxuICAgKiBSZXF1aXJlZCB3aGVuIHVzaW5nIEZhcmdhdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgZXhlY3V0aW9uUm9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdG8gYXNzaWduIGEgcHVibGljIElQIHRvIHRoZSBqb2JcbiAgICpcbiAgICogQGRlZmF1bHQgLSBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgYXNzaWduUHVibGljSXA/OiBib29sZWFuXG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgb2YgdGhlIHtAbGluayBKb2JEZWZpbml0aW9ufSBjb25zdHJ1Y3QuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSm9iRGVmaW5pdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBqb2IgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogVXAgdG8gMTI4IGxldHRlcnMgKHVwcGVyY2FzZSBhbmQgbG93ZXJjYXNlKSwgbnVtYmVycywgaHlwaGVucywgYW5kIHVuZGVyc2NvcmVzIGFyZSBhbGxvd2VkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBDbG91ZGZvcm1hdGlvbi1nZW5lcmF0ZWQgbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgam9iRGVmaW5pdGlvbk5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFuIG9iamVjdCB3aXRoIHZhcmlvdXMgcHJvcGVydGllcyBzcGVjaWZpYyB0byBjb250YWluZXItYmFzZWQgam9icy5cbiAgICovXG4gIHJlYWRvbmx5IGNvbnRhaW5lcjogSm9iRGVmaW5pdGlvbkNvbnRhaW5lcjtcblxuICAvKipcbiAgICogQW4gb2JqZWN0IHdpdGggdmFyaW91cyBwcm9wZXJ0aWVzIHNwZWNpZmljIHRvIG11bHRpLW5vZGUgcGFyYWxsZWwgam9icy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB1bmRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IG5vZGVQcm9wcz86IElNdWx0aU5vZGVQcm9wcztcblxuICAvKipcbiAgICogV2hlbiB5b3Ugc3VibWl0IGEgam9iLCB5b3UgY2FuIHNwZWNpZnkgcGFyYW1ldGVycyB0aGF0IHNob3VsZCByZXBsYWNlIHRoZVxuICAgKiBwbGFjZWhvbGRlcnMgb3Igb3ZlcnJpZGUgdGhlIGRlZmF1bHQgam9iIGRlZmluaXRpb24gcGFyYW1ldGVycy4gUGFyYW1ldGVyc1xuICAgKiBpbiBqb2Igc3VibWlzc2lvbiByZXF1ZXN0cyB0YWtlIHByZWNlZGVuY2Ugb3ZlciB0aGUgZGVmYXVsdHMgaW4gYSBqb2IgZGVmaW5pdGlvbi5cbiAgICogVGhpcyBhbGxvd3MgeW91IHRvIHVzZSB0aGUgc2FtZSBqb2IgZGVmaW5pdGlvbiBmb3IgbXVsdGlwbGUgam9icyB0aGF0IHVzZSB0aGUgc2FtZVxuICAgKiBmb3JtYXQsIGFuZCBwcm9ncmFtbWF0aWNhbGx5IGNoYW5nZSB2YWx1ZXMgaW4gdGhlIGNvbW1hbmQgYXQgc3VibWlzc2lvbiB0aW1lLlxuICAgKlxuICAgKiBAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L3VzZXJndWlkZS9qb2JfZGVmaW5pdGlvbl9wYXJhbWV0ZXJzLmh0bWxcbiAgICogQGRlZmF1bHQgLSB1bmRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IHBhcmFtZXRlcnM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIHRpbWVzIHRvIG1vdmUgYSBqb2IgdG8gdGhlIFJVTk5BQkxFIHN0YXR1cy4gWW91IG1heSBzcGVjaWZ5IGJldHdlZW4gMSBhbmRcbiAgICogMTAgYXR0ZW1wdHMuIElmIHRoZSB2YWx1ZSBvZiBhdHRlbXB0cyBpcyBncmVhdGVyIHRoYW4gb25lLCB0aGUgam9iIGlzIHJldHJpZWQgb24gZmFpbHVyZVxuICAgKiB0aGUgc2FtZSBudW1iZXIgb2YgYXR0ZW1wdHMgYXMgdGhlIHZhbHVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAxXG4gICAqL1xuICByZWFkb25seSByZXRyeUF0dGVtcHRzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgdGltZW91dCBjb25maWd1cmF0aW9uIGZvciBqb2JzIHRoYXQgYXJlIHN1Ym1pdHRlZCB3aXRoIHRoaXMgam9iIGRlZmluaXRpb24uIFlvdSBjYW4gc3BlY2lmeVxuICAgKiBhIHRpbWVvdXQgZHVyYXRpb24gYWZ0ZXIgd2hpY2ggQVdTIEJhdGNoIHRlcm1pbmF0ZXMgeW91ciBqb2JzIGlmIHRoZXkgaGF2ZSBub3QgZmluaXNoZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdW5kZWZpbmVkXG4gICAqL1xuICByZWFkb25seSB0aW1lb3V0PzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBwbGF0Zm9ybSBjYXBhYmlsaXRpZXMgcmVxdWlyZWQgYnkgdGhlIGpvYiBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEVDMlxuICAgKi9cbiAgcmVhZG9ubHkgcGxhdGZvcm1DYXBhYmlsaXRpZXM/OiBQbGF0Zm9ybUNhcGFiaWxpdGllc1tdO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIHNwZWNpZnlpbmcgbXVsdGktbm9kZSBwcm9wZXJ0aWVzIGZvciBjb21wdXRlIHJlc291cmNlcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJTXVsdGlOb2RlUHJvcHMge1xuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBub2RlIGluZGV4IGZvciB0aGUgbWFpbiBub2RlIG9mIGEgbXVsdGktbm9kZSBwYXJhbGxlbCBqb2IuIFRoaXMgbm9kZSBpbmRleCB2YWx1ZSBtdXN0IGJlIGZld2VyIHRoYW4gdGhlIG51bWJlciBvZiBub2Rlcy5cbiAgICovXG4gIG1haW5Ob2RlOiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBub2RlIHJhbmdlcyBhbmQgdGhlaXIgcHJvcGVydGllcyBhc3NvY2lhdGVkIHdpdGggYSBtdWx0aS1ub2RlIHBhcmFsbGVsIGpvYi5cbiAgICovXG4gIHJhbmdlUHJvcHM6IElOb2RlUmFuZ2VQcm9wc1tdO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIG5vZGVzIGFzc29jaWF0ZWQgd2l0aCBhIG11bHRpLW5vZGUgcGFyYWxsZWwgam9iLlxuICAgKi9cbiAgY291bnQ6IG51bWJlcjtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIG11bHRpLW5vZGUgYmF0Y2ggam9iLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElOb2RlUmFuZ2VQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgY29udGFpbmVyIGRldGFpbHMgZm9yIHRoZSBub2RlIHJhbmdlLlxuICAgKi9cbiAgY29udGFpbmVyOiBKb2JEZWZpbml0aW9uQ29udGFpbmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbWluaW11bSBub2RlIGluZGV4IHZhbHVlIHRvIGFwcGx5IHRoaXMgY29udGFpbmVyIGRlZmluaXRpb24gYWdhaW5zdC5cbiAgICpcbiAgICogWW91IG1heSBuZXN0IG5vZGUgcmFuZ2VzLCBmb3IgZXhhbXBsZSAwOjEwIGFuZCA0OjUsIGluIHdoaWNoIGNhc2UgdGhlIDQ6NSByYW5nZSBwcm9wZXJ0aWVzIG92ZXJyaWRlIHRoZSAwOjEwIHByb3BlcnRpZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IDBcbiAgICovXG4gIGZyb21Ob2RlSW5kZXg/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIG5vZGUgaW5kZXggdmFsdWUgdG8gYXBwbHkgdGhpcyBjb250YWluZXIgZGVmaW5pdGlvbiBhZ2FpbnN0LiBJZiBvbWl0dGVkLCB0aGUgaGlnaGVzdCB2YWx1ZSBpcyB1c2VkIHJlbGF0aXZlLlxuICAgKlxuICAgKiB0byB0aGUgbnVtYmVyIG9mIG5vZGVzIGFzc29jaWF0ZWQgd2l0aCB0aGUgam9iLiBZb3UgbWF5IG5lc3Qgbm9kZSByYW5nZXMsIGZvciBleGFtcGxlIDA6MTAgYW5kIDQ6NSxcbiAgICogaW4gd2hpY2ggY2FzZSB0aGUgNDo1IHJhbmdlIHByb3BlcnRpZXMgb3ZlcnJpZGUgdGhlIDA6MTAgcHJvcGVydGllcy5cbiAgICpcbiAgICogQGRlZmF1bHQge0BsaW5rIElNdWx0aU5vZGVwcm9wcy5jb3VudH1cbiAgICovXG4gIHRvTm9kZUluZGV4PzogbnVtYmVyO1xufVxuXG4vKipcbiAqIEFuIGludGVyZmFjZSByZXByZXNlbnRpbmcgYSBqb2IgZGVmaW5pdGlvbiAtIGVpdGhlciBhIG5ldyBvbmUsIGNyZWF0ZWQgd2l0aCB0aGUgQ0RLLCAqdXNpbmcgdGhlXG4gKiB7QGxpbmsgSm9iRGVmaW5pdGlvbn0gY2xhc3MsIG9yIGV4aXN0aW5nIG9uZXMsIHJlZmVyZW5jZWQgdXNpbmcgdGhlIHtAbGluayBKb2JEZWZpbml0aW9uLmZyb21Kb2JEZWZpbml0aW9uQXJufSBtZXRob2QuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUpvYkRlZmluaXRpb24gZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGlzIGJhdGNoIGpvYiBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBqb2JEZWZpbml0aW9uQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBiYXRjaCBqb2IgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgam9iRGVmaW5pdGlvbk5hbWU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBCYXRjaCBKb2IgRGVmaW5pdGlvbi5cbiAqXG4gKiBEZWZpbmVzIGEgYmF0Y2ggam9iIGRlZmluaXRpb24gdG8gZXhlY3V0ZSBhIHNwZWNpZmljIGJhdGNoIGpvYi5cbiAqL1xuZXhwb3J0IGNsYXNzIEpvYkRlZmluaXRpb24gZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElKb2JEZWZpbml0aW9uIHtcbiAgLyoqXG4gICAqIEltcG9ydHMgYW4gZXhpc3RpbmcgYmF0Y2ggam9iIGRlZmluaXRpb24gYnkgaXRzIGFtYXpvbiByZXNvdXJjZSBuYW1lLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGVcbiAgICogQHBhcmFtIGlkXG4gICAqIEBwYXJhbSBqb2JEZWZpbml0aW9uQXJuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Kb2JEZWZpbml0aW9uQXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGpvYkRlZmluaXRpb25Bcm46IHN0cmluZyk6IElKb2JEZWZpbml0aW9uIHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHNjb3BlKTtcbiAgICBjb25zdCBqb2JEZWZOYW1lID0gc3RhY2suc3BsaXRBcm4oam9iRGVmaW5pdGlvbkFybiwgQXJuRm9ybWF0LlNMQVNIX1JFU09VUkNFX05BTUUpLnJlc291cmNlTmFtZSE7XG5cbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElKb2JEZWZpbml0aW9uIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBqb2JEZWZpbml0aW9uQXJuID0gam9iRGVmaW5pdGlvbkFybjtcbiAgICAgIHB1YmxpYyByZWFkb25seSBqb2JEZWZpbml0aW9uTmFtZSA9IGpvYkRlZk5hbWU7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbXBvcnRzIGFuIGV4aXN0aW5nIGJhdGNoIGpvYiBkZWZpbml0aW9uIGJ5IGl0cyBuYW1lLlxuICAgKiBJZiBuYW1lIGlzIHNwZWNpZmllZCB3aXRob3V0IGEgcmV2aXNpb24gdGhlbiB0aGUgbGF0ZXN0IGFjdGl2ZSByZXZpc2lvbiBpcyB1c2VkLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGVcbiAgICogQHBhcmFtIGlkXG4gICAqIEBwYXJhbSBqb2JEZWZpbml0aW9uTmFtZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tSm9iRGVmaW5pdGlvbk5hbWUoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgam9iRGVmaW5pdGlvbk5hbWU6IHN0cmluZyk6IElKb2JEZWZpbml0aW9uIHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHNjb3BlKTtcbiAgICBjb25zdCBqb2JEZWZBcm4gPSBzdGFjay5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ2JhdGNoJyxcbiAgICAgIHJlc291cmNlOiAnam9iLWRlZmluaXRpb24nLFxuICAgICAgYXJuRm9ybWF0OiBBcm5Gb3JtYXQuU0xBU0hfUkVTT1VSQ0VfTkFNRSxcbiAgICAgIHJlc291cmNlTmFtZTogam9iRGVmaW5pdGlvbk5hbWUsXG4gICAgfSk7XG5cbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElKb2JEZWZpbml0aW9uIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBqb2JEZWZpbml0aW9uQXJuID0gam9iRGVmQXJuO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGpvYkRlZmluaXRpb25OYW1lID0gam9iRGVmaW5pdGlvbk5hbWU7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSBqb2JEZWZpbml0aW9uQXJuOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBqb2JEZWZpbml0aW9uTmFtZTogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGltYWdlQ29uZmlnOiBKb2JEZWZpbml0aW9uSW1hZ2VDb25maWc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEpvYkRlZmluaXRpb25Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy5qb2JEZWZpbml0aW9uTmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMudmFsaWRhdGVQcm9wcyhwcm9wcyk7XG5cbiAgICB0aGlzLmltYWdlQ29uZmlnID0gbmV3IEpvYkRlZmluaXRpb25JbWFnZUNvbmZpZyh0aGlzLCBwcm9wcy5jb250YWluZXIpO1xuXG4gICAgY29uc3QgaXNGYXJnYXRlID0gISFwcm9wcy5wbGF0Zm9ybUNhcGFiaWxpdGllcz8uaW5jbHVkZXMoUGxhdGZvcm1DYXBhYmlsaXRpZXMuRkFSR0FURSk7XG5cbiAgICBjb25zdCBqb2JEZWYgPSBuZXcgQ2ZuSm9iRGVmaW5pdGlvbih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBqb2JEZWZpbml0aW9uTmFtZTogcHJvcHMuam9iRGVmaW5pdGlvbk5hbWUsXG4gICAgICBjb250YWluZXJQcm9wZXJ0aWVzOiB0aGlzLmJ1aWxkSm9iQ29udGFpbmVyKHByb3BzLmNvbnRhaW5lciwgaXNGYXJnYXRlKSxcbiAgICAgIHR5cGU6ICdjb250YWluZXInLFxuICAgICAgbm9kZVByb3BlcnRpZXM6IHByb3BzLm5vZGVQcm9wc1xuICAgICAgICA/IHtcbiAgICAgICAgICBtYWluTm9kZTogcHJvcHMubm9kZVByb3BzLm1haW5Ob2RlLFxuICAgICAgICAgIG5vZGVSYW5nZVByb3BlcnRpZXM6IHRoaXMuYnVpbGROb2RlUmFuZ2VQcm9wcyhwcm9wcy5ub2RlUHJvcHMsIGlzRmFyZ2F0ZSksXG4gICAgICAgICAgbnVtTm9kZXM6IHByb3BzLm5vZGVQcm9wcy5jb3VudCxcbiAgICAgICAgfVxuICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgIHBhcmFtZXRlcnM6IHByb3BzLnBhcmFtZXRlcnMsXG4gICAgICByZXRyeVN0cmF0ZWd5OiB7XG4gICAgICAgIGF0dGVtcHRzOiBwcm9wcy5yZXRyeUF0dGVtcHRzIHx8IDEsXG4gICAgICB9LFxuICAgICAgdGltZW91dDoge1xuICAgICAgICBhdHRlbXB0RHVyYXRpb25TZWNvbmRzOiBwcm9wcy50aW1lb3V0ID8gcHJvcHMudGltZW91dC50b1NlY29uZHMoKSA6IHVuZGVmaW5lZCxcbiAgICAgIH0sXG4gICAgICBwbGF0Zm9ybUNhcGFiaWxpdGllczogcHJvcHMucGxhdGZvcm1DYXBhYmlsaXRpZXMgPz8gW1BsYXRmb3JtQ2FwYWJpbGl0aWVzLkVDMl0sXG4gICAgfSk7XG5cbiAgICB0aGlzLmpvYkRlZmluaXRpb25Bcm4gPSB0aGlzLmdldFJlc291cmNlQXJuQXR0cmlidXRlKGpvYkRlZi5yZWYsIHtcbiAgICAgIHNlcnZpY2U6ICdiYXRjaCcsXG4gICAgICByZXNvdXJjZTogJ2pvYi1kZWZpbml0aW9uJyxcbiAgICAgIHJlc291cmNlTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgfSk7XG4gICAgdGhpcy5qb2JEZWZpbml0aW9uTmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKGpvYkRlZi5yZWYpO1xuICB9XG5cbiAgcHJpdmF0ZSBkZXNlcmlhbGl6ZUVudlZhcmlhYmxlcyhlbnY/OiB7IFtuYW1lOiBzdHJpbmddOiBzdHJpbmcgfSk6IENmbkpvYkRlZmluaXRpb24uRW52aXJvbm1lbnRQcm9wZXJ0eVtdIHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCB2YXJzID0gbmV3IEFycmF5PENmbkpvYkRlZmluaXRpb24uRW52aXJvbm1lbnRQcm9wZXJ0eT4oKTtcblxuICAgIGlmIChlbnYgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICBPYmplY3Qua2V5cyhlbnYpLm1hcCgobmFtZTogc3RyaW5nKSA9PiB7XG4gICAgICB2YXJzLnB1c2goeyBuYW1lLCB2YWx1ZTogZW52W25hbWVdIH0pO1xuICAgIH0pO1xuXG4gICAgcmV0dXJuIHZhcnM7XG4gIH1cblxuICAvKipcbiAgICogVmFsaWRhdGVzIHRoZSBwcm9wZXJ0aWVzIHByb3ZpZGVkIGZvciBhIG5ldyBqb2IgZGVmaW5pdGlvbi5cbiAgICovXG4gIHByaXZhdGUgdmFsaWRhdGVQcm9wcyhwcm9wczogSm9iRGVmaW5pdGlvblByb3BzKSB7XG4gICAgaWYgKHByb3BzID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMucGxhdGZvcm1DYXBhYmlsaXRpZXMgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5wbGF0Zm9ybUNhcGFiaWxpdGllcy5pbmNsdWRlcyhQbGF0Zm9ybUNhcGFiaWxpdGllcy5GQVJHQVRFKVxuICAgICAgJiYgcHJvcHMuY29udGFpbmVyLmV4ZWN1dGlvblJvbGUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdGYXJnYXRlIGpvYiBtdXN0IGhhdmUgZXhlY3V0aW9uUm9sZSBzZXQnKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMucGxhdGZvcm1DYXBhYmlsaXRpZXMgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5wbGF0Zm9ybUNhcGFiaWxpdGllcy5pbmNsdWRlcyhQbGF0Zm9ybUNhcGFiaWxpdGllcy5GQVJHQVRFKVxuICAgICAgJiYgcHJvcHMuY29udGFpbmVyLmdwdUNvdW50ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRmFyZ2F0ZSBqb2IgbXVzdCBub3QgaGF2ZSBncHVDb3VudCBzZXQnKTtcbiAgICB9XG5cbiAgICBpZiAoKHByb3BzLnBsYXRmb3JtQ2FwYWJpbGl0aWVzID09PSB1bmRlZmluZWQgfHwgcHJvcHMucGxhdGZvcm1DYXBhYmlsaXRpZXMuaW5jbHVkZXMoUGxhdGZvcm1DYXBhYmlsaXRpZXMuRUMyKSlcbiAgICAgICYmIHByb3BzLmNvbnRhaW5lci5hc3NpZ25QdWJsaWNJcCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0VDMiBqb2IgbXVzdCBub3QgaGF2ZSBhc3NpZ25QdWJsaWNJcCBzZXQnKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGJ1aWxkSm9iQ29udGFpbmVyKGNvbnRhaW5lcjogSm9iRGVmaW5pdGlvbkNvbnRhaW5lciwgaXNGYXJnYXRlOiBib29sZWFuKTogQ2ZuSm9iRGVmaW5pdGlvbi5Db250YWluZXJQcm9wZXJ0aWVzUHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgIGlmIChjb250YWluZXIgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgY29tbWFuZDogY29udGFpbmVyLmNvbW1hbmQsXG4gICAgICBlbnZpcm9ubWVudDogdGhpcy5kZXNlcmlhbGl6ZUVudlZhcmlhYmxlcyhjb250YWluZXIuZW52aXJvbm1lbnQpLFxuICAgICAgaW1hZ2U6IHRoaXMuaW1hZ2VDb25maWcuaW1hZ2VOYW1lLFxuICAgICAgaW5zdGFuY2VUeXBlOiBjb250YWluZXIuaW5zdGFuY2VUeXBlICYmIGNvbnRhaW5lci5pbnN0YW5jZVR5cGUudG9TdHJpbmcoKSxcbiAgICAgIGpvYlJvbGVBcm46IGNvbnRhaW5lci5qb2JSb2xlICYmIGNvbnRhaW5lci5qb2JSb2xlLnJvbGVBcm4sXG4gICAgICBleGVjdXRpb25Sb2xlQXJuOiBjb250YWluZXIuZXhlY3V0aW9uUm9sZSAmJiBjb250YWluZXIuZXhlY3V0aW9uUm9sZS5yb2xlQXJuLFxuICAgICAgbGludXhQYXJhbWV0ZXJzOiBjb250YWluZXIubGludXhQYXJhbXNcbiAgICAgICAgPyB7IGRldmljZXM6IGNvbnRhaW5lci5saW51eFBhcmFtcy5yZW5kZXJMaW51eFBhcmFtZXRlcnMoKS5kZXZpY2VzIH1cbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICBsb2dDb25maWd1cmF0aW9uOiBjb250YWluZXIubG9nQ29uZmlndXJhdGlvbiA/IHtcbiAgICAgICAgbG9nRHJpdmVyOiBjb250YWluZXIubG9nQ29uZmlndXJhdGlvbi5sb2dEcml2ZXIsXG4gICAgICAgIG9wdGlvbnM6IGNvbnRhaW5lci5sb2dDb25maWd1cmF0aW9uLm9wdGlvbnMsXG4gICAgICAgIHNlY3JldE9wdGlvbnM6IGNvbnRhaW5lci5sb2dDb25maWd1cmF0aW9uLnNlY3JldE9wdGlvbnNcbiAgICAgICAgICA/IHRoaXMuYnVpbGRMb2dDb25maWd1cmF0aW9uU2VjcmV0T3B0aW9ucyhjb250YWluZXIubG9nQ29uZmlndXJhdGlvbi5zZWNyZXRPcHRpb25zKVxuICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgIG1vdW50UG9pbnRzOiBjb250YWluZXIubW91bnRQb2ludHMsXG4gICAgICBwcml2aWxlZ2VkOiBjb250YWluZXIucHJpdmlsZWdlZCB8fCBmYWxzZSxcbiAgICAgIG5ldHdvcmtDb25maWd1cmF0aW9uOiBjb250YWluZXIuYXNzaWduUHVibGljSXAgPyB7XG4gICAgICAgIGFzc2lnblB1YmxpY0lwOiBjb250YWluZXIuYXNzaWduUHVibGljSXAgPyAnRU5BQkxFRCcgOiAnRElTQUJMRUQnLFxuICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgIHJlYWRvbmx5Um9vdEZpbGVzeXN0ZW06IGNvbnRhaW5lci5yZWFkT25seSB8fCBmYWxzZSxcbiAgICAgIHVsaW1pdHM6IGNvbnRhaW5lci51bGltaXRzLFxuICAgICAgdXNlcjogY29udGFpbmVyLnVzZXIsXG4gICAgICB2b2x1bWVzOiBjb250YWluZXIudm9sdW1lcyxcbiAgICAgIGZhcmdhdGVQbGF0Zm9ybUNvbmZpZ3VyYXRpb246IGNvbnRhaW5lci5wbGF0Zm9ybVZlcnNpb24gPyB7XG4gICAgICAgIHBsYXRmb3JtVmVyc2lvbjogY29udGFpbmVyLnBsYXRmb3JtVmVyc2lvbixcbiAgICAgIH0gOiB1bmRlZmluZWQsXG4gICAgICByZXNvdXJjZVJlcXVpcmVtZW50czogW1xuICAgICAgICB7IHR5cGU6ICdWQ1BVJywgdmFsdWU6IFN0cmluZyhjb250YWluZXIudmNwdXMgfHwgKGlzRmFyZ2F0ZSA/IDAuMjUgOiAxKSkgfSxcbiAgICAgICAgeyB0eXBlOiAnTUVNT1JZJywgdmFsdWU6IFN0cmluZyhjb250YWluZXIubWVtb3J5TGltaXRNaUIgfHwgKGlzRmFyZ2F0ZSA/IDUxMiA6IDQpKSB9LFxuICAgICAgXS5jb25jYXQoY29udGFpbmVyLmdwdUNvdW50ID8gW3sgdHlwZTogJ0dQVScsIHZhbHVlOiBTdHJpbmcoY29udGFpbmVyLmdwdUNvdW50KSB9XSA6IFtdKSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBidWlsZE5vZGVSYW5nZVByb3BzKG11bHRpTm9kZVByb3BzOiBJTXVsdGlOb2RlUHJvcHMsIGlzRmFyZ2F0ZTogYm9vbGVhbik6IENmbkpvYkRlZmluaXRpb24uTm9kZVJhbmdlUHJvcGVydHlQcm9wZXJ0eVtdIHtcbiAgICBjb25zdCByYW5nZVByb3BzID0gbmV3IEFycmF5PENmbkpvYkRlZmluaXRpb24uTm9kZVJhbmdlUHJvcGVydHlQcm9wZXJ0eT4oKTtcblxuICAgIGZvciAoY29uc3QgcHJvcCBvZiBtdWx0aU5vZGVQcm9wcy5yYW5nZVByb3BzKSB7XG4gICAgICByYW5nZVByb3BzLnB1c2goe1xuICAgICAgICBjb250YWluZXI6IHRoaXMuYnVpbGRKb2JDb250YWluZXIocHJvcC5jb250YWluZXIsIGlzRmFyZ2F0ZSksXG4gICAgICAgIHRhcmdldE5vZGVzOiBgJHtwcm9wLmZyb21Ob2RlSW5kZXggfHwgMH06JHtwcm9wLnRvTm9kZUluZGV4IHx8IG11bHRpTm9kZVByb3BzLmNvdW50fWAsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICByZXR1cm4gcmFuZ2VQcm9wcztcbiAgfVxuXG4gIHByaXZhdGUgYnVpbGRMb2dDb25maWd1cmF0aW9uU2VjcmV0T3B0aW9ucyhzZWNyZXRPcHRpb25zOiBFeHBvc2VkU2VjcmV0W10pOiBDZm5Kb2JEZWZpbml0aW9uLlNlY3JldFByb3BlcnR5W10ge1xuICAgIHJldHVybiBzZWNyZXRPcHRpb25zLm1hcChzZWNyZXRPcHRpb24gPT4ge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgbmFtZTogc2VjcmV0T3B0aW9uLm9wdGlvbk5hbWUsXG4gICAgICAgIHZhbHVlRnJvbTogc2VjcmV0T3B0aW9uLnNlY3JldEFybixcbiAgICAgIH07XG4gICAgfSk7XG4gIH1cbn1cbiJdfQ==