"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
describe('fromBucket', () => {
    test('as bucket, renders all properties, including S3Origin config', () => {
        const bucket = new s3.Bucket(stack, 'Bucket');
        const origin = lib_1.Origin.fromBucket(bucket);
        origin._bind(stack, { originIndex: 0 });
        expect(origin._renderOrigin()).toEqual({
            id: 'StackOrigin029E19582',
            domainName: bucket.bucketRegionalDomainName,
            s3OriginConfig: {
                originAccessIdentity: 'origin-access-identity/cloudfront/${Token[TOKEN.69]}',
            },
        });
    });
    test('as bucket, creates an OriginAccessIdentity and grants read permissions on the bucket', () => {
        const bucket = new s3.Bucket(stack, 'Bucket');
        const origin = lib_1.Origin.fromBucket(bucket);
        new lib_1.Distribution(stack, 'Dist', { defaultBehavior: { origin } });
        expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity', {
            CloudFrontOriginAccessIdentityConfig: {
                Comment: 'Allows CloudFront to reach the bucket',
            },
        });
        expect(stack).toHaveResourceLike('AWS::S3::BucketPolicy', {
            PolicyDocument: {
                Statement: [{
                        Principal: {
                            CanonicalUser: { 'Fn::GetAtt': ['DistS3Origin1C4519663', 'S3CanonicalUserId'] },
                        },
                    }],
            },
        });
    });
    test('as website buvcket, renders all properties, including custom origin config', () => {
        const bucket = new s3.Bucket(stack, 'Bucket', {
            websiteIndexDocument: 'index.html',
        });
        const origin = lib_1.Origin.fromBucket(bucket);
        origin._bind(stack, { originIndex: 0 });
        expect(origin._renderOrigin()).toEqual({
            id: 'StackOrigin029E19582',
            domainName: bucket.bucketWebsiteDomainName,
            customOriginConfig: {
                originProtocolPolicy: 'http-only',
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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