"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const acm = require("@aws-cdk/aws-certificatemanager");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
test('minimal example renders correctly', () => {
    const origin = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket'));
    new lib_1.Distribution(stack, 'MyDist', { defaultBehavior: { origin } });
    expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                ForwardedValues: { QueryString: false },
                TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                ViewerProtocolPolicy: 'allow-all',
            },
            Enabled: true,
            Origins: [{
                    DomainName: { 'Fn::GetAtt': ['Bucket83908E77', 'RegionalDomainName'] },
                    Id: 'StackMyDistOrigin1D6D5E535',
                    S3OriginConfig: {
                        OriginAccessIdentity: { 'Fn::Join': ['',
                                ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin1ED86A27E' }],
                            ] },
                    },
                }],
        },
    });
});
describe('multiple behaviors', () => {
    test('a second behavior can\'t be specified with the catch-all path pattern', () => {
        const origin = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket'));
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: { origin },
                additionalBehaviors: {
                    '*': { origin },
                },
            });
        }).toThrow(/Only the default behavior can have a path pattern of \'*\'/);
    });
    test('a second behavior can be added to the original origin', () => {
        const origin = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket'));
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                Origins: [{
                        DomainName: { 'Fn::GetAtt': ['Bucket83908E77', 'RegionalDomainName'] },
                        Id: 'StackMyDistOrigin1D6D5E535',
                        S3OriginConfig: {
                            OriginAccessIdentity: { 'Fn::Join': ['',
                                    ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin1ED86A27E' }],
                                ] },
                        },
                    }],
            },
        });
    });
    test('a second behavior can be added to a secondary origin', () => {
        const origin = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket'));
        const origin2 = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket2'));
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin: origin2 },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                Origins: [{
                        DomainName: { 'Fn::GetAtt': ['Bucket83908E77', 'RegionalDomainName'] },
                        Id: 'StackMyDistOrigin1D6D5E535',
                        S3OriginConfig: {
                            OriginAccessIdentity: { 'Fn::Join': ['',
                                    ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin1ED86A27E' }],
                                ] },
                        },
                    },
                    {
                        DomainName: { 'Fn::GetAtt': ['Bucket25524B414', 'RegionalDomainName'] },
                        Id: 'StackMyDistOrigin20B96F3AD',
                        S3OriginConfig: {
                            OriginAccessIdentity: { 'Fn::Join': ['',
                                    ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin2E88F08BB' }],
                                ] },
                        },
                    }],
            },
        });
    });
    test('behavior creation order is preserved', () => {
        const origin = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket'));
        const origin2 = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket2'));
        const dist = new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/1*': { origin: origin2 },
            },
        });
        dist.addBehavior('api/2*', origin);
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/1*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    },
                    {
                        PathPattern: 'api/2*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                Origins: [{
                        DomainName: { 'Fn::GetAtt': ['Bucket83908E77', 'RegionalDomainName'] },
                        Id: 'StackMyDistOrigin1D6D5E535',
                        S3OriginConfig: {
                            OriginAccessIdentity: { 'Fn::Join': ['',
                                    ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin1ED86A27E' }],
                                ] },
                        },
                    },
                    {
                        DomainName: { 'Fn::GetAtt': ['Bucket25524B414', 'RegionalDomainName'] },
                        Id: 'StackMyDistOrigin20B96F3AD',
                        S3OriginConfig: {
                            OriginAccessIdentity: { 'Fn::Join': ['',
                                    ['origin-access-identity/cloudfront/', { Ref: 'MyDistS3Origin2E88F08BB' }],
                                ] },
                        },
                    }],
            },
        });
    });
});
describe('certificates', () => {
    test('should fail if using an imported certificate from outside of us-east-1', () => {
        const origin = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket'));
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:eu-west-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                certificate,
            });
        }).toThrow(/Distribution certificates must be in the us-east-1 region/);
    });
    test('adding a certificate renders the correct ViewerCertificate property', () => {
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin: lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket')) },
            certificate,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                ViewerCertificate: {
                    AcmCertificateArn: 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012',
                },
            },
        });
    });
});
describe('custom error responses', () => {
    test('should fail if responsePagePath is defined but responseCode is not', () => {
        const origin = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket'));
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{
                        httpStatus: 404,
                        responsePagePath: '/errors/404.html',
                    }],
            });
        }).toThrow(/\'responseCode\' must be provided if \'responsePagePath\' is defined/);
    });
    test('should fail if only the error code is provided', () => {
        const origin = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket'));
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{ httpStatus: 404 }],
            });
        }).toThrow(/A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid./);
    });
    test('should render the array of error configs if provided', () => {
        const origin = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket'));
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin },
            errorResponses: [{
                    httpStatus: 404,
                    responseHttpStatus: 404,
                    responsePagePath: '/errors/404.html',
                },
                {
                    httpStatus: 500,
                    ttl: core_1.Duration.seconds(2),
                }],
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                CustomErrorResponses: [
                    {
                        ErrorCode: 404,
                        ResponseCode: 404,
                        ResponsePagePath: '/errors/404.html',
                    },
                    {
                        ErrorCachingMinTTL: 2,
                        ErrorCode: 500,
                    },
                ],
            },
        });
    });
});
test('price class is included if provided', () => {
    const origin = lib_1.Origin.fromBucket(new s3.Bucket(stack, 'Bucket'));
    new lib_1.Distribution(stack, 'Dist', {
        defaultBehavior: { origin },
        priceClass: lib_1.PriceClass.PRICE_CLASS_200,
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            PriceClass: 'PriceClass_200',
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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