"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpOrigin = exports.S3Origin = exports.Origin = void 0;
const core_1 = require("@aws-cdk/core");
const distribution_1 = require("./distribution");
const origin_access_identity_1 = require("./origin_access_identity");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
class Origin {
    constructor(props) {
        this.domainName = props.domainName;
    }
    /**
     * Creates a pre-configured origin for a S3 bucket.
     * If this bucket has been configured for static website hosting, then `fromWebsiteBucket` should be used instead.
     *
     * An Origin Access Identity will be created and granted read access to the bucket.
     *
     * @param bucket the bucket to act as an origin.
     */
    static fromBucket(bucket) {
        if (bucket.isWebsite) {
            return new HttpOrigin({
                domainName: bucket.bucketWebsiteDomainName,
                protocolPolicy: distribution_1.OriginProtocolPolicy.HTTP_ONLY,
            });
        }
        else {
            return new S3Origin({ domainName: bucket.bucketRegionalDomainName, bucket });
        }
    }
    /**
     * The unique id for this origin.
     *
     * Cannot be accesed until bind() is called.
     */
    get id() {
        if (!this.originId) {
            throw new Error('Cannot access originId until `bind` is called.');
        }
        return this.originId;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     *
     * @internal
     */
    _bind(scope, options) {
        this.originId = new core_1.Construct(scope, `Origin${options.originIndex}`).node.uniqueId;
    }
    /**
     * Creates and returns the CloudFormation representation of this origin.
     *
     * @internal
     */
    _renderOrigin() {
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            domainName: this.domainName,
            id: this.id,
            s3OriginConfig,
            customOriginConfig,
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
}
exports.Origin = Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 *
 * @experimental
 */
class S3Origin extends Origin {
    constructor(props) {
        super(props);
        this.bucket = props.bucket;
    }
    /** @internal */
    _bind(scope, options) {
        super._bind(scope, options);
        if (!this.originAccessIdentity) {
            this.originAccessIdentity = new origin_access_identity_1.OriginAccessIdentity(scope, `S3Origin${options.originIndex}`);
            this.bucket.grantRead(this.originAccessIdentity);
        }
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin for an HTTP server or S3 bucket configured for website hosting.
 *
 * @experimental
 */
class HttpOrigin extends Origin {
    constructor(props) {
        super(props);
        this.protocolPolicy = props.protocolPolicy;
    }
    renderCustomOriginConfig() {
        var _a;
        return {
            originProtocolPolicy: (_a = this.protocolPolicy) !== null && _a !== void 0 ? _a : distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
        };
    }
}
exports.HttpOrigin = HttpOrigin;
//# sourceMappingURL=data:application/json;base64,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