"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NewRelicStack = exports.EndpointUrlMetrics = exports.EndpointUrlLogs = exports.EndpointType = exports.NEW_RELIC_AWS_ACCOUNT_ID = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const iam = require("aws-cdk-lib/aws-iam");
const firehose = require("aws-cdk-lib/aws-kinesisfirehose");
const s3 = require("aws-cdk-lib/aws-s3");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const utils_1 = require("../common/utils");
const stack_1 = require("../stack");
exports.NEW_RELIC_AWS_ACCOUNT_ID = '754728514883';
var EndpointType;
(function (EndpointType) {
    EndpointType["METRICS"] = "metrics";
    EndpointType["LOGS"] = "logs";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
var EndpointUrlLogs;
(function (EndpointUrlLogs) {
    EndpointUrlLogs["EU_LOGS"] = "https://aws-api.eu.newrelic.com/firehose/v1";
    EndpointUrlLogs["US_LOGS"] = "https://aws-api.newrelic.com/firehose/v1";
})(EndpointUrlLogs = exports.EndpointUrlLogs || (exports.EndpointUrlLogs = {}));
var EndpointUrlMetrics;
(function (EndpointUrlMetrics) {
    EndpointUrlMetrics["EU_METRICS"] = "https://aws-api.eu01.nr-data.net/cloudwatch-metrics/v1";
    EndpointUrlMetrics["US_METRICS"] = "https://aws-api.newrelic.com/cloudwatch-metrics/v1";
})(EndpointUrlMetrics = exports.EndpointUrlMetrics || (exports.EndpointUrlMetrics = {}));
class NewRelicStack extends stack_1.BaseStack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.newRelicIntegrationRole = this.createNewRelicRole(props.newRelicAccountId);
        this.newRelicSecret = this.createSecrets(props.newRelicAccountId, props.newRelicLicenseKey);
        this.newRelicBucket = this.createFirehoseBucket(props.newRelicBucketName);
        this.newRelicFirehoseRole = this.createFirehoseRole(this.newRelicBucket);
        if (props.newRelicApiUrlLogs) {
            this.newRelicFirehoseLogs = this.createFirehoseStream(this.newRelicBucket, this.newRelicFirehoseRole, EndpointType.LOGS, props.newRelicApiUrlLogs, props.newRelicLicenseKey);
            this.newRelicCloudwatchLogsStreamRole = this.createCloudwatchLogsStreamRole();
        }
        if (props.newRelicApiUrlMetrics) {
            this.newRelicFirehoseMetrics = this.createFirehoseStream(this.newRelicBucket, this.newRelicFirehoseRole, EndpointType.METRICS, props.newRelicApiUrlMetrics, props.newRelicLicenseKey);
        }
    }
    createCloudwatchMetricStream(roleArn, firehoseArn) {
        return new cloudwatch.CfnMetricStream(this, 'newrelic-cloudwatch-stream-metrics', {
            firehoseArn: firehoseArn,
            outputFormat: 'opentelemetry0.7',
            roleArn: roleArn,
            name: 'newelic-stream-metrics',
        });
    }
    createCloudwatchLogsStreamRole() {
        let role = new iam.Role(this, 'newrelic-logstream-role', {
            roleName: 'NewRelicInfrastructure-CloudwatchStream',
            assumedBy: new iam.ServicePrincipal('logs.amazonaws.com'),
        });
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'firehose:PutRecord',
                'firehose:PutRecordBatch',
                'kinesis:PutRecords',
                'kinesis:PutRecord',
            ],
            resources: ['*'],
        }));
        utils_1.addBaseTags(role);
        return role;
    }
    createNewRelicRole(newRelicAccountId) {
        let role = new iam.Role(this, 'newrelic-role', {
            roleName: 'NewRelicInfrastructure-Integrations',
            assumedBy: new iam.AccountPrincipal(exports.NEW_RELIC_AWS_ACCOUNT_ID),
            externalIds: [
                newRelicAccountId,
            ],
        });
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'));
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'budgets:ViewBudget',
            ],
            resources: ['*'],
        }));
        utils_1.addBaseTags(role);
        new aws_cdk_lib_1.CfnOutput(this, 'newrelic-role-output', {
            value: role.roleArn,
            description: 'New Relic role arn',
            exportName: 'newRelicRole',
        });
        return role;
    }
    createFirehoseStream(newRelicBucket, role, endpointType, endpointUrl, newRelicLicenseLey) {
        if (this.stage == 'production') {
            // Minute in one day: 1440
            // Interval: 5min
            // Sends per day: 1440/5 = 288
            // Usage per day: 288*5mb = 1.5gb
            var bufferingHints = {
                intervalInSeconds: 300,
                sizeInMBs: 5,
            };
        }
        else {
            // Usage per day: 144*3mb = 0.432gb
            var bufferingHints = {
                intervalInSeconds: 600,
                sizeInMBs: 3,
            };
        }
        let httpEndpointMetrics = {
            bufferingHints: bufferingHints,
            endpointConfiguration: {
                url: endpointUrl,
                accessKey: newRelicLicenseLey,
                name: endpointType,
            },
            s3Configuration: {
                bucketArn: newRelicBucket.bucketArn,
                roleArn: role.roleArn,
            },
            requestConfiguration: {
                contentEncoding: 'GZIP',
            },
            roleArn: role.roleArn,
        };
        let firehoseStream = new firehose.CfnDeliveryStream(this, `newrelic-firehose-${endpointType}`, {
            deliveryStreamName: `NewRelic-stream-${endpointType}`,
            deliveryStreamType: 'DirectPut',
            httpEndpointDestinationConfiguration: httpEndpointMetrics,
        });
        utils_1.addBaseTags(firehoseStream);
        return firehoseStream;
    }
    createSecrets(newRelicAccountId, newRelicLicenseLey) {
        let secret = new secretsmanager.Secret(this, 'newrelic-secret', {
            secretName: 'NEW_RELIC_LICENSE_KEY',
            secretObjectValue: {
                NrAccountId: aws_cdk_lib_1.SecretValue.unsafePlainText(newRelicAccountId),
                LicenseKey: aws_cdk_lib_1.SecretValue.unsafePlainText(newRelicLicenseLey),
            },
        });
        utils_1.addBaseTags(secret);
        return secret;
    }
    createFirehoseBucket(newRelicBucketName) {
        let bucket = new s3.Bucket(this, 'newrelic-bucket', {
            bucketName: newRelicBucketName,
            versioned: true,
            lifecycleRules: [
                {
                    id: 'raw-bucket-lifecycle-rule',
                    enabled: true,
                    noncurrentVersionExpiration: aws_cdk_lib_1.Duration.days(30),
                    transitions: [
                        {
                            storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: aws_cdk_lib_1.Duration.days(30),
                        },
                    ],
                },
            ],
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        utils_1.addBaseTags(bucket);
        return bucket;
    }
    createFirehoseRole(newRelicFirehoseBucket) {
        let role = new iam.Role(this, 'newrelic-firehose-role', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
        });
        utils_1.addBaseTags(role);
        // TODO: create more restrictive policy
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'glue:GetTable',
                'glue:GetTableVersion',
                'glue:GetTableVersions',
                'glue:GetSchemaByDefinition',
                'glue:GetSchemaVersion',
                'logs:PutLogEvents',
                'kinesis:DescribeStream',
                'kinesis:GetShardIterator',
                'kinesis:GetRecords',
                'kinesis:ListShards',
                'kms:Decrypt',
                'kms:GenerateDataKey',
                'kms:Decrypt',
                'kafka:GetBootstrapBrokers',
                'kafka:DescribeCluster',
                'kafka:DescribeClusterV2',
                'kafka-cluster:Connect',
                'kafka-cluster:DescribeTopic',
                'kafka-cluster:DescribeTopicDynamicConfiguration',
                'kafka-cluster:ReadData',
                'kafka-cluster:DescribeGroup',
                'lambda:InvokeFunction',
                'lambda:GetFunctionConfiguration',
            ],
            resources: ['*'],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                's3:AbortMultipartUpload',
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:PutObject',
                'S3:PutObjectAcl',
            ],
            resources: [
                `${newRelicFirehoseBucket.bucketArn}`,
                `${newRelicFirehoseBucket.bucketArn}/*`,
            ],
        }));
        return role;
    }
}
exports.NewRelicStack = NewRelicStack;
_a = JSII_RTTI_SYMBOL_1;
NewRelicStack[_a] = { fqn: "neulabs-cdk-constructs.newrelic.NewRelicStack", version: "0.1.11" };
//# sourceMappingURL=data:application/json;base64,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