"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const widget_1 = require("./widget");
// This file contains widgets that exist for layout purposes
/**
 * A widget that contains other widgets in a horizontal row
 *
 * Widgets will be laid out next to each other
 */
class Row {
    constructor(...widgets) {
        /**
         * Relative position of each widget inside this row
         */
        this.offsets = [];
        this.widgets = widgets;
        this.width = 0;
        this.height = 0;
        let x = 0;
        let y = 0;
        for (const widget of widgets) {
            // See if we need to horizontally wrap to add this widget
            if (x + widget.width > widget_1.GRID_WIDTH) {
                y = this.height;
                x = 0;
            }
            this.offsets.push({ x, y });
            this.width = Math.max(this.width, x + widget.width);
            this.height = Math.max(this.height, y + widget.height);
            x += widget.width;
        }
    }
    position(x, y) {
        for (let i = 0; i < this.widgets.length; i++) {
            this.widgets[i].position(x + this.offsets[i].x, y + this.offsets[i].y);
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Row = Row;
/**
 * A widget that contains other widgets in a vertical column
 *
 * Widgets will be laid out next to each other
 */
class Column {
    constructor(...widgets) {
        this.widgets = widgets;
        // There's no vertical wrapping so this one's a lot easier
        this.width = Math.max(...this.widgets.map(w => w.width));
        this.height = sum(...this.widgets.map(w => w.height));
    }
    position(x, y) {
        let widgetY = y;
        for (const widget of this.widgets) {
            widget.position(x, widgetY);
            widgetY += widget.height;
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Column = Column;
/**
 * A widget that doesn't display anything but takes up space
 */
class Spacer {
    constructor(props = {}) {
        this.width = props.width || 1;
        this.height = props.height || 1;
    }
    position(_x, _y) {
        // Don't need to do anything, not a physical widget
    }
    toJson() {
        return [];
    }
}
exports.Spacer = Spacer;
/**
 * Return the sum of a list of numbers
 */
function sum(...xs) {
    let ret = 0;
    for (const x of xs) {
        ret += x;
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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