"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const rendering_1 = require("./private/rendering");
const widget_1 = require("./widget");
/**
 * Display the metric associated with an alarm, including the alarm line
 */
class AlarmWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    annotations: {
                        alarms: [this.props.alarm.alarmArn],
                    },
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                    },
                },
            }];
    }
}
exports.AlarmWidget = AlarmWidget;
/**
 * A dashboard widget that displays metrics
 */
class GraphWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        const horizontalAnnotations = [
            ...(this.props.leftAnnotations || []).map(mapAnnotation('left')),
            ...(this.props.rightAnnotations || []).map(mapAnnotation('right')),
        ];
        const metrics = rendering_1.allMetricsGraphJson(this.props.left || [], this.props.right || []);
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    stacked: this.props.stacked,
                    metrics: metrics.length > 0 ? metrics : undefined,
                    annotations: horizontalAnnotations.length > 0 ? { horizontal: horizontalAnnotations } : undefined,
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                        right: this.props.rightYAxis !== undefined ? this.props.rightYAxis : undefined,
                    },
                    legend: this.props.legendPosition !== undefined ? { position: this.props.legendPosition } : undefined,
                },
            }];
    }
}
exports.GraphWidget = GraphWidget;
/**
 * A dashboard widget that displays the most recent value for every metric
 */
class SingleValueWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 3);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'singleValue',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    metrics: rendering_1.allMetricsGraphJson(this.props.metrics, []),
                    setPeriodToTimeRange: this.props.setPeriodToTimeRange,
                },
            }];
    }
}
exports.SingleValueWidget = SingleValueWidget;
/**
 * Fill shading options that will be used with an annotation
 */
var Shading;
(function (Shading) {
    /**
     * Don't add shading
     */
    Shading["NONE"] = "none";
    /**
     * Add shading above the annotation
     */
    Shading["ABOVE"] = "above";
    /**
     * Add shading below the annotation
     */
    Shading["BELOW"] = "below";
})(Shading = exports.Shading || (exports.Shading = {}));
/**
 * A set of standard colours that can be used in annotations in a GraphWidget.
 */
class Color {
}
exports.Color = Color;
/** blue - hex #1f77b4 */
Color.BLUE = '#1f77b4';
/** brown - hex #8c564b */
Color.BROWN = '#8c564b';
/** green - hex #2ca02c */
Color.GREEN = '#2ca02c';
/** grey - hex #7f7f7f */
Color.GREY = '#7f7f7f';
/** orange - hex #ff7f0e */
Color.ORANGE = '#ff7f0e';
/** pink - hex #e377c2 */
Color.PINK = '#e377c2';
/** purple - hex #9467bd */
Color.PURPLE = '#9467bd';
/** red - hex #d62728 */
Color.RED = '#d62728';
/**
 * The position of the legend on a GraphWidget.
 */
var LegendPosition;
(function (LegendPosition) {
    /**
     * Legend appears below the graph (default).
     */
    LegendPosition["BOTTOM"] = "bottom";
    /**
     * Add shading above the annotation
     */
    LegendPosition["RIGHT"] = "right";
    /**
     * Add shading below the annotation
     */
    LegendPosition["HIDDEN"] = "hidden";
})(LegendPosition = exports.LegendPosition || (exports.LegendPosition = {}));
function mapAnnotation(yAxis) {
    return (a) => {
        return { ...a, yAxis };
    };
}
//# sourceMappingURL=data:application/json;base64,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