#!/usr/bin/env python3

import os
import sys
import argparse
from argparse import RawTextHelpFormatter
from collections import OrderedDict
import matplotlib.pyplot as plt

from mpet.version import __version__
import mpet.plot.outmat2txt as outmat2txt
import mpet.plot.plot_data as plot_data

#Ordered dictionary of plot types
plotTypes=OrderedDict([
    ('v','voltage vs filling fraction'),
    ('vt','voltage vs time'),
    ('curr','current vs time'),
    ('elytec','electrolyte concentration (movie)'),
    ('elytecf','electrolyte concentration final snapshot'),
    ('elytep','electrolyte potential (movie)'),
    ('elytepf','electroloyte potential final snapshot'),
    ('elytei','electrolyte potential (movie)'),
    ('elyteif','electrolyte potential final snapshot'),
    ('surf_c','cathode surface concentrations'),
    ('surf_a','anode surface concentrations'),
    ('soc_c','cathode state of charge'),
    ('soc_a','anode state of charge'),
    ('csld_c','solid concentrations of cathode particles'),
    ('csld_a','solid concentrations of anode particles'),
    ('cbarLine_c','average concentration in each cathode particle'),
    ('cbarLine_a','average concentration in each anode particle'),
    ('cbar_full','average solid concentrations (movie)'),
    ('cbar_c','average cathode solid concentrations (movie)'),
    ('cbar_a','average anode solid concentrations (movie)'),
    ('bulkp_c','macroscopic cathode solid phase potential(movie)'),
    ('bulkp_a','macroscopic anode solid phase potential (movie)'),
    ('text','convert the output to plain text (csv)')
])

#Generate help description for plotTypes
plotTypesHelp="""The type of plot to be generated:"""
for key,value in plotTypes.items():
    plotTypesHelp = plotTypesHelp + '\n' + key
    if len(key)<7:
        plotTypesHelp = plotTypesHelp + '\t'
    plotTypesHelp = plotTypesHelp + '\t' + value

parser = argparse.ArgumentParser(description='Process and plot results generated by mpetrun.py.', formatter_class=RawTextHelpFormatter)
parser.add_argument('directory', help='location of the mpet results')
parser.add_argument('plotType', metavar='plotType', help=plotTypesHelp, choices=plotTypes.keys())
parser.add_argument('save',nargs='?',choices=['save'],help='Optionally save the output')

parser.add_argument('-v','--version', action='version',
                    version='%(prog)s '+__version__)
args = parser.parse_args()


# Get input file from script parameters
if len(sys.argv) < 2:
    raise Exception("Need input data directory name")
indir = sys.argv[1]
if not os.path.exists(os.path.join(os.getcwd(), indir)):
    raise Exception("Input file doesn't exist")
# Optionally just convert output to text
if len(sys.argv) == 3 and sys.argv[2] == "text":
    outmat2txt.main(indir)
    sys.exit()
# Get plot type from script parameters
plots = []
if len(sys.argv) > 2:
    plots.append(sys.argv[2])
else:
    plots.append("v")
# Save the plot instead of showing on screen?
# Get from script parameters
save_flag = False
print_flag = True
data_only = False
save_only = False
if len(sys.argv) > 3:
    if sys.argv[3] in ["save", "saveonly"]:
        save_flag = True
        if sys.argv[3] == "saveonly":
            save_only = True
            print_flag = False
    else:
        for i in range(3, len(sys.argv)):
            plots.append(sys.argv[i])
out = []
for plot_type in plots:
    out.append(plot_data.show_data(
        indir, plot_type, print_flag, save_flag, data_only))
if not save_only:
    plt.show()
