################################################################################
#
# Copyright 2021-2022 Rocco Matano
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
#
################################################################################

from types import SimpleNamespace as _namespace
import ctypes
from .wtypes import (
    DWORD,
    HANDLE,
    PWSTR,
    UINT,
    WCHAR,
    )
ref = ctypes.byref

################################################################################

version = "1.10.0"

################################################################################

def raise_if(condition):
    if condition:
        raise ctypes.WinError()

################################################################################

def raise_on_zero(value):
    if value == 0 or getattr(value, "value", 1) is None:
        raise ctypes.WinError()

################################################################################

def raise_on_err(err):
    if err:
        raise ctypes.WinError(err)

################################################################################

def raise_on_hr(hr):
    if hr < 0:
        raise ctypes.WinError(hr)

################################################################################

def fun_fact(function, signature):
    function.restype = signature[0]
    function.argtypes = signature[1:]
    return function

################################################################################

def multi_str_from_addr(addr):
    WCHAR_SIZE = ctypes.sizeof(WCHAR)
    end = addr
    while True:
        slen = len(ctypes.cast(end, PWSTR).value)
        if slen:
            end += (slen + 1) * WCHAR_SIZE
        else:
            # +WCHAR_SIZE for final null
            size = (end + WCHAR_SIZE - addr) // WCHAR_SIZE
            return ctypes.wstring_at(addr, size)

################################################################################

def cmdline_from_args(args):
    BS = "\\"
    parts = []
    for arg in args:
        bs_accu = []
        if parts:
            parts.append(" ")
        need_qmark = (" " in arg) or ("\t" in arg) or not arg
        if need_qmark:
            parts.append('"')
        for c in arg:
            if c == BS:
                bs_accu.append(c)
            elif c == '"':
                parts.append(BS * len(bs_accu) * 2)
                bs_accu = []
                parts.append(r'\"')
            else:
                if bs_accu:
                    parts.extend(bs_accu)
                    bs_accu = []
                parts.append(c)
        if bs_accu:
            parts.extend(bs_accu)
        if need_qmark:
            parts.extend(bs_accu)
            parts.append('"')
    return "".join(parts)

################################################################################

def ns_from_struct(cts):
    return _namespace(**{f: getattr(cts, f) for f, _ in cts._fields_})

################################################################################

INVALID_HANDLE_VALUE = HANDLE(-1).value
HGDI_ERROR = INVALID_HANDLE_VALUE

DBT_CONFIGCHANGECANCELED = 25
DBT_CONFIGCHANGED = 24
DBT_CUSTOMEVENT = 32774
DBT_DEVICEARRIVAL = 32768
DBT_DEVICEQUERYREMOVE = 32769
DBT_DEVICEQUERYREMOVEFAILED = 32770
DBT_DEVICEREMOVECOMPLETE = 32772
DBT_DEVICEREMOVEPENDING = 32771
DBT_DEVICETYPESPECIFIC = 32773
DBT_QUERYCHANGECONFIG = 23

SC_ACTION_NONE = 0
SC_ACTION_REBOOT = 2
SC_ACTION_RESTART = 1
SC_ACTION_RUN_COMMAND = 3
SC_ENUM_PROCESS_INFO = 0
SC_GROUP_IDENTIFIER = 43
SC_MANAGER_ALL_ACCESS = 983103
SC_MANAGER_CONNECT = 1
SC_MANAGER_CREATE_SERVICE = 2
SC_MANAGER_ENUMERATE_SERVICE = 4
SC_MANAGER_LOCK = 8
SC_MANAGER_MODIFY_BOOT_CONFIG = 32
SC_MANAGER_QUERY_LOCK_STATUS = 16
SC_STATUS_PROCESS_INFO = 0
SERVICE_ACCEPT_HARDWAREPROFILECHANGE = 32
SERVICE_ACCEPT_NETBINDCHANGE = 16
SERVICE_ACCEPT_PARAMCHANGE = 8
SERVICE_ACCEPT_PAUSE_CONTINUE = 2
SERVICE_ACCEPT_POWEREVENT = 64
SERVICE_ACCEPT_PRESHUTDOWN = 256
SERVICE_ACCEPT_SESSIONCHANGE = 128
SERVICE_ACCEPT_SHUTDOWN = 4
SERVICE_ACCEPT_STOP = 1
SERVICE_ACTIVE = 1
SERVICE_ALL_ACCESS = 983551
SERVICE_CHANGE_CONFIG = 2
SERVICE_CONFIG_DELAYED_AUTO_START_INFO = 3
SERVICE_CONFIG_DESCRIPTION = 1
SERVICE_CONFIG_FAILURE_ACTIONS = 2
SERVICE_CONFIG_FAILURE_ACTIONS_FLAG = 4
SERVICE_CONFIG_PRESHUTDOWN_INFO = 7
SERVICE_CONFIG_REQUIRED_PRIVILEGES_INFO = 6
SERVICE_CONFIG_SERVICE_SID_INFO = 5
SERVICE_CONTINUE_PENDING = 5
SERVICE_CONTROL_CONTINUE = 3
SERVICE_CONTROL_DEVICEEVENT = 11
SERVICE_CONTROL_HARDWAREPROFILECHANGE = 12
SERVICE_CONTROL_INTERROGATE = 4
SERVICE_CONTROL_NETBINDADD = 7
SERVICE_CONTROL_NETBINDDISABLE = 10
SERVICE_CONTROL_NETBINDENABLE = 9
SERVICE_CONTROL_NETBINDREMOVE = 8
SERVICE_CONTROL_PARAMCHANGE = 6
SERVICE_CONTROL_PAUSE = 2
SERVICE_CONTROL_POWEREVENT = 13
SERVICE_CONTROL_PRESHUTDOWN = 15
SERVICE_CONTROL_SESSIONCHANGE = 14
SERVICE_CONTROL_SHUTDOWN = 5
SERVICE_CONTROL_STOP = 1
SERVICE_DRIVER = 11
SERVICE_ENUMERATE_DEPENDENTS = 8
SERVICE_FILE_SYSTEM_DRIVER = 2
SERVICE_INACTIVE = 2
SERVICE_INTERACTIVE_PROCESS = 256
SERVICE_INTERROGATE = 128
SERVICE_KERNEL_DRIVER = 1
SERVICE_NO_CHANGE = -1
SERVICE_PAUSED = 7
SERVICE_PAUSE_CONTINUE = 64
SERVICE_PAUSE_PENDING = 6
SERVICE_QUERY_CONFIG = 1
SERVICE_QUERY_STATUS = 4
SERVICE_RUNNING = 4
SERVICE_SID_TYPE_NONE = 0
SERVICE_SID_TYPE_RESTRICTED = 3
SERVICE_SID_TYPE_UNRESTRICTED = 1
SERVICE_SPECIFIC_ERROR = 1066
SERVICE_START = 16
SERVICE_START_PENDING = 2
SERVICE_STATE_ALL = 3
SERVICE_STOP = 32
SERVICE_STOPPED = 1
SERVICE_STOP_PENDING = 3
SERVICE_USER_DEFINED_CONTROL = 256
SERVICE_WIN32_OWN_PROCESS = 16
SERVICE_WIN32_SHARE_PROCESS = 32
SERVICE_WIN32 = SERVICE_WIN32_OWN_PROCESS | SERVICE_WIN32_SHARE_PROCESS
SERVICE_BOOT_START = 0
SERVICE_SYSTEM_START = 1
SERVICE_AUTO_START = 2
SERVICE_DEMAND_START = 3
SERVICE_DISABLED = 4
SERVICE_ERROR_IGNORE = 0
SERVICE_ERROR_NORMAL = 1
SERVICE_ERROR_SEVERE = 2
SERVICE_ERROR_CRITICAL = 3

UOI_FLAGS = 1
UOI_NAME = 2
UOI_TYPE = 3
UOI_USER_SID = 4
WSF_VISIBLE = 1

SE_MIN_WELL_KNOWN_PRIVILEGE = 2
SE_CREATE_TOKEN_PRIVILEGE = 2
SE_ASSIGNPRIMARYTOKEN_PRIVILEGE = 3
SE_LOCK_MEMORY_PRIVILEGE = 4
SE_INCREASE_QUOTA_PRIVILEGE = 5
SE_MACHINE_ACCOUNT_PRIVILEGE = 6
SE_TCB_PRIVILEGE = 7
SE_SECURITY_PRIVILEGE = 8
SE_TAKE_OWNERSHIP_PRIVILEGE = 9
SE_LOAD_DRIVER_PRIVILEGE = 10
SE_SYSTEM_PROFILE_PRIVILEGE = 11
SE_SYSTEMTIME_PRIVILEGE = 12
SE_PROF_SINGLE_PROCESS_PRIVILEGE = 13
SE_INC_BASE_PRIORITY_PRIVILEGE = 14
SE_CREATE_PAGEFILE_PRIVILEGE = 15
SE_CREATE_PERMANENT_PRIVILEGE = 16
SE_BACKUP_PRIVILEGE = 17
SE_RESTORE_PRIVILEGE = 18
SE_SHUTDOWN_PRIVILEGE = 19
SE_DEBUG_PRIVILEGE = 20
SE_AUDIT_PRIVILEGE = 21
SE_SYSTEM_ENVIRONMENT_PRIVILEGE = 22
SE_CHANGE_NOTIFY_PRIVILEGE = 23
SE_REMOTE_SHUTDOWN_PRIVILEGE = 24
SE_UNDOCK_PRIVILEGE = 25
SE_SYNC_AGENT_PRIVILEGE = 26
SE_ENABLE_DELEGATION_PRIVILEGE = 27
SE_MANAGE_VOLUME_PRIVILEGE = 28
SE_IMPERSONATE_PRIVILEGE = 29
SE_CREATE_GLOBAL_PRIVILEGE = 30
SE_TRUSTED_CREDMAN_ACCESS_PRIVILEGE = 31
SE_RELABEL_PRIVILEGE = 32
SE_INC_WORKING_SET_PRIVILEGE = 33
SE_TIME_ZONE_PRIVILEGE = 34
SE_CREATE_SYMBOLIC_LINK_PRIVILEGE = 35
SE_DELEGATE_SESSION_USER_IMPERSONATE_PRIVILEGE = 36
SE_MAX_WELL_KNOWN_PRIVILEGE = SE_DELEGATE_SESSION_USER_IMPERSONATE_PRIVILEGE

DELETE = 65536
READ_CONTROL = 131072
WRITE_DAC = 262144
WRITE_OWNER = 524288
SYNCHRONIZE = 1048576
STANDARD_RIGHTS_REQUIRED = 983040
STANDARD_RIGHTS_READ = READ_CONTROL
STANDARD_RIGHTS_WRITE = READ_CONTROL
STANDARD_RIGHTS_EXECUTE = READ_CONTROL
STANDARD_RIGHTS_ALL = 2031616
SPECIFIC_RIGHTS_ALL = 65535
ACCESS_SYSTEM_SECURITY = 16777216
MAXIMUM_ALLOWED = 33554432
GENERIC_READ = -2147483648
GENERIC_WRITE = 1073741824
GENERIC_EXECUTE = 536870912
GENERIC_ALL = 268435456

FILE_READ_DATA = 1
FILE_LIST_DIRECTORY = 1
FILE_WRITE_DATA = 2
FILE_ADD_FILE = 2
FILE_APPEND_DATA = 4
FILE_ADD_SUBDIRECTORY = 4
FILE_CREATE_PIPE_INSTANCE = 4
FILE_READ_EA = 8
FILE_WRITE_EA = 16
FILE_EXECUTE = 32
FILE_TRAVERSE = 32
FILE_DELETE_CHILD = 64
FILE_READ_ATTRIBUTES = 128
FILE_WRITE_ATTRIBUTES = 256
FILE_ALL_ACCESS = STANDARD_RIGHTS_REQUIRED | SYNCHRONIZE | 511
FILE_GENERIC_READ = (
    STANDARD_RIGHTS_READ
    | FILE_READ_DATA
    | FILE_READ_ATTRIBUTES
    | FILE_READ_EA
    | SYNCHRONIZE
    )
FILE_GENERIC_WRITE = (
    STANDARD_RIGHTS_WRITE
    | FILE_WRITE_DATA
    | FILE_WRITE_ATTRIBUTES
    | FILE_WRITE_EA
    | FILE_APPEND_DATA
    | SYNCHRONIZE
    )
FILE_GENERIC_EXECUTE = (
    STANDARD_RIGHTS_EXECUTE | FILE_READ_ATTRIBUTES | FILE_EXECUTE | SYNCHRONIZE
    )

SECURITY_NULL_SID_AUTHORITY = (0, 0, 0, 0, 0, 0)
SECURITY_WORLD_SID_AUTHORITY = (0, 0, 0, 0, 0, 1)
SECURITY_LOCAL_SID_AUTHORITY = (0, 0, 0, 0, 0, 2)
SECURITY_CREATOR_SID_AUTHORITY = (0, 0, 0, 0, 0, 3)
SECURITY_NON_UNIQUE_AUTHORITY = (0, 0, 0, 0, 0, 4)
SECURITY_RESOURCE_MANAGER_AUTHORITY = (0, 0, 0, 0, 0, 9)

SECURITY_NULL_RID = 0
SECURITY_WORLD_RID = 0
SECURITY_LOCAL_RID = 0x00000000

SECURITY_CREATOR_OWNER_RID = 0
SECURITY_CREATOR_GROUP_RID = 1

SECURITY_CREATOR_OWNER_SERVER_RID = 2
SECURITY_CREATOR_GROUP_SERVER_RID = 3
SECURITY_CREATOR_OWNER_RIGHTS_RID = 4

SECURITY_NT_AUTHORITY = (0, 0, 0, 0, 0, 5)

SECURITY_DIALUP_RID = 1
SECURITY_NETWORK_RID = 2
SECURITY_BATCH_RID = 3
SECURITY_INTERACTIVE_RID = 4
SECURITY_SERVICE_RID = 6
SECURITY_ANONYMOUS_LOGON_RID = 7
SECURITY_PROXY_RID = 8
SECURITY_SERVER_LOGON_RID = 9

SECURITY_LOGON_IDS_RID = 5
SECURITY_LOGON_IDS_RID_COUNT = 3

SECURITY_LOCAL_SYSTEM_RID = 18

SECURITY_NT_NON_UNIQUE = 21

SECURITY_BUILTIN_DOMAIN_RID = 32

DOMAIN_USER_RID_ADMIN = 500
DOMAIN_USER_RID_GUEST = 501
DOMAIN_USER_RID_KRBTGT = 502
DOMAIN_USER_RID_MAX = 999

DOMAIN_GROUP_RID_ADMINS = 512
DOMAIN_GROUP_RID_USERS = 513
DOMAIN_GROUP_RID_GUESTS = 514
DOMAIN_GROUP_RID_COMPUTERS = 515
DOMAIN_GROUP_RID_CONTROLLERS = 516
DOMAIN_GROUP_RID_CERT_ADMINS = 517
DOMAIN_GROUP_RID_SCHEMA_ADMINS = 518
DOMAIN_GROUP_RID_ENTERPRISE_ADMINS = 519
DOMAIN_GROUP_RID_POLICY_ADMINS = 520
DOMAIN_GROUP_RID_READONLY_CONTROLLERS = 521

DOMAIN_ALIAS_RID_ADMINS = 544
DOMAIN_ALIAS_RID_USERS = 545
DOMAIN_ALIAS_RID_GUESTS = 546
DOMAIN_ALIAS_RID_POWER_USERS = 547
DOMAIN_ALIAS_RID_ACCOUNT_OPS = 548
DOMAIN_ALIAS_RID_SYSTEM_OPS = 549
DOMAIN_ALIAS_RID_PRINT_OPS = 550
DOMAIN_ALIAS_RID_BACKUP_OPS = 551
DOMAIN_ALIAS_RID_REPLICATOR = 552
DOMAIN_ALIAS_RID_RAS_SERVERS = 553
DOMAIN_ALIAS_RID_PREW2KCOMPACCESS = 554
DOMAIN_ALIAS_RID_REMOTE_DESKTOP_USERS = 555
DOMAIN_ALIAS_RID_NETWORK_CONFIGURATION_OPS = 556
DOMAIN_ALIAS_RID_INCOMING_FOREST_TRUST_BUILDERS = 557
DOMAIN_ALIAS_RID_MONITORING_USERS = 558
DOMAIN_ALIAS_RID_LOGGING_USERS = 559
DOMAIN_ALIAS_RID_AUTHORIZATIONACCESS = 560
DOMAIN_ALIAS_RID_TS_LICENSE_SERVERS = 561
DOMAIN_ALIAS_RID_DCOM_USERS = 562
DOMAIN_ALIAS_RID_IUSERS = 568
DOMAIN_ALIAS_RID_CRYPTO_OPERATORS = 569
DOMAIN_ALIAS_RID_CACHEABLE_PRINCIPALS_GROUP = 571
DOMAIN_ALIAS_RID_NON_CACHEABLE_PRINCIPALS_GROUP = 572
DOMAIN_ALIAS_RID_EVENT_LOG_READERS_GROUP = 573

SECURITY_MANDATORY_LABEL_AUTHORITY = (0, 0, 0, 0, 0, 16)
SECURITY_MANDATORY_UNTRUSTED_RID = 0x00000000
SECURITY_MANDATORY_LOW_RID = 0x00001000
SECURITY_MANDATORY_MEDIUM_RID = 0x00002000
SECURITY_MANDATORY_HIGH_RID = 0x00003000
SECURITY_MANDATORY_SYSTEM_RID = 0x00004000
SECURITY_MANDATORY_PROTECTED_PROCESS_RID = 0x00005000
SECURITY_MANDATORY_MAXIMUM_USER_RID = SECURITY_MANDATORY_SYSTEM_RID

SYSTEM_LUID = (999, 0)
ANONYMOUS_LOGON_LUID = (998, 0)
LOCALSERVICE_LUID = (997, 0)
NETWORKSERVICE_LUID = (996, 0)
IUSER_LUID = (995, 0)

SE_GROUP_MANDATORY = 1
SE_GROUP_ENABLED_BY_DEFAULT = 2
SE_GROUP_ENABLED = 4
SE_GROUP_OWNER = 8
SE_GROUP_USE_FOR_DENY_ONLY = 16
SE_GROUP_INTEGRITY = 32
SE_GROUP_INTEGRITY_ENABLED = 64
SE_GROUP_RESOURCE = 536870912
SE_GROUP_LOGON_ID = -1073741824

ACCESS_MIN_MS_ACE_TYPE = 0
ACCESS_ALLOWED_ACE_TYPE = 0
ACCESS_DENIED_ACE_TYPE = 1
SYSTEM_AUDIT_ACE_TYPE = 2
SYSTEM_ALARM_ACE_TYPE = 3
ACCESS_MAX_MS_V2_ACE_TYPE = 3
ACCESS_ALLOWED_COMPOUND_ACE_TYPE = 4
ACCESS_MAX_MS_V3_ACE_TYPE = 4
ACCESS_MIN_MS_OBJECT_ACE_TYPE = 5
ACCESS_ALLOWED_OBJECT_ACE_TYPE = 5
ACCESS_DENIED_OBJECT_ACE_TYPE = 6
SYSTEM_AUDIT_OBJECT_ACE_TYPE = 7
SYSTEM_ALARM_OBJECT_ACE_TYPE = 8
ACCESS_MAX_MS_OBJECT_ACE_TYPE = 8
ACCESS_MAX_MS_V4_ACE_TYPE = 8
ACCESS_MAX_MS_ACE_TYPE = 8
ACCESS_ALLOWED_CALLBACK_ACE_TYPE = 9
ACCESS_DENIED_CALLBACK_ACE_TYPE = 10
ACCESS_ALLOWED_CALLBACK_OBJECT_ACE_TYPE = 11
ACCESS_DENIED_CALLBACK_OBJECT_ACE_TYPE = 12
SYSTEM_AUDIT_CALLBACK_ACE_TYPE = 13
SYSTEM_ALARM_CALLBACK_ACE_TYPE = 14
SYSTEM_AUDIT_CALLBACK_OBJECT_ACE_TYPE = 15
SYSTEM_ALARM_CALLBACK_OBJECT_ACE_TYPE = 16
SYSTEM_MANDATORY_LABEL_ACE_TYPE = 17
ACCESS_MAX_MS_V5_ACE_TYPE = 17

OBJECT_INHERIT_ACE = 1
CONTAINER_INHERIT_ACE = 2
NO_PROPAGATE_INHERIT_ACE = 4
INHERIT_ONLY_ACE = 8
VALID_INHERIT_FLAGS = 15

SUCCESSFUL_ACCESS_ACE_FLAG = 64
FAILED_ACCESS_ACE_FLAG = 128

SE_OWNER_DEFAULTED = 1
SE_GROUP_DEFAULTED = 2
SE_DACL_PRESENT = 4
SE_DACL_DEFAULTED = 8
SE_SACL_PRESENT = 16
SE_SACL_DEFAULTED = 32
SE_SELF_RELATIVE = 32768

SE_PRIVILEGE_ENABLED_BY_DEFAULT = 1
SE_PRIVILEGE_ENABLED = 2
SE_PRIVILEGE_USED_FOR_ACCESS = -2147483648

PRIVILEGE_SET_ALL_NECESSARY = 1

SE_CREATE_TOKEN_NAME = "SeCreateTokenPrivilege"
SE_ASSIGNPRIMARYTOKEN_NAME = "SeAssignPrimaryTokenPrivilege"
SE_LOCK_MEMORY_NAME = "SeLockMemoryPrivilege"
SE_INCREASE_QUOTA_NAME = "SeIncreaseQuotaPrivilege"
SE_UNSOLICITED_INPUT_NAME = "SeUnsolicitedInputPrivilege"
SE_MACHINE_ACCOUNT_NAME = "SeMachineAccountPrivilege"
SE_TCB_NAME = "SeTcbPrivilege"
SE_SECURITY_NAME = "SeSecurityPrivilege"
SE_TAKE_OWNERSHIP_NAME = "SeTakeOwnershipPrivilege"
SE_LOAD_DRIVER_NAME = "SeLoadDriverPrivilege"
SE_SYSTEM_PROFILE_NAME = "SeSystemProfilePrivilege"
SE_SYSTEMTIME_NAME = "SeSystemtimePrivilege"
SE_PROF_SINGLE_PROCESS_NAME = "SeProfileSingleProcessPrivilege"
SE_INC_BASE_PRIORITY_NAME = "SeIncreaseBasePriorityPrivilege"
SE_CREATE_PAGEFILE_NAME = "SeCreatePagefilePrivilege"
SE_CREATE_PERMANENT_NAME = "SeCreatePermanentPrivilege"
SE_BACKUP_NAME = "SeBackupPrivilege"
SE_RESTORE_NAME = "SeRestorePrivilege"
SE_SHUTDOWN_NAME = "SeShutdownPrivilege"
SE_DEBUG_NAME = "SeDebugPrivilege"
SE_AUDIT_NAME = "SeAuditPrivilege"
SE_SYSTEM_ENVIRONMENT_NAME = "SeSystemEnvironmentPrivilege"
SE_CHANGE_NOTIFY_NAME = "SeChangeNotifyPrivilege"
SE_REMOTE_SHUTDOWN_NAME = "SeRemoteShutdownPrivilege"
SE_UNDOCK_NAME = "SeUndockPrivilege"
SE_SYNC_AGENT_NAME = "SeSyncAgentPrivilege"
SE_ENABLE_DELEGATION_NAME = "SeEnableDelegationPrivilege"
SE_MANAGE_VOLUME_NAME = "SeManageVolumePrivilege"
SE_IMPERSONATE_NAME = "SeImpersonatePrivilege"
SE_CREATE_GLOBAL_NAME = "SeCreateGlobalPrivilege"
SE_TRUSTED_CREDMAN_ACCESS_NAME = "SeTrustedCredManAccessPrivilege"
SE_RELABEL_NAME = "SeRelabelPrivilege"
SE_INC_WORKING_SET_NAME = "SeIncreaseWorkingSetPrivilege"
SE_TIME_ZONE_NAME = "SeTimeZonePrivilege"
SE_CREATE_SYMBOLIC_LINK_NAME = "SeCreateSymbolicLinkPrivilege"
SE_DELEGATE_SESSION_USER_IMPERSONATE_NAME = (
    "SeDelegateSessionUserImpersonatePrivilege"
    )

SecurityAnonymous = 0
SecurityIdentification = 1
SecurityImpersonation = 2
SecurityDelegation = 3

SECURITY_MAX_IMPERSONATION_LEVEL = SecurityDelegation

DEFAULT_IMPERSONATION_LEVEL = SecurityImpersonation

TOKEN_ASSIGN_PRIMARY = 1
TOKEN_DUPLICATE = 2
TOKEN_IMPERSONATE = 4
TOKEN_QUERY = 8
TOKEN_QUERY_SOURCE = 16
TOKEN_ADJUST_PRIVILEGES = 32
TOKEN_ADJUST_GROUPS = 64
TOKEN_ADJUST_DEFAULT = 128

TOKEN_ALL_ACCESS = (
    STANDARD_RIGHTS_REQUIRED
    | TOKEN_ASSIGN_PRIMARY
    | TOKEN_DUPLICATE
    | TOKEN_IMPERSONATE
    | TOKEN_QUERY
    | TOKEN_QUERY_SOURCE
    | TOKEN_ADJUST_PRIVILEGES
    | TOKEN_ADJUST_GROUPS
    | TOKEN_ADJUST_DEFAULT
    )

TOKEN_READ = STANDARD_RIGHTS_READ | TOKEN_QUERY

TOKEN_WRITE = (
    STANDARD_RIGHTS_WRITE
    | TOKEN_ADJUST_PRIVILEGES
    | TOKEN_ADJUST_GROUPS
    | TOKEN_ADJUST_DEFAULT
    )

TOKEN_EXECUTE = STANDARD_RIGHTS_EXECUTE

SidTypeUser = 1
SidTypeGroup = 2
SidTypeDomain = 3
SidTypeAlias = 4
SidTypeWellKnownGroup = 5
SidTypeDeletedAccount = 6
SidTypeInvalid = 7
SidTypeUnknown = 8
SidTypeComputer = 9
SidTypeLabel = 10

TokenPrimary = 1
TokenImpersonation = 2

TokenUser = 1
TokenGroups = 2
TokenPrivileges = 3
TokenOwner = 4
TokenPrimaryGroup = 5
TokenDefaultDacl = 6
TokenSource = 7
TokenType = 8
TokenImpersonationLevel = 9
TokenStatistics = 10
TokenRestrictedSids = 11
TokenSessionId = 12
TokenGroupsAndPrivileges = 13
TokenSessionReference = 14
TokenSandBoxInert = 15
TokenAuditPolicy = 16
TokenOrigin = 17
TokenElevationType = 18
TokenLinkedToken = 19
TokenElevation = 20
TokenHasRestrictions = 21
TokenAccessInformation = 22
TokenVirtualizationAllowed = 23
TokenVirtualizationEnabled = 24
TokenIntegrityLevel = 25
TokenUIAccess = 26
TokenMandatoryPolicy = 27
TokenLogonSid = 28

WAIT_FAILED = 0xFFFFFFFF
WAIT_OBJECT_0 = 0
WAIT_ABANDONED = 0x00000080
WAIT_ABANDONED_0 = WAIT_ABANDONED
WAIT_TIMEOUT = 258
WAIT_IO_COMPLETION = 0x000000C0
MAXIMUM_WAIT_OBJECTS = 64
INFINITE = 0xFFFFFFFF

ERROR_SUCCESS = 0
NO_ERROR = 0
SEC_E_OK = 0x00000000
ERROR_INVALID_FUNCTION = 1
ERROR_FILE_NOT_FOUND = 2
ERROR_PATH_NOT_FOUND = 3
ERROR_TOO_MANY_OPEN_FILES = 4
ERROR_ACCESS_DENIED = 5
ERROR_INVALID_HANDLE = 6
ERROR_ARENA_TRASHED = 7
ERROR_NOT_ENOUGH_MEMORY = 8
ERROR_INVALID_BLOCK = 9
ERROR_BAD_ENVIRONMENT = 10
ERROR_BAD_FORMAT = 11
ERROR_INVALID_ACCESS = 12
ERROR_INVALID_DATA = 13
ERROR_OUTOFMEMORY = 14
ERROR_INVALID_DRIVE = 15
ERROR_CURRENT_DIRECTORY = 16
ERROR_NOT_SAME_DEVICE = 17
ERROR_NO_MORE_FILES = 18
ERROR_WRITE_PROTECT = 19
ERROR_BAD_UNIT = 20
ERROR_NOT_READY = 21
ERROR_BAD_COMMAND = 22
ERROR_CRC = 23
ERROR_BAD_LENGTH = 24
ERROR_SEEK = 25
ERROR_NOT_DOS_DISK = 26
ERROR_SECTOR_NOT_FOUND = 27
ERROR_OUT_OF_PAPER = 28
ERROR_WRITE_FAULT = 29
ERROR_READ_FAULT = 30
ERROR_GEN_FAILURE = 31
ERROR_SHARING_VIOLATION = 32
ERROR_LOCK_VIOLATION = 33
ERROR_WRONG_DISK = 34
ERROR_SHARING_BUFFER_EXCEEDED = 36
ERROR_HANDLE_EOF = 38
ERROR_HANDLE_DISK_FULL = 39
ERROR_NOT_SUPPORTED = 50
ERROR_REM_NOT_LIST = 51
ERROR_DUP_NAME = 52
ERROR_BAD_NETPATH = 53
ERROR_NETWORK_BUSY = 54
ERROR_DEV_NOT_EXIST = 55
ERROR_TOO_MANY_CMDS = 56
ERROR_ADAP_HDW_ERR = 57
ERROR_BAD_NET_RESP = 58
ERROR_UNEXP_NET_ERR = 59
ERROR_BAD_REM_ADAP = 60
ERROR_PRINTQ_FULL = 61
ERROR_NO_SPOOL_SPACE = 62
ERROR_PRINT_CANCELLED = 63
ERROR_NETNAME_DELETED = 64
ERROR_NETWORK_ACCESS_DENIED = 65
ERROR_BAD_DEV_TYPE = 66
ERROR_BAD_NET_NAME = 67
ERROR_TOO_MANY_NAMES = 68
ERROR_TOO_MANY_SESS = 69
ERROR_SHARING_PAUSED = 70
ERROR_REQ_NOT_ACCEP = 71
ERROR_REDIR_PAUSED = 72
ERROR_FILE_EXISTS = 80
ERROR_CANNOT_MAKE = 82
ERROR_FAIL_I24 = 83
ERROR_OUT_OF_STRUCTURES = 84
ERROR_ALREADY_ASSIGNED = 85
ERROR_INVALID_PASSWORD = 86
ERROR_INVALID_PARAMETER = 87
ERROR_NET_WRITE_FAULT = 88
ERROR_NO_PROC_SLOTS = 89
ERROR_TOO_MANY_SEMAPHORES = 100
ERROR_EXCL_SEM_ALREADY_OWNED = 101
ERROR_SEM_IS_SET = 102
ERROR_TOO_MANY_SEM_REQUESTS = 103
ERROR_INVALID_AT_INTERRUPT_TIME = 104
ERROR_SEM_OWNER_DIED = 105
ERROR_SEM_USER_LIMIT = 106
ERROR_DISK_CHANGE = 107
ERROR_DRIVE_LOCKED = 108
ERROR_BROKEN_PIPE = 109
ERROR_OPEN_FAILED = 110
ERROR_BUFFER_OVERFLOW = 111
ERROR_DISK_FULL = 112
ERROR_NO_MORE_SEARCH_HANDLES = 113
ERROR_INVALID_TARGET_HANDLE = 114
ERROR_INVALID_CATEGORY = 117
ERROR_INVALID_VERIFY_SWITCH = 118
ERROR_BAD_DRIVER_LEVEL = 119
ERROR_CALL_NOT_IMPLEMENTED = 120
ERROR_SEM_TIMEOUT = 121
ERROR_INSUFFICIENT_BUFFER = 122
ERROR_INVALID_NAME = 123
ERROR_INVALID_LEVEL = 124
ERROR_NO_VOLUME_LABEL = 125
ERROR_MOD_NOT_FOUND = 126
ERROR_PROC_NOT_FOUND = 127
ERROR_WAIT_NO_CHILDREN = 128
ERROR_CHILD_NOT_COMPLETE = 129
ERROR_DIRECT_ACCESS_HANDLE = 130
ERROR_NEGATIVE_SEEK = 131
ERROR_SEEK_ON_DEVICE = 132
ERROR_IS_JOIN_TARGET = 133
ERROR_IS_JOINED = 134
ERROR_IS_SUBSTED = 135
ERROR_NOT_JOINED = 136
ERROR_NOT_SUBSTED = 137
ERROR_JOIN_TO_JOIN = 138
ERROR_SUBST_TO_SUBST = 139
ERROR_JOIN_TO_SUBST = 140
ERROR_SUBST_TO_JOIN = 141
ERROR_BUSY_DRIVE = 142
ERROR_SAME_DRIVE = 143
ERROR_DIR_NOT_ROOT = 144
ERROR_DIR_NOT_EMPTY = 145
ERROR_IS_SUBST_PATH = 146
ERROR_IS_JOIN_PATH = 147
ERROR_PATH_BUSY = 148
ERROR_IS_SUBST_TARGET = 149
ERROR_SYSTEM_TRACE = 150
ERROR_INVALID_EVENT_COUNT = 151
ERROR_TOO_MANY_MUXWAITERS = 152
ERROR_INVALID_LIST_FORMAT = 153
ERROR_LABEL_TOO_LONG = 154
ERROR_TOO_MANY_TCBS = 155
ERROR_SIGNAL_REFUSED = 156
ERROR_DISCARDED = 157
ERROR_NOT_LOCKED = 158
ERROR_BAD_THREADID_ADDR = 159
ERROR_BAD_ARGUMENTS = 160
ERROR_BAD_PATHNAME = 161
ERROR_SIGNAL_PENDING = 162
ERROR_MAX_THRDS_REACHED = 164
ERROR_LOCK_FAILED = 167
ERROR_BUSY = 170
ERROR_DEVICE_SUPPORT_IN_PROGRESS = 171
ERROR_CANCEL_VIOLATION = 173
ERROR_ATOMIC_LOCKS_NOT_SUPPORTED = 174
ERROR_INVALID_SEGMENT_NUMBER = 180
ERROR_INVALID_ORDINAL = 182
ERROR_ALREADY_EXISTS = 183
ERROR_INVALID_FLAG_NUMBER = 186
ERROR_SEM_NOT_FOUND = 187
ERROR_INVALID_STARTING_CODESEG = 188
ERROR_INVALID_STACKSEG = 189
ERROR_INVALID_MODULETYPE = 190
ERROR_INVALID_EXE_SIGNATURE = 191
ERROR_EXE_MARKED_INVALID = 192
ERROR_BAD_EXE_FORMAT = 193
ERROR_ITERATED_DATA_EXCEEDS_64k = 194
ERROR_INVALID_MINALLOCSIZE = 195
ERROR_DYNLINK_FROM_INVALID_RING = 196
ERROR_IOPL_NOT_ENABLED = 197
ERROR_INVALID_SEGDPL = 198
ERROR_AUTODATASEG_EXCEEDS_64k = 199
ERROR_RING2SEG_MUST_BE_MOVABLE = 200
ERROR_RELOC_CHAIN_XEEDS_SEGLIM = 201
ERROR_INFLOOP_IN_RELOC_CHAIN = 202
ERROR_ENVVAR_NOT_FOUND = 203
ERROR_NO_SIGNAL_SENT = 205
ERROR_FILENAME_EXCED_RANGE = 206
ERROR_RING2_STACK_IN_USE = 207
ERROR_META_EXPANSION_TOO_LONG = 208
ERROR_INVALID_SIGNAL_NUMBER = 209
ERROR_THREAD_1_INACTIVE = 210
ERROR_LOCKED = 212
ERROR_TOO_MANY_MODULES = 214
ERROR_NESTING_NOT_ALLOWED = 215
ERROR_EXE_MACHINE_TYPE_MISMATCH = 216
ERROR_EXE_CANNOT_MODIFY_SIGNED_BINARY = 217
ERROR_EXE_CANNOT_MODIFY_STRONG_SIGNED_BINARY = 218
ERROR_FILE_CHECKED_OUT = 220
ERROR_CHECKOUT_REQUIRED = 221
ERROR_BAD_FILE_TYPE = 222
ERROR_FILE_TOO_LARGE = 223
ERROR_FORMS_AUTH_REQUIRED = 224
ERROR_VIRUS_INFECTED = 225
ERROR_VIRUS_DELETED = 226
ERROR_PIPE_LOCAL = 229
ERROR_BAD_PIPE = 230
ERROR_PIPE_BUSY = 231
ERROR_NO_DATA = 232
ERROR_PIPE_NOT_CONNECTED = 233
ERROR_MORE_DATA = 234
ERROR_NO_WORK_DONE = 235
ERROR_VC_DISCONNECTED = 240
ERROR_INVALID_EA_NAME = 254
ERROR_EA_LIST_INCONSISTENT = 255
ERROR_NO_MORE_ITEMS = 259
ERROR_CANNOT_COPY = 266
ERROR_DIRECTORY = 267
ERROR_EAS_DIDNT_FIT = 275
ERROR_EA_FILE_CORRUPT = 276
ERROR_EA_TABLE_FULL = 277
ERROR_INVALID_EA_HANDLE = 278
ERROR_EAS_NOT_SUPPORTED = 282
ERROR_NOT_OWNER = 288
ERROR_TOO_MANY_POSTS = 298
ERROR_PARTIAL_COPY = 299
ERROR_OPLOCK_NOT_GRANTED = 300
ERROR_INVALID_OPLOCK_PROTOCOL = 301
ERROR_DISK_TOO_FRAGMENTED = 302
ERROR_DELETE_PENDING = 303
ERROR_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING = 304
ERROR_SHORT_NAMES_NOT_ENABLED_ON_VOLUME = 305
ERROR_SECURITY_STREAM_IS_INCONSISTENT = 306
ERROR_INVALID_LOCK_RANGE = 307
ERROR_IMAGE_SUBSYSTEM_NOT_PRESENT = 308
ERROR_NOTIFICATION_GUID_ALREADY_DEFINED = 309
ERROR_INVALID_EXCEPTION_HANDLER = 310
ERROR_DUPLICATE_PRIVILEGES = 311
ERROR_NO_RANGES_PROCESSED = 312
ERROR_NOT_ALLOWED_ON_SYSTEM_FILE = 313
ERROR_DISK_RESOURCES_EXHAUSTED = 314
ERROR_INVALID_TOKEN = 315
ERROR_DEVICE_FEATURE_NOT_SUPPORTED = 316
ERROR_MR_MID_NOT_FOUND = 317
ERROR_SCOPE_NOT_FOUND = 318
ERROR_UNDEFINED_SCOPE = 319
ERROR_INVALID_CAP = 320
ERROR_DEVICE_UNREACHABLE = 321
ERROR_DEVICE_NO_RESOURCES = 322
ERROR_DATA_CHECKSUM_ERROR = 323
ERROR_INTERMIXED_KERNEL_EA_OPERATION = 324
ERROR_FILE_LEVEL_TRIM_NOT_SUPPORTED = 326
ERROR_OFFSET_ALIGNMENT_VIOLATION = 327
ERROR_INVALID_FIELD_IN_PARAMETER_LIST = 328
ERROR_OPERATION_IN_PROGRESS = 329
ERROR_BAD_DEVICE_PATH = 330
ERROR_TOO_MANY_DESCRIPTORS = 331
ERROR_SCRUB_DATA_DISABLED = 332
ERROR_NOT_REDUNDANT_STORAGE = 333
ERROR_RESIDENT_FILE_NOT_SUPPORTED = 334
ERROR_COMPRESSED_FILE_NOT_SUPPORTED = 335
ERROR_DIRECTORY_NOT_SUPPORTED = 336
ERROR_NOT_READ_FROM_COPY = 337
ERROR_FT_WRITE_FAILURE = 338
ERROR_FT_DI_SCAN_REQUIRED = 339
ERROR_INVALID_KERNEL_INFO_VERSION = 340
ERROR_INVALID_PEP_INFO_VERSION = 341
ERROR_OBJECT_NOT_EXTERNALLY_BACKED = 342
ERROR_EXTERNAL_BACKING_PROVIDER_UNKNOWN = 343
ERROR_COMPRESSION_NOT_BENEFICIAL = 344
ERROR_STORAGE_TOPOLOGY_ID_MISMATCH = 345
ERROR_BLOCKED_BY_PARENTAL_CONTROLS = 346
ERROR_BLOCK_TOO_MANY_REFERENCES = 347
ERROR_MARKED_TO_DISALLOW_WRITES = 348
ERROR_ENCLAVE_FAILURE = 349
ERROR_FAIL_NOACTION_REBOOT = 350
ERROR_FAIL_SHUTDOWN = 351
ERROR_FAIL_RESTART = 352
ERROR_MAX_SESSIONS_REACHED = 353
ERROR_NETWORK_ACCESS_DENIED_EDP = 354
ERROR_DEVICE_HINT_NAME_BUFFER_TOO_SMALL = 355
ERROR_EDP_POLICY_DENIES_OPERATION = 356
ERROR_EDP_DPL_POLICY_CANT_BE_SATISFIED = 357
ERROR_CLOUD_FILE_SYNC_ROOT_METADATA_CORRUPT = 358
ERROR_DEVICE_IN_MAINTENANCE = 359
ERROR_NOT_SUPPORTED_ON_DAX = 360
ERROR_DAX_MAPPING_EXISTS = 361
ERROR_CLOUD_FILE_PROVIDER_NOT_RUNNING = 362
ERROR_CLOUD_FILE_METADATA_CORRUPT = 363
ERROR_CLOUD_FILE_METADATA_TOO_LARGE = 364
ERROR_CLOUD_FILE_PROPERTY_BLOB_TOO_LARGE = 365
ERROR_CLOUD_FILE_PROPERTY_BLOB_CHECKSUM_MISMATCH = 366
ERROR_CHILD_PROCESS_BLOCKED = 367
ERROR_STORAGE_LOST_DATA_PERSISTENCE = 368
ERROR_FILE_SYSTEM_VIRTUALIZATION_UNAVAILABLE = 369
ERROR_FILE_SYSTEM_VIRTUALIZATION_METADATA_CORRUPT = 370
ERROR_FILE_SYSTEM_VIRTUALIZATION_BUSY = 371
ERROR_FILE_SYSTEM_VIRTUALIZATION_PROVIDER_UNKNOWN = 372
ERROR_GDI_HANDLE_LEAK = 373
ERROR_CLOUD_FILE_TOO_MANY_PROPERTY_BLOBS = 374
ERROR_CLOUD_FILE_PROPERTY_VERSION_NOT_SUPPORTED = 375
ERROR_NOT_A_CLOUD_FILE = 376
ERROR_CLOUD_FILE_NOT_IN_SYNC = 377
ERROR_CLOUD_FILE_ALREADY_CONNECTED = 378
ERROR_CLOUD_FILE_NOT_SUPPORTED = 379
ERROR_CLOUD_FILE_INVALID_REQUEST = 380
ERROR_CLOUD_FILE_READ_ONLY_VOLUME = 381
ERROR_CLOUD_FILE_CONNECTED_PROVIDER_ONLY = 382
ERROR_CLOUD_FILE_VALIDATION_FAILED = 383
ERROR_SMB1_NOT_AVAILABLE = 384
ERROR_FILE_SYSTEM_VIRTUALIZATION_INVALID_OPERATION = 385
ERROR_CLOUD_FILE_AUTHENTICATION_FAILED = 386
ERROR_CLOUD_FILE_INSUFFICIENT_RESOURCES = 387
ERROR_CLOUD_FILE_NETWORK_UNAVAILABLE = 388
ERROR_CLOUD_FILE_UNSUCCESSFUL = 389
ERROR_CLOUD_FILE_NOT_UNDER_SYNC_ROOT = 390
ERROR_CLOUD_FILE_IN_USE = 391
ERROR_CLOUD_FILE_PINNED = 392
ERROR_CLOUD_FILE_REQUEST_ABORTED = 393
ERROR_CLOUD_FILE_PROPERTY_CORRUPT = 394
ERROR_CLOUD_FILE_ACCESS_DENIED = 395
ERROR_CLOUD_FILE_INCOMPATIBLE_HARDLINKS = 396
ERROR_CLOUD_FILE_PROPERTY_LOCK_CONFLICT = 397
ERROR_CLOUD_FILE_REQUEST_CANCELED = 398
ERROR_EXTERNAL_SYSKEY_NOT_SUPPORTED = 399
ERROR_THREAD_MODE_ALREADY_BACKGROUND = 400
ERROR_THREAD_MODE_NOT_BACKGROUND = 401
ERROR_PROCESS_MODE_ALREADY_BACKGROUND = 402
ERROR_PROCESS_MODE_NOT_BACKGROUND = 403
ERROR_CLOUD_FILE_PROVIDER_TERMINATED = 404
ERROR_NOT_A_CLOUD_SYNC_ROOT = 405
ERROR_FILE_PROTECTED_UNDER_DPL = 406
ERROR_VOLUME_NOT_CLUSTER_ALIGNED = 407
ERROR_NO_PHYSICALLY_ALIGNED_FREE_SPACE_FOUND = 408
ERROR_APPX_FILE_NOT_ENCRYPTED = 409
ERROR_RWRAW_ENCRYPTED_FILE_NOT_ENCRYPTED = 410
ERROR_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILEOFFSET = 411
ERROR_RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILERANGE = 412
ERROR_RWRAW_ENCRYPTED_INVALID_EDATAINFO_PARAMETER = 413
ERROR_LINUX_SUBSYSTEM_NOT_PRESENT = 414
ERROR_FT_READ_FAILURE = 415
ERROR_STORAGE_RESERVE_ID_INVALID = 416
ERROR_STORAGE_RESERVE_DOES_NOT_EXIST = 417
ERROR_STORAGE_RESERVE_ALREADY_EXISTS = 418
ERROR_STORAGE_RESERVE_NOT_EMPTY = 419
ERROR_NOT_A_DAX_VOLUME = 420
ERROR_NOT_DAX_MAPPABLE = 421
ERROR_TIME_SENSITIVE_THREAD = 422
ERROR_DPL_NOT_SUPPORTED_FOR_USER = 423
ERROR_CASE_DIFFERING_NAMES_IN_DIR = 424
ERROR_FILE_NOT_SUPPORTED = 425
ERROR_CLOUD_FILE_REQUEST_TIMEOUT = 426
ERROR_NO_TASK_QUEUE = 427
ERROR_SRC_SRV_DLL_LOAD_FAILED = 428
ERROR_NOT_SUPPORTED_WITH_BTT = 429
ERROR_ENCRYPTION_DISABLED = 430
ERROR_ENCRYPTING_METADATA_DISALLOWED = 431
ERROR_CANT_CLEAR_ENCRYPTION_FLAG = 432
ERROR_NO_SUCH_DEVICE = 433
ERROR_CLOUD_FILE_DEHYDRATION_DISALLOWED = 434
ERROR_FILE_SNAP_IN_PROGRESS = 435
ERROR_FILE_SNAP_USER_SECTION_NOT_SUPPORTED = 436
ERROR_FILE_SNAP_MODIFY_NOT_SUPPORTED = 437
ERROR_FILE_SNAP_IO_NOT_COORDINATED = 438
ERROR_FILE_SNAP_UNEXPECTED_ERROR = 439
ERROR_FILE_SNAP_INVALID_PARAMETER = 440
ERROR_UNSATISFIED_DEPENDENCIES = 441
ERROR_CASE_SENSITIVE_PATH = 442
ERROR_UNEXPECTED_NTCACHEMANAGER_ERROR = 443
ERROR_CAPAUTHZ_NOT_DEVUNLOCKED = 450
ERROR_CAPAUTHZ_CHANGE_TYPE = 451
ERROR_CAPAUTHZ_NOT_PROVISIONED = 452
ERROR_CAPAUTHZ_NOT_AUTHORIZED = 453
ERROR_CAPAUTHZ_NO_POLICY = 454
ERROR_CAPAUTHZ_DB_CORRUPTED = 455
ERROR_CAPAUTHZ_SCCD_INVALID_CATALOG = 456
ERROR_CAPAUTHZ_SCCD_NO_AUTH_ENTITY = 457
ERROR_CAPAUTHZ_SCCD_PARSE_ERROR = 458
ERROR_CAPAUTHZ_SCCD_DEV_MODE_REQUIRED = 459
ERROR_CAPAUTHZ_SCCD_NO_CAPABILITY_MATCH = 460
ERROR_CIMFS_IMAGE_CORRUPT = 470
ERROR_PNP_QUERY_REMOVE_DEVICE_TIMEOUT = 480
ERROR_PNP_QUERY_REMOVE_RELATED_DEVICE_TIMEOUT = 481
ERROR_PNP_QUERY_REMOVE_UNRELATED_DEVICE_TIMEOUT = 482
ERROR_DEVICE_HARDWARE_ERROR = 483
ERROR_INVALID_ADDRESS = 487
ERROR_VRF_CFG_AND_IO_ENABLED = 1183
ERROR_PARTITION_TERMINATING = 1184
ERROR_USER_PROFILE_LOAD = 500
ERROR_ARITHMETIC_OVERFLOW = 534
ERROR_PIPE_CONNECTED = 535
ERROR_PIPE_LISTENING = 536
ERROR_VERIFIER_STOP = 537
ERROR_ABIOS_ERROR = 538
ERROR_WX86_WARNING = 539
ERROR_WX86_ERROR = 540
ERROR_TIMER_NOT_CANCELED = 541
ERROR_UNWIND = 542
ERROR_BAD_STACK = 543
ERROR_INVALID_UNWIND_TARGET = 544
ERROR_INVALID_PORT_ATTRIBUTES = 545
ERROR_PORT_MESSAGE_TOO_LONG = 546
ERROR_INVALID_QUOTA_LOWER = 547
ERROR_DEVICE_ALREADY_ATTACHED = 548
ERROR_INSTRUCTION_MISALIGNMENT = 549
ERROR_PROFILING_NOT_STARTED = 550
ERROR_PROFILING_NOT_STOPPED = 551
ERROR_COULD_NOT_INTERPRET = 552
ERROR_PROFILING_AT_LIMIT = 553
ERROR_CANT_WAIT = 554
ERROR_CANT_TERMINATE_SELF = 555
ERROR_UNEXPECTED_MM_CREATE_ERR = 556
ERROR_UNEXPECTED_MM_MAP_ERROR = 557
ERROR_UNEXPECTED_MM_EXTEND_ERR = 558
ERROR_BAD_FUNCTION_TABLE = 559
ERROR_NO_GUID_TRANSLATION = 560
ERROR_INVALID_LDT_SIZE = 561
ERROR_INVALID_LDT_OFFSET = 563
ERROR_INVALID_LDT_DESCRIPTOR = 564
ERROR_TOO_MANY_THREADS = 565
ERROR_THREAD_NOT_IN_PROCESS = 566
ERROR_PAGEFILE_QUOTA_EXCEEDED = 567
ERROR_LOGON_SERVER_CONFLICT = 568
ERROR_SYNCHRONIZATION_REQUIRED = 569
ERROR_NET_OPEN_FAILED = 570
ERROR_IO_PRIVILEGE_FAILED = 571
ERROR_CONTROL_C_EXIT = 572
ERROR_MISSING_SYSTEMFILE = 573
ERROR_UNHANDLED_EXCEPTION = 574
ERROR_APP_INIT_FAILURE = 575
ERROR_PAGEFILE_CREATE_FAILED = 576
ERROR_INVALID_IMAGE_HASH = 577
ERROR_NO_PAGEFILE = 578
ERROR_ILLEGAL_FLOAT_CONTEXT = 579
ERROR_NO_EVENT_PAIR = 580
ERROR_DOMAIN_CTRLR_CONFIG_ERROR = 581
ERROR_ILLEGAL_CHARACTER = 582
ERROR_UNDEFINED_CHARACTER = 583
ERROR_FLOPPY_VOLUME = 584
ERROR_BIOS_FAILED_TO_CONNECT_INTERRUPT = 585
ERROR_BACKUP_CONTROLLER = 586
ERROR_MUTANT_LIMIT_EXCEEDED = 587
ERROR_FS_DRIVER_REQUIRED = 588
ERROR_CANNOT_LOAD_REGISTRY_FILE = 589
ERROR_DEBUG_ATTACH_FAILED = 590
ERROR_SYSTEM_PROCESS_TERMINATED = 591
ERROR_DATA_NOT_ACCEPTED = 592
ERROR_VDM_HARD_ERROR = 593
ERROR_DRIVER_CANCEL_TIMEOUT = 594
ERROR_REPLY_MESSAGE_MISMATCH = 595
ERROR_LOST_WRITEBEHIND_DATA = 596
ERROR_CLIENT_SERVER_PARAMETERS_INVALID = 597
ERROR_NOT_TINY_STREAM = 598
ERROR_STACK_OVERFLOW_READ = 599
ERROR_CONVERT_TO_LARGE = 600
ERROR_FOUND_OUT_OF_SCOPE = 601
ERROR_ALLOCATE_BUCKET = 602
ERROR_MARSHALL_OVERFLOW = 603
ERROR_INVALID_VARIANT = 604
ERROR_BAD_COMPRESSION_BUFFER = 605
ERROR_AUDIT_FAILED = 606
ERROR_TIMER_RESOLUTION_NOT_SET = 607
ERROR_INSUFFICIENT_LOGON_INFO = 608
ERROR_BAD_DLL_ENTRYPOINT = 609
ERROR_BAD_SERVICE_ENTRYPOINT = 610
ERROR_IP_ADDRESS_CONFLICT1 = 611
ERROR_IP_ADDRESS_CONFLICT2 = 612
ERROR_REGISTRY_QUOTA_LIMIT = 613
ERROR_NO_CALLBACK_ACTIVE = 614
ERROR_PWD_TOO_SHORT = 615
ERROR_PWD_TOO_RECENT = 616
ERROR_PWD_HISTORY_CONFLICT = 617
ERROR_UNSUPPORTED_COMPRESSION = 618
ERROR_INVALID_HW_PROFILE = 619
ERROR_INVALID_PLUGPLAY_DEVICE_PATH = 620
ERROR_QUOTA_LIST_INCONSISTENT = 621
ERROR_EVALUATION_EXPIRATION = 622
ERROR_ILLEGAL_DLL_RELOCATION = 623
ERROR_DLL_INIT_FAILED_LOGOFF = 624
ERROR_VALIDATE_CONTINUE = 625
ERROR_NO_MORE_MATCHES = 626
ERROR_RANGE_LIST_CONFLICT = 627
ERROR_SERVER_SID_MISMATCH = 628
ERROR_CANT_ENABLE_DENY_ONLY = 629
ERROR_FLOAT_MULTIPLE_FAULTS = 630
ERROR_FLOAT_MULTIPLE_TRAPS = 631
ERROR_NOINTERFACE = 632
ERROR_DRIVER_FAILED_SLEEP = 633
ERROR_CORRUPT_SYSTEM_FILE = 634
ERROR_COMMITMENT_MINIMUM = 635
ERROR_PNP_RESTART_ENUMERATION = 636
ERROR_SYSTEM_IMAGE_BAD_SIGNATURE = 637
ERROR_PNP_REBOOT_REQUIRED = 638
ERROR_INSUFFICIENT_POWER = 639
ERROR_MULTIPLE_FAULT_VIOLATION = 640
ERROR_SYSTEM_SHUTDOWN = 641
ERROR_PORT_NOT_SET = 642
ERROR_DS_VERSION_CHECK_FAILURE = 643
ERROR_RANGE_NOT_FOUND = 644
ERROR_NOT_SAFE_MODE_DRIVER = 646
ERROR_FAILED_DRIVER_ENTRY = 647
ERROR_DEVICE_ENUMERATION_ERROR = 648
ERROR_MOUNT_POINT_NOT_RESOLVED = 649
ERROR_INVALID_DEVICE_OBJECT_PARAMETER = 650
ERROR_MCA_OCCURED = 651
ERROR_DRIVER_DATABASE_ERROR = 652
ERROR_SYSTEM_HIVE_TOO_LARGE = 653
ERROR_DRIVER_FAILED_PRIOR_UNLOAD = 654
ERROR_VOLSNAP_PREPARE_HIBERNATE = 655
ERROR_HIBERNATION_FAILURE = 656
ERROR_PWD_TOO_LONG = 657
ERROR_FILE_SYSTEM_LIMITATION = 665
ERROR_ASSERTION_FAILURE = 668
ERROR_ACPI_ERROR = 669
ERROR_WOW_ASSERTION = 670
ERROR_PNP_BAD_MPS_TABLE = 671
ERROR_PNP_TRANSLATION_FAILED = 672
ERROR_PNP_IRQ_TRANSLATION_FAILED = 673
ERROR_PNP_INVALID_ID = 674
ERROR_WAKE_SYSTEM_DEBUGGER = 675
ERROR_HANDLES_CLOSED = 676
ERROR_EXTRANEOUS_INFORMATION = 677
ERROR_RXACT_COMMIT_NECESSARY = 678
ERROR_MEDIA_CHECK = 679
ERROR_GUID_SUBSTITUTION_MADE = 680
ERROR_STOPPED_ON_SYMLINK = 681
ERROR_LONGJUMP = 682
ERROR_PLUGPLAY_QUERY_VETOED = 683
ERROR_UNWIND_CONSOLIDATE = 684
ERROR_REGISTRY_HIVE_RECOVERED = 685
ERROR_DLL_MIGHT_BE_INSECURE = 686
ERROR_DLL_MIGHT_BE_INCOMPATIBLE = 687
ERROR_DBG_EXCEPTION_NOT_HANDLED = 688
ERROR_DBG_REPLY_LATER = 689
ERROR_DBG_UNABLE_TO_PROVIDE_HANDLE = 690
ERROR_DBG_TERMINATE_THREAD = 691
ERROR_DBG_TERMINATE_PROCESS = 692
ERROR_DBG_CONTROL_C = 693
ERROR_DBG_PRINTEXCEPTION_C = 694
ERROR_DBG_RIPEXCEPTION = 695
ERROR_DBG_CONTROL_BREAK = 696
ERROR_DBG_COMMAND_EXCEPTION = 697
ERROR_OBJECT_NAME_EXISTS = 698
ERROR_THREAD_WAS_SUSPENDED = 699
ERROR_IMAGE_NOT_AT_BASE = 700
ERROR_RXACT_STATE_CREATED = 701
ERROR_SEGMENT_NOTIFICATION = 702
ERROR_BAD_CURRENT_DIRECTORY = 703
ERROR_FT_READ_RECOVERY_FROM_BACKUP = 704
ERROR_FT_WRITE_RECOVERY = 705
ERROR_IMAGE_MACHINE_TYPE_MISMATCH = 706
ERROR_RECEIVE_PARTIAL = 707
ERROR_RECEIVE_EXPEDITED = 708
ERROR_RECEIVE_PARTIAL_EXPEDITED = 709
ERROR_EVENT_DONE = 710
ERROR_EVENT_PENDING = 711
ERROR_CHECKING_FILE_SYSTEM = 712
ERROR_FATAL_APP_EXIT = 713
ERROR_PREDEFINED_HANDLE = 714
ERROR_WAS_UNLOCKED = 715
ERROR_SERVICE_NOTIFICATION = 716
ERROR_WAS_LOCKED = 717
ERROR_LOG_HARD_ERROR = 718
ERROR_ALREADY_WIN32 = 719
ERROR_IMAGE_MACHINE_TYPE_MISMATCH_EXE = 720
ERROR_NO_YIELD_PERFORMED = 721
ERROR_TIMER_RESUME_IGNORED = 722
ERROR_ARBITRATION_UNHANDLED = 723
ERROR_CARDBUS_NOT_SUPPORTED = 724
ERROR_MP_PROCESSOR_MISMATCH = 725
ERROR_HIBERNATED = 726
ERROR_RESUME_HIBERNATION = 727
ERROR_FIRMWARE_UPDATED = 728
ERROR_DRIVERS_LEAKING_LOCKED_PAGES = 729
ERROR_WAKE_SYSTEM = 730
ERROR_WAIT_1 = 731
ERROR_WAIT_2 = 732
ERROR_WAIT_3 = 733
ERROR_WAIT_63 = 734
ERROR_ABANDONED_WAIT_0 = 735
ERROR_ABANDONED_WAIT_63 = 736
ERROR_USER_APC = 737
ERROR_KERNEL_APC = 738
ERROR_ALERTED = 739
ERROR_ELEVATION_REQUIRED = 740
ERROR_REPARSE = 741
ERROR_OPLOCK_BREAK_IN_PROGRESS = 742
ERROR_VOLUME_MOUNTED = 743
ERROR_RXACT_COMMITTED = 744
ERROR_NOTIFY_CLEANUP = 745
ERROR_PRIMARY_TRANSPORT_CONNECT_FAILED = 746
ERROR_PAGE_FAULT_TRANSITION = 747
ERROR_PAGE_FAULT_DEMAND_ZERO = 748
ERROR_PAGE_FAULT_COPY_ON_WRITE = 749
ERROR_PAGE_FAULT_GUARD_PAGE = 750
ERROR_PAGE_FAULT_PAGING_FILE = 751
ERROR_CACHE_PAGE_LOCKED = 752
ERROR_CRASH_DUMP = 753
ERROR_BUFFER_ALL_ZEROS = 754
ERROR_REPARSE_OBJECT = 755
ERROR_RESOURCE_REQUIREMENTS_CHANGED = 756
ERROR_TRANSLATION_COMPLETE = 757
ERROR_NOTHING_TO_TERMINATE = 758
ERROR_PROCESS_NOT_IN_JOB = 759
ERROR_PROCESS_IN_JOB = 760
ERROR_VOLSNAP_HIBERNATE_READY = 761
ERROR_FSFILTER_OP_COMPLETED_SUCCESSFULLY = 762
ERROR_INTERRUPT_VECTOR_ALREADY_CONNECTED = 763
ERROR_INTERRUPT_STILL_CONNECTED = 764
ERROR_WAIT_FOR_OPLOCK = 765
ERROR_DBG_EXCEPTION_HANDLED = 766
ERROR_DBG_CONTINUE = 767
ERROR_CALLBACK_POP_STACK = 768
ERROR_COMPRESSION_DISABLED = 769
ERROR_CANTFETCHBACKWARDS = 770
ERROR_CANTSCROLLBACKWARDS = 771
ERROR_ROWSNOTRELEASED = 772
ERROR_BAD_ACCESSOR_FLAGS = 773
ERROR_ERRORS_ENCOUNTERED = 774
ERROR_NOT_CAPABLE = 775
ERROR_REQUEST_OUT_OF_SEQUENCE = 776
ERROR_VERSION_PARSE_ERROR = 777
ERROR_BADSTARTPOSITION = 778
ERROR_MEMORY_HARDWARE = 779
ERROR_DISK_REPAIR_DISABLED = 780
ERROR_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE = 781
ERROR_SYSTEM_POWERSTATE_TRANSITION = 782
ERROR_SYSTEM_POWERSTATE_COMPLEX_TRANSITION = 783
ERROR_MCA_EXCEPTION = 784
ERROR_ACCESS_AUDIT_BY_POLICY = 785
ERROR_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY = 786
ERROR_ABANDON_HIBERFILE = 787
ERROR_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED = 788
ERROR_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR = 789
ERROR_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR = 790
ERROR_BAD_MCFG_TABLE = 791
ERROR_DISK_REPAIR_REDIRECTED = 792
ERROR_DISK_REPAIR_UNSUCCESSFUL = 793
ERROR_CORRUPT_LOG_OVERFULL = 794
ERROR_CORRUPT_LOG_CORRUPTED = 795
ERROR_CORRUPT_LOG_UNAVAILABLE = 796
ERROR_CORRUPT_LOG_DELETED_FULL = 797
ERROR_CORRUPT_LOG_CLEARED = 798
ERROR_ORPHAN_NAME_EXHAUSTED = 799
ERROR_OPLOCK_SWITCHED_TO_NEW_HANDLE = 800
ERROR_CANNOT_GRANT_REQUESTED_OPLOCK = 801
ERROR_CANNOT_BREAK_OPLOCK = 802
ERROR_OPLOCK_HANDLE_CLOSED = 803
ERROR_NO_ACE_CONDITION = 804
ERROR_INVALID_ACE_CONDITION = 805
ERROR_FILE_HANDLE_REVOKED = 806
ERROR_IMAGE_AT_DIFFERENT_BASE = 807
ERROR_ENCRYPTED_IO_NOT_POSSIBLE = 808
ERROR_FILE_METADATA_OPTIMIZATION_IN_PROGRESS = 809
ERROR_QUOTA_ACTIVITY = 810
ERROR_HANDLE_REVOKED = 811
ERROR_CALLBACK_INVOKE_INLINE = 812
ERROR_CPU_SET_INVALID = 813
ERROR_ENCLAVE_NOT_TERMINATED = 814
ERROR_ENCLAVE_VIOLATION = 815
ERROR_EA_ACCESS_DENIED = 994
ERROR_OPERATION_ABORTED = 995
ERROR_IO_INCOMPLETE = 996
ERROR_IO_PENDING = 997
ERROR_NOACCESS = 998
ERROR_SWAPERROR = 999
ERROR_STACK_OVERFLOW = 1001
ERROR_INVALID_MESSAGE = 1002
ERROR_CAN_NOT_COMPLETE = 1003
ERROR_INVALID_FLAGS = 1004
ERROR_UNRECOGNIZED_VOLUME = 1005
ERROR_FILE_INVALID = 1006
ERROR_FULLSCREEN_MODE = 1007
ERROR_NO_TOKEN = 1008
ERROR_BADDB = 1009
ERROR_BADKEY = 1010
ERROR_CANTOPEN = 1011
ERROR_CANTREAD = 1012
ERROR_CANTWRITE = 1013
ERROR_REGISTRY_RECOVERED = 1014
ERROR_REGISTRY_CORRUPT = 1015
ERROR_REGISTRY_IO_FAILED = 1016
ERROR_NOT_REGISTRY_FILE = 1017
ERROR_KEY_DELETED = 1018
ERROR_NO_LOG_SPACE = 1019
ERROR_KEY_HAS_CHILDREN = 1020
ERROR_CHILD_MUST_BE_VOLATILE = 1021
ERROR_NOTIFY_ENUM_DIR = 1022
ERROR_DEPENDENT_SERVICES_RUNNING = 1051
ERROR_INVALID_SERVICE_CONTROL = 1052
ERROR_SERVICE_REQUEST_TIMEOUT = 1053
ERROR_SERVICE_NO_THREAD = 1054
ERROR_SERVICE_DATABASE_LOCKED = 1055
ERROR_SERVICE_ALREADY_RUNNING = 1056
ERROR_INVALID_SERVICE_ACCOUNT = 1057
ERROR_SERVICE_DISABLED = 1058
ERROR_CIRCULAR_DEPENDENCY = 1059
ERROR_SERVICE_DOES_NOT_EXIST = 1060
ERROR_SERVICE_CANNOT_ACCEPT_CTRL = 1061
ERROR_SERVICE_NOT_ACTIVE = 1062
ERROR_FAILED_SERVICE_CONTROLLER_CONNECT = 1063
ERROR_EXCEPTION_IN_SERVICE = 1064
ERROR_DATABASE_DOES_NOT_EXIST = 1065
ERROR_SERVICE_SPECIFIC_ERROR = 1066
ERROR_PROCESS_ABORTED = 1067
ERROR_SERVICE_DEPENDENCY_FAIL = 1068
ERROR_SERVICE_LOGON_FAILED = 1069
ERROR_SERVICE_START_HANG = 1070
ERROR_INVALID_SERVICE_LOCK = 1071
ERROR_SERVICE_MARKED_FOR_DELETE = 1072
ERROR_SERVICE_EXISTS = 1073
ERROR_ALREADY_RUNNING_LKG = 1074
ERROR_SERVICE_DEPENDENCY_DELETED = 1075
ERROR_BOOT_ALREADY_ACCEPTED = 1076
ERROR_SERVICE_NEVER_STARTED = 1077
ERROR_DUPLICATE_SERVICE_NAME = 1078
ERROR_DIFFERENT_SERVICE_ACCOUNT = 1079
ERROR_CANNOT_DETECT_DRIVER_FAILURE = 1080
ERROR_CANNOT_DETECT_PROCESS_ABORT = 1081
ERROR_NO_RECOVERY_PROGRAM = 1082
ERROR_SERVICE_NOT_IN_EXE = 1083
ERROR_NOT_SAFEBOOT_SERVICE = 1084
ERROR_END_OF_MEDIA = 1100
ERROR_FILEMARK_DETECTED = 1101
ERROR_BEGINNING_OF_MEDIA = 1102
ERROR_SETMARK_DETECTED = 1103
ERROR_NO_DATA_DETECTED = 1104
ERROR_PARTITION_FAILURE = 1105
ERROR_INVALID_BLOCK_LENGTH = 1106
ERROR_DEVICE_NOT_PARTITIONED = 1107
ERROR_UNABLE_TO_LOCK_MEDIA = 1108
ERROR_UNABLE_TO_UNLOAD_MEDIA = 1109
ERROR_MEDIA_CHANGED = 1110
ERROR_BUS_RESET = 1111
ERROR_NO_MEDIA_IN_DRIVE = 1112
ERROR_NO_UNICODE_TRANSLATION = 1113
ERROR_DLL_INIT_FAILED = 1114
ERROR_SHUTDOWN_IN_PROGRESS = 1115
ERROR_NO_SHUTDOWN_IN_PROGRESS = 1116
ERROR_IO_DEVICE = 1117
ERROR_SERIAL_NO_DEVICE = 1118
ERROR_IRQ_BUSY = 1119
ERROR_MORE_WRITES = 1120
ERROR_COUNTER_TIMEOUT = 1121
ERROR_FLOPPY_ID_MARK_NOT_FOUND = 1122
ERROR_FLOPPY_WRONG_CYLINDER = 1123
ERROR_FLOPPY_UNKNOWN_ERROR = 1124
ERROR_FLOPPY_BAD_REGISTERS = 1125
ERROR_DISK_RECALIBRATE_FAILED = 1126
ERROR_DISK_OPERATION_FAILED = 1127
ERROR_DISK_RESET_FAILED = 1128
ERROR_EOM_OVERFLOW = 1129
ERROR_NOT_ENOUGH_SERVER_MEMORY = 1130
ERROR_POSSIBLE_DEADLOCK = 1131
ERROR_MAPPED_ALIGNMENT = 1132
ERROR_SET_POWER_STATE_VETOED = 1140
ERROR_SET_POWER_STATE_FAILED = 1141
ERROR_TOO_MANY_LINKS = 1142
ERROR_OLD_WIN_VERSION = 1150
ERROR_APP_WRONG_OS = 1151
ERROR_SINGLE_INSTANCE_APP = 1152
ERROR_RMODE_APP = 1153
ERROR_INVALID_DLL = 1154
ERROR_NO_ASSOCIATION = 1155
ERROR_DDE_FAIL = 1156
ERROR_DLL_NOT_FOUND = 1157
ERROR_NO_MORE_USER_HANDLES = 1158
ERROR_MESSAGE_SYNC_ONLY = 1159
ERROR_SOURCE_ELEMENT_EMPTY = 1160
ERROR_DESTINATION_ELEMENT_FULL = 1161
ERROR_ILLEGAL_ELEMENT_ADDRESS = 1162
ERROR_MAGAZINE_NOT_PRESENT = 1163
ERROR_DEVICE_REINITIALIZATION_NEEDED = 1164
ERROR_DEVICE_REQUIRES_CLEANING = 1165
ERROR_DEVICE_DOOR_OPEN = 1166
ERROR_DEVICE_NOT_CONNECTED = 1167
ERROR_NOT_FOUND = 1168
ERROR_NO_MATCH = 1169
ERROR_SET_NOT_FOUND = 1170
ERROR_POINT_NOT_FOUND = 1171
ERROR_NO_TRACKING_SERVICE = 1172
ERROR_NO_VOLUME_ID = 1173
ERROR_UNABLE_TO_REMOVE_REPLACED = 1175
ERROR_UNABLE_TO_MOVE_REPLACEMENT = 1176
ERROR_UNABLE_TO_MOVE_REPLACEMENT_2 = 1177
ERROR_JOURNAL_DELETE_IN_PROGRESS = 1178
ERROR_JOURNAL_NOT_ACTIVE = 1179
ERROR_POTENTIAL_FILE_FOUND = 1180
ERROR_JOURNAL_ENTRY_DELETED = 1181
ERROR_SHUTDOWN_IS_SCHEDULED = 1190
ERROR_SHUTDOWN_USERS_LOGGED_ON = 1191
ERROR_BAD_DEVICE = 1200
ERROR_CONNECTION_UNAVAIL = 1201
ERROR_DEVICE_ALREADY_REMEMBERED = 1202
ERROR_NO_NET_OR_BAD_PATH = 1203
ERROR_BAD_PROVIDER = 1204
ERROR_CANNOT_OPEN_PROFILE = 1205
ERROR_BAD_PROFILE = 1206
ERROR_NOT_CONTAINER = 1207
ERROR_EXTENDED_ERROR = 1208
ERROR_INVALID_GROUPNAME = 1209
ERROR_INVALID_COMPUTERNAME = 1210
ERROR_INVALID_EVENTNAME = 1211
ERROR_INVALID_DOMAINNAME = 1212
ERROR_INVALID_SERVICENAME = 1213
ERROR_INVALID_NETNAME = 1214
ERROR_INVALID_SHARENAME = 1215
ERROR_INVALID_PASSWORDNAME = 1216
ERROR_INVALID_MESSAGENAME = 1217
ERROR_INVALID_MESSAGEDEST = 1218
ERROR_SESSION_CREDENTIAL_CONFLICT = 1219
ERROR_REMOTE_SESSION_LIMIT_EXCEEDED = 1220
ERROR_DUP_DOMAINNAME = 1221
ERROR_NO_NETWORK = 1222
ERROR_CANCELLED = 1223
ERROR_USER_MAPPED_FILE = 1224
ERROR_CONNECTION_REFUSED = 1225
ERROR_GRACEFUL_DISCONNECT = 1226
ERROR_ADDRESS_ALREADY_ASSOCIATED = 1227
ERROR_ADDRESS_NOT_ASSOCIATED = 1228
ERROR_CONNECTION_INVALID = 1229
ERROR_CONNECTION_ACTIVE = 1230
ERROR_NETWORK_UNREACHABLE = 1231
ERROR_HOST_UNREACHABLE = 1232
ERROR_PROTOCOL_UNREACHABLE = 1233
ERROR_PORT_UNREACHABLE = 1234
ERROR_REQUEST_ABORTED = 1235
ERROR_CONNECTION_ABORTED = 1236
ERROR_RETRY = 1237
ERROR_CONNECTION_COUNT_LIMIT = 1238
ERROR_LOGIN_TIME_RESTRICTION = 1239
ERROR_LOGIN_WKSTA_RESTRICTION = 1240
ERROR_INCORRECT_ADDRESS = 1241
ERROR_ALREADY_REGISTERED = 1242
ERROR_SERVICE_NOT_FOUND = 1243
ERROR_NOT_AUTHENTICATED = 1244
ERROR_NOT_LOGGED_ON = 1245
ERROR_CONTINUE = 1246
ERROR_ALREADY_INITIALIZED = 1247
ERROR_NO_MORE_DEVICES = 1248
ERROR_NO_SUCH_SITE = 1249
ERROR_DOMAIN_CONTROLLER_EXISTS = 1250
ERROR_ONLY_IF_CONNECTED = 1251
ERROR_OVERRIDE_NOCHANGES = 1252
ERROR_BAD_USER_PROFILE = 1253
ERROR_NOT_SUPPORTED_ON_SBS = 1254
ERROR_SERVER_SHUTDOWN_IN_PROGRESS = 1255
ERROR_HOST_DOWN = 1256
ERROR_NON_ACCOUNT_SID = 1257
ERROR_NON_DOMAIN_SID = 1258
ERROR_APPHELP_BLOCK = 1259
ERROR_ACCESS_DISABLED_BY_POLICY = 1260
ERROR_REG_NAT_CONSUMPTION = 1261
ERROR_CSCSHARE_OFFLINE = 1262
ERROR_PKINIT_FAILURE = 1263
ERROR_SMARTCARD_SUBSYSTEM_FAILURE = 1264
ERROR_DOWNGRADE_DETECTED = 1265
ERROR_MACHINE_LOCKED = 1271
ERROR_SMB_GUEST_LOGON_BLOCKED = 1272
ERROR_CALLBACK_SUPPLIED_INVALID_DATA = 1273
ERROR_SYNC_FOREGROUND_REFRESH_REQUIRED = 1274
ERROR_DRIVER_BLOCKED = 1275
ERROR_INVALID_IMPORT_OF_NON_DLL = 1276
ERROR_ACCESS_DISABLED_WEBBLADE = 1277
ERROR_ACCESS_DISABLED_WEBBLADE_TAMPER = 1278
ERROR_RECOVERY_FAILURE = 1279
ERROR_ALREADY_FIBER = 1280
ERROR_ALREADY_THREAD = 1281
ERROR_STACK_BUFFER_OVERRUN = 1282
ERROR_PARAMETER_QUOTA_EXCEEDED = 1283
ERROR_DEBUGGER_INACTIVE = 1284
ERROR_DELAY_LOAD_FAILED = 1285
ERROR_VDM_DISALLOWED = 1286
ERROR_UNIDENTIFIED_ERROR = 1287
ERROR_INVALID_CRUNTIME_PARAMETER = 1288
ERROR_BEYOND_VDL = 1289
ERROR_INCOMPATIBLE_SERVICE_SID_TYPE = 1290
ERROR_DRIVER_PROCESS_TERMINATED = 1291
ERROR_IMPLEMENTATION_LIMIT = 1292
ERROR_PROCESS_IS_PROTECTED = 1293
ERROR_SERVICE_NOTIFY_CLIENT_LAGGING = 1294
ERROR_DISK_QUOTA_EXCEEDED = 1295
ERROR_CONTENT_BLOCKED = 1296
ERROR_INCOMPATIBLE_SERVICE_PRIVILEGE = 1297
ERROR_APP_HANG = 1298
ERROR_INVALID_LABEL = 1299
ERROR_NOT_ALL_ASSIGNED = 1300
ERROR_SOME_NOT_MAPPED = 1301
ERROR_NO_QUOTAS_FOR_ACCOUNT = 1302
ERROR_LOCAL_USER_SESSION_KEY = 1303
ERROR_NULL_LM_PASSWORD = 1304
ERROR_UNKNOWN_REVISION = 1305
ERROR_REVISION_MISMATCH = 1306
ERROR_INVALID_OWNER = 1307
ERROR_INVALID_PRIMARY_GROUP = 1308
ERROR_NO_IMPERSONATION_TOKEN = 1309
ERROR_CANT_DISABLE_MANDATORY = 1310
ERROR_NO_LOGON_SERVERS = 1311
ERROR_NO_SUCH_LOGON_SESSION = 1312
ERROR_NO_SUCH_PRIVILEGE = 1313
ERROR_PRIVILEGE_NOT_HELD = 1314
ERROR_INVALID_ACCOUNT_NAME = 1315
ERROR_USER_EXISTS = 1316
ERROR_NO_SUCH_USER = 1317
ERROR_GROUP_EXISTS = 1318
ERROR_NO_SUCH_GROUP = 1319
ERROR_MEMBER_IN_GROUP = 1320
ERROR_MEMBER_NOT_IN_GROUP = 1321
ERROR_LAST_ADMIN = 1322
ERROR_WRONG_PASSWORD = 1323
ERROR_ILL_FORMED_PASSWORD = 1324
ERROR_PASSWORD_RESTRICTION = 1325
ERROR_LOGON_FAILURE = 1326
ERROR_ACCOUNT_RESTRICTION = 1327
ERROR_INVALID_LOGON_HOURS = 1328
ERROR_INVALID_WORKSTATION = 1329
ERROR_PASSWORD_EXPIRED = 1330
ERROR_ACCOUNT_DISABLED = 1331
ERROR_NONE_MAPPED = 1332
ERROR_TOO_MANY_LUIDS_REQUESTED = 1333
ERROR_LUIDS_EXHAUSTED = 1334
ERROR_INVALID_SUB_AUTHORITY = 1335
ERROR_INVALID_ACL = 1336
ERROR_INVALID_SID = 1337
ERROR_INVALID_SECURITY_DESCR = 1338
ERROR_BAD_INHERITANCE_ACL = 1340
ERROR_SERVER_DISABLED = 1341
ERROR_SERVER_NOT_DISABLED = 1342
ERROR_INVALID_ID_AUTHORITY = 1343
ERROR_ALLOTTED_SPACE_EXCEEDED = 1344
ERROR_INVALID_GROUP_ATTRIBUTES = 1345
ERROR_BAD_IMPERSONATION_LEVEL = 1346
ERROR_CANT_OPEN_ANONYMOUS = 1347
ERROR_BAD_VALIDATION_CLASS = 1348
ERROR_BAD_TOKEN_TYPE = 1349
ERROR_NO_SECURITY_ON_OBJECT = 1350
ERROR_CANT_ACCESS_DOMAIN_INFO = 1351
ERROR_INVALID_SERVER_STATE = 1352
ERROR_INVALID_DOMAIN_STATE = 1353
ERROR_INVALID_DOMAIN_ROLE = 1354
ERROR_NO_SUCH_DOMAIN = 1355
ERROR_DOMAIN_EXISTS = 1356
ERROR_DOMAIN_LIMIT_EXCEEDED = 1357
ERROR_INTERNAL_DB_CORRUPTION = 1358
ERROR_INTERNAL_ERROR = 1359
ERROR_GENERIC_NOT_MAPPED = 1360
ERROR_BAD_DESCRIPTOR_FORMAT = 1361
ERROR_NOT_LOGON_PROCESS = 1362
ERROR_LOGON_SESSION_EXISTS = 1363
ERROR_NO_SUCH_PACKAGE = 1364
ERROR_BAD_LOGON_SESSION_STATE = 1365
ERROR_LOGON_SESSION_COLLISION = 1366
ERROR_INVALID_LOGON_TYPE = 1367
ERROR_CANNOT_IMPERSONATE = 1368
ERROR_RXACT_INVALID_STATE = 1369
ERROR_RXACT_COMMIT_FAILURE = 1370
ERROR_SPECIAL_ACCOUNT = 1371
ERROR_SPECIAL_GROUP = 1372
ERROR_SPECIAL_USER = 1373
ERROR_MEMBERS_PRIMARY_GROUP = 1374
ERROR_TOKEN_ALREADY_IN_USE = 1375
ERROR_NO_SUCH_ALIAS = 1376
ERROR_MEMBER_NOT_IN_ALIAS = 1377
ERROR_MEMBER_IN_ALIAS = 1378
ERROR_ALIAS_EXISTS = 1379
ERROR_LOGON_NOT_GRANTED = 1380
ERROR_TOO_MANY_SECRETS = 1381
ERROR_SECRET_TOO_LONG = 1382
ERROR_INTERNAL_DB_ERROR = 1383
ERROR_TOO_MANY_CONTEXT_IDS = 1384
ERROR_LOGON_TYPE_NOT_GRANTED = 1385
ERROR_NT_CROSS_ENCRYPTION_REQUIRED = 1386
ERROR_NO_SUCH_MEMBER = 1387
ERROR_INVALID_MEMBER = 1388
ERROR_TOO_MANY_SIDS = 1389
ERROR_LM_CROSS_ENCRYPTION_REQUIRED = 1390
ERROR_NO_INHERITANCE = 1391
ERROR_FILE_CORRUPT = 1392
ERROR_DISK_CORRUPT = 1393
ERROR_NO_USER_SESSION_KEY = 1394
ERROR_LICENSE_QUOTA_EXCEEDED = 1395
ERROR_WRONG_TARGET_NAME = 1396
ERROR_MUTUAL_AUTH_FAILED = 1397
ERROR_TIME_SKEW = 1398
ERROR_CURRENT_DOMAIN_NOT_ALLOWED = 1399
ERROR_INVALID_WINDOW_HANDLE = 1400
ERROR_INVALID_MENU_HANDLE = 1401
ERROR_INVALID_CURSOR_HANDLE = 1402
ERROR_INVALID_ACCEL_HANDLE = 1403
ERROR_INVALID_HOOK_HANDLE = 1404
ERROR_INVALID_DWP_HANDLE = 1405
ERROR_TLW_WITH_WSCHILD = 1406
ERROR_CANNOT_FIND_WND_CLASS = 1407
ERROR_WINDOW_OF_OTHER_THREAD = 1408
ERROR_HOTKEY_ALREADY_REGISTERED = 1409
ERROR_CLASS_ALREADY_EXISTS = 1410
ERROR_CLASS_DOES_NOT_EXIST = 1411
ERROR_CLASS_HAS_WINDOWS = 1412
ERROR_INVALID_INDEX = 1413
ERROR_INVALID_ICON_HANDLE = 1414
ERROR_PRIVATE_DIALOG_INDEX = 1415
ERROR_LISTBOX_ID_NOT_FOUND = 1416
ERROR_NO_WILDCARD_CHARACTERS = 1417
ERROR_CLIPBOARD_NOT_OPEN = 1418
ERROR_HOTKEY_NOT_REGISTERED = 1419
ERROR_WINDOW_NOT_DIALOG = 1420
ERROR_CONTROL_ID_NOT_FOUND = 1421
ERROR_INVALID_COMBOBOX_MESSAGE = 1422
ERROR_WINDOW_NOT_COMBOBOX = 1423
ERROR_INVALID_EDIT_HEIGHT = 1424
ERROR_DC_NOT_FOUND = 1425
ERROR_INVALID_HOOK_FILTER = 1426
ERROR_INVALID_FILTER_PROC = 1427
ERROR_HOOK_NEEDS_HMOD = 1428
ERROR_GLOBAL_ONLY_HOOK = 1429
ERROR_JOURNAL_HOOK_SET = 1430
ERROR_HOOK_NOT_INSTALLED = 1431
ERROR_INVALID_LB_MESSAGE = 1432
ERROR_SETCOUNT_ON_BAD_LB = 1433
ERROR_LB_WITHOUT_TABSTOPS = 1434
ERROR_DESTROY_OBJECT_OF_OTHER_THREAD = 1435
ERROR_CHILD_WINDOW_MENU = 1436
ERROR_NO_SYSTEM_MENU = 1437
ERROR_INVALID_MSGBOX_STYLE = 1438
ERROR_INVALID_SPI_VALUE = 1439
ERROR_SCREEN_ALREADY_LOCKED = 1440
ERROR_HWNDS_HAVE_DIFF_PARENT = 1441
ERROR_NOT_CHILD_WINDOW = 1442
ERROR_INVALID_GW_COMMAND = 1443
ERROR_INVALID_THREAD_ID = 1444
ERROR_NON_MDICHILD_WINDOW = 1445
ERROR_POPUP_ALREADY_ACTIVE = 1446
ERROR_NO_SCROLLBARS = 1447
ERROR_INVALID_SCROLLBAR_RANGE = 1448
ERROR_INVALID_SHOWWIN_COMMAND = 1449
ERROR_NO_SYSTEM_RESOURCES = 1450
ERROR_NONPAGED_SYSTEM_RESOURCES = 1451
ERROR_PAGED_SYSTEM_RESOURCES = 1452
ERROR_WORKING_SET_QUOTA = 1453
ERROR_PAGEFILE_QUOTA = 1454
ERROR_COMMITMENT_LIMIT = 1455
ERROR_MENU_ITEM_NOT_FOUND = 1456
ERROR_INVALID_KEYBOARD_HANDLE = 1457
ERROR_HOOK_TYPE_NOT_ALLOWED = 1458
ERROR_REQUIRES_INTERACTIVE_WINDOWSTATION = 1459
ERROR_TIMEOUT = 1460
ERROR_INVALID_MONITOR_HANDLE = 1461
ERROR_INCORRECT_SIZE = 1462
ERROR_SYMLINK_CLASS_DISABLED = 1463
ERROR_SYMLINK_NOT_SUPPORTED = 1464
ERROR_XML_PARSE_ERROR = 1465
ERROR_XMLDSIG_ERROR = 1466
ERROR_RESTART_APPLICATION = 1467
ERROR_WRONG_COMPARTMENT = 1468
ERROR_AUTHIP_FAILURE = 1469
ERROR_NO_NVRAM_RESOURCES = 1470
ERROR_NOT_GUI_PROCESS = 1471
ERROR_EVENTLOG_FILE_CORRUPT = 1500
ERROR_EVENTLOG_CANT_START = 1501
ERROR_LOG_FILE_FULL = 1502
ERROR_EVENTLOG_FILE_CHANGED = 1503
ERROR_CONTAINER_ASSIGNED = 1504
ERROR_JOB_NO_CONTAINER = 1505
ERROR_INVALID_TASK_NAME = 1550
ERROR_INVALID_TASK_INDEX = 1551
ERROR_THREAD_ALREADY_IN_TASK = 1552
ERROR_INSTALL_SERVICE_FAILURE = 1601
ERROR_INSTALL_USEREXIT = 1602
ERROR_INSTALL_FAILURE = 1603
ERROR_INSTALL_SUSPEND = 1604
ERROR_UNKNOWN_PRODUCT = 1605
ERROR_UNKNOWN_FEATURE = 1606
ERROR_UNKNOWN_COMPONENT = 1607
ERROR_UNKNOWN_PROPERTY = 1608
ERROR_INVALID_HANDLE_STATE = 1609
ERROR_BAD_CONFIGURATION = 1610
ERROR_INDEX_ABSENT = 1611
ERROR_INSTALL_SOURCE_ABSENT = 1612
ERROR_INSTALL_PACKAGE_VERSION = 1613
ERROR_PRODUCT_UNINSTALLED = 1614
ERROR_BAD_QUERY_SYNTAX = 1615
ERROR_INVALID_FIELD = 1616
ERROR_DEVICE_REMOVED = 1617
ERROR_INSTALL_ALREADY_RUNNING = 1618
ERROR_INSTALL_PACKAGE_OPEN_FAILED = 1619
ERROR_INSTALL_PACKAGE_INVALID = 1620
ERROR_INSTALL_UI_FAILURE = 1621
ERROR_INSTALL_LOG_FAILURE = 1622
ERROR_INSTALL_LANGUAGE_UNSUPPORTED = 1623
ERROR_INSTALL_TRANSFORM_FAILURE = 1624
ERROR_INSTALL_PACKAGE_REJECTED = 1625
ERROR_FUNCTION_NOT_CALLED = 1626
ERROR_FUNCTION_FAILED = 1627
ERROR_INVALID_TABLE = 1628
ERROR_DATATYPE_MISMATCH = 1629
ERROR_UNSUPPORTED_TYPE = 1630
ERROR_CREATE_FAILED = 1631
ERROR_INSTALL_TEMP_UNWRITABLE = 1632
ERROR_INSTALL_PLATFORM_UNSUPPORTED = 1633
ERROR_INSTALL_NOTUSED = 1634
ERROR_PATCH_PACKAGE_OPEN_FAILED = 1635
ERROR_PATCH_PACKAGE_INVALID = 1636
ERROR_PATCH_PACKAGE_UNSUPPORTED = 1637
ERROR_PRODUCT_VERSION = 1638
ERROR_INVALID_COMMAND_LINE = 1639
ERROR_INSTALL_REMOTE_DISALLOWED = 1640
ERROR_SUCCESS_REBOOT_INITIATED = 1641
ERROR_PATCH_TARGET_NOT_FOUND = 1642
ERROR_PATCH_PACKAGE_REJECTED = 1643
ERROR_INSTALL_TRANSFORM_REJECTED = 1644
ERROR_INSTALL_REMOTE_PROHIBITED = 1645
ERROR_PATCH_REMOVAL_UNSUPPORTED = 1646
ERROR_UNKNOWN_PATCH = 1647
ERROR_PATCH_NO_SEQUENCE = 1648
ERROR_PATCH_REMOVAL_DISALLOWED = 1649
ERROR_INVALID_PATCH_XML = 1650
ERROR_PATCH_MANAGED_ADVERTISED_PRODUCT = 1651
ERROR_INSTALL_SERVICE_SAFEBOOT = 1652
ERROR_FAIL_FAST_EXCEPTION = 1653
ERROR_INSTALL_REJECTED = 1654
ERROR_DYNAMIC_CODE_BLOCKED = 1655
ERROR_NOT_SAME_OBJECT = 1656
ERROR_STRICT_CFG_VIOLATION = 1657
ERROR_SET_CONTEXT_DENIED = 1660
ERROR_CROSS_PARTITION_VIOLATION = 1661
ERROR_RETURN_ADDRESS_HIJACK_ATTEMPT = 1662
RPC_S_INVALID_STRING_BINDING = 1700
RPC_S_WRONG_KIND_OF_BINDING = 1701
RPC_S_INVALID_BINDING = 1702
RPC_S_PROTSEQ_NOT_SUPPORTED = 1703
RPC_S_INVALID_RPC_PROTSEQ = 1704
RPC_S_INVALID_STRING_UUID = 1705
RPC_S_INVALID_ENDPOINT_FORMAT = 1706
RPC_S_INVALID_NET_ADDR = 1707
RPC_S_NO_ENDPOINT_FOUND = 1708
RPC_S_INVALID_TIMEOUT = 1709
RPC_S_OBJECT_NOT_FOUND = 1710
RPC_S_ALREADY_REGISTERED = 1711
RPC_S_TYPE_ALREADY_REGISTERED = 1712
RPC_S_ALREADY_LISTENING = 1713
RPC_S_NO_PROTSEQS_REGISTERED = 1714
RPC_S_NOT_LISTENING = 1715
RPC_S_UNKNOWN_MGR_TYPE = 1716
RPC_S_UNKNOWN_IF = 1717
RPC_S_NO_BINDINGS = 1718
RPC_S_NO_PROTSEQS = 1719
RPC_S_CANT_CREATE_ENDPOINT = 1720
RPC_S_OUT_OF_RESOURCES = 1721
RPC_S_SERVER_UNAVAILABLE = 1722
RPC_S_SERVER_TOO_BUSY = 1723
RPC_S_INVALID_NETWORK_OPTIONS = 1724
RPC_S_NO_CALL_ACTIVE = 1725
RPC_S_CALL_FAILED = 1726
RPC_S_CALL_FAILED_DNE = 1727
RPC_S_PROTOCOL_ERROR = 1728
RPC_S_PROXY_ACCESS_DENIED = 1729
RPC_S_UNSUPPORTED_TRANS_SYN = 1730
RPC_S_UNSUPPORTED_TYPE = 1732
RPC_S_INVALID_TAG = 1733
RPC_S_INVALID_BOUND = 1734
RPC_S_NO_ENTRY_NAME = 1735
RPC_S_INVALID_NAME_SYNTAX = 1736
RPC_S_UNSUPPORTED_NAME_SYNTAX = 1737
RPC_S_UUID_NO_ADDRESS = 1739
RPC_S_DUPLICATE_ENDPOINT = 1740
RPC_S_UNKNOWN_AUTHN_TYPE = 1741
RPC_S_MAX_CALLS_TOO_SMALL = 1742
RPC_S_STRING_TOO_LONG = 1743
RPC_S_PROTSEQ_NOT_FOUND = 1744
RPC_S_PROCNUM_OUT_OF_RANGE = 1745
RPC_S_BINDING_HAS_NO_AUTH = 1746
RPC_S_UNKNOWN_AUTHN_SERVICE = 1747
RPC_S_UNKNOWN_AUTHN_LEVEL = 1748
RPC_S_INVALID_AUTH_IDENTITY = 1749
RPC_S_UNKNOWN_AUTHZ_SERVICE = 1750
EPT_S_INVALID_ENTRY = 1751
EPT_S_CANT_PERFORM_OP = 1752
EPT_S_NOT_REGISTERED = 1753
RPC_S_NOTHING_TO_EXPORT = 1754
RPC_S_INCOMPLETE_NAME = 1755
RPC_S_INVALID_VERS_OPTION = 1756
RPC_S_NO_MORE_MEMBERS = 1757
RPC_S_NOT_ALL_OBJS_UNEXPORTED = 1758
RPC_S_INTERFACE_NOT_FOUND = 1759
RPC_S_ENTRY_ALREADY_EXISTS = 1760
RPC_S_ENTRY_NOT_FOUND = 1761
RPC_S_NAME_SERVICE_UNAVAILABLE = 1762
RPC_S_INVALID_NAF_ID = 1763
RPC_S_CANNOT_SUPPORT = 1764
RPC_S_NO_CONTEXT_AVAILABLE = 1765
RPC_S_INTERNAL_ERROR = 1766
RPC_S_ZERO_DIVIDE = 1767
RPC_S_ADDRESS_ERROR = 1768
RPC_S_FP_DIV_ZERO = 1769
RPC_S_FP_UNDERFLOW = 1770
RPC_S_FP_OVERFLOW = 1771
RPC_X_NO_MORE_ENTRIES = 1772
RPC_X_SS_CHAR_TRANS_OPEN_FAIL = 1773
RPC_X_SS_CHAR_TRANS_SHORT_FILE = 1774
RPC_X_SS_IN_NULL_CONTEXT = 1775
RPC_X_SS_CONTEXT_DAMAGED = 1777
RPC_X_SS_HANDLES_MISMATCH = 1778
RPC_X_SS_CANNOT_GET_CALL_HANDLE = 1779
RPC_X_NULL_REF_POINTER = 1780
RPC_X_ENUM_VALUE_OUT_OF_RANGE = 1781
RPC_X_BYTE_COUNT_TOO_SMALL = 1782
RPC_X_BAD_STUB_DATA = 1783
ERROR_INVALID_USER_BUFFER = 1784
ERROR_UNRECOGNIZED_MEDIA = 1785
ERROR_NO_TRUST_LSA_SECRET = 1786
ERROR_NO_TRUST_SAM_ACCOUNT = 1787
ERROR_TRUSTED_DOMAIN_FAILURE = 1788
ERROR_TRUSTED_RELATIONSHIP_FAILURE = 1789
ERROR_TRUST_FAILURE = 1790
RPC_S_CALL_IN_PROGRESS = 1791
ERROR_NETLOGON_NOT_STARTED = 1792
ERROR_ACCOUNT_EXPIRED = 1793
ERROR_REDIRECTOR_HAS_OPEN_HANDLES = 1794
ERROR_PRINTER_DRIVER_ALREADY_INSTALLED = 1795
ERROR_UNKNOWN_PORT = 1796
ERROR_UNKNOWN_PRINTER_DRIVER = 1797
ERROR_UNKNOWN_PRINTPROCESSOR = 1798
ERROR_INVALID_SEPARATOR_FILE = 1799
ERROR_INVALID_PRIORITY = 1800
ERROR_INVALID_PRINTER_NAME = 1801
ERROR_PRINTER_ALREADY_EXISTS = 1802
ERROR_INVALID_PRINTER_COMMAND = 1803
ERROR_INVALID_DATATYPE = 1804
ERROR_INVALID_ENVIRONMENT = 1805
RPC_S_NO_MORE_BINDINGS = 1806
ERROR_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT = 1807
ERROR_NOLOGON_WORKSTATION_TRUST_ACCOUNT = 1808
ERROR_NOLOGON_SERVER_TRUST_ACCOUNT = 1809
ERROR_DOMAIN_TRUST_INCONSISTENT = 1810
ERROR_SERVER_HAS_OPEN_HANDLES = 1811
ERROR_RESOURCE_DATA_NOT_FOUND = 1812
ERROR_RESOURCE_TYPE_NOT_FOUND = 1813
ERROR_RESOURCE_NAME_NOT_FOUND = 1814
ERROR_RESOURCE_LANG_NOT_FOUND = 1815
ERROR_NOT_ENOUGH_QUOTA = 1816
RPC_S_NO_INTERFACES = 1817
RPC_S_CALL_CANCELLED = 1818
RPC_S_BINDING_INCOMPLETE = 1819
RPC_S_COMM_FAILURE = 1820
RPC_S_UNSUPPORTED_AUTHN_LEVEL = 1821
RPC_S_NO_PRINC_NAME = 1822
RPC_S_NOT_RPC_ERROR = 1823
RPC_S_UUID_LOCAL_ONLY = 1824
RPC_S_SEC_PKG_ERROR = 1825
RPC_S_NOT_CANCELLED = 1826
RPC_X_INVALID_ES_ACTION = 1827
RPC_X_WRONG_ES_VERSION = 1828
RPC_X_WRONG_STUB_VERSION = 1829
RPC_X_INVALID_PIPE_OBJECT = 1830
RPC_X_WRONG_PIPE_ORDER = 1831
RPC_X_WRONG_PIPE_VERSION = 1832
RPC_S_COOKIE_AUTH_FAILED = 1833
RPC_S_DO_NOT_DISTURB = 1834
RPC_S_SYSTEM_HANDLE_COUNT_EXCEEDED = 1835
RPC_S_SYSTEM_HANDLE_TYPE_MISMATCH = 1836
RPC_S_GROUP_MEMBER_NOT_FOUND = 1898
EPT_S_CANT_CREATE = 1899
RPC_S_INVALID_OBJECT = 1900
ERROR_INVALID_TIME = 1901
ERROR_INVALID_FORM_NAME = 1902
ERROR_INVALID_FORM_SIZE = 1903
ERROR_ALREADY_WAITING = 1904
ERROR_PRINTER_DELETED = 1905
ERROR_INVALID_PRINTER_STATE = 1906
ERROR_PASSWORD_MUST_CHANGE = 1907
ERROR_DOMAIN_CONTROLLER_NOT_FOUND = 1908
ERROR_ACCOUNT_LOCKED_OUT = 1909
OR_INVALID_OXID = 1910
OR_INVALID_OID = 1911
OR_INVALID_SET = 1912
RPC_S_SEND_INCOMPLETE = 1913
RPC_S_INVALID_ASYNC_HANDLE = 1914
RPC_S_INVALID_ASYNC_CALL = 1915
RPC_X_PIPE_CLOSED = 1916
RPC_X_PIPE_DISCIPLINE_ERROR = 1917
RPC_X_PIPE_EMPTY = 1918
ERROR_NO_SITENAME = 1919
ERROR_CANT_ACCESS_FILE = 1920
ERROR_CANT_RESOLVE_FILENAME = 1921
RPC_S_ENTRY_TYPE_MISMATCH = 1922
RPC_S_NOT_ALL_OBJS_EXPORTED = 1923
RPC_S_INTERFACE_NOT_EXPORTED = 1924
RPC_S_PROFILE_NOT_ADDED = 1925
RPC_S_PRF_ELT_NOT_ADDED = 1926
RPC_S_PRF_ELT_NOT_REMOVED = 1927
RPC_S_GRP_ELT_NOT_ADDED = 1928
RPC_S_GRP_ELT_NOT_REMOVED = 1929
ERROR_KM_DRIVER_BLOCKED = 1930
ERROR_CONTEXT_EXPIRED = 1931
ERROR_PER_USER_TRUST_QUOTA_EXCEEDED = 1932
ERROR_ALL_USER_TRUST_QUOTA_EXCEEDED = 1933
ERROR_USER_DELETE_TRUST_QUOTA_EXCEEDED = 1934
ERROR_AUTHENTICATION_FIREWALL_FAILED = 1935
ERROR_REMOTE_PRINT_CONNECTIONS_BLOCKED = 1936
ERROR_NTLM_BLOCKED = 1937
ERROR_PASSWORD_CHANGE_REQUIRED = 1938
ERROR_LOST_MODE_LOGON_RESTRICTION = 1939
ERROR_INVALID_PIXEL_FORMAT = 2000
ERROR_BAD_DRIVER = 2001
ERROR_INVALID_WINDOW_STYLE = 2002
ERROR_METAFILE_NOT_SUPPORTED = 2003
ERROR_TRANSFORM_NOT_SUPPORTED = 2004
ERROR_CLIPPING_NOT_SUPPORTED = 2005
ERROR_INVALID_CMM = 2010
ERROR_INVALID_PROFILE = 2011
ERROR_TAG_NOT_FOUND = 2012
ERROR_TAG_NOT_PRESENT = 2013
ERROR_DUPLICATE_TAG = 2014
ERROR_PROFILE_NOT_ASSOCIATED_WITH_DEVICE = 2015
ERROR_PROFILE_NOT_FOUND = 2016
ERROR_INVALID_COLORSPACE = 2017
ERROR_ICM_NOT_ENABLED = 2018
ERROR_DELETING_ICM_XFORM = 2019
ERROR_INVALID_TRANSFORM = 2020
ERROR_COLORSPACE_MISMATCH = 2021
ERROR_INVALID_COLORINDEX = 2022
ERROR_PROFILE_DOES_NOT_MATCH_DEVICE = 2023
ERROR_CONNECTED_OTHER_PASSWORD = 2108
ERROR_CONNECTED_OTHER_PASSWORD_DEFAULT = 2109
ERROR_BAD_USERNAME = 2202
ERROR_NOT_CONNECTED = 2250
ERROR_OPEN_FILES = 2401
ERROR_ACTIVE_CONNECTIONS = 2402
ERROR_DEVICE_IN_USE = 2404
ERROR_UNKNOWN_PRINT_MONITOR = 3000
ERROR_PRINTER_DRIVER_IN_USE = 3001
ERROR_SPOOL_FILE_NOT_FOUND = 3002
ERROR_SPL_NO_STARTDOC = 3003
ERROR_SPL_NO_ADDJOB = 3004
ERROR_PRINT_PROCESSOR_ALREADY_INSTALLED = 3005
ERROR_PRINT_MONITOR_ALREADY_INSTALLED = 3006
ERROR_INVALID_PRINT_MONITOR = 3007
ERROR_PRINT_MONITOR_IN_USE = 3008
ERROR_PRINTER_HAS_JOBS_QUEUED = 3009
ERROR_SUCCESS_REBOOT_REQUIRED = 3010
ERROR_SUCCESS_RESTART_REQUIRED = 3011
ERROR_PRINTER_NOT_FOUND = 3012
ERROR_PRINTER_DRIVER_WARNED = 3013
ERROR_PRINTER_DRIVER_BLOCKED = 3014
ERROR_PRINTER_DRIVER_PACKAGE_IN_USE = 3015
ERROR_CORE_DRIVER_PACKAGE_NOT_FOUND = 3016
ERROR_FAIL_REBOOT_REQUIRED = 3017
ERROR_FAIL_REBOOT_INITIATED = 3018
ERROR_PRINTER_DRIVER_DOWNLOAD_NEEDED = 3019
ERROR_PRINT_JOB_RESTART_REQUIRED = 3020
ERROR_INVALID_PRINTER_DRIVER_MANIFEST = 3021
ERROR_PRINTER_NOT_SHAREABLE = 3022
ERROR_REQUEST_PAUSED = 3050
ERROR_APPEXEC_CONDITION_NOT_SATISFIED = 3060
ERROR_APPEXEC_HANDLE_INVALIDATED = 3061
ERROR_APPEXEC_INVALID_HOST_GENERATION = 3062
ERROR_APPEXEC_UNEXPECTED_PROCESS_REGISTRATION = 3063
ERROR_APPEXEC_INVALID_HOST_STATE = 3064
ERROR_APPEXEC_NO_DONOR = 3065
ERROR_APPEXEC_HOST_ID_MISMATCH = 3066
ERROR_APPEXEC_UNKNOWN_USER = 3067
ERROR_IO_REISSUE_AS_CACHED = 3950
ERROR_WINS_INTERNAL = 4000
ERROR_CAN_NOT_DEL_LOCAL_WINS = 4001
ERROR_STATIC_INIT = 4002
ERROR_INC_BACKUP = 4003
ERROR_FULL_BACKUP = 4004
ERROR_REC_NON_EXISTENT = 4005
ERROR_RPL_NOT_ALLOWED = 4006
PEERDIST_ERROR_CONTENTINFO_VERSION_UNSUPPORTED = 4050
PEERDIST_ERROR_CANNOT_PARSE_CONTENTINFO = 4051
PEERDIST_ERROR_MISSING_DATA = 4052
PEERDIST_ERROR_NO_MORE = 4053
PEERDIST_ERROR_NOT_INITIALIZED = 4054
PEERDIST_ERROR_ALREADY_INITIALIZED = 4055
PEERDIST_ERROR_SHUTDOWN_IN_PROGRESS = 4056
PEERDIST_ERROR_INVALIDATED = 4057
PEERDIST_ERROR_ALREADY_EXISTS = 4058
PEERDIST_ERROR_OPERATION_NOTFOUND = 4059
PEERDIST_ERROR_ALREADY_COMPLETED = 4060
PEERDIST_ERROR_OUT_OF_BOUNDS = 4061
PEERDIST_ERROR_VERSION_UNSUPPORTED = 4062
PEERDIST_ERROR_INVALID_CONFIGURATION = 4063
PEERDIST_ERROR_NOT_LICENSED = 4064
PEERDIST_ERROR_SERVICE_UNAVAILABLE = 4065
PEERDIST_ERROR_TRUST_FAILURE = 4066
ERROR_DHCP_ADDRESS_CONFLICT = 4100
ERROR_WMI_GUID_NOT_FOUND = 4200
ERROR_WMI_INSTANCE_NOT_FOUND = 4201
ERROR_WMI_ITEMID_NOT_FOUND = 4202
ERROR_WMI_TRY_AGAIN = 4203
ERROR_WMI_DP_NOT_FOUND = 4204
ERROR_WMI_UNRESOLVED_INSTANCE_REF = 4205
ERROR_WMI_ALREADY_ENABLED = 4206
ERROR_WMI_GUID_DISCONNECTED = 4207
ERROR_WMI_SERVER_UNAVAILABLE = 4208
ERROR_WMI_DP_FAILED = 4209
ERROR_WMI_INVALID_MOF = 4210
ERROR_WMI_INVALID_REGINFO = 4211
ERROR_WMI_ALREADY_DISABLED = 4212
ERROR_WMI_READ_ONLY = 4213
ERROR_WMI_SET_FAILURE = 4214
ERROR_NOT_APPCONTAINER = 4250
ERROR_APPCONTAINER_REQUIRED = 4251
ERROR_NOT_SUPPORTED_IN_APPCONTAINER = 4252
ERROR_INVALID_PACKAGE_SID_LENGTH = 4253
ERROR_INVALID_MEDIA = 4300
ERROR_INVALID_LIBRARY = 4301
ERROR_INVALID_MEDIA_POOL = 4302
ERROR_DRIVE_MEDIA_MISMATCH = 4303
ERROR_MEDIA_OFFLINE = 4304
ERROR_LIBRARY_OFFLINE = 4305
ERROR_EMPTY = 4306
ERROR_NOT_EMPTY = 4307
ERROR_MEDIA_UNAVAILABLE = 4308
ERROR_RESOURCE_DISABLED = 4309
ERROR_INVALID_CLEANER = 4310
ERROR_UNABLE_TO_CLEAN = 4311
ERROR_OBJECT_NOT_FOUND = 4312
ERROR_DATABASE_FAILURE = 4313
ERROR_DATABASE_FULL = 4314
ERROR_MEDIA_INCOMPATIBLE = 4315
ERROR_RESOURCE_NOT_PRESENT = 4316
ERROR_INVALID_OPERATION = 4317
ERROR_MEDIA_NOT_AVAILABLE = 4318
ERROR_DEVICE_NOT_AVAILABLE = 4319
ERROR_REQUEST_REFUSED = 4320
ERROR_INVALID_DRIVE_OBJECT = 4321
ERROR_LIBRARY_FULL = 4322
ERROR_MEDIUM_NOT_ACCESSIBLE = 4323
ERROR_UNABLE_TO_LOAD_MEDIUM = 4324
ERROR_UNABLE_TO_INVENTORY_DRIVE = 4325
ERROR_UNABLE_TO_INVENTORY_SLOT = 4326
ERROR_UNABLE_TO_INVENTORY_TRANSPORT = 4327
ERROR_TRANSPORT_FULL = 4328
ERROR_CONTROLLING_IEPORT = 4329
ERROR_UNABLE_TO_EJECT_MOUNTED_MEDIA = 4330
ERROR_CLEANER_SLOT_SET = 4331
ERROR_CLEANER_SLOT_NOT_SET = 4332
ERROR_CLEANER_CARTRIDGE_SPENT = 4333
ERROR_UNEXPECTED_OMID = 4334
ERROR_CANT_DELETE_LAST_ITEM = 4335
ERROR_MESSAGE_EXCEEDS_MAX_SIZE = 4336
ERROR_VOLUME_CONTAINS_SYS_FILES = 4337
ERROR_INDIGENOUS_TYPE = 4338
ERROR_NO_SUPPORTING_DRIVES = 4339
ERROR_CLEANER_CARTRIDGE_INSTALLED = 4340
ERROR_IEPORT_FULL = 4341
ERROR_FILE_OFFLINE = 4350
ERROR_REMOTE_STORAGE_NOT_ACTIVE = 4351
ERROR_REMOTE_STORAGE_MEDIA_ERROR = 4352
ERROR_NOT_A_REPARSE_POINT = 4390
ERROR_REPARSE_ATTRIBUTE_CONFLICT = 4391
ERROR_INVALID_REPARSE_DATA = 4392
ERROR_REPARSE_TAG_INVALID = 4393
ERROR_REPARSE_TAG_MISMATCH = 4394
ERROR_REPARSE_POINT_ENCOUNTERED = 4395
ERROR_APP_DATA_NOT_FOUND = 4400
ERROR_APP_DATA_EXPIRED = 4401
ERROR_APP_DATA_CORRUPT = 4402
ERROR_APP_DATA_LIMIT_EXCEEDED = 4403
ERROR_APP_DATA_REBOOT_REQUIRED = 4404
ERROR_SECUREBOOT_ROLLBACK_DETECTED = 4420
ERROR_SECUREBOOT_POLICY_VIOLATION = 4421
ERROR_SECUREBOOT_INVALID_POLICY = 4422
ERROR_SECUREBOOT_POLICY_PUBLISHER_NOT_FOUND = 4423
ERROR_SECUREBOOT_POLICY_NOT_SIGNED = 4424
ERROR_SECUREBOOT_NOT_ENABLED = 4425
ERROR_SECUREBOOT_FILE_REPLACED = 4426
ERROR_SECUREBOOT_POLICY_NOT_AUTHORIZED = 4427
ERROR_SECUREBOOT_POLICY_UNKNOWN = 4428
ERROR_SECUREBOOT_POLICY_MISSING_ANTIROLLBACKVERSION = 4429
ERROR_SECUREBOOT_PLATFORM_ID_MISMATCH = 4430
ERROR_SECUREBOOT_POLICY_ROLLBACK_DETECTED = 4431
ERROR_SECUREBOOT_POLICY_UPGRADE_MISMATCH = 4432
ERROR_SECUREBOOT_REQUIRED_POLICY_FILE_MISSING = 4433
ERROR_SECUREBOOT_NOT_BASE_POLICY = 4434
ERROR_SECUREBOOT_NOT_SUPPLEMENTAL_POLICY = 4435
ERROR_OFFLOAD_READ_FLT_NOT_SUPPORTED = 4440
ERROR_OFFLOAD_WRITE_FLT_NOT_SUPPORTED = 4441
ERROR_OFFLOAD_READ_FILE_NOT_SUPPORTED = 4442
ERROR_OFFLOAD_WRITE_FILE_NOT_SUPPORTED = 4443
ERROR_ALREADY_HAS_STREAM_ID = 4444
ERROR_SMR_GARBAGE_COLLECTION_REQUIRED = 4445
ERROR_WOF_WIM_HEADER_CORRUPT = 4446
ERROR_WOF_WIM_RESOURCE_TABLE_CORRUPT = 4447
ERROR_WOF_FILE_RESOURCE_TABLE_CORRUPT = 4448
ERROR_VOLUME_NOT_SIS_ENABLED = 4500
ERROR_SYSTEM_INTEGRITY_ROLLBACK_DETECTED = 4550
ERROR_SYSTEM_INTEGRITY_POLICY_VIOLATION = 4551
ERROR_SYSTEM_INTEGRITY_INVALID_POLICY = 4552
ERROR_SYSTEM_INTEGRITY_POLICY_NOT_SIGNED = 4553
ERROR_SYSTEM_INTEGRITY_TOO_MANY_POLICIES = 4554
ERROR_SYSTEM_INTEGRITY_SUPPLEMENTAL_POLICY_NOT_AUTHORIZED = 4555
ERROR_VSM_NOT_INITIALIZED = 4560
ERROR_VSM_DMA_PROTECTION_NOT_IN_USE = 4561
ERROR_PLATFORM_MANIFEST_NOT_AUTHORIZED = 4570
ERROR_PLATFORM_MANIFEST_INVALID = 4571
ERROR_PLATFORM_MANIFEST_FILE_NOT_AUTHORIZED = 4572
ERROR_PLATFORM_MANIFEST_CATALOG_NOT_AUTHORIZED = 4573
ERROR_PLATFORM_MANIFEST_BINARY_ID_NOT_FOUND = 4574
ERROR_PLATFORM_MANIFEST_NOT_ACTIVE = 4575
ERROR_PLATFORM_MANIFEST_NOT_SIGNED = 4576
ERROR_DEPENDENT_RESOURCE_EXISTS = 5001
ERROR_DEPENDENCY_NOT_FOUND = 5002
ERROR_DEPENDENCY_ALREADY_EXISTS = 5003
ERROR_RESOURCE_NOT_ONLINE = 5004
ERROR_HOST_NODE_NOT_AVAILABLE = 5005
ERROR_RESOURCE_NOT_AVAILABLE = 5006
ERROR_RESOURCE_NOT_FOUND = 5007
ERROR_SHUTDOWN_CLUSTER = 5008
ERROR_CANT_EVICT_ACTIVE_NODE = 5009
ERROR_OBJECT_ALREADY_EXISTS = 5010
ERROR_OBJECT_IN_LIST = 5011
ERROR_GROUP_NOT_AVAILABLE = 5012
ERROR_GROUP_NOT_FOUND = 5013
ERROR_GROUP_NOT_ONLINE = 5014
ERROR_HOST_NODE_NOT_RESOURCE_OWNER = 5015
ERROR_HOST_NODE_NOT_GROUP_OWNER = 5016
ERROR_RESMON_CREATE_FAILED = 5017
ERROR_RESMON_ONLINE_FAILED = 5018
ERROR_RESOURCE_ONLINE = 5019
ERROR_QUORUM_RESOURCE = 5020
ERROR_NOT_QUORUM_CAPABLE = 5021
ERROR_CLUSTER_SHUTTING_DOWN = 5022
ERROR_INVALID_STATE = 5023
ERROR_RESOURCE_PROPERTIES_STORED = 5024
ERROR_NOT_QUORUM_CLASS = 5025
ERROR_CORE_RESOURCE = 5026
ERROR_QUORUM_RESOURCE_ONLINE_FAILED = 5027
ERROR_QUORUMLOG_OPEN_FAILED = 5028
ERROR_CLUSTERLOG_CORRUPT = 5029
ERROR_CLUSTERLOG_RECORD_EXCEEDS_MAXSIZE = 5030
ERROR_CLUSTERLOG_EXCEEDS_MAXSIZE = 5031
ERROR_CLUSTERLOG_CHKPOINT_NOT_FOUND = 5032
ERROR_CLUSTERLOG_NOT_ENOUGH_SPACE = 5033
ERROR_QUORUM_OWNER_ALIVE = 5034
ERROR_NETWORK_NOT_AVAILABLE = 5035
ERROR_NODE_NOT_AVAILABLE = 5036
ERROR_ALL_NODES_NOT_AVAILABLE = 5037
ERROR_RESOURCE_FAILED = 5038
ERROR_CLUSTER_INVALID_NODE = 5039
ERROR_CLUSTER_NODE_EXISTS = 5040
ERROR_CLUSTER_JOIN_IN_PROGRESS = 5041
ERROR_CLUSTER_NODE_NOT_FOUND = 5042
ERROR_CLUSTER_LOCAL_NODE_NOT_FOUND = 5043
ERROR_CLUSTER_NETWORK_EXISTS = 5044
ERROR_CLUSTER_NETWORK_NOT_FOUND = 5045
ERROR_CLUSTER_NETINTERFACE_EXISTS = 5046
ERROR_CLUSTER_NETINTERFACE_NOT_FOUND = 5047
ERROR_CLUSTER_INVALID_REQUEST = 5048
ERROR_CLUSTER_INVALID_NETWORK_PROVIDER = 5049
ERROR_CLUSTER_NODE_DOWN = 5050
ERROR_CLUSTER_NODE_UNREACHABLE = 5051
ERROR_CLUSTER_NODE_NOT_MEMBER = 5052
ERROR_CLUSTER_JOIN_NOT_IN_PROGRESS = 5053
ERROR_CLUSTER_INVALID_NETWORK = 5054
ERROR_CLUSTER_NODE_UP = 5056
ERROR_CLUSTER_IPADDR_IN_USE = 5057
ERROR_CLUSTER_NODE_NOT_PAUSED = 5058
ERROR_CLUSTER_NO_SECURITY_CONTEXT = 5059
ERROR_CLUSTER_NETWORK_NOT_INTERNAL = 5060
ERROR_CLUSTER_NODE_ALREADY_UP = 5061
ERROR_CLUSTER_NODE_ALREADY_DOWN = 5062
ERROR_CLUSTER_NETWORK_ALREADY_ONLINE = 5063
ERROR_CLUSTER_NETWORK_ALREADY_OFFLINE = 5064
ERROR_CLUSTER_NODE_ALREADY_MEMBER = 5065
ERROR_CLUSTER_LAST_INTERNAL_NETWORK = 5066
ERROR_CLUSTER_NETWORK_HAS_DEPENDENTS = 5067
ERROR_INVALID_OPERATION_ON_QUORUM = 5068
ERROR_DEPENDENCY_NOT_ALLOWED = 5069
ERROR_CLUSTER_NODE_PAUSED = 5070
ERROR_NODE_CANT_HOST_RESOURCE = 5071
ERROR_CLUSTER_NODE_NOT_READY = 5072
ERROR_CLUSTER_NODE_SHUTTING_DOWN = 5073
ERROR_CLUSTER_JOIN_ABORTED = 5074
ERROR_CLUSTER_INCOMPATIBLE_VERSIONS = 5075
ERROR_CLUSTER_MAXNUM_OF_RESOURCES_EXCEEDED = 5076
ERROR_CLUSTER_SYSTEM_CONFIG_CHANGED = 5077
ERROR_CLUSTER_RESOURCE_TYPE_NOT_FOUND = 5078
ERROR_CLUSTER_RESTYPE_NOT_SUPPORTED = 5079
ERROR_CLUSTER_RESNAME_NOT_FOUND = 5080
ERROR_CLUSTER_NO_RPC_PACKAGES_REGISTERED = 5081
ERROR_CLUSTER_OWNER_NOT_IN_PREFLIST = 5082
ERROR_CLUSTER_DATABASE_SEQMISMATCH = 5083
ERROR_RESMON_INVALID_STATE = 5084
ERROR_CLUSTER_GUM_NOT_LOCKER = 5085
ERROR_QUORUM_DISK_NOT_FOUND = 5086
ERROR_DATABASE_BACKUP_CORRUPT = 5087
ERROR_CLUSTER_NODE_ALREADY_HAS_DFS_ROOT = 5088
ERROR_RESOURCE_PROPERTY_UNCHANGEABLE = 5089
ERROR_NO_ADMIN_ACCESS_POINT = 5090
ERROR_CLUSTER_MEMBERSHIP_INVALID_STATE = 5890
ERROR_CLUSTER_QUORUMLOG_NOT_FOUND = 5891
ERROR_CLUSTER_MEMBERSHIP_HALT = 5892
ERROR_CLUSTER_INSTANCE_ID_MISMATCH = 5893
ERROR_CLUSTER_NETWORK_NOT_FOUND_FOR_IP = 5894
ERROR_CLUSTER_PROPERTY_DATA_TYPE_MISMATCH = 5895
ERROR_CLUSTER_EVICT_WITHOUT_CLEANUP = 5896
ERROR_CLUSTER_PARAMETER_MISMATCH = 5897
ERROR_NODE_CANNOT_BE_CLUSTERED = 5898
ERROR_CLUSTER_WRONG_OS_VERSION = 5899
ERROR_CLUSTER_CANT_CREATE_DUP_CLUSTER_NAME = 5900
ERROR_CLUSCFG_ALREADY_COMMITTED = 5901
ERROR_CLUSCFG_ROLLBACK_FAILED = 5902
ERROR_CLUSCFG_SYSTEM_DISK_DRIVE_LETTER_CONFLICT = 5903
ERROR_CLUSTER_OLD_VERSION = 5904
ERROR_CLUSTER_MISMATCHED_COMPUTER_ACCT_NAME = 5905
ERROR_CLUSTER_NO_NET_ADAPTERS = 5906
ERROR_CLUSTER_POISONED = 5907
ERROR_CLUSTER_GROUP_MOVING = 5908
ERROR_CLUSTER_RESOURCE_TYPE_BUSY = 5909
ERROR_RESOURCE_CALL_TIMED_OUT = 5910
ERROR_INVALID_CLUSTER_IPV6_ADDRESS = 5911
ERROR_CLUSTER_INTERNAL_INVALID_FUNCTION = 5912
ERROR_CLUSTER_PARAMETER_OUT_OF_BOUNDS = 5913
ERROR_CLUSTER_PARTIAL_SEND = 5914
ERROR_CLUSTER_REGISTRY_INVALID_FUNCTION = 5915
ERROR_CLUSTER_INVALID_STRING_TERMINATION = 5916
ERROR_CLUSTER_INVALID_STRING_FORMAT = 5917
ERROR_CLUSTER_DATABASE_TRANSACTION_IN_PROGRESS = 5918
ERROR_CLUSTER_DATABASE_TRANSACTION_NOT_IN_PROGRESS = 5919
ERROR_CLUSTER_NULL_DATA = 5920
ERROR_CLUSTER_PARTIAL_READ = 5921
ERROR_CLUSTER_PARTIAL_WRITE = 5922
ERROR_CLUSTER_CANT_DESERIALIZE_DATA = 5923
ERROR_DEPENDENT_RESOURCE_PROPERTY_CONFLICT = 5924
ERROR_CLUSTER_NO_QUORUM = 5925
ERROR_CLUSTER_INVALID_IPV6_NETWORK = 5926
ERROR_CLUSTER_INVALID_IPV6_TUNNEL_NETWORK = 5927
ERROR_QUORUM_NOT_ALLOWED_IN_THIS_GROUP = 5928
ERROR_DEPENDENCY_TREE_TOO_COMPLEX = 5929
ERROR_EXCEPTION_IN_RESOURCE_CALL = 5930
ERROR_CLUSTER_RHS_FAILED_INITIALIZATION = 5931
ERROR_CLUSTER_NOT_INSTALLED = 5932
ERROR_CLUSTER_RESOURCES_MUST_BE_ONLINE_ON_THE_SAME_NODE = 5933
ERROR_CLUSTER_MAX_NODES_IN_CLUSTER = 5934
ERROR_CLUSTER_TOO_MANY_NODES = 5935
ERROR_CLUSTER_OBJECT_ALREADY_USED = 5936
ERROR_NONCORE_GROUPS_FOUND = 5937
ERROR_FILE_SHARE_RESOURCE_CONFLICT = 5938
ERROR_CLUSTER_EVICT_INVALID_REQUEST = 5939
ERROR_CLUSTER_SINGLETON_RESOURCE = 5940
ERROR_CLUSTER_GROUP_SINGLETON_RESOURCE = 5941
ERROR_CLUSTER_RESOURCE_PROVIDER_FAILED = 5942
ERROR_CLUSTER_RESOURCE_CONFIGURATION_ERROR = 5943
ERROR_CLUSTER_GROUP_BUSY = 5944
ERROR_CLUSTER_NOT_SHARED_VOLUME = 5945
ERROR_CLUSTER_INVALID_SECURITY_DESCRIPTOR = 5946
ERROR_CLUSTER_SHARED_VOLUMES_IN_USE = 5947
ERROR_CLUSTER_USE_SHARED_VOLUMES_API = 5948
ERROR_CLUSTER_BACKUP_IN_PROGRESS = 5949
ERROR_NON_CSV_PATH = 5950
ERROR_CSV_VOLUME_NOT_LOCAL = 5951
ERROR_CLUSTER_WATCHDOG_TERMINATING = 5952
ERROR_CLUSTER_RESOURCE_VETOED_MOVE_INCOMPATIBLE_NODES = 5953
ERROR_CLUSTER_INVALID_NODE_WEIGHT = 5954
ERROR_CLUSTER_RESOURCE_VETOED_CALL = 5955
ERROR_RESMON_SYSTEM_RESOURCES_LACKING = 5956
ERROR_CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_DESTINATION = 5957
ERROR_CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_SOURCE = 5958
ERROR_CLUSTER_GROUP_QUEUED = 5959
ERROR_CLUSTER_RESOURCE_LOCKED_STATUS = 5960
ERROR_CLUSTER_SHARED_VOLUME_FAILOVER_NOT_ALLOWED = 5961
ERROR_CLUSTER_NODE_DRAIN_IN_PROGRESS = 5962
ERROR_CLUSTER_DISK_NOT_CONNECTED = 5963
ERROR_DISK_NOT_CSV_CAPABLE = 5964
ERROR_RESOURCE_NOT_IN_AVAILABLE_STORAGE = 5965
ERROR_CLUSTER_SHARED_VOLUME_REDIRECTED = 5966
ERROR_CLUSTER_SHARED_VOLUME_NOT_REDIRECTED = 5967
ERROR_CLUSTER_CANNOT_RETURN_PROPERTIES = 5968
ERROR_CLUSTER_RESOURCE_CONTAINS_UNSUPPORTED_DIFF_AREA_FOR_SHARED_VOLUMES = 5969
ERROR_CLUSTER_RESOURCE_IS_IN_MAINTENANCE_MODE = 5970
ERROR_CLUSTER_AFFINITY_CONFLICT = 5971
ERROR_CLUSTER_RESOURCE_IS_REPLICA_VIRTUAL_MACHINE = 5972
ERROR_CLUSTER_UPGRADE_INCOMPATIBLE_VERSIONS = 5973
ERROR_CLUSTER_UPGRADE_FIX_QUORUM_NOT_SUPPORTED = 5974
ERROR_CLUSTER_UPGRADE_RESTART_REQUIRED = 5975
ERROR_CLUSTER_UPGRADE_IN_PROGRESS = 5976
ERROR_CLUSTER_UPGRADE_INCOMPLETE = 5977
ERROR_CLUSTER_NODE_IN_GRACE_PERIOD = 5978
ERROR_CLUSTER_CSV_IO_PAUSE_TIMEOUT = 5979
ERROR_NODE_NOT_ACTIVE_CLUSTER_MEMBER = 5980
ERROR_CLUSTER_RESOURCE_NOT_MONITORED = 5981
ERROR_CLUSTER_RESOURCE_DOES_NOT_SUPPORT_UNMONITORED = 5982
ERROR_CLUSTER_RESOURCE_IS_REPLICATED = 5983
ERROR_CLUSTER_NODE_ISOLATED = 5984
ERROR_CLUSTER_NODE_QUARANTINED = 5985
ERROR_CLUSTER_DATABASE_UPDATE_CONDITION_FAILED = 5986
ERROR_CLUSTER_SPACE_DEGRADED = 5987
ERROR_CLUSTER_TOKEN_DELEGATION_NOT_SUPPORTED = 5988
ERROR_CLUSTER_CSV_INVALID_HANDLE = 5989
ERROR_CLUSTER_CSV_SUPPORTED_ONLY_ON_COORDINATOR = 5990
ERROR_GROUPSET_NOT_AVAILABLE = 5991
ERROR_GROUPSET_NOT_FOUND = 5992
ERROR_GROUPSET_CANT_PROVIDE = 5993
ERROR_CLUSTER_FAULT_DOMAIN_PARENT_NOT_FOUND = 5994
ERROR_CLUSTER_FAULT_DOMAIN_INVALID_HIERARCHY = 5995
ERROR_CLUSTER_FAULT_DOMAIN_FAILED_S2D_VALIDATION = 5996
ERROR_CLUSTER_FAULT_DOMAIN_S2D_CONNECTIVITY_LOSS = 5997
ERROR_CLUSTER_INVALID_INFRASTRUCTURE_FILESERVER_NAME = 5998
ERROR_CLUSTERSET_MANAGEMENT_CLUSTER_UNREACHABLE = 5999
ERROR_ENCRYPTION_FAILED = 6000
ERROR_DECRYPTION_FAILED = 6001
ERROR_FILE_ENCRYPTED = 6002
ERROR_NO_RECOVERY_POLICY = 6003
ERROR_NO_EFS = 6004
ERROR_WRONG_EFS = 6005
ERROR_NO_USER_KEYS = 6006
ERROR_FILE_NOT_ENCRYPTED = 6007
ERROR_NOT_EXPORT_FORMAT = 6008
ERROR_FILE_READ_ONLY = 6009
ERROR_DIR_EFS_DISALLOWED = 6010
ERROR_EFS_SERVER_NOT_TRUSTED = 6011
ERROR_BAD_RECOVERY_POLICY = 6012
ERROR_EFS_ALG_BLOB_TOO_BIG = 6013
ERROR_VOLUME_NOT_SUPPORT_EFS = 6014
ERROR_EFS_DISABLED = 6015
ERROR_EFS_VERSION_NOT_SUPPORT = 6016
ERROR_CS_ENCRYPTION_INVALID_SERVER_RESPONSE = 6017
ERROR_CS_ENCRYPTION_UNSUPPORTED_SERVER = 6018
ERROR_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE = 6019
ERROR_CS_ENCRYPTION_NEW_ENCRYPTED_FILE = 6020
ERROR_CS_ENCRYPTION_FILE_NOT_CSE = 6021
ERROR_ENCRYPTION_POLICY_DENIES_OPERATION = 6022
ERROR_WIP_ENCRYPTION_FAILED = 6023
ERROR_NO_BROWSER_SERVERS_FOUND = 6118
SCHED_E_SERVICE_NOT_LOCALSYSTEM = 6200
ERROR_CLUSTER_OBJECT_IS_CLUSTER_SET_VM = 6250
ERROR_LOG_SECTOR_INVALID = 6600
ERROR_LOG_SECTOR_PARITY_INVALID = 6601
ERROR_LOG_SECTOR_REMAPPED = 6602
ERROR_LOG_BLOCK_INCOMPLETE = 6603
ERROR_LOG_INVALID_RANGE = 6604
ERROR_LOG_BLOCKS_EXHAUSTED = 6605
ERROR_LOG_READ_CONTEXT_INVALID = 6606
ERROR_LOG_RESTART_INVALID = 6607
ERROR_LOG_BLOCK_VERSION = 6608
ERROR_LOG_BLOCK_INVALID = 6609
ERROR_LOG_READ_MODE_INVALID = 6610
ERROR_LOG_NO_RESTART = 6611
ERROR_LOG_METADATA_CORRUPT = 6612
ERROR_LOG_METADATA_INVALID = 6613
ERROR_LOG_METADATA_INCONSISTENT = 6614
ERROR_LOG_RESERVATION_INVALID = 6615
ERROR_LOG_CANT_DELETE = 6616
ERROR_LOG_CONTAINER_LIMIT_EXCEEDED = 6617
ERROR_LOG_START_OF_LOG = 6618
ERROR_LOG_POLICY_ALREADY_INSTALLED = 6619
ERROR_LOG_POLICY_NOT_INSTALLED = 6620
ERROR_LOG_POLICY_INVALID = 6621
ERROR_LOG_POLICY_CONFLICT = 6622
ERROR_LOG_PINNED_ARCHIVE_TAIL = 6623
ERROR_LOG_RECORD_NONEXISTENT = 6624
ERROR_LOG_RECORDS_RESERVED_INVALID = 6625
ERROR_LOG_SPACE_RESERVED_INVALID = 6626
ERROR_LOG_TAIL_INVALID = 6627
ERROR_LOG_FULL = 6628
ERROR_COULD_NOT_RESIZE_LOG = 6629
ERROR_LOG_MULTIPLEXED = 6630
ERROR_LOG_DEDICATED = 6631
ERROR_LOG_ARCHIVE_NOT_IN_PROGRESS = 6632
ERROR_LOG_ARCHIVE_IN_PROGRESS = 6633
ERROR_LOG_EPHEMERAL = 6634
ERROR_LOG_NOT_ENOUGH_CONTAINERS = 6635
ERROR_LOG_CLIENT_ALREADY_REGISTERED = 6636
ERROR_LOG_CLIENT_NOT_REGISTERED = 6637
ERROR_LOG_FULL_HANDLER_IN_PROGRESS = 6638
ERROR_LOG_CONTAINER_READ_FAILED = 6639
ERROR_LOG_CONTAINER_WRITE_FAILED = 6640
ERROR_LOG_CONTAINER_OPEN_FAILED = 6641
ERROR_LOG_CONTAINER_STATE_INVALID = 6642
ERROR_LOG_STATE_INVALID = 6643
ERROR_LOG_PINNED = 6644
ERROR_LOG_METADATA_FLUSH_FAILED = 6645
ERROR_LOG_INCONSISTENT_SECURITY = 6646
ERROR_LOG_APPENDED_FLUSH_FAILED = 6647
ERROR_LOG_PINNED_RESERVATION = 6648
ERROR_INVALID_TRANSACTION = 6700
ERROR_TRANSACTION_NOT_ACTIVE = 6701
ERROR_TRANSACTION_REQUEST_NOT_VALID = 6702
ERROR_TRANSACTION_NOT_REQUESTED = 6703
ERROR_TRANSACTION_ALREADY_ABORTED = 6704
ERROR_TRANSACTION_ALREADY_COMMITTED = 6705
ERROR_TM_INITIALIZATION_FAILED = 6706
ERROR_RESOURCEMANAGER_READ_ONLY = 6707
ERROR_TRANSACTION_NOT_JOINED = 6708
ERROR_TRANSACTION_SUPERIOR_EXISTS = 6709
ERROR_CRM_PROTOCOL_ALREADY_EXISTS = 6710
ERROR_TRANSACTION_PROPAGATION_FAILED = 6711
ERROR_CRM_PROTOCOL_NOT_FOUND = 6712
ERROR_TRANSACTION_INVALID_MARSHALL_BUFFER = 6713
ERROR_CURRENT_TRANSACTION_NOT_VALID = 6714
ERROR_TRANSACTION_NOT_FOUND = 6715
ERROR_RESOURCEMANAGER_NOT_FOUND = 6716
ERROR_ENLISTMENT_NOT_FOUND = 6717
ERROR_TRANSACTIONMANAGER_NOT_FOUND = 6718
ERROR_TRANSACTIONMANAGER_NOT_ONLINE = 6719
ERROR_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION = 6720
ERROR_TRANSACTION_NOT_ROOT = 6721
ERROR_TRANSACTION_OBJECT_EXPIRED = 6722
ERROR_TRANSACTION_RESPONSE_NOT_ENLISTED = 6723
ERROR_TRANSACTION_RECORD_TOO_LONG = 6724
ERROR_IMPLICIT_TRANSACTION_NOT_SUPPORTED = 6725
ERROR_TRANSACTION_INTEGRITY_VIOLATED = 6726
ERROR_TRANSACTIONMANAGER_IDENTITY_MISMATCH = 6727
ERROR_RM_CANNOT_BE_FROZEN_FOR_SNAPSHOT = 6728
ERROR_TRANSACTION_MUST_WRITETHROUGH = 6729
ERROR_TRANSACTION_NO_SUPERIOR = 6730
ERROR_HEURISTIC_DAMAGE_POSSIBLE = 6731
ERROR_TRANSACTIONAL_CONFLICT = 6800
ERROR_RM_NOT_ACTIVE = 6801
ERROR_RM_METADATA_CORRUPT = 6802
ERROR_DIRECTORY_NOT_RM = 6803
ERROR_TRANSACTIONS_UNSUPPORTED_REMOTE = 6805
ERROR_LOG_RESIZE_INVALID_SIZE = 6806
ERROR_OBJECT_NO_LONGER_EXISTS = 6807
ERROR_STREAM_MINIVERSION_NOT_FOUND = 6808
ERROR_STREAM_MINIVERSION_NOT_VALID = 6809
ERROR_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION = 6810
ERROR_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT = 6811
ERROR_CANT_CREATE_MORE_STREAM_MINIVERSIONS = 6812
ERROR_REMOTE_FILE_VERSION_MISMATCH = 6814
ERROR_HANDLE_NO_LONGER_VALID = 6815
ERROR_NO_TXF_METADATA = 6816
ERROR_LOG_CORRUPTION_DETECTED = 6817
ERROR_CANT_RECOVER_WITH_HANDLE_OPEN = 6818
ERROR_RM_DISCONNECTED = 6819
ERROR_ENLISTMENT_NOT_SUPERIOR = 6820
ERROR_RECOVERY_NOT_NEEDED = 6821
ERROR_RM_ALREADY_STARTED = 6822
ERROR_FILE_IDENTITY_NOT_PERSISTENT = 6823
ERROR_CANT_BREAK_TRANSACTIONAL_DEPENDENCY = 6824
ERROR_CANT_CROSS_RM_BOUNDARY = 6825
ERROR_TXF_DIR_NOT_EMPTY = 6826
ERROR_INDOUBT_TRANSACTIONS_EXIST = 6827
ERROR_TM_VOLATILE = 6828
ERROR_ROLLBACK_TIMER_EXPIRED = 6829
ERROR_TXF_ATTRIBUTE_CORRUPT = 6830
ERROR_EFS_NOT_ALLOWED_IN_TRANSACTION = 6831
ERROR_TRANSACTIONAL_OPEN_NOT_ALLOWED = 6832
ERROR_LOG_GROWTH_FAILED = 6833
ERROR_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE = 6834
ERROR_TXF_METADATA_ALREADY_PRESENT = 6835
ERROR_TRANSACTION_SCOPE_CALLBACKS_NOT_SET = 6836
ERROR_TRANSACTION_REQUIRED_PROMOTION = 6837
ERROR_CANNOT_EXECUTE_FILE_IN_TRANSACTION = 6838
ERROR_TRANSACTIONS_NOT_FROZEN = 6839
ERROR_TRANSACTION_FREEZE_IN_PROGRESS = 6840
ERROR_NOT_SNAPSHOT_VOLUME = 6841
ERROR_NO_SAVEPOINT_WITH_OPEN_FILES = 6842
ERROR_DATA_LOST_REPAIR = 6843
ERROR_SPARSE_NOT_ALLOWED_IN_TRANSACTION = 6844
ERROR_TM_IDENTITY_MISMATCH = 6845
ERROR_FLOATED_SECTION = 6846
ERROR_CANNOT_ACCEPT_TRANSACTED_WORK = 6847
ERROR_CANNOT_ABORT_TRANSACTIONS = 6848
ERROR_BAD_CLUSTERS = 6849
ERROR_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION = 6850
ERROR_VOLUME_DIRTY = 6851
ERROR_NO_LINK_TRACKING_IN_TRANSACTION = 6852
ERROR_OPERATION_NOT_SUPPORTED_IN_TRANSACTION = 6853
ERROR_EXPIRED_HANDLE = 6854
ERROR_TRANSACTION_NOT_ENLISTED = 6855
ERROR_CTX_WINSTATION_NAME_INVALID = 7001
ERROR_CTX_INVALID_PD = 7002
ERROR_CTX_PD_NOT_FOUND = 7003
ERROR_CTX_WD_NOT_FOUND = 7004
ERROR_CTX_CANNOT_MAKE_EVENTLOG_ENTRY = 7005
ERROR_CTX_SERVICE_NAME_COLLISION = 7006
ERROR_CTX_CLOSE_PENDING = 7007
ERROR_CTX_NO_OUTBUF = 7008
ERROR_CTX_MODEM_INF_NOT_FOUND = 7009
ERROR_CTX_INVALID_MODEMNAME = 7010
ERROR_CTX_MODEM_RESPONSE_ERROR = 7011
ERROR_CTX_MODEM_RESPONSE_TIMEOUT = 7012
ERROR_CTX_MODEM_RESPONSE_NO_CARRIER = 7013
ERROR_CTX_MODEM_RESPONSE_NO_DIALTONE = 7014
ERROR_CTX_MODEM_RESPONSE_BUSY = 7015
ERROR_CTX_MODEM_RESPONSE_VOICE = 7016
ERROR_CTX_TD_ERROR = 7017
ERROR_CTX_WINSTATION_NOT_FOUND = 7022
ERROR_CTX_WINSTATION_ALREADY_EXISTS = 7023
ERROR_CTX_WINSTATION_BUSY = 7024
ERROR_CTX_BAD_VIDEO_MODE = 7025
ERROR_CTX_GRAPHICS_INVALID = 7035
ERROR_CTX_LOGON_DISABLED = 7037
ERROR_CTX_NOT_CONSOLE = 7038
ERROR_CTX_CLIENT_QUERY_TIMEOUT = 7040
ERROR_CTX_CONSOLE_DISCONNECT = 7041
ERROR_CTX_CONSOLE_CONNECT = 7042
ERROR_CTX_SHADOW_DENIED = 7044
ERROR_CTX_WINSTATION_ACCESS_DENIED = 7045
ERROR_CTX_INVALID_WD = 7049
ERROR_CTX_SHADOW_INVALID = 7050
ERROR_CTX_SHADOW_DISABLED = 7051
ERROR_CTX_CLIENT_LICENSE_IN_USE = 7052
ERROR_CTX_CLIENT_LICENSE_NOT_SET = 7053
ERROR_CTX_LICENSE_NOT_AVAILABLE = 7054
ERROR_CTX_LICENSE_CLIENT_INVALID = 7055
ERROR_CTX_LICENSE_EXPIRED = 7056
ERROR_CTX_SHADOW_NOT_RUNNING = 7057
ERROR_CTX_SHADOW_ENDED_BY_MODE_CHANGE = 7058
ERROR_ACTIVATION_COUNT_EXCEEDED = 7059
ERROR_CTX_WINSTATIONS_DISABLED = 7060
ERROR_CTX_ENCRYPTION_LEVEL_REQUIRED = 7061
ERROR_CTX_SESSION_IN_USE = 7062
ERROR_CTX_NO_FORCE_LOGOFF = 7063
ERROR_CTX_ACCOUNT_RESTRICTION = 7064
ERROR_RDP_PROTOCOL_ERROR = 7065
ERROR_CTX_CDM_CONNECT = 7066
ERROR_CTX_CDM_DISCONNECT = 7067
ERROR_CTX_SECURITY_LAYER_ERROR = 7068
ERROR_TS_INCOMPATIBLE_SESSIONS = 7069
ERROR_TS_VIDEO_SUBSYSTEM_ERROR = 7070
ERROR_RESOURCE_ENUM_USER_STOP = 15106

WSABASEERR = 10000
WSAEINTR = 10004
WSAEBADF = 10009
WSAEACCES = 10013
WSAEFAULT = 10014
WSAEINVAL = 10022
WSAEMFILE = 10024
WSAEWOULDBLOCK = 10035
WSAEINPROGRESS = 10036
WSAEALREADY = 10037
WSAENOTSOCK = 10038
WSAEDESTADDRREQ = 10039
WSAEMSGSIZE = 10040
WSAEPROTOTYPE = 10041
WSAENOPROTOOPT = 10042
WSAEPROTONOSUPPORT = 10043
WSAESOCKTNOSUPPORT = 10044
WSAEOPNOTSUPP = 10045
WSAEPFNOSUPPORT = 10046
WSAEAFNOSUPPORT = 10047
WSAEADDRINUSE = 10048
WSAEADDRNOTAVAIL = 10049
WSAENETDOWN = 10050
WSAENETUNREACH = 10051
WSAENETRESET = 10052
WSAECONNABORTED = 10053
WSAECONNRESET = 10054
WSAENOBUFS = 10055
WSAEISCONN = 10056
WSAENOTCONN = 10057
WSAESHUTDOWN = 10058
WSAETOOMANYREFS = 10059
WSAETIMEDOUT = 10060
WSAECONNREFUSED = 10061
WSAELOOP = 10062
WSAENAMETOOLONG = 10063
WSAEHOSTDOWN = 10064
WSAEHOSTUNREACH = 10065
WSAENOTEMPTY = 10066
WSAEPROCLIM = 10067
WSAEUSERS = 10068
WSAEDQUOT = 10069
WSAESTALE = 10070
WSAEREMOTE = 10071
WSASYSNOTREADY = 10091
WSAVERNOTSUPPORTED = 10092
WSANOTINITIALISED = 10093
WSAEDISCON = 10101
WSAENOMORE = 10102
WSAECANCELLED = 10103
WSAEINVALIDPROCTABLE = 10104
WSAEINVALIDPROVIDER = 10105
WSAEPROVIDERFAILEDINIT = 10106
WSASYSCALLFAILURE = 10107
WSASERVICE_NOT_FOUND = 10108
WSATYPE_NOT_FOUND = 10109
WSA_E_NO_MORE = 10110
WSA_E_CANCELLED = 10111
WSAEREFUSED = 10112
WSAHOST_NOT_FOUND = 11001
WSATRY_AGAIN = 11002
WSANO_RECOVERY = 11003
WSANO_DATA = 11004
WSA_QOS_RECEIVERS = 11005
WSA_QOS_SENDERS = 11006
WSA_QOS_NO_SENDERS = 11007
WSA_QOS_NO_RECEIVERS = 11008
WSA_QOS_REQUEST_CONFIRMED = 11009
WSA_QOS_ADMISSION_FAILURE = 11010
WSA_QOS_POLICY_FAILURE = 11011
WSA_QOS_BAD_STYLE = 11012
WSA_QOS_BAD_OBJECT = 11013
WSA_QOS_TRAFFIC_CTRL_ERROR = 11014
WSA_QOS_GENERIC_ERROR = 11015
WSA_QOS_ESERVICETYPE = 11016
WSA_QOS_EFLOWSPEC = 11017
WSA_QOS_EPROVSPECBUF = 11018
WSA_QOS_EFILTERSTYLE = 11019
WSA_QOS_EFILTERTYPE = 11020
WSA_QOS_EFILTERCOUNT = 11021
WSA_QOS_EOBJLENGTH = 11022
WSA_QOS_EFLOWCOUNT = 11023
WSA_QOS_EUNKOWNPSOBJ = 11024
WSA_QOS_EPOLICYOBJ = 11025
WSA_QOS_EFLOWDESC = 11026
WSA_QOS_EPSFLOWSPEC = 11027
WSA_QOS_EPSFILTERSPEC = 11028
WSA_QOS_ESDMODEOBJ = 11029
WSA_QOS_ESHAPERATEOBJ = 11030
WSA_QOS_RESERVED_PETYPE = 11031
WSA_SECURE_HOST_NOT_FOUND = 11032
WSA_IPSEC_NAME_POLICY_ERROR = 11033

SEVERITY_SUCCESS = 0
SEVERITY_ERROR = 1
FACILITY_NT_BIT = 0x10000000

NOERROR = 0
E_UNEXPECTED = 0x8000FFFF
E_NOTIMPL = 0x80004001
E_OUTOFMEMORY = 0x8007000E
E_INVALIDARG = 0x80070057
E_NOINTERFACE = 0x80004002
E_POINTER = 0x80004003
E_HANDLE = 0x80070006
E_ABORT = 0x80004004
E_FAIL = 0x80004005
E_ACCESSDENIED = 0x80070005
E_PENDING = 0x8000000A
E_BOUNDS = 0x8000000B
E_CHANGED_STATE = 0x8000000C
E_ILLEGAL_STATE_CHANGE = 0x8000000D
E_ILLEGAL_METHOD_CALL = 0x8000000E
RO_E_METADATA_NAME_NOT_FOUND = 0x8000000F
RO_E_METADATA_NAME_IS_NAMESPACE = 0x80000010
RO_E_METADATA_INVALID_TYPE_FORMAT = 0x80000011
RO_E_INVALID_METADATA_FILE = 0x80000012
RO_E_CLOSED = 0x80000013
RO_E_EXCLUSIVE_WRITE = 0x80000014
RO_E_CHANGE_NOTIFICATION_IN_PROGRESS = 0x80000015
RO_E_ERROR_STRING_NOT_FOUND = 0x80000016
E_STRING_NOT_NULL_TERMINATED = 0x80000017
E_ILLEGAL_DELEGATE_ASSIGNMENT = 0x80000018
E_ASYNC_OPERATION_NOT_STARTED = 0x80000019
E_APPLICATION_EXITING = 0x8000001A
E_APPLICATION_VIEW_EXITING = 0x8000001B
RO_E_MUST_BE_AGILE = 0x8000001C
RO_E_UNSUPPORTED_FROM_MTA = 0x8000001D
RO_E_COMMITTED = 0x8000001E
RO_E_BLOCKED_CROSS_ASTA_CALL = 0x8000001F
RO_E_CANNOT_ACTIVATE_FULL_TRUST_SERVER = 0x80000020
RO_E_CANNOT_ACTIVATE_UNIVERSAL_APPLICATION_SERVER = 0x80000021
CO_E_INIT_TLS = 0x80004006
CO_E_INIT_SHARED_ALLOCATOR = 0x80004007
CO_E_INIT_MEMORY_ALLOCATOR = 0x80004008
CO_E_INIT_CLASS_CACHE = 0x80004009
CO_E_INIT_RPC_CHANNEL = 0x8000400A
CO_E_INIT_TLS_SET_CHANNEL_CONTROL = 0x8000400B
CO_E_INIT_TLS_CHANNEL_CONTROL = 0x8000400C
CO_E_INIT_UNACCEPTED_USER_ALLOCATOR = 0x8000400D
CO_E_INIT_SCM_MUTEX_EXISTS = 0x8000400E
CO_E_INIT_SCM_FILE_MAPPING_EXISTS = 0x8000400F
CO_E_INIT_SCM_MAP_VIEW_OF_FILE = 0x80004010
CO_E_INIT_SCM_EXEC_FAILURE = 0x80004011
CO_E_INIT_ONLY_SINGLE_THREADED = 0x80004012
CO_E_CANT_REMOTE = 0x80004013
CO_E_BAD_SERVER_NAME = 0x80004014
CO_E_WRONG_SERVER_IDENTITY = 0x80004015
CO_E_OLE1DDE_DISABLED = 0x80004016
CO_E_RUNAS_SYNTAX = 0x80004017
CO_E_CREATEPROCESS_FAILURE = 0x80004018
CO_E_RUNAS_CREATEPROCESS_FAILURE = 0x80004019
CO_E_RUNAS_LOGON_FAILURE = 0x8000401A
CO_E_LAUNCH_PERMSSION_DENIED = 0x8000401B
CO_E_START_SERVICE_FAILURE = 0x8000401C
CO_E_REMOTE_COMMUNICATION_FAILURE = 0x8000401D
CO_E_SERVER_START_TIMEOUT = 0x8000401E
CO_E_CLSREG_INCONSISTENT = 0x8000401F
CO_E_IIDREG_INCONSISTENT = 0x80004020
CO_E_NOT_SUPPORTED = 0x80004021
CO_E_RELOAD_DLL = 0x80004022
CO_E_MSI_ERROR = 0x80004023
CO_E_ATTEMPT_TO_CREATE_OUTSIDE_CLIENT_CONTEXT = 0x80004024
CO_E_SERVER_PAUSED = 0x80004025
CO_E_SERVER_NOT_PAUSED = 0x80004026
CO_E_CLASS_DISABLED = 0x80004027
CO_E_CLRNOTAVAILABLE = 0x80004028
CO_E_ASYNC_WORK_REJECTED = 0x80004029
CO_E_SERVER_INIT_TIMEOUT = 0x8000402A
CO_E_NO_SECCTX_IN_ACTIVATE = 0x8000402B
CO_E_TRACKER_CONFIG = 0x80004030
CO_E_THREADPOOL_CONFIG = 0x80004031
CO_E_SXS_CONFIG = 0x80004032
CO_E_MALFORMED_SPN = 0x80004033
CO_E_UNREVOKED_REGISTRATION_ON_APARTMENT_SHUTDOWN = 0x80004034
CO_E_PREMATURE_STUB_RUNDOWN = 0x80004035
S_OK = 0
S_FALSE = 1
OLE_E_FIRST = 0x80040000
OLE_E_LAST = 0x800400FF
OLE_S_FIRST = 0x00040000
OLE_S_LAST = 0x000400FF
OLE_E_OLEVERB = 0x80040000
OLE_E_ADVF = 0x80040001
OLE_E_ENUM_NOMORE = 0x80040002
OLE_E_ADVISENOTSUPPORTED = 0x80040003
OLE_E_NOCONNECTION = 0x80040004
OLE_E_NOTRUNNING = 0x80040005
OLE_E_NOCACHE = 0x80040006
OLE_E_BLANK = 0x80040007
OLE_E_CLASSDIFF = 0x80040008
OLE_E_CANT_GETMONIKER = 0x80040009
OLE_E_CANT_BINDTOSOURCE = 0x8004000A
OLE_E_STATIC = 0x8004000B
OLE_E_PROMPTSAVECANCELLED = 0x8004000C
OLE_E_INVALIDRECT = 0x8004000D
OLE_E_WRONGCOMPOBJ = 0x8004000E
OLE_E_INVALIDHWND = 0x8004000F
OLE_E_NOT_INPLACEACTIVE = 0x80040010
OLE_E_CANTCONVERT = 0x80040011
OLE_E_NOSTORAGE = 0x80040012
DV_E_FORMATETC = 0x80040064
DV_E_DVTARGETDEVICE = 0x80040065
DV_E_STGMEDIUM = 0x80040066
DV_E_STATDATA = 0x80040067
DV_E_LINDEX = 0x80040068
DV_E_TYMED = 0x80040069
DV_E_CLIPFORMAT = 0x8004006A
DV_E_DVASPECT = 0x8004006B
DV_E_DVTARGETDEVICE_SIZE = 0x8004006C
DV_E_NOIVIEWOBJECT = 0x8004006D
DRAGDROP_E_FIRST = 0x80040100
DRAGDROP_E_LAST = 0x8004010F
DRAGDROP_S_FIRST = 0x00040100
DRAGDROP_S_LAST = 0x0004010F
DRAGDROP_E_NOTREGISTERED = 0x80040100
DRAGDROP_E_ALREADYREGISTERED = 0x80040101
DRAGDROP_E_INVALIDHWND = 0x80040102
DRAGDROP_E_CONCURRENT_DRAG_ATTEMPTED = 0x80040103
CLASSFACTORY_E_FIRST = 0x80040110
CLASSFACTORY_E_LAST = 0x8004011F
CLASSFACTORY_S_FIRST = 0x00040110
CLASSFACTORY_S_LAST = 0x0004011F
CLASS_E_NOAGGREGATION = 0x80040110
CLASS_E_CLASSNOTAVAILABLE = 0x80040111
CLASS_E_NOTLICENSED = 0x80040112
MARSHAL_E_FIRST = 0x80040120
MARSHAL_E_LAST = 0x8004012F
MARSHAL_S_FIRST = 0x00040120
MARSHAL_S_LAST = 0x0004012F
DATA_E_FIRST = 0x80040130
DATA_E_LAST = 0x8004013F
DATA_S_FIRST = 0x00040130
DATA_S_LAST = 0x0004013F
VIEW_E_FIRST = 0x80040140
VIEW_E_LAST = 0x8004014F
VIEW_S_FIRST = 0x00040140
VIEW_S_LAST = 0x0004014F
VIEW_E_DRAW = 0x80040140
REGDB_E_FIRST = 0x80040150
REGDB_E_LAST = 0x8004015F
REGDB_S_FIRST = 0x00040150
REGDB_S_LAST = 0x0004015F
REGDB_E_READREGDB = 0x80040150
REGDB_E_WRITEREGDB = 0x80040151
REGDB_E_KEYMISSING = 0x80040152
REGDB_E_INVALIDVALUE = 0x80040153
REGDB_E_CLASSNOTREG = 0x80040154
REGDB_E_IIDNOTREG = 0x80040155
REGDB_E_BADTHREADINGMODEL = 0x80040156
REGDB_E_PACKAGEPOLICYVIOLATION = 0x80040157
CAT_E_FIRST = 0x80040160
CAT_E_LAST = 0x80040161
CAT_E_CATIDNOEXIST = 0x80040160
CAT_E_NODESCRIPTION = 0x80040161
CS_E_FIRST = 0x80040164
CS_E_LAST = 0x8004016F
CS_E_PACKAGE_NOTFOUND = 0x80040164
CS_E_NOT_DELETABLE = 0x80040165
CS_E_CLASS_NOTFOUND = 0x80040166
CS_E_INVALID_VERSION = 0x80040167
CS_E_NO_CLASSSTORE = 0x80040168
CS_E_OBJECT_NOTFOUND = 0x80040169
CS_E_OBJECT_ALREADY_EXISTS = 0x8004016A
CS_E_INVALID_PATH = 0x8004016B
CS_E_NETWORK_ERROR = 0x8004016C
CS_E_ADMIN_LIMIT_EXCEEDED = 0x8004016D
CS_E_SCHEMA_MISMATCH = 0x8004016E
CS_E_INTERNAL_ERROR = 0x8004016F
CACHE_E_FIRST = 0x80040170
CACHE_E_LAST = 0x8004017F
CACHE_S_FIRST = 0x00040170
CACHE_S_LAST = 0x0004017F
CACHE_E_NOCACHE_UPDATED = 0x80040170
OLEOBJ_E_FIRST = 0x80040180
OLEOBJ_E_LAST = 0x8004018F
OLEOBJ_S_FIRST = 0x00040180
OLEOBJ_S_LAST = 0x0004018F
OLEOBJ_E_NOVERBS = 0x80040180
OLEOBJ_E_INVALIDVERB = 0x80040181
CLIENTSITE_E_FIRST = 0x80040190
CLIENTSITE_E_LAST = 0x8004019F
CLIENTSITE_S_FIRST = 0x00040190
CLIENTSITE_S_LAST = 0x0004019F
INPLACE_E_NOTUNDOABLE = 0x800401A0
INPLACE_E_NOTOOLSPACE = 0x800401A1
INPLACE_E_FIRST = 0x800401A0
INPLACE_E_LAST = 0x800401AF
INPLACE_S_FIRST = 0x000401A0
INPLACE_S_LAST = 0x000401AF
ENUM_E_FIRST = 0x800401B0
ENUM_E_LAST = 0x800401BF
ENUM_S_FIRST = 0x000401B0
ENUM_S_LAST = 0x000401BF
CONVERT10_E_FIRST = 0x800401C0
CONVERT10_E_LAST = 0x800401CF
CONVERT10_S_FIRST = 0x000401C0
CONVERT10_S_LAST = 0x000401CF
CONVERT10_E_OLESTREAM_GET = 0x800401C0
CONVERT10_E_OLESTREAM_PUT = 0x800401C1
CONVERT10_E_OLESTREAM_FMT = 0x800401C2
CONVERT10_E_OLESTREAM_BITMAP_TO_DIB = 0x800401C3
CONVERT10_E_STG_FMT = 0x800401C4
CONVERT10_E_STG_NO_STD_STREAM = 0x800401C5
CONVERT10_E_STG_DIB_TO_BITMAP = 0x800401C6
CLIPBRD_E_FIRST = 0x800401D0
CLIPBRD_E_LAST = 0x800401DF
CLIPBRD_S_FIRST = 0x000401D0
CLIPBRD_S_LAST = 0x000401DF
CLIPBRD_E_CANT_OPEN = 0x800401D0
CLIPBRD_E_CANT_EMPTY = 0x800401D1
CLIPBRD_E_CANT_SET = 0x800401D2
CLIPBRD_E_BAD_DATA = 0x800401D3
CLIPBRD_E_CANT_CLOSE = 0x800401D4
MK_E_FIRST = 0x800401E0
MK_E_LAST = 0x800401EF
MK_S_FIRST = 0x000401E0
MK_S_LAST = 0x000401EF
MK_E_CONNECTMANUALLY = 0x800401E0
MK_E_EXCEEDEDDEADLINE = 0x800401E1
MK_E_NEEDGENERIC = 0x800401E2
MK_E_UNAVAILABLE = 0x800401E3
MK_E_SYNTAX = 0x800401E4
MK_E_NOOBJECT = 0x800401E5
MK_E_INVALIDEXTENSION = 0x800401E6
MK_E_INTERMEDIATEINTERFACENOTSUPPORTED = 0x800401E7
MK_E_NOTBINDABLE = 0x800401E8
MK_E_NOTBOUND = 0x800401E9
MK_E_CANTOPENFILE = 0x800401EA
MK_E_MUSTBOTHERUSER = 0x800401EB
MK_E_NOINVERSE = 0x800401EC
MK_E_NOSTORAGE = 0x800401ED
MK_E_NOPREFIX = 0x800401EE
MK_E_ENUMERATION_FAILED = 0x800401EF
CO_E_FIRST = 0x800401F0
CO_E_LAST = 0x800401FF
CO_S_FIRST = 0x000401F0
CO_S_LAST = 0x000401FF
CO_E_NOTINITIALIZED = 0x800401F0
CO_E_ALREADYINITIALIZED = 0x800401F1
CO_E_CANTDETERMINECLASS = 0x800401F2
CO_E_CLASSSTRING = 0x800401F3
CO_E_IIDSTRING = 0x800401F4
CO_E_APPNOTFOUND = 0x800401F5
CO_E_APPSINGLEUSE = 0x800401F6
CO_E_ERRORINAPP = 0x800401F7
CO_E_DLLNOTFOUND = 0x800401F8
CO_E_ERRORINDLL = 0x800401F9
CO_E_WRONGOSFORAPP = 0x800401FA
CO_E_OBJNOTREG = 0x800401FB
CO_E_OBJISREG = 0x800401FC
CO_E_OBJNOTCONNECTED = 0x800401FD
CO_E_APPDIDNTREG = 0x800401FE
CO_E_RELEASED = 0x800401FF
EVENT_E_FIRST = 0x80040200
EVENT_E_LAST = 0x8004021F
EVENT_S_FIRST = 0x00040200
EVENT_S_LAST = 0x0004021F
EVENT_S_SOME_SUBSCRIBERS_FAILED = 0x00040200
EVENT_E_ALL_SUBSCRIBERS_FAILED = 0x80040201
EVENT_S_NOSUBSCRIBERS = 0x00040202
EVENT_E_QUERYSYNTAX = 0x80040203
EVENT_E_QUERYFIELD = 0x80040204
EVENT_E_INTERNALEXCEPTION = 0x80040205
EVENT_E_INTERNALERROR = 0x80040206
EVENT_E_INVALID_PER_USER_SID = 0x80040207
EVENT_E_USER_EXCEPTION = 0x80040208
EVENT_E_TOO_MANY_METHODS = 0x80040209
EVENT_E_MISSING_EVENTCLASS = 0x8004020A
EVENT_E_NOT_ALL_REMOVED = 0x8004020B
EVENT_E_COMPLUS_NOT_INSTALLED = 0x8004020C
EVENT_E_CANT_MODIFY_OR_DELETE_UNCONFIGURED_OBJECT = 0x8004020D
EVENT_E_CANT_MODIFY_OR_DELETE_CONFIGURED_OBJECT = 0x8004020E
EVENT_E_INVALID_EVENT_CLASS_PARTITION = 0x8004020F
EVENT_E_PER_USER_SID_NOT_LOGGED_ON = 0x80040210
TPC_E_INVALID_PROPERTY = 0x80040241
TPC_E_NO_DEFAULT_TABLET = 0x80040212
TPC_E_UNKNOWN_PROPERTY = 0x8004021B
TPC_E_INVALID_INPUT_RECT = 0x80040219
TPC_E_INVALID_STROKE = 0x80040222
TPC_E_INITIALIZE_FAIL = 0x80040223
TPC_E_NOT_RELEVANT = 0x80040232
TPC_E_INVALID_PACKET_DESCRIPTION = 0x80040233
TPC_E_RECOGNIZER_NOT_REGISTERED = 0x80040235
TPC_E_INVALID_RIGHTS = 0x80040236
TPC_E_OUT_OF_ORDER_CALL = 0x80040237
TPC_E_QUEUE_FULL = 0x80040238
TPC_E_INVALID_CONFIGURATION = 0x80040239
TPC_E_INVALID_DATA_FROM_RECOGNIZER = 0x8004023A
TPC_S_TRUNCATED = 0x00040252
TPC_S_INTERRUPTED = 0x00040253
TPC_S_NO_DATA_TO_PROCESS = 0x00040254
XACT_E_FIRST = 0x8004D000
XACT_E_LAST = 0x8004D02B
XACT_S_FIRST = 0x0004D000
XACT_S_LAST = 0x0004D010
XACT_E_ALREADYOTHERSINGLEPHASE = 0x8004D000
XACT_E_CANTRETAIN = 0x8004D001
XACT_E_COMMITFAILED = 0x8004D002
XACT_E_COMMITPREVENTED = 0x8004D003
XACT_E_HEURISTICABORT = 0x8004D004
XACT_E_HEURISTICCOMMIT = 0x8004D005
XACT_E_HEURISTICDAMAGE = 0x8004D006
XACT_E_HEURISTICDANGER = 0x8004D007
XACT_E_ISOLATIONLEVEL = 0x8004D008
XACT_E_NOASYNC = 0x8004D009
XACT_E_NOENLIST = 0x8004D00A
XACT_E_NOISORETAIN = 0x8004D00B
XACT_E_NORESOURCE = 0x8004D00C
XACT_E_NOTCURRENT = 0x8004D00D
XACT_E_NOTRANSACTION = 0x8004D00E
XACT_E_NOTSUPPORTED = 0x8004D00F
XACT_E_UNKNOWNRMGRID = 0x8004D010
XACT_E_WRONGSTATE = 0x8004D011
XACT_E_WRONGUOW = 0x8004D012
XACT_E_XTIONEXISTS = 0x8004D013
XACT_E_NOIMPORTOBJECT = 0x8004D014
XACT_E_INVALIDCOOKIE = 0x8004D015
XACT_E_INDOUBT = 0x8004D016
XACT_E_NOTIMEOUT = 0x8004D017
XACT_E_ALREADYINPROGRESS = 0x8004D018
XACT_E_ABORTED = 0x8004D019
XACT_E_LOGFULL = 0x8004D01A
XACT_E_TMNOTAVAILABLE = 0x8004D01B
XACT_E_CONNECTION_DOWN = 0x8004D01C
XACT_E_CONNECTION_DENIED = 0x8004D01D
XACT_E_REENLISTTIMEOUT = 0x8004D01E
XACT_E_TIP_CONNECT_FAILED = 0x8004D01F
XACT_E_TIP_PROTOCOL_ERROR = 0x8004D020
XACT_E_TIP_PULL_FAILED = 0x8004D021
XACT_E_DEST_TMNOTAVAILABLE = 0x8004D022
XACT_E_TIP_DISABLED = 0x8004D023
XACT_E_NETWORK_TX_DISABLED = 0x8004D024
XACT_E_PARTNER_NETWORK_TX_DISABLED = 0x8004D025
XACT_E_XA_TX_DISABLED = 0x8004D026
XACT_E_UNABLE_TO_READ_DTC_CONFIG = 0x8004D027
XACT_E_UNABLE_TO_LOAD_DTC_PROXY = 0x8004D028
XACT_E_ABORTING = 0x8004D029
XACT_E_PUSH_COMM_FAILURE = 0x8004D02A
XACT_E_PULL_COMM_FAILURE = 0x8004D02B
XACT_E_LU_TX_DISABLED = 0x8004D02C
XACT_E_CLERKNOTFOUND = 0x8004D080
XACT_E_CLERKEXISTS = 0x8004D081
XACT_E_RECOVERYINPROGRESS = 0x8004D082
XACT_E_TRANSACTIONCLOSED = 0x8004D083
XACT_E_INVALIDLSN = 0x8004D084
XACT_E_REPLAYREQUEST = 0x8004D085
XACT_S_ASYNC = 0x0004D000
XACT_S_DEFECT = 0x0004D001
XACT_S_READONLY = 0x0004D002
XACT_S_SOMENORETAIN = 0x0004D003
XACT_S_OKINFORM = 0x0004D004
XACT_S_MADECHANGESCONTENT = 0x0004D005
XACT_S_MADECHANGESINFORM = 0x0004D006
XACT_S_ALLNORETAIN = 0x0004D007
XACT_S_ABORTING = 0x0004D008
XACT_S_SINGLEPHASE = 0x0004D009
XACT_S_LOCALLY_OK = 0x0004D00A
XACT_S_LASTRESOURCEMANAGER = 0x0004D010
CONTEXT_E_FIRST = 0x8004E000
CONTEXT_E_LAST = 0x8004E02F
CONTEXT_S_FIRST = 0x0004E000
CONTEXT_S_LAST = 0x0004E02F
CONTEXT_E_ABORTED = 0x8004E002
CONTEXT_E_ABORTING = 0x8004E003
CONTEXT_E_NOCONTEXT = 0x8004E004
CONTEXT_E_WOULD_DEADLOCK = 0x8004E005
CONTEXT_E_SYNCH_TIMEOUT = 0x8004E006
CONTEXT_E_OLDREF = 0x8004E007
CONTEXT_E_ROLENOTFOUND = 0x8004E00C
CONTEXT_E_TMNOTAVAILABLE = 0x8004E00F
CO_E_ACTIVATIONFAILED = 0x8004E021
CO_E_ACTIVATIONFAILED_EVENTLOGGED = 0x8004E022
CO_E_ACTIVATIONFAILED_CATALOGERROR = 0x8004E023
CO_E_ACTIVATIONFAILED_TIMEOUT = 0x8004E024
CO_E_INITIALIZATIONFAILED = 0x8004E025
CONTEXT_E_NOJIT = 0x8004E026
CONTEXT_E_NOTRANSACTION = 0x8004E027
CO_E_THREADINGMODEL_CHANGED = 0x8004E028
CO_E_NOIISINTRINSICS = 0x8004E029
CO_E_NOCOOKIES = 0x8004E02A
CO_E_DBERROR = 0x8004E02B
CO_E_NOTPOOLED = 0x8004E02C
CO_E_NOTCONSTRUCTED = 0x8004E02D
CO_E_NOSYNCHRONIZATION = 0x8004E02E
CO_E_ISOLEVELMISMATCH = 0x8004E02F
CO_E_CALL_OUT_OF_TX_SCOPE_NOT_ALLOWED = 0x8004E030
CO_E_EXIT_TRANSACTION_SCOPE_NOT_CALLED = 0x8004E031
OLE_S_USEREG = 0x00040000
OLE_S_STATIC = 0x00040001
OLE_S_MAC_CLIPFORMAT = 0x00040002
DRAGDROP_S_DROP = 0x00040100
DRAGDROP_S_CANCEL = 0x00040101
DRAGDROP_S_USEDEFAULTCURSORS = 0x00040102
DATA_S_SAMEFORMATETC = 0x00040130
VIEW_S_ALREADY_FROZEN = 0x00040140
CACHE_S_FORMATETC_NOTSUPPORTED = 0x00040170
CACHE_S_SAMECACHE = 0x00040171
CACHE_S_SOMECACHES_NOTUPDATED = 0x00040172
OLEOBJ_S_INVALIDVERB = 0x00040180
OLEOBJ_S_CANNOT_DOVERB_NOW = 0x00040181
OLEOBJ_S_INVALIDHWND = 0x00040182
INPLACE_S_TRUNCATED = 0x000401A0
CONVERT10_S_NO_PRESENTATION = 0x000401C0
MK_S_REDUCED_TO_SELF = 0x000401E2
MK_S_ME = 0x000401E4
MK_S_HIM = 0x000401E5
MK_S_US = 0x000401E6
MK_S_MONIKERALREADYREGISTERED = 0x000401E7
SCHED_S_TASK_READY = 0x00041300
SCHED_S_TASK_RUNNING = 0x00041301
SCHED_S_TASK_DISABLED = 0x00041302
SCHED_S_TASK_HAS_NOT_RUN = 0x00041303
SCHED_S_TASK_NO_MORE_RUNS = 0x00041304
SCHED_S_TASK_NOT_SCHEDULED = 0x00041305
SCHED_S_TASK_TERMINATED = 0x00041306
SCHED_S_TASK_NO_VALID_TRIGGERS = 0x00041307
SCHED_S_EVENT_TRIGGER = 0x00041308
SCHED_E_TRIGGER_NOT_FOUND = 0x80041309
SCHED_E_TASK_NOT_READY = 0x8004130A
SCHED_E_TASK_NOT_RUNNING = 0x8004130B
SCHED_E_SERVICE_NOT_INSTALLED = 0x8004130C
SCHED_E_CANNOT_OPEN_TASK = 0x8004130D
SCHED_E_INVALID_TASK = 0x8004130E
SCHED_E_ACCOUNT_INFORMATION_NOT_SET = 0x8004130F
SCHED_E_ACCOUNT_NAME_NOT_FOUND = 0x80041310
SCHED_E_ACCOUNT_DBASE_CORRUPT = 0x80041311
SCHED_E_NO_SECURITY_SERVICES = 0x80041312
SCHED_E_UNKNOWN_OBJECT_VERSION = 0x80041313
SCHED_E_UNSUPPORTED_ACCOUNT_OPTION = 0x80041314
SCHED_E_SERVICE_NOT_RUNNING = 0x80041315
SCHED_E_UNEXPECTEDNODE = 0x80041316
SCHED_E_NAMESPACE = 0x80041317
SCHED_E_INVALIDVALUE = 0x80041318
SCHED_E_MISSINGNODE = 0x80041319
SCHED_E_MALFORMEDXML = 0x8004131A
SCHED_S_SOME_TRIGGERS_FAILED = 0x0004131B
SCHED_S_BATCH_LOGON_PROBLEM = 0x0004131C
SCHED_E_TOO_MANY_NODES = 0x8004131D
SCHED_E_PAST_END_BOUNDARY = 0x8004131E
SCHED_E_ALREADY_RUNNING = 0x8004131F
SCHED_E_USER_NOT_LOGGED_ON = 0x80041320
SCHED_E_INVALID_TASK_HASH = 0x80041321
SCHED_E_SERVICE_NOT_AVAILABLE = 0x80041322
SCHED_E_SERVICE_TOO_BUSY = 0x80041323
SCHED_E_TASK_ATTEMPTED = 0x80041324
SCHED_S_TASK_QUEUED = 0x00041325
SCHED_E_TASK_DISABLED = 0x80041326
SCHED_E_TASK_NOT_V1_COMPAT = 0x80041327
SCHED_E_START_ON_DEMAND = 0x80041328
SCHED_E_TASK_NOT_UBPM_COMPAT = 0x80041329
SCHED_E_DEPRECATED_FEATURE_USED = 0x80041330
CO_E_CLASS_CREATE_FAILED = 0x80080001
CO_E_SCM_ERROR = 0x80080002
CO_E_SCM_RPC_FAILURE = 0x80080003
CO_E_BAD_PATH = 0x80080004
CO_E_SERVER_EXEC_FAILURE = 0x80080005
CO_E_OBJSRV_RPC_FAILURE = 0x80080006
MK_E_NO_NORMALIZED = 0x80080007
CO_E_SERVER_STOPPING = 0x80080008
MEM_E_INVALID_ROOT = 0x80080009
MEM_E_INVALID_LINK = 0x80080010
MEM_E_INVALID_SIZE = 0x80080011
CO_S_NOTALLINTERFACES = 0x00080012
CO_S_MACHINENAMENOTFOUND = 0x00080013
CO_E_MISSING_DISPLAYNAME = 0x80080015
CO_E_RUNAS_VALUE_MUST_BE_AAA = 0x80080016
CO_E_ELEVATION_DISABLED = 0x80080017
APPX_E_PACKAGING_INTERNAL = 0x80080200
APPX_E_INTERLEAVING_NOT_ALLOWED = 0x80080201
APPX_E_RELATIONSHIPS_NOT_ALLOWED = 0x80080202
APPX_E_MISSING_REQUIRED_FILE = 0x80080203
APPX_E_INVALID_MANIFEST = 0x80080204
APPX_E_INVALID_BLOCKMAP = 0x80080205
APPX_E_CORRUPT_CONTENT = 0x80080206
APPX_E_BLOCK_HASH_INVALID = 0x80080207
APPX_E_REQUESTED_RANGE_TOO_LARGE = 0x80080208
APPX_E_INVALID_SIP_CLIENT_DATA = 0x80080209
APPX_E_INVALID_KEY_INFO = 0x8008020A
APPX_E_INVALID_CONTENTGROUPMAP = 0x8008020B
APPX_E_INVALID_APPINSTALLER = 0x8008020C
APPX_E_DELTA_BASELINE_VERSION_MISMATCH = 0x8008020D
APPX_E_DELTA_PACKAGE_MISSING_FILE = 0x8008020E
APPX_E_INVALID_DELTA_PACKAGE = 0x8008020F
APPX_E_DELTA_APPENDED_PACKAGE_NOT_ALLOWED = 0x80080210
APPX_E_INVALID_PACKAGING_LAYOUT = 0x80080211
APPX_E_INVALID_PACKAGESIGNCONFIG = 0x80080212
APPX_E_RESOURCESPRI_NOT_ALLOWED = 0x80080213
APPX_E_FILE_COMPRESSION_MISMATCH = 0x80080214
APPX_E_INVALID_PAYLOAD_PACKAGE_EXTENSION = 0x80080215
APPX_E_INVALID_ENCRYPTION_EXCLUSION_FILE_LIST = 0x80080216
BT_E_SPURIOUS_ACTIVATION = 0x80080300
DISP_E_UNKNOWNINTERFACE = 0x80020001
DISP_E_MEMBERNOTFOUND = 0x80020003
DISP_E_PARAMNOTFOUND = 0x80020004
DISP_E_TYPEMISMATCH = 0x80020005
DISP_E_UNKNOWNNAME = 0x80020006
DISP_E_NONAMEDARGS = 0x80020007
DISP_E_BADVARTYPE = 0x80020008
DISP_E_EXCEPTION = 0x80020009
DISP_E_OVERFLOW = 0x8002000A
DISP_E_BADINDEX = 0x8002000B
DISP_E_UNKNOWNLCID = 0x8002000C
DISP_E_ARRAYISLOCKED = 0x8002000D
DISP_E_BADPARAMCOUNT = 0x8002000E
DISP_E_PARAMNOTOPTIONAL = 0x8002000F
DISP_E_BADCALLEE = 0x80020010
DISP_E_NOTACOLLECTION = 0x80020011
DISP_E_DIVBYZERO = 0x80020012
DISP_E_BUFFERTOOSMALL = 0x80020013
TYPE_E_BUFFERTOOSMALL = 0x80028016
TYPE_E_FIELDNOTFOUND = 0x80028017
TYPE_E_INVDATAREAD = 0x80028018
TYPE_E_UNSUPFORMAT = 0x80028019
TYPE_E_REGISTRYACCESS = 0x8002801C
TYPE_E_LIBNOTREGISTERED = 0x8002801D
TYPE_E_UNDEFINEDTYPE = 0x80028027
TYPE_E_QUALIFIEDNAMEDISALLOWED = 0x80028028
TYPE_E_INVALIDSTATE = 0x80028029
TYPE_E_WRONGTYPEKIND = 0x8002802A
TYPE_E_ELEMENTNOTFOUND = 0x8002802B
TYPE_E_AMBIGUOUSNAME = 0x8002802C
TYPE_E_NAMECONFLICT = 0x8002802D
TYPE_E_UNKNOWNLCID = 0x8002802E
TYPE_E_DLLFUNCTIONNOTFOUND = 0x8002802F
TYPE_E_BADMODULEKIND = 0x800288BD
TYPE_E_SIZETOOBIG = 0x800288C5
TYPE_E_DUPLICATEID = 0x800288C6
TYPE_E_INVALIDID = 0x800288CF
TYPE_E_TYPEMISMATCH = 0x80028CA0
TYPE_E_OUTOFBOUNDS = 0x80028CA1
TYPE_E_IOERROR = 0x80028CA2
TYPE_E_CANTCREATETMPFILE = 0x80028CA3
TYPE_E_CANTLOADLIBRARY = 0x80029C4A
TYPE_E_INCONSISTENTPROPFUNCS = 0x80029C83
TYPE_E_CIRCULARTYPE = 0x80029C84
STG_E_INVALIDFUNCTION = 0x80030001
STG_E_FILENOTFOUND = 0x80030002
STG_E_PATHNOTFOUND = 0x80030003
STG_E_TOOMANYOPENFILES = 0x80030004
STG_E_ACCESSDENIED = 0x80030005
STG_E_INVALIDHANDLE = 0x80030006
STG_E_INSUFFICIENTMEMORY = 0x80030008
STG_E_INVALIDPOINTER = 0x80030009
STG_E_NOMOREFILES = 0x80030012
STG_E_DISKISWRITEPROTECTED = 0x80030013
STG_E_SEEKERROR = 0x80030019
STG_E_WRITEFAULT = 0x8003001D
STG_E_READFAULT = 0x8003001E
STG_E_SHAREVIOLATION = 0x80030020
STG_E_LOCKVIOLATION = 0x80030021
STG_E_FILEALREADYEXISTS = 0x80030050
STG_E_INVALIDPARAMETER = 0x80030057
STG_E_MEDIUMFULL = 0x80030070
STG_E_PROPSETMISMATCHED = 0x800300F0
STG_E_ABNORMALAPIEXIT = 0x800300FA
STG_E_INVALIDHEADER = 0x800300FB
STG_E_INVALIDNAME = 0x800300FC
STG_E_UNKNOWN = 0x800300FD
STG_E_UNIMPLEMENTEDFUNCTION = 0x800300FE
STG_E_INVALIDFLAG = 0x800300FF
STG_E_INUSE = 0x80030100
STG_E_NOTCURRENT = 0x80030101
STG_E_REVERTED = 0x80030102
STG_E_CANTSAVE = 0x80030103
STG_E_OLDFORMAT = 0x80030104
STG_E_OLDDLL = 0x80030105
STG_E_SHAREREQUIRED = 0x80030106
STG_E_NOTFILEBASEDSTORAGE = 0x80030107
STG_E_EXTANTMARSHALLINGS = 0x80030108
STG_E_DOCFILECORRUPT = 0x80030109
STG_E_BADBASEADDRESS = 0x80030110
STG_E_DOCFILETOOLARGE = 0x80030111
STG_E_NOTSIMPLEFORMAT = 0x80030112
STG_E_INCOMPLETE = 0x80030201
STG_E_TERMINATED = 0x80030202
STG_S_CONVERTED = 0x00030200
STG_S_BLOCK = 0x00030201
STG_S_RETRYNOW = 0x00030202
STG_S_MONITORING = 0x00030203
STG_S_MULTIPLEOPENS = 0x00030204
STG_S_CONSOLIDATIONFAILED = 0x00030205
STG_S_CANNOTCONSOLIDATE = 0x00030206
STG_S_POWER_CYCLE_REQUIRED = 0x00030207
STG_E_FIRMWARE_SLOT_INVALID = 0x80030208
STG_E_FIRMWARE_IMAGE_INVALID = 0x80030209
STG_E_DEVICE_UNRESPONSIVE = 0x8003020A
STG_E_STATUS_COPY_PROTECTION_FAILURE = 0x80030305
STG_E_CSS_AUTHENTICATION_FAILURE = 0x80030306
STG_E_CSS_KEY_NOT_PRESENT = 0x80030307
STG_E_CSS_KEY_NOT_ESTABLISHED = 0x80030308
STG_E_CSS_SCRAMBLED_SECTOR = 0x80030309
STG_E_CSS_REGION_MISMATCH = 0x8003030A
STG_E_RESETS_EXHAUSTED = 0x8003030B
RPC_E_CALL_REJECTED = 0x80010001
RPC_E_CALL_CANCELED = 0x80010002
RPC_E_CANTPOST_INSENDCALL = 0x80010003
RPC_E_CANTCALLOUT_INASYNCCALL = 0x80010004
RPC_E_CANTCALLOUT_INEXTERNALCALL = 0x80010005
RPC_E_CONNECTION_TERMINATED = 0x80010006
RPC_E_SERVER_DIED = 0x80010007
RPC_E_CLIENT_DIED = 0x80010008
RPC_E_INVALID_DATAPACKET = 0x80010009
RPC_E_CANTTRANSMIT_CALL = 0x8001000A
RPC_E_CLIENT_CANTMARSHAL_DATA = 0x8001000B
RPC_E_CLIENT_CANTUNMARSHAL_DATA = 0x8001000C
RPC_E_SERVER_CANTMARSHAL_DATA = 0x8001000D
RPC_E_SERVER_CANTUNMARSHAL_DATA = 0x8001000E
RPC_E_INVALID_DATA = 0x8001000F
RPC_E_INVALID_PARAMETER = 0x80010010
RPC_E_CANTCALLOUT_AGAIN = 0x80010011
RPC_E_SERVER_DIED_DNE = 0x80010012
RPC_E_SYS_CALL_FAILED = 0x80010100
RPC_E_OUT_OF_RESOURCES = 0x80010101
RPC_E_ATTEMPTED_MULTITHREAD = 0x80010102
RPC_E_NOT_REGISTERED = 0x80010103
RPC_E_FAULT = 0x80010104
RPC_E_SERVERFAULT = 0x80010105
RPC_E_CHANGED_MODE = 0x80010106
RPC_E_INVALIDMETHOD = 0x80010107
RPC_E_DISCONNECTED = 0x80010108
RPC_E_RETRY = 0x80010109
RPC_E_SERVERCALL_RETRYLATER = 0x8001010A
RPC_E_SERVERCALL_REJECTED = 0x8001010B
RPC_E_INVALID_CALLDATA = 0x8001010C
RPC_E_CANTCALLOUT_ININPUTSYNCCALL = 0x8001010D
RPC_E_WRONG_THREAD = 0x8001010E
RPC_E_THREAD_NOT_INIT = 0x8001010F
RPC_E_VERSION_MISMATCH = 0x80010110
RPC_E_INVALID_HEADER = 0x80010111
RPC_E_INVALID_EXTENSION = 0x80010112
RPC_E_INVALID_IPID = 0x80010113
RPC_E_INVALID_OBJECT = 0x80010114
RPC_S_CALLPENDING = 0x80010115
RPC_S_WAITONTIMER = 0x80010116
RPC_E_CALL_COMPLETE = 0x80010117
RPC_E_UNSECURE_CALL = 0x80010118
RPC_E_TOO_LATE = 0x80010119
RPC_E_NO_GOOD_SECURITY_PACKAGES = 0x8001011A
RPC_E_ACCESS_DENIED = 0x8001011B
RPC_E_REMOTE_DISABLED = 0x8001011C
RPC_E_INVALID_OBJREF = 0x8001011D
RPC_E_NO_CONTEXT = 0x8001011E
RPC_E_TIMEOUT = 0x8001011F
RPC_E_NO_SYNC = 0x80010120
RPC_E_FULLSIC_REQUIRED = 0x80010121
RPC_E_INVALID_STD_NAME = 0x80010122
CO_E_FAILEDTOIMPERSONATE = 0x80010123
CO_E_FAILEDTOGETSECCTX = 0x80010124
CO_E_FAILEDTOOPENTHREADTOKEN = 0x80010125
CO_E_FAILEDTOGETTOKENINFO = 0x80010126
CO_E_TRUSTEEDOESNTMATCHCLIENT = 0x80010127
CO_E_FAILEDTOQUERYCLIENTBLANKET = 0x80010128
CO_E_FAILEDTOSETDACL = 0x80010129
CO_E_ACCESSCHECKFAILED = 0x8001012A
CO_E_NETACCESSAPIFAILED = 0x8001012B
CO_E_WRONGTRUSTEENAMESYNTAX = 0x8001012C
CO_E_INVALIDSID = 0x8001012D
CO_E_CONVERSIONFAILED = 0x8001012E
CO_E_NOMATCHINGSIDFOUND = 0x8001012F
CO_E_LOOKUPACCSIDFAILED = 0x80010130
CO_E_NOMATCHINGNAMEFOUND = 0x80010131
CO_E_LOOKUPACCNAMEFAILED = 0x80010132
CO_E_SETSERLHNDLFAILED = 0x80010133
CO_E_FAILEDTOGETWINDIR = 0x80010134
CO_E_PATHTOOLONG = 0x80010135
CO_E_FAILEDTOGENUUID = 0x80010136
CO_E_FAILEDTOCREATEFILE = 0x80010137
CO_E_FAILEDTOCLOSEHANDLE = 0x80010138
CO_E_EXCEEDSYSACLLIMIT = 0x80010139
CO_E_ACESINWRONGORDER = 0x8001013A
CO_E_INCOMPATIBLESTREAMVERSION = 0x8001013B
CO_E_FAILEDTOOPENPROCESSTOKEN = 0x8001013C
CO_E_DECODEFAILED = 0x8001013D
CO_E_ACNOTINITIALIZED = 0x8001013F
CO_E_CANCEL_DISABLED = 0x80010140
RPC_E_UNEXPECTED = 0x8001FFFF

SW_HIDE = 0
SW_SHOWNORMAL = 1
SW_NORMAL = 1
SW_SHOWMINIMIZED = 2
SW_SHOWMAXIMIZED = 3
SW_MAXIMIZE = 3
SW_SHOWNOACTIVATE = 4
SW_SHOW = 5
SW_MINIMIZE = 6
SW_SHOWMINNOACTIVE = 7
SW_SHOWNA = 8
SW_RESTORE = 9
SW_SHOWDEFAULT = 10
SW_FORCEMINIMIZE = 11
SW_MAX = 11
SW_PARENTCLOSING = 1
SW_OTHERZOOM = 2
SW_PARENTOPENING = 3
SW_OTHERUNZOOM = 4

HWND_BROADCAST = 65535
HWND_DESKTOP = 0
HWND_TOP = 0
HWND_BOTTOM = 1
HWND_TOPMOST = -1
HWND_NOTOPMOST = -2
HWND_MESSAGE = -3

SM_CXSCREEN = 0
SM_CYSCREEN = 1
SM_CXVSCROLL = 2
SM_CYHSCROLL = 3
SM_CYCAPTION = 4
SM_CXBORDER = 5
SM_CYBORDER = 6
SM_CXDLGFRAME = 7
SM_CYDLGFRAME = 8
SM_CYVTHUMB = 9
SM_CXHTHUMB = 10
SM_CXICON = 11
SM_CYICON = 12
SM_CXCURSOR = 13
SM_CYCURSOR = 14
SM_CYMENU = 15
SM_CXFULLSCREEN = 16
SM_CYFULLSCREEN = 17
SM_CYKANJIWINDOW = 18
SM_MOUSEPRESENT = 19
SM_CYVSCROLL = 20
SM_CXHSCROLL = 21
SM_DEBUG = 22
SM_SWAPBUTTON = 23
SM_RESERVED1 = 24
SM_RESERVED2 = 25
SM_RESERVED3 = 26
SM_RESERVED4 = 27
SM_CXMIN = 28
SM_CYMIN = 29
SM_CXSIZE = 30
SM_CYSIZE = 31
SM_CXFRAME = 32
SM_CYFRAME = 33
SM_CXMINTRACK = 34
SM_CYMINTRACK = 35
SM_CXDOUBLECLK = 36
SM_CYDOUBLECLK = 37
SM_CXICONSPACING = 38
SM_CYICONSPACING = 39
SM_MENUDROPALIGNMENT = 40
SM_PENWINDOWS = 41
SM_DBCSENABLED = 42
SM_CMOUSEBUTTONS = 43
SM_CXFIXEDFRAME = SM_CXDLGFRAME
SM_CYFIXEDFRAME = SM_CYDLGFRAME
SM_CXSIZEFRAME = SM_CXFRAME
SM_CYSIZEFRAME = SM_CYFRAME
SM_SECURE = 44
SM_CXEDGE = 45
SM_CYEDGE = 46
SM_CXMINSPACING = 47
SM_CYMINSPACING = 48
SM_CXSMICON = 49
SM_CYSMICON = 50
SM_CYSMCAPTION = 51
SM_CXSMSIZE = 52
SM_CYSMSIZE = 53
SM_CXMENUSIZE = 54
SM_CYMENUSIZE = 55
SM_ARRANGE = 56
SM_CXMINIMIZED = 57
SM_CYMINIMIZED = 58
SM_CXMAXTRACK = 59
SM_CYMAXTRACK = 60
SM_CXMAXIMIZED = 61
SM_CYMAXIMIZED = 62
SM_NETWORK = 63
SM_CLEANBOOT = 67
SM_CXDRAG = 68
SM_CYDRAG = 69
SM_SHOWSOUNDS = 70
SM_CXMENUCHECK = 71
SM_CYMENUCHECK = 72
SM_SLOWMACHINE = 73
SM_MIDEASTENABLED = 74
SM_MOUSEWHEELPRESENT = 75
SM_XVIRTUALSCREEN = 76
SM_YVIRTUALSCREEN = 77
SM_CXVIRTUALSCREEN = 78
SM_CYVIRTUALSCREEN = 79
SM_CMONITORS = 80
SM_SAMEDISPLAYFORMAT = 81
SM_CMETRICS = 83

MNC_IGNORE = 0
MNC_CLOSE = 1
MNC_EXECUTE = 2
MNC_SELECT = 3
MNS_NOCHECK = -2147483648
MNS_MODELESS = 1073741824
MNS_DRAGDROP = 536870912
MNS_AUTODISMISS = 268435456
MNS_NOTIFYBYPOS = 134217728
MNS_CHECKORBMP = 67108864
MIM_MAXHEIGHT = 1
MIM_BACKGROUND = 2
MIM_HELPID = 4
MIM_MENUDATA = 8
MIM_STYLE = 16
MIM_APPLYTOSUBMENUS = -2147483648
MND_CONTINUE = 0
MND_ENDMENU = 1
MNGOF_GAP = 3
MNGO_NOINTERFACE = 0
MNGO_NOERROR = 1
MIIM_STATE = 1
MIIM_ID = 2
MIIM_SUBMENU = 4
MIIM_CHECKMARKS = 8
MIIM_TYPE = 16
MIIM_DATA = 32
MIIM_STRING = 64
MIIM_BITMAP = 128
MIIM_FTYPE = 256
HBMMENU_CALLBACK = -1
HBMMENU_SYSTEM = 1
HBMMENU_MBAR_RESTORE = 2
HBMMENU_MBAR_MINIMIZE = 3
HBMMENU_MBAR_CLOSE = 5
HBMMENU_MBAR_CLOSE_D = 6
HBMMENU_MBAR_MINIMIZE_D = 7
HBMMENU_POPUP_CLOSE = 8
HBMMENU_POPUP_RESTORE = 9
HBMMENU_POPUP_MAXIMIZE = 10
HBMMENU_POPUP_MINIMIZE = 11
GMDI_USEDISABLED = 1
GMDI_GOINTOPOPUPS = 2
TPM_LEFTBUTTON = 0
TPM_RIGHTBUTTON = 2
TPM_LEFTALIGN = 0
TPM_CENTERALIGN = 4
TPM_RIGHTALIGN = 8
TPM_TOPALIGN = 0
TPM_VCENTERALIGN = 16
TPM_BOTTOMALIGN = 32
TPM_HORIZONTAL = 0
TPM_VERTICAL = 64
TPM_NONOTIFY = 128
TPM_RETURNCMD = 256
TPM_RECURSE = 1
DOF_EXECUTABLE = 32769
DOF_DOCUMENT = 32770
DOF_DIRECTORY = 32771
DOF_MULTIPLE = 32772
DOF_PROGMAN = 1
DOF_SHELLDATA = 2
DO_DROPFILE = 1162627398
DO_PRINTFILE = 1414419024
DT_TOP = 0
DT_LEFT = 0
DT_CENTER = 1
DT_RIGHT = 2
DT_VCENTER = 4
DT_BOTTOM = 8
DT_WORDBREAK = 16
DT_SINGLELINE = 32
DT_EXPANDTABS = 64
DT_TABSTOP = 128
DT_NOCLIP = 256
DT_EXTERNALLEADING = 512
DT_CALCRECT = 1024
DT_NOPREFIX = 2048
DT_INTERNAL = 4096
DT_EDITCONTROL = 8192
DT_PATH_ELLIPSIS = 16384
DT_END_ELLIPSIS = 32768
DT_MODIFYSTRING = 65536
DT_RTLREADING = 131072
DT_WORD_ELLIPSIS = 262144
DST_COMPLEX = 0
DST_TEXT = 1
DST_PREFIXTEXT = 2
DST_ICON = 3
DST_BITMAP = 4
DSS_NORMAL = 0
DSS_UNION = 16
DSS_DISABLED = 32
DSS_MONO = 128
DSS_RIGHT = 32768

SW_SCROLLCHILDREN = 1
SW_INVALIDATE = 2
SW_ERASE = 4
SW_SMOOTHSCROLL = 16

MB_OK = 0
MB_OKCANCEL = 1
MB_ABORTRETRYIGNORE = 2
MB_YESNOCANCEL = 3
MB_YESNO = 4
MB_RETRYCANCEL = 5
MB_ICONHAND = 16
MB_ICONQUESTION = 32
MB_ICONEXCLAMATION = 48
MB_ICONASTERISK = 64
MB_ICONWARNING = MB_ICONEXCLAMATION
MB_ICONERROR = MB_ICONHAND
MB_ICONINFORMATION = MB_ICONASTERISK
MB_ICONSTOP = MB_ICONHAND
MB_DEFBUTTON1 = 0
MB_DEFBUTTON2 = 256
MB_DEFBUTTON3 = 512
MB_DEFBUTTON4 = 768
MB_APPLMODAL = 0
MB_SYSTEMMODAL = 4096
MB_TASKMODAL = 8192
MB_HELP = 16384
MB_NOFOCUS = 32768
MB_SETFOREGROUND = 65536
MB_DEFAULT_DESKTOP_ONLY = 131072
MB_TOPMOST = 262144
MB_RIGHT = 524288
MB_RTLREADING = 1048576
MB_SERVICE_NOTIFICATION = 2097152
MB_TYPEMASK = 15
MB_USERICON = 128
MB_ICONMASK = 240
MB_DEFMASK = 3840
MB_MODEMASK = 12288
MB_MISCMASK = 49152

WM_NULL = 0x0000
WM_CREATE = 0x0001
WM_DESTROY = 0x0002
WM_MOVE = 0x0003
WM_SIZE = 0x0005
WM_ACTIVATE = 0x0006
WA_INACTIVE = 0
WA_ACTIVE = 1
WA_CLICKACTIVE = 2
WM_SETFOCUS = 0x0007
WM_KILLFOCUS = 0x0008
WM_ENABLE = 0x000A
WM_SETREDRAW = 0x000B
WM_SETTEXT = 0x000C
WM_GETTEXT = 0x000D
WM_GETTEXTLENGTH = 0x000E
WM_PAINT = 0x000F
WM_CLOSE = 0x0010
WM_QUERYENDSESSION = 0x0011
WM_QUERYOPEN = 0x0013
WM_ENDSESSION = 0x0016
WM_QUIT = 0x0012
WM_ERASEBKGND = 0x0014
WM_SYSCOLORCHANGE = 0x0015
WM_SHOWWINDOW = 0x0018
WM_WININICHANGE = 0x001A
WM_SETTINGCHANGE = WM_WININICHANGE
WM_DEVMODECHANGE = 0x001B
WM_ACTIVATEAPP = 0x001C
WM_FONTCHANGE = 0x001D
WM_TIMECHANGE = 0x001E
WM_CANCELMODE = 0x001F
WM_SETCURSOR = 0x0020
WM_MOUSEACTIVATE = 0x0021
WM_CHILDACTIVATE = 0x0022
WM_QUEUESYNC = 0x0023
WM_GETMINMAXINFO = 0x0024
WM_PAINTICON = 0x0026
WM_ICONERASEBKGND = 0x0027
WM_NEXTDLGCTL = 0x0028
WM_SPOOLERSTATUS = 0x002A
WM_DRAWITEM = 0x002B
WM_MEASUREITEM = 0x002C
WM_DELETEITEM = 0x002D
WM_VKEYTOITEM = 0x002E
WM_CHARTOITEM = 0x002F
WM_SETFONT = 0x0030
WM_GETFONT = 0x0031
WM_SETHOTKEY = 0x0032
WM_GETHOTKEY = 0x0033
WM_QUERYDRAGICON = 0x0037
WM_COMPAREITEM = 0x0039
WM_GETOBJECT = 0x003D
WM_COMPACTING = 0x0041
WM_COMMNOTIFY = 0x0044
WM_WINDOWPOSCHANGING = 0x0046
WM_WINDOWPOSCHANGED = 0x0047
WM_POWER = 0x0048
PWR_OK = 1
PWR_FAIL = -1
PWR_SUSPENDREQUEST = 1
PWR_SUSPENDRESUME = 2
PWR_CRITICALRESUME = 3
WM_COPYDATA = 0x004A
WM_CANCELJOURNAL = 0x004B
WM_NOTIFY = 0x004E
WM_INPUTLANGCHANGEREQUEST = 0x0050
WM_INPUTLANGCHANGE = 0x0051
WM_TCARD = 0x0052
WM_HELP = 0x0053
WM_USERCHANGED = 0x0054
WM_NOTIFYFORMAT = 0x0055
NFR_ANSI = 1
NFR_UNICODE = 2
NF_QUERY = 3
NF_REQUERY = 4
WM_CONTEXTMENU = 0x007B
WM_STYLECHANGING = 0x007C
WM_STYLECHANGED = 0x007D
WM_DISPLAYCHANGE = 0x007E
WM_GETICON = 0x007F
WM_SETICON = 0x0080
WM_NCCREATE = 0x0081
WM_NCDESTROY = 0x0082
WM_NCCALCSIZE = 0x0083
WM_NCHITTEST = 0x0084
WM_NCPAINT = 0x0085
WM_NCACTIVATE = 0x0086
WM_GETDLGCODE = 0x0087
WM_SYNCPAINT = 0x0088
WM_NCMOUSEMOVE = 0x00A0
WM_NCLBUTTONDOWN = 0x00A1
WM_NCLBUTTONUP = 0x00A2
WM_NCLBUTTONDBLCLK = 0x00A3
WM_NCRBUTTONDOWN = 0x00A4
WM_NCRBUTTONUP = 0x00A5
WM_NCRBUTTONDBLCLK = 0x00A6
WM_NCMBUTTONDOWN = 0x00A7
WM_NCMBUTTONUP = 0x00A8
WM_NCMBUTTONDBLCLK = 0x00A9
WM_NCXBUTTONDOWN = 0x00AB
WM_NCXBUTTONUP = 0x00AC
WM_NCXBUTTONDBLCLK = 0x00AD
WM_INPUT_DEVICE_CHANGE = 0x00FE
WM_INPUT = 0x00FF
WM_KEYFIRST = 0x0100
WM_KEYDOWN = 0x0100
WM_KEYUP = 0x0101
WM_CHAR = 0x0102
WM_DEADCHAR = 0x0103
WM_SYSKEYDOWN = 0x0104
WM_SYSKEYUP = 0x0105
WM_SYSCHAR = 0x0106
WM_SYSDEADCHAR = 0x0107
WM_UNICHAR = 0x0109
WM_KEYLAST = 0x0109
UNICODE_NOCHAR = 0xFFFF
WM_IME_STARTCOMPOSITION = 0x010D
WM_IME_ENDCOMPOSITION = 0x010E
WM_IME_COMPOSITION = 0x010F
WM_IME_KEYLAST = 0x010F
WM_INITDIALOG = 0x0110
WM_COMMAND = 0x0111
WM_SYSCOMMAND = 0x0112
WM_TIMER = 0x0113
WM_HSCROLL = 0x0114
WM_VSCROLL = 0x0115
WM_INITMENU = 0x0116
WM_INITMENUPOPUP = 0x0117
WM_GESTURE = 0x0119
WM_GESTURENOTIFY = 0x011A
WM_MENUSELECT = 0x011F
WM_MENUCHAR = 0x0120
WM_ENTERIDLE = 0x0121
WM_MENURBUTTONUP = 0x0122
WM_MENUDRAG = 0x0123
WM_MENUGETOBJECT = 0x0124
WM_UNINITMENUPOPUP = 0x0125
WM_MENUCOMMAND = 0x0126
WM_CHANGEUISTATE = 0x0127
WM_UPDATEUISTATE = 0x0128
WM_QUERYUISTATE = 0x0129
UIS_SET = 1
UIS_CLEAR = 2
UIS_INITIALIZE = 3
UISF_HIDEFOCUS = 0x1
UISF_HIDEACCEL = 0x2
UISF_ACTIVE = 0x4
WM_CTLCOLORMSGBOX = 0x0132
WM_CTLCOLOREDIT = 0x0133
WM_CTLCOLORLISTBOX = 0x0134
WM_CTLCOLORBTN = 0x0135
WM_CTLCOLORDLG = 0x0136
WM_CTLCOLORSCROLLBAR = 0x0137
WM_CTLCOLORSTATIC = 0x0138
MN_GETHMENU = 0x01E1
WM_MOUSEFIRST = 0x0200
WM_MOUSEMOVE = 0x0200
WM_LBUTTONDOWN = 0x0201
WM_LBUTTONUP = 0x0202
WM_LBUTTONDBLCLK = 0x0203
WM_RBUTTONDOWN = 0x0204
WM_RBUTTONUP = 0x0205
WM_RBUTTONDBLCLK = 0x0206
WM_MBUTTONDOWN = 0x0207
WM_MBUTTONUP = 0x0208
WM_MBUTTONDBLCLK = 0x0209
WM_MOUSEWHEEL = 0x020A
WM_XBUTTONDOWN = 0x020B
WM_XBUTTONUP = 0x020C
WM_XBUTTONDBLCLK = 0x020D
WM_MOUSEHWHEEL = 0x020E
WM_MOUSELAST = 0x020E
WHEEL_DELTA = 120
WHEEL_PAGESCROLL = UINT(-1).value
XBUTTON1 = 0x0001
XBUTTON2 = 0x0002
WM_PARENTNOTIFY = 0x0210
WM_ENTERMENULOOP = 0x0211
WM_EXITMENULOOP = 0x0212
WM_NEXTMENU = 0x0213
WM_SIZING = 0x0214
WM_CAPTURECHANGED = 0x0215
WM_MOVING = 0x0216
WM_POWERBROADCAST = 0x0218
PBT_APMQUERYSUSPEND = 0x0000
PBT_APMQUERYSTANDBY = 0x0001
PBT_APMQUERYSUSPENDFAILED = 0x0002
PBT_APMQUERYSTANDBYFAILED = 0x0003
PBT_APMSUSPEND = 0x0004
PBT_APMSTANDBY = 0x0005
PBT_APMRESUMECRITICAL = 0x0006
PBT_APMRESUMESUSPEND = 0x0007
PBT_APMRESUMESTANDBY = 0x0008
PBTF_APMRESUMEFROMFAILURE = 0x00000001
PBT_APMBATTERYLOW = 0x0009
PBT_APMPOWERSTATUSCHANGE = 0x000A
PBT_APMOEMEVENT = 0x000B
PBT_APMRESUMEAUTOMATIC = 0x0012
PBT_POWERSETTINGCHANGE = 0x8013
WM_DEVICECHANGE = 0x0219
WM_MDICREATE = 0x0220
WM_MDIDESTROY = 0x0221
WM_MDIACTIVATE = 0x0222
WM_MDIRESTORE = 0x0223
WM_MDINEXT = 0x0224
WM_MDIMAXIMIZE = 0x0225
WM_MDITILE = 0x0226
WM_MDICASCADE = 0x0227
WM_MDIICONARRANGE = 0x0228
WM_MDIGETACTIVE = 0x0229
WM_MDISETMENU = 0x0230
WM_ENTERSIZEMOVE = 0x0231
WM_EXITSIZEMOVE = 0x0232
WM_DROPFILES = 0x0233
WM_MDIREFRESHMENU = 0x0234
WM_POINTERDEVICECHANGE = 0x238
WM_POINTERDEVICEINRANGE = 0x239
WM_POINTERDEVICEOUTOFRANGE = 0x23A
WM_TOUCH = 0x0240
WM_NCPOINTERUPDATE = 0x0241
WM_NCPOINTERDOWN = 0x0242
WM_NCPOINTERUP = 0x0243
WM_POINTERUPDATE = 0x0245
WM_POINTERDOWN = 0x0246
WM_POINTERUP = 0x0247
WM_POINTERENTER = 0x0249
WM_POINTERLEAVE = 0x024A
WM_POINTERACTIVATE = 0x024B
WM_POINTERCAPTURECHANGED = 0x024C
WM_TOUCHHITTESTING = 0x024D
WM_POINTERWHEEL = 0x024E
WM_POINTERHWHEEL = 0x024F
DM_POINTERHITTEST = 0x0250
WM_POINTERROUTEDTO = 0x0251
WM_POINTERROUTEDAWAY = 0x0252
WM_POINTERROUTEDRELEASED = 0x0253
WM_IME_SETCONTEXT = 0x0281
WM_IME_NOTIFY = 0x0282
WM_IME_CONTROL = 0x0283
WM_IME_COMPOSITIONFULL = 0x0284
WM_IME_SELECT = 0x0285
WM_IME_CHAR = 0x0286
WM_IME_REQUEST = 0x0288
WM_IME_KEYDOWN = 0x0290
WM_IME_KEYUP = 0x0291
WM_MOUSEHOVER = 0x02A1
WM_MOUSELEAVE = 0x02A3
WM_NCMOUSEHOVER = 0x02A0
WM_NCMOUSELEAVE = 0x02A2
WM_WTSSESSION_CHANGE = 0x02B1
WM_TABLET_FIRST = 0x02C0
WM_TABLET_LAST = 0x02DF
WM_DPICHANGED = 0x02E0
WM_DPICHANGED_BEFOREPARENT = 0x02E2
WM_DPICHANGED_AFTERPARENT = 0x02E3
WM_GETDPISCALEDSIZE = 0x02E4
WM_CUT = 0x0300
WM_COPY = 0x0301
WM_PASTE = 0x0302
WM_CLEAR = 0x0303
WM_UNDO = 0x0304
WM_RENDERFORMAT = 0x0305
WM_RENDERALLFORMATS = 0x0306
WM_DESTROYCLIPBOARD = 0x0307
WM_DRAWCLIPBOARD = 0x0308
WM_PAINTCLIPBOARD = 0x0309
WM_VSCROLLCLIPBOARD = 0x030A
WM_SIZECLIPBOARD = 0x030B
WM_ASKCBFORMATNAME = 0x030C
WM_CHANGECBCHAIN = 0x030D
WM_HSCROLLCLIPBOARD = 0x030E
WM_QUERYNEWPALETTE = 0x030F
WM_PALETTEISCHANGING = 0x0310
WM_PALETTECHANGED = 0x0311
WM_HOTKEY = 0x0312
WM_PRINT = 0x0317
WM_PRINTCLIENT = 0x0318
WM_APPCOMMAND = 0x0319
WM_THEMECHANGED = 0x031A
WM_CLIPBOARDUPDATE = 0x031D
WM_DWMCOMPOSITIONCHANGED = 0x031E
WM_DWMNCRENDERINGCHANGED = 0x031F
WM_DWMCOLORIZATIONCOLORCHANGED = 0x0320
WM_DWMWINDOWMAXIMIZEDCHANGE = 0x0321
WM_DWMSENDICONICTHUMBNAIL = 0x0323
WM_DWMSENDICONICLIVEPREVIEWBITMAP = 0x0326
WM_GETTITLEBARINFOEX = 0x033F
WM_HANDHELDFIRST = 0x0358
WM_HANDHELDLAST = 0x035F
WM_AFXFIRST = 0x0360
WM_AFXLAST = 0x037F
WM_PENWINFIRST = 0x0380
WM_PENWINLAST = 0x038F
WM_APP = 0x8000
WM_USER = 0x0400
WMSZ_LEFT = 1
WMSZ_RIGHT = 2
WMSZ_TOP = 3
WMSZ_TOPLEFT = 4
WMSZ_TOPRIGHT = 5
WMSZ_BOTTOM = 6
WMSZ_BOTTOMLEFT = 7
WMSZ_BOTTOMRIGHT = 8
HTERROR = -2
HTTRANSPARENT = -1
HTNOWHERE = 0
HTCLIENT = 1
HTCAPTION = 2
HTSYSMENU = 3
HTGROWBOX = 4
HTSIZE = HTGROWBOX
HTMENU = 5
HTHSCROLL = 6
HTVSCROLL = 7
HTMINBUTTON = 8
HTMAXBUTTON = 9
HTLEFT = 10
HTRIGHT = 11
HTTOP = 12
HTTOPLEFT = 13
HTTOPRIGHT = 14
HTBOTTOM = 15
HTBOTTOMLEFT = 16
HTBOTTOMRIGHT = 17
HTBORDER = 18
HTREDUCE = HTMINBUTTON
HTZOOM = HTMAXBUTTON
HTSIZEFIRST = HTLEFT
HTSIZELAST = HTBOTTOMRIGHT
HTOBJECT = 19
HTCLOSE = 20
HTHELP = 21
SMTO_NORMAL = 0x0000
SMTO_BLOCK = 0x0001
SMTO_ABORTIFHUNG = 0x0002
SMTO_NOTIMEOUTIFNOTHUNG = 0x0008
SMTO_ERRORONEXIT = 0x0020
MA_ACTIVATE = 1
MA_ACTIVATEANDEAT = 2
MA_NOACTIVATE = 3
MA_NOACTIVATEANDEAT = 4
ICON_SMALL = 0
ICON_BIG = 1
ICON_SMALL2 = 2
SIZE_RESTORED = 0
SIZE_MINIMIZED = 1
SIZE_MAXIMIZED = 2
SIZE_MAXSHOW = 3
SIZE_MAXHIDE = 4
SIZENORMAL = SIZE_RESTORED
SIZEICONIC = SIZE_MINIMIZED
SIZEFULLSCREEN = SIZE_MAXIMIZED
SIZEZOOMSHOW = SIZE_MAXSHOW
SIZEZOOMHIDE = SIZE_MAXHIDE
WVR_ALIGNTOP = 0x0010
WVR_ALIGNLEFT = 0x0020
WVR_ALIGNBOTTOM = 0x0040
WVR_ALIGNRIGHT = 0x0080
WVR_HREDRAW = 0x0100
WVR_VREDRAW = 0x0200
WVR_REDRAW = WVR_HREDRAW | WVR_VREDRAW
WVR_VALIDRECTS = 0x0400
MK_LBUTTON = 0x0001
MK_RBUTTON = 0x0002
MK_SHIFT = 0x0004
MK_CONTROL = 0x0008
MK_MBUTTON = 0x0010
MK_XBUTTON1 = 0x0020
MK_XBUTTON2 = 0x0040
TME_HOVER = 0x00000001
TME_LEAVE = 0x00000002
TME_NONCLIENT = 0x00000010
TME_QUERY = 0x40000000
TME_CANCEL = 0x80000000
HOVER_DEFAULT = 0xFFFFFFFF

WS_OVERLAPPED = 0
WS_POPUP = -2147483648
WS_CHILD = 1073741824
WS_MINIMIZE = 536870912
WS_VISIBLE = 268435456
WS_DISABLED = 134217728
WS_CLIPSIBLINGS = 67108864
WS_CLIPCHILDREN = 33554432
WS_MAXIMIZE = 16777216
WS_CAPTION = 12582912
WS_BORDER = 8388608
WS_DLGFRAME = 4194304
WS_VSCROLL = 2097152
WS_HSCROLL = 1048576
WS_SYSMENU = 524288
WS_THICKFRAME = 262144
WS_GROUP = 131072
WS_TABSTOP = 65536
WS_MINIMIZEBOX = 131072
WS_MAXIMIZEBOX = 65536
WS_TILED = WS_OVERLAPPED
WS_ICONIC = WS_MINIMIZE
WS_SIZEBOX = WS_THICKFRAME
WS_OVERLAPPEDWINDOW = (
    WS_OVERLAPPED
    | WS_CAPTION
    | WS_SYSMENU
    | WS_THICKFRAME
    | WS_MINIMIZEBOX
    | WS_MAXIMIZEBOX
    )
WS_POPUPWINDOW = WS_POPUP | WS_BORDER | WS_SYSMENU
WS_CHILDWINDOW = WS_CHILD
WS_TILEDWINDOW = WS_OVERLAPPEDWINDOW
WS_EX_DLGMODALFRAME = 1
WS_EX_NOPARENTNOTIFY = 4
WS_EX_TOPMOST = 8
WS_EX_ACCEPTFILES = 16
WS_EX_TRANSPARENT = 32
WS_EX_MDICHILD = 64
WS_EX_TOOLWINDOW = 128
WS_EX_WINDOWEDGE = 256
WS_EX_CLIENTEDGE = 512
WS_EX_CONTEXTHELP = 1024
WS_EX_RIGHT = 4096
WS_EX_LEFT = 0
WS_EX_RTLREADING = 8192
WS_EX_LTRREADING = 0
WS_EX_LEFTSCROLLBAR = 16384
WS_EX_RIGHTSCROLLBAR = 0
WS_EX_CONTROLPARENT = 65536
WS_EX_STATICEDGE = 131072
WS_EX_APPWINDOW = 262144
WS_EX_OVERLAPPEDWINDOW = WS_EX_WINDOWEDGE | WS_EX_CLIENTEDGE
WS_EX_PALETTEWINDOW = WS_EX_WINDOWEDGE | WS_EX_TOOLWINDOW | WS_EX_TOPMOST
WS_EX_LAYERED = 0x00080000
WS_EX_NOINHERITLAYOUT = 0x00100000
WS_EX_LAYOUTRTL = 0x00400000
WS_EX_COMPOSITED = 0x02000000
WS_EX_NOACTIVATE = 0x08000000

CS_VREDRAW = 1
CS_HREDRAW = 2
CS_DBLCLKS = 8
CS_OWNDC = 32
CS_CLASSDC = 64
CS_PARENTDC = 128
CS_NOCLOSE = 512
CS_SAVEBITS = 2048
CS_BYTEALIGNCLIENT = 4096
CS_BYTEALIGNWINDOW = 8192
CS_GLOBALCLASS = 16384
CS_IME = 65536
PRF_CHECKVISIBLE = 1
PRF_NONCLIENT = 2
PRF_CLIENT = 4
PRF_ERASEBKGND = 8
PRF_CHILDREN = 16
PRF_OWNED = 32
BDR_RAISEDOUTER = 1
BDR_SUNKENOUTER = 2
BDR_RAISEDINNER = 4
BDR_SUNKENINNER = 8

BDR_OUTER = BDR_RAISEDOUTER | BDR_SUNKENOUTER
BDR_INNER = BDR_RAISEDINNER | BDR_SUNKENINNER
BDR_RAISED = BDR_RAISEDOUTER | BDR_RAISEDINNER
BDR_SUNKEN = BDR_SUNKENOUTER | BDR_SUNKENINNER
EDGE_RAISED = BDR_RAISEDOUTER | BDR_RAISEDINNER
EDGE_SUNKEN = BDR_SUNKENOUTER | BDR_SUNKENINNER
EDGE_ETCHED = BDR_SUNKENOUTER | BDR_RAISEDINNER
EDGE_BUMP = BDR_RAISEDOUTER | BDR_SUNKENINNER

ISMEX_NOSEND = 0
ISMEX_SEND = 1
ISMEX_NOTIFY = 2
ISMEX_CALLBACK = 4
ISMEX_REPLIED = 8
CW_USEDEFAULT = -2147483648
FLASHW_STOP = 0
FLASHW_CAPTION = 1
FLASHW_TRAY = 2
FLASHW_ALL = FLASHW_CAPTION | FLASHW_TRAY
FLASHW_TIMER = 4
FLASHW_TIMERNOFG = 12

DS_ABSALIGN = 1
DS_SYSMODAL = 2
DS_LOCALEDIT = 32
DS_SETFONT = 64
DS_MODALFRAME = 128
DS_NOIDLEMSG = 256
DS_SETFOREGROUND = 512
DS_3DLOOK = 4
DS_FIXEDSYS = 8
DS_NOFAILCREATE = 16
DS_CONTROL = 1024
DS_CENTER = 2048
DS_CENTERMOUSE = 4096
DS_CONTEXTHELP = 8192
DM_GETDEFID = WM_USER + 0
DM_SETDEFID = WM_USER + 1
DM_REPOSITION = WM_USER + 2
DC_HASDEFID = 21323
DLGC_WANTARROWS = 1
DLGC_WANTTAB = 2
DLGC_WANTALLKEYS = 4
DLGC_WANTMESSAGE = 4
DLGC_HASSETSEL = 8
DLGC_DEFPUSHBUTTON = 16
DLGC_UNDEFPUSHBUTTON = 32
DLGC_RADIOBUTTON = 64
DLGC_WANTCHARS = 128
DLGC_STATIC = 256
DLGC_BUTTON = 8192
LB_CTLCODE = 0
LB_OKAY = 0
LB_ERR = -1
LB_ERRSPACE = -2
LBN_ERRSPACE = -2
LBN_SELCHANGE = 1
LBN_DBLCLK = 2
LBN_SELCANCEL = 3
LBN_SETFOCUS = 4
LBN_KILLFOCUS = 5
LB_ADDSTRING = 384
LB_INSERTSTRING = 385
LB_DELETESTRING = 386
LB_SELITEMRANGEEX = 387
LB_RESETCONTENT = 388
LB_SETSEL = 389
LB_SETCURSEL = 390
LB_GETSEL = 391
LB_GETCURSEL = 392
LB_GETTEXT = 393
LB_GETTEXTLEN = 394
LB_GETCOUNT = 395
LB_SELECTSTRING = 396
LB_DIR = 397
LB_GETTOPINDEX = 398
LB_FINDSTRING = 399
LB_GETSELCOUNT = 400
LB_GETSELITEMS = 401
LB_SETTABSTOPS = 402
LB_GETHORIZONTALEXTENT = 403
LB_SETHORIZONTALEXTENT = 404
LB_SETCOLUMNWIDTH = 405
LB_ADDFILE = 406
LB_SETTOPINDEX = 407
LB_GETITEMRECT = 408
LB_GETITEMDATA = 409
LB_SETITEMDATA = 410
LB_SELITEMRANGE = 411
LB_SETANCHORINDEX = 412
LB_GETANCHORINDEX = 413
LB_SETCARETINDEX = 414
LB_GETCARETINDEX = 415
LB_SETITEMHEIGHT = 416
LB_GETITEMHEIGHT = 417
LB_FINDSTRINGEXACT = 418
LB_SETLOCALE = 421
LB_GETLOCALE = 422
LB_SETCOUNT = 423
LB_INITSTORAGE = 424
LB_ITEMFROMPOINT = 425
LB_MSGMAX = 432
LBS_NOTIFY = 1
LBS_SORT = 2
LBS_NOREDRAW = 4
LBS_MULTIPLESEL = 8
LBS_OWNERDRAWFIXED = 16
LBS_OWNERDRAWVARIABLE = 32
LBS_HASSTRINGS = 64
LBS_USETABSTOPS = 128
LBS_NOINTEGRALHEIGHT = 256
LBS_MULTICOLUMN = 512
LBS_WANTKEYBOARDINPUT = 1024
LBS_EXTENDEDSEL = 2048
LBS_DISABLENOSCROLL = 4096
LBS_NODATA = 8192
LBS_NOSEL = 16384
LBS_STANDARD = LBS_NOTIFY | LBS_SORT | WS_VSCROLL | WS_BORDER
CB_OKAY = 0
CB_ERR = -1
CB_ERRSPACE = -2
CBN_ERRSPACE = -1
CBN_SELCHANGE = 1
CBN_DBLCLK = 2
CBN_SETFOCUS = 3
CBN_KILLFOCUS = 4
CBN_EDITCHANGE = 5
CBN_EDITUPDATE = 6
CBN_DROPDOWN = 7
CBN_CLOSEUP = 8
CBN_SELENDOK = 9
CBN_SELENDCANCEL = 10
CBS_SIMPLE = 1
CBS_DROPDOWN = 2
CBS_DROPDOWNLIST = 3
CBS_OWNERDRAWFIXED = 16
CBS_OWNERDRAWVARIABLE = 32
CBS_AUTOHSCROLL = 64
CBS_OEMCONVERT = 128
CBS_SORT = 256
CBS_HASSTRINGS = 512
CBS_NOINTEGRALHEIGHT = 1024
CBS_DISABLENOSCROLL = 2048
CBS_UPPERCASE = 8192
CBS_LOWERCASE = 16384
CB_GETEDITSEL = 320
CB_LIMITTEXT = 321
CB_SETEDITSEL = 322
CB_ADDSTRING = 323
CB_DELETESTRING = 324
CB_DIR = 325
CB_GETCOUNT = 326
CB_GETCURSEL = 327
CB_GETLBTEXT = 328
CB_GETLBTEXTLEN = 329
CB_INSERTSTRING = 330
CB_RESETCONTENT = 331
CB_FINDSTRING = 332
CB_SELECTSTRING = 333
CB_SETCURSEL = 334
CB_SHOWDROPDOWN = 335
CB_GETITEMDATA = 336
CB_SETITEMDATA = 337
CB_GETDROPPEDCONTROLRECT = 338
CB_SETITEMHEIGHT = 339
CB_GETITEMHEIGHT = 340
CB_SETEXTENDEDUI = 341
CB_GETEXTENDEDUI = 342
CB_GETDROPPEDSTATE = 343
CB_FINDSTRINGEXACT = 344
CB_SETLOCALE = 345
CB_GETLOCALE = 346
CB_GETTOPINDEX = 347
CB_SETTOPINDEX = 348
CB_GETHORIZONTALEXTENT = 349
CB_SETHORIZONTALEXTENT = 350
CB_GETDROPPEDWIDTH = 351
CB_SETDROPPEDWIDTH = 352
CB_INITSTORAGE = 353
CB_MSGMAX = 354
SBS_HORZ = 0
SBS_VERT = 1
SBS_TOPALIGN = 2
SBS_LEFTALIGN = 2
SBS_BOTTOMALIGN = 4
SBS_RIGHTALIGN = 4
SBS_SIZEBOXTOPLEFTALIGN = 2
SBS_SIZEBOXBOTTOMRIGHTALIGN = 4
SBS_SIZEBOX = 8
SBS_SIZEGRIP = 16
SBM_SETPOS = 224
SBM_GETPOS = 225
SBM_SETRANGE = 226
SBM_SETRANGEREDRAW = 230
SBM_GETRANGE = 227
SBM_ENABLE_ARROWS = 228
SBM_SETSCROLLINFO = 233
SBM_GETSCROLLINFO = 234
SIF_RANGE = 1
SIF_PAGE = 2
SIF_POS = 4
SIF_DISABLENOSCROLL = 8
SIF_TRACKPOS = 16
SIF_ALL = SIF_RANGE | SIF_PAGE | SIF_POS | SIF_TRACKPOS
MDIS_ALLCHILDSTYLES = 1
MDITILE_VERTICAL = 0
MDITILE_HORIZONTAL = 1
MDITILE_SKIPDISABLED = 2

IMC_GETCANDIDATEPOS = 7
IMC_SETCANDIDATEPOS = 8
IMC_GETCOMPOSITIONFONT = 9
IMC_SETCOMPOSITIONFONT = 10
IMC_GETCOMPOSITIONWINDOW = 11
IMC_SETCOMPOSITIONWINDOW = 12
IMC_GETSTATUSWINDOWPOS = 15
IMC_SETSTATUSWINDOWPOS = 16
IMC_CLOSESTATUSWINDOW = 33
IMC_OPENSTATUSWINDOW = 34

APPLICATION_ERROR_MASK = 536870912
ERROR_SEVERITY_SUCCESS = 0
ERROR_SEVERITY_INFORMATIONAL = 1073741824
ERROR_SEVERITY_WARNING = -2147483648
ERROR_SEVERITY_ERROR = -1073741824
MINCHAR = 128
MAXCHAR = 127
MINSHORT = 32768
MAXSHORT = 32767
MINLONG = -2147483648
MAXLONG = 2147483647
MAXBYTE = 255
MAXWORD = 65535
MAXDWORD = -1
LANG_NEUTRAL = 0
LANG_BULGARIAN = 2
LANG_CHINESE = 4
LANG_CROATIAN = 26
LANG_CZECH = 5
LANG_DANISH = 6
LANG_DUTCH = 19
LANG_ENGLISH = 9
LANG_FINNISH = 11
LANG_FRENCH = 12
LANG_GERMAN = 7
LANG_GREEK = 8
LANG_HUNGARIAN = 14
LANG_ICELANDIC = 15
LANG_ITALIAN = 16
LANG_JAPANESE = 17
LANG_KOREAN = 18
LANG_NORWEGIAN = 20
LANG_POLISH = 21
LANG_PORTUGUESE = 22
LANG_ROMANIAN = 24
LANG_RUSSIAN = 25
LANG_SLOVAK = 27
LANG_SLOVENIAN = 36
LANG_SPANISH = 10
LANG_SWEDISH = 29
LANG_TURKISH = 31
SUBLANG_NEUTRAL = 0
SUBLANG_DEFAULT = 1
SUBLANG_SYS_DEFAULT = 2
SUBLANG_CHINESE_TRADITIONAL = 1
SUBLANG_CHINESE_SIMPLIFIED = 2
SUBLANG_CHINESE_HONGKONG = 3
SUBLANG_CHINESE_SINGAPORE = 4
SUBLANG_DUTCH = 1
SUBLANG_DUTCH_BELGIAN = 2
SUBLANG_ENGLISH_US = 1
SUBLANG_ENGLISH_UK = 2
SUBLANG_ENGLISH_AUS = 3
SUBLANG_ENGLISH_CAN = 4
SUBLANG_ENGLISH_NZ = 5
SUBLANG_ENGLISH_EIRE = 6
SUBLANG_FRENCH = 1
SUBLANG_FRENCH_BELGIAN = 2
SUBLANG_FRENCH_CANADIAN = 3
SUBLANG_FRENCH_SWISS = 4
SUBLANG_GERMAN = 1
SUBLANG_GERMAN_SWISS = 2
SUBLANG_GERMAN_AUSTRIAN = 3
SUBLANG_ITALIAN = 1
SUBLANG_ITALIAN_SWISS = 2
SUBLANG_NORWEGIAN_BOKMAL = 1
SUBLANG_NORWEGIAN_NYNORSK = 2
SUBLANG_PORTUGUESE = 2
SUBLANG_PORTUGUESE_BRAZILIAN = 1
SUBLANG_SPANISH = 1
SUBLANG_SPANISH_MEXICAN = 2
SUBLANG_SPANISH_MODERN = 3
SORT_DEFAULT = 0
SORT_JAPANESE_XJIS = 0
SORT_JAPANESE_UNICODE = 1
SORT_CHINESE_BIG5 = 0
SORT_CHINESE_UNICODE = 1
SORT_KOREAN_KSC = 0
SORT_KOREAN_UNICODE = 1

NLS_VALID_LOCALE_MASK = 1048575
PROCESS_TERMINATE = 1
PROCESS_CREATE_THREAD = 2
PROCESS_VM_OPERATION = 8
PROCESS_VM_READ = 16
PROCESS_VM_WRITE = 32
PROCESS_DUP_HANDLE = 64
PROCESS_CREATE_PROCESS = 128
PROCESS_SET_QUOTA = 256
PROCESS_SET_INFORMATION = 512
PROCESS_QUERY_INFORMATION = 1024
PROCESS_SUSPEND_RESUME = 2048
PROCESS_QUERY_LIMITED_INFORMATION = 4096
PROCESS_SET_LIMITED_INFORMATION = 8192
PROCESS_ALL_ACCESS = STANDARD_RIGHTS_REQUIRED | SYNCHRONIZE | 4095
THREAD_TERMINATE = 1
THREAD_SUSPEND_RESUME = 2
THREAD_GET_CONTEXT = 8
THREAD_SET_CONTEXT = 16
THREAD_SET_INFORMATION = 32
THREAD_QUERY_INFORMATION = 64
THREAD_SET_THREAD_TOKEN = 128
THREAD_IMPERSONATE = 256
THREAD_DIRECT_IMPERSONATION = 512
THREAD_SET_LIMITED_INFORMATION = 1024
THREAD_QUERY_LIMITED_INFORMATION = 2048
THREAD_RESUME = 4096
TLS_MINIMUM_AVAILABLE = 64
EVENT_MODIFY_STATE = 2
MUTANT_QUERY_STATE = 1
SEMAPHORE_MODIFY_STATE = 2
TIME_ZONE_ID_UNKNOWN = 0
TIME_ZONE_ID_STANDARD = 1
TIME_ZONE_ID_DAYLIGHT = 2
PROCESSOR_INTEL_386 = 386
PROCESSOR_INTEL_486 = 486
PROCESSOR_INTEL_PENTIUM = 586
PROCESSOR_INTEL_860 = 860
PROCESSOR_MIPS_R2000 = 2000
PROCESSOR_MIPS_R3000 = 3000
PROCESSOR_MIPS_R4000 = 4000
PROCESSOR_ALPHA_21064 = 21064
PROCESSOR_PPC_601 = 601
PROCESSOR_PPC_603 = 603
PROCESSOR_PPC_604 = 604
PROCESSOR_PPC_620 = 620
SECTION_QUERY = 1
SECTION_MAP_WRITE = 2
SECTION_MAP_READ = 4
SECTION_MAP_EXECUTE = 8
SECTION_EXTEND_SIZE = 16
PAGE_NOACCESS = 1
PAGE_READONLY = 2
PAGE_READWRITE = 4
PAGE_WRITECOPY = 8
PAGE_EXECUTE = 16
PAGE_EXECUTE_READ = 32
PAGE_EXECUTE_READWRITE = 64
PAGE_EXECUTE_WRITECOPY = 128
PAGE_GUARD = 256
PAGE_NOCACHE = 512
MEM_COMMIT = 4096
MEM_RESERVE = 8192
MEM_DECOMMIT = 16384
MEM_RELEASE = 32768
MEM_FREE = 65536
MEM_PRIVATE = 131072
MEM_MAPPED = 262144
MEM_TOP_DOWN = 1048576

SEC_FILE = 8388608
SEC_IMAGE = 16777216
SEC_RESERVE = 67108864
SEC_COMMIT = 134217728
SEC_NOCACHE = 268435456
MEM_IMAGE = SEC_IMAGE
FILE_SHARE_READ = 1
FILE_SHARE_WRITE = 2
FILE_SHARE_DELETE = 4

FILE_NOTIFY_CHANGE_FILE_NAME = 1
FILE_NOTIFY_CHANGE_DIR_NAME = 2
FILE_NOTIFY_CHANGE_ATTRIBUTES = 4
FILE_NOTIFY_CHANGE_SIZE = 8
FILE_NOTIFY_CHANGE_LAST_WRITE = 16
FILE_NOTIFY_CHANGE_SECURITY = 256
FILE_CASE_SENSITIVE_SEARCH = 1
FILE_CASE_PRESERVED_NAMES = 2
FILE_FILE_COMPRESSION = 16
FILE_NAMED_STREAMS = 262144
FILE_PERSISTENT_ACLS = 0x00000008
FILE_READ_ONLY_VOLUME = 0x00080000
FILE_SEQUENTIAL_WRITE_ONCE = 0x00100000
FILE_SUPPORTS_ENCRYPTION = 0x00020000
FILE_SUPPORTS_EXTENDED_ATTRIBUTES = 0x00800000
FILE_SUPPORTS_HARD_LINKS = 0x00400000
FILE_SUPPORTS_OBJECT_IDS = 0x00010000
FILE_SUPPORTS_OPEN_BY_FILE_ID = 0x01000000
FILE_SUPPORTS_REPARSE_POINTS = 0x00000080
FILE_SUPPORTS_SPARSE_FILES = 0x00000040
FILE_SUPPORTS_TRANSACTIONS = 0x00200000
FILE_SUPPORTS_USN_JOURNAL = 0x02000000
FILE_UNICODE_ON_DISK = 0x00000004
FILE_VOLUME_IS_COMPRESSED = 0x00008000
FILE_VOLUME_QUOTAS = 0x00000020
IO_COMPLETION_MODIFY_STATE = 2
DUPLICATE_CLOSE_SOURCE = 1
DUPLICATE_SAME_ACCESS = 2

KEY_QUERY_VALUE = 1
KEY_SET_VALUE = 2
KEY_CREATE_SUB_KEY = 4
KEY_ENUMERATE_SUB_KEYS = 8
KEY_NOTIFY = 16
KEY_CREATE_LINK = 32
KEY_WOW64_32KEY = 512
KEY_WOW64_64KEY = 256
KEY_WOW64_RES = 768
KEY_READ = (
    STANDARD_RIGHTS_READ | KEY_QUERY_VALUE | KEY_ENUMERATE_SUB_KEYS | KEY_NOTIFY
    ) & (~SYNCHRONIZE)
KEY_WRITE = (STANDARD_RIGHTS_WRITE | KEY_SET_VALUE | KEY_CREATE_SUB_KEY) & (
    ~SYNCHRONIZE
)
KEY_EXECUTE = KEY_READ & (~SYNCHRONIZE)
KEY_ALL_ACCESS = (
    STANDARD_RIGHTS_ALL
    | KEY_QUERY_VALUE
    | KEY_SET_VALUE
    | KEY_CREATE_SUB_KEY
    | KEY_ENUMERATE_SUB_KEYS
    | KEY_NOTIFY
    | KEY_CREATE_LINK
    ) & (~SYNCHRONIZE)
REG_NOTIFY_CHANGE_ATTRIBUTES = 2
REG_NOTIFY_CHANGE_SECURITY = 8

REG_NONE = 0
REG_SZ = 1
REG_EXPAND_SZ = 2
REG_BINARY = 3
REG_DWORD = 4
REG_DWORD_LITTLE_ENDIAN = 4
REG_DWORD_BIG_ENDIAN = 5
REG_LINK = 6
REG_MULTI_SZ = 7
REG_RESOURCE_LIST = 8
REG_FULL_RESOURCE_DESCRIPTOR = 9
REG_RESOURCE_REQUIREMENTS_LIST = 10
REG_QWORD = 11
REG_QWORD_LITTLE_ENDIAN = 11

EVENTLOG_SEQUENTIAL_READ = 0x0001
EVENTLOG_SEEK_READ = 0x0002
EVENTLOG_FORWARDS_READ = 0x0004
EVENTLOG_BACKWARDS_READ = 0x0008
EVENTLOG_SUCCESS = 0x0000
EVENTLOG_ERROR_TYPE = 1
EVENTLOG_WARNING_TYPE = 2
EVENTLOG_INFORMATION_TYPE = 4
EVENTLOG_AUDIT_SUCCESS = 8
EVENTLOG_AUDIT_FAILURE = 16
EVENTLOG_START_PAIRED_EVENT = 1
EVENTLOG_END_PAIRED_EVENT = 2
EVENTLOG_END_ALL_PAIRED_EVENTS = 4
EVENTLOG_PAIRED_EVENT_ACTIVE = 8
EVENTLOG_PAIRED_EVENT_INACTIVE = 16

OWNER_SECURITY_INFORMATION = 0x00000001
GROUP_SECURITY_INFORMATION = 0x00000002
DACL_SECURITY_INFORMATION = 0x00000004
SACL_SECURITY_INFORMATION = 0x00000008
LABEL_SECURITY_INFORMATION = 0x00000010
ATTRIBUTE_SECURITY_INFORMATION = 0x00000020
SCOPE_SECURITY_INFORMATION = 0x00000040
PROCESS_TRUST_LABEL_SECURITY_INFORMATION = 0x00000080
ACCESS_FILTER_SECURITY_INFORMATION = 0x00000100
BACKUP_SECURITY_INFORMATION = 0x00010000

PROTECTED_DACL_SECURITY_INFORMATION = 0x80000000
PROTECTED_SACL_SECURITY_INFORMATION = 0x40000000
UNPROTECTED_DACL_SECURITY_INFORMATION = 0x20000000
UNPROTECTED_SACL_SECURITY_INFORMATION = 0x10000000

NEWFRAME = 1
ABORTDOC = 2
NEXTBAND = 3
SETCOLORTABLE = 4
GETCOLORTABLE = 5
FLUSHOUTPUT = 6
DRAFTMODE = 7
QUERYESCSUPPORT = 8
SETABORTPROC = 9
STARTDOC = 10
ENDDOC = 11
GETPHYSPAGESIZE = 12
GETPRINTINGOFFSET = 13
GETSCALINGFACTOR = 14
MFCOMMENT = 15
GETPENWIDTH = 16
SETCOPYCOUNT = 17
SELECTPAPERSOURCE = 18
DEVICEDATA = 19
PASSTHROUGH = 19
GETTECHNOLGY = 20
GETTECHNOLOGY = 20
SETLINECAP = 21
SETLINEJOIN = 22
SETMITERLIMIT = 23
BANDINFO = 24
DRAWPATTERNRECT = 25
GETVECTORPENSIZE = 26
GETVECTORBRUSHSIZE = 27
ENABLEDUPLEX = 28
GETSETPAPERBINS = 29
GETSETPRINTORIENT = 30
ENUMPAPERBINS = 31
SETDIBSCALING = 32
EPSPRINTING = 33
ENUMPAPERMETRICS = 34
GETSETPAPERMETRICS = 35
POSTSCRIPT_DATA = 37
POSTSCRIPT_IGNORE = 38
MOUSETRAILS = 39
GETDEVICEUNITS = 42
GETEXTENDEDTEXTMETRICS = 256
GETEXTENTTABLE = 257
GETPAIRKERNTABLE = 258
GETTRACKKERNTABLE = 259
EXTTEXTOUT = 512
GETFACENAME = 513
DOWNLOADFACE = 514
ENABLERELATIVEWIDTHS = 768
ENABLEPAIRKERNING = 769
SETKERNTRACK = 770
SETALLJUSTVALUES = 771
SETCHARSET = 772
STRETCHBLT = 2048
GETSETSCREENPARAMS = 3072
BEGIN_PATH = 4096
CLIP_TO_PATH = 4097
END_PATH = 4098
EXT_DEVICE_CAPS = 4099
RESTORE_CTM = 4100
SAVE_CTM = 4101
SET_ARC_DIRECTION = 4102
SET_BACKGROUND_COLOR = 4103
SET_POLY_MODE = 4104
SET_SCREEN_ANGLE = 4105
SET_SPREAD = 4106
TRANSFORM_CTM = 4107
SET_CLIP_BOX = 4108
SET_BOUNDS = 4109
SET_MIRROR_MODE = 4110
OPENCHANNEL = 4110
DOWNLOADHEADER = 4111
CLOSECHANNEL = 4112
POSTSCRIPT_PASSTHROUGH = 4115
ENCAPSULATED_POSTSCRIPT = 4116
SP_NOTREPORTED = 16384
SP_ERROR = -1
SP_APPABORT = -2
SP_USERABORT = -3
SP_OUTOFDISK = -4
SP_OUTOFMEMORY = -5
PR_JOBSTATUS = 0

OBJ_PEN = 1
OBJ_BRUSH = 2
OBJ_DC = 3
OBJ_METADC = 4
OBJ_PAL = 5
OBJ_FONT = 6
OBJ_BITMAP = 7
OBJ_REGION = 8
OBJ_METAFILE = 9
OBJ_MEMDC = 10
OBJ_EXTPEN = 11
OBJ_ENHMETADC = 12
OBJ_ENHMETAFILE = 13
OBJ_COLORSPACE = 14

MWT_IDENTITY = 1
MWT_LEFTMULTIPLY = 2
MWT_RIGHTMULTIPLY = 3
MWT_MIN = MWT_IDENTITY
MWT_MAX = MWT_RIGHTMULTIPLY
BI_RGB = 0
BI_RLE8 = 1
BI_RLE4 = 2
BI_BITFIELDS = 3
TMPF_FIXED_PITCH = 1
TMPF_VECTOR = 2
TMPF_DEVICE = 8
TMPF_TRUETYPE = 4
NTM_REGULAR = 64
NTM_BOLD = 32
NTM_ITALIC = 1
LF_FACESIZE = 32
LF_FULLFACESIZE = 64
OUT_DEFAULT_PRECIS = 0
OUT_STRING_PRECIS = 1
OUT_CHARACTER_PRECIS = 2
OUT_STROKE_PRECIS = 3
OUT_TT_PRECIS = 4
OUT_DEVICE_PRECIS = 5
OUT_RASTER_PRECIS = 6
OUT_TT_ONLY_PRECIS = 7
OUT_OUTLINE_PRECIS = 8
CLIP_DEFAULT_PRECIS = 0
CLIP_CHARACTER_PRECIS = 1
CLIP_STROKE_PRECIS = 2
CLIP_MASK = 15
CLIP_LH_ANGLES = 1 << 4
CLIP_TT_ALWAYS = 2 << 4
CLIP_EMBEDDED = 8 << 4
DEFAULT_QUALITY = 0
DRAFT_QUALITY = 1
PROOF_QUALITY = 2
NONANTIALIASED_QUALITY = 3
ANTIALIASED_QUALITY = 4
CLEARTYPE_QUALITY = 5
CLEARTYPE_NATURAL_QUALITY = 6
DEFAULT_PITCH = 0
FIXED_PITCH = 1
VARIABLE_PITCH = 2
ANSI_CHARSET = 0
DEFAULT_CHARSET = 1
SYMBOL_CHARSET = 2
SHIFTJIS_CHARSET = 128
HANGEUL_CHARSET = 129
CHINESEBIG5_CHARSET = 136
OEM_CHARSET = 255
JOHAB_CHARSET = 130
HEBREW_CHARSET = 177
ARABIC_CHARSET = 178
GREEK_CHARSET = 161
TURKISH_CHARSET = 162
VIETNAMESE_CHARSET = 163
THAI_CHARSET = 222
EASTEUROPE_CHARSET = 238
RUSSIAN_CHARSET = 204
MAC_CHARSET = 77
BALTIC_CHARSET = 186
FF_DONTCARE = 0 << 4
FF_ROMAN = 1 << 4
FF_SWISS = 2 << 4
FF_MODERN = 3 << 4
FF_SCRIPT = 4 << 4
FF_DECORATIVE = 5 << 4
FW_DONTCARE = 0
FW_THIN = 100
FW_EXTRALIGHT = 200
FW_LIGHT = 300
FW_NORMAL = 400
FW_MEDIUM = 500
FW_SEMIBOLD = 600
FW_BOLD = 700
FW_EXTRABOLD = 800
FW_HEAVY = 900
FW_ULTRALIGHT = FW_EXTRALIGHT
FW_REGULAR = FW_NORMAL
FW_DEMIBOLD = FW_SEMIBOLD
FW_ULTRABOLD = FW_EXTRABOLD
FW_BLACK = FW_HEAVY

BS_SOLID = 0
BS_NULL = 1
BS_HOLLOW = BS_NULL
BS_HATCHED = 2
BS_PATTERN = 3
BS_INDEXED = 4
BS_DIBPATTERN = 5
BS_DIBPATTERNPT = 6
BS_PATTERN8X8 = 7
BS_DIBPATTERN8X8 = 8
HS_HORIZONTAL = 0
HS_VERTICAL = 1
HS_FDIAGONAL = 2
HS_BDIAGONAL = 3
HS_CROSS = 4
HS_DIAGCROSS = 5
HS_FDIAGONAL1 = 6
HS_BDIAGONAL1 = 7
HS_SOLID = 8
HS_DENSE1 = 9
HS_DENSE2 = 10
HS_DENSE3 = 11
HS_DENSE4 = 12
HS_DENSE5 = 13
HS_DENSE6 = 14
HS_DENSE7 = 15
HS_DENSE8 = 16
HS_NOSHADE = 17
HS_HALFTONE = 18
HS_SOLIDCLR = 19
HS_DITHEREDCLR = 20
HS_SOLIDTEXTCLR = 21
HS_DITHEREDTEXTCLR = 22
HS_SOLIDBKCLR = 23
HS_DITHEREDBKCLR = 24
HS_API_MAX = 25
PS_SOLID = 0
PS_DASH = 1
PS_DOT = 2
PS_DASHDOT = 3
PS_DASHDOTDOT = 4
PS_NULL = 5
PS_INSIDEFRAME = 6
PS_USERSTYLE = 7
PS_ALTERNATE = 8
PS_STYLE_MASK = 15
PS_ENDCAP_ROUND = 0
PS_ENDCAP_SQUARE = 256
PS_ENDCAP_FLAT = 512
PS_ENDCAP_MASK = 3840
PS_JOIN_ROUND = 0
PS_JOIN_BEVEL = 4096
PS_JOIN_MITER = 8192
PS_JOIN_MASK = 61440
PS_COSMETIC = 0
PS_GEOMETRIC = 65536
PS_TYPE_MASK = 983040
AD_COUNTERCLOCKWISE = 1
AD_CLOCKWISE = 2
DRIVERVERSION = 0
TECHNOLOGY = 2
HORZSIZE = 4
VERTSIZE = 6
HORZRES = 8
VERTRES = 10
BITSPIXEL = 12
PLANES = 14
NUMBRUSHES = 16
NUMPENS = 18
NUMMARKERS = 20
NUMFONTS = 22
NUMCOLORS = 24
PDEVICESIZE = 26
CURVECAPS = 28
LINECAPS = 30
POLYGONALCAPS = 32
TEXTCAPS = 34
CLIPCAPS = 36
RASTERCAPS = 38
ASPECTX = 40
ASPECTY = 42
ASPECTXY = 44
LOGPIXELSX = 88
LOGPIXELSY = 90
SIZEPALETTE = 104
NUMRESERVED = 106
COLORRES = 108

PHYSICALWIDTH = 110
PHYSICALHEIGHT = 111
PHYSICALOFFSETX = 112
PHYSICALOFFSETY = 113
SCALINGFACTORX = 114
SCALINGFACTORY = 115
VREFRESH = 116
DESKTOPVERTRES = 117
DESKTOPHORZRES = 118
BLTALIGNMENT = 119
SHADEBLENDCAPS = 120
COLORMGMTCAPS = 121

DT_PLOTTER = 0
DT_RASDISPLAY = 1
DT_RASPRINTER = 2
DT_RASCAMERA = 3
DT_CHARSTREAM = 4
DT_METAFILE = 5
DT_DISPFILE = 6
CC_NONE = 0
CC_CIRCLES = 1
CC_PIE = 2
CC_CHORD = 4
CC_ELLIPSES = 8
CC_WIDE = 16
CC_STYLED = 32
CC_WIDESTYLED = 64
CC_INTERIORS = 128
CC_ROUNDRECT = 256
LC_NONE = 0
LC_POLYLINE = 2
LC_MARKER = 4
LC_POLYMARKER = 8
LC_WIDE = 16
LC_STYLED = 32
LC_WIDESTYLED = 64
LC_INTERIORS = 128
PC_NONE = 0
PC_POLYGON = 1
PC_RECTANGLE = 2
PC_WINDPOLYGON = 4
PC_TRAPEZOID = 4
PC_SCANLINE = 8
PC_WIDE = 16
PC_STYLED = 32
PC_WIDESTYLED = 64
PC_INTERIORS = 128
CP_NONE = 0
CP_RECTANGLE = 1
CP_REGION = 2
TC_OP_CHARACTER = 1
TC_OP_STROKE = 2
TC_CP_STROKE = 4
TC_CR_90 = 8
TC_CR_ANY = 16
TC_SF_X_YINDEP = 32
TC_SA_DOUBLE = 64
TC_SA_INTEGER = 128
TC_SA_CONTIN = 256
TC_EA_DOUBLE = 512
TC_IA_ABLE = 1024
TC_UA_ABLE = 2048
TC_SO_ABLE = 4096
TC_RA_ABLE = 8192
TC_VA_ABLE = 16384
TC_RESERVED = 32768
TC_SCROLLBLT = 65536
RC_BITBLT = 1
RC_BANDING = 2
RC_SCALING = 4
RC_BITMAP64 = 8
RC_GDI20_OUTPUT = 16
RC_GDI20_STATE = 32
RC_SAVEBITMAP = 64
RC_DI_BITMAP = 128
RC_PALETTE = 256
RC_DIBTODEV = 512
RC_BIGFONT = 1024
RC_STRETCHBLT = 2048
RC_FLOODFILL = 4096
RC_STRETCHDIB = 8192
RC_OP_DX_OUTPUT = 16384
RC_DEVBITS = 32768
DIB_RGB_COLORS = 0
DIB_PAL_COLORS = 1
DIB_PAL_INDICES = 2
DIB_PAL_PHYSINDICES = 2
DIB_PAL_LOGINDICES = 4
SYSPAL_ERROR = 0
SYSPAL_STATIC = 1
SYSPAL_NOSTATIC = 2
CBM_CREATEDIB = 2
CBM_INIT = 4
FLOODFILLBORDER = 0
FLOODFILLSURFACE = 1
CCHDEVICENAME = 32
CCHFORMNAME = 32

DM_SPECVERSION = 800
DM_ORIENTATION = 1
DM_PAPERSIZE = 2
DM_PAPERLENGTH = 4
DM_PAPERWIDTH = 8
DM_SCALE = 16
DM_POSITION = 32
DM_NUP = 64
DM_DISPLAYORIENTATION = 128
DM_COPIES = 256
DM_DEFAULTSOURCE = 512
DM_PRINTQUALITY = 1024
DM_COLOR = 2048
DM_DUPLEX = 4096
DM_YRESOLUTION = 8192
DM_TTOPTION = 16384
DM_COLLATE = 32768
DM_FORMNAME = 65536
DM_LOGPIXELS = 131072
DM_BITSPERPEL = 262144
DM_PELSWIDTH = 524288
DM_PELSHEIGHT = 1048576
DM_DISPLAYFLAGS = 2097152
DM_DISPLAYFREQUENCY = 4194304
DM_ICMMETHOD = 8388608
DM_ICMINTENT = 16777216
DM_MEDIATYPE = 33554432
DM_DITHERTYPE = 67108864
DM_PANNINGWIDTH = 134217728
DM_PANNINGHEIGHT = 268435456
DM_DISPLAYFIXEDOUTPUT = 536870912

DMORIENT_PORTRAIT = 1
DMORIENT_LANDSCAPE = 2

DMDO_DEFAULT = 0
DMDO_90 = 1
DMDO_180 = 2
DMDO_270 = 3

DMDFO_DEFAULT = 0
DMDFO_STRETCH = 1
DMDFO_CENTER = 2

DMPAPER_LETTER = 1
DMPAPER_LETTERSMALL = 2
DMPAPER_TABLOID = 3
DMPAPER_LEDGER = 4
DMPAPER_LEGAL = 5
DMPAPER_STATEMENT = 6
DMPAPER_EXECUTIVE = 7
DMPAPER_A3 = 8
DMPAPER_A4 = 9
DMPAPER_A4SMALL = 10
DMPAPER_A5 = 11
DMPAPER_B4 = 12
DMPAPER_B5 = 13
DMPAPER_FOLIO = 14
DMPAPER_QUARTO = 15
DMPAPER_10X14 = 16
DMPAPER_11X17 = 17
DMPAPER_NOTE = 18
DMPAPER_ENV_9 = 19
DMPAPER_ENV_10 = 20
DMPAPER_ENV_11 = 21
DMPAPER_ENV_12 = 22
DMPAPER_ENV_14 = 23
DMPAPER_CSHEET = 24
DMPAPER_DSHEET = 25
DMPAPER_ESHEET = 26
DMPAPER_ENV_DL = 27
DMPAPER_ENV_C5 = 28
DMPAPER_ENV_C3 = 29
DMPAPER_ENV_C4 = 30
DMPAPER_ENV_C6 = 31
DMPAPER_ENV_C65 = 32
DMPAPER_ENV_B4 = 33
DMPAPER_ENV_B5 = 34
DMPAPER_ENV_B6 = 35
DMPAPER_ENV_ITALY = 36
DMPAPER_ENV_MONARCH = 37
DMPAPER_ENV_PERSONAL = 38
DMPAPER_FANFOLD_US = 39
DMPAPER_FANFOLD_STD_GERMAN = 40
DMPAPER_FANFOLD_LGL_GERMAN = 41
DMPAPER_ISO_B4 = 42
DMPAPER_JAPANESE_POSTCARD = 43
DMPAPER_9X11 = 44
DMPAPER_10X11 = 45
DMPAPER_15X11 = 46
DMPAPER_ENV_INVITE = 47
DMPAPER_RESERVED_48 = 48
DMPAPER_RESERVED_49 = 49
DMPAPER_LETTER_EXTRA = 50
DMPAPER_LEGAL_EXTRA = 51
DMPAPER_TABLOID_EXTRA = 52
DMPAPER_A4_EXTRA = 53
DMPAPER_LETTER_TRANSVERSE = 54
DMPAPER_A4_TRANSVERSE = 55
DMPAPER_LETTER_EXTRA_TRANSVERSE = 56
DMPAPER_A_PLUS = 57
DMPAPER_B_PLUS = 58
DMPAPER_LETTER_PLUS = 59
DMPAPER_A4_PLUS = 60
DMPAPER_A5_TRANSVERSE = 61
DMPAPER_B5_TRANSVERSE = 62
DMPAPER_A3_EXTRA = 63
DMPAPER_A5_EXTRA = 64
DMPAPER_B5_EXTRA = 65
DMPAPER_A2 = 66
DMPAPER_A3_TRANSVERSE = 67
DMPAPER_A3_EXTRA_TRANSVERSE = 68
DMPAPER_DBL_JAPANESE_POSTCARD = 69
DMPAPER_A6 = 70
DMPAPER_JENV_KAKU2 = 71
DMPAPER_JENV_KAKU3 = 72
DMPAPER_JENV_CHOU3 = 73
DMPAPER_JENV_CHOU4 = 74
DMPAPER_LETTER_ROTATED = 75
DMPAPER_A3_ROTATED = 76
DMPAPER_A4_ROTATED = 77
DMPAPER_A5_ROTATED = 78
DMPAPER_B4_JIS_ROTATED = 79
DMPAPER_B5_JIS_ROTATED = 80
DMPAPER_JAPANESE_POSTCARD_ROTATED = 81
DMPAPER_DBL_JAPANESE_POSTCARD_ROTATED = 82
DMPAPER_A6_ROTATED = 83
DMPAPER_JENV_KAKU2_ROTATED = 84
DMPAPER_JENV_KAKU3_ROTATED = 85
DMPAPER_JENV_CHOU3_ROTATED = 86
DMPAPER_JENV_CHOU4_ROTATED = 87
DMPAPER_B6_JIS = 88
DMPAPER_B6_JIS_ROTATED = 89
DMPAPER_12X11 = 90
DMPAPER_JENV_YOU4 = 91
DMPAPER_JENV_YOU4_ROTATED = 92
DMPAPER_P16K = 93
DMPAPER_P32K = 94
DMPAPER_P32KBIG = 95
DMPAPER_PENV_1 = 96
DMPAPER_PENV_2 = 97
DMPAPER_PENV_3 = 98
DMPAPER_PENV_4 = 99
DMPAPER_PENV_5 = 100
DMPAPER_PENV_6 = 101
DMPAPER_PENV_7 = 102
DMPAPER_PENV_8 = 103
DMPAPER_PENV_9 = 104
DMPAPER_PENV_10 = 105
DMPAPER_P16K_ROTATED = 106
DMPAPER_P32K_ROTATED = 107
DMPAPER_P32KBIG_ROTATED = 108
DMPAPER_PENV_1_ROTATED = 109
DMPAPER_PENV_2_ROTATED = 110
DMPAPER_PENV_3_ROTATED = 111
DMPAPER_PENV_4_ROTATED = 112
DMPAPER_PENV_5_ROTATED = 113
DMPAPER_PENV_6_ROTATED = 114
DMPAPER_PENV_7_ROTATED = 115
DMPAPER_PENV_8_ROTATED = 116
DMPAPER_PENV_9_ROTATED = 117
DMPAPER_PENV_10_ROTATED = 118
DMPAPER_LAST = DMPAPER_PENV_10_ROTATED
DMPAPER_USER = 256

DMBIN_UPPER = 1
DMBIN_ONLYONE = 1
DMBIN_LOWER = 2
DMBIN_MIDDLE = 3
DMBIN_MANUAL = 4
DMBIN_ENVELOPE = 5
DMBIN_ENVMANUAL = 6
DMBIN_AUTO = 7
DMBIN_TRACTOR = 8
DMBIN_SMALLFMT = 9
DMBIN_LARGEFMT = 10
DMBIN_LARGECAPACITY = 11
DMBIN_CASSETTE = 14
DMBIN_FORMSOURCE = 15
DMBIN_LAST = DMBIN_FORMSOURCE
DMBIN_USER = 256

DMRES_DRAFT = -1
DMRES_LOW = -2
DMRES_MEDIUM = -3
DMRES_HIGH = -4

DMCOLOR_MONOCHROME = 1
DMCOLOR_COLOR = 2

DMDUP_SIMPLEX = 1
DMDUP_VERTICAL = 2
DMDUP_HORIZONTAL = 3

DMTT_BITMAP = 1
DMTT_DOWNLOAD = 2
DMTT_SUBDEV = 3
DMTT_DOWNLOAD_OUTLINE = 4

DMCOLLATE_FALSE = 0
DMCOLLATE_TRUE = 1

DM_GRAYSCALE = 1
DM_INTERLACED = 2

DMICMMETHOD_NONE = 1
DMICMMETHOD_SYSTEM = 2
DMICMMETHOD_DRIVER = 3
DMICMMETHOD_DEVICE = 4
DMICMMETHOD_USER = 256

DMICM_SATURATE = 1
DMICM_CONTRAST = 2
DMICM_COLORIMETRIC = 3
DMICM_ABS_COLORIMETRIC = 4
DMICM_USER = 256

DMMEDIA_STANDARD = 1
DMMEDIA_TRANSPARENCY = 2
DMMEDIA_GLOSSY = 3
DMMEDIA_USER = 256

DMDITHER_NONE = 1
DMDITHER_COARSE = 2
DMDITHER_FINE = 3
DMDITHER_LINEART = 4
DMDITHER_ERRORDIFFUSION = 5
DMDITHER_RESERVED6 = 6
DMDITHER_RESERVED7 = 7
DMDITHER_RESERVED8 = 8
DMDITHER_RESERVED9 = 9
DMDITHER_GRAYSCALE = 10
DMDITHER_USER = 256

DMNUP_SYSTEM = 1
DMNUP_ONEUP = 2

FEATURESETTING_NUP = 0
FEATURESETTING_OUTPUT = 1
FEATURESETTING_PSLEVEL = 2
FEATURESETTING_CUSTPAPER = 3
FEATURESETTING_MIRROR = 4
FEATURESETTING_NEGATIVE = 5
FEATURESETTING_PROTOCOL = 6
FEATURESETTING_PRIVATE_BEGIN = 0x1000
FEATURESETTING_PRIVATE_END = 0x1FFF

RDH_RECTANGLES = 1
GGO_METRICS = 0
GGO_BITMAP = 1
GGO_NATIVE = 2
TT_POLYGON_TYPE = 24
TT_PRIM_LINE = 1
TT_PRIM_QSPLINE = 2
TT_AVAILABLE = 1
TT_ENABLED = 2
DM_UPDATE = 1
DM_COPY = 2
DM_PROMPT = 4
DM_MODIFY = 8
DM_IN_BUFFER = DM_MODIFY
DM_IN_PROMPT = DM_PROMPT
DM_OUT_BUFFER = DM_COPY
DM_OUT_DEFAULT = DM_UPDATE

DISPLAY_DEVICE_ATTACHED_TO_DESKTOP = 1
DISPLAY_DEVICE_MULTI_DRIVER = 2
DISPLAY_DEVICE_PRIMARY_DEVICE = 4
DISPLAY_DEVICE_MIRRORING_DRIVER = 8
DISPLAY_DEVICE_VGA_COMPATIBLE = 16
DISPLAY_DEVICE_REMOVABLE = 32
DISPLAY_DEVICE_MODESPRUNED = 134217728
DISPLAY_DEVICE_REMOTE = 67108864
DISPLAY_DEVICE_DISCONNECT = 33554432

DC_FIELDS = 1
DC_PAPERS = 2
DC_PAPERSIZE = 3
DC_MINEXTENT = 4
DC_MAXEXTENT = 5
DC_BINS = 6
DC_DUPLEX = 7
DC_SIZE = 8
DC_EXTRA = 9
DC_VERSION = 10
DC_DRIVER = 11
DC_BINNAMES = 12
DC_ENUMRESOLUTIONS = 13
DC_FILEDEPENDENCIES = 14
DC_TRUETYPE = 15
DC_PAPERNAMES = 16
DC_ORIENTATION = 17
DC_COPIES = 18
DC_BINADJUST = 19
DC_EMF_COMPLIANT = 20
DC_DATATYPE_PRODUCED = 21
DC_COLLATE = 22
DC_MANUFACTURER = 23
DC_MODEL = 24
DC_PERSONALITY = 25
DC_PRINTRATE = 26
DC_PRINTRATEUNIT = 27
DC_PRINTERMEM = 28
DC_MEDIAREADY = 29
DC_STAPLE = 30
DC_PRINTRATEPPM = 31
DC_COLORDEVICE = 32
DC_NUP = 33
DC_MEDIATYPENAMES = 34
DC_MEDIATYPES = 35

PRINTRATEUNIT_PPM = 1
PRINTRATEUNIT_CPS = 2
PRINTRATEUNIT_LPM = 3
PRINTRATEUNIT_IPM = 4

DCTT_BITMAP = 1
DCTT_DOWNLOAD = 2
DCTT_SUBDEV = 4
DCTT_DOWNLOAD_OUTLINE = 8

DCBA_FACEUPNONE = 0
DCBA_FACEUPCENTER = 1
DCBA_FACEUPLEFT = 2
DCBA_FACEUPRIGHT = 3
DCBA_FACEDOWNNONE = 256
DCBA_FACEDOWNCENTER = 257
DCBA_FACEDOWNLEFT = 258
DCBA_FACEDOWNRIGHT = 259

CA_NEGATIVE = 1
CA_LOG_FILTER = 2
ILLUMINANT_DEVICE_DEFAULT = 0
ILLUMINANT_A = 1
ILLUMINANT_B = 2
ILLUMINANT_C = 3
ILLUMINANT_D50 = 4
ILLUMINANT_D55 = 5
ILLUMINANT_D65 = 6
ILLUMINANT_D75 = 7
ILLUMINANT_F2 = 8
ILLUMINANT_MAX_INDEX = ILLUMINANT_F2
ILLUMINANT_TUNGSTEN = ILLUMINANT_A
ILLUMINANT_DAYLIGHT = ILLUMINANT_C
ILLUMINANT_FLUORESCENT = ILLUMINANT_F2
ILLUMINANT_NTSC = ILLUMINANT_C

FONTMAPPER_MAX = 10
ENHMETA_SIGNATURE = 1179469088
ENHMETA_STOCK_OBJECT = -2147483648
EMR_HEADER = 1
EMR_POLYBEZIER = 2
EMR_POLYGON = 3
EMR_POLYLINE = 4
EMR_POLYBEZIERTO = 5
EMR_POLYLINETO = 6
EMR_POLYPOLYLINE = 7
EMR_POLYPOLYGON = 8
EMR_SETWINDOWEXTEX = 9
EMR_SETWINDOWORGEX = 10
EMR_SETVIEWPORTEXTEX = 11
EMR_SETVIEWPORTORGEX = 12
EMR_SETBRUSHORGEX = 13
EMR_EOF = 14
EMR_SETPIXELV = 15
EMR_SETMAPPERFLAGS = 16
EMR_SETMAPMODE = 17
EMR_SETBKMODE = 18
EMR_SETPOLYFILLMODE = 19
EMR_SETROP2 = 20
EMR_SETSTRETCHBLTMODE = 21
EMR_SETTEXTALIGN = 22
EMR_SETCOLORADJUSTMENT = 23
EMR_SETTEXTCOLOR = 24
EMR_SETBKCOLOR = 25
EMR_OFFSETCLIPRGN = 26
EMR_MOVETOEX = 27
EMR_SETMETARGN = 28
EMR_EXCLUDECLIPRECT = 29
EMR_INTERSECTCLIPRECT = 30
EMR_SCALEVIEWPORTEXTEX = 31
EMR_SCALEWINDOWEXTEX = 32
EMR_SAVEDC = 33
EMR_RESTOREDC = 34
EMR_SETWORLDTRANSFORM = 35
EMR_MODIFYWORLDTRANSFORM = 36
EMR_SELECTOBJECT = 37
EMR_CREATEPEN = 38
EMR_CREATEBRUSHINDIRECT = 39
EMR_DELETEOBJECT = 40
EMR_ANGLEARC = 41
EMR_ELLIPSE = 42
EMR_RECTANGLE = 43
EMR_ROUNDRECT = 44
EMR_ARC = 45
EMR_CHORD = 46
EMR_PIE = 47
EMR_SELECTPALETTE = 48
EMR_CREATEPALETTE = 49
EMR_SETPALETTEENTRIES = 50
EMR_RESIZEPALETTE = 51
EMR_REALIZEPALETTE = 52
EMR_EXTFLOODFILL = 53
EMR_LINETO = 54
EMR_ARCTO = 55
EMR_POLYDRAW = 56
EMR_SETARCDIRECTION = 57
EMR_SETMITERLIMIT = 58
EMR_BEGINPATH = 59
EMR_ENDPATH = 60
EMR_CLOSEFIGURE = 61
EMR_FILLPATH = 62
EMR_STROKEANDFILLPATH = 63
EMR_STROKEPATH = 64
EMR_FLATTENPATH = 65
EMR_WIDENPATH = 66
EMR_SELECTCLIPPATH = 67
EMR_ABORTPATH = 68
EMR_GDICOMMENT = 70
EMR_FILLRGN = 71
EMR_FRAMERGN = 72
EMR_INVERTRGN = 73
EMR_PAINTRGN = 74
EMR_EXTSELECTCLIPRGN = 75
EMR_BITBLT = 76
EMR_STRETCHBLT = 77
EMR_MASKBLT = 78
EMR_PLGBLT = 79
EMR_SETDIBITSTODEVICE = 80
EMR_STRETCHDIBITS = 81
EMR_EXTCREATEFONTINDIRECTW = 82
EMR_EXTTEXTOUTA = 83
EMR_EXTTEXTOUTW = 84
EMR_POLYBEZIER16 = 85
EMR_POLYGON16 = 86
EMR_POLYLINE16 = 87
EMR_POLYBEZIERTO16 = 88
EMR_POLYLINETO16 = 89
EMR_POLYPOLYLINE16 = 90
EMR_POLYPOLYGON16 = 91
EMR_POLYDRAW16 = 92
EMR_CREATEMONOBRUSH = 93
EMR_CREATEDIBPATTERNBRUSHPT = 94
EMR_EXTCREATEPEN = 95
EMR_POLYTEXTOUTA = 96
EMR_POLYTEXTOUTW = 97
EMR_MIN = 1
EMR_MAX = 97

PANOSE_COUNT = 10
PAN_FAMILYTYPE_INDEX = 0
PAN_SERIFSTYLE_INDEX = 1
PAN_WEIGHT_INDEX = 2
PAN_PROPORTION_INDEX = 3
PAN_CONTRAST_INDEX = 4
PAN_STROKEVARIATION_INDEX = 5
PAN_ARMSTYLE_INDEX = 6
PAN_LETTERFORM_INDEX = 7
PAN_MIDLINE_INDEX = 8
PAN_XHEIGHT_INDEX = 9
PAN_CULTURE_LATIN = 0
PAN_ANY = 0
PAN_NO_FIT = 1
PAN_FAMILY_TEXT_DISPLAY = 2
PAN_FAMILY_SCRIPT = 3
PAN_FAMILY_DECORATIVE = 4
PAN_FAMILY_PICTORIAL = 5
PAN_SERIF_COVE = 2
PAN_SERIF_OBTUSE_COVE = 3
PAN_SERIF_SQUARE_COVE = 4
PAN_SERIF_OBTUSE_SQUARE_COVE = 5
PAN_SERIF_SQUARE = 6
PAN_SERIF_THIN = 7
PAN_SERIF_BONE = 8
PAN_SERIF_EXAGGERATED = 9
PAN_SERIF_TRIANGLE = 10
PAN_SERIF_NORMAL_SANS = 11
PAN_SERIF_OBTUSE_SANS = 12
PAN_SERIF_PERP_SANS = 13
PAN_SERIF_FLARED = 14
PAN_SERIF_ROUNDED = 15
PAN_WEIGHT_VERY_LIGHT = 2
PAN_WEIGHT_LIGHT = 3
PAN_WEIGHT_THIN = 4
PAN_WEIGHT_BOOK = 5
PAN_WEIGHT_MEDIUM = 6
PAN_WEIGHT_DEMI = 7
PAN_WEIGHT_BOLD = 8
PAN_WEIGHT_HEAVY = 9
PAN_WEIGHT_BLACK = 10
PAN_WEIGHT_NORD = 11
PAN_PROP_OLD_STYLE = 2
PAN_PROP_MODERN = 3
PAN_PROP_EVEN_WIDTH = 4
PAN_PROP_EXPANDED = 5
PAN_PROP_CONDENSED = 6
PAN_PROP_VERY_EXPANDED = 7
PAN_PROP_VERY_CONDENSED = 8
PAN_PROP_MONOSPACED = 9
PAN_CONTRAST_NONE = 2
PAN_CONTRAST_VERY_LOW = 3
PAN_CONTRAST_LOW = 4
PAN_CONTRAST_MEDIUM_LOW = 5
PAN_CONTRAST_MEDIUM = 6
PAN_CONTRAST_MEDIUM_HIGH = 7
PAN_CONTRAST_HIGH = 8
PAN_CONTRAST_VERY_HIGH = 9
PAN_STROKE_GRADUAL_DIAG = 2
PAN_STROKE_GRADUAL_TRAN = 3
PAN_STROKE_GRADUAL_VERT = 4
PAN_STROKE_GRADUAL_HORZ = 5
PAN_STROKE_RAPID_VERT = 6
PAN_STROKE_RAPID_HORZ = 7
PAN_STROKE_INSTANT_VERT = 8
PAN_STRAIGHT_ARMS_HORZ = 2
PAN_STRAIGHT_ARMS_WEDGE = 3
PAN_STRAIGHT_ARMS_VERT = 4
PAN_STRAIGHT_ARMS_SINGLE_SERIF = 5
PAN_STRAIGHT_ARMS_DOUBLE_SERIF = 6
PAN_BENT_ARMS_HORZ = 7
PAN_BENT_ARMS_WEDGE = 8
PAN_BENT_ARMS_VERT = 9
PAN_BENT_ARMS_SINGLE_SERIF = 10
PAN_BENT_ARMS_DOUBLE_SERIF = 11
PAN_LETT_NORMAL_CONTACT = 2
PAN_LETT_NORMAL_WEIGHTED = 3
PAN_LETT_NORMAL_BOXED = 4
PAN_LETT_NORMAL_FLATTENED = 5
PAN_LETT_NORMAL_ROUNDED = 6
PAN_LETT_NORMAL_OFF_CENTER = 7
PAN_LETT_NORMAL_SQUARE = 8
PAN_LETT_OBLIQUE_CONTACT = 9
PAN_LETT_OBLIQUE_WEIGHTED = 10
PAN_LETT_OBLIQUE_BOXED = 11
PAN_LETT_OBLIQUE_FLATTENED = 12
PAN_LETT_OBLIQUE_ROUNDED = 13
PAN_LETT_OBLIQUE_OFF_CENTER = 14
PAN_LETT_OBLIQUE_SQUARE = 15
PAN_MIDLINE_STANDARD_TRIMMED = 2
PAN_MIDLINE_STANDARD_POINTED = 3
PAN_MIDLINE_STANDARD_SERIFED = 4
PAN_MIDLINE_HIGH_TRIMMED = 5
PAN_MIDLINE_HIGH_POINTED = 6
PAN_MIDLINE_HIGH_SERIFED = 7
PAN_MIDLINE_CONSTANT_TRIMMED = 8
PAN_MIDLINE_CONSTANT_POINTED = 9
PAN_MIDLINE_CONSTANT_SERIFED = 10
PAN_MIDLINE_LOW_TRIMMED = 11
PAN_MIDLINE_LOW_POINTED = 12
PAN_MIDLINE_LOW_SERIFED = 13
PAN_XHEIGHT_CONSTANT_SMALL = 2
PAN_XHEIGHT_CONSTANT_STD = 3
PAN_XHEIGHT_CONSTANT_LARGE = 4
PAN_XHEIGHT_DUCKING_SMALL = 5
PAN_XHEIGHT_DUCKING_STD = 6
PAN_XHEIGHT_DUCKING_LARGE = 7
ELF_VENDOR_SIZE = 4
ELF_VERSION = 0
ELF_CULTURE_LATIN = 0
RASTER_FONTTYPE = 1
DEVICE_FONTTYPE = 2
TRUETYPE_FONTTYPE = 4

PC_RESERVED = 1
PC_EXPLICIT = 2
PC_NOCOLLAPSE = 4

TRANSPARENT = 1
OPAQUE = 2
BKMODE_LAST = 2
GM_COMPATIBLE = 1
GM_ADVANCED = 2
GM_LAST = 2
PT_CLOSEFIGURE = 1
PT_LINETO = 2
PT_BEZIERTO = 4
PT_MOVETO = 6
MM_TEXT = 1
MM_LOMETRIC = 2
MM_HIMETRIC = 3
MM_LOENGLISH = 4
MM_HIENGLISH = 5
MM_TWIPS = 6
MM_ISOTROPIC = 7
MM_ANISOTROPIC = 8
MM_MIN = MM_TEXT
MM_MAX = MM_ANISOTROPIC
MM_MAX_FIXEDSCALE = MM_TWIPS
ABSOLUTE = 1
RELATIVE = 2
WHITE_BRUSH = 0
LTGRAY_BRUSH = 1
GRAY_BRUSH = 2
DKGRAY_BRUSH = 3
BLACK_BRUSH = 4
NULL_BRUSH = 5
HOLLOW_BRUSH = NULL_BRUSH
WHITE_PEN = 6
BLACK_PEN = 7
NULL_PEN = 8
OEM_FIXED_FONT = 10
ANSI_FIXED_FONT = 11
ANSI_VAR_FONT = 12
SYSTEM_FONT = 13
DEVICE_DEFAULT_FONT = 14
DEFAULT_PALETTE = 15
SYSTEM_FIXED_FONT = 16
STOCK_LAST = 16
CLR_INVALID = -1

DC_BRUSH = 18
DC_PEN = 19

SPI_GETBEEP = 1
SPI_SETBEEP = 2
SPI_GETMOUSE = 3
SPI_SETMOUSE = 4
SPI_GETBORDER = 5
SPI_SETBORDER = 6
SPI_GETKEYBOARDSPEED = 10
SPI_SETKEYBOARDSPEED = 11
SPI_LANGDRIVER = 12
SPI_ICONHORIZONTALSPACING = 13
SPI_GETSCREENSAVETIMEOUT = 14
SPI_SETSCREENSAVETIMEOUT = 15
SPI_GETSCREENSAVEACTIVE = 16
SPI_SETSCREENSAVEACTIVE = 17
SPI_GETGRIDGRANULARITY = 18
SPI_SETGRIDGRANULARITY = 19
SPI_SETDESKWALLPAPER = 20
SPI_SETDESKPATTERN = 21
SPI_GETKEYBOARDDELAY = 22
SPI_SETKEYBOARDDELAY = 23
SPI_ICONVERTICALSPACING = 24
SPI_GETICONTITLEWRAP = 25
SPI_SETICONTITLEWRAP = 26
SPI_GETMENUDROPALIGNMENT = 27
SPI_SETMENUDROPALIGNMENT = 28
SPI_SETDOUBLECLKWIDTH = 29
SPI_SETDOUBLECLKHEIGHT = 30
SPI_GETICONTITLELOGFONT = 31
SPI_SETDOUBLECLICKTIME = 32
SPI_SETMOUSEBUTTONSWAP = 33
SPI_SETICONTITLELOGFONT = 34
SPI_GETFASTTASKSWITCH = 35
SPI_SETFASTTASKSWITCH = 36
SPI_SETDRAGFULLWINDOWS = 37
SPI_GETDRAGFULLWINDOWS = 38
SPI_GETNONCLIENTMETRICS = 41
SPI_SETNONCLIENTMETRICS = 42
SPI_GETMINIMIZEDMETRICS = 43
SPI_SETMINIMIZEDMETRICS = 44
SPI_GETICONMETRICS = 45
SPI_SETICONMETRICS = 46
SPI_SETWORKAREA = 47
SPI_GETWORKAREA = 48
SPI_SETPENWINDOWS = 49
SPI_GETFILTERKEYS = 50
SPI_SETFILTERKEYS = 51
SPI_GETTOGGLEKEYS = 52
SPI_SETTOGGLEKEYS = 53
SPI_GETMOUSEKEYS = 54
SPI_SETMOUSEKEYS = 55
SPI_GETSHOWSOUNDS = 56
SPI_SETSHOWSOUNDS = 57
SPI_GETSTICKYKEYS = 58
SPI_SETSTICKYKEYS = 59
SPI_GETACCESSTIMEOUT = 60
SPI_SETACCESSTIMEOUT = 61
SPI_GETSERIALKEYS = 62
SPI_SETSERIALKEYS = 63
SPI_GETSOUNDSENTRY = 64
SPI_SETSOUNDSENTRY = 65
SPI_GETHIGHCONTRAST = 66
SPI_SETHIGHCONTRAST = 67
SPI_GETKEYBOARDPREF = 68
SPI_SETKEYBOARDPREF = 69
SPI_GETSCREENREADER = 70
SPI_SETSCREENREADER = 71
SPI_GETANIMATION = 72
SPI_SETANIMATION = 73
SPI_GETFONTSMOOTHING = 74
SPI_SETFONTSMOOTHING = 75
SPI_SETDRAGWIDTH = 76
SPI_SETDRAGHEIGHT = 77
SPI_SETHANDHELD = 78
SPI_GETLOWPOWERTIMEOUT = 79
SPI_GETPOWEROFFTIMEOUT = 80
SPI_SETLOWPOWERTIMEOUT = 81
SPI_SETPOWEROFFTIMEOUT = 82
SPI_GETLOWPOWERACTIVE = 83
SPI_GETPOWEROFFACTIVE = 84
SPI_SETLOWPOWERACTIVE = 85
SPI_SETPOWEROFFACTIVE = 86
SPI_SETCURSORS = 87
SPI_SETICONS = 88
SPI_GETDEFAULTINPUTLANG = 89
SPI_SETDEFAULTINPUTLANG = 90
SPI_SETLANGTOGGLE = 91
SPI_GETWINDOWSEXTENSION = 92
SPI_SETMOUSETRAILS = 93
SPI_GETMOUSETRAILS = 94
SPI_GETSNAPTODEFBUTTON = 95
SPI_SETSNAPTODEFBUTTON = 96
SPI_SETSCREENSAVERRUNNING = 97
SPI_SCREENSAVERRUNNING = SPI_SETSCREENSAVERRUNNING
SPI_GETMOUSEHOVERWIDTH = 98
SPI_SETMOUSEHOVERWIDTH = 99
SPI_GETMOUSEHOVERHEIGHT = 100
SPI_SETMOUSEHOVERHEIGHT = 101
SPI_GETMOUSEHOVERTIME = 102
SPI_SETMOUSEHOVERTIME = 103
SPI_GETWHEELSCROLLLINES = 104
SPI_SETWHEELSCROLLLINES = 105
SPI_GETMENUSHOWDELAY = 106
SPI_SETMENUSHOWDELAY = 107

SPI_GETSHOWIMEUI = 110
SPI_SETSHOWIMEUI = 111
SPI_GETMOUSESPEED = 112
SPI_SETMOUSESPEED = 113
SPI_GETSCREENSAVERRUNNING = 114
SPI_GETDESKWALLPAPER = 115

SPI_GETACTIVEWINDOWTRACKING = 4096
SPI_SETACTIVEWINDOWTRACKING = 4097
SPI_GETMENUANIMATION = 4098
SPI_SETMENUANIMATION = 4099
SPI_GETCOMBOBOXANIMATION = 4100
SPI_SETCOMBOBOXANIMATION = 4101
SPI_GETLISTBOXSMOOTHSCROLLING = 4102
SPI_SETLISTBOXSMOOTHSCROLLING = 4103
SPI_GETGRADIENTCAPTIONS = 4104
SPI_SETGRADIENTCAPTIONS = 4105
SPI_GETKEYBOARDCUES = 4106
SPI_SETKEYBOARDCUES = 4107
SPI_GETMENUUNDERLINES = 4106
SPI_SETMENUUNDERLINES = 4107
SPI_GETACTIVEWNDTRKZORDER = 4108
SPI_SETACTIVEWNDTRKZORDER = 4109
SPI_GETHOTTRACKING = 4110
SPI_SETHOTTRACKING = 4111

SPI_GETMENUFADE = 4114
SPI_SETMENUFADE = 4115
SPI_GETSELECTIONFADE = 4116
SPI_SETSELECTIONFADE = 4117
SPI_GETTOOLTIPANIMATION = 4118
SPI_SETTOOLTIPANIMATION = 4119
SPI_GETTOOLTIPFADE = 4120
SPI_SETTOOLTIPFADE = 4121
SPI_GETCURSORSHADOW = 4122
SPI_SETCURSORSHADOW = 4123
SPI_GETMOUSESONAR = 4124
SPI_SETMOUSESONAR = 4125
SPI_GETMOUSECLICKLOCK = 4126
SPI_SETMOUSECLICKLOCK = 4127
SPI_GETMOUSEVANISH = 4128
SPI_SETMOUSEVANISH = 4129
SPI_GETFLATMENU = 4130
SPI_SETFLATMENU = 4131
SPI_GETDROPSHADOW = 4132
SPI_SETDROPSHADOW = 4133
SPI_GETBLOCKSENDINPUTRESETS = 4134
SPI_SETBLOCKSENDINPUTRESETS = 4135
SPI_GETUIEFFECTS = 4158
SPI_SETUIEFFECTS = 4159

SPI_GETFOREGROUNDLOCKTIMEOUT = 8192
SPI_SETFOREGROUNDLOCKTIMEOUT = 8193
SPI_GETACTIVEWNDTRKTIMEOUT = 8194
SPI_SETACTIVEWNDTRKTIMEOUT = 8195
SPI_GETFOREGROUNDFLASHCOUNT = 8196
SPI_SETFOREGROUNDFLASHCOUNT = 8197
SPI_GETCARETWIDTH = 8198
SPI_SETCARETWIDTH = 8199
SPI_GETMOUSECLICKLOCKTIME = 8200
SPI_SETMOUSECLICKLOCKTIME = 8201
SPI_GETFONTSMOOTHINGTYPE = 8202
SPI_SETFONTSMOOTHINGTYPE = 8203
SPI_GETFONTSMOOTHINGCONTRAST = 8204
SPI_SETFONTSMOOTHINGCONTRAST = 8205
SPI_GETFOCUSBORDERWIDTH = 8206
SPI_SETFOCUSBORDERWIDTH = 8207
SPI_GETFOCUSBORDERHEIGHT = 8208
SPI_SETFOCUSBORDERHEIGHT = 8209
SPI_GETFONTSMOOTHINGORIENTATION = 8210
SPI_SETFONTSMOOTHINGORIENTATION = 8211

SPIF_UPDATEINIFILE = 1
SPIF_SENDWININICHANGE = 2
SPIF_SENDCHANGE = SPIF_SENDWININICHANGE

FE_FONTSMOOTHINGSTANDARD = 1
FE_FONTSMOOTHINGCLEARTYPE = 2
FE_FONTSMOOTHINGDOCKING = 32768

DISP_CHANGE_SUCCESSFUL = 0
DISP_CHANGE_RESTART = 1
DISP_CHANGE_FAILED = -1
DISP_CHANGE_BADMODE = -2
DISP_CHANGE_NOTUPDATED = -3
DISP_CHANGE_BADFLAGS = -4
DISP_CHANGE_BADPARAM = -5
DISP_CHANGE_BADDUALVIEW = -6

BF_LEFT = 1
BF_TOP = 2
BF_RIGHT = 4
BF_BOTTOM = 8
BF_TOPLEFT = BF_TOP | BF_LEFT
BF_TOPRIGHT = BF_TOP | BF_RIGHT
BF_BOTTOMLEFT = BF_BOTTOM | BF_LEFT
BF_BOTTOMRIGHT = BF_BOTTOM | BF_RIGHT
BF_RECT = BF_LEFT | BF_TOP | BF_RIGHT | BF_BOTTOM
BF_DIAGONAL = 16
BF_DIAGONAL_ENDTOPRIGHT = BF_DIAGONAL | BF_TOP | BF_RIGHT
BF_DIAGONAL_ENDTOPLEFT = BF_DIAGONAL | BF_TOP | BF_LEFT
BF_DIAGONAL_ENDBOTTOMLEFT = BF_DIAGONAL | BF_BOTTOM | BF_LEFT
BF_DIAGONAL_ENDBOTTOMRIGHT = BF_DIAGONAL | BF_BOTTOM | BF_RIGHT
BF_MIDDLE = 2048
BF_SOFT = 4096
BF_ADJUST = 8192
BF_FLAT = 16384
BF_MONO = 32768
DFC_CAPTION = 1
DFC_MENU = 2
DFC_SCROLL = 3
DFC_BUTTON = 4
DFC_POPUPMENU = 5
DFCS_CAPTIONCLOSE = 0
DFCS_CAPTIONMIN = 1
DFCS_CAPTIONMAX = 2
DFCS_CAPTIONRESTORE = 3
DFCS_CAPTIONHELP = 4
DFCS_MENUARROW = 0
DFCS_MENUCHECK = 1
DFCS_MENUBULLET = 2
DFCS_MENUARROWRIGHT = 4
DFCS_SCROLLUP = 0
DFCS_SCROLLDOWN = 1
DFCS_SCROLLLEFT = 2
DFCS_SCROLLRIGHT = 3
DFCS_SCROLLCOMBOBOX = 5
DFCS_SCROLLSIZEGRIP = 8
DFCS_SCROLLSIZEGRIPRIGHT = 16
DFCS_BUTTONCHECK = 0
DFCS_BUTTONRADIOIMAGE = 1
DFCS_BUTTONRADIOMASK = 2
DFCS_BUTTONRADIO = 4
DFCS_BUTTON3STATE = 8
DFCS_BUTTONPUSH = 16
DFCS_INACTIVE = 256
DFCS_PUSHED = 512
DFCS_CHECKED = 1024
DFCS_TRANSPARENT = 2048
DFCS_HOT = 4096
DFCS_ADJUSTRECT = 8192
DFCS_FLAT = 16384
DFCS_MONO = 32768
DC_ACTIVE = 1
DC_SMALLCAP = 2
DC_ICON = 4
DC_TEXT = 8
DC_INBUTTON = 16
DC_GRADIENT = 32
IDANI_OPEN = 1
IDANI_CLOSE = 2
IDANI_CAPTION = 3
CF_TEXT = 1
CF_BITMAP = 2
CF_METAFILEPICT = 3
CF_SYLK = 4
CF_DIF = 5
CF_TIFF = 6
CF_OEMTEXT = 7
CF_DIB = 8
CF_PALETTE = 9
CF_PENDATA = 10
CF_RIFF = 11
CF_WAVE = 12
CF_UNICODETEXT = 13
CF_ENHMETAFILE = 14
CF_HDROP = 15
CF_LOCALE = 16
CF_DIBV5 = 17
CF_MAX = 18
CF_OWNERDISPLAY = 128
CF_DSPTEXT = 129
CF_DSPBITMAP = 130
CF_DSPMETAFILEPICT = 131
CF_DSPENHMETAFILE = 142
CF_PRIVATEFIRST = 512
CF_PRIVATELAST = 767
CF_GDIOBJFIRST = 768
CF_GDIOBJLAST = 1023
FVIRTKEY = 1
FNOINVERT = 2
FSHIFT = 4
FCONTROL = 8
FALT = 16
WPF_SETMINPOSITION = 1
WPF_RESTORETOMAXIMIZED = 2
ODT_MENU = 1
ODT_LISTBOX = 2
ODT_COMBOBOX = 3
ODT_BUTTON = 4
ODT_STATIC = 5
ODA_DRAWENTIRE = 1
ODA_SELECT = 2
ODA_FOCUS = 4
ODS_SELECTED = 1
ODS_GRAYED = 2
ODS_DISABLED = 4
ODS_CHECKED = 8
ODS_FOCUS = 16
ODS_DEFAULT = 32
ODS_COMBOBOXEDIT = 4096
ODS_HOTLIGHT = 64
ODS_INACTIVE = 128
PM_NOREMOVE = 0
PM_REMOVE = 1
PM_NOYIELD = 2
MOD_ALT = 1
MOD_CONTROL = 2
MOD_SHIFT = 4
MOD_WIN = 8
IDHOT_SNAPWINDOW = -1
IDHOT_SNAPDESKTOP = -2
ENDSESSION_LOGOFF = -2147483648
BSM_ALLCOMPONENTS = 0
BSM_VXDS = 1
BSM_NETDRIVER = 2
BSM_INSTALLABLEDRIVERS = 4
BSM_APPLICATIONS = 8
BSM_ALLDESKTOPS = 16
BSF_QUERY = 1
BSF_IGNORECURRENTTASK = 2
BSF_FLUSHDISK = 4
BSF_NOHANG = 8
BSF_POSTMESSAGE = 16
BSF_FORCEIFHUNG = 32
BSF_NOTIMEOUTIFNOTHUNG = 64
BROADCAST_QUERY_DENY = 1112363332  # Return this value to deny a query.

DBWF_LPARAMPOINTER = 32768

SWP_NOSIZE = 1
SWP_NOMOVE = 2
SWP_NOZORDER = 4
SWP_NOREDRAW = 8
SWP_NOACTIVATE = 16
SWP_FRAMECHANGED = 32
SWP_SHOWWINDOW = 64
SWP_HIDEWINDOW = 128
SWP_NOCOPYBITS = 256
SWP_NOOWNERZORDER = 512
SWP_NOSENDCHANGING = 1024
SWP_DRAWFRAME = SWP_FRAMECHANGED
SWP_NOREPOSITION = SWP_NOOWNERZORDER
SWP_DEFERERASE = 8192
SWP_ASYNCWINDOWPOS = 16384

DLGWINDOWEXTRA = 30
MWMO_WAITALL = 1
MWMO_ALERTABLE = 2
MWMO_INPUTAVAILABLE = 4
QS_KEY = 1
QS_MOUSEMOVE = 2
QS_MOUSEBUTTON = 4
QS_POSTMESSAGE = 8
QS_TIMER = 16
QS_PAINT = 32
QS_SENDMESSAGE = 64
QS_HOTKEY = 128
QS_MOUSE = QS_MOUSEMOVE | QS_MOUSEBUTTON
QS_INPUT = QS_MOUSE | QS_KEY
QS_ALLEVENTS = QS_INPUT | QS_POSTMESSAGE | QS_TIMER | QS_PAINT | QS_HOTKEY
QS_ALLINPUT = (
    QS_INPUT | QS_POSTMESSAGE | QS_TIMER | QS_PAINT | QS_HOTKEY | QS_SENDMESSAGE
    )

TA_NOUPDATECP = 0
TA_UPDATECP = 1
TA_LEFT = 0
TA_RIGHT = 2
TA_CENTER = 6
TA_TOP = 0
TA_BOTTOM = 8
TA_BASELINE = 24
TA_MASK = TA_BASELINE + TA_CENTER + TA_UPDATECP
VTA_BASELINE = TA_BASELINE
VTA_LEFT = TA_BOTTOM
VTA_RIGHT = TA_TOP
VTA_CENTER = TA_CENTER
VTA_BOTTOM = TA_RIGHT
VTA_TOP = TA_LEFT
ETO_GRAYED = 1
ETO_OPAQUE = 2
ETO_CLIPPED = 4
ASPECT_FILTERING = 1
DCB_RESET = 1
DCB_ACCUMULATE = 2
DCB_DIRTY = DCB_ACCUMULATE
DCB_SET = DCB_RESET | DCB_ACCUMULATE
DCB_ENABLE = 4
DCB_DISABLE = 8

FILE_BEGIN = 0
FILE_CURRENT = 1
FILE_END = 2
FILE_FLAG_WRITE_THROUGH = -2147483648
FILE_FLAG_OVERLAPPED = 1073741824
FILE_FLAG_NO_BUFFERING = 536870912
FILE_FLAG_RANDOM_ACCESS = 268435456
FILE_FLAG_SEQUENTIAL_SCAN = 134217728
FILE_FLAG_DELETE_ON_CLOSE = 67108864
FILE_FLAG_BACKUP_SEMANTICS = 33554432
FILE_FLAG_POSIX_SEMANTICS = 16777216
CREATE_NEW = 1
CREATE_ALWAYS = 2
OPEN_EXISTING = 3
OPEN_ALWAYS = 4
TRUNCATE_EXISTING = 5
PIPE_ACCESS_INBOUND = 1
PIPE_ACCESS_OUTBOUND = 2
PIPE_ACCESS_DUPLEX = 3
PIPE_CLIENT_END = 0
PIPE_SERVER_END = 1
PIPE_WAIT = 0
PIPE_NOWAIT = 1
PIPE_READMODE_BYTE = 0
PIPE_READMODE_MESSAGE = 2
PIPE_TYPE_BYTE = 0
PIPE_TYPE_MESSAGE = 4
PIPE_UNLIMITED_INSTANCES = 255

DEBUG_PROCESS = 0x00000001
DEBUG_ONLY_THIS_PROCESS = 0x00000002
CREATE_SUSPENDED = 0x00000004
DETACHED_PROCESS = 0x00000008
CREATE_NEW_CONSOLE = 0x00000010
NORMAL_PRIORITY_CLASS = 0x00000020
IDLE_PRIORITY_CLASS = 0x00000040
HIGH_PRIORITY_CLASS = 0x00000080
REALTIME_PRIORITY_CLASS = 0x00000100
CREATE_NEW_PROCESS_GROUP = 0x00000200
CREATE_UNICODE_ENVIRONMENT = 0x00000400
CREATE_SEPARATE_WOW_VDM = 0x00000800
CREATE_SHARED_WOW_VDM = 0x00001000
CREATE_FORCEDOS = 0x00002000
BELOW_NORMAL_PRIORITY_CLASS = 0x00004000
ABOVE_NORMAL_PRIORITY_CLASS = 0x00008000
INHERIT_PARENT_AFFINITY = 0x00010000
INHERIT_CALLER_PRIORITY = 0x00020000
CREATE_PROTECTED_PROCESS = 0x00040000
EXTENDED_STARTUPINFO_PRESENT = 0x00080000
PROCESS_MODE_BACKGROUND_BEGIN = 0x00100000
PROCESS_MODE_BACKGROUND_END = 0x00200000
CREATE_SECURE_PROCESS = 0x00400000

CREATE_BREAKAWAY_FROM_JOB = 0x01000000
CREATE_PRESERVE_CODE_AUTHZ_LEVEL = 0x02000000
CREATE_DEFAULT_ERROR_MODE = 0x04000000
CREATE_NO_WINDOW = 0x08000000
PROFILE_USER = 0x10000000
PROFILE_KERNEL = 0x20000000
PROFILE_SERVER = 0x40000000
CREATE_IGNORE_SYSTEM_DEFAULT = 0x80000000

THREAD_BASE_PRIORITY_LOWRT = 15
THREAD_BASE_PRIORITY_MAX = 2
THREAD_BASE_PRIORITY_MIN = -2
THREAD_BASE_PRIORITY_IDLE = -15
THREAD_PRIORITY_LOWEST = THREAD_BASE_PRIORITY_MIN
THREAD_PRIORITY_BELOW_NORMAL = THREAD_PRIORITY_LOWEST + 1
THREAD_PRIORITY_HIGHEST = THREAD_BASE_PRIORITY_MAX
THREAD_PRIORITY_ABOVE_NORMAL = THREAD_PRIORITY_HIGHEST - 1
THREAD_PRIORITY_ERROR_RETURN = MAXLONG
THREAD_PRIORITY_TIME_CRITICAL = THREAD_BASE_PRIORITY_LOWRT
THREAD_PRIORITY_IDLE = THREAD_BASE_PRIORITY_IDLE
THREAD_PRIORITY_NORMAL = 0
THREAD_MODE_BACKGROUND_BEGIN = 0x00010000
THREAD_MODE_BACKGROUND_END = 0x00020000

EXCEPTION_DEBUG_EVENT = 1
CREATE_THREAD_DEBUG_EVENT = 2
CREATE_PROCESS_DEBUG_EVENT = 3
EXIT_THREAD_DEBUG_EVENT = 4
EXIT_PROCESS_DEBUG_EVENT = 5
LOAD_DLL_DEBUG_EVENT = 6
UNLOAD_DLL_DEBUG_EVENT = 7
OUTPUT_DEBUG_STRING_EVENT = 8
RIP_EVENT = 9

PROCESS_HEAP_REGION = 1
PROCESS_HEAP_UNCOMMITTED_RANGE = 2
PROCESS_HEAP_ENTRY_BUSY = 4
PROCESS_HEAP_ENTRY_MOVEABLE = 16
PROCESS_HEAP_ENTRY_DDESHARE = 32
SCS_32BIT_BINARY = 0
SCS_DOS_BINARY = 1
SCS_WOW_BINARY = 2
SCS_PIF_BINARY = 3
SCS_POSIX_BINARY = 4
SCS_OS216_BINARY = 5
SEM_FAILCRITICALERRORS = 1
SEM_NOGPFAULTERRORBOX = 2
SEM_NOALIGNMENTFAULTEXCEPT = 4
SEM_NOOPENFILEERRORBOX = 32768
LOCKFILE_FAIL_IMMEDIATELY = 1
LOCKFILE_EXCLUSIVE_LOCK = 2
HANDLE_FLAG_INHERIT = 1
HANDLE_FLAG_PROTECT_FROM_CLOSE = 2
HINSTANCE_ERROR = 32
GET_TAPE_MEDIA_INFORMATION = 0
GET_TAPE_DRIVE_INFORMATION = 1
SET_TAPE_MEDIA_INFORMATION = 0
SET_TAPE_DRIVE_INFORMATION = 1
FORMAT_MESSAGE_ALLOCATE_BUFFER = 256
FORMAT_MESSAGE_IGNORE_INSERTS = 512
FORMAT_MESSAGE_FROM_STRING = 1024
FORMAT_MESSAGE_FROM_HMODULE = 2048
FORMAT_MESSAGE_FROM_SYSTEM = 4096
FORMAT_MESSAGE_ARGUMENT_ARRAY = 8192
FORMAT_MESSAGE_MAX_WIDTH_MASK = 255
BACKUP_INVALID = 0
BACKUP_DATA = 1
BACKUP_EA_DATA = 2
BACKUP_SECURITY_DATA = 3
BACKUP_ALTERNATE_DATA = 4
BACKUP_LINK = 5
BACKUP_PROPERTY_DATA = 6
BACKUP_OBJECT_ID = 7
BACKUP_REPARSE_DATA = 8
BACKUP_SPARSE_BLOCK = 9

STREAM_NORMAL_ATTRIBUTE = 0
STREAM_MODIFIED_WHEN_READ = 1
STREAM_CONTAINS_SECURITY = 2
STREAM_CONTAINS_PROPERTIES = 4
STARTF_USESHOWWINDOW = 1
STARTF_USESIZE = 2
STARTF_USEPOSITION = 4
STARTF_USECOUNTCHARS = 8
STARTF_USEFILLATTRIBUTE = 16
STARTF_FORCEONFEEDBACK = 64
STARTF_FORCEOFFFEEDBACK = 128
STARTF_USESTDHANDLES = 256
STARTF_USEHOTKEY = 512
SHUTDOWN_NORETRY = 1
DDD_RAW_TARGET_PATH = 1
DDD_REMOVE_DEFINITION = 2
DDD_EXACT_MATCH_ON_REMOVE = 4
MOVEFILE_REPLACE_EXISTING = 1
MOVEFILE_COPY_ALLOWED = 2
MOVEFILE_DELAY_UNTIL_REBOOT = 4
MAX_COMPUTERNAME_LENGTH = 15
LOGON32_LOGON_INTERACTIVE = 2
LOGON32_LOGON_NETWORK = 3
LOGON32_LOGON_BATCH = 4
LOGON32_LOGON_SERVICE = 5
LOGON32_LOGON_UNLOCK = 7
LOGON32_LOGON_NETWORK_CLEARTEXT = 8
LOGON32_LOGON_NEW_CREDENTIALS = 9
LOGON32_PROVIDER_DEFAULT = 0
LOGON32_PROVIDER_WINNT35 = 1
LOGON32_PROVIDER_WINNT40 = 2
LOGON32_PROVIDER_WINNT50 = 3
VER_PLATFORM_WIN32s = 0
VER_PLATFORM_WIN32_WINDOWS = 1
VER_PLATFORM_WIN32_NT = 2
TC_NORMAL = 0
TC_HARDERR = 1
TC_GP_TRAP = 2
TC_SIGNAL = 3
AC_LINE_OFFLINE = 0
AC_LINE_ONLINE = 1
AC_LINE_BACKUP_POWER = 2
AC_LINE_UNKNOWN = 255
BATTERY_FLAG_HIGH = 1
BATTERY_FLAG_LOW = 2
BATTERY_FLAG_CRITICAL = 4
BATTERY_FLAG_CHARGING = 8
BATTERY_FLAG_NO_BATTERY = 128
BATTERY_FLAG_UNKNOWN = 255
BATTERY_PERCENTAGE_UNKNOWN = 255
BATTERY_LIFE_UNKNOWN = -1

CTRL_C_EVENT = 0
CTRL_BREAK_EVENT = 1
CTRL_CLOSE_EVENT = 2
CTRL_LOGOFF_EVENT = 5
CTRL_SHUTDOWN_EVENT = 6

MOUSE_MOVED = 1
DOUBLE_CLICK = 2
MOUSE_WHEELED = 4

NameUnknown = 0
NameFullyQualifiedDN = 1
NameSamCompatible = 2
NameDisplay = 3
NameUniqueId = 6
NameCanonical = 7
NameUserPrincipal = 8
NameCanonicalEx = 9
NameServicePrincipal = 10
NameDnsDomain = 12

ComputerNameNetBIOS = 0
ComputerNameDnsHostname = 1
ComputerNameDnsDomain = 2
ComputerNameDnsFullyQualified = 3
ComputerNamePhysicalNetBIOS = 4
ComputerNamePhysicalDnsHostname = 5
ComputerNamePhysicalDnsDomain = 6
ComputerNamePhysicalDnsFullyQualified = 7

TRUE = 1
FALSE = 0
MAX_PATH = 260

AC_SRC_OVER = 0
AC_SRC_ALPHA = 1
GRADIENT_FILL_RECT_H = 0
GRADIENT_FILL_RECT_V = 1
GRADIENT_FILL_TRIANGLE = 2
GRADIENT_FILL_OP_FLAG = 255

VOLUME_NAME_DOS = 0
VOLUME_NAME_GUID = 1
VOLUME_NAME_NT = 2
VOLUME_NAME_NONE = 4
FILE_NAME_NORMALIZED = 0
FILE_NAME_OPENED = 8

DEVICE_NOTIFY_WINDOW_HANDLE = 0x00000000
DEVICE_NOTIFY_SERVICE_HANDLE = 0x00000001

SEE_MASK_DEFAULT = 0x00000000
SEE_MASK_CLASSNAME = 0x00000001
SEE_MASK_CLASSKEY = 0x00000003
SEE_MASK_IDLIST = 0x00000004
SEE_MASK_INVOKEIDLIST = 0x0000000C
SEE_MASK_ICON = 0x00000010
SEE_MASK_HOTKEY = 0x00000020
SEE_MASK_NOCLOSEPROCESS = 0x00000040
SEE_MASK_CONNECTNETDRV = 0x00000080
SEE_MASK_NOASYNC = 0x00000100
SEE_MASK_FLAG_DDEWAIT = SEE_MASK_NOASYNC
SEE_MASK_DOENVSUBST = 0x00000200
SEE_MASK_FLAG_NO_UI = 0x00000400
SEE_MASK_UNICODE = 0x00004000
SEE_MASK_NO_CONSOLE = 0x00008000
SEE_MASK_ASYNCOK = 0x00100000
SEE_MASK_HMONITOR = 0x00200000
SEE_MASK_NOZONECHECKS = 0x00800000
SEE_MASK_NOQUERYCLASSSTORE = 0x01000000
SEE_MASK_WAITFORINPUTIDLE = 0x02000000
SEE_MASK_FLAG_LOG_USAGE = 0x04000000
SEE_MASK_FLAG_HINST_IS_SITE = 0x08000000

GWL_WNDPROC = -4
GWL_HINSTANCE = -6
GWL_HWNDPARENT = -8
GWL_STYLE = -16
GWL_EXSTYLE = -20
GWL_USERDATA = -21
GWL_ID = -12

GWLP_WNDPROC = -4
GWLP_HINSTANCE = -6
GWLP_HWNDPARENT = -8
GWLP_USERDATA = -21
GWLP_ID = -12

DWLP_MSGRESULT = 0

GCL_MENUNAME = -8
GCL_HBRBACKGROUND = -10
GCL_HCURSOR = -12
GCL_HICON = -14
GCL_HMODULE = -16
GCL_CBWNDEXTRA = -18
GCL_CBCLSEXTRA = -20
GCL_WNDPROC = -24
GCL_STYLE = -26
GCW_ATOM = -32
GCL_HICONSM = -34

GCLP_MENUNAME = -8
GCLP_HBRBACKGROUND = -10
GCLP_HCURSOR = -12
GCLP_HICON = -14
GCLP_HMODULE = -16
GCLP_WNDPROC = -24
GCLP_HICONSM = -34

GW_HWNDFIRST = 0
GW_HWNDLAST = 1
GW_HWNDNEXT = 2
GW_HWNDPREV = 3
GW_OWNER = 4
GW_CHILD = 5
GW_ENABLEDPOPUP = 6
GW_MAX = 6

VK_LBUTTON = 0x01
VK_RBUTTON = 0x02
VK_CANCEL = 0x03
VK_MBUTTON = 0x04
VK_XBUTTON1 = 0x05
VK_XBUTTON2 = 0x06
VK_BACK = 0x08
VK_TAB = 0x09
VK_CLEAR = 0x0C
VK_RETURN = 0x0D
VK_SHIFT = 0x10
VK_CONTROL = 0x11
VK_MENU = 0x12
VK_PAUSE = 0x13
VK_CAPITAL = 0x14
VK_KANA = 0x15
VK_HANGEUL = 0x15
VK_HANGUL = 0x15
VK_JUNJA = 0x17
VK_FINAL = 0x18
VK_HANJA = 0x19
VK_KANJI = 0x19
VK_ESCAPE = 0x1B
VK_CONVERT = 0x1C
VK_NONCONVERT = 0x1D
VK_ACCEPT = 0x1E
VK_MODECHANGE = 0x1F
VK_SPACE = 0x20
VK_PRIOR = 0x21
VK_NEXT = 0x22
VK_END = 0x23
VK_HOME = 0x24
VK_LEFT = 0x25
VK_UP = 0x26
VK_RIGHT = 0x27
VK_DOWN = 0x28
VK_SELECT = 0x29
VK_PRINT = 0x2A
VK_EXECUTE = 0x2B
VK_SNAPSHOT = 0x2C
VK_INSERT = 0x2D
VK_DELETE = 0x2E
VK_HELP = 0x2F
VK_LWIN = 0x5B
VK_RWIN = 0x5C
VK_APPS = 0x5D
VK_SLEEP = 0x5F
VK_NUMPAD0 = 0x60
VK_NUMPAD1 = 0x61
VK_NUMPAD2 = 0x62
VK_NUMPAD3 = 0x63
VK_NUMPAD4 = 0x64
VK_NUMPAD5 = 0x65
VK_NUMPAD6 = 0x66
VK_NUMPAD7 = 0x67
VK_NUMPAD8 = 0x68
VK_NUMPAD9 = 0x69
VK_MULTIPLY = 0x6A
VK_ADD = 0x6B
VK_SEPARATOR = 0x6C
VK_SUBTRACT = 0x6D
VK_DECIMAL = 0x6E
VK_DIVIDE = 0x6F
VK_F1 = 0x70
VK_F2 = 0x71
VK_F3 = 0x72
VK_F4 = 0x73
VK_F5 = 0x74
VK_F6 = 0x75
VK_F7 = 0x76
VK_F8 = 0x77
VK_F9 = 0x78
VK_F10 = 0x79
VK_F11 = 0x7A
VK_F12 = 0x7B
VK_F13 = 0x7C
VK_F14 = 0x7D
VK_F15 = 0x7E
VK_F16 = 0x7F
VK_F17 = 0x80
VK_F18 = 0x81
VK_F19 = 0x82
VK_F20 = 0x83
VK_F21 = 0x84
VK_F22 = 0x85
VK_F23 = 0x86
VK_F24 = 0x87
VK_NAVIGATION_VIEW = 0x88
VK_NAVIGATION_MENU = 0x89
VK_NAVIGATION_UP = 0x8A
VK_NAVIGATION_DOWN = 0x8B
VK_NAVIGATION_LEFT = 0x8C
VK_NAVIGATION_RIGHT = 0x8D
VK_NAVIGATION_ACCEPT = 0x8E
VK_NAVIGATION_CANCEL = 0x8F
VK_NUMLOCK = 0x90
VK_SCROLL = 0x91
VK_OEM_NEC_EQUAL = 0x92
VK_OEM_FJ_JISHO = 0x92
VK_OEM_FJ_MASSHOU = 0x93
VK_OEM_FJ_TOUROKU = 0x94
VK_OEM_FJ_LOYA = 0x95
VK_OEM_FJ_ROYA = 0x96
VK_LSHIFT = 0xA0
VK_RSHIFT = 0xA1
VK_LCONTROL = 0xA2
VK_RCONTROL = 0xA3
VK_LMENU = 0xA4
VK_RMENU = 0xA5
VK_BROWSER_BACK = 0xA6
VK_BROWSER_FORWARD = 0xA7
VK_BROWSER_REFRESH = 0xA8
VK_BROWSER_STOP = 0xA9
VK_BROWSER_SEARCH = 0xAA
VK_BROWSER_FAVORITES = 0xAB
VK_BROWSER_HOME = 0xAC
VK_VOLUME_MUTE = 0xAD
VK_VOLUME_DOWN = 0xAE
VK_VOLUME_UP = 0xAF
VK_MEDIA_NEXT_TRACK = 0xB0
VK_MEDIA_PREV_TRACK = 0xB1
VK_MEDIA_STOP = 0xB2
VK_MEDIA_PLAY_PAUSE = 0xB3
VK_LAUNCH_MAIL = 0xB4
VK_LAUNCH_MEDIA_SELECT = 0xB5
VK_LAUNCH_APP1 = 0xB6
VK_LAUNCH_APP2 = 0xB7
VK_OEM_1 = 0xBA
VK_OEM_PLUS = 0xBB
VK_OEM_COMMA = 0xBC
VK_OEM_MINUS = 0xBD
VK_OEM_PERIOD = 0xBE
VK_OEM_2 = 0xBF
VK_OEM_3 = 0xC0
VK_GAMEPAD_A = 0xC3
VK_GAMEPAD_B = 0xC4
VK_GAMEPAD_X = 0xC5
VK_GAMEPAD_Y = 0xC6
VK_GAMEPAD_RIGHT_SHOULDER = 0xC7
VK_GAMEPAD_LEFT_SHOULDER = 0xC8
VK_GAMEPAD_LEFT_TRIGGER = 0xC9
VK_GAMEPAD_RIGHT_TRIGGER = 0xCA
VK_GAMEPAD_DPAD_UP = 0xCB
VK_GAMEPAD_DPAD_DOWN = 0xCC
VK_GAMEPAD_DPAD_LEFT = 0xCD
VK_GAMEPAD_DPAD_RIGHT = 0xCE
VK_GAMEPAD_MENU = 0xCF
VK_GAMEPAD_VIEW = 0xD0
VK_GAMEPAD_LEFT_THUMBSTICK_BUTTON = 0xD1
VK_GAMEPAD_RIGHT_THUMBSTICK_BUTTON = 0xD2
VK_GAMEPAD_LEFT_THUMBSTICK_UP = 0xD3
VK_GAMEPAD_LEFT_THUMBSTICK_DOWN = 0xD4
VK_GAMEPAD_LEFT_THUMBSTICK_RIGHT = 0xD5
VK_GAMEPAD_LEFT_THUMBSTICK_LEFT = 0xD6
VK_GAMEPAD_RIGHT_THUMBSTICK_UP = 0xD7
VK_GAMEPAD_RIGHT_THUMBSTICK_DOWN = 0xD8
VK_GAMEPAD_RIGHT_THUMBSTICK_RIGHT = 0xD9
VK_GAMEPAD_RIGHT_THUMBSTICK_LEFT = 0xDA
VK_OEM_4 = 0xDB
VK_OEM_5 = 0xDC
VK_OEM_6 = 0xDD
VK_OEM_7 = 0xDE
VK_OEM_8 = 0xDF
VK_OEM_AX = 0xE1
VK_OEM_102 = 0xE2
VK_ICO_HELP = 0xE3
VK_ICO_00 = 0xE4
VK_PROCESSKEY = 0xE5
VK_ICO_CLEAR = 0xE6
VK_PACKET = 0xE7
VK_OEM_RESET = 0xE9
VK_OEM_JUMP = 0xEA
VK_OEM_PA1 = 0xEB
VK_OEM_PA2 = 0xEC
VK_OEM_PA3 = 0xED
VK_OEM_WSCTRL = 0xEE
VK_OEM_CUSEL = 0xEF
VK_OEM_ATTN = 0xF0
VK_OEM_FINISH = 0xF1
VK_OEM_COPY = 0xF2
VK_OEM_AUTO = 0xF3
VK_OEM_ENLW = 0xF4
VK_OEM_BACKTAB = 0xF5
VK_ATTN = 0xF6
VK_CRSEL = 0xF7
VK_EXSEL = 0xF8
VK_EREOF = 0xF9
VK_PLAY = 0xFA
VK_ZOOM = 0xFB
VK_NONAME = 0xFC
VK_PA1 = 0xFD
VK_OEM_CLEAR = 0xFE

CRED_TYPE_GENERIC = 1
CRED_TYPE_DOMAIN_PASSWORD = 2
CRED_TYPE_DOMAIN_CERTIFICATE = 3
CRED_TYPE_DOMAIN_VISIBLE_PASSWORD = 4
CRED_TYPE_GENERIC_CERTIFICATE = 5
CRED_TYPE_DOMAIN_EXTENDED = 6
CRED_TYPE_MAXIMUM = 7
CRED_TYPE_MAXIMUM_EX = CRED_TYPE_MAXIMUM + 1000

SC_SIZE = 0xF000
SC_MOVE = 0xF010
SC_MINIMIZE = 0xF020
SC_MAXIMIZE = 0xF030
SC_NEXTWINDOW = 0xF040
SC_PREVWINDOW = 0xF050
SC_CLOSE = 0xF060
SC_VSCROLL = 0xF070
SC_HSCROLL = 0xF080
SC_MOUSEMENU = 0xF090
SC_KEYMENU = 0xF100
SC_ARRANGE = 0xF110
SC_RESTORE = 0xF120
SC_TASKLIST = 0xF130
SC_SCREENSAVE = 0xF140
SC_HOTKEY = 0xF150
SC_DEFAULT = 0xF160
SC_MONITORPOWER = 0xF170
SC_CONTEXTHELP = 0xF180
SC_SEPARATOR = 0xF00F

INVALID_FILE_ATTRIBUTES = -1
FILE_ATTRIBUTE_READONLY = 0x00000001
FILE_ATTRIBUTE_HIDDEN = 0x00000002
FILE_ATTRIBUTE_SYSTEM = 0x00000004
FILE_ATTRIBUTE_DIRECTORY = 0x00000010
FILE_ATTRIBUTE_ARCHIVE = 0x00000020
FILE_ATTRIBUTE_DEVICE = 0x00000040
FILE_ATTRIBUTE_NORMAL = 0x00000080
FILE_ATTRIBUTE_TEMPORARY = 0x00000100
FILE_ATTRIBUTE_SPARSE_FILE = 0x00000200
FILE_ATTRIBUTE_REPARSE_POINT = 0x00000400
FILE_ATTRIBUTE_COMPRESSED = 0x00000800
FILE_ATTRIBUTE_OFFLINE = 0x00001000
FILE_ATTRIBUTE_NOT_CONTENT_INDEXED = 0x00002000
FILE_ATTRIBUTE_ENCRYPTED = 0x00004000
FILE_ATTRIBUTE_INTEGRITY_STREAM = 0x00008000
FILE_ATTRIBUTE_VIRTUAL = 0x00010000
FILE_ATTRIBUTE_NO_SCRUB_DATA = 0x00020000
FILE_ATTRIBUTE_EA = 0x00040000
FILE_ATTRIBUTE_PINNED = 0x00080000
FILE_ATTRIBUTE_UNPINNED = 0x00100000
FILE_ATTRIBUTE_RECALL_ON_OPEN = 0x00040000
FILE_ATTRIBUTE_RECALL_ON_DATA_ACCESS = 0x00400000

# Object attributes

OBJ_INHERIT = 0x00000002
OBJ_PERMANENT = 0x00000010
OBJ_EXCLUSIVE = 0x00000020
OBJ_CASE_INSENSITIVE = 0x00000040
OBJ_OPENIF = 0x00000080
OBJ_OPENLINK = 0x00000100
OBJ_KERNEL_HANDLE = 0x00000200
OBJ_FORCE_ACCESS_CHECK = 0x00000400
OBJ_IGNORE_IMPERSONATED_DEVICEMAP = 0x00000800
OBJ_DONT_REPARSE = 0x00001000
OBJ_VALID_ATTRIBUTES = 0x00001FF2

# Thread execution states

ES_SYSTEM_REQUIRED = 0x00000001
ES_DISPLAY_REQUIRED = 0x00000002
ES_USER_PRESENT = 0x00000004
ES_AWAYMODE_REQUIRED = 0x00000040
ES_CONTINUOUS = 0x80000000

# SYSTEM_INFORMATION_CLASS
SystemBasicInformation = 0
SystemProcessorInformation = 1
SystemPerformanceInformation = 2
SystemTimeOfDayInformation = 3
SystemPathInformation = 4
SystemProcessInformation = 5
SystemCallCountInformation = 6
SystemDeviceInformation = 7
SystemProcessorPerformanceInformation = 8
SystemFlagsInformation = 9
SystemCallTimeInformation = 10
SystemModuleInformation = 11
SystemLocksInformation = 12
SystemStackTraceInformation = 13
SystemPagedPoolInformation = 14
SystemNonPagedPoolInformation = 15
SystemHandleInformation = 16
SystemObjectInformation = 17
SystemPageFileInformation = 18
SystemVdmInstemulInformation = 19
SystemVdmBopInformation = 20
SystemFileCacheInformation = 21
SystemPoolTagInformation = 22
SystemInterruptInformation = 23
SystemDpcBehaviorInformation = 24
SystemFullMemoryInformation = 25
SystemLoadGdiDriverInformation = 26
SystemUnloadGdiDriverInformation = 27
SystemTimeAdjustmentInformation = 28
SystemSummaryMemoryInformation = 29
SystemMirrorMemoryInformation = 30
SystemPerformanceTraceInformation = 31
SystemObsolete0 = 32
SystemExceptionInformation = 33
SystemCrashDumpStateInformation = 34
SystemKernelDebuggerInformation = 35
SystemContextSwitchInformation = 36
SystemRegistryQuotaInformation = 37
SystemExtendServiceTableInformation = 38
SystemPrioritySeperation = 39
SystemVerifierAddDriverInformation = 40
SystemVerifierRemoveDriverInformation = 41
SystemProcessorIdleInformation = 42
SystemLegacyDriverInformation = 43
SystemCurrentTimeZoneInformation = 44
SystemLookasideInformation = 45
SystemTimeSlipNotification = 46
SystemSessionCreate = 47
SystemSessionDetach = 48
SystemSessionInformation = 49
SystemRangeStartInformation = 50
SystemVerifierInformation = 51
SystemVerifierThunkExtend = 52
SystemSessionProcessInformation = 53
SystemLoadGdiDriverInSystemSpace = 54
SystemNumaProcessorMap = 55
SystemPrefetcherInformation = 56
SystemExtendedProcessInformation = 57
SystemRecommendedSharedDataAlignment = 58
SystemComPlusPackage = 59
SystemNumaAvailableMemory = 60
SystemProcessorPowerInformation = 61
SystemEmulationBasicInformation = 62
SystemEmulationProcessorInformation = 63
SystemExtendedHandleInformation = 64
SystemLostDelayedWriteInformation = 65
SystemBigPoolInformation = 66
SystemSessionPoolTagInformation = 67
SystemSessionMappedViewInformation = 68
SystemHotpatchInformation = 69
SystemObjectSecurityMode = 70
SystemWatchdogTimerHandler = 71
SystemWatchdogTimerInformation = 72
SystemLogicalProcessorInformation = 73
SystemWow64SharedInformationObsolete = 74
SystemRegisterFirmwareTableInformationHandler = 75
SystemFirmwareTableInformation = 76
SystemModuleInformationEx = 77
SystemVerifierTriageInformation = 78
SystemSuperfetchInformation = 79
SystemMemoryListInformation = 80
SystemFileCacheInformationEx = 81
SystemThreadPriorityClientIdInformation = 82
SystemProcessorIdleCycleTimeInformation = 83
SystemVerifierCancellationInformation = 84
SystemProcessorPowerInformationEx = 85
SystemRefTraceInformation = 86
SystemSpecialPoolInformation = 87
SystemProcessIdInformation = 88
SystemErrorPortInformation = 89
SystemBootEnvironmentInformation = 90
SystemHypervisorInformation = 91
SystemVerifierInformationEx = 92
SystemTimeZoneInformation = 93
SystemImageFileExecutionOptionsInformation = 94
SystemCoverageInformation = 95
SystemPrefetchPatchInformation = 96
SystemVerifierFaultsInformation = 97
SystemSystemPartitionInformation = 98
SystemSystemDiskInformation = 99
SystemProcessorPerformanceDistribution = 100
SystemNumaProximityNodeInformation = 101
SystemDynamicTimeZoneInformation = 102
SystemCodeIntegrityInformation = 103
SystemProcessorMicrocodeUpdateInformation = 104
SystemProcessorBrandString = 105
SystemVirtualAddressInformation = 106
SystemLogicalProcessorAndGroupInformation = 107
SystemProcessorCycleTimeInformation = 108
SystemStoreInformation = 109
SystemRegistryAppendString = 110
SystemAitSamplingValue = 111
SystemVhdBootInformation = 112
SystemCpuQuotaInformation = 113
SystemNativeBasicInformation = 114
SystemErrorPortTimeouts = 115
SystemLowPriorityIoInformation = 116
SystemTpmBootEntropyInformation = 117
SystemVerifierCountersInformation = 118
SystemPagedPoolInformationEx = 119
SystemSystemPtesInformationEx = 120
SystemNodeDistanceInformation = 121
SystemAcpiAuditInformation = 122
SystemBasicPerformanceInformation = 123
SystemQueryPerformanceCounterInformation = 124
SystemSessionBigPoolInformation = 125
SystemBootGraphicsInformation = 126
SystemScrubPhysicalMemoryInformation = 127
SystemBadPageInformation = 128
SystemProcessorProfileControlArea = 129
SystemCombinePhysicalMemoryInformation = 130
SystemEntropyInterruptTimingInformation = 131
SystemConsoleInformation = 132
SystemPlatformBinaryInformation = 133
SystemPolicyInformation = 134
SystemHypervisorProcessorCountInformation = 135
SystemDeviceDataInformation = 136
SystemDeviceDataEnumerationInformation = 137
SystemMemoryTopologyInformation = 138
SystemMemoryChannelInformation = 139
SystemBootLogoInformation = 140
SystemProcessorPerformanceInformationEx = 141
SystemCriticalProcessErrorLogInformation = 142
SystemSecureBootPolicyInformation = 143
SystemPageFileInformationEx = 144
SystemSecureBootInformation = 145
SystemEntropyInterruptTimingRawInformation = 146
SystemPortableWorkspaceEfiLauncherInformation = 147
SystemFullProcessInformation = 148
SystemKernelDebuggerInformationEx = 149
SystemBootMetadataInformation = 150
SystemSoftRebootInformation = 151
SystemElamCertificateInformation = 152
SystemOfflineDumpConfigInformation = 153
SystemProcessorFeaturesInformation = 154
SystemRegistryReconciliationInformation = 155
SystemEdidInformation = 156
SystemManufacturingInformation = 157
SystemEnergyEstimationConfigInformation = 158
SystemHypervisorDetailInformation = 159
SystemProcessorCycleStatsInformation = 160
SystemVmGenerationCountInformation = 161
SystemTrustedPlatformModuleInformation = 162
SystemKernelDebuggerFlags = 163
SystemCodeIntegrityPolicyInformation = 164
SystemIsolatedUserModeInformation = 165
SystemHardwareSecurityTestInterfaceResultsInformation = 166
SystemSingleModuleInformation = 167
SystemAllowedCpuSetsInformation = 168
SystemVsmProtectionInformation = 169
SystemInterruptCpuSetsInformation = 170
SystemSecureBootPolicyFullInformation = 171
SystemCodeIntegrityPolicyFullInformation = 172
SystemAffinitizedInterruptProcessorInformation = 173
SystemRootSiloInformation = 174
SystemCpuSetInformation = 175
SystemCpuSetTagInformation = 176
SystemWin32WerStartCallout = 177
SystemSecureKernelProfileInformation = 178
SystemCodeIntegrityPlatformManifestInformation = 179
SystemInterruptSteeringInformation = 180
SystemSupportedProcessorArchitectures = 181
SystemMemoryUsageInformation = 182
SystemCodeIntegrityCertificateInformation = 183
SystemPhysicalMemoryInformation = 184
SystemControlFlowTransition = 185
SystemKernelDebuggingAllowed = 186
SystemActivityModerationExeState = 187
SystemActivityModerationUserSettings = 188
SystemCodeIntegrityPoliciesFullInformation = 189
SystemCodeIntegrityUnlockInformation = 190
SystemIntegrityQuotaInformation = 191
SystemFlushInformation = 192
SystemProcessorIdleMaskInformation = 193
SystemSecureDumpEncryptionInformation = 194
SystemWriteConstraintInformation = 195
SystemKernelVaShadowInformation = 196
SystemHypervisorSharedPageInformation = 197
SystemFirmwareBootPerformanceInformation = 198
SystemCodeIntegrityVerificationInformation = 199
SystemFirmwarePartitionInformation = 200
SystemSpeculationControlInformation = 201
SystemDmaGuardPolicyInformation = 202
SystemEnclaveLaunchControlInformation = 203
SystemWorkloadAllowedCpuSetsInformation = 204
SystemCodeIntegrityUnlockModeInformation = 205
SystemLeapSecondInformation = 206
SystemFlags2Information = 207
SystemSecurityModelInformation = 208
SystemCodeIntegritySyntheticCacheInformation = 209
SystemFeatureConfigurationInformation = 210
SystemFeatureConfigurationSectionInformation = 211
SystemFeatureUsageSubscriptionInformation = 212
SystemSecureSpeculationControlInformation = 213
MaxSystemInfoClass = 214

# PROCESSINFOCLASS
ProcessBasicInformation = 0
ProcessQuotaLimits = 1
ProcessIoCounters = 2
ProcessVmCounters = 3
ProcessTimes = 4
ProcessBasePriority = 5
ProcessRaisePriority = 6
ProcessDebugPort = 7
ProcessExceptionPort = 8
ProcessAccessToken = 9
ProcessLdtInformation = 10
ProcessLdtSize = 11
ProcessDefaultHardErrorMode = 12
ProcessIoPortHandlers = 13
ProcessPooledUsageAndLimits = 14
ProcessWorkingSetWatch = 15
ProcessUserModeIOPL = 16
ProcessEnableAlignmentFaultFixup = 17
ProcessPriorityClass = 18
ProcessWx86Information = 19
ProcessHandleCount = 20
ProcessAffinityMask = 21
ProcessPriorityBoost = 22
ProcessDeviceMap = 23
ProcessSessionInformation = 24
ProcessForegroundInformation = 25
ProcessWow64Information = 26
ProcessImageFileName = 27
ProcessLUIDDeviceMapsEnabled = 28
ProcessBreakOnTermination = 29
ProcessDebugObjectHandle = 30
ProcessDebugFlags = 31
ProcessHandleTracing = 32
ProcessIoPriority = 33
ProcessExecuteFlags = 34
ProcessResourceManagement = 35
ProcessCookie = 36
ProcessImageInformation = 37
ProcessCycleTime = 38
ProcessPagePriority = 39
ProcessInstrumentationCallback = 40
ProcessThreadStackAllocation = 41
ProcessWorkingSetWatchEx = 42
ProcessImageFileNameWin32 = 43
ProcessImageFileMapping = 44
ProcessAffinityUpdateMode = 45
ProcessMemoryAllocationMode = 46
ProcessGroupInformation = 47
ProcessTokenVirtualizationEnabled = 48
ProcessConsoleHostProcess = 49
ProcessWindowInformation = 50
ProcessHandleInformation = 51
ProcessMitigationPolicy = 52
ProcessDynamicFunctionTableInformation = 53
ProcessHandleCheckingMode = 54
ProcessKeepAliveCount = 55
ProcessRevokeFileHandles = 56
ProcessWorkingSetControl = 57
ProcessHandleTable = 58
ProcessCheckStackExtentsMode = 59
ProcessCommandLineInformation = 60
ProcessProtectionInformation = 61
ProcessMemoryExhaustion = 62
ProcessFaultInformation = 63
ProcessTelemetryIdInformation = 64
ProcessCommitReleaseInformation = 65
ProcessDefaultCpuSetsInformation = 66
ProcessAllowedCpuSetsInformation = 67
ProcessSubsystemProcess = 68
ProcessJobMemoryInformation = 69
ProcessInPrivate = 70
ProcessRaiseUMExceptionOnInvalidHandleClose = 71
ProcessIumChallengeResponse = 72
ProcessChildProcessInformation = 73
ProcessHighGraphicsPriorityInformation = 74
ProcessSubsystemInformation = 75
ProcessEnergyValues = 76
ProcessActivityThrottleState = 77
ProcessActivityThrottlePolicy = 78
ProcessWin32kSyscallFilterInformation = 79
ProcessDisableSystemAllowedCpuSets = 80
ProcessWakeInformation = 81
ProcessEnergyTrackingState = 82
ProcessManageWritesToExecutableMemory = 83
ProcessCaptureTrustletLiveDump = 84
ProcessTelemetryCoverage = 85
ProcessEnclaveInformation = 86
ProcessEnableReadWriteVmLogging = 87
ProcessUptimeInformation = 88
ProcessImageSection = 89
ProcessDebugAuthInformation = 90
ProcessSystemResourceManagement = 91
ProcessSequenceNumber = 92
ProcessLoaderDetour = 93
ProcessSecurityDomainInformation = 94
ProcessCombineSecurityDomainsInformation = 95
ProcessEnableLogging = 96
ProcessLeapSecondInformation = 97
ProcessFiberShadowStackAllocation = 98
ProcessFreeFiberShadowStackAllocation = 99
ProcessAltSystemCallInformation = 100
ProcessDynamicEHContinuationTargets = 101
MaxProcessInfoClass = 102

DIGCF_DEFAULT = 0x00000001  # only valid with DIGCF_DEVICEINTERFACE
DIGCF_PRESENT = 0x00000002
DIGCF_ALLCLASSES = 0x00000004
DIGCF_PROFILE = 0x00000008
DIGCF_DEVICEINTERFACE = 0x00000010

DICS_ENABLE = 0x00000001
DICS_DISABLE = 0x00000002
DICS_PROPCHANGE = 0x00000003
DICS_START = 0x00000004
DICS_STOP = 0x00000005

DICS_FLAG_GLOBAL = 0x00000001
DICS_FLAG_CONFIGSPECIFIC = 0x00000002
DICS_FLAG_CONFIGGENERAL = 0x00000004

MAX_DEVICE_ID_LEN = 200
MAX_DEVNODE_ID_LEN = MAX_DEVICE_ID_LEN
MAX_GUID_STRING_LEN = 39  # 38 chars + terminator null
MAX_CLASS_NAME_LEN = 32
MAX_PROFILE_LEN = 80
MAX_CONFIG_VALUE = 9999
MAX_INSTANCE_VALUE = 9999

DIF_SELECTDEVICE = 0x00000001
DIF_INSTALLDEVICE = 0x00000002
DIF_ASSIGNRESOURCES = 0x00000003
DIF_PROPERTIES = 0x00000004
DIF_REMOVE = 0x00000005
DIF_FIRSTTIMESETUP = 0x00000006
DIF_FOUNDDEVICE = 0x00000007
DIF_SELECTCLASSDRIVERS = 0x00000008
DIF_VALIDATECLASSDRIVERS = 0x00000009
DIF_INSTALLCLASSDRIVERS = 0x0000000A
DIF_CALCDISKSPACE = 0x0000000B
DIF_DESTROYPRIVATEDATA = 0x0000000C
DIF_VALIDATEDRIVER = 0x0000000D
DIF_DETECT = 0x0000000F
DIF_INSTALLWIZARD = 0x00000010
DIF_DESTROYWIZARDDATA = 0x00000011
DIF_PROPERTYCHANGE = 0x00000012
DIF_ENABLECLASS = 0x00000013
DIF_DETECTVERIFY = 0x00000014
DIF_INSTALLDEVICEFILES = 0x00000015
DIF_UNREMOVE = 0x00000016
DIF_SELECTBESTCOMPATDRV = 0x00000017
DIF_ALLOW_INSTALL = 0x00000018
DIF_REGISTERDEVICE = 0x00000019
DIF_NEWDEVICEWIZARD_PRESELECT = 0x0000001A
DIF_NEWDEVICEWIZARD_SELECT = 0x0000001B
DIF_NEWDEVICEWIZARD_PREANALYZE = 0x0000001C
DIF_NEWDEVICEWIZARD_POSTANALYZE = 0x0000001D
DIF_NEWDEVICEWIZARD_FINISHINSTALL = 0x0000001E
DIF_UNUSED1 = 0x0000001F
DIF_INSTALLINTERFACES = 0x00000020
DIF_DETECTCANCEL = 0x00000021
DIF_REGISTER_COINSTALLERS = 0x00000022
DIF_ADDPROPERTYPAGE_ADVANCED = 0x00000023
DIF_ADDPROPERTYPAGE_BASIC = 0x00000024
DIF_RESERVED1 = 0x00000025
DIF_TROUBLESHOOTER = 0x00000026
DIF_POWERMESSAGEWAKE = 0x00000027
DIF_ADDREMOTEPROPERTYPAGE_ADVANCED = 0x00000028
DIF_UPDATEDRIVER_UI = 0x00000029
DIF_FINISHINSTALL_ACTION = 0x0000002A
DIF_RESERVED2 = 0x00000030

CR_SUCCESS = 0x00000000
CR_DEFAULT = 0x00000001
CR_OUT_OF_MEMORY = 0x00000002
CR_INVALID_POINTER = 0x00000003
CR_INVALID_FLAG = 0x00000004
CR_INVALID_DEVNODE = 0x00000005
CR_INVALID_DEVINST = CR_INVALID_DEVNODE
CR_INVALID_RES_DES = 0x00000006
CR_INVALID_LOG_CONF = 0x00000007
CR_INVALID_ARBITRATOR = 0x00000008
CR_INVALID_NODELIST = 0x00000009
CR_DEVNODE_HAS_REQS = 0x0000000A
CR_DEVINST_HAS_REQS = CR_DEVNODE_HAS_REQS
CR_INVALID_RESOURCEID = 0x0000000B
CR_NO_SUCH_DEVNODE = 0x0000000D
CR_NO_SUCH_DEVINST = CR_NO_SUCH_DEVNODE
CR_NO_MORE_LOG_CONF = 0x0000000E
CR_NO_MORE_RES_DES = 0x0000000F
CR_ALREADY_SUCH_DEVNODE = 0x00000010
CR_ALREADY_SUCH_DEVINST = CR_ALREADY_SUCH_DEVNODE
CR_INVALID_RANGE_LIST = 0x00000011
CR_INVALID_RANGE = 0x00000012
CR_FAILURE = 0x00000013
CR_NO_SUCH_LOGICAL_DEV = 0x00000014
CR_CREATE_BLOCKED = 0x00000015
CR_REMOVE_VETOED = 0x00000017
CR_APM_VETOED = 0x00000018
CR_INVALID_LOAD_TYPE = 0x00000019
CR_BUFFER_SMALL = 0x0000001A
CR_NO_ARBITRATOR = 0x0000001B
CR_NO_REGISTRY_HANDLE = 0x0000001C
CR_REGISTRY_ERROR = 0x0000001D
CR_INVALID_DEVICE_ID = 0x0000001E
CR_INVALID_DATA = 0x0000001F
CR_INVALID_API = 0x00000020
CR_DEVLOADER_NOT_READY = 0x00000021
CR_NEED_RESTART = 0x00000022
CR_NO_MORE_HW_PROFILES = 0x00000023
CR_DEVICE_NOT_THERE = 0x00000024
CR_NO_SUCH_VALUE = 0x00000025
CR_WRONG_TYPE = 0x00000026
CR_INVALID_PRIORITY = 0x00000027
CR_NOT_DISABLEABLE = 0x00000028
CR_FREE_RESOURCES = 0x00000029
CR_QUERY_VETOED = 0x0000002A
CR_CANT_SHARE_IRQ = 0x0000002B
CR_NO_DEPENDENT = 0x0000002C
CR_SAME_RESOURCES = 0x0000002D
CR_NO_SUCH_REGISTRY_KEY = 0x0000002E
CR_INVALID_MACHINENAME = 0x0000002F
CR_REMOTE_COMM_FAILURE = 0x00000030
CR_MACHINE_UNAVAILABLE = 0x00000031
CR_NO_CM_SERVICES = 0x00000032
CR_ACCESS_DENIED = 0x00000033
CR_CALL_NOT_IMPLEMENTED = 0x00000034
CR_INVALID_PROPERTY = 0x00000035
CR_DEVICE_INTERFACE_ACTIVE = 0x00000036
CR_NO_SUCH_DEVICE_INTERFACE = 0x00000037
CR_INVALID_REFERENCE_STRING = 0x00000038
CR_INVALID_CONFLICT_LIST = 0x00000039
CR_INVALID_INDEX = 0x0000003A
CR_INVALID_STRUCTURE_SIZE = 0x0000003B
NUM_CR_RESULTS = 0x0000003C

CM_ENUMERATE_CLASSES_INSTALLER = 0x00000000
CM_ENUMERATE_CLASSES_INTERFACE = 0x00000001
CM_ENUMERATE_CLASSES_BITS = 0x00000001

SPDRP_DEVICEDESC = 0x00000000  # DeviceDesc (R/W)
SPDRP_HARDWAREID = 0x00000001  # HardwareID (R/W)
SPDRP_COMPATIBLEIDS = 0x00000002  # CompatibleIDs (R/W)
SPDRP_UNUSED0 = 0x00000003  # unused
SPDRP_SERVICE = 0x00000004  # Service (R/W)
SPDRP_UNUSED1 = 0x00000005  # unused
SPDRP_UNUSED2 = 0x00000006  # unused
SPDRP_CLASS = 0x00000007  # Class (R--tied to ClassGUID)
SPDRP_CLASSGUID = 0x00000008  # ClassGUID (R/W)
SPDRP_DRIVER = 0x00000009  # Driver (R/W)
SPDRP_CONFIGFLAGS = 0x0000000A  # ConfigFlags (R/W)
SPDRP_MFG = 0x0000000B  # Mfg (R/W)
SPDRP_FRIENDLYNAME = 0x0000000C  # FriendlyName (R/W)
SPDRP_LOCATION_INFORMATION = 0x0000000D  # LocationInformation (R/W)
SPDRP_PHYSICAL_DEVICE_OBJECT_NAME = 0x0000000E  # PhysicalDeviceObjectName (R)
SPDRP_CAPABILITIES = 0x0000000F  # Capabilities (R)
SPDRP_UI_NUMBER = 0x00000010  # UiNumber (R)
SPDRP_UPPERFILTERS = 0x00000011  # UpperFilters (R/W)
SPDRP_LOWERFILTERS = 0x00000012  # LowerFilters (R/W)
SPDRP_BUSTYPEGUID = 0x00000013  # BusTypeGUID (R)
SPDRP_LEGACYBUSTYPE = 0x00000014  # LegacyBusType (R)
SPDRP_BUSNUMBER = 0x00000015  # BusNumber (R)
SPDRP_ENUMERATOR_NAME = 0x00000016  # Enumerator Name (R)
SPDRP_SECURITY = 0x00000017  # Security (R/W, binary form)
SPDRP_SECURITY_SDS = 0x00000018  # Security (W, SDS form)
SPDRP_DEVTYPE = 0x00000019  # Device Type (R/W)
SPDRP_EXCLUSIVE = 0x0000001A  # Device is exclusive-access (R/W)
SPDRP_CHARACTERISTICS = 0x0000001B  # Device Characteristics (R/W)
SPDRP_ADDRESS = 0x0000001C  # Device Address (R)
SPDRP_UI_NUMBER_DESC_FORMAT = 0x0000001D  # UiNumberDescFormat (R/W)
SPDRP_DEVICE_POWER_DATA = 0x0000001E  # Device Power Data (R)
SPDRP_REMOVAL_POLICY = 0x0000001F  # Removal Policy (R)
SPDRP_REMOVAL_POLICY_HW_DEFAULT = 0x00000020  # Hardware Removal Policy (R)
SPDRP_REMOVAL_POLICY_OVERRIDE = 0x00000021  # Removal Policy Override (RW)
SPDRP_INSTALL_STATE = 0x00000022  # Device Install State (R)
SPDRP_LOCATION_PATHS = 0x00000023  # Device Location Paths (R)
SPDRP_BASE_CONTAINERID = 0x00000024  # Base ContainerID (R)
SPDRP_MAXIMUM_PROPERTY = 0x00000025  # Upper bound on ordinals

IF_TYPE_SOFTWARE_LOOPBACK = 24

AF_UNSPEC = 0  # unspecified
AF_UNIX = 1  # local to host (pipes, portals)
AF_INET = 2  # internetwork: UDP, TCP, etc.
AF_IMPLINK = 3  # arpanet imp addresses
AF_PUP = 4  # pup protocols: e.g. BSP
AF_CHAOS = 5  # mit CHAOS protocols
AF_NS = 6  # XEROX NS protocols
AF_IPX = AF_NS  # IPX protocols: IPX, SPX, etc.
AF_ISO = 7  # ISO protocols
AF_OSI = AF_ISO  # OSI is ISO
AF_ECMA = 8  # european computer manufacturers
AF_DATAKIT = 9  # datakit protocols
AF_CCITT = 10  # CCITT protocols, X.25 etc
AF_SNA = 11  # IBM SNA
AF_DECnet = 12  # DECnet
AF_DLI = 13  # Direct data link interface
AF_LAT = 14  # LAT
AF_HYLINK = 15  # NSC Hyperchannel
AF_APPLETALK = 16  # AppleTalk
AF_NETBIOS = 17  # NetBios-style addresses
AF_VOICEVIEW = 18  # VoiceView
AF_FIREFOX = 19  # Protocols from Firefox
AF_UNKNOWN1 = 20  # Somebody is using this!
AF_BAN = 21  # Banyan
AF_ATM = 22  # Native ATM Services
AF_INET6 = 23  # Internetwork Version 6
AF_CLUSTER = 24  # Microsoft Wolfpack
AF_12844 = 25  # IEEE 1284.4 WG AF
AF_IRDA = 26  # IrDA
AF_NETDES = 28  # Network Designers OSI & gateway

INPUT_MOUSE = 0
INPUT_KEYBOARD = 1
INPUT_HARDWARE = 2

KEYEVENTF_EXTENDEDKEY = 0x0001
KEYEVENTF_KEYUP = 0x0002
KEYEVENTF_UNICODE = 0x0004
KEYEVENTF_SCANCODE = 0x0008

MOUSEEVENTF_MOVE = 0x0001
MOUSEEVENTF_LEFTDOWN = 0x0002
MOUSEEVENTF_LEFTUP = 0x0004
MOUSEEVENTF_RIGHTDOWN = 0x0008
MOUSEEVENTF_RIGHTUP = 0x0010
MOUSEEVENTF_MIDDLEDOWN = 0x0020
MOUSEEVENTF_MIDDLEUP = 0x0040
MOUSEEVENTF_XDOWN = 0x0080
MOUSEEVENTF_XUP = 0x0100
MOUSEEVENTF_WHEEL = 0x0800
MOUSEEVENTF_HWHEEL = 0x1000
MOUSEEVENTF_MOVE_NOCOALESCE = 0x2000
MOUSEEVENTF_VIRTUALDESK = 0x4000
MOUSEEVENTF_ABSOLUTE = 0x8000

SystemPowerPolicyAc = 0
SystemPowerPolicyDc = 1
VerifySystemPolicyAc = 2
VerifySystemPolicyDc = 3
SystemPowerCapabilities = 4
SystemBatteryState = 5
SystemPowerStateHandler = 6
ProcessorStateHandler = 7
SystemPowerPolicyCurrent = 8
AdministratorPowerPolicy = 9
SystemReserveHiberFile = 10
ProcessorInformation = 11
SystemPowerInformation = 12
ProcessorStateHandler2 = 13
LastWakeTime = 14
LastSleepTime = 15
SystemExecutionState = 16
SystemPowerStateNotifyHandler = 17
ProcessorPowerPolicyAc = 18
ProcessorPowerPolicyDc = 19
VerifyProcessorPowerPolicyAc = 20
VerifyProcessorPowerPolicyDc = 21
ProcessorPowerPolicyCurrent = 22
SystemPowerStateLogging = 23
SystemPowerLoggingEntry = 24
SetPowerSettingValue = 25
NotifyUserPowerSetting = 26
PowerInformationLevelUnused0 = 27
SystemMonitorHiberBootPowerOff = 28
SystemVideoState = 29
TraceApplicationPowerMessage = 30
TraceApplicationPowerMessageEnd = 31
ProcessorPerfStates = 32
ProcessorIdleStates = 33
ProcessorCap = 34
SystemWakeSource = 35
SystemHiberFileInformation = 36
TraceServicePowerMessage = 37
ProcessorLoad = 38
PowerShutdownNotification = 39
MonitorCapabilities = 40
SessionPowerInit = 41
SessionDisplayState = 42
PowerRequestCreate = 43
PowerRequestAction = 44
GetPowerRequestList = 45
ProcessorInformationEx = 46
NotifyUserModeLegacyPowerEvent = 47
GroupPark = 48
ProcessorIdleDomains = 49
WakeTimerList = 50
SystemHiberFileSize = 51
ProcessorIdleStatesHv = 52
ProcessorPerfStatesHv = 53
ProcessorPerfCapHv = 54
ProcessorSetIdle = 55
LogicalProcessorIdling = 56
UserPresence = 57
PowerSettingNotificationName = 58
GetPowerSettingValue = 59
IdleResiliency = 60
SessionRITState = 61
SessionConnectNotification = 62
SessionPowerCleanup = 63
SessionLockState = 64
SystemHiberbootState = 65
PlatformInformation = 66
PdcInvocation = 67
MonitorInvocation = 68
FirmwareTableInformationRegistered = 69
SetShutdownSelectedTime = 70
SuspendResumeInvocation = 71
PlmPowerRequestCreate = 72
ScreenOff = 73
CsDeviceNotification = 74
PlatformRole = 75
LastResumePerformance = 76
DisplayBurst = 77
ExitLatencySamplingPercentage = 78
RegisterSpmPowerSettings = 79
PlatformIdleStates = 80
ProcessorIdleVeto = 81
PlatformIdleVeto = 82
SystemBatteryStatePrecise = 83
ThermalEvent = 84
PowerRequestActionInternal = 85
BatteryDeviceState = 86
PowerInformationInternal = 87
ThermalStandby = 88
SystemHiberFileType = 89
PhysicalPowerButtonPress = 90
QueryPotentialDripsConstraint = 91
EnergyTrackerCreate = 92
EnergyTrackerQuery = 93
UpdateBlackBoxRecorder = 94
PowerInformationLevelMaximum = 95

WTSActive = 0
WTSConnected = 1
WTSConnectQuery = 2
WTSShadow = 3
WTSDisconnected = 4
WTSIdle = 5
WTSListen = 6
WTSReset = 7
WTSDown = 8
WTSInit = 9

EWX_LOGOFF = 0x00000000
EWX_SHUTDOWN = 0x00000001
EWX_REBOOT = 0x00000002
EWX_FORCE = 0x00000004
EWX_POWEROFF = 0x00000008
EWX_FORCEIFHUNG = 0x00000010
EWX_QUICKRESOLVE = 0x00000020
EWX_RESTARTAPPS = 0x00000040
EWX_HYBRID_SHUTDOWN = 0x00400000
EWX_BOOTOPTIONS = 0x01000000
EWX_ARSO = 0x04000000

SHTDN_REASON_FLAG_COMMENT_REQUIRED = 0x01000000
SHTDN_REASON_FLAG_DIRTY_PROBLEM_ID_REQUIRED = 0x02000000
SHTDN_REASON_FLAG_CLEAN_UI = 0x04000000
SHTDN_REASON_FLAG_DIRTY_UI = 0x08000000
SHTDN_REASON_FLAG_MOBILE_UI_RESERVED = 0x10000000
SHTDN_REASON_FLAG_USER_DEFINED = 0x40000000
SHTDN_REASON_FLAG_PLANNED = 0x80000000
SHTDN_REASON_MAJOR_OTHER = 0x00000000
SHTDN_REASON_MAJOR_NONE = 0x00000000
SHTDN_REASON_MAJOR_HARDWARE = 0x00010000
SHTDN_REASON_MAJOR_OPERATINGSYSTEM = 0x00020000
SHTDN_REASON_MAJOR_SOFTWARE = 0x00030000
SHTDN_REASON_MAJOR_APPLICATION = 0x00040000
SHTDN_REASON_MAJOR_SYSTEM = 0x00050000
SHTDN_REASON_MAJOR_POWER = 0x00060000
SHTDN_REASON_MAJOR_LEGACY_API = 0x00070000
SHTDN_REASON_MINOR_OTHER = 0x00000000
SHTDN_REASON_MINOR_NONE = 0x000000FF
SHTDN_REASON_MINOR_MAINTENANCE = 0x00000001
SHTDN_REASON_MINOR_INSTALLATION = 0x00000002
SHTDN_REASON_MINOR_UPGRADE = 0x00000003
SHTDN_REASON_MINOR_RECONFIG = 0x00000004
SHTDN_REASON_MINOR_HUNG = 0x00000005
SHTDN_REASON_MINOR_UNSTABLE = 0x00000006
SHTDN_REASON_MINOR_DISK = 0x00000007
SHTDN_REASON_MINOR_PROCESSOR = 0x00000008
SHTDN_REASON_MINOR_NETWORKCARD = 0x00000009
SHTDN_REASON_MINOR_POWER_SUPPLY = 0x0000000A
SHTDN_REASON_MINOR_CORDUNPLUGGED = 0x0000000B
SHTDN_REASON_MINOR_ENVIRONMENT = 0x0000000C
SHTDN_REASON_MINOR_HARDWARE_DRIVER = 0x0000000D
SHTDN_REASON_MINOR_OTHERDRIVER = 0x0000000E
SHTDN_REASON_MINOR_BLUESCREEN = 0x0000000F
SHTDN_REASON_MINOR_SERVICEPACK = 0x00000010
SHTDN_REASON_MINOR_HOTFIX = 0x00000011
SHTDN_REASON_MINOR_SECURITYFIX = 0x00000012
SHTDN_REASON_MINOR_SECURITY = 0x00000013
SHTDN_REASON_MINOR_NETWORK_CONNECTIVITY = 0x00000014
SHTDN_REASON_MINOR_WMI = 0x00000015
SHTDN_REASON_MINOR_SERVICEPACK_UNINSTALL = 0x00000016
SHTDN_REASON_MINOR_HOTFIX_UNINSTALL = 0x00000017
SHTDN_REASON_MINOR_SECURITYFIX_UNINSTALL = 0x00000018
SHTDN_REASON_MINOR_MMC = 0x00000019
SHTDN_REASON_MINOR_SYSTEMRESTORE = 0x0000001A
SHTDN_REASON_MINOR_TERMSRV = 0x00000020
SHTDN_REASON_MINOR_DC_PROMOTION = 0x00000021
SHTDN_REASON_MINOR_DC_DEMOTION = 0x00000022
SHTDN_REASON_UNKNOWN = SHTDN_REASON_MINOR_NONE
SHTDN_REASON_LEGACY_API = (
    SHTDN_REASON_MAJOR_LEGACY_API | SHTDN_REASON_FLAG_PLANNED
    )
SHTDN_REASON_VALID_BIT_MASK = 0xC0FFFFFF

DN_ROOT_ENUMERATED = 0x00000001
DN_DRIVER_LOADED = 0x00000002
DN_ENUM_LOADED = 0x00000004
DN_STARTED = 0x00000008
DN_MANUAL = 0x00000010
DN_NEED_TO_ENUM = 0x00000020
DN_NOT_FIRST_TIME = 0x00000040
DN_HARDWARE_ENUM = 0x00000080
DN_LIAR = 0x00000100
DN_HAS_MARK = 0x00000200
DN_HAS_PROBLEM = 0x00000400
DN_FILTERED = 0x00000800
DN_MOVED = 0x00001000
DN_DISABLEABLE = 0x00002000
DN_REMOVABLE = 0x00004000
DN_PRIVATE_PROBLEM = 0x00008000
DN_MF_PARENT = 0x00010000
DN_MF_CHILD = 0x00020000
DN_WILL_BE_REMOVED = 0x00040000
DN_NOT_FIRST_TIMEE = 0x00080000
DN_STOP_FREE_RES = 0x00100000
DN_REBAL_CANDIDATE = 0x00200000
DN_BAD_PARTIAL = 0x00400000
DN_NT_ENUMERATOR = 0x00800000
DN_NT_DRIVER = 0x01000000
DN_NEEDS_LOCKING = 0x02000000
DN_ARM_WAKEUP = 0x04000000
DN_APM_ENUMERATOR = 0x08000000
DN_APM_DRIVER = 0x10000000
DN_SILENT_INSTALL = 0x20000000
DN_NO_SHOW_IN_DM = 0x40000000
DN_BOOT_LOG_PROB = 0x80000000

DRIVE_UNKNOWN = 0
DRIVE_NO_ROOT_DIR = 1
DRIVE_REMOVABLE = 2
DRIVE_FIXED = 3
DRIVE_REMOTE = 4
DRIVE_CDROM = 5
DRIVE_RAMDISK = 6

FILE_TYPE_UNKNOWN = 0x0000
FILE_TYPE_DISK = 0x0001
FILE_TYPE_CHAR = 0x0002
FILE_TYPE_PIPE = 0x0003
FILE_TYPE_REMOTE = 0x8000

FILE_DEVICE_BEEP = 0x00000001
FILE_DEVICE_CD_ROM = 0x00000002
FILE_DEVICE_CD_ROM_FILE_SYSTEM = 0x00000003
FILE_DEVICE_CONTROLLER = 0x00000004
FILE_DEVICE_DATALINK = 0x00000005
FILE_DEVICE_DFS = 0x00000006
FILE_DEVICE_DISK = 0x00000007
FILE_DEVICE_DISK_FILE_SYSTEM = 0x00000008
FILE_DEVICE_FILE_SYSTEM = 0x00000009
FILE_DEVICE_INPORT_PORT = 0x0000000A
FILE_DEVICE_KEYBOARD = 0x0000000B
FILE_DEVICE_MAILSLOT = 0x0000000C
FILE_DEVICE_MIDI_IN = 0x0000000D
FILE_DEVICE_MIDI_OUT = 0x0000000E
FILE_DEVICE_MOUSE = 0x0000000F
FILE_DEVICE_MULTI_UNC_PROVIDER = 0x00000010
FILE_DEVICE_NAMED_PIPE = 0x00000011
FILE_DEVICE_NETWORK = 0x00000012
FILE_DEVICE_NETWORK_BROWSER = 0x00000013
FILE_DEVICE_NETWORK_FILE_SYSTEM = 0x00000014
FILE_DEVICE_NULL = 0x00000015
FILE_DEVICE_PARALLEL_PORT = 0x00000016
FILE_DEVICE_PHYSICAL_NETCARD = 0x00000017
FILE_DEVICE_PRINTER = 0x00000018
FILE_DEVICE_SCANNER = 0x00000019
FILE_DEVICE_SERIAL_MOUSE_PORT = 0x0000001A
FILE_DEVICE_SERIAL_PORT = 0x0000001B
FILE_DEVICE_SCREEN = 0x0000001C
FILE_DEVICE_SOUND = 0x0000001D
FILE_DEVICE_STREAMS = 0x0000001E
FILE_DEVICE_TAPE = 0x0000001F
FILE_DEVICE_TAPE_FILE_SYSTEM = 0x00000020
FILE_DEVICE_TRANSPORT = 0x00000021
FILE_DEVICE_UNKNOWN = 0x00000022
FILE_DEVICE_VIDEO = 0x00000023
FILE_DEVICE_VIRTUAL_DISK = 0x00000024
FILE_DEVICE_WAVE_IN = 0x00000025
FILE_DEVICE_WAVE_OUT = 0x00000026
FILE_DEVICE_8042_PORT = 0x00000027
FILE_DEVICE_NETWORK_REDIRECTOR = 0x00000028
FILE_DEVICE_BATTERY = 0x00000029
FILE_DEVICE_BUS_EXTENDER = 0x0000002A
FILE_DEVICE_MODEM = 0x0000002B
FILE_DEVICE_VDM = 0x0000002C
FILE_DEVICE_MASS_STORAGE = 0x0000002D
FILE_DEVICE_SMB = 0x0000002E
FILE_DEVICE_KS = 0x0000002F
FILE_DEVICE_CHANGER = 0x00000030
FILE_DEVICE_SMARTCARD = 0x00000031
FILE_DEVICE_ACPI = 0x00000032
FILE_DEVICE_DVD = 0x00000033
FILE_DEVICE_FULLSCREEN_VIDEO = 0x00000034
FILE_DEVICE_DFS_FILE_SYSTEM = 0x00000035
FILE_DEVICE_DFS_VOLUME = 0x00000036
FILE_DEVICE_SERENUM = 0x00000037
FILE_DEVICE_TERMSRV = 0x00000038
FILE_DEVICE_KSEC = 0x00000039
FILE_DEVICE_FIPS = 0x0000003A
FILE_DEVICE_INFINIBAND = 0x0000003B
FILE_DEVICE_VMBUS = 0x0000003E
FILE_DEVICE_CRYPT_PROVIDER = 0x0000003F
FILE_DEVICE_WPD = 0x00000040
FILE_DEVICE_BLUETOOTH = 0x00000041
FILE_DEVICE_MT_COMPOSITE = 0x00000042
FILE_DEVICE_MT_TRANSPORT = 0x00000043
FILE_DEVICE_BIOMETRIC = 0x00000044
FILE_DEVICE_PMI = 0x00000045
FILE_DEVICE_EHSTOR = 0x00000046
FILE_DEVICE_DEVAPI = 0x00000047
FILE_DEVICE_GPIO = 0x00000048
FILE_DEVICE_USBEX = 0x00000049
FILE_DEVICE_CONSOLE = 0x00000050
FILE_DEVICE_NFP = 0x00000051
FILE_DEVICE_SYSENV = 0x00000052
FILE_DEVICE_VIRTUAL_BLOCK = 0x00000053
FILE_DEVICE_POINT_OF_SERVICE = 0x00000054
FILE_DEVICE_STORAGE_REPLICATION = 0x00000055
FILE_DEVICE_TRUST_ENV = 0x00000056
FILE_DEVICE_UCM = 0x00000057
FILE_DEVICE_UCMTCPCI = 0x00000058
FILE_DEVICE_PERSISTENT_MEMORY = 0x00000059
FILE_DEVICE_NVDIMM = 0x0000005A
FILE_DEVICE_HOLOGRAPHIC = 0x0000005B
FILE_DEVICE_SDFXHCI = 0x0000005C
FILE_DEVICE_UCMUCSI = 0x0000005D

MONITOR_DEFAULTTONULL = 0x00000000
MONITOR_DEFAULTTOPRIMARY = 0x00000001
MONITOR_DEFAULTTONEAREST = 0x00000002

IDC_ARROW = 32512
IDC_IBEAM = 32513
IDC_WAIT = 32514
IDC_CROSS = 32515
IDC_UPARROW = 32516
IDC_SIZE = 32640
IDC_ICON = 32641
IDC_SIZENWSE = 32642
IDC_SIZENESW = 32643
IDC_SIZEWE = 32644
IDC_SIZENS = 32645
IDC_SIZEALL = 32646
IDC_NO = 32648
IDC_HAND = 32649
IDC_APPSTARTING = 32650
IDC_HELP = 32651
IDC_PIN = 32671
IDC_PERSON = 32672

COLOR_SCROLLBAR = 0
COLOR_BACKGROUND = 1
COLOR_ACTIVECAPTION = 2
COLOR_INACTIVECAPTION = 3
COLOR_MENU = 4
COLOR_WINDOW = 5
COLOR_WINDOWFRAME = 6
COLOR_MENUTEXT = 7
COLOR_WINDOWTEXT = 8
COLOR_CAPTIONTEXT = 9
COLOR_ACTIVEBORDER = 10
COLOR_INACTIVEBORDER = 11
COLOR_APPWORKSPACE = 12
COLOR_HIGHLIGHT = 13
COLOR_HIGHLIGHTTEXT = 14
COLOR_BTNFACE = 15
COLOR_BTNSHADOW = 16
COLOR_GRAYTEXT = 17
COLOR_BTNTEXT = 18
COLOR_INACTIVECAPTIONTEXT = 19
COLOR_BTNHIGHLIGHT = 20
COLOR_3DDKSHADOW = 21
COLOR_3DLIGHT = 22
COLOR_INFOTEXT = 23
COLOR_INFOBK = 24
COLOR_HOTLIGHT = 26
COLOR_GRADIENTACTIVECAPTION = 27
COLOR_GRADIENTINACTIVECAPTION = 28
COLOR_MENUHILIGHT = 29
COLOR_MENUBAR = 30
COLOR_DESKTOP = COLOR_BACKGROUND
COLOR_3DFACE = COLOR_BTNFACE
COLOR_3DSHADOW = COLOR_BTNSHADOW
COLOR_3DHIGHLIGHT = COLOR_BTNHIGHLIGHT
COLOR_3DHILIGHT = COLOR_BTNHIGHLIGHT
COLOR_BTNHILIGHT = COLOR_BTNHIGHLIGHT

BS_PUSHBUTTON = 0x00000000
BS_DEFPUSHBUTTON = 0x00000001
BS_CHECKBOX = 0x00000002
BS_AUTOCHECKBOX = 0x00000003
BS_RADIOBUTTON = 0x00000004
BS_3STATE = 0x00000005
BS_AUTO3STATE = 0x00000006
BS_GROUPBOX = 0x00000007
BS_USERBUTTON = 0x00000008
BS_AUTORADIOBUTTON = 0x00000009
BS_PUSHBOX = 0x0000000A
BS_OWNERDRAW = 0x0000000B
BS_TYPEMASK = 0x0000000F
BS_LEFTTEXT = 0x00000020
BS_TEXT = 0x00000000
BS_ICON = 0x00000040
BS_BITMAP = 0x00000080
BS_LEFT = 0x00000100
BS_RIGHT = 0x00000200
BS_CENTER = 0x00000300
BS_TOP = 0x00000400
BS_BOTTOM = 0x00000800
BS_VCENTER = 0x00000C00
BS_PUSHLIKE = 0x00001000
BS_MULTILINE = 0x00002000
BS_NOTIFY = 0x00004000
BS_FLAT = 0x00008000
BS_RIGHTBUTTON = BS_LEFTTEXT

BN_CLICKED = 0
BN_PAINT = 1
BN_HILITE = 2
BN_UNHILITE = 3
BN_DISABLE = 4
BN_DOUBLECLICKED = 5
BN_PUSHED = BN_HILITE
BN_UNPUSHED = BN_UNHILITE
BN_DBLCLK = BN_DOUBLECLICKED
BN_SETFOCUS = 6
BN_KILLFOCUS = 7

BM_GETCHECK = 0x00F0
BM_SETCHECK = 0x00F1
BM_GETSTATE = 0x00F2
BM_SETSTATE = 0x00F3
BM_SETSTYLE = 0x00F4
BM_CLICK = 0x00F5
BM_GETIMAGE = 0x00F6
BM_SETIMAGE = 0x00F7
BM_SETDONTCLICK = 0x00F8

BST_UNCHECKED = 0x0000
BST_CHECKED = 0x0001
BST_INDETERMINATE = 0x0002
BST_PUSHED = 0x0004
BST_FOCUS = 0x0008

GMEM_FIXED = 0x0000
GMEM_MOVEABLE = 0x0002
GMEM_NOCOMPACT = 0x0010
GMEM_NODISCARD = 0x0020
GMEM_ZEROINIT = 0x0040
GMEM_MODIFY = 0x0080
GMEM_DISCARDABLE = 0x0100
GMEM_NOT_BANKED = 0x1000
GMEM_SHARE = 0x2000
GMEM_DDESHARE = 0x2000
GMEM_NOTIFY = 0x4000
GMEM_LOWER = GMEM_NOT_BANKED
GMEM_VALID_FLAGS = 0x7F72
GMEM_INVALID_HANDLE = 0x8000
GHND = GMEM_MOVEABLE | GMEM_ZEROINIT
GPTR = GMEM_FIXED | GMEM_ZEROINIT

PROC_THREAD_ATTRIBUTE_NUMBER = 0x0000FFFF
PROC_THREAD_ATTRIBUTE_THREAD = 0x00010000
PROC_THREAD_ATTRIBUTE_INPUT = 0x00020000
PROC_THREAD_ATTRIBUTE_ADDITIVE = 0x00040000

PROC_THREAD_ATTRIBUTE_ALL_APPLICATION_PACKAGES_POLICY = 0x0002000F
PROC_THREAD_ATTRIBUTE_CHILD_PROCESS_POLICY = 0x0002000E
PROC_THREAD_ATTRIBUTE_DESKTOP_APP_POLICY = 0x00020012
PROC_THREAD_ATTRIBUTE_EXTENDED_FLAGS = 0x00060001
PROC_THREAD_ATTRIBUTE_GROUP_AFFINITY = 0x00030003
PROC_THREAD_ATTRIBUTE_HANDLE_LIST = 0x00020002
PROC_THREAD_ATTRIBUTE_IDEAL_PROCESSOR = 0x00030005
PROC_THREAD_ATTRIBUTE_JOB_LIST = 0x0002000D
PROC_THREAD_ATTRIBUTE_MITIGATION_AUDIT_POLICY = 0x00020018
PROC_THREAD_ATTRIBUTE_MITIGATION_POLICY = 0x00020007
PROC_THREAD_ATTRIBUTE_PARENT_PROCESS = 0x00020000
PROC_THREAD_ATTRIBUTE_PREFERRED_NODE = 0x00020004
PROC_THREAD_ATTRIBUTE_PROTECTION_LEVEL = 0x0002000B
PROC_THREAD_ATTRIBUTE_PSEUDOCONSOLE = 0x00020016
PROC_THREAD_ATTRIBUTE_SECURITY_CAPABILITIES = 0x00020009
PROC_THREAD_ATTRIBUTE_UMS_THREAD = 0x00030006
PROC_THREAD_ATTRIBUTE_WIN32K_FILTER = 0x00020010

TDF_ENABLE_HYPERLINKS = 0x0001
TDF_USE_HICON_MAIN = 0x0002
TDF_USE_HICON_FOOTER = 0x0004
TDF_ALLOW_DIALOG_CANCELLATION = 0x0008
TDF_USE_COMMAND_LINKS = 0x0010
TDF_USE_COMMAND_LINKS_NO_ICON = 0x0020
TDF_EXPAND_FOOTER_AREA = 0x0040
TDF_EXPANDED_BY_DEFAULT = 0x0080
TDF_VERIFICATION_FLAG_CHECKED = 0x0100
TDF_SHOW_PROGRESS_BAR = 0x0200
TDF_SHOW_MARQUEE_PROGRESS_BAR = 0x0400
TDF_CALLBACK_TIMER = 0x0800
TDF_POSITION_RELATIVE_TO_WINDOW = 0x1000
TDF_RTL_LAYOUT = 0x2000
TDF_NO_DEFAULT_RADIO_BUTTON = 0x4000
TDF_CAN_BE_MINIMIZED = 0x8000
TDF_NO_SET_FOREGROUND = 0x00010000
TDF_SIZE_TO_CONTENT = 0x01000000

TDN_CREATED = 0
TDN_NAVIGATED = 1
TDN_BUTTON_CLICKED = 2
TDN_HYPERLINK_CLICKED = 3
TDN_TIMER = 4
TDN_DESTROYED = 5
TDN_RADIO_BUTTON_CLICKED = 6
TDN_DIALOG_CONSTRUCTED = 7
TDN_VERIFICATION_CLICKED = 8
TDN_HELP = 9
TDN_EXPANDO_BUTTON_CLICKED = 10

TD_WARNING_ICON = PWSTR(0xFFFF)
TD_ERROR_ICON = PWSTR(0xFFFE)
TD_INFORMATION_ICON = PWSTR(0xFFFD)
TD_SHIELD_ICON = PWSTR(0xFFFC)

TDCBF_OK_BUTTON = 0x0001
TDCBF_YES_BUTTON = 0x0002
TDCBF_NO_BUTTON = 0x0004
TDCBF_CANCEL_BUTTON = 0x0008
TDCBF_RETRY_BUTTON = 0x0010
TDCBF_CLOSE_BUTTON = 0x0020

IDOK = 1
IDCANCEL = 2
IDABORT = 3
IDRETRY = 4
IDIGNORE = 5
IDYES = 6
IDNO = 7
IDCLOSE = 8
IDHELP = 9
IDTRYAGAIN = 10
IDCONTINUE = 11
IDTIMEOUT = 32000

ACTCTX_FLAG_PROCESSOR_ARCHITECTURE_VALID = 0x00000001
ACTCTX_FLAG_LANGID_VALID = 0x00000002
ACTCTX_FLAG_ASSEMBLY_DIRECTORY_VALID = 0x00000004
ACTCTX_FLAG_RESOURCE_NAME_VALID = 0x00000008
ACTCTX_FLAG_SET_PROCESS_DEFAULT = 0x00000010
ACTCTX_FLAG_APPLICATION_NAME_VALID = 0x00000020
ACTCTX_FLAG_SOURCE_IS_ASSEMBLYREF = 0x00000040
ACTCTX_FLAG_HMODULE_VALID = 0x00000080

SE_UNKNOWN_OBJECT_TYPE = 0
SE_FILE_OBJECT = 1
SE_SERVICE = 2
SE_PRINTER = 3
SE_REGISTRY_KEY = 4
SE_LMSHARE = 5
SE_KERNEL_OBJECT = 6
SE_WINDOW_OBJECT = 7
SE_DS_OBJECT = 8
SE_DS_OBJECT_ALL = 9
SE_PROVIDER_DEFINED_OBJECT = 10
SE_WMIGUID_OBJECT = 11
SE_REGISTRY_WOW64_32KEY = 12
SE_REGISTRY_WOW64_64KEY = 13

LR_DEFAULTCOLOR = 0x00000000
LR_MONOCHROME = 0x00000001
LR_COLOR = 0x00000002
LR_COPYRETURNORG = 0x00000004
LR_COPYDELETEORG = 0x00000008
LR_LOADFROMFILE = 0x00000010
LR_LOADTRANSPARENT = 0x00000020
LR_DEFAULTSIZE = 0x00000040
LR_VGACOLOR = 0x00000080
LR_LOADMAP3DCOLORS = 0x00001000
LR_CREATEDIBSECTION = 0x00002000
LR_COPYFROMRESOURCE = 0x00004000
LR_SHARED = 0x00008000

DONT_RESOLVE_DLL_REFERENCES = 0x00000001
LOAD_LIBRARY_AS_DATAFILE = 0x00000002
LOAD_WITH_ALTERED_SEARCH_PATH = 0x00000008
LOAD_IGNORE_CODE_AUTHZ_LEVEL = 0x00000010
LOAD_LIBRARY_AS_IMAGE_RESOURCE = 0x00000020
LOAD_LIBRARY_AS_DATAFILE_EXCLUSIVE = 0x00000040
LOAD_LIBRARY_REQUIRE_SIGNED_TARGET = 0x00000080
LOAD_LIBRARY_SEARCH_DLL_LOAD_DIR = 0x00000100
LOAD_LIBRARY_SEARCH_APPLICATION_DIR = 0x00000200
LOAD_LIBRARY_SEARCH_USER_DIRS = 0x00000400
LOAD_LIBRARY_SEARCH_SYSTEM32 = 0x00000800
LOAD_LIBRARY_SEARCH_DEFAULT_DIRS = 0x00001000
LOAD_LIBRARY_SAFE_CURRENT_DIRS = 0x00002000
LOAD_LIBRARY_SEARCH_SYSTEM32_NO_FORWARDER = 0x00004000
LOAD_LIBRARY_OS_INTEGRITY_CONTINUITY = 0x00008000

RT_CURSOR = PWSTR(1)
RT_BITMAP = PWSTR(2)
RT_ICON = PWSTR(3)
RT_MENU = PWSTR(4)
RT_DIALOG = PWSTR(5)
RT_STRING = PWSTR(6)
RT_FONTDIR = PWSTR(7)
RT_FONT = PWSTR(8)
RT_ACCELERATOR = PWSTR(9)
RT_RCDATA = PWSTR(10)
RT_MESSAGETABLE = PWSTR(11)
RT_GROUP_CURSOR = PWSTR(12)
RT_GROUP_ICON = PWSTR(14)
RT_VERSION = PWSTR(16)
RT_DLGINCLUDE = PWSTR(17)
RT_PLUGPLAY = PWSTR(19)
RT_VXD = PWSTR(20)
RT_ANICURSOR = PWSTR(21)
RT_ANIICON = PWSTR(22)
RT_HTML = PWSTR(23)
RT_MANIFEST = PWSTR(24)
CREATEPROCESS_MANIFEST_RESOURCE_ID = PWSTR(1)
ISOLATIONAWARE_MANIFEST_RESOURCE_ID = PWSTR(2)
ISOLATIONAWARE_NOSTATICIMPORT_MANIFEST_RESOURCE_ID = PWSTR(3)
ISOLATIONPOLICY_MANIFEST_RESOURCE_ID = PWSTR(4)
ISOLATIONPOLICY_BROWSER_MANIFEST_RESOURCE_ID = PWSTR(5)
MINIMUM_RESERVED_MANIFEST_RESOURCE_ID = PWSTR(1)
MAXIMUM_RESERVED_MANIFEST_RESOURCE_ID = PWSTR(16)

IMAGE_FILE_MACHINE_UNKNOWN = 0
IMAGE_FILE_MACHINE_TARGET_HOST = 0x0001
IMAGE_FILE_MACHINE_I386 = 0x014c
IMAGE_FILE_MACHINE_R3000 = 0x0162
IMAGE_FILE_MACHINE_R4000 = 0x0166
IMAGE_FILE_MACHINE_R10000 = 0x0168
IMAGE_FILE_MACHINE_WCEMIPSV2 = 0x0169
IMAGE_FILE_MACHINE_ALPHA = 0x0184
IMAGE_FILE_MACHINE_SH3 = 0x01a2
IMAGE_FILE_MACHINE_SH3DSP = 0x01a3
IMAGE_FILE_MACHINE_SH3E = 0x01a4
IMAGE_FILE_MACHINE_SH4 = 0x01a6
IMAGE_FILE_MACHINE_SH5 = 0x01a8
IMAGE_FILE_MACHINE_ARM = 0x01c0
IMAGE_FILE_MACHINE_THUMB = 0x01c2
IMAGE_FILE_MACHINE_ARMNT = 0x01c4
IMAGE_FILE_MACHINE_AM33 = 0x01d3
IMAGE_FILE_MACHINE_POWERPC = 0x01F0
IMAGE_FILE_MACHINE_POWERPCFP = 0x01f1
IMAGE_FILE_MACHINE_IA64 = 0x0200
IMAGE_FILE_MACHINE_MIPS16 = 0x0266
IMAGE_FILE_MACHINE_ALPHA64 = 0x0284
IMAGE_FILE_MACHINE_MIPSFPU = 0x0366
IMAGE_FILE_MACHINE_MIPSFPU16 = 0x0466
IMAGE_FILE_MACHINE_AXP64 = IMAGE_FILE_MACHINE_ALPHA64
IMAGE_FILE_MACHINE_TRICORE = 0x0520
IMAGE_FILE_MACHINE_CEF = 0x0CEF
IMAGE_FILE_MACHINE_EBC = 0x0EBC
IMAGE_FILE_MACHINE_AMD64 = 0x8664
IMAGE_FILE_MACHINE_M32R = 0x9041
IMAGE_FILE_MACHINE_ARM64 = 0xAA64
IMAGE_FILE_MACHINE_CEE = 0xC0EE

ES_LEFT = 0x0000
ES_CENTER = 0x0001
ES_RIGHT = 0x0002
ES_MULTILINE = 0x0004
ES_UPPERCASE = 0x0008
ES_LOWERCASE = 0x0010
ES_PASSWORD = 0x0020
ES_AUTOVSCROLL = 0x0040
ES_AUTOHSCROLL = 0x0080
ES_NOHIDESEL = 0x0100
ES_OEMCONVERT = 0x0400
ES_READONLY = 0x0800
ES_WANTRETURN = 0x1000
ES_NUMBER = 0x2000

EM_GETSEL = 0x00B0
EM_SETSEL = 0x00B1
EM_GETRECT = 0x00B2
EM_SETRECT = 0x00B3
EM_SETRECTNP = 0x00B4
EM_SCROLL = 0x00B5
EM_LINESCROLL = 0x00B6
EM_SCROLLCARET = 0x00B7
EM_GETMODIFY = 0x00B8
EM_SETMODIFY = 0x00B9
EM_GETLINECOUNT = 0x00BA
EM_LINEINDEX = 0x00BB
EM_SETHANDLE = 0x00BC
EM_GETHANDLE = 0x00BD
EM_GETTHUMB = 0x00BE
EM_LINELENGTH = 0x00C1
EM_REPLACESEL = 0x00C2
EM_GETLINE = 0x00C4
EM_LIMITTEXT = 0x00C5
EM_CANUNDO = 0x00C6
EM_UNDO = 0x00C7
EM_FMTLINES = 0x00C8
EM_LINEFROMCHAR = 0x00C9
EM_SETTABSTOPS = 0x00CB
EM_SETPASSWORDCHAR = 0x00CC
EM_EMPTYUNDOBUFFER = 0x00CD
EM_GETFIRSTVISIBLELINE = 0x00CE
EM_SETREADONLY = 0x00CF
EM_SETWORDBREAKPROC = 0x00D0
EM_GETWORDBREAKPROC = 0x00D1
EM_GETPASSWORDCHAR = 0x00D2
EM_SETMARGINS = 0x00D3
EM_GETMARGINS = 0x00D4
EM_SETLIMITTEXT = EM_LIMITTEXT
EM_GETLIMITTEXT = 0x00D5
EM_POSFROMCHAR = 0x00D6
EM_CHARFROMPOS = 0x00D7
EM_SETIMESTATUS = 0x00D8
EM_GETIMESTATUS = 0x00D9
EM_ENABLEFEATURE = 0x00DA

SS_LEFT = 0x00000000
SS_CENTER = 0x00000001
SS_RIGHT = 0x00000002
SS_ICON = 0x00000003
SS_BLACKRECT = 0x00000004
SS_GRAYRECT = 0x00000005
SS_WHITERECT = 0x00000006
SS_BLACKFRAME = 0x00000007
SS_GRAYFRAME = 0x00000008
SS_WHITEFRAME = 0x00000009
SS_USERITEM = 0x0000000A
SS_SIMPLE = 0x0000000B
SS_LEFTNOWORDWRAP = 0x0000000C
SS_OWNERDRAW = 0x0000000D
SS_BITMAP = 0x0000000E
SS_ENHMETAFILE = 0x0000000F
SS_ETCHEDHORZ = 0x00000010
SS_ETCHEDVERT = 0x00000011
SS_ETCHEDFRAME = 0x00000012
SS_TYPEMASK = 0x0000001F
SS_REALSIZECONTROL = 0x00000040
SS_NOPREFIX = 0x00000080
SS_NOTIFY = 0x00000100
SS_CENTERIMAGE = 0x00000200
SS_RIGHTJUST = 0x00000400
SS_REALSIZEIMAGE = 0x00000800
SS_SUNKEN = 0x00001000
SS_EDITCONTROL = 0x00002000
SS_ENDELLIPSIS = 0x00004000
SS_PATHELLIPSIS = 0x00008000
SS_WORDELLIPSIS = 0x0000C000
SS_ELLIPSISMASK = 0x0000C000

BS_PUSHBUTTON = 0x00000000
BS_DEFPUSHBUTTON = 0x00000001
BS_CHECKBOX = 0x00000002
BS_AUTOCHECKBOX = 0x00000003
BS_RADIOBUTTON = 0x00000004
BS_3STATE = 0x00000005
BS_AUTO3STATE = 0x00000006
BS_GROUPBOX = 0x00000007
BS_USERBUTTON = 0x00000008
BS_AUTORADIOBUTTON = 0x00000009
BS_PUSHBOX = 0x0000000A
BS_OWNERDRAW = 0x0000000B
BS_TYPEMASK = 0x0000000F
BS_LEFTTEXT = 0x00000020
BS_TEXT = 0x00000000
BS_ICON = 0x00000040
BS_BITMAP = 0x00000080
BS_LEFT = 0x00000100
BS_RIGHT = 0x00000200
BS_CENTER = 0x00000300
BS_TOP = 0x00000400
BS_BOTTOM = 0x00000800
BS_VCENTER = 0x00000C00
BS_PUSHLIKE = 0x00001000
BS_MULTILINE = 0x00002000
BS_NOTIFY = 0x00004000
BS_FLAT = 0x00008000
BS_RIGHTBUTTON = BS_LEFTTEXT

DS_ABSALIGN = 0x01
DS_SYSMODAL = 0x02
DS_LOCALEDIT = 0x20
DS_SETFONT = 0x40
DS_MODALFRAME = 0x80
DS_NOIDLEMSG = 0x100
DS_SETFOREGROUND = 0x200
DS_3DLOOK = 0x0004
DS_FIXEDSYS = 0x0008
DS_NOFAILCREATE = 0x0010
DS_CONTROL = 0x0400
DS_CENTER = 0x0800
DS_CENTERMOUSE = 0x1000
DS_CONTEXTHELP = 0x2000
DS_SHELLFONT = (DS_SETFONT | DS_FIXEDSYS)

STD_INPUT_HANDLE = DWORD(-10).value
STD_OUTPUT_HANDLE = DWORD(-11).value
STD_ERROR_HANDLE = DWORD(-12).value

FILE_TYPE_UNKNOWN = 0x0000
FILE_TYPE_DISK = 0x0001
FILE_TYPE_CHAR = 0x0002
FILE_TYPE_PIPE = 0x0003
FILE_TYPE_REMOTE = 0x8000
