
from rengine.workouts import BaseWorkout
from .config import *
from .data_structures import StrengthExerciseQueue
from .workouts import AutoGeneratedWorkout, dictionary_addition


def get_strenth_exercise_queue(
    muscle_groups: list[MuscleGroup],
    equipment_available: list[EquipmentAvailable] = EquipmentAvailable.ALL,
):
    potential_strength_exercises = []

    if EquipmentAvailable.BARBELL in equipment_available or equipment_available == EquipmentAvailable.ALL:
        if (
            MuscleGroup.HAMSTRINGS in muscle_groups
            or MuscleGroup.QUAD in muscle_groups
        ):
            potential_strength_exercises.append(("Barbell Squat", 1))
            potential_strength_exercises.append(("Romanian Deadlift", 1))
        if MuscleGroup.CHEST in muscle_groups:
            potential_strength_exercises.append(("Barbell Bench Press", 2))
        if MuscleGroup.DELTOIDS in muscle_groups:
            potential_strength_exercises.append(("Barbell Shoulder Press", 3))
        if MuscleGroup.BACK in muscle_groups:
            potential_strength_exercises.append(("Bent-Over Barbell Row", 3))
    elif EquipmentAvailable.SMITH_MACHINE in equipment_available:
        if (
            MuscleGroup.HAMSTRINGS in muscle_groups
            or MuscleGroup.QUAD in muscle_groups
        ):
            potential_strength_exercises.append(("Smith Machine Squat", 1))
            potential_strength_exercises.append(("Smith Machine Deadlift", 1))
        if MuscleGroup.CHEST in muscle_groups:
            potential_strength_exercises.append(("Smith Machine Bench Press", 2))
        if MuscleGroup.DELTOIDS in muscle_groups:
            potential_strength_exercises.append(("Smith Machine Shoulder Press", 3))
        if MuscleGroup.BACK in muscle_groups:
            potential_strength_exercises.append(("Bent-Over Smith Machine Rows", 3))
    if (
        EquipmentAvailable.DIP_BAR in equipment_available
        and MuscleGroup.TRICEPS in muscle_groups
    ):
        potential_strength_exercises.append(("Weighted Dips", 3))
    if (
        EquipmentAvailable.PULLUP_BAR in equipment_available
        and MuscleGroup.BACK in muscle_groups
    ):
        potential_strength_exercises.append(("Weighted Pull-Ups", 3))
    return StrengthExerciseQueue(potential_strength_exercises)


class BaseWorkoutPlan:
    """
    General workout plan class

    Responsibilities
        - Add workouts
        - Delete workouts
        - Export to excel


    """

    def __init__(self, workouts=None):
        self.workouts = workouts or []

    def add_workout(self, workout: BaseWorkout):
        """Simply adds workout to workout plan."""
        assert isinstance(workout, BaseWorkout)
        self.workouts.append(workout)

    def export_to_excel(self):
        """Exports workout plan to a nice looking excel sheet."""
        pass

    @property
    def cummative_load_per_muscle_group(self):
        """Calculates the cummulative load per muscle group."""
        return dictionary_addition([workout.load_per_muscle_group for workout in self.workouts])

    @property
    def unique_exercise_names(self):
        """Returns the set of all exercise names in workout plan."""
        return list(set(
            exercise.exercise_name for workout in self.workouts for exercise in workout.workout
        ))


class AutoGeneratedWorkoutPlan(BaseWorkoutPlan):
    def __init__(
        self,
        target_time_per_workout,
        workout_generation_blueprint: list[dict],
        equipment_available: list[EquipmentAvailable] = EquipmentAvailable.ALL,
        workouts=None,
    ):
        """
        Adds support for auto generating workout plans based on workout_generation_blueprint. This list holds dictionaries with following form:

        workout_generation_blueprint = [
            ("Back", "Biceps", "Calves"),
            ("Chest", "Triceps")
        ]

        This will generate 1 back and bicep workouts then 1 chest and tricep workouts.
        """
        self.target_time_per_workout = target_time_per_workout
        self.workout_generation_blueprint = workout_generation_blueprint
        self.equipment_available = equipment_available
        self._strength_exercise_queues = {}

        super().__init__(workouts)

    def generate(self):
        """Auto generates the workouts."""
        for muscle_groups in self.workout_generation_blueprint:
            self._create_strength_exercise_queue_if_valid(muscle_groups)
            workout = AutoGeneratedWorkout(
                    self.target_time_per_workout,
                    muscle_groups,
                    self._strength_exercise_queues[muscle_groups].get().exercise_name,
                    self.equipment_available,
                    self.unique_exercise_names,
                    self.cummative_load_per_muscle_group,
                )
            workout.create()
            self.add_workout(
                workout
            )

    def _create_strength_exercise_queue_if_valid(self, muscle_groups):
        """If _strength_exercise_queue does not have a key with a group of muscles then create a queue and add to dictionary."""
        if muscle_groups not in self._strength_exercise_queues:
            self._strength_exercise_queues[muscle_groups] = get_strenth_exercise_queue(
                muscle_groups, self.equipment_available
            )
