# -*- coding: utf-8 -*-
#
# This file is part of the parce Python package.
#
# Copyright © 2019-2020 by Wilbert Berendsen <info@wilbertberendsen.nl>
#
# This module is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This module is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.


"""
Updates the language stub files in source/lang and the listing in source/langs.inc
"""

EXAMPLES_DIRECTORY = "../tests/lang/"


import collections
import glob
import io
import os
import sys

sys.path.insert(0, "..")

import parce
import parce.registry
from parce.language import get_all_modules, get_languages
from parce.out.html import HtmlFormatter


STUB = r"""
.. Generated by ../update_languages.py -- do not edit!

{title}

.. automodule:: parce.lang.{module}
   :members:
   :undoc-members:
   :show-inheritance:

   In this module:
   ---------------

{registry}

{examples}

"""

REGISTRY_STUB_HEADER = r"""
.. list-table::
   :header-rows: 1
   :widths: 10, 10, 40, 20, 20

   * - Language
     - Name (Aliases)
     - Description
     - Filename(s)
     - Mime Type(s)
"""

REGISTRY_STUB = r"""
   * - :class:`~parce.lang.{module}.{lang}`
     - {name}
     - {desc}
     - {fnames}
     - {mtypes}
"""

EXAMPLE_STUB = r"""
Root lexicon ``{root}`` and text:

.. admonition:: Text rendered using default theme

   .. raw:: html

{code}

Result tree:

.. code-block:: none
   :class: token-tree

{result}
"""

HTML_STUB = r"""<pre style="overflow: scroll; white-space: pre; {baseformat}">{html}</pre>"""

LANGS_INC_HEADER = r"""
.. Generated by ../update_languages.py -- do not edit!

.. list-table::
   :header-rows: 1
   :widths: 20 70

   * - Module
     - Languages

"""

# mapping from the example files to their Language
all_examples = collections.defaultdict(list)


# formatter for the examples
formatter = HtmlFormatter(parce.theme_by_name('default'))


def title(text, char='-'):
    """Return text with a line of hyphens of the same length below it."""
    return text + '\n' + char * len(text)


def indent(text, indent=3):
    """Indent text."""
    return '\n'.join(' ' * indent + line for line in text.splitlines())


def make_stub(module):
    """Writes a documentation page for a language module."""
    # registry listing, if any
    registered = []
    langs = list(get_languages(module))
    if langs:
        registered.append(REGISTRY_STUB_HEADER)
        langs.sort(key=lambda l: l.__name__)
        for l in langs:
            rootname = parce.registry.find(l.__name__)
            if rootname:
                r = parce.registry.registry[rootname]
                name = r.name
                if r.aliases:
                    name += ' ({})'.format(', '.join(r.aliases))
                desc = r.desc
                fnames = ', '.join('``{}``'.format(fname) for fname, weight in r.filenames)
                mtypes = ', '.join('``{}``'.format(mtype) for mtype, weight in r.mimetypes)
            else:
                name, desc, fnames, mtypes = '', '(not registered)', '', ''
            registered.append(REGISTRY_STUB.format(module=module, lang=l.__name__,
                name=name, desc=desc, fnames=fnames, mtypes=mtypes))

    # examples, if any
    examples = sum((all_examples.get(lang, []) for lang in langs), [])
    example = []
    if examples:
        example.append(title("Examples:" if len(examples) > 1 else "Example:"))

        for root_lexicon, text in examples:
            doc = parce.Document(root_lexicon, text)
            buf = io.StringIO()
            doc.get_root(True).dump(buf)
            cursor = parce.Cursor(doc, 0, None)
            html = HTML_STUB.format(
                baseformat=formatter.baseformat() or "",
                html=formatter.html(cursor))
            example.append(EXAMPLE_STUB.format(root=root_lexicon, language=module,
                code=indent(html, 6), result=indent(buf.getvalue())))

    text = STUB.format(
        title=module + '\n' + '=' * len(module),
        module=module,
        registry=indent(''.join(registered)),
        examples=''.join(example))

    with open("source/lang/{0}.rst".format(module), "w") as f:
        f.write(text)


def get_examples():
    """Return a list of all example documents in tests/lang/example.*."""
    pattern = os.path.join(EXAMPLES_DIRECTORY, "example*.*")
    return glob.glob(pattern)


def load_examples():
    """Load the examples in the global ``all_examples`` variable."""
    # make a mapping from the example files to their Language
    for filename in get_examples():
        text = open(filename).read()    # TODO encoding?
        root_lexicon = parce.find(filename=filename, contents=text)
        all_examples[root_lexicon.language].append((root_lexicon, text))


def main():
    """Main function."""
    load_examples()

    # write the source/langs.inc file and all the module doc files
    with open("source/langs.inc", "w") as f:
        f.write(LANGS_INC_HEADER)
        for name in get_all_modules():
            langs = list(get_languages(name))
            if langs:
                clss = ", ".join(":class:`~parce.lang.{0}.{1}`".format(name, lang.__name__)
                    for lang in langs)
                make_stub(name)
                f.write("   * - :mod:`~parce.lang.{0}`\n".format(name))
                f.write("     - {0}\n\n".format(clss))

if __name__ == "__main__":
    main()

