from typing import Any, Dict, List, Optional, Tuple, Union

import logging

from np_lims_tk.core import get_project_asset, put_project_asset

from .exceptions import NpValidatorError, ParsingError
from .models import (
    BuiltinCallable,
    Processor,
    ValidationResult,
    ValidationStep,
    Validator,
    ValidatorError,
    ValidatorResult,
    dump_ValidationResults,
    dump_ValidationStep,
    load_ValidationStep,
    path_to_ValidationStep,
)
from .path_matching import path_matches_pattern

logger = logging.getLogger(__name__)


def run_BuiltinCallable(target: Any, builtin: BuiltinCallable) -> Any:
    """Runs a BuiltinCallable on an arbitrary value."""
    return builtin.resolved_function(
        target,
        **builtin.args,
    )


def run_ValidationStep(filepath: str, step: ValidationStep) -> ValidationResult:
    """Runs a single validation step.

    Args:
        filepath (str): filepath of data to validate.
        step (ValidationStep): validation step to run.

    Returns:
        result (ValidationResult): result of the validation
    """
    if step.processor:
        try:
            processed = run_BuiltinCallable(
                filepath,
                step.processor.builtin,
            )  # type: Union[Any, str]
        except Exception as e:
            return ValidationResult(
                filepath=filepath,
                errors=[
                    ValidatorError(
                        str(e),
                        processor=step.processor,
                    )
                ],
            )
    else:
        processed = filepath

    results = []
    errors = []
    for validator in step.validators:
        try:
            value, passed = run_BuiltinCallable(
                processed,
                validator.builtin,
            )
            results.append(
                ValidatorResult(
                    value=value,
                    passed=passed,
                    validator=validator,
                )
            )
        except Exception as e:
            errors.append(
                ValidatorError(
                    message=str(e),
                    validator=validator,
                )
            )

    if step.processor:
        return ValidationResult(
            filepath=filepath,
            processor=step.processor,
            results=results,
            errors=errors,
        )
    else:
        return ValidationResult(
            filepath=filepath,
            results=results,
            errors=errors,
        )


def run_validation(
    file_list: List[str],
    validation_steps: List[ValidationStep] = [],
    autogen_steps: bool = True,
    project: str = "default",
    raise_if_errors: bool = False,
) -> List[ValidationResult]:
    """Validates a list of files.

    Args:
        file_list (:obj:`list` of :obj:`str`): List of filepaths to validate.
        validation_steps: List of validation steps to run.
        autogen_steps: Add additional autogenerated ValidationSteps based on file types in `file_list`.
        project: Project id associated with the validation.
        raise_if_errors: Raise an NpValidatorError if ValidationStep errors occur.

    Returns:
        results: List of validation results.

    Raises:
        NpValidatorError: One or more validation steps had an unexpected error.
    """
    if autogen_steps:
        logger.debug("Autogenerating validation steps: project_name=%s" % project)
        autogenerated = autogenerate_validation_steps(project)
        logger.debug("Autogenerated: %s" % autogenerated)
        validation_steps = list(validation_steps)  # cast in case of map
        validation_steps.extend(autogenerated)

    results = []
    for validation_step in list(validation_steps):  # lazy way to accept a map type
        for filepath in file_list:
            if path_matches_pattern(filepath, validation_step.path_suffix):
                results.append(
                    run_ValidationStep(
                        filepath,
                        validation_step,
                    )
                )

    if raise_if_errors:
        for result in results:
            if len(result.errors) > 0:
                raise NpValidatorError("A validation step failed.")

    return results


def update_project_validation_steps(
    project_name: str,
    steps: List[ValidationStep],
    client: Optional[Any] = None,  # for integration testing, remove later
) -> None:
    dumped = [dump_ValidationStep(step) for step in steps]
    put_project_asset(
        project_name,
        "validation_steps",
        dumped,
        client=client,
    )


def autogenerate_validation_steps(
    project_name: str = "default",
    client: Optional[Any] = None,  # for integration testing, remove later
) -> List[ValidationStep]:
    asset = get_project_asset(
        project_name,
        "validation_steps",
        client=client,
    )
    return [load_ValidationStep(step_data) for step_data in asset]
