import { v4 as uuidv4 } from 'uuid'
import { Jwt, ProjectScopedToken, Iota } from './helpers'
import { EnvironmentUtils } from '@affinidi-tdk/common'

export interface IotaTokenOutput {
  readonly iotaJwt: string
  readonly iotaSessionId: string
}

export interface IAuthProviderParams {
  projectId: string
  tokenId: string
  keyId?: string
  privateKey: string
  passphrase?: string
  apiGatewayUrl?: string
  tokenEndpoint?: string
}

export class AuthProvider {
  // NOTE: publicKey to validate projectScopedToken generated by Affinidi,
  //       need to fetch from Affinidi service
  private publicKey: string = ''
  private projectScopedToken: string = ''

  private readonly keyId: string
  private readonly tokenId: string
  private readonly passphrase: string
  private readonly privateKey: string
  private readonly projectId: string

  private readonly apiGatewayUrl: string
  private readonly tokenEndpoint: string

  private readonly projectScopedTokenInstance: ProjectScopedToken
  private readonly jwt: Jwt
  private readonly iotaInstance: Iota

  constructor(param: { [key: string]: string }) {
    this.apiGatewayUrl = param?.apiGW ?? EnvironmentUtils.fetchApiGwUrl()
    this.tokenEndpoint =
      param?.tokenEndpoint ?? EnvironmentUtils.fetchElementsAuthTokenUrl()

    if (!param.privateKey || !param.projectId || !param.tokenId) {
      throw new Error(
        'Missing parameters. Please provide privateKey, projectId and tokenId.',
      )
    }
    this.projectId = param.projectId
    this.tokenId = param.tokenId
    this.keyId = param.keyId ?? param.tokenId
    this.privateKey = param.privateKey
    this.passphrase = param.passphrase

    this.projectScopedTokenInstance = new ProjectScopedToken()
    this.jwt = new Jwt()
    this.iotaInstance = new Iota()
  }

  private async shouldRefreshToken(): Promise<boolean> {
    if (!this.publicKey) {
      this.publicKey = await this.jwt.fetchPublicKey(this.apiGatewayUrl)
    }

    const itExistsAndExpired =
      !!this.projectScopedToken &&
      this.jwt.validateToken(this.projectScopedToken, this.publicKey).isExpired

    return !this.projectScopedToken || itExistsAndExpired
  }

  public async fetchProjectScopedToken(): Promise<string> {
    const shouldRefreshToken = await this.shouldRefreshToken()

    if (shouldRefreshToken) {
      this.projectScopedToken =
        await this.projectScopedTokenInstance.fetchProjectScopedToken({
          apiGatewayUrl: this.apiGatewayUrl,
          keyId: this.keyId,
          tokenId: this.tokenId,
          passphrase: this.passphrase,
          privateKey: this.privateKey,
          projectId: this.projectId,
          audience: this.tokenEndpoint,
        })
    }

    return this.projectScopedToken
  }

  public createIotaToken(
    iotaConfigId: string,
    did: string,
    iotaSessionId?: string,
  ): IotaTokenOutput {
    const sessionId = iotaSessionId ?? uuidv4()

    return {
      iotaJwt: this.iotaInstance.signIotaJwt(
        this.projectId,
        iotaConfigId,
        sessionId,
        {
          keyId: this.keyId,
          tokenId: `token/${this.tokenId}`,
          passphrase: this.passphrase,
          privateKey: this.privateKey,
          audience: did,
        },
      ),
      iotaSessionId: sessionId,
    }
  }
}
