import { Duration } from '@aws-cdk/core';
/**
 * Interface for metrics.
 *
 * @stability stable
 */
export interface IMetric {
    /**
     * Inspect the details of the metric object.
     *
     * @stability stable
     */
    toMetricConfig(): MetricConfig;
    /**
     * (deprecated) Turn this metric object into an alarm configuration.
     *
     * @deprecated Use `toMetricsConfig()` instead.
     */
    toAlarmConfig(): MetricAlarmConfig;
    /**
     * (deprecated) Turn this metric object into a graph configuration.
     *
     * @deprecated Use `toMetricsConfig()` instead.
     */
    toGraphConfig(): MetricGraphConfig;
}
/**
 * Metric dimension.
 *
 * @stability stable
 */
export interface Dimension {
    /**
     * Name of the dimension.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * Value of the dimension.
     *
     * @stability stable
     */
    readonly value: any;
}
/**
 * Statistic to use over the aggregation period.
 *
 * @stability stable
 */
export declare enum Statistic {
    /**
     * The count (number) of data points used for the statistical calculation.
     *
     * @stability stable
     */
    SAMPLE_COUNT = "SampleCount",
    /**
     * The value of Sum / SampleCount during the specified period.
     *
     * @stability stable
     */
    AVERAGE = "Average",
    /**
     * All values submitted for the matching metric added together.
     *
     * This statistic can be useful for determining the total volume of a metric.
     *
     * @stability stable
     */
    SUM = "Sum",
    /**
     * The lowest value observed during the specified period.
     *
     * You can use this value to determine low volumes of activity for your application.
     *
     * @stability stable
     */
    MINIMUM = "Minimum",
    /**
     * The highest value observed during the specified period.
     *
     * You can use this value to determine high volumes of activity for your application.
     *
     * @stability stable
     */
    MAXIMUM = "Maximum"
}
/**
 * Unit for metric.
 *
 * @stability stable
 */
export declare enum Unit {
    /**
     * Seconds.
     *
     * @stability stable
     */
    SECONDS = "Seconds",
    /**
     * Microseconds.
     *
     * @stability stable
     */
    MICROSECONDS = "Microseconds",
    /**
     * Milliseconds.
     *
     * @stability stable
     */
    MILLISECONDS = "Milliseconds",
    /**
     * Bytes.
     *
     * @stability stable
     */
    BYTES = "Bytes",
    /**
     * Kilobytes.
     *
     * @stability stable
     */
    KILOBYTES = "Kilobytes",
    /**
     * Megabytes.
     *
     * @stability stable
     */
    MEGABYTES = "Megabytes",
    /**
     * Gigabytes.
     *
     * @stability stable
     */
    GIGABYTES = "Gigabytes",
    /**
     * Terabytes.
     *
     * @stability stable
     */
    TERABYTES = "Terabytes",
    /**
     * Bits.
     *
     * @stability stable
     */
    BITS = "Bits",
    /**
     * Kilobits.
     *
     * @stability stable
     */
    KILOBITS = "Kilobits",
    /**
     * Megabits.
     *
     * @stability stable
     */
    MEGABITS = "Megabits",
    /**
     * Gigabits.
     *
     * @stability stable
     */
    GIGABITS = "Gigabits",
    /**
     * Terabits.
     *
     * @stability stable
     */
    TERABITS = "Terabits",
    /**
     * Percent.
     *
     * @stability stable
     */
    PERCENT = "Percent",
    /**
     * Count.
     *
     * @stability stable
     */
    COUNT = "Count",
    /**
     * Bytes/second (B/s).
     *
     * @stability stable
     */
    BYTES_PER_SECOND = "Bytes/Second",
    /**
     * Kilobytes/second (kB/s).
     *
     * @stability stable
     */
    KILOBYTES_PER_SECOND = "Kilobytes/Second",
    /**
     * Megabytes/second (MB/s).
     *
     * @stability stable
     */
    MEGABYTES_PER_SECOND = "Megabytes/Second",
    /**
     * Gigabytes/second (GB/s).
     *
     * @stability stable
     */
    GIGABYTES_PER_SECOND = "Gigabytes/Second",
    /**
     * Terabytes/second (TB/s).
     *
     * @stability stable
     */
    TERABYTES_PER_SECOND = "Terabytes/Second",
    /**
     * Bits/second (b/s).
     *
     * @stability stable
     */
    BITS_PER_SECOND = "Bits/Second",
    /**
     * Kilobits/second (kb/s).
     *
     * @stability stable
     */
    KILOBITS_PER_SECOND = "Kilobits/Second",
    /**
     * Megabits/second (Mb/s).
     *
     * @stability stable
     */
    MEGABITS_PER_SECOND = "Megabits/Second",
    /**
     * Gigabits/second (Gb/s).
     *
     * @stability stable
     */
    GIGABITS_PER_SECOND = "Gigabits/Second",
    /**
     * Terabits/second (Tb/s).
     *
     * @stability stable
     */
    TERABITS_PER_SECOND = "Terabits/Second",
    /**
     * Count/second.
     *
     * @stability stable
     */
    COUNT_PER_SECOND = "Count/Second",
    /**
     * No unit.
     *
     * @stability stable
     */
    NONE = "None"
}
/**
 * Properties of a rendered metric.
 *
 * @stability stable
 */
export interface MetricConfig {
    /**
     * In case the metric represents a query, the details of the query.
     *
     * @default - None
     * @stability stable
     */
    readonly metricStat?: MetricStatConfig;
    /**
     * In case the metric is a math expression, the details of the math expression.
     *
     * @default - None
     * @stability stable
     */
    readonly mathExpression?: MetricExpressionConfig;
    /**
     * Additional properties which will be rendered if the metric is used in a dashboard.
     *
     * Examples are 'label' and 'color', but any key in here will be
     * added to dashboard graphs.
     *
     * @default - None
     * @stability stable
     */
    readonly renderingProperties?: Record<string, unknown>;
}
/**
 * Properties for a concrete metric.
 *
 * NOTE: `unit` is no longer on this object since it is only used for `Alarms`, and doesn't mean what one
 * would expect it to mean there anyway. It is most likely to be misused.
 *
 * @stability stable
 */
export interface MetricStatConfig {
    /**
     * The dimensions to apply to the alarm.
     *
     * @default []
     * @stability stable
     */
    readonly dimensions?: Dimension[];
    /**
     * Namespace of the metric.
     *
     * @stability stable
     */
    readonly namespace: string;
    /**
     * Name of the metric.
     *
     * @stability stable
     */
    readonly metricName: string;
    /**
     * How many seconds to aggregate over.
     *
     * @stability stable
     */
    readonly period: Duration;
    /**
     * Aggregation function to use (can be either simple or a percentile).
     *
     * @stability stable
     */
    readonly statistic: string;
    /**
     * Unit used to filter the metric stream.
     *
     * Only refer to datums emitted to the metric stream with the given unit and
     * ignore all others. Only useful when datums are being emitted to the same
     * metric stream under different units.
     *
     * This field has been renamed from plain `unit` to clearly communicate
     * its purpose.
     *
     * @default - Refer to all metric datums
     * @stability stable
     */
    readonly unitFilter?: Unit;
    /**
     * Region which this metric comes from.
     *
     * @default Deployment region.
     * @stability stable
     */
    readonly region?: string;
    /**
     * Account which this metric comes from.
     *
     * @default Deployment account.
     * @stability stable
     */
    readonly account?: string;
}
/**
 * Properties for a concrete metric.
 *
 * @stability stable
 */
export interface MetricExpressionConfig {
    /**
     * Math expression for the metric.
     *
     * @stability stable
     */
    readonly expression: string;
    /**
     * Metrics used in the math expression.
     *
     * @stability stable
     */
    readonly usingMetrics: Record<string, IMetric>;
    /**
     * How many seconds to aggregate over.
     *
     * @stability stable
     */
    readonly period: number;
}
/**
 * (deprecated) Properties used to construct the Metric identifying part of an Alarm.
 *
 * @deprecated Replaced by MetricConfig
 */
export interface MetricAlarmConfig {
    /**
     * (deprecated) The dimensions to apply to the alarm.
     *
     * @deprecated
     */
    readonly dimensions?: Dimension[];
    /**
     * (deprecated) Namespace of the metric.
     *
     * @deprecated
     */
    readonly namespace: string;
    /**
     * (deprecated) Name of the metric.
     *
     * @deprecated
     */
    readonly metricName: string;
    /**
     * (deprecated) How many seconds to aggregate over.
     *
     * @deprecated
     */
    readonly period: number;
    /**
     * (deprecated) Simple aggregation function to use.
     *
     * @deprecated
     */
    readonly statistic?: Statistic;
    /**
     * (deprecated) Percentile aggregation function to use.
     *
     * @deprecated
     */
    readonly extendedStatistic?: string;
    /**
     * (deprecated) The unit of the alarm.
     *
     * @deprecated
     */
    readonly unit?: Unit;
}
/**
 * (deprecated) Properties used to construct the Metric identifying part of a Graph.
 *
 * @deprecated Replaced by MetricConfig
 */
export interface MetricGraphConfig {
    /**
     * (deprecated) The dimensions to apply to the alarm.
     *
     * @deprecated
     */
    readonly dimensions?: Dimension[];
    /**
     * (deprecated) Namespace of the metric.
     *
     * @deprecated
     */
    readonly namespace: string;
    /**
     * (deprecated) Name of the metric.
     *
     * @deprecated
     */
    readonly metricName: string;
    /**
     * (deprecated) Rendering properties override yAxis parameter of the widget object.
     *
     * @deprecated
     */
    readonly renderingProperties: MetricRenderingProperties;
    /**
     * (deprecated) How many seconds to aggregate over.
     *
     * @deprecated Use `period` in `renderingProperties`
     */
    readonly period: number;
    /**
     * (deprecated) Label for the metric.
     *
     * @deprecated Use `label` in `renderingProperties`
     */
    readonly label?: string;
    /**
     * (deprecated) Color for the graph line.
     *
     * @deprecated Use `color` in `renderingProperties`
     */
    readonly color?: string;
    /**
     * (deprecated) Aggregation function to use (can be either simple or a percentile).
     *
     * @deprecated Use `stat` in `renderingProperties`
     */
    readonly statistic?: string;
    /**
     * (deprecated) The unit of the alarm.
     *
     * @deprecated not used in dashboard widgets
     */
    readonly unit?: Unit;
}
/**
 * (deprecated) Custom rendering properties that override the default rendering properties specified in the yAxis parameter of the widget object.
 *
 * @deprecated Replaced by MetricConfig.
 */
export interface MetricRenderingProperties {
    /**
     * (deprecated) How many seconds to aggregate over.
     *
     * @deprecated
     */
    readonly period: number;
    /**
     * (deprecated) Label for the metric.
     *
     * @deprecated
     */
    readonly label?: string;
    /**
     * (deprecated) The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The `Color` class has a set of standard colors that can be used here.
     *
     * @deprecated
     */
    readonly color?: string;
    /**
     * (deprecated) Aggregation function to use (can be either simple or a percentile).
     *
     * @deprecated
     */
    readonly stat?: string;
}
