"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ModelAuthTransformer = void 0;
const graphql_transformer_core_1 = require("graphql-transformer-core");
const cloudform_types_1 = require("cloudform-types");
const resources_1 = require("./resources");
const graphql_1 = require("graphql");
const graphql_transformer_common_1 = require("graphql-transformer-common");
const graphql_mapping_template_1 = require("graphql-mapping-template");
const ModelDirectiveConfiguration_1 = require("./ModelDirectiveConfiguration");
const constants_1 = require("./constants");
const validateAuthModes = (authConfig) => {
    let additionalAuthModes = [];
    if (authConfig.additionalAuthenticationProviders) {
        additionalAuthModes = authConfig.additionalAuthenticationProviders.map(p => p.authenticationType).filter(t => !!t);
    }
    const authModes = [...additionalAuthModes, authConfig.defaultAuthentication.authenticationType];
    for (let i = 0; i < authModes.length; i++) {
        const mode = authModes[i];
        if (mode !== 'API_KEY' && mode !== 'AMAZON_COGNITO_USER_POOLS' && mode !== 'AWS_IAM' && mode !== 'OPENID_CONNECT') {
            throw new Error(`Invalid auth mode ${mode}`);
        }
    }
};
class ModelAuthTransformer extends graphql_transformer_core_1.Transformer {
    constructor(config) {
        super('ModelAuthTransformer', graphql_transformer_core_1.gql `
        directive @auth(rules: [AuthRule!]!) on OBJECT | FIELD_DEFINITION
        input AuthRule {
          # Specifies the auth rule's strategy. Allowed values are 'owner', 'groups', 'public', 'private'.
          allow: AuthStrategy!

          # Legacy name for identityClaim
          identityField: String @deprecated(reason: "The 'identityField' argument is replaced by the 'identityClaim'.")

          # Specifies the name of the provider to use for the rule. This overrides the default provider
          # when 'public' and 'private' AuthStrategy is used. Specifying a provider for 'owner' or 'groups'
          # are not allowed.
          provider: AuthProvider

          # Specifies the name of the claim to look for on the request's JWT token
          # from Cognito User Pools (and in the future OIDC) that contains the identity
          # of the user. If 'allow' is 'groups', this value should point to a list of groups
          # in the claims. If 'allow' is 'owner', this value should point to the logged in user identity string.
          # Defaults to "cognito:username" for Cognito User Pools auth.
          identityClaim: String

          # Allows for custom config of 'groups' which is validated against the JWT
          # Specifies a static list of groups that should have access to the object
          groupClaim: String

          # Allowed when the 'allow' argument is 'owner'.
          # Specifies the field of type String or [String] that contains owner(s) that can access the object.
          ownerField: String # defaults to "owner"
          # Allowed when the 'allow' argument is 'groups'.
          # Specifies the field of type String or [String] that contains group(s) that can access the object.
          groupsField: String

          # Allowed when the 'allow' argument is 'groups'.
          # Specifies a static list of groups that should have access to the object.
          groups: [String]

          # Specifies operations to which this auth rule should be applied.
          operations: [ModelOperation]

          # Deprecated. It is recommended to use the 'operations' arguments.
          queries: [ModelQuery]
            @deprecated(reason: "The 'queries' argument will be replaced by the 'operations' argument in a future release.")

          # Deprecated. It is recommended to use the 'operations' arguments.
          mutations: [ModelMutation]
            @deprecated(reason: "The 'mutations' argument will be replaced by the 'operations' argument in a future release.")
        }
        enum AuthStrategy {
          owner
          groups
          private
          public
        }
        enum AuthProvider {
          apiKey
          iam
          oidc
          userPools
        }
        enum ModelOperation {
          create
          update
          delete
          read
        }
        enum ModelQuery @deprecated(reason: "ModelQuery will be replaced by the 'ModelOperation' in a future release.") {
          get
          list
        }
        enum ModelMutation @deprecated(reason: "ModelMutation will be replaced by the 'ModelOperation' in a future release.") {
          create
          update
          delete
        }
      `);
        this.updateAPIAuthentication = (ctx) => {
            const apiRecord = ctx.getResource(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID);
            const updated = this.resources.updateGraphQLAPIWithAuth(apiRecord, this.config.authConfig);
            ctx.setResource(graphql_transformer_common_1.ResourceConstants.RESOURCES.GraphQLAPILogicalID, updated);
        };
        this.before = (ctx) => {
            const template = this.resources.initTemplate(this.getApiKeyConfig());
            ctx.mergeResources(template.Resources);
            ctx.mergeParameters(template.Parameters);
            ctx.mergeOutputs(template.Outputs);
            ctx.mergeConditions(template.Conditions);
            this.updateAPIAuthentication(ctx);
            if (!ctx.metadata.has(constants_1.AUTH_NON_MODEL_TYPES)) {
                ctx.metadata.set(constants_1.AUTH_NON_MODEL_TYPES, new Set());
            }
        };
        this.after = (ctx) => {
            if (this.generateIAMPolicyforAuthRole === true) {
                if (this.authPolicyResources.size === 0) {
                    throw new Error('AuthRole policies should be generated, but no resources were added');
                }
                ctx.mergeParameters({
                    [graphql_transformer_common_1.ResourceConstants.PARAMETERS.AuthRoleName]: new cloudform_types_1.StringParameter({
                        Description: 'Reference to the name of the Auth Role created for the project.',
                    }),
                });
                const authPolicies = this.resources.makeIAMPolicyForRole(true, this.authPolicyResources);
                for (let i = 0; i < authPolicies.length; i++) {
                    const paddedIndex = `${i + 1}`.padStart(2, '0');
                    const resourceName = `${graphql_transformer_common_1.ResourceConstants.RESOURCES.AuthRolePolicy}${paddedIndex}`;
                    ctx.mergeResources({
                        [resourceName]: authPolicies[i],
                    });
                }
            }
            if (this.generateIAMPolicyforUnauthRole === true) {
                if (this.unauthPolicyResources.size === 0) {
                    throw new Error('UnauthRole policies should be generated, but no resources were added');
                }
                ctx.mergeParameters({
                    [graphql_transformer_common_1.ResourceConstants.PARAMETERS.UnauthRoleName]: new cloudform_types_1.StringParameter({
                        Description: 'Reference to the name of the Unauth Role created for the project.',
                    }),
                });
                const unauthPolicies = this.resources.makeIAMPolicyForRole(false, this.unauthPolicyResources);
                for (let i = 0; i < unauthPolicies.length; i++) {
                    const paddedIndex = `${i + 1}`.padStart(2, '0');
                    const resourceName = `${graphql_transformer_common_1.ResourceConstants.RESOURCES.UnauthRolePolicy}${paddedIndex}`;
                    ctx.mergeResources({
                        [resourceName]: unauthPolicies[i],
                    });
                }
            }
        };
        this.object = (def, directive, ctx) => {
            const modelDirective = def.directives.find(dir => dir.name.value === 'model');
            if (!modelDirective) {
                throw new graphql_transformer_core_1.InvalidDirectiveError('Types annotated with @auth must also be annotated with @model.');
            }
            const searchableDirective = def.directives.find(dir => dir.name.value === 'searchable');
            const rules = this.getAuthRulesFromDirective(directive);
            this.ensureDefaultAuthProviderAssigned(rules);
            this.validateRules(rules);
            this.addOwnerFieldsToObject(ctx, def.name.value, rules);
            this.setAuthPolicyFlag(rules);
            this.setUnauthPolicyFlag(rules);
            this.propagateAuthDirectivesToNestedTypes(def, rules, ctx);
            const { operationRules, queryRules } = this.splitRules(rules);
            const modelConfiguration = new ModelDirectiveConfiguration_1.ModelDirectiveConfiguration(modelDirective, def);
            const directives = this.getDirectivesForRules(rules, false);
            if (directives.length > 0) {
                this.extendTypeWithDirectives(ctx, def.name.value, directives);
            }
            this.addTypeToResourceReferences(def.name.value, rules);
            this.protectCreateMutation(ctx, graphql_transformer_common_1.ResolverResourceIDs.DynamoDBCreateResolverResourceID(def.name.value), operationRules.create, def, modelConfiguration);
            this.protectUpdateMutation(ctx, graphql_transformer_common_1.ResolverResourceIDs.DynamoDBUpdateResolverResourceID(def.name.value), operationRules.update, def, modelConfiguration);
            this.protectDeleteMutation(ctx, graphql_transformer_common_1.ResolverResourceIDs.DynamoDBDeleteResolverResourceID(def.name.value), operationRules.delete, def, modelConfiguration);
            this.protectGetQuery(ctx, graphql_transformer_common_1.ResolverResourceIDs.DynamoDBGetResolverResourceID(def.name.value), queryRules.get, def, modelConfiguration);
            this.protectListQuery(ctx, graphql_transformer_common_1.ResolverResourceIDs.DynamoDBListResolverResourceID(def.name.value), queryRules.list, def, modelConfiguration);
            this.protectConnections(ctx, def, operationRules.read, modelConfiguration);
            this.protectQueries(ctx, def, operationRules.read, modelConfiguration);
            if (searchableDirective) {
                this.protectSearchQuery(ctx, def, graphql_transformer_common_1.ResolverResourceIDs.ElasticsearchSearchResolverResourceID(def.name.value), operationRules.read);
            }
            if (this.isSyncEnabled(ctx, def.name.value)) {
                this.protectSyncQuery(ctx, def, graphql_transformer_common_1.ResolverResourceIDs.SyncResolverResourceID(def.name.value), operationRules.read);
            }
            if (modelConfiguration.getName('level') !== 'off') {
                this.protectOnCreateSubscription(ctx, operationRules.read, def, modelConfiguration);
                this.protectOnUpdateSubscription(ctx, operationRules.read, def, modelConfiguration);
                this.protectOnDeleteSubscription(ctx, operationRules.read, def, modelConfiguration);
            }
            this.updateMutationConditionInput(ctx, def, rules);
        };
        this.field = (parent, definition, directive, ctx) => {
            if (parent.kind === graphql_1.Kind.INTERFACE_TYPE_DEFINITION) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`The @auth directive cannot be placed on an interface's field. See ${parent.name.value}${definition.name.value}`);
            }
            const modelDirective = parent.directives.find(dir => dir.name.value === 'model');
            const isParentTypeBuiltinType = parent.name.value === ctx.getQueryTypeName() ||
                parent.name.value === ctx.getMutationTypeName() ||
                parent.name.value === ctx.getSubscriptionTypeName();
            if (isParentTypeBuiltinType) {
                console.warn(`Be careful when using @auth directives on a field in a root type. @auth directives on field definitions use the source \
object to perform authorization logic and the source will be an empty object for fields on root types. \
Static group authorization should perform as expected.`);
            }
            const rules = this.getAuthRulesFromDirective(directive);
            if (!isParentTypeBuiltinType) {
                this.addOwnerFieldsToObject(ctx, parent.name.value, rules);
            }
            this.ensureDefaultAuthProviderAssigned(rules);
            this.validateFieldRules(rules, isParentTypeBuiltinType, modelDirective !== undefined);
            this.setAuthPolicyFlag(rules);
            this.setUnauthPolicyFlag(rules);
            this.addFieldToResourceReferences(parent.name.value, definition.name.value, rules);
            const includeDefault = this.isTypeNeedsDefaultProviderAccess(parent);
            const typeDirectives = isParentTypeBuiltinType ? [] : this.getDirectivesForRules(rules, includeDefault);
            if (typeDirectives.length > 0) {
                this.extendTypeWithDirectives(ctx, parent.name.value, typeDirectives);
            }
            const isOpRule = (op) => (rule) => {
                if (rule.operations) {
                    const matchesOp = rule.operations.find(o => o === op);
                    return Boolean(matchesOp);
                }
                if (rule.operations === null) {
                    return false;
                }
                return true;
            };
            if (modelDirective) {
                const isReadRule = isOpRule('read');
                const isCreateRule = isOpRule('create');
                const isUpdateRule = isOpRule('update');
                const isDeleteRule = isOpRule('delete');
                const modelConfiguration = new ModelDirectiveConfiguration_1.ModelDirectiveConfiguration(modelDirective, parent);
                const readRules = rules.filter((rule) => isReadRule(rule));
                this.protectReadForField(ctx, parent, definition, readRules, modelConfiguration);
                const createRules = rules.filter((rule) => isCreateRule(rule));
                this.protectCreateForField(ctx, parent, definition, createRules, modelConfiguration);
                const updateRules = rules.filter((rule) => isUpdateRule(rule));
                this.protectUpdateForField(ctx, parent, definition, updateRules, modelConfiguration);
                const deleteRules = rules.filter((rule) => isDeleteRule(rule));
                this.protectDeleteForField(ctx, parent, definition, deleteRules, modelConfiguration);
            }
            else {
                const directives = this.getDirectivesForRules(rules, false);
                if (directives.length > 0) {
                    this.addDirectivesToField(ctx, parent.name.value, definition.name.value, directives);
                }
                const staticGroupRules = rules.filter((rule) => rule.groups);
                this.protectField(ctx, parent, definition, staticGroupRules);
            }
        };
        if (config && config.authConfig) {
            this.config = config;
            if (!this.config.authConfig.additionalAuthenticationProviders) {
                this.config.authConfig.additionalAuthenticationProviders = [];
            }
        }
        else {
            this.config = { authConfig: { defaultAuthentication: { authenticationType: 'API_KEY' }, additionalAuthenticationProviders: [] } };
        }
        validateAuthModes(this.config.authConfig);
        this.resources = new resources_1.ResourceFactory();
        this.configuredAuthProviders = this.getConfiguredAuthProviders();
        this.generateIAMPolicyforUnauthRole = false;
        this.generateIAMPolicyforAuthRole = false;
        this.authPolicyResources = new Set();
        this.unauthPolicyResources = new Set();
    }
    getApiKeyConfig() {
        let authProviders = [];
        if (this.config.authConfig.additionalAuthenticationProviders) {
            authProviders = authProviders.concat(this.config.authConfig.additionalAuthenticationProviders.filter(p => !!p.authenticationType));
        }
        authProviders.push(this.config.authConfig.defaultAuthentication);
        const apiKeyAuthProvider = authProviders.find(p => p.authenticationType === 'API_KEY');
        return apiKeyAuthProvider ? apiKeyAuthProvider.apiKeyConfig : { apiKeyExpirationDays: 7 };
    }
    propagateAuthDirectivesToNestedTypes(type, rules, ctx) {
        const seenNonModelTypes = ctx.metadata.get(constants_1.AUTH_NON_MODEL_TYPES);
        const nonModelTypePredicate = (fieldType) => {
            if (fieldType) {
                if (fieldType.kind !== 'ObjectTypeDefinition') {
                    return undefined;
                }
                const typeModel = fieldType.directives.find(dir => dir.name.value === 'model');
                return typeModel !== undefined ? undefined : fieldType;
            }
            return fieldType;
        };
        const nonModelFieldTypes = type.fields.map(f => ctx.getType(graphql_transformer_common_1.getBaseType(f.type))).filter(nonModelTypePredicate);
        for (const nonModelFieldType of nonModelFieldTypes) {
            if (!seenNonModelTypes.has(nonModelFieldType.name.value)) {
                seenNonModelTypes.add(nonModelFieldType.name.value);
                const directives = this.getDirectivesForRules(rules, false);
                if (directives.length > 0) {
                    this.extendTypeWithDirectives(ctx, nonModelFieldType.name.value, directives);
                }
                const hasIAM = directives.filter(directive => directive.name.value === 'aws_iam') || this.configuredAuthProviders.default === 'iam';
                if (hasIAM) {
                    this.unauthPolicyResources.add(`${nonModelFieldType.name.value}/null`);
                    this.authPolicyResources.add(`${nonModelFieldType.name.value}/null`);
                }
                this.propagateAuthDirectivesToNestedTypes(nonModelFieldType, rules, ctx);
            }
        }
    }
    protectField(ctx, parent, field, staticGroupRules) {
        const typeName = parent.name.value;
        const fieldName = field.name.value;
        const resolverResourceId = graphql_transformer_common_1.ResolverResourceIDs.ResolverResourceID(typeName, fieldName);
        let fieldResolverResource = ctx.getResource(resolverResourceId);
        const templateParts = [];
        if (staticGroupRules && staticGroupRules.length) {
            const staticGroupAuthorizationRules = this.getStaticGroupRules(staticGroupRules);
            const staticGroupAuthorizationExpression = this.resources.staticGroupAuthorizationExpression(staticGroupAuthorizationRules, field);
            const throwIfUnauthorizedExpression = this.resources.throwIfStaticGroupUnauthorized(field);
            const authModesToCheck = new Set();
            const expressions = new Array();
            if (staticGroupAuthorizationRules.find(r => r.provider === 'userPools')) {
                authModesToCheck.add('userPools');
            }
            if (staticGroupAuthorizationRules.find(r => r.provider === 'oidc')) {
                authModesToCheck.add('oidc');
            }
            if (authModesToCheck.size > 0) {
                const isUserPoolTheDefault = this.configuredAuthProviders.default === 'userPools';
                expressions.push(this.resources.getAuthModeDeterminationExpression(authModesToCheck, isUserPoolTheDefault));
            }
            const authCheckExpressions = [staticGroupAuthorizationExpression, graphql_mapping_template_1.newline(), throwIfUnauthorizedExpression];
            expressions.push(this.resources.getAuthModeCheckWrappedExpression(authModesToCheck, graphql_mapping_template_1.compoundExpression(authCheckExpressions)));
            templateParts.push(graphql_mapping_template_1.print(graphql_mapping_template_1.compoundExpression(expressions)));
        }
        if (!fieldResolverResource) {
            fieldResolverResource = this.resources.blankResolver(typeName, fieldName);
            ctx.setResource(resolverResourceId, fieldResolverResource);
            const noneDS = ctx.getResource(graphql_transformer_common_1.ResourceConstants.RESOURCES.NoneDataSource);
            if (!noneDS) {
                ctx.setResource(graphql_transformer_common_1.ResourceConstants.RESOURCES.NoneDataSource, this.resources.noneDataSource());
            }
        }
        templateParts.push(fieldResolverResource.Properties.RequestMappingTemplate);
        fieldResolverResource.Properties.RequestMappingTemplate = templateParts.join('\n\n');
        ctx.setResource(resolverResourceId, fieldResolverResource);
    }
    protectReadForField(ctx, parent, field, rules, modelConfiguration) {
        if (rules && rules.length) {
            const directives = this.getDirectivesForRules(rules, false);
            if (directives.length > 0) {
                this.addDirectivesToField(ctx, parent.name.value, field.name.value, directives);
                const addDirectivesForOperation = (operationType) => {
                    if (modelConfiguration.shouldHave(operationType)) {
                        const operationName = modelConfiguration.getName(operationType);
                        const includeDefault = this.isTypeHasRulesForOperation(parent, operationType);
                        const operationDirectives = this.getDirectivesForRules(rules, includeDefault);
                        this.addDirectivesToOperation(ctx, ctx.getQueryTypeName(), operationName, operationDirectives);
                    }
                };
                addDirectivesForOperation('get');
                addDirectivesForOperation('list');
            }
            const addResourceReference = (operationType) => {
                if (modelConfiguration.shouldHave(operationType)) {
                    const operationName = modelConfiguration.getName(operationType);
                    this.addFieldToResourceReferences(ctx.getQueryTypeName(), operationName, rules);
                }
            };
            addResourceReference('get');
            addResourceReference('list');
            const resolverResourceId = graphql_transformer_common_1.ResolverResourceIDs.ResolverResourceID(parent.name.value, field.name.value);
            let resolver = ctx.getResource(resolverResourceId);
            if (!resolver) {
                const noneDS = ctx.getResource(graphql_transformer_common_1.ResourceConstants.RESOURCES.NoneDataSource);
                if (!noneDS) {
                    ctx.setResource(graphql_transformer_common_1.ResourceConstants.RESOURCES.NoneDataSource, this.resources.noneDataSource());
                }
                ctx.mapResourceToStack(parent.name.value, resolverResourceId);
                resolver = this.resources.blankResolver(parent.name.value, field.name.value);
            }
            const authExpression = this.authorizationExpressionOnSingleObject(rules, 'ctx.source');
            if (modelConfiguration.getName('level') === 'on') {
                if (field.type.kind === graphql_1.Kind.NON_NULL_TYPE) {
                    throw new graphql_transformer_core_1.InvalidDirectiveError(`\nPer-field auth on the required field ${field.name.value} is not supported with subscriptions.
Either make the field optional, set auth on the object and not the field, or disable subscriptions for the object (setting level to off or public)\n`);
                }
                resolver.Properties.ResponseMappingTemplate = graphql_mapping_template_1.print(this.resources.operationCheckExpression(ctx.getMutationTypeName(), field.name.value));
            }
            const templateParts = [graphql_mapping_template_1.print(authExpression), resolver.Properties.RequestMappingTemplate];
            resolver.Properties.RequestMappingTemplate = templateParts.join('\n\n');
            ctx.setResource(resolverResourceId, resolver);
        }
    }
    protectUpdateForField(ctx, parent, field, rules, modelConfiguration) {
        const resolverResourceId = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBUpdateResolverResourceID(parent.name.value);
        const subscriptionOperation = 'onUpdate';
        this.protectUpdateMutation(ctx, resolverResourceId, rules, parent, modelConfiguration, field, subscriptionOperation);
    }
    protectDeleteForField(ctx, parent, field, rules, modelConfiguration) {
        const resolverResourceId = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBUpdateResolverResourceID(parent.name.value);
        const subscriptionOperation = 'onDelete';
        this.protectDeleteMutation(ctx, resolverResourceId, rules, parent, modelConfiguration, field, subscriptionOperation);
    }
    protectCreateForField(ctx, parent, field, rules, modelConfiguration) {
        const typeName = parent.name.value;
        const resolverResourceId = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBCreateResolverResourceID(typeName);
        const createResolverResource = ctx.getResource(resolverResourceId);
        const mutationTypeName = ctx.getMutationTypeName();
        if (rules && rules.length && createResolverResource) {
            const directives = this.getDirectivesForRules(rules, false);
            let operationName = undefined;
            if (directives.length > 0) {
                this.addDirectivesToField(ctx, typeName, field.name.value, directives);
                if (modelConfiguration.shouldHave('create')) {
                    const includeDefault = this.isTypeHasRulesForOperation(parent, 'create');
                    const operationDirectives = this.getDirectivesForRules(rules, includeDefault);
                    operationName = modelConfiguration.getName('create');
                    this.addDirectivesToOperation(ctx, mutationTypeName, operationName, operationDirectives);
                }
            }
            if (operationName) {
                this.addFieldToResourceReferences(mutationTypeName, operationName, rules);
            }
            const staticGroupAuthorizationRules = this.getStaticGroupRules(rules);
            const dynamicGroupAuthorizationRules = this.getDynamicGroupRules(rules);
            const ownerAuthorizationRules = this.getOwnerRules(rules);
            const providerAuthorization = this.hasProviderAuthRules(rules);
            if ((staticGroupAuthorizationRules.length > 0 || dynamicGroupAuthorizationRules.length > 0 || ownerAuthorizationRules.length > 0) &&
                providerAuthorization === false) {
                const staticGroupAuthorizationExpression = this.resources.staticGroupAuthorizationExpression(staticGroupAuthorizationRules, field);
                const dynamicGroupAuthorizationExpression = this.resources.dynamicGroupAuthorizationExpressionForCreateOperationsByField(dynamicGroupAuthorizationRules, field.name.value);
                const fieldIsList = (fieldName) => {
                    const field = parent.fields.find(field => field.name.value === fieldName);
                    if (field) {
                        return graphql_transformer_common_1.isListType(field.type);
                    }
                    return false;
                };
                const ownerAuthorizationExpression = this.resources.ownerAuthorizationExpressionForCreateOperationsByField(ownerAuthorizationRules, field.name.value, fieldIsList);
                const throwIfUnauthorizedExpression = this.resources.throwIfUnauthorized(field);
                const authModesToCheck = new Set();
                const expressions = new Array();
                if (ownerAuthorizationRules.find(r => r.provider === 'userPools') ||
                    staticGroupAuthorizationRules.find(r => r.provider === 'userPools') ||
                    dynamicGroupAuthorizationRules.find(r => r.provider === 'userPools')) {
                    authModesToCheck.add('userPools');
                }
                if (ownerAuthorizationRules.find(r => r.provider === 'oidc') ||
                    staticGroupAuthorizationRules.find(r => r.provider === 'oidc') ||
                    dynamicGroupAuthorizationRules.find(r => r.provider === 'oidc')) {
                    authModesToCheck.add('oidc');
                }
                if (authModesToCheck.size > 0) {
                    const isUserPoolTheDefault = this.configuredAuthProviders.default === 'userPools';
                    expressions.push(this.resources.getAuthModeDeterminationExpression(authModesToCheck, isUserPoolTheDefault));
                }
                const authCheckExpressions = [
                    staticGroupAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    dynamicGroupAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    ownerAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    throwIfUnauthorizedExpression,
                ];
                expressions.push(this.resources.getAuthModeCheckWrappedExpression(authModesToCheck, graphql_mapping_template_1.compoundExpression(authCheckExpressions)));
                const templateParts = [
                    graphql_mapping_template_1.print(graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`$ctx.args.input.containsKey("${field.name.value}")`), graphql_mapping_template_1.compoundExpression(expressions))),
                    createResolverResource.Properties.RequestMappingTemplate,
                ];
                createResolverResource.Properties.RequestMappingTemplate = templateParts.join('\n\n');
                ctx.setResource(resolverResourceId, createResolverResource);
            }
            if (modelConfiguration.shouldHave('onCreate') && modelConfiguration.getName('level') === 'on') {
                const getTemplateParts = [createResolverResource.Properties.ResponseMappingTemplate];
                if (!this.isOperationExpressionSet(mutationTypeName, createResolverResource.Properties.ResponseMappingTemplate)) {
                    getTemplateParts.unshift(this.resources.setOperationExpression(mutationTypeName));
                }
                createResolverResource.Properties.ResponseMappingTemplate = getTemplateParts.join('\n\n');
                ctx.setResource(resolverResourceId, createResolverResource);
            }
        }
    }
    splitRules(rules) {
        const queryRules = {
            get: [],
            list: [],
        };
        const operationRules = {
            create: [],
            update: [],
            delete: [],
            read: [],
        };
        const matchQuery = (op) => (rule) => {
            if (rule.queries) {
                const matchesOp = rule.queries.find(o => o === op);
                return Boolean(matchesOp);
            }
            else if (rule.queries === null) {
                return false;
            }
            return true;
        };
        const matchMutation = (op) => (rule) => {
            if (rule.mutations) {
                const matchesOp = rule.mutations.find(o => o === op);
                return Boolean(matchesOp);
            }
            else if (rule.mutations === null) {
                return false;
            }
            return true;
        };
        const matchOperation = (op) => (rule) => {
            if (rule.operations) {
                const matchesOp = rule.operations.find(o => o === op);
                return Boolean(matchesOp);
            }
            else if (rule.operations === null) {
                return false;
            }
            return true;
        };
        for (const rule of rules) {
            if (isTruthyOrNull(rule.operations)) {
                if (matchOperation('read')(rule)) {
                    queryRules.get.push(rule);
                    queryRules.list.push(rule);
                    operationRules.read.push(rule);
                }
                if (matchOperation('create')(rule)) {
                    operationRules.create.push(rule);
                }
                if (matchOperation('update')(rule)) {
                    operationRules.update.push(rule);
                }
                if (matchOperation('delete')(rule)) {
                    operationRules.delete.push(rule);
                }
            }
            else {
                if (isUndefined(rule.queries)) {
                    queryRules.get.push(rule);
                    queryRules.list.push(rule);
                    operationRules.read.push(rule);
                }
                else {
                    if (matchQuery('get')(rule)) {
                        queryRules.get.push(rule);
                    }
                    if (matchQuery('list')(rule)) {
                        queryRules.list.push(rule);
                    }
                }
                if (isUndefined(rule.mutations)) {
                    operationRules.create.push(rule);
                    operationRules.update.push(rule);
                    operationRules.delete.push(rule);
                }
                else {
                    if (matchMutation('create')(rule)) {
                        operationRules.create.push(rule);
                    }
                    if (matchMutation('update')(rule)) {
                        operationRules.update.push(rule);
                    }
                    if (matchMutation('delete')(rule)) {
                        operationRules.delete.push(rule);
                    }
                }
            }
        }
        return {
            operationRules,
            queryRules,
        };
    }
    validateRules(rules) {
        for (const rule of rules) {
            this.validateRuleAuthStrategy(rule);
            const { queries, mutations, operations } = rule;
            if (mutations && operations) {
                console.warn(`It is not recommended to use 'mutations' and 'operations'. The 'operations' argument will be used.`);
            }
            if (queries && operations) {
                console.warn(`It is not recommended to use 'queries' and 'operations'. The 'operations' argument will be used.`);
            }
            this.commonRuleValidation(rule);
        }
    }
    validateFieldRules(rules, isParentTypeBuiltinType, parentHasModelDirective) {
        for (const rule of rules) {
            this.validateRuleAuthStrategy(rule);
            const { queries, mutations } = rule;
            if (queries || mutations) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`@auth directives used on field definitions may not specify the 'queries' or 'mutations' arguments. \
All @auth directives used on field definitions are performed when the field is resolved and can be thought of as 'read' operations.`);
            }
            if (isParentTypeBuiltinType && rule.operations && rule.operations.length > 0) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`@auth rules on fields within Query, Mutation, Subscription cannot specify 'operations' argument as these rules \
are already on an operation already.`);
            }
            if (!parentHasModelDirective && rule.operations && rule.operations.length > 0) {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`@auth rules on fields within types that does not have @model directive cannot specify 'operations' argument as there are \
operations will be generated by the CLI.`);
            }
            this.commonRuleValidation(rule);
        }
    }
    commonRuleValidation(rule) {
        const { identityField, identityClaim, allow, groups, groupsField, groupClaim } = rule;
        if (allow === 'groups' && (identityClaim || identityField)) {
            throw new graphql_transformer_core_1.InvalidDirectiveError(`
            @auth identityField/Claim can only be used for 'allow: owner'`);
        }
        if (allow === 'owner' && groupClaim) {
            throw new graphql_transformer_core_1.InvalidDirectiveError(`
            @auth groupClaim can only be used 'allow: groups'`);
        }
        if (groupsField && groups) {
            throw new graphql_transformer_core_1.InvalidDirectiveError('This rule has groupsField and groups, please use one or the other');
        }
        if (identityField && identityClaim) {
            throw new graphql_transformer_core_1.InvalidDirectiveError('Please use consider IdentifyClaim over IdentityField as it is deprecated.');
        }
    }
    protectGetQuery(ctx, resolverResourceId, rules, parent, modelConfiguration) {
        const resolver = ctx.getResource(resolverResourceId);
        if (!rules || rules.length === 0 || !resolver) {
            return;
        }
        else {
            let operationName = undefined;
            if (modelConfiguration.shouldHave('get')) {
                operationName = modelConfiguration.getName('get');
                const includeDefault = parent !== null ? this.isTypeHasRulesForOperation(parent, 'get') : false;
                const operationDirectives = this.getDirectivesForRules(rules, includeDefault);
                if (operationDirectives.length > 0) {
                    this.addDirectivesToOperation(ctx, ctx.getQueryTypeName(), operationName, operationDirectives);
                }
            }
            if (operationName) {
                this.addFieldToResourceReferences(ctx.getQueryTypeName(), operationName, rules);
            }
            const authExpression = this.authorizationExpressionOnSingleObject(rules);
            if (authExpression) {
                const templateParts = [graphql_mapping_template_1.print(authExpression), resolver.Properties.ResponseMappingTemplate];
                resolver.Properties.ResponseMappingTemplate = templateParts.join('\n\n');
                ctx.setResource(resolverResourceId, resolver);
            }
        }
    }
    authorizationExpressionOnSingleObject(rules, objectPath = 'ctx.result') {
        const staticGroupAuthorizationRules = this.getStaticGroupRules(rules);
        const dynamicGroupAuthorizationRules = this.getDynamicGroupRules(rules);
        const ownerAuthorizationRules = this.getOwnerRules(rules);
        const providerAuthorization = this.hasProviderAuthRules(rules);
        if ((staticGroupAuthorizationRules.length > 0 || dynamicGroupAuthorizationRules.length > 0 || ownerAuthorizationRules.length > 0) &&
            providerAuthorization === false) {
            const staticGroupAuthorizationExpression = this.resources.staticGroupAuthorizationExpression(staticGroupAuthorizationRules);
            const dynamicGroupAuthorizationExpression = this.resources.dynamicGroupAuthorizationExpressionForReadOperations(dynamicGroupAuthorizationRules, objectPath);
            const ownerAuthorizationExpression = this.resources.ownerAuthorizationExpressionForReadOperations(ownerAuthorizationRules, objectPath);
            const throwIfUnauthorizedExpression = this.resources.throwIfUnauthorized();
            const authModesToCheck = new Set();
            const expressions = new Array();
            expressions.push(this.resources.returnIfEmpty(objectPath));
            if (ownerAuthorizationRules.find(r => r.provider === 'userPools') ||
                staticGroupAuthorizationRules.find(r => r.provider === 'userPools') ||
                dynamicGroupAuthorizationRules.find(r => r.provider === 'userPools')) {
                authModesToCheck.add('userPools');
            }
            if (ownerAuthorizationRules.find(r => r.provider === 'oidc') ||
                staticGroupAuthorizationRules.find(r => r.provider === 'oidc') ||
                dynamicGroupAuthorizationRules.find(r => r.provider === 'oidc')) {
                authModesToCheck.add('oidc');
            }
            if (authModesToCheck.size > 0) {
                const isUserPoolTheDefault = this.configuredAuthProviders.default === 'userPools';
                expressions.push(this.resources.getAuthModeDeterminationExpression(authModesToCheck, isUserPoolTheDefault));
            }
            const templateExpressions = [
                staticGroupAuthorizationExpression,
                graphql_mapping_template_1.newline(),
                dynamicGroupAuthorizationExpression,
                graphql_mapping_template_1.newline(),
                ownerAuthorizationExpression,
                graphql_mapping_template_1.newline(),
                throwIfUnauthorizedExpression,
            ];
            expressions.push(this.resources.getAuthModeCheckWrappedExpression(authModesToCheck, graphql_mapping_template_1.compoundExpression(templateExpressions)));
            return graphql_mapping_template_1.compoundExpression(expressions);
        }
    }
    protectListQuery(ctx, resolverResourceId, rules, parent, modelConfiguration, explicitOperationName = undefined) {
        const resolver = ctx.getResource(resolverResourceId);
        if (!rules || rules.length === 0 || !resolver) {
            return;
        }
        if (modelConfiguration.shouldHave('list')) {
            const operationName = explicitOperationName ? explicitOperationName : modelConfiguration.getName('list');
            const includeDefault = parent !== null ? this.isTypeHasRulesForOperation(parent, 'list') : false;
            const operationDirectives = this.getDirectivesForRules(rules, includeDefault);
            if (operationDirectives.length > 0) {
                this.addDirectivesToOperation(ctx, ctx.getQueryTypeName(), operationName, operationDirectives);
            }
            this.addFieldToResourceReferences(ctx.getQueryTypeName(), operationName, rules);
        }
        const authExpression = this.authorizationExpressionForListResult(rules);
        if (authExpression) {
            const templateParts = [graphql_mapping_template_1.print(authExpression), resolver.Properties.ResponseMappingTemplate];
            resolver.Properties.ResponseMappingTemplate = templateParts.join('\n\n');
            ctx.setResource(resolverResourceId, resolver);
        }
    }
    authorizationExpressionForListResult(rules, itemList = 'ctx.result.items') {
        const staticGroupAuthorizationRules = this.getStaticGroupRules(rules);
        const dynamicGroupAuthorizationRules = this.getDynamicGroupRules(rules);
        const ownerAuthorizationRules = this.getOwnerRules(rules);
        const providerAuthorization = this.hasProviderAuthRules(rules);
        if ((staticGroupAuthorizationRules.length > 0 || dynamicGroupAuthorizationRules.length > 0 || ownerAuthorizationRules.length > 0) &&
            providerAuthorization === false) {
            const staticGroupAuthorizationExpression = this.resources.staticGroupAuthorizationExpression(staticGroupAuthorizationRules);
            const dynamicGroupAuthorizationExpression = this.resources.dynamicGroupAuthorizationExpressionForReadOperations(dynamicGroupAuthorizationRules, 'item', graphql_transformer_common_1.ResourceConstants.SNIPPETS.IsLocalDynamicGroupAuthorizedVariable, graphql_mapping_template_1.raw(`false`));
            const ownerAuthorizationExpression = this.resources.ownerAuthorizationExpressionForReadOperations(ownerAuthorizationRules, 'item', graphql_transformer_common_1.ResourceConstants.SNIPPETS.IsLocalOwnerAuthorizedVariable, graphql_mapping_template_1.raw(`false`));
            const appendIfLocallyAuthorized = this.resources.appendItemIfLocallyAuthorized();
            const ifNotStaticallyAuthedFilterObjects = graphql_mapping_template_1.iff(graphql_mapping_template_1.not(graphql_mapping_template_1.ref(graphql_transformer_common_1.ResourceConstants.SNIPPETS.IsStaticGroupAuthorizedVariable)), graphql_mapping_template_1.compoundExpression([
                graphql_mapping_template_1.set(graphql_mapping_template_1.ref('items'), graphql_mapping_template_1.list([])),
                graphql_mapping_template_1.forEach(graphql_mapping_template_1.ref('item'), graphql_mapping_template_1.ref(itemList), [
                    dynamicGroupAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    ownerAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    appendIfLocallyAuthorized,
                ]),
                graphql_mapping_template_1.set(graphql_mapping_template_1.ref(itemList), graphql_mapping_template_1.ref('items')),
            ]));
            const authModesToCheck = new Set();
            const expressions = new Array();
            if (ownerAuthorizationRules.find(r => r.provider === 'userPools') ||
                staticGroupAuthorizationRules.find(r => r.provider === 'userPools') ||
                dynamicGroupAuthorizationRules.find(r => r.provider === 'userPools')) {
                authModesToCheck.add('userPools');
            }
            if (ownerAuthorizationRules.find(r => r.provider === 'oidc') ||
                staticGroupAuthorizationRules.find(r => r.provider === 'oidc') ||
                dynamicGroupAuthorizationRules.find(r => r.provider === 'oidc')) {
                authModesToCheck.add('oidc');
            }
            if (authModesToCheck.size > 0) {
                const isUserPoolTheDefault = this.configuredAuthProviders.default === 'userPools';
                expressions.push(this.resources.getAuthModeDeterminationExpression(authModesToCheck, isUserPoolTheDefault));
            }
            const templateExpressions = [
                staticGroupAuthorizationExpression,
                graphql_mapping_template_1.newline(),
                graphql_mapping_template_1.comment('[Start] If not static group authorized, filter items'),
                ifNotStaticallyAuthedFilterObjects,
                graphql_mapping_template_1.comment('[End] If not static group authorized, filter items'),
            ];
            expressions.push(this.resources.getAuthModeCheckWrappedExpression(authModesToCheck, graphql_mapping_template_1.compoundExpression(templateExpressions)));
            return graphql_mapping_template_1.compoundExpression(expressions);
        }
    }
    protectCreateMutation(ctx, resolverResourceId, rules, parent, modelConfiguration) {
        const resolver = ctx.getResource(resolverResourceId);
        if (!rules || rules.length === 0 || !resolver) {
            return;
        }
        else {
            const mutationTypeName = ctx.getMutationTypeName();
            if (modelConfiguration.shouldHave('create')) {
                const operationName = modelConfiguration.getName('create');
                const includeDefault = this.isTypeHasRulesForOperation(parent, 'create');
                const operationDirectives = this.getDirectivesForRules(rules, includeDefault);
                if (operationDirectives.length > 0) {
                    this.addDirectivesToOperation(ctx, mutationTypeName, operationName, operationDirectives);
                }
                this.addFieldToResourceReferences(mutationTypeName, operationName, rules);
            }
            const staticGroupAuthorizationRules = this.getStaticGroupRules(rules);
            const dynamicGroupAuthorizationRules = this.getDynamicGroupRules(rules);
            const ownerAuthorizationRules = this.getOwnerRules(rules);
            const providerAuthorization = this.hasProviderAuthRules(rules);
            if ((staticGroupAuthorizationRules.length > 0 || dynamicGroupAuthorizationRules.length > 0 || ownerAuthorizationRules.length > 0) &&
                providerAuthorization === false) {
                const staticGroupAuthorizationExpression = this.resources.staticGroupAuthorizationExpression(staticGroupAuthorizationRules);
                const dynamicGroupAuthorizationExpression = this.resources.dynamicGroupAuthorizationExpressionForCreateOperations(dynamicGroupAuthorizationRules);
                const fieldIsList = (fieldName) => {
                    const field = parent.fields.find(field => field.name.value === fieldName);
                    if (field) {
                        return graphql_transformer_common_1.isListType(field.type);
                    }
                    return false;
                };
                const ownerAuthorizationExpression = this.resources.ownerAuthorizationExpressionForCreateOperations(ownerAuthorizationRules, fieldIsList);
                const throwIfUnauthorizedExpression = this.resources.throwIfUnauthorized();
                const authModesToCheck = new Set();
                const expressions = new Array();
                if (ownerAuthorizationRules.find(r => r.provider === 'userPools') ||
                    staticGroupAuthorizationRules.find(r => r.provider === 'userPools') ||
                    dynamicGroupAuthorizationRules.find(r => r.provider === 'userPools')) {
                    authModesToCheck.add('userPools');
                }
                if (ownerAuthorizationRules.find(r => r.provider === 'oidc') ||
                    staticGroupAuthorizationRules.find(r => r.provider === 'oidc') ||
                    dynamicGroupAuthorizationRules.find(r => r.provider === 'oidc')) {
                    authModesToCheck.add('oidc');
                }
                if (authModesToCheck.size > 0) {
                    const isUserPoolTheDefault = this.configuredAuthProviders.default === 'userPools';
                    expressions.push(this.resources.getAuthModeDeterminationExpression(authModesToCheck, isUserPoolTheDefault));
                }
                const authCheckExpressions = [
                    staticGroupAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    dynamicGroupAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    ownerAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    throwIfUnauthorizedExpression,
                ];
                expressions.push(this.resources.getAuthModeCheckWrappedExpression(authModesToCheck, graphql_mapping_template_1.compoundExpression(authCheckExpressions)));
                const templateParts = [graphql_mapping_template_1.print(graphql_mapping_template_1.compoundExpression(expressions)), resolver.Properties.RequestMappingTemplate];
                resolver.Properties.RequestMappingTemplate = templateParts.join('\n\n');
                ctx.setResource(resolverResourceId, resolver);
            }
        }
    }
    protectUpdateOrDeleteMutation(ctx, resolverResourceId, rules, parent, modelConfiguration, isUpdate, field, ifCondition, subscriptionOperation) {
        const resolver = ctx.getResource(resolverResourceId);
        if (!rules || rules.length === 0 || !resolver) {
            return;
        }
        else {
            const mutationTypeName = ctx.getMutationTypeName();
            if (modelConfiguration.shouldHave(isUpdate ? 'update' : 'delete')) {
                const operationName = modelConfiguration.getName(isUpdate ? 'update' : 'delete');
                const includeDefault = Boolean(!field && this.isTypeHasRulesForOperation(parent, isUpdate ? 'update' : 'delete'));
                const operationDirectives = this.getDirectivesForRules(rules, includeDefault);
                if (operationDirectives.length > 0) {
                    this.addDirectivesToOperation(ctx, mutationTypeName, operationName, operationDirectives);
                }
                this.addFieldToResourceReferences(mutationTypeName, operationName, rules);
            }
            const staticGroupAuthorizationRules = this.getStaticGroupRules(rules);
            const dynamicGroupAuthorizationRules = this.getDynamicGroupRules(rules);
            const ownerAuthorizationRules = this.getOwnerRules(rules);
            const providerAuthorization = this.hasProviderAuthRules(rules);
            if ((staticGroupAuthorizationRules.length > 0 || dynamicGroupAuthorizationRules.length > 0 || ownerAuthorizationRules.length > 0) &&
                providerAuthorization === false) {
                const staticGroupAuthorizationExpression = this.resources.staticGroupAuthorizationExpression(staticGroupAuthorizationRules, field);
                const fieldIsList = (fieldName) => {
                    const field = parent.fields.find(field => field.name.value === fieldName);
                    if (field) {
                        return graphql_transformer_common_1.isListType(field.type);
                    }
                    return false;
                };
                const dynamicGroupAuthorizationExpression = this.resources.dynamicGroupAuthorizationExpressionForUpdateOrDeleteOperations(dynamicGroupAuthorizationRules, fieldIsList, field ? field.name.value : undefined);
                const ownerAuthorizationExpression = this.resources.ownerAuthorizationExpressionForUpdateOrDeleteOperations(ownerAuthorizationRules, fieldIsList, field ? field.name.value : undefined);
                const collectAuthCondition = this.resources.collectAuthCondition();
                const staticGroupAuthorizedVariable = this.resources.getStaticAuthorizationVariable(field);
                const ifNotStaticallyAuthedCreateAuthCondition = graphql_mapping_template_1.iff(graphql_mapping_template_1.raw(`! $${staticGroupAuthorizedVariable}`), graphql_mapping_template_1.compoundExpression([
                    dynamicGroupAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    ownerAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    collectAuthCondition,
                ]));
                const throwIfNotStaticGroupAuthorizedOrAuthConditionIsEmpty = this.resources.throwIfNotStaticGroupAuthorizedOrAuthConditionIsEmpty(field);
                const authModesToCheck = new Set();
                const expressions = new Array();
                if (ownerAuthorizationRules.find(r => r.provider === 'userPools') ||
                    staticGroupAuthorizationRules.find(r => r.provider === 'userPools') ||
                    dynamicGroupAuthorizationRules.find(r => r.provider === 'userPools')) {
                    authModesToCheck.add('userPools');
                }
                if (ownerAuthorizationRules.find(r => r.provider === 'oidc') ||
                    staticGroupAuthorizationRules.find(r => r.provider === 'oidc') ||
                    dynamicGroupAuthorizationRules.find(r => r.provider === 'oidc')) {
                    authModesToCheck.add('oidc');
                }
                if (authModesToCheck.size > 0) {
                    const isUserPoolTheDefault = this.configuredAuthProviders.default === 'userPools';
                    expressions.push(this.resources.getAuthModeDeterminationExpression(authModesToCheck, isUserPoolTheDefault));
                }
                const authorizationLogic = graphql_mapping_template_1.compoundExpression([
                    staticGroupAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    ifNotStaticallyAuthedCreateAuthCondition,
                    graphql_mapping_template_1.newline(),
                    throwIfNotStaticGroupAuthorizedOrAuthConditionIsEmpty,
                ]);
                expressions.push(this.resources.getAuthModeCheckWrappedExpression(authModesToCheck, authorizationLogic));
                const templateParts = [
                    graphql_mapping_template_1.print(field && ifCondition ? graphql_mapping_template_1.iff(ifCondition, graphql_mapping_template_1.compoundExpression(expressions)) : graphql_mapping_template_1.compoundExpression(expressions)),
                    resolver.Properties.RequestMappingTemplate,
                ];
                resolver.Properties.RequestMappingTemplate = templateParts.join('\n\n');
                ctx.setResource(resolverResourceId, resolver);
            }
            if (field &&
                subscriptionOperation &&
                modelConfiguration.shouldHave(subscriptionOperation) &&
                modelConfiguration.getName('level') === 'on') {
                let mutationResolver = resolver;
                let mutationResolverResourceID = resolverResourceId;
                if (subscriptionOperation === 'onDelete') {
                    mutationResolverResourceID = graphql_transformer_common_1.ResolverResourceIDs.DynamoDBDeleteResolverResourceID(parent.name.value);
                    mutationResolver = ctx.getResource(mutationResolverResourceID);
                }
                const getTemplateParts = [mutationResolver.Properties.ResponseMappingTemplate];
                if (!this.isOperationExpressionSet(mutationTypeName, mutationResolver.Properties.ResponseMappingTemplate)) {
                    getTemplateParts.unshift(this.resources.setOperationExpression(mutationTypeName));
                }
                mutationResolver.Properties.ResponseMappingTemplate = getTemplateParts.join('\n\n');
                ctx.setResource(mutationResolverResourceID, mutationResolver);
            }
        }
    }
    protectUpdateMutation(ctx, resolverResourceId, rules, parent, modelConfiguration, field, subscriptionOperation) {
        return this.protectUpdateOrDeleteMutation(ctx, resolverResourceId, rules, parent, modelConfiguration, true, field, field ? graphql_mapping_template_1.raw(`$ctx.args.input.containsKey("${field.name.value}")`) : undefined, subscriptionOperation);
    }
    protectDeleteMutation(ctx, resolverResourceId, rules, parent, modelConfiguration, field, subscriptionOperation) {
        return this.protectUpdateOrDeleteMutation(ctx, resolverResourceId, rules, parent, modelConfiguration, false, field, field
            ? graphql_mapping_template_1.raw(`$ctx.args.input.containsKey("${field.name.value}") && $util.isNull($ctx.args.input.get("${field.name.value}"))`)
            : undefined, subscriptionOperation);
    }
    protectConnections(ctx, def, rules, modelConfiguration) {
        const thisModelName = def.name.value;
        for (const inputDef of ctx.inputDocument.definitions) {
            if (inputDef.kind === graphql_1.Kind.OBJECT_TYPE_DEFINITION) {
                for (const field of inputDef.fields) {
                    const returnTypeName = graphql_transformer_common_1.getBaseType(field.type);
                    if (fieldHasDirective(field, 'connection') && returnTypeName === thisModelName) {
                        const resolverResourceId = graphql_transformer_common_1.ResolverResourceIDs.ResolverResourceID(inputDef.name.value, field.name.value);
                        const directives = this.getDirectivesForRules(rules, false);
                        if (directives.length > 0) {
                            this.addDirectivesToField(ctx, inputDef.name.value, field.name.value, directives);
                        }
                        if (graphql_transformer_common_1.isListType(field.type)) {
                            this.protectListQuery(ctx, resolverResourceId, rules, null, modelConfiguration);
                        }
                        else {
                            this.protectGetQuery(ctx, resolverResourceId, rules, null, modelConfiguration);
                        }
                    }
                }
            }
        }
    }
    protectQueries(ctx, def, rules, modelConfiguration) {
        const secondaryKeyDirectivesWithQueries = (def.directives || []).filter(d => {
            const isKey = d.name.value === 'key';
            const args = graphql_transformer_core_1.getDirectiveArguments(d);
            const isSecondaryKey = Boolean(args.name);
            const hasQueryField = Boolean(args.queryField);
            return isKey && isSecondaryKey && hasQueryField;
        });
        for (const keyWithQuery of secondaryKeyDirectivesWithQueries) {
            const args = graphql_transformer_core_1.getDirectiveArguments(keyWithQuery);
            const resolverResourceId = graphql_transformer_common_1.ResolverResourceIDs.ResolverResourceID(ctx.getQueryTypeName(), args.queryField);
            this.protectListQuery(ctx, resolverResourceId, rules, null, modelConfiguration, args.queryField);
        }
    }
    protectSearchQuery(ctx, def, resolverResourceId, rules) {
        const resolver = ctx.getResource(resolverResourceId);
        if (!rules || rules.length === 0 || !resolver) {
            return;
        }
        else {
            const operationName = resolver.Properties.FieldName;
            const includeDefault = def !== null ? this.isTypeHasRulesForOperation(def, 'list') : false;
            const operationDirectives = this.getDirectivesForRules(rules, includeDefault);
            if (operationDirectives.length > 0) {
                this.addDirectivesToOperation(ctx, ctx.getQueryTypeName(), operationName, operationDirectives);
            }
            this.addFieldToResourceReferences(ctx.getQueryTypeName(), operationName, rules);
            const authExpression = this.authorizationExpressionForListResult(rules, 'es_items');
            if (authExpression) {
                const templateParts = [
                    graphql_mapping_template_1.print(this.resources.makeESItemsExpression(ctx.isProjectUsingDataStore())),
                    graphql_mapping_template_1.print(authExpression),
                    graphql_mapping_template_1.print(this.resources.makeESToGQLExpression()),
                ];
                resolver.Properties.ResponseMappingTemplate = templateParts.join('\n\n');
                ctx.setResource(resolverResourceId, resolver);
            }
        }
    }
    protectSyncQuery(ctx, def, resolverResourceID, rules) {
        const resolver = ctx.getResource(resolverResourceID);
        if (!rules || rules.length === 0 || !resolver) {
            return;
        }
        const operationName = resolver.Properties.FieldName;
        const includeDefault = def !== null ? this.isTypeHasRulesForOperation(def, 'list') : false;
        const operationDirectives = this.getDirectivesForRules(rules, includeDefault);
        if (operationDirectives.length > 0) {
            this.addDirectivesToOperation(ctx, ctx.getQueryTypeName(), operationName, operationDirectives);
        }
        this.addFieldToResourceReferences(ctx.getQueryTypeName(), operationName, rules);
        const authExpression = this.authorizationExpressionForListResult(rules);
        if (authExpression) {
            const templateParts = [graphql_mapping_template_1.print(authExpression), resolver.Properties.ResponseMappingTemplate];
            resolver.Properties.ResponseMappingTemplate = templateParts.join('\n\n');
            ctx.setResource(resolverResourceID, resolver);
        }
    }
    protectOnCreateSubscription(ctx, rules, parent, modelConfiguration) {
        const names = modelConfiguration.getNames('onCreate');
        const level = modelConfiguration.getName('level');
        if (names) {
            names.forEach(name => {
                this.addSubscriptionResolvers(ctx, rules, parent, level, name);
            });
        }
    }
    protectOnUpdateSubscription(ctx, rules, parent, modelConfiguration) {
        const names = modelConfiguration.getNames('onUpdate');
        const level = modelConfiguration.getName('level');
        if (names) {
            names.forEach(name => {
                this.addSubscriptionResolvers(ctx, rules, parent, level, name);
            });
        }
    }
    protectOnDeleteSubscription(ctx, rules, parent, modelConfiguration) {
        const names = modelConfiguration.getNames('onDelete');
        const level = modelConfiguration.getName('level');
        if (names) {
            names.forEach(name => {
                this.addSubscriptionResolvers(ctx, rules, parent, level, name);
            });
        }
    }
    addSubscriptionResolvers(ctx, rules, parent, level, fieldName) {
        const resolverResourceId = graphql_transformer_common_1.ResolverResourceIDs.ResolverResourceID('Subscription', fieldName);
        const resolver = this.resources.generateSubscriptionResolver(fieldName);
        const noneDS = ctx.getResource(graphql_transformer_common_1.ResourceConstants.RESOURCES.NoneDataSource);
        if (!rules || rules.length === 0) {
            return;
        }
        else if (level === 'public') {
            ctx.setResource(resolverResourceId, resolver);
        }
        else {
            const includeDefault = parent !== null ? this.isTypeHasRulesForOperation(parent, 'get') : false;
            const directives = this.getDirectivesForRules(rules, includeDefault);
            if (directives.length > 0) {
                this.addDirectivesToField(ctx, ctx.getSubscriptionTypeName(), fieldName, directives);
            }
            this.addFieldToResourceReferences(ctx.getSubscriptionTypeName(), fieldName, rules);
            const staticGroupAuthorizationRules = this.getStaticGroupRules(rules);
            const ownerAuthorizationRules = this.getOwnerRules(rules);
            const providerAuthorization = this.hasProviderAuthRules(rules);
            if ((staticGroupAuthorizationRules.length > 0 || ownerAuthorizationRules.length > 0) && providerAuthorization === false) {
                const staticGroupAuthorizationExpression = this.resources.staticGroupAuthorizationExpression(staticGroupAuthorizationRules);
                const ownerAuthorizationExpression = this.resources.ownerAuthorizationExpressionForSubscriptions(ownerAuthorizationRules);
                const throwIfUnauthorizedExpression = this.resources.throwIfSubscriptionUnauthorized();
                const authModesToCheck = new Set();
                const expressions = new Array();
                if (ownerAuthorizationRules.find(r => r.provider === 'userPools') ||
                    staticGroupAuthorizationRules.find(r => r.provider === 'userPools')) {
                    authModesToCheck.add('userPools');
                }
                if (ownerAuthorizationRules.find(r => r.provider === 'oidc') || staticGroupAuthorizationRules.find(r => r.provider === 'oidc')) {
                    authModesToCheck.add('oidc');
                }
                if (authModesToCheck.size > 0) {
                    const isUserPoolTheDefault = this.configuredAuthProviders.default === 'userPools';
                    expressions.push(this.resources.getAuthModeDeterminationExpression(authModesToCheck, isUserPoolTheDefault));
                }
                const authCheckExpressions = [
                    staticGroupAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    ownerAuthorizationExpression,
                    graphql_mapping_template_1.newline(),
                    throwIfUnauthorizedExpression,
                ];
                expressions.push(this.resources.getAuthModeCheckWrappedExpression(authModesToCheck, graphql_mapping_template_1.compoundExpression(authCheckExpressions)));
                const templateParts = [graphql_mapping_template_1.print(graphql_mapping_template_1.compoundExpression(expressions)), resolver.Properties.ResponseMappingTemplate];
                resolver.Properties.ResponseMappingTemplate = templateParts.join('\n\n');
                ctx.setResource(resolverResourceId, resolver);
                const ownerRules = rules.filter(rule => rule.allow === constants_1.OWNER_AUTH_STRATEGY);
                const needsDefaultOwnerField = ownerRules.find(rule => !rule.ownerField);
                const needsOwnerArgument = rules.find(rule => (rule.allow === constants_1.GROUPS_AUTH_STRATEGY && !rule.groupsField) || rule.allow === 'private' || rule.allow === 'public');
                if (ownerRules) {
                    if (needsDefaultOwnerField) {
                        this.addOwner(ctx, parent.name.value);
                    }
                    const makeNonNull = needsOwnerArgument ? false : true;
                    this.addSubscriptionOwnerArgument(ctx, resolver, ownerRules, makeNonNull);
                }
            }
        }
        if (!noneDS) {
            ctx.setResource(graphql_transformer_common_1.ResourceConstants.RESOURCES.NoneDataSource, this.resources.noneDataSource());
        }
        ctx.mapResourceToStack(parent.name.value, resolverResourceId);
    }
    addSubscriptionOwnerArgument(ctx, resolver, ownerRules, makeNonNull = false) {
        let subscription = ctx.getSubscription();
        let createField = subscription.fields.find(field => field.name.value === resolver.Properties.FieldName);
        const nameNode = makeNonNull ? graphql_transformer_common_1.makeNonNullType(graphql_transformer_common_1.makeNamedType('String')) : graphql_transformer_common_1.makeNamedType('String');
        const ownerArgumentList = ownerRules.map(rule => {
            return graphql_transformer_common_1.makeInputValueDefinition(rule.ownerField || constants_1.DEFAULT_OWNER_FIELD, nameNode);
        });
        createField = {
            ...createField,
            arguments: ownerArgumentList,
        };
        subscription = {
            ...subscription,
            fields: subscription.fields.map(field => (field.name.value === resolver.Properties.FieldName ? createField : field)),
        };
        ctx.putType(subscription);
    }
    addOwnerFieldsToObject(ctx, typeName, rules) {
        if (ctx.featureFlags.getBoolean('addMissingOwnerFields', true)) {
            const object = ctx.getObject(typeName);
            const ownerRules = rules.filter(rule => rule.allow === 'owner');
            const existingFields = Object.keys(graphql_transformer_core_1.getFieldArguments(object));
            const ownerFields = Array.from(new Set(ownerRules.map(rule => rule.ownerField || constants_1.DEFAULT_OWNER_FIELD)).values());
            const fieldsToAdd = ownerFields.filter(field => !existingFields.includes(field));
            const modelType = ctx.getType(object.name.value);
            if (fieldsToAdd.length) {
                for (let ownerField of fieldsToAdd) {
                    modelType.fields.push(graphql_transformer_common_1.makeField(ownerField, [], graphql_transformer_common_1.makeNamedType('String')));
                }
                ctx.putType(modelType);
            }
        }
    }
    addOwner(ctx, parent) {
        if (ctx.featureFlags.getBoolean('addMissingOwnerFields', true)) {
            return;
        }
        const modelType = ctx.getType(parent);
        const fields = graphql_transformer_core_1.getFieldArguments(modelType);
        if (!('owner' in fields)) {
            modelType.fields.push(graphql_transformer_common_1.makeField(constants_1.DEFAULT_OWNER_FIELD, [], graphql_transformer_common_1.makeNamedType('String')));
        }
        ctx.putType(modelType);
    }
    getOwnerRules(rules) {
        return rules.filter(rule => rule.allow === 'owner');
    }
    getStaticGroupRules(rules) {
        return rules.filter(rule => rule.allow === 'groups' && Boolean(rule.groups));
    }
    getDynamicGroupRules(rules) {
        return rules.filter(rule => rule.allow === 'groups' && !Boolean(rule.groups));
    }
    hasProviderAuthRules(rules) {
        return rules.filter(rule => rule.provider === 'userPools' && (rule.allow === 'public' || rule.allow === 'private')).length > 0;
    }
    extendTypeWithDirectives(ctx, typeName, directives) {
        let objectTypeExtension = graphql_transformer_common_1.blankObjectExtension(typeName);
        objectTypeExtension = graphql_transformer_common_1.extensionWithDirectives(objectTypeExtension, directives);
        ctx.addObjectExtension(objectTypeExtension);
    }
    addDirectivesToOperation(ctx, typeName, operationName, directives) {
        this.addDirectivesToField(ctx, typeName, operationName, directives);
        const type = ctx.getType(typeName);
        if (type) {
            const field = type.fields.find(f => f.name.value === operationName);
            if (field) {
                const returnFieldType = field.type;
                if (returnFieldType.name) {
                    const returnTypeName = returnFieldType.name.value;
                    this.extendTypeWithDirectives(ctx, returnTypeName, directives);
                }
            }
        }
    }
    addDirectivesToField(ctx, typeName, fieldName, directives) {
        const type = ctx.getType(typeName);
        if (type) {
            const field = type.fields.find(f => f.name.value === fieldName);
            if (field) {
                const newFields = [...type.fields.filter(f => f.name.value !== field.name.value), graphql_transformer_common_1.extendFieldWithDirectives(field, directives)];
                const newMutation = {
                    ...type,
                    fields: newFields,
                };
                ctx.putType(newMutation);
            }
        }
    }
    getDirectivesForRules(rules, addDefaultIfNeeded = true) {
        if (!rules || rules.length === 0) {
            return [];
        }
        if (this.configuredAuthProviders.hasIAM && this.config.addAwsIamAuthInOutputSchema) {
            rules.push({ allow: 'private', provider: 'iam' });
        }
        const directives = new Array();
        const addDirectiveIfNeeded = (provider, directiveName) => {
            if ((this.configuredAuthProviders.default !== provider && Boolean(rules.find(r => r.provider === provider))) ||
                (this.configuredAuthProviders.default === provider &&
                    Boolean(rules.find(r => r.provider !== provider && addDefaultIfNeeded === true)))) {
                directives.push(graphql_transformer_common_1.makeDirective(directiveName, []));
            }
        };
        const authProviderDirectiveMap = new Map([
            ['apiKey', 'aws_api_key'],
            ['iam', 'aws_iam'],
            ['oidc', 'aws_oidc'],
            ['userPools', 'aws_cognito_user_pools'],
        ]);
        for (const entry of authProviderDirectiveMap.entries()) {
            addDirectiveIfNeeded(entry[0], entry[1]);
        }
        if (Boolean(rules.find(r => r.provider === this.configuredAuthProviders.default)) &&
            Boolean(rules.find(r => r.provider !== this.configuredAuthProviders.default) &&
                !Boolean(directives.find(d => d.name.value === authProviderDirectiveMap.get(this.configuredAuthProviders.default))))) {
            directives.push(graphql_transformer_common_1.makeDirective(authProviderDirectiveMap.get(this.configuredAuthProviders.default), []));
        }
        return directives;
    }
    ensureDefaultAuthProviderAssigned(rules) {
        for (const rule of rules) {
            if (!rule.provider) {
                switch (rule.allow) {
                    case 'owner':
                    case 'groups':
                        rule.provider = 'userPools';
                        break;
                    case 'private':
                        rule.provider = 'userPools';
                        break;
                    case 'public':
                        rule.provider = 'apiKey';
                        break;
                    default:
                        rule.provider = null;
                        break;
                }
            }
        }
    }
    validateRuleAuthStrategy(rule) {
        if (rule.allow === 'groups' && rule.provider !== 'userPools' && rule.provider !== 'oidc') {
            throw new graphql_transformer_core_1.InvalidDirectiveError(`@auth directive with 'groups' strategy only supports 'userPools' and 'oidc' providers, but found '${rule.provider}' assigned.`);
        }
        if (rule.allow === 'owner') {
            if (rule.provider !== null && rule.provider !== 'userPools' && rule.provider !== 'oidc') {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`@auth directive with 'owner' strategy only supports 'userPools' (default) and 'oidc' providers, but \
found '${rule.provider}' assigned.`);
            }
        }
        if (rule.allow === 'public') {
            if (rule.provider !== null && rule.provider !== 'apiKey' && rule.provider !== 'iam') {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`@auth directive with 'public' strategy only supports 'apiKey' (default) and 'iam' providers, but \
found '${rule.provider}' assigned.`);
            }
        }
        if (rule.allow === 'private') {
            if (rule.provider !== null && rule.provider !== 'userPools' && rule.provider !== 'iam') {
                throw new graphql_transformer_core_1.InvalidDirectiveError(`@auth directive with 'private' strategy only supports 'userPools' (default) and 'iam' providers, but \
found '${rule.provider}' assigned.`);
            }
        }
        if (rule.provider === 'apiKey' && this.configuredAuthProviders.hasApiKey === false) {
            throw new graphql_transformer_core_1.InvalidDirectiveError(`@auth directive with 'apiKey' provider found, but the project has no API Key authentication provider configured.`);
        }
        else if (rule.provider === 'oidc' && this.configuredAuthProviders.hasOIDC === false) {
            throw new graphql_transformer_core_1.InvalidDirectiveError(`@auth directive with 'oidc' provider found, but the project has no OPENID_CONNECT authentication provider configured.`);
        }
        else if (rule.provider === 'userPools' && this.configuredAuthProviders.hasUserPools === false) {
            throw new graphql_transformer_core_1.InvalidDirectiveError(`@auth directive with 'userPools' provider found, but the project has no Cognito User Pools authentication provider configured.`);
        }
        else if (rule.provider === 'iam' && this.configuredAuthProviders.hasIAM === false) {
            throw new graphql_transformer_core_1.InvalidDirectiveError(`@auth directive with 'iam' provider found, but the project has no IAM authentication provider configured.`);
        }
    }
    getConfiguredAuthProviders() {
        const providers = [
            this.config.authConfig.defaultAuthentication.authenticationType,
            ...this.config.authConfig.additionalAuthenticationProviders.map(p => p.authenticationType),
        ];
        const getAuthProvider = (authType) => {
            switch (authType) {
                case 'AMAZON_COGNITO_USER_POOLS':
                    return 'userPools';
                case 'API_KEY':
                    return 'apiKey';
                case 'AWS_IAM':
                    return 'iam';
                case 'OPENID_CONNECT':
                    return 'oidc';
            }
        };
        return {
            default: getAuthProvider(this.config.authConfig.defaultAuthentication.authenticationType),
            onlyDefaultAuthProviderConfigured: this.config.authConfig.additionalAuthenticationProviders.length === 0,
            hasApiKey: providers.find(p => p === 'API_KEY') ? true : false,
            hasUserPools: providers.find(p => p === 'AMAZON_COGNITO_USER_POOLS') ? true : false,
            hasOIDC: providers.find(p => p === 'OPENID_CONNECT') ? true : false,
            hasIAM: providers.find(p => p === 'AWS_IAM') ? true : false,
        };
    }
    setAuthPolicyFlag(rules) {
        if (!rules || rules.length === 0 || this.generateIAMPolicyforAuthRole === true) {
            return;
        }
        for (const rule of rules) {
            if ((rule.allow === 'private' || rule.allow === 'public') && rule.provider === 'iam') {
                this.generateIAMPolicyforAuthRole = true;
                return;
            }
        }
    }
    setUnauthPolicyFlag(rules) {
        if (!rules || rules.length === 0 || this.generateIAMPolicyforUnauthRole === true) {
            return;
        }
        for (const rule of rules) {
            if (rule.allow === 'public' && rule.provider === 'iam') {
                this.generateIAMPolicyforUnauthRole = true;
                return;
            }
        }
    }
    getAuthRulesFromDirective(directive) {
        const get = (s) => (arg) => arg.name.value === s;
        const getArg = (arg, dflt) => {
            const argument = directive.arguments.find(get(arg));
            return argument ? graphql_1.valueFromASTUntyped(argument.value) : dflt;
        };
        return getArg('rules', []);
    }
    isTypeNeedsDefaultProviderAccess(def) {
        const authDirective = def.directives.find(dir => dir.name.value === 'auth');
        if (!authDirective) {
            return true;
        }
        const rules = this.getAuthRulesFromDirective(authDirective);
        this.ensureDefaultAuthProviderAssigned(rules);
        return Boolean(rules.find(r => r.provider === this.configuredAuthProviders.default));
    }
    isTypeHasRulesForOperation(def, operation) {
        const authDirective = def.directives.find(dir => dir.name.value === 'auth');
        if (!authDirective) {
            return false;
        }
        const rules = this.getAuthRulesFromDirective(authDirective);
        this.ensureDefaultAuthProviderAssigned(rules);
        const { operationRules, queryRules } = this.splitRules(rules);
        const hasRulesForDefaultProvider = (operationRules) => {
            return Boolean(operationRules.find(r => r.provider === this.configuredAuthProviders.default));
        };
        switch (operation) {
            case 'create':
                return hasRulesForDefaultProvider(operationRules.create);
            case 'update':
                return hasRulesForDefaultProvider(operationRules.update);
            case 'delete':
                return hasRulesForDefaultProvider(operationRules.delete);
            case 'get':
                return hasRulesForDefaultProvider(operationRules.read) || hasRulesForDefaultProvider(queryRules.get);
            case 'list':
                return hasRulesForDefaultProvider(operationRules.read) || hasRulesForDefaultProvider(queryRules.list);
        }
        return false;
    }
    addTypeToResourceReferences(typeName, rules) {
        const iamPublicRules = rules.filter(r => r.allow === 'public' && r.provider === 'iam');
        const iamPrivateRules = rules.filter(r => r.allow === 'private' && r.provider === 'iam');
        if (iamPublicRules.length > 0) {
            this.unauthPolicyResources.add(`${typeName}/null`);
            this.authPolicyResources.add(`${typeName}/null`);
        }
        if (iamPrivateRules.length > 0) {
            this.authPolicyResources.add(`${typeName}/null`);
        }
    }
    addFieldToResourceReferences(typeName, fieldName, rules) {
        const iamPublicRules = rules.filter(r => r.allow === 'public' && r.provider === 'iam');
        const iamPrivateRules = rules.filter(r => r.allow === 'private' && r.provider === 'iam');
        if (iamPublicRules.length > 0) {
            this.unauthPolicyResources.add(`${typeName}/${fieldName}`);
            this.authPolicyResources.add(`${typeName}/${fieldName}`);
        }
        if (iamPrivateRules.length > 0) {
            this.authPolicyResources.add(`${typeName}/${fieldName}`);
        }
    }
    isOperationExpressionSet(operationTypeName, template) {
        return template.includes(`$ctx.result.put("operation", "${operationTypeName}")`);
    }
    updateMutationConditionInput(ctx, type, rules) {
        const tableXMutationConditionInputName = graphql_transformer_common_1.ModelResourceIDs.ModelConditionInputTypeName(type.name.value);
        if (this.typeExist(tableXMutationConditionInputName, ctx)) {
            const tableXMutationConditionInput = ctx.getType(tableXMutationConditionInputName);
            const fieldNames = new Set();
            const getAuthFieldNames = () => {
                if (rules.length > 0) {
                    const ownerRules = this.getOwnerRules(rules);
                    const ownerFieldNameArgs = ownerRules.filter(rule => !!rule.ownerField).map(rule => rule.ownerField);
                    ownerFieldNameArgs.forEach((f) => fieldNames.add(f));
                    if (ownerRules.find(rule => !rule.ownerField)) {
                        fieldNames.add('owner');
                    }
                    const groupsRules = rules.filter(rule => rule.allow === 'groups');
                    const groupFieldNameArgs = groupsRules.filter(rule => !!rule.groupsField).map(rule => rule.groupsField);
                    groupFieldNameArgs.forEach((f) => fieldNames.add(f));
                    if (groupsRules.find(rule => !rule.groupsField)) {
                        fieldNames.add('groups');
                    }
                }
            };
            getAuthFieldNames();
            if (fieldNames.size > 0) {
                const reducedFields = tableXMutationConditionInput.fields.filter(field => !fieldNames.has(field.name.value));
                const updatedInput = {
                    ...tableXMutationConditionInput,
                    fields: reducedFields,
                };
                ctx.putType(updatedInput);
            }
        }
    }
    typeExist(type, ctx) {
        return Boolean(type in ctx.nodeMap);
    }
    isSyncEnabled(ctx, typeName) {
        const resolverConfig = ctx.getResolverConfig();
        if (resolverConfig && resolverConfig.project) {
            return true;
        }
        if (resolverConfig && resolverConfig.models && resolverConfig.models[typeName]) {
            return true;
        }
        return false;
    }
}
exports.ModelAuthTransformer = ModelAuthTransformer;
function fieldHasDirective(field, directiveName) {
    return (field.directives && field.directives.length && Boolean(field.directives.find((d) => d.name.value === directiveName)));
}
function isTruthyOrNull(obj) {
    return obj || obj === null;
}
function isUndefined(obj) {
    return obj === undefined;
}
//# sourceMappingURL=ModelAuthTransformer.js.map