"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackResourceRenamer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
/**
 * StackResourceRenamer renames stack name and stack's subordinate resources' physical names, so that a CDK stack can be used to create multiple stacks in same AWS environment.
 *
 * @stability stable
 */
class StackResourceRenamer {
    /**
     * Construct a new StackResourceRenamer.
     *
     * @param renameOper RenameOperation is used to rename stack name and resources' physical names.
     * @param props Properties are set to customize rename operations.
     * @stability stable
     */
    constructor(renameOper, props = {}) {
        this.renameOper = renameOper;
        //mapping for resources whose physical names donot follow
        //the regular naming conventions: `${resourceType}`+'Name'
        this.irregularNames = {
            Output: 'exportName',
            ScalingPolicy: 'policyName',
            SlackChannelConfiguration: 'configurationName',
            CompositeAlarm: 'alarmName',
            SecurityGroup: 'groupName',
            DBProxy: 'dbProxyName',
        };
        //by default, only rename user provdied custom names
        this.customNameOnly = true;
        this.defaultNameField = 'name';
        if (props.irregularResourceNames) {
            this.irregularNames = {
                ...this.irregularNames,
                ...props.irregularResourceNames,
            };
        }
        this.excludeResTypes = props.excludeResourceTypes;
        this.includeResTypes = props.includeResourceTypes;
        if (props.userCustomNameOnly !== undefined) {
            this.customNameOnly = props.userCustomNameOnly;
        }
    }
    /**
     * Static method to rename a stack and all its subordinate resources.
     *
     * @param stack The stack (and all its children resources) to be renamed.
     * @param renameOper RenameOperation is used to rename stack name and resources' physical names.
     * @param props Properties are set to customize rename operations.
     * @stability stable
     */
    static rename(stack, renameOper, props = {}) {
        cdk.Aspects.of(stack).add(new StackResourceRenamer(renameOper, props));
    }
    /**
     * Implement core.IAspect interface.
     *
     * @param node CFN resources to be renamed.
     * @stability stable
     */
    visit(node) {
        if (node instanceof cdk.Stack) {
            //rename stack
            this.renameResource(node, 'Stack');
        }
        else {
            //rename CFN resources
            let ctorName = node.constructor.name;
            if (ctorName.startsWith('Cfn')) {
                this.renameResource(node, ctorName.substring(3));
            }
        }
    }
    /**
     * Rename a CFN resource or stack.
     *
     * @param node CFN resource or stack.
     * @param resTypeName The type name of CFN resource.
     * @stability stable
     */
    renameResource(node, resTypeName) {
        //check include/exclude
        if (this.excludeResTypes && this.excludeResTypes.length > 0 &&
            this.excludeResTypes.indexOf(resTypeName) !== -1) {
            return;
        }
        if (this.includeResTypes && this.includeResTypes.length > 0 &&
            this.includeResTypes.indexOf(resTypeName) === -1) {
            return;
        }
        //find the specific "name" field for CFN resources
        let physicalName = 'name';
        if (this.irregularNames[resTypeName]) {
            physicalName = this.irregularNames[resTypeName];
        }
        else {
            //decapitalize regular resource names
            let [first, ...rest] = resTypeName;
            let decapName = first.toLowerCase() + rest.join('');
            physicalName = `${decapName}Name`;
        }
        if (physicalName.length === 0) {
            return;
        }
        //some names (eg. stackName, exportName) only has getter,
        //need access protected fields starting with underscore
        let underscoreName = '_' + physicalName;
        //rename
        let b = node;
        if (b[physicalName] && b[physicalName].length > 0) {
            if (isWritable(b, physicalName) && this.isTarget(b[physicalName])) {
                b[physicalName] = this.renameOper.rename(b[physicalName], resTypeName);
            }
            else if (b[underscoreName] && b[underscoreName].length > 0 && isWritable(b, underscoreName) && this.isTarget(b[underscoreName])) {
                b[underscoreName] = this.renameOper.rename(b[underscoreName], resTypeName);
            }
        }
        else if (b[this.defaultNameField] && b[this.defaultNameField].length > 0 &&
            isWritable(b, this.defaultNameField) && this.isTarget(b[this.defaultNameField])) {
            b[this.defaultNameField] = this.renameOper.rename(b[this.defaultNameField], resTypeName);
        }
    }
    /**
     * check if a resName(resource name) is a valid target for rename;
     *
     * @stability stable
     */
    isTarget(resName) {
        let isAWSGenerated = cdk.Token.isUnresolved(resName);
        if (this.customNameOnly && isAWSGenerated) {
            return false;
        }
        return true;
    }
}
exports.StackResourceRenamer = StackResourceRenamer;
_a = JSII_RTTI_SYMBOL_1;
StackResourceRenamer[_a] = { fqn: "cdk-stack-resource-rename.StackResourceRenamer", version: "0.0.17" };
function isWritable(obj, key) {
    let desc;
    for (let o = obj; o != Object.prototype; o = Object.getPrototypeOf(o)) {
        desc = Object.getOwnPropertyDescriptor(o, key);
        if (desc !== undefined) {
            break;
        }
    }
    if (desc === undefined) {
        desc = {};
    }
    return Boolean(desc.writable);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxxQ0FBcUM7Ozs7OztBQW1FckMsTUFBYSxvQkFBb0I7Ozs7Ozs7O0lBcUMvQixZQUFvQixVQUE0QixFQUFFLFFBQXFCLEVBQUU7UUFBckQsZUFBVSxHQUFWLFVBQVUsQ0FBa0I7UUF2QmhELHlEQUF5RDtRQUN6RCwwREFBMEQ7UUFDbEQsbUJBQWMsR0FBOEI7WUFDbEQsTUFBTSxFQUFFLFlBQVk7WUFDcEIsYUFBYSxFQUFFLFlBQVk7WUFDM0IseUJBQXlCLEVBQUUsbUJBQW1CO1lBQzlDLGNBQWMsRUFBRSxXQUFXO1lBQzNCLGFBQWEsRUFBRSxXQUFXO1lBQzFCLE9BQU8sRUFBRSxhQUFhO1NBQ3ZCLENBQUM7UUFHRixvREFBb0Q7UUFDNUMsbUJBQWMsR0FBRyxJQUFJLENBQUM7UUFDdEIscUJBQWdCLEdBQUcsTUFBTSxDQUFDO1FBVWhDLElBQUksS0FBSyxDQUFDLHNCQUFzQixFQUFFO1lBQ2hDLElBQUksQ0FBQyxjQUFjLEdBQUc7Z0JBQ3BCLEdBQUcsSUFBSSxDQUFDLGNBQWM7Z0JBQ3RCLEdBQUcsS0FBSyxDQUFDLHNCQUFzQjthQUNoQyxDQUFDO1NBQ0g7UUFDRCxJQUFJLENBQUMsZUFBZSxHQUFHLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQztRQUNsRCxJQUFJLENBQUMsZUFBZSxHQUFHLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQztRQUNsRCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsS0FBRyxTQUFTLEVBQUU7WUFDeEMsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7U0FDaEQ7SUFDSCxDQUFDOzs7Ozs7Ozs7SUF2Q0QsTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFxQixFQUFFLFVBQTRCLEVBQUUsUUFBcUIsRUFBRTtRQUN4RixHQUFHLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxvQkFBb0IsQ0FBQyxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQztJQUN6RSxDQUFDOzs7Ozs7O0lBMENELEtBQUssQ0FBQyxJQUFvQjtRQUN4QixJQUFJLElBQUksWUFBWSxHQUFHLENBQUMsS0FBSyxFQUFFO1lBQzdCLGNBQWM7WUFDZCxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztTQUNwQzthQUFNO1lBQ0wsc0JBQXNCO1lBQ3RCLElBQUksUUFBUSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDO1lBQ3JDLElBQUksUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsRUFBRTtnQkFDOUIsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQ2xEO1NBQ0Y7SUFDSCxDQUFDOzs7Ozs7OztJQU1TLGNBQWMsQ0FBQyxJQUFvQixFQUFFLFdBQW1CO1FBQ2hFLHVCQUF1QjtRQUN2QixJQUFJLElBQUksQ0FBQyxlQUFlLElBQUksSUFBSSxDQUFDLGVBQWUsQ0FBQyxNQUFNLEdBQUcsQ0FBQztZQUN6RCxJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRTtZQUNsRCxPQUFPO1NBQ1I7UUFDRCxJQUFJLElBQUksQ0FBQyxlQUFlLElBQUksSUFBSSxDQUFDLGVBQWUsQ0FBQyxNQUFNLEdBQUcsQ0FBQztZQUN6RCxJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRTtZQUNsRCxPQUFPO1NBQ1I7UUFDRCxrREFBa0Q7UUFDbEQsSUFBSSxZQUFZLEdBQUcsTUFBTSxDQUFDO1FBQzFCLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsRUFBRTtZQUNwQyxZQUFZLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNqRDthQUFNO1lBQ0wscUNBQXFDO1lBQ3JDLElBQUksQ0FBQyxLQUFLLEVBQUUsR0FBRyxJQUFJLENBQUMsR0FBRyxXQUFXLENBQUM7WUFDbkMsSUFBSSxTQUFTLEdBQUcsS0FBSyxDQUFDLFdBQVcsRUFBRSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDcEQsWUFBWSxHQUFHLEdBQUcsU0FBUyxNQUFNLENBQUM7U0FDbkM7UUFDRCxJQUFJLFlBQVksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzdCLE9BQU87U0FDUjtRQUNELHlEQUF5RDtRQUN6RCx1REFBdUQ7UUFDdkQsSUFBSSxjQUFjLEdBQUcsR0FBRyxHQUFHLFlBQVksQ0FBQztRQUN4QyxRQUFRO1FBQ1IsSUFBSSxDQUFDLEdBQUksSUFBWSxDQUFDO1FBQ3RCLElBQUksQ0FBQyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ2pELElBQUksVUFBVSxDQUFDLENBQUMsRUFBRSxZQUFZLENBQUMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxFQUFFO2dCQUNqRSxDQUFDLENBQUMsWUFBWSxDQUFDLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxFQUFFLFdBQVcsQ0FBQyxDQUFDO2FBQ3hFO2lCQUFNLElBQUksQ0FBQyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxJQUFJLFVBQVUsQ0FBQyxDQUFDLEVBQUUsY0FBYyxDQUFDLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsRUFBRTtnQkFDakksQ0FBQyxDQUFDLGNBQWMsQ0FBQyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsRUFBRSxXQUFXLENBQUMsQ0FBQzthQUM1RTtTQUNGO2FBQU0sSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDO1lBQ3hFLFVBQVUsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsRUFBRTtZQUNqRixDQUFDLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1NBQzFGO0lBQ0gsQ0FBQzs7Ozs7O0lBSVMsUUFBUSxDQUFDLE9BQVk7UUFDN0IsSUFBSSxjQUFjLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDckQsSUFBSSxJQUFJLENBQUMsY0FBYyxJQUFJLGNBQWMsRUFBRTtZQUN6QyxPQUFPLEtBQUssQ0FBQztTQUNkO1FBQ0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDOztBQXZISCxvREF3SEM7OztBQUVELFNBQVMsVUFBVSxDQUFtQixHQUFNLEVBQUUsR0FBWTtJQUN4RCxJQUFJLElBQWtDLENBQUM7SUFDdkMsS0FBSyxJQUFJLENBQUMsR0FBRyxHQUFHLEVBQUUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxTQUFTLEVBQUUsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLEVBQUU7UUFDckUsSUFBSSxHQUFHLE1BQU0sQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDL0MsSUFBSSxJQUFJLEtBQUssU0FBUyxFQUFFO1lBQ3RCLE1BQU07U0FDUDtLQUNGO0lBQ0QsSUFBSSxJQUFJLEtBQUcsU0FBUyxFQUFFO1FBQ3BCLElBQUksR0FBQyxFQUFFLENBQUM7S0FDVDtJQUNELE9BQU8sT0FBTyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUNoQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSVJlbmFtZU9wZXJhdGlvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZW5hbWUocmVzb3VyY2VOYW1lOiBzdHJpbmcsIHJlc291cmNlVHlwZTogc3RyaW5nKTogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgUmVuYW1lUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdXNlckN1c3RvbU5hbWVPbmx5PzogYm9vbGVhbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpcnJlZ3VsYXJSZXNvdXJjZU5hbWVzPzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBleGNsdWRlUmVzb3VyY2VUeXBlcz86IHN0cmluZ1tdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbmNsdWRlUmVzb3VyY2VUeXBlcz86IHN0cmluZ1tdO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgU3RhY2tSZXNvdXJjZVJlbmFtZXIgaW1wbGVtZW50cyBjZGsuSUFzcGVjdCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgc3RhdGljIHJlbmFtZShzdGFjazogY2RrLklDb25zdHJ1Y3QsIHJlbmFtZU9wZXI6IElSZW5hbWVPcGVyYXRpb24sIHByb3BzOiBSZW5hbWVQcm9wcyA9IHt9KSB7XG4gICAgY2RrLkFzcGVjdHMub2Yoc3RhY2spLmFkZChuZXcgU3RhY2tSZXNvdXJjZVJlbmFtZXIocmVuYW1lT3BlciwgcHJvcHMpKTtcbiAgfVxuXG4gIC8vbWFwcGluZyBmb3IgcmVzb3VyY2VzIHdob3NlIHBoeXNpY2FsIG5hbWVzIGRvbm90IGZvbGxvd1xuICAvL3RoZSByZWd1bGFyIG5hbWluZyBjb252ZW50aW9uczogYCR7cmVzb3VyY2VUeXBlfWArJ05hbWUnXG4gIHByaXZhdGUgaXJyZWd1bGFyTmFtZXM6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH0gPSB7XG4gICAgT3V0cHV0OiAnZXhwb3J0TmFtZScsXG4gICAgU2NhbGluZ1BvbGljeTogJ3BvbGljeU5hbWUnLFxuICAgIFNsYWNrQ2hhbm5lbENvbmZpZ3VyYXRpb246ICdjb25maWd1cmF0aW9uTmFtZScsXG4gICAgQ29tcG9zaXRlQWxhcm06ICdhbGFybU5hbWUnLFxuICAgIFNlY3VyaXR5R3JvdXA6ICdncm91cE5hbWUnLFxuICAgIERCUHJveHk6ICdkYlByb3h5TmFtZScsXG4gIH07XG4gIHByaXZhdGUgaW5jbHVkZVJlc1R5cGVzOiBzdHJpbmdbXSB8IHVuZGVmaW5lZDtcbiAgcHJpdmF0ZSBleGNsdWRlUmVzVHlwZXM6IHN0cmluZ1tdIHwgdW5kZWZpbmVkO1xuICAvL2J5IGRlZmF1bHQsIG9ubHkgcmVuYW1lIHVzZXIgcHJvdmRpZWQgY3VzdG9tIG5hbWVzXG4gIHByaXZhdGUgY3VzdG9tTmFtZU9ubHkgPSB0cnVlO1xuICBwcml2YXRlIGRlZmF1bHROYW1lRmllbGQgPSAnbmFtZSc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZW5hbWVPcGVyOiBJUmVuYW1lT3BlcmF0aW9uLCBwcm9wczogUmVuYW1lUHJvcHMgPSB7fSkge1xuICAgIGlmIChwcm9wcy5pcnJlZ3VsYXJSZXNvdXJjZU5hbWVzKSB7XG4gICAgICB0aGlzLmlycmVndWxhck5hbWVzID0ge1xuICAgICAgICAuLi50aGlzLmlycmVndWxhck5hbWVzLFxuICAgICAgICAuLi5wcm9wcy5pcnJlZ3VsYXJSZXNvdXJjZU5hbWVzLFxuICAgICAgfTtcbiAgICB9XG4gICAgdGhpcy5leGNsdWRlUmVzVHlwZXMgPSBwcm9wcy5leGNsdWRlUmVzb3VyY2VUeXBlcztcbiAgICB0aGlzLmluY2x1ZGVSZXNUeXBlcyA9IHByb3BzLmluY2x1ZGVSZXNvdXJjZVR5cGVzO1xuICAgIGlmIChwcm9wcy51c2VyQ3VzdG9tTmFtZU9ubHkhPT11bmRlZmluZWQpIHtcbiAgICAgIHRoaXMuY3VzdG9tTmFtZU9ubHkgPSBwcm9wcy51c2VyQ3VzdG9tTmFtZU9ubHk7XG4gICAgfVxuICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICB2aXNpdChub2RlOiBjZGsuSUNvbnN0cnVjdCk6IHZvaWQge1xuICAgIGlmIChub2RlIGluc3RhbmNlb2YgY2RrLlN0YWNrKSB7XG4gICAgICAvL3JlbmFtZSBzdGFja1xuICAgICAgdGhpcy5yZW5hbWVSZXNvdXJjZShub2RlLCAnU3RhY2snKTtcbiAgICB9IGVsc2Uge1xuICAgICAgLy9yZW5hbWUgQ0ZOIHJlc291cmNlc1xuICAgICAgbGV0IGN0b3JOYW1lID0gbm9kZS5jb25zdHJ1Y3Rvci5uYW1lO1xuICAgICAgaWYgKGN0b3JOYW1lLnN0YXJ0c1dpdGgoJ0NmbicpKSB7XG4gICAgICAgIHRoaXMucmVuYW1lUmVzb3VyY2Uobm9kZSwgY3Rvck5hbWUuc3Vic3RyaW5nKDMpKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHByb3RlY3RlZCByZW5hbWVSZXNvdXJjZShub2RlOiBjZGsuSUNvbnN0cnVjdCwgcmVzVHlwZU5hbWU6IHN0cmluZykge1xuICAgIC8vY2hlY2sgaW5jbHVkZS9leGNsdWRlXG4gICAgaWYgKHRoaXMuZXhjbHVkZVJlc1R5cGVzICYmIHRoaXMuZXhjbHVkZVJlc1R5cGVzLmxlbmd0aCA+IDAgJiZcbiAgICAgIHRoaXMuZXhjbHVkZVJlc1R5cGVzLmluZGV4T2YocmVzVHlwZU5hbWUpICE9PSAtMSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgICBpZiAodGhpcy5pbmNsdWRlUmVzVHlwZXMgJiYgdGhpcy5pbmNsdWRlUmVzVHlwZXMubGVuZ3RoID4gMCAmJlxuICAgICAgdGhpcy5pbmNsdWRlUmVzVHlwZXMuaW5kZXhPZihyZXNUeXBlTmFtZSkgPT09IC0xKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIC8vZmluZCB0aGUgc3BlY2lmaWMgXCJuYW1lXCIgZmllbGQgZm9yIENGTiByZXNvdXJjZXNcbiAgICBsZXQgcGh5c2ljYWxOYW1lID0gJ25hbWUnO1xuICAgIGlmICh0aGlzLmlycmVndWxhck5hbWVzW3Jlc1R5cGVOYW1lXSkge1xuICAgICAgcGh5c2ljYWxOYW1lID0gdGhpcy5pcnJlZ3VsYXJOYW1lc1tyZXNUeXBlTmFtZV07XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vZGVjYXBpdGFsaXplIHJlZ3VsYXIgcmVzb3VyY2UgbmFtZXNcbiAgICAgIGxldCBbZmlyc3QsIC4uLnJlc3RdID0gcmVzVHlwZU5hbWU7XG4gICAgICBsZXQgZGVjYXBOYW1lID0gZmlyc3QudG9Mb3dlckNhc2UoKSArIHJlc3Quam9pbignJyk7XG4gICAgICBwaHlzaWNhbE5hbWUgPSBgJHtkZWNhcE5hbWV9TmFtZWA7XG4gICAgfVxuICAgIGlmIChwaHlzaWNhbE5hbWUubGVuZ3RoID09PSAwKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIC8vc29tZSBuYW1lcyAoZWcuIHN0YWNrTmFtZSwgZXhwb3J0TmFtZSkgb25seSBoYXMgZ2V0dGVyLFxuICAgIC8vbmVlZCBhY2Nlc3MgcHJvdGVjdGVkIGZpZWxkcyBzdGFydGluZyB3aXRoIHVuZGVyc2NvcmVcbiAgICBsZXQgdW5kZXJzY29yZU5hbWUgPSAnXycgKyBwaHlzaWNhbE5hbWU7XG4gICAgLy9yZW5hbWVcbiAgICBsZXQgYiA9IChub2RlIGFzIGFueSk7XG4gICAgaWYgKGJbcGh5c2ljYWxOYW1lXSAmJiBiW3BoeXNpY2FsTmFtZV0ubGVuZ3RoID4gMCkge1xuICAgICAgaWYgKGlzV3JpdGFibGUoYiwgcGh5c2ljYWxOYW1lKSAmJiB0aGlzLmlzVGFyZ2V0KGJbcGh5c2ljYWxOYW1lXSkpIHtcbiAgICAgICAgYltwaHlzaWNhbE5hbWVdID0gdGhpcy5yZW5hbWVPcGVyLnJlbmFtZShiW3BoeXNpY2FsTmFtZV0sIHJlc1R5cGVOYW1lKTtcbiAgICAgIH0gZWxzZSBpZiAoYlt1bmRlcnNjb3JlTmFtZV0gJiYgYlt1bmRlcnNjb3JlTmFtZV0ubGVuZ3RoID4gMCAmJiBpc1dyaXRhYmxlKGIsIHVuZGVyc2NvcmVOYW1lKSAmJiB0aGlzLmlzVGFyZ2V0KGJbdW5kZXJzY29yZU5hbWVdKSkge1xuICAgICAgICBiW3VuZGVyc2NvcmVOYW1lXSA9IHRoaXMucmVuYW1lT3Blci5yZW5hbWUoYlt1bmRlcnNjb3JlTmFtZV0sIHJlc1R5cGVOYW1lKTtcbiAgICAgIH1cbiAgICB9IGVsc2UgaWYgKGJbdGhpcy5kZWZhdWx0TmFtZUZpZWxkXSAmJiBiW3RoaXMuZGVmYXVsdE5hbWVGaWVsZF0ubGVuZ3RoID4gMCAmJlxuICAgICAgaXNXcml0YWJsZShiLCB0aGlzLmRlZmF1bHROYW1lRmllbGQpICYmIHRoaXMuaXNUYXJnZXQoYlt0aGlzLmRlZmF1bHROYW1lRmllbGRdKSkge1xuICAgICAgYlt0aGlzLmRlZmF1bHROYW1lRmllbGRdID0gdGhpcy5yZW5hbWVPcGVyLnJlbmFtZShiW3RoaXMuZGVmYXVsdE5hbWVGaWVsZF0sIHJlc1R5cGVOYW1lKTtcbiAgICB9XG4gIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHByb3RlY3RlZCBpc1RhcmdldChyZXNOYW1lOiBhbnkpOiBib29sZWFuIHtcbiAgICBsZXQgaXNBV1NHZW5lcmF0ZWQgPSBjZGsuVG9rZW4uaXNVbnJlc29sdmVkKHJlc05hbWUpO1xuICAgIGlmICh0aGlzLmN1c3RvbU5hbWVPbmx5ICYmIGlzQVdTR2VuZXJhdGVkKSB7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuICAgIHJldHVybiB0cnVlO1xuICB9XG59XG5cbmZ1bmN0aW9uIGlzV3JpdGFibGU8VCBleHRlbmRzIE9iamVjdD4ob2JqOiBULCBrZXk6IGtleW9mIFQpOiBib29sZWFuIHtcbiAgbGV0IGRlc2M6IFByb3BlcnR5RGVzY3JpcHRvcnx1bmRlZmluZWQ7XG4gIGZvciAobGV0IG8gPSBvYmo7IG8gIT0gT2JqZWN0LnByb3RvdHlwZTsgbyA9IE9iamVjdC5nZXRQcm90b3R5cGVPZihvKSkge1xuICAgIGRlc2MgPSBPYmplY3QuZ2V0T3duUHJvcGVydHlEZXNjcmlwdG9yKG8sIGtleSk7XG4gICAgaWYgKGRlc2MgIT09IHVuZGVmaW5lZCkge1xuICAgICAgYnJlYWs7XG4gICAgfVxuICB9XG4gIGlmIChkZXNjPT09dW5kZWZpbmVkKSB7XG4gICAgZGVzYz17fTtcbiAgfVxuICByZXR1cm4gQm9vbGVhbihkZXNjLndyaXRhYmxlKTtcbn1cbiJdfQ==