import * as cdk from '@aws-cdk/core';
/**
 * Interface of operation used to rename stack and its resources.
 *
 * @stability stable
 */
export interface IRenameOperation {
    /**
     * Rename method to rename stack and its resources' physical names.
     *
     * AWS generated physical names are not changed unless StackResourceRenamer
     * is created with RenameProps{ userCustomNameOnly:false }.
     * The updated stack name or resource's name is returned.
     *
     * @param resourceName The original resource physical name (if it is not user specified custom name, it is a AWS generated unique id/token, can be checked with cdk.Token.isUnresolved()). If you want to keep original resource name, simply return it in "rename" function.
     * @param resourceType The type name of CFN resource.
     * @returns updatedResourceName This is final resource name to be used.
     * @stability stable
     */
    rename(resourceName: string, resourceType: string): string;
}
/**
 * Properties to control rename process.
 *
 * @stability stable
 */
export interface RenameProps {
    /**
     * Only rename user provided custom names.
     *
     * If set to false, rename() will be invoked for all resources names with or without custom names.
     *
     * @default True
     * @stability stable
     */
    readonly userCustomNameOnly?: boolean;
    /**
     * Mapping of resourceType names to physicalName fields for resources whose physicalName field donot follow the regular naming conventions: `${resourceType}`+'Name'.
     *
     * @default {}
     * @stability stable
     */
    readonly irregularResourceNames?: {
        [key: string]: string;
    };
    /**
     * An array of Resource Types whose physical names could not be changed.
     *
     * An empty array will allow the renaming for all resources. A non-empty
     * array will apply rename operation only if the Resource type is not in
     * this array.
     *
     * @default []
     * @stability stable
     */
    readonly excludeResourceTypes?: string[];
    /**
     * An array of Resource Types whose physical names could be updated.
     *
     * An empty array will not allow any renaming to resources. A
     * non-empty array will allow renaming only if the Resource type is in
     * this array.
     *
     * @default []
     * @stability stable
     */
    readonly includeResourceTypes?: string[];
}
/**
 * StackResourceRenamer renames stack name and stack's subordinate resources' physical names, so that a CDK stack can be used to create multiple stacks in same AWS environment.
 *
 * @stability stable
 */
export declare class StackResourceRenamer implements cdk.IAspect {
    private renameOper;
    /**
     * Static method to rename a stack and all its subordinate resources.
     *
     * @param stack The stack (and all its children resources) to be renamed.
     * @param renameOper RenameOperation is used to rename stack name and resources' physical names.
     * @param props Properties are set to customize rename operations.
     * @stability stable
     */
    static rename(stack: cdk.IConstruct, renameOper: IRenameOperation, props?: RenameProps): void;
    private irregularNames;
    private includeResTypes;
    private excludeResTypes;
    private customNameOnly;
    private defaultNameField;
    /**
     * Construct a new StackResourceRenamer.
     *
     * @param renameOper RenameOperation is used to rename stack name and resources' physical names.
     * @param props Properties are set to customize rename operations.
     * @stability stable
     */
    constructor(renameOper: IRenameOperation, props?: RenameProps);
    /**
     * Implement core.IAspect interface.
     *
     * @param node CFN resources to be renamed.
     * @stability stable
     */
    visit(node: cdk.IConstruct): void;
    /**
     * Rename a CFN resource or stack.
     *
     * @param node CFN resource or stack.
     * @param resTypeName The type name of CFN resource.
     * @stability stable
     */
    protected renameResource(node: cdk.IConstruct, resTypeName: string): void;
    /**
     * check if a resName(resource name) is a valid target for rename;
     *
     * @stability stable
     */
    protected isTarget(resName: any): boolean;
}
