# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['tpkutils']

package_data = \
{'': ['*']}

install_requires = \
['click>=8.1.3,<9.0.0',
 'mercantile>=1.2.1,<2.0.0',
 'pymbtiles>=0.5.0,<0.6.0',
 'six>=1.16.0,<2.0.0']

entry_points = \
{'console_scripts': ['tpk = tpkutils.cli:cli']}

setup_kwargs = {
    'name': 'tpkutils',
    'version': '0.8.2',
    'description': 'ArcGIS Tile Package Utilities',
    'long_description': '# ArcGIS Tile Package Utilities\n\nA Python 3 library for reading tiles and exporting tools from\n[ArcGIS Tile Packages](http://desktop.arcgis.com/en/arcmap/10.3/map/working-with-arcmap/about-tile-packages.htm)\nwhich contain tile caches using the\n[ArcGIS Compact Tile Cache](https://server.arcgis.com/en/server/10.3/publish-services/windows/inside-the-compact-cache-storage-format.htm)\n\n![Python Tests](https://github.com/consbio/tpkutils/actions/workflows/python-tests.yml/badge.svg)\n\n## Goals\n\n-   easy access to tiles in a tile package\n-   export to mbtiles, for hosting on any of a variety of mbtiles servers,\n    such as [mbtileserver](https://github.com/consbio/mbtileserver)\n\n## Our workflow\n\n-   create beautiful maps in ArcGIS Desktop\n-   [export to ArcGIS tile package](http://desktop.arcgis.com/en/arcmap/10.3/map/working-with-arcmap/how-to-create-a-tile-package.htm)\n-   convert to mbtiles format using this package\n-   host as an XYZ tile service using [mbtileserver](https://github.com/consbio/mbtileserver)\n\n## Installation\n\n```\npip install tpkutils --upgrade\n```\n\n## Usage\n\n### Python API\n\nOpen a tile package:\n\n```\nfrom tpkutils import TPK\n\ntpk = TPK(\'my_tiles.tpk\')\n\n# close when done\ntpk.close()\n```\n\nOr use `with`:\n\n```\nwith TPK(\'my_tiles.tpk\') as tpk:\n```\n\nYou can query basic information about the tile package:\n\n```\ntpk.bounds  # tuple of (xmin, ymin, xmax, ymax) in geographic coordinates\ntpk.zoom_levels  # list of zoom levels available in package [0,1,...]\n```\n\n#### Tile access\n\nYou can iterate over individual tiles - for instance, to save to disk.\nTiles are returned as a\n`namedtuple`: `Tile(z, x, y, data)`:\n\n```\nfor tile in tpk.read_tiles():\n    with open(\'{0}_{1}_{2}.png\'.format(tile.x, tile.y, tile.z), \'wb\') as outfile:\n        outfile.write(tile.data)\n```\n\nYou can also just read tiles for a given zoom level or levels:\n\n```\ntpk.read_tiles(zoom=[4])\n```\n\nBy default, tiles are returned according to the ArcGIS tile scheme.\nTo output tiles in xyz scheme, the y-value needs to be flipped:\n\n```\ntpk.read_tiles(flip_y=True)\n```\n\n_Note:_ no direct interface to read a single tile or tiles specified by\nx or y is currently provided.\n\n### Export to mbtiles\n\nYou can export a tile package to a [MapBox mbtiles v1.1](https://github.com/mapbox/mbtiles-spec/blob/master/1.1/spec.md) file:\n\n```\ntpk.to_mbtiles(\'my_tiles.mbtiles\')\n```\n\nOr just export a subset of zoom levels:\n\n```\ntpk.to_mbtiles(\'fewer_tiles.mbtiles\', zoom=[0,1,2,3,4])\n```\n\n_Note:_\n\n-   tiles are output to mbtiles format in xyz tile scheme.\n-   [mixed format](http://desktop.arcgis.com/en/arcmap/10.3/map/working-with-arcmap/about-tile-packages.htm)\n    tiles are not supported for export to mbtiles.\n\n### Export to disk\n\nYou can export the tile package to disk. Files are written to\n\'[z]/[x]/[y].[ext]\' where [ext] is one png or jpg. Alternative file\nnames can be provided using the `--path-format` option.\n\nBy default, tiles will be written in the \'arcgis\' tile scheme.\nIf using tiles in an XYZ tilevserver or client, use the \'xyz\' tile\nscheme.\n\nOutput directory must be empty.\n\n```\ntpk.to_disk(\'my_tiles\')\n```\n\nYou can export a subset of zoom levels, use the \'xyz\' scheme, and\nomit empty (completely blank PNG or completely white JPG) tiles:\n\n```\ntpk.to_disk(\'my_tiles\', zoom=[0,1,2], scheme=\'xyz\', drop_empty=True)\n```\n\n_Note:_\n\n-   not recommended for large tile packages, as this will\n    potentially create a large number of directories and files.\n-   \'mixed\' format is not supported\n\n### Metadata / descriptive attributes\n\nBasic attributes describing the tile package are extracted from\nconfiguration files in the tile package. These are typically populated\nfrom the user interface for the ArcGIS tile package tool:\n\n-   name: autopopulated by ArcGIS tile package tool, based on filename of map document\n-   description: optional field in ArcGIS tile package tool\n-   summary: required field in ArcGIS tile package tool\n-   tags: required field in ArcGIS tile package tool\n-   credits: optional field in ArcGIS tile package tool\n-   use_constraints: optional field in ArcGIS tile package tool\n\n#### MBtiles metadata\n\nThe metadata table in the mbtiles file is created from the attributes\nof the tile package. Right now, any of these attributes can be\noverwritten to control the contents of this table:\n\n```\ntpk.name = \'Some new name\'\ntpk.description = \'This is a much better description\'\ntpk.to_mbtiles(...)\n```\n\nTwo additional attributes are exposed specifically for use in mbtiles:\n\n```\ntpk.version  # version of tileset, defaults to 1.0.0\ntpk.attribution  # copyright / attribution statement.  Used by some\n                 # clients for attribution info shown on map.\n```\n\n## Command line interface\n\nYou can also use the command line to perform export operations:\n\n```\n$ tpk export mbtiles --help\nUsage: tpk export mbtiles [OPTIONS] TPK_FILENAME MBTILES_FILENAME\n\n  Export the tile package to mbtiles format\n\nOptions:\n  -z, --zoom TEXT  Limit zoom levels to export: "0,1,2"\n  --overwrite      Overwrite existing mbtiles file  [default: False]\n  -v, --verbose    Verbose output\n  --help           Show this message and exit.\n```\n\n```\n$ tpk export disk --help\nUsage: tpk export disk [OPTIONS] TPK_FILENAME PATH\n\n  Export the tile package to disk: z/x/y.<ext> or pattern specified using\n  --path-format option.\n\n  Will use the \'arcgis\' tile scheme by default.  If using with an XYZ tile\n  server or client, use the \'xyz\' tile scheme.\n\n  Not recommended for higher zoom levels as this will produce large\n  directory trees.\n\nOptions:\n  -z, --zoom TEXT        Limit zoom levels to export: "0,1,2"\n  --scheme [xyz|arcgis]  Tile numbering scheme: xyz or arcgis  [default:\n                         arcgis]\n  --drop-empty           Drop empty tiles from output\n  --path-format TEXT     Format expression for output tile files, within\n                         output path. Must contain parameters for z, x, y, and\n                         ext (extension).  [default: {z}/{x}/{y}.{ext}]\n  -p, --preview          Preview the exported tiles in a simple map.\n  -v, --verbose          Verbose output\n  --help                 Show this message and exit.\n```\n\n## Note\n\nAll tile packages are assumed to follow the Web Mercator Tiling Scheme\n(Google/Bing/etc), and be in the Web Mercator coordinate reference system.\n\nDeveloped and tested using image tile packages created using ArcGIS 10.3;\nhowever, these appear to use the 10.1 compact bundle format.\n\nArcGIS Server 10.3 introduced a new version of the compact bundle,\nwhich is not handled here yet. If you want this, please submit an issue\nwith a small test file in 10.3 format.\n\nTile packages created using other versions may not work correctly\n(please log an issue with test data).\n\nVersions from ArcGIS older than 10.1 are unlikely to be supported.\n\n\n## Changes:\n\n### 0.8.0 (unreleased)\n- removed Python 2 support\n\n\n### 0.7.0\n\n-   added ability to drop empty tiles when exporting to mbtiles via `drop_empty` option\n-   added ability to drop completely transparent tiles in addition to completely white or black tiles\n-   added ability to calculate tile bounds from highest zoom level exported to mbtiles\n-   corrected zoom levels for tilesets where tiles start at zoom levels greater than 0\n-   added `--tile-bounds` option to command line interface to calculate bounds from tiles available at highest exported zoom level\n-   added `--drop-empty` option to command line interface to drop empty tiles when creating mbtiles\n\n## Credits:\n\nTile package format is described [here](https://gdbgeek.wordpress.com/2012/08/09/demystifying-the-esri-compact-cache/).\n\nInspired by:\n\n-   [tiler-arcgis-bundle](https://github.com/FuZhenn/tiler-arcgis-bundle)\n-   [mbutil](https://github.com/mapbox/mbutil)\n-   [node-mbtiles](https://github.com/mapbox/node-mbtiles)\n\nSQL for creating mbtiles database derived from\n[node-mbtiles](https://github.com/mapbox/node-mbtiles)\n\nArcGIS is a trademark of of [ESRI](http://esri.com) and is used here\nto refer to specific technologies. No endorsement by ESRI is implied.\n\n## License:\n\nSee LICENSE.md\n',
    'author': 'Brendan Ward',
    'author_email': 'bcward@astutespruce.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
