"""
deepdataspace.environs

| The environment variables read from environments on program starts up.
| If not found, dds will try to read the default value.
| If no default value is specified, dds will raise an exception.

If you start dds services by ``dds`` CLI tool,
these variables are generated by ``dds`` with different default values,
which are written to ``~/.deepdataspace/dds.env``.

.. data:: DDS_DEPLOY

    Where the program is running? local, dev, test or prod?

    :default: `'local'`

.. data:: DDS_DEBUG

    Whether the program is running in debug mode?

    :default: `False`

.. data:: DDS_VERBOSE_LOG

    Whether the program is running in verbose mode?

    :default: `False`

.. data:: DDS_RUNTIME_DIR

    The runtime directory of dds.

    :default: ``~/.deepdataspace``

.. data:: DDS_DJANGO_DIR

    The runtime directory of django which includes django log files.

    :default: ``$DDS_RUNTIME_DIR/django``

.. data:: DDS_CELERY_DIR

    The runtime directory if celery which includes celery log files.

.. data:: DDS_REDIS_DIR

    The runtime directory of redis which includes redis bin, conf, log, pid, rdb and aof files.

    :default: ``$DDS_RUNTIME_DIR/redis``

.. data:: DDS_MONGODB_DIR

    The runtime directory of mongodb which includes mongodb bin, conf, log, pid, db files.

    :default: ``$DDS_RUNTIME_DIR/mongodb``

.. data:: DDS_DATA_DIR

    | The dataset directory which holds datasets.
    | All datasets in this directory will be imported by dds on program start up.

.. data:: DDS_DJANGO_KEY

    The django secret key.

.. data:: DDS_DJANGO_LOG_PATH

    The django log file path.

    :default: ``$DDS_DJANGO_DIR/django.log``

.. data:: DDS_CELERY_WORKERPOOL

    The celery worker pool type.

    :default: `'prefork'`
"""

import os
from pathlib import Path

from deepdataspace.constants import RunningEnv

ENV = os.environ.get("DDS_DEPLOY", RunningEnv.Local)

DEBUG = bool(os.environ.get("DDS_DEBUG", False))

VERBOSE = bool(os.environ.get("DDS_VERBOSE_LOG", False))

# All directories
HOME_DIR = str(Path.home())

RUNTIME_DIR = str(os.environ.get("DDS_RUNTIME_DIR", Path(HOME_DIR, ".deepdataspace")))
REDIS_DIR = str(os.environ.get("DDS_REDIS_DIR", Path(RUNTIME_DIR, "redis")))
DJANGO_DIR = str(os.environ.get("DDS_DJANGO_DIR", Path(RUNTIME_DIR, "django")))
CELERY_DIR = str(os.environ.get("DDS_CELERY_DIR", Path(RUNTIME_DIR, "celery")))
MONGODB_DIR = str(os.environ.get("DDS_MONGODB_DIR", Path(RUNTIME_DIR, "mongodb")))
DATASET_DIR = os.environ["DDS_DATA_DIR"]

# database configuration
DB_ENGIN = str(os.environ.get("DDS_DB_ENGINE", "sqlite3"))
DB_HOST = str(os.environ.get("DDS_DB_HOST", "127.0.0.1"))
DB_PORT = int(os.environ.get("DDS_DB_PORT", 3306))
DB_NAME = str(os.environ.get("DDS_DB_NAME", "dds"))
DB_USER = str(os.environ.get("DDS_DB_USER", "dds"))
DB_PASS = str(os.environ.get("DDS_DB_PASS", "dds"))

# redis configurations
REDIS_HOST = os.environ["DDS_REDIS_HOST"]
REDIS_PORT = os.environ["DDS_REDIS_PORT"]
REDIS_PASS = os.environ["DDS_REDIS_PASS"]
REDIS_DBNAME = os.environ["DDS_REDIS_DBNAME"]

# mongodb configurations
MONGODB_HOST = os.environ["DDS_MONGODB_HOST"]
MONGODB_PORT = os.environ["DDS_MONGODB_PORT"]
MONGODB_USER = os.environ["DDS_MONGODB_USER"]
MONGODB_PASS = os.environ["DDS_MONGODB_PASS"]
MONGODB_DBNAME = os.environ["DDS_MONGODB_DBNAME"]

# django configurations
DJANGO_SECRET = os.environ["DDS_DJANGO_KEY"]
DJANGO_LOG_PATH = str(os.environ.get("DDS_DJANGO_LOG_PATH", Path(DJANGO_DIR, "django.log")))

# celery configurations
CELERY_BROKER = f"redis://:{REDIS_PASS}@{REDIS_HOST}:{REDIS_PORT}/{REDIS_DBNAME}"
CELERY_WORKER_POOL = os.environ.get("DDS_CELERY_WORKER_POOL", "prefork")

SENTRY_DSN: str = os.environ.get("DDS_SENTRY_DSN", None)
