"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const testMetric = new lib_1.Metric({
    namespace: 'CDK/Test',
    metricName: 'Metric',
});
class TestAlarmAction {
    constructor(arn) {
        this.arn = arn;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.arn };
    }
}
module.exports = {
    'can make simple alarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            threshold: 1000,
            evaluationPeriods: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 3,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 300,
            Statistic: 'Average',
            Threshold: 1000,
        }));
        test.done();
    },
    'override metric period in Alarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            period: core_1.Duration.minutes(10),
            threshold: 1000,
            evaluationPeriods: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 3,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 600,
            Statistic: 'Average',
            Threshold: 1000,
        }));
        test.done();
    },
    'override statistic Alarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            statistic: 'max',
            threshold: 1000,
            evaluationPeriods: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 3,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 300,
            Statistic: 'Maximum',
            Threshold: 1000,
        }));
        test.done();
    },
    'can use percentile in Alarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            statistic: 'P99',
            threshold: 1000,
            evaluationPeriods: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 3,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 300,
            ExtendedStatistic: 'p99',
            Threshold: 1000,
        }));
        test.done();
    },
    'can set DatapointsToAlarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            threshold: 1000,
            evaluationPeriods: 3,
            datapointsToAlarm: 2,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 3,
            DatapointsToAlarm: 2,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 300,
            Statistic: 'Average',
            Threshold: 1000,
        }));
        test.done();
    },
    'can add actions to alarms'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const alarm = new lib_1.Alarm(stack, 'Alarm', {
            metric: testMetric,
            threshold: 1000,
            evaluationPeriods: 2
        });
        alarm.addAlarmAction(new TestAlarmAction('A'));
        alarm.addInsufficientDataAction(new TestAlarmAction('B'));
        alarm.addOkAction(new TestAlarmAction('C'));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            AlarmActions: ['A'],
            InsufficientDataActions: ['B'],
            OKActions: ['C'],
        }));
        test.done();
    },
    'can make alarm directly from metric'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        testMetric.createAlarm(stack, 'Alarm', {
            threshold: 1000,
            evaluationPeriods: 2,
            statistic: 'min',
            period: core_1.Duration.seconds(10),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanOrEqualToThreshold',
            EvaluationPeriods: 2,
            MetricName: 'Metric',
            Namespace: 'CDK/Test',
            Period: 10,
            Statistic: 'Minimum',
            Threshold: 1000,
        }));
        test.done();
    },
    'can use percentile string to make alarm'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        testMetric.createAlarm(stack, 'Alarm', {
            threshold: 1000,
            evaluationPeriods: 2,
            statistic: 'p99.9'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            ExtendedStatistic: 'p99.9',
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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