"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const layout_1 = require("./layout");
/**
 * Specify the period for graphs when the CloudWatch dashboard loads
 */
var PeriodOverride;
(function (PeriodOverride) {
    /**
     * Period of all graphs on the dashboard automatically adapt to the time range of the dashboard.
     */
    PeriodOverride["AUTO"] = "auto";
    /**
     * Period set for each graph will be used
     */
    PeriodOverride["INHERIT"] = "inherit";
})(PeriodOverride = exports.PeriodOverride || (exports.PeriodOverride = {}));
/**
 * A CloudWatch dashboard
 */
class Dashboard extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.dashboardName,
        });
        this.rows = [];
        {
            const { dashboardName } = props;
            if (dashboardName && !core_1.Token.isUnresolved(dashboardName) && !dashboardName.match(/^[\w-]+$/)) {
                throw new Error([
                    `The value ${dashboardName} for field dashboardName contains invalid characters.`,
                    'It can only contain alphanumerics, dash (-) and underscore (_).'
                ].join(' '));
            }
        }
        new cloudwatch_generated_1.CfnDashboard(this, 'Resource', {
            dashboardName: this.physicalName,
            dashboardBody: core_1.Lazy.stringValue({ produce: () => {
                    const column = new layout_1.Column(...this.rows);
                    column.position(0, 0);
                    return core_1.Stack.of(this).toJsonString({
                        start: props.start,
                        end: props.end,
                        periodOverride: props.periodOverride,
                        widgets: column.toJson(),
                    });
                } })
        });
        (props.widgets || []).forEach(row => {
            this.addWidgets(...row);
        });
    }
    /**
     * Add a widget to the dashboard.
     *
     * Widgets given in multiple calls to add() will be laid out stacked on
     * top of each other.
     *
     * Multiple widgets added in the same call to add() will be laid out next
     * to each other.
     */
    addWidgets(...widgets) {
        if (widgets.length === 0) {
            return;
        }
        const w = widgets.length > 1 ? new layout_1.Row(...widgets) : widgets[0];
        this.rows.push(w);
    }
}
exports.Dashboard = Dashboard;
//# sourceMappingURL=data:application/json;base64,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