'''
Class for a bipartite network in extended event study format.
'''
import bipartitepandas as bpd

class BipartiteExtendedEventStudy(bpd.BipartiteExtendedEventStudyBase):
    '''
    Class for bipartite networks of firms and workers in extended event study form. Inherits from BipartiteExtendedEventStudyBase.

    Arguments:
        *args: arguments for BipartiteExtendedEventStudyBase
        n_periods (int): number of periods in extended event study
        col_reference_dict (dict or None): clarify which columns are associated with a general column name, e.g. {'i': 'i', 'j': ['j1', 'j2']}; None is equivalent to {}
        col_collapse_dict (dict or None): how to collapse column (None indicates the column should be dropped), e.g. {'y': 'mean'}; None is equivalent to {}
        **kwargs: keyword arguments for BipartiteExtendedEventStudyBase
    '''

    def __init__(self, *args, n_periods=4, col_reference_dict=None, col_collapse_dict=None, **kwargs):
        # Update parameters to be lists/dictionaries instead of None (source: https://stackoverflow.com/a/54781084/17333120)
        if col_reference_dict is None:
            col_reference_dict = {}
        if col_collapse_dict is None:
            col_collapse_dict = {}
        col_reference_dict = bpd.util.update_dict({'t': [f't{t + 1}' for t in range(n_periods)]}, col_reference_dict)
        col_collapse_dict = bpd.util.update_dict({'m': 'sum'}, col_collapse_dict)
        # Initialize DataFrame
        super().__init__(*args, n_periods=n_periods, col_reference_dict=col_reference_dict, col_collapse_dict=col_collapse_dict, **kwargs)

        # self.log('BipartiteExtendedEventStudy object initialized', level='info')

    @property
    def _constructor(self):
        '''
        For inheritance from Pandas.

        Returns:
            (class): BipartiteExtendedEventStudy class
        '''
        return BipartiteExtendedEventStudy

    @property
    def _constructor_long(self):
        '''
        For .to_long(), tells BipartiteExtendedEventStudyBase which long format to use.

        Returns:
            (class): BipartiteLong class
        '''
        return bpd.BipartiteLong

    def collapse(self, level='spell', is_sorted=False, copy=True):
        '''
        Collapse extended event study data at the worker-firm spell level (so each spell for a particular worker at a particular firm becomes one observation).

        Arguments:
            level (str): if 'spell', collapse at the worker-firm spell level; if 'match', collapse at the worker-firm match level ('spell' and 'match' will differ if a worker leaves then returns to a firm)
            is_sorted (bool): if False, dataframe will be sorted by i (and t, if included). Returned dataframe will be sorted. Sorting may alter original dataframe if copy is set to False. Set is_sorted to True if dataframe is already sorted.
            copy (bool): if False, avoid copy

        Returns:
            (BipartiteEventStudyCollapsed): collapsed event study data generated by collapsing event study data at the worker-firm spell level
        '''
        raise NotImplementedError('.collapse() is not implemented for extended event study format data. Please convert to long format with the method .to_long().')

    def get_worker_m(self, is_sorted=False):
        '''
        Get NumPy array indicating whether the worker associated with each observation is a mover.

        Arguments:
            is_sorted (bool): if False, dataframe will be sorted by i in a groupby (but self will not be not sorted). Set to True if dataframe is already sorted.

        Returns:
            (NumPy Array): indicates whether the worker associated with each observation is a mover
        '''
        return self.groupby('i', sort=(not is_sorted))['m'].transform('max').to_numpy() > 0
