"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SPADeploy = void 0;
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_patterns_1 = require("@aws-cdk/aws-route53-patterns");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const s3 = require("@aws-cdk/aws-s3");
class SPADeploy extends cdk.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        if (typeof config !== 'undefined') {
            this.globalConfig = config;
        }
        else {
            this.globalConfig = {
                encryptBucket: false,
                ipFilter: false,
            };
        }
    }
    /**
     * Helper method to provide a configured s3 bucket
     */
    getS3Bucket(config, isForCloudFront) {
        const bucketConfig = {
            websiteIndexDocument: config.indexDoc,
            websiteErrorDocument: config.errorDoc,
            publicReadAccess: true,
        };
        if (this.globalConfig.encryptBucket === true) {
            bucketConfig.encryption = s3.BucketEncryption.S3_MANAGED;
        }
        if (this.globalConfig.ipFilter === true || isForCloudFront === true) {
            bucketConfig.publicReadAccess = false;
            if (typeof config.blockPublicAccess !== 'undefined') {
                bucketConfig.blockPublicAccess = config.blockPublicAccess;
            }
        }
        const bucket = new s3.Bucket(this, 'WebsiteBucket', bucketConfig);
        if (this.globalConfig.ipFilter === true && isForCloudFront === false) {
            if (typeof this.globalConfig.ipList === 'undefined') {
                this.node.addError('When IP Filter is true then the IP List is required');
            }
            const bucketPolicy = new aws_iam_1.PolicyStatement();
            bucketPolicy.addAnyPrincipal();
            bucketPolicy.addActions('s3:GetObject');
            bucketPolicy.addResources(`${bucket.bucketArn}/*`);
            bucketPolicy.addCondition('IpAddress', {
                'aws:SourceIp': this.globalConfig.ipList,
            });
            bucket.addToResourcePolicy(bucketPolicy);
        }
        return bucket;
    }
    /**
     * Helper method to provide configuration for cloudfront
     */
    getCFConfig(websiteBucket, config, accessIdentity, cert) {
        const cfConfig = {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: websiteBucket,
                        originAccessIdentity: accessIdentity,
                    },
                    behaviors: config.cfBehaviors ? config.cfBehaviors : [{ isDefaultBehavior: true }],
                },
            ],
            // We need to redirect all unknown routes back to index.html for angular routing to work
            errorConfigurations: [{
                    errorCode: 403,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                },
                {
                    errorCode: 404,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                }],
        };
        if (typeof config.certificateARN !== 'undefined' && typeof config.cfAliases !== 'undefined') {
            cfConfig.aliasConfiguration = {
                acmCertRef: config.certificateARN,
                names: config.cfAliases,
            };
        }
        if (typeof config.sslMethod !== 'undefined') {
            cfConfig.aliasConfiguration.sslMethod = config.sslMethod;
        }
        if (typeof config.securityPolicy !== 'undefined') {
            cfConfig.aliasConfiguration.securityPolicy = config.securityPolicy;
        }
        if (typeof config.zoneName !== 'undefined' && typeof cert !== 'undefined') {
            cfConfig.viewerCertificate = aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(cert, {
                aliases: [config.zoneName],
            });
        }
        return cfConfig;
    }
    /**
     * Basic setup needed for a non-ssl, non vanity url, non cached s3 website
     */
    createBasicSite(config) {
        const websiteBucket = this.getS3Bucket(config, false);
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        const cfnOutputConfig = {
            description: 'The url of the website',
            value: websiteBucket.bucketWebsiteUrl,
        };
        if (config.exportWebsiteUrlOutput === true) {
            if (typeof config.exportWebsiteUrlName === 'undefined' || config.exportWebsiteUrlName === '') {
                this.node.addError('When Output URL as AWS Export property is true then the output name is required');
            }
            cfnOutputConfig.exportName = config.exportWebsiteUrlName;
        }
        new cdk.CfnOutput(this, 'URL', cfnOutputConfig);
        return { websiteBucket };
    }
    /**
     * This will create an s3 deployment fronted by a cloudfront distribution
     * It will also setup error forwarding and unauth forwarding back to indexDoc
     */
    createSiteWithCloudfront(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new cdk.CfnOutput(this, 'cloudfront domain', {
            description: 'The domain of the website',
            value: distribution.distributionDomainName,
        });
        return { websiteBucket, distribution };
    }
    /**
     * S3 Deployment, cloudfront distribution, ssl cert and error forwarding auto
     * configured by using the details in the hosted zone provided
     */
    createSiteFromHostedZone(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const zone = aws_route53_1.HostedZone.fromLookup(this, 'HostedZone', { domainName: config.zoneName });
        const cert = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
            hostedZone: zone,
            domainName: config.zoneName,
            region: 'us-east-1',
        });
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity, cert));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new aws_route53_1.ARecord(this, 'Alias', {
            zone,
            recordName: config.zoneName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(distribution)),
        });
        new aws_route53_patterns_1.HttpsRedirect(this, 'Redirect', {
            zone,
            recordNames: [`www.${config.zoneName}`],
            targetDomain: config.zoneName,
        });
        return { websiteBucket, distribution };
    }
}
exports.SPADeploy = SPADeploy;
//# sourceMappingURL=data:application/json;base64,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