"use strict";
var _a, _b, _c, _d, _e, _f, _g;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LookupMachineImage = exports.OperatingSystemType = exports.GenericWindowsImage = exports.GenericLinuxImage = exports.AmazonLinuxStorage = exports.AmazonLinuxVirt = exports.AmazonLinuxEdition = exports.AmazonLinuxGeneration = exports.AmazonLinuxImage = exports.AmazonLinuxCpuType = exports.WindowsImage = exports.GenericSSMParameterImage = exports.MachineImage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ssm = require("@aws-cdk/aws-ssm");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const user_data_1 = require("./user-data");
/**
 * Factory functions for standard Amazon Machine Image objects.
 *
 * @stability stable
 */
class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @stability stable
     */
    static latestWindows(version, props) {
        return new WindowsImage(version, props);
    }
    /**
     * An Amazon Linux image that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @stability stable
     */
    static latestAmazonLinux(props) {
        return new AmazonLinuxImage(props);
    }
    /**
     * A Linux image where you specify the AMI ID for every region.
     *
     * @param amiMap For every region where you are deploying the stack, specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props.
     * @stability stable
     */
    static genericLinux(amiMap, props) {
        return new GenericLinuxImage(amiMap, props);
    }
    /**
     * A Windows image where you specify the AMI ID for every region.
     *
     * @param amiMap For every region where you are deploying the stack, specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props.
     * @stability stable
     */
    static genericWindows(amiMap, props) {
        return new GenericWindowsImage(amiMap, props);
    }
    /**
     * An image specified in SSM parameter store that is automatically kept up-to-date.
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @param parameterName The name of SSM parameter containing the AMi id.
     * @param os The operating system type of the AMI.
     * @param userData optional user data for the given image.
     * @stability stable
     */
    static fromSSMParameter(parameterName, os, userData) {
        return new GenericSSMParameterImage(parameterName, os, userData);
    }
    /**
     * Look up a shared Machine Image using DescribeImages.
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     *
     * @stability stable
     */
    static lookup(props) {
        return new LookupMachineImage(props);
    }
}
exports.MachineImage = MachineImage;
_a = JSII_RTTI_SYMBOL_1;
MachineImage[_a] = { fqn: "@aws-cdk/aws-ec2.MachineImage", version: "1.119.0" };
/**
 * Select the image based on a given SSM parameter.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * @stability stable
 */
class GenericSSMParameterImage {
    /**
     * @stability stable
     */
    constructor(parameterName, os, userData) {
        this.parameterName = parameterName;
        this.os = os;
        this.userData = userData;
    }
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope) {
        var _h;
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            osType: this.os,
            userData: (_h = this.userData) !== null && _h !== void 0 ? _h : (this.os === OperatingSystemType.WINDOWS ? user_data_1.UserData.forWindows() : user_data_1.UserData.forLinux()),
        };
    }
}
exports.GenericSSMParameterImage = GenericSSMParameterImage;
_b = JSII_RTTI_SYMBOL_1;
GenericSSMParameterImage[_b] = { fqn: "@aws-cdk/aws-ec2.GenericSSMParameterImage", version: "1.119.0" };
/**
 * Select the latest version of the indicated Windows version.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 *
 * @stability stable
 */
class WindowsImage extends GenericSSMParameterImage {
    /**
     * @stability stable
     */
    constructor(version, props = {}) {
        super('/aws/service/ami-windows-latest/' + version, OperatingSystemType.WINDOWS, props.userData);
    }
}
exports.WindowsImage = WindowsImage;
_c = JSII_RTTI_SYMBOL_1;
WindowsImage[_c] = { fqn: "@aws-cdk/aws-ec2.WindowsImage", version: "1.119.0" };
/**
 * CPU type.
 *
 * @stability stable
 */
var AmazonLinuxCpuType;
(function (AmazonLinuxCpuType) {
    AmazonLinuxCpuType["ARM_64"] = "arm64";
    AmazonLinuxCpuType["X86_64"] = "x86_64";
})(AmazonLinuxCpuType = exports.AmazonLinuxCpuType || (exports.AmazonLinuxCpuType = {}));
/**
 * Selects the latest version of Amazon Linux.
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * @stability stable
 */
class AmazonLinuxImage extends GenericSSMParameterImage {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        const generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        const edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        const virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
        const storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
        const cpu = (props && props.cpuType) || AmazonLinuxCpuType.X86_64;
        const parts = [
            generation,
            'ami',
            edition !== AmazonLinuxEdition.STANDARD ? edition : undefined,
            virtualization,
            cpu,
            storage,
        ].filter(x => x !== undefined); // Get rid of undefineds
        const parameterName = '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
        super(parameterName, OperatingSystemType.LINUX, props.userData);
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
_d = JSII_RTTI_SYMBOL_1;
AmazonLinuxImage[_d] = { fqn: "@aws-cdk/aws-ec2.AmazonLinuxImage", version: "1.119.0" };
/**
 * What generation of Amazon Linux to use.
 *
 * @stability stable
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux edition.
 *
 * @stability stable
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    AmazonLinuxEdition["STANDARD"] = "standard";
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux.
 *
 * @stability stable
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    AmazonLinuxVirt["HVM"] = "hvm";
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
/**
 * @stability stable
 */
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "ebs";
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map.
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 *
 * @stability stable
 */
class GenericLinuxImage {
    /**
     * @stability stable
     */
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope) {
        var _h;
        const userData = (_h = this.props.userData) !== null && _h !== void 0 ? _h : user_data_1.UserData.forLinux();
        const osType = OperatingSystemType.LINUX;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            const mapping = {};
            for (const [rgn, ami] of Object.entries(this.amiMap)) {
                mapping[rgn] = { ami };
            }
            const amiMap = new core_1.CfnMapping(scope, 'AmiMap', { mapping });
            return {
                imageId: amiMap.findInMap(core_1.Aws.REGION, 'ami'),
                userData,
                osType,
            };
        }
        const imageId = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!imageId) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId,
            userData,
            osType,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
_e = JSII_RTTI_SYMBOL_1;
GenericLinuxImage[_e] = { fqn: "@aws-cdk/aws-ec2.GenericLinuxImage", version: "1.119.0" };
/**
 * Construct a Windows machine image from an AMI map.
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 *
 * @stability stable
 */
class GenericWindowsImage {
    /**
     * @stability stable
     */
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope) {
        var _h;
        const userData = (_h = this.props.userData) !== null && _h !== void 0 ? _h : user_data_1.UserData.forWindows();
        const osType = OperatingSystemType.WINDOWS;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            const mapping = {};
            for (const [rgn, ami] of Object.entries(this.amiMap)) {
                mapping[rgn] = { ami };
            }
            const amiMap = new core_1.CfnMapping(scope, 'AmiMap', { mapping });
            return {
                imageId: amiMap.findInMap(core_1.Aws.REGION, 'ami'),
                userData,
                osType,
            };
        }
        const imageId = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!imageId) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId,
            userData,
            osType,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
_f = JSII_RTTI_SYMBOL_1;
GenericWindowsImage[_f] = { fqn: "@aws-cdk/aws-ec2.GenericWindowsImage", version: "1.119.0" };
/**
 * The OS type of a particular image.
 *
 * @stability stable
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
    OperatingSystemType[OperatingSystemType["UNKNOWN"] = 2] = "UNKNOWN";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 *
 * @stability stable
 */
class LookupMachineImage {
    /**
     * @stability stable
     */
    constructor(props) {
        this.props = props;
    }
    /**
     * Return the image to use in the given context.
     *
     * @stability stable
     */
    getImage(scope) {
        var _h;
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        const osType = this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX;
        return {
            imageId: value,
            osType,
            userData: (_h = this.props.userData) !== null && _h !== void 0 ? _h : user_data_1.UserData.forOperatingSystem(osType),
        };
    }
}
exports.LookupMachineImage = LookupMachineImage;
_g = JSII_RTTI_SYMBOL_1;
LookupMachineImage[_g] = { fqn: "@aws-cdk/aws-ec2.LookupMachineImage", version: "1.119.0" };
//# sourceMappingURL=data:application/json;base64,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