# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fossology']

package_data = \
{'': ['*']}

install_requires = \
['click>=8.1.3,<9.0.0', 'requests>=2.28.1', 'tenacity>=8.1.0']

entry_points = \
{'console_scripts': ['foss_cli = fossology.foss_cli:main']}

setup_kwargs = {
    'name': 'fossology',
    'version': '2.0.0',
    'description': 'A library to automate Fossology from Python scripts',
    'long_description': '|License| |PyPI Version| |Python Version| |Static Checks| |Fossology Tests| |Coverage|\n\n.. |License| image:: https://img.shields.io/badge/license-MIT-blue.svg\n   :target: https://github.com/deveaud-m/fossology-python/LICENSE.md\n\n.. |PyPI Version| image:: https://badge.fury.io/py/fossology.svg\n   :target: https://pypi.org/project/fossology\n\n.. |Python Version| image:: https://img.shields.io/badge/python-3.7%2C3.8%2C3.9-blue?logo=python\n   :target: https://www.python.org/doc/versions/\n\n.. |Static Checks| image:: https://github.com/deveaud-m/fossology-python/workflows/Static%20Checks/badge.svg\n   :target: https://github.com/deveaud-m/fossology-python/actions?query=workflow%3A%22Static+Checks%22\n\n.. |Fossology Tests| image:: https://github.com/deveaud-m/fossology-python/workflows/API%20Tests/badge.svg\n   :target: https://github.com/deveaud-m/fossology-python/actions?query=workflow%3A%22API+Tests%22\n\n.. |Coverage| image:: https://codecov.io/gh/fossology/fossology-python/branch/master/graph/badge.svg\n   :target: https://codecov.io/gh/fossology/fossology-python\n\nA simple wrapper for the Fossology REST API.\n\nSee `the OpenAPI specification <https://raw.githubusercontent.com/fossology/fossology/master/src/www/ui/api/documentation/openapi.yaml>`_ used to implement this library.\n\n   Compatible API versions:\n\n   - 1.2.1 (Fossology 3.10.0)\n   - 1.3.2 (Fossology 3.11.0)\n   - 1.4.0 (Fossology 4.0.0)\n   - 1.4.3 (Fossology 4.1.0)\n\n**NOTE**\n\n   Version 2.0.0 of `fossology-python` only supports Fossology API version 1.4.3 onwards because of a breaking change in\n   the version format returned by the API. Other earlier version of the wrapper support a wider range of API versions,\n   e.g. 1.5.0 supports Fossology API 1.2.1 to 1.4.0.\n\nDocumentation\n=============\n\nSee `fossology-python on Github Pages <https://fossology.github.io/fossology-python>`_.\n\nUsage\n=====\n\nInstallation\n------------\n\nThis project is available as `Python package on PyPi.org <https://pypi.org/project/fossology/>`_.\n\n-  Install fossology and required dependencies:\n\n   .. code:: shell\n\n      pip install fossology requests\n\nUsing the API\n-------------\n\n-  Get a REST API token either from the Fossology server under **``User->Edit user account``** or generate a token using the method available in this library:\n\n   .. code:: python\n\n      from fossology import fossology_token\n      from fossology.obj import TokenScope\n\n      FOSSOLOGY_SERVER = "https://fossology.example.com/"\n      FOSSOLOGY_USER = "fossy"\n      FOSSOLOGY_PASSWORD = "fossy"\n      TOKEN_NAME = "fossy_token"\n\n      token = fossology_token(\n            FOSSOLOGY_SERVER,\n            FOSSOLOGY_USER,\n            FOSSOLOGY_PASSWORD,\n            TOKEN_NAME,\n            TokenScope.WRITE\n      )\n\n-  Start using the API:\n\n   .. code:: python\n\n      from fossology import Fossology\n\n      # Starting from API version 1.2.3, the `FOSSOLOGY_USER` option is not needed anymore\n      foss = Fossology(FOSSOLOGY_SERVER, token, FOSSOLOGY_USER)\n      print(f"Logged in as user {foss.user.name}")\n\n\nUsing the CLI\n-------------\n\nFossology Python also offers a command line interface to simplify interactions with your Fossology server.\n\n- To get a list of available commands, run:\n\n   .. code:: bash\n\n      $ foss_cli --help\n      Usage: foss_cli [OPTIONS] COMMAND [ARGS]...\n\n- Generate a configuration file:\n\n   .. code:: bash\n\n      $ foss_cli config\n      Enter the URL to your Fossology server: e.g. http://fossology/repo\n      Fossology URL: http://fossology/repo\n      Enter Username and Password: e.g. fossy/fossy (in the default environment)\n      Username: fossy\n      Password: \n      Enter a scope for your Fossology token: either \'read\' or \'write\'\n      Token scope: write\n\n   This will get a token from Fossology server and store it within the local ``.foss_cli.ini`` file. \n\n   On subsequent foss_cli calls those values will be reused.\n\n   Re-run the config command to **create a new token** once it expired.\n\n- Verbosity of all foss_cli commands could be increased using the ``-v`` verbosity option:\n\n   .. code:: bash\n\n      $ foss_cli -vv [COMMAND]\n\n   This runs the given command with verbosity level 2 (all debug statements will be logged).\n\n   A log file in directory ``.foss_cli_results`` named ``.foss_cli.log`` will be created.\n\n- To create a group:\n\n   .. code:: bash\n\n      $ foss_cli -vv create_group FossGroup\n\n- To create a a folder:\n\n   .. code:: bash\n\n      $ foss_cli -vv create_folder FossFolder \\\n         --folder_group FossGroup \\\n         --folder_description "Description of FossFolder"\n\n- To upload a file:\n\n   .. code:: bash\n\n      $ foss_cli -vv upload_file tests/files/zlib_1.2.11.dfsg-0ubuntu2.debian.tar.xz \\\n            --folder_name FossFolder\n            --access_level public\n\n- To upload a source package to the server and initialize a scan workflow including report generation:\n\n   .. code:: bash\n\n      $ foss_cli -vv start_workflow --help \n      Usage: foss_cli start_workflow [OPTIONS] FILE_NAME\n      The foss_cli start_workflow command.\n      Options:\n            --folder_name TEXT            The name of the folder to upload to.\n            --file_description TEXT       The description of the upload.\n            --dry_run / --no_dry_run      Do not upload but show what would be done.\n                                          Use -vv to see output.\n            --reuse_newest_upload / --no_reuse_newest_upload\n                                          Reuse newest upload if available.\n            --reuse_newest_job / --no_reuse_newest_job\n                                          Reuse newest scheduled job for the upload if\n                                          available.\n            --report_format TEXT          The name of the reportformat. [dep5,\n                                          spdx2,spdxtv,readmeoss,unifiedreport]\n            --access_level TEXT           The access level of the\n                                          upload.[private,protected,public]\n            --help                        Show this message and exit.\n\nContribute\n==========\n\nDevelop\n-------\n\n-  All contributions in form of bug reports, feature requests or merge requests!\n\n-  Use proper\n   `docstrings <https://realpython.com/documenting-python-code/>`__ to\n   document functions and classes\n\n-  Extend the testsuite **poetry run pytest** with the new functions/classes\n\n-  The **documentation website** can automatically be generated by the `Sphinx autodoc\n   extension <http://www.sphinx-doc.org/en/master/usage/extensions/autodoc.html>`_\n\n**HINT**\n\n   To avoid running the whole testsuite during development of a new branch with changing only touching the code related\n   to the CLI, name your branch ``feat/cli-{something}`` and only the ``test_foss_cli_*`` will run in the pull request context.\n\nBuild\n-----\n\n- You can build the PyPi package using `poetry <https://poetry.eustace.io/>`_:\n\n  .. code:: shell\n\n    poetry build\n\n- Build documentation:\n\n  The static site is generated automatically by\n  `GitHub Actions <https://github.com/fossology/fossology-python/actions/workflows/doc-deploy.yml>`_\n  on every merge to main branch and pushed to **gh-pages** branch. The action uses\n  `JamesIves/github-pages-deploy-action <https://github.com/JamesIves/github-pages-deploy-action>`_\n  to deploy the static pages.\n\n  To build it locally\n\n  .. code:: shell\n\n     poetry run sphinx-build -b html docs-source docs/\n\n- Cleanup builds:\n\n  .. code:: shell\n\n     rm -r dist/ docs/\n\nTag\n----\n\nEach new release gets a new tag with important information about the changes added to the new release:\n\n.. code:: shell\n\n   git tag -a vx.x.x -m "New major/minor/patch release x.x.x"\n   git push origin vx.x.x\n\nAdd required information in the corresponding `release in the Github project <https://github.com/fossology/fossology-python/releases>`_.\n\n\nTest\n----\n\nThe testsuite available in this project expects a running Fossology instance under the hostname **fossology** with the default admin user "fossy".\n\n- Use the latest Fossology container from `Docker hub <https://hub.docker.com/r/fossology/fossology>`_:\n\n  .. code:: shell\n\n    docker pull fossology/fossology\n    tar xJf tests/files/base-files_11.tar.xz -C /tmp\n    docker run --mount src="/tmp",dst=/tmp,type=bind --name fossology -p 80:80 fossology/fossology\n\n- Start the complete test suite or a specific test case (and generate coverage report):\n\n  .. code:: shell\n\n     poetry run coverage run --source=fossology -m pytest\n     poetry run coverage report -m\n     poetry run coverage html\n',
    'author': 'Marion Deveaud',
    'author_email': 'marion.deveaud@siemens.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/fossology/fossology-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
