from typing import Callable, Dict, Optional
from selenium import webdriver
from selenium.common.exceptions import TimeoutException
from selenium.webdriver.firefox.firefox_binary import FirefoxBinary
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.common.by import By
from selenium.webdriver.common.proxy import Proxy, ProxyType
from selenium.webdriver.common.keys import Keys
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
import time
import random
import requests


class SelectBy(By):
    pass


class ExploitBuilder:
    def __init__(
        self,
        protocol: str,
        hostname: str,
        **options
    ):
        # Configure Options
        stealth = options.get("stealth", False)
        browser = options.get("browser", "firefox")
        use_proxy = options.get("use_proxy", False)
        proxy_address = options.get("proxy_address", "127.0.0.1:8080")

        self.protocol = protocol
        self.hostname = hostname
        self.stealth = stealth
        self.use_proxy = use_proxy
        self.proxy_address = proxy_address

        additional_options = options.copy()
        additional_options.pop("use_proxy", None)
        additional_options.pop("proxy_address", None)
        additional_options.pop("stealth", None)

        if browser == "firefox":
            self._init_firefox(use_proxy, proxy_address, **additional_options)
        elif browser == "chrome":
            self._init_chrome(use_proxy, proxy_address, **additional_options)

    def _init_firefox(self, use_proxy, proxy_address, **options):
        firefox_binary_path = options.get(
            "firefox_binary_path", "/opt/firefox/firefox")
        binary = FirefoxBinary(firefox_binary_path)
        if use_proxy and proxy_address:
            proxy = Proxy({
                "proxyType": ProxyType.MANUAL,
                "httpProxy": proxy_address,
                "sslProxy": proxy_address,
                "noProxy": ''
            })
            proxy.add_to_capabilities(webdriver.DesiredCapabilities.FIREFOX)
            fp = webdriver.FirefoxProfile()
            fp.set_preference("network.proxy.type", 1)
            fp.set_preference("network.proxy.http",
                              proxy_address.split(":")[0])
            fp.set_preference("network.proxy.http_port",
                              int(proxy_address.split(":")[1]))
            fp.set_preference('network.proxy.allow_hijacking_localhost', True)
            fp.update_preferences()
            fp.accept_untrusted_certs = True
            fp.assume_untrusted_cert_issuer = True
            self.driver = webdriver.Firefox(
                firefox_binary=binary,
                proxy=proxy,
                firefox_profile=fp
            )
        else:
            self.driver = webdriver.Firefox(firefox_binary=binary)

    def _init_chrome(self, use_proxy, proxy_address, **options):
        chromedriver_path = options.get(
            "chromedriver_path", "/opt/chromedriver")
        chrome_binary_path = options.get(
            "chrome_binary_path", "/opt/google/chrome/google-chrome")

        options = Options()
        options.binary_location = chrome_binary_path
        if use_proxy and proxy_address:
            options.add_argument(f'--proxy-server={proxy_address}')
            options.add_argument('ignore-certificate-errors')
        self.driver = webdriver.Chrome(
            chrome_options=options, executable_path=chromedriver_path)

    def api(self, req_method: Callable, endpoint: str, api_protocol: str = None, api_hostname: str = None, **kwargs) -> requests.Response:
        """api
        Wrapper for Python's Requests library, enabling automatic proxy usage and cookie sharing when interfacing with APIs. 
        kwargs supports same options as Requests.

        Parameters:
            req_method (Callable): requests function to use (requests.get, requests.posts, etc.)
            endpoint (str): the API endpoint to request
            api_protocol (str): the protocol to use for the API. Effectively defaults to the same protocol used in the ExploitBuilder
            api_hostname (str): the hostname to use for the API. Effectively defaults to the same hostname used in the ExploitBuilder
            kwargs: interoperable with Python Requests (e.g. headers parameter will be passed into the requests call)
        
        Returns:
        requests.Response
        """
        url = self._format(endpoint, api_protocol, api_hostname)
        cookies = self._merge_cookies(**kwargs)
        req_options = kwargs.copy()
        req_options.pop("cookies", None)
        if self.use_proxy:
            proxy_url = self._format("/", "http", self.proxy_address)
            req_options['proxies'] = {
                'http': proxy_url,
                'https': proxy_url
            }
        return req_method(url, cookies=cookies, verify=False, **req_options)

    def api_get(self, endpoint: str, api_protocol: str = None, api_hostname: str = None, **kwargs) -> requests.Response:
        """api_get
        Wrapper for ExploitBuilder.api(requests.get, ...).

        Parameters:
            req_method (Callable): requests function to use (requests.get, requests.posts, etc.)
            endpoint (str): the API endpoint to request
            api_protocol (str): the protocol to use for the API. Effectively defaults to the same protocol used in the ExploitBuilder
            api_hostname (str): the hostname to use for the API. Effectively defaults to the same hostname used in the ExploitBuilder
            kwargs: interoperable with Python Requests (e.g. headers parameter will be passed into the requests call)
        
        Returns:
        requests.Response
        """
        return self.api(requests.get, endpoint, api_protocol, api_hostname, **kwargs)

    def api_post(self, endpoint: str, api_protocol: str = None, api_hostname: str = None, **kwargs) -> requests.Response:
        """api_post
        Wrapper for ExploitBuilder.api(requests.post, ...).

        Parameters:
            req_method (Callable): requests function to use (requests.get, requests.posts, etc.)
            endpoint (str): the API endpoint to request
            api_protocol (str): the protocol to use for the API. Effectively defaults to the same protocol used in the ExploitBuilder
            api_hostname (str): the hostname to use for the API. Effectively defaults to the same hostname used in the ExploitBuilder
            kwargs: interoperable with Python Requests (e.g. headers parameter will be passed into the requests call)
        
        Returns:
        requests.Response
        """
        return self.api(requests.post, endpoint, api_protocol, api_hostname, **kwargs)

    def api_put(self, endpoint: str, api_protocol: str = None, api_hostname: str = None, **kwargs) -> requests.Response:
        """api_put
        Wrapper for ExploitBuilder.api(requests.put, ...).

        Parameters:
            req_method (Callable): requests function to use (requests.get, requests.posts, etc.)
            endpoint (str): the API endpoint to request
            api_protocol (str): the protocol to use for the API. Effectively defaults to the same protocol used in the ExploitBuilder
            api_hostname (str): the hostname to use for the API. Effectively defaults to the same hostname used in the ExploitBuilder
            kwargs: interoperable with Python Requests (e.g. headers parameter will be passed into the requests call)

        Returns:
        requests.Response
        """
        return self.api(requests.put, endpoint, api_protocol, api_hostname, **kwargs)

    def api_delete(self, endpoint: str, api_protocol: str = None, api_hostname: str = None, **kwargs) -> requests.Response:
        """api_delete
        Wrapper for ExploitBuilder.api(requests.delete, ...).

        Parameters:
            req_method (Callable): requests function to use (requests.get, requests.posts, etc.)
            endpoint (str): the API endpoint to request
            api_protocol (str): the protocol to use for the API. Effectively defaults to the same protocol used in the ExploitBuilder
            api_hostname (str): the hostname to use for the API. Effectively defaults to the same hostname used in the ExploitBuilder
            kwargs: interoperable with Python Requests (e.g. headers parameter will be passed into the requests call)

        Returns:
        requests.Response
        """
        return self.api(requests.delete, endpoint, api_protocol, api_hostname, **kwargs)

    def click(self, button: str, by: SelectBy = SelectBy.XPATH):
        """click
        Clicks an element.

        Parameters:
            button (str): The selector for the element to click
            by (SelectBy): The selector type to use (default: SelectBy.XPATH)

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.find_element(by, button).click()
        return self

    def click_by_class(self, button_class: str):
        """click_by_class
        Clicks an element based on CSS class name.

        Parameters:
            button_class (str): The element class name

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.find_element_by_class_name(button_class).click()
        return self

    def click_by_id(self, button_id: str):
        """click_by_id
        Clicks an element based on HTML ID.

        Parameters:
            button_id (str): The element HTML ID

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.find_element_by_id(button_id).click()
        return self

    def click_by_xpath(self, xpath: str):
        """click_by_xpath
        Clicks an element based on XPath.

        Parameters:
            xpath (str): The element XPath

        Returns:
        exploit_builder.ExploitBuilder"""
        self.driver.find_element(By.XPATH, xpath).click()
        return self

    def dump_cookies(self):
        """dump_cookies
        Dumps the cookies into the terminal.

        Returns:
        exploit_builder.ExploitBuilder"""

        for cookie in self.get_cookies():
            print(cookie)
        return self

    def execute_script(self, script, *args):
        """execute_script
        Wrapper for Selenium's webdriver.execute_script().

        Parameters:
            script (str): The JavaScript to execute
        
        Returns:
        Any: the return value of the JavaScript
        """
        return self.driver.execute_script(script, *args)

    def get(self, endpoint: str):
        """get
        Makes a GET request to the specified endpoint in the browser.

        Parameters:
            endpoint (str): The endpoint to which the browser navigates

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.get(self._format(endpoint))
        return self

    def get_contents(self, field: str, by: SelectBy = SelectBy.XPATH):
        """get_contents
        Get the contents of a specified field.

        Parameters:
            field (str): The selector to extract text from
            by (SelectBy): The type of selector to use (default: SelectBy.XPATH)

        Returns:
        str: The text in the given field"""
        return self.driver.find_element(by, field).text

    def get_contents_by_id(self, field: str):
        """get_contents_by_id
        Get the contents of a specified field, based on the HTML ID

        Parameters:
            field (str): The HTML ID to extract text from

        Returns:
        str: The text in the given field"""

        return self.driver.find_element_by_id(field).text

    def get_contents_by_class(self, field: str):
        """get_contents_by_class
        Get the contents of a specified field, based on the CSS class name

        Parameters:
            field (str): The CSS class name to extract text from

        Returns:
        str: The text in the given field"""

        return self.driver.find_element_by_class_name(field).text

    def get_contents_by_xpath(self, xpath: str) -> str:
        """get_contents_by_xpath
        Get the contents of a specified field, based on the XPath

        Parameters:
            xpath (str): The XPath to extract text from

        Returns:
        str: The text in the given field"""

        return self.driver.find_element(By.XPATH, xpath).text

    def get_cookie_by_name(self, name: str) -> Optional[dict]:
        """get_cookie_by_name
        Gets a browser cookie by its name.

        Parameters:
            name (str): The name of the cookie

        Returns:
        Optional(dict): The cookie, or None"""

        for cookie in self.get_cookies():
            if cookie["name"] == name:
                return cookie
        return None

    def get_cookies(self):
        """get_cookies
        Get all cookies in the browser for the current page.

        Returns:
        list(dict): A list of cookies in the browser"""
        return self.driver.get_cookies()

    def login(self, endpoint: str, username: str, password: str, username_xpath: str, password_xpath: str, submit_xpath: str):
        """login
        Performs a full login process for a given endpoint. Convenience function.

        Parameters:
            endpoint (str): The location of the login page
            username (str): The username to use
            password (str): The password to use
            username_xpath (str): The XPath of the username field
            password_xpath (str): The XPath of the password field
            submit_xpath (str): The XPath of the submit button

        Returns:
        exploit_builder.ExploitBuilder"""

        (
            self.get(endpoint)
                .type_entry(username_xpath, username)
                .type_entry(password_xpath, password)
                .click(submit_xpath)
        )
        return self

    def send_enter(self, field: str, by: SelectBy = SelectBy.XPATH):
        """send_enter
        Simulates sending the enter key to a particular element.

        Parameters:
            field (str): The selector of the field
            by (SelectBy): The selector type to use (default: SelectBy.XPATH)

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.find_element(by, field).send_keys(Keys.ENTER)
        return self

    def send_enter_by_class(self, field: str):
        """send_enter_by_class
        Simulates sending the enter key to a particular element selected by CSS class name.

        Parameters:
            field (str): The CSS Class name of the field to send enter in

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.find_element_by_class_name(field).send_keys(Keys.ENTER)
        return self

    def send_enter_by_id(self, field: str):
        """send_enter_by_id
        Simulates sending the enter key to a particular element selected by HTML ID.

        Parameters:
            field (str): The HTML ID of the field to send enter in

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.find_element_by_id(field).send_keys(Keys.ENTER)
        return self

    def send_enter_by_xpath(self, xpath: str):
        """send_enter_by_xpath
        Simulates sending the enter key to a particular element selected by XPath.

        Parameters:
            field (str): The XPath of the field to send enter in

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.find_element(By.XPATH, xpath).send_keys(Keys.ENTER)
        return self

    def set_cookie(self, name: str, value: str, path: str = "/", secure: bool = False):
        """set_cookie
        Set a cookie in the browser.

        Parameters:
            name (str): The name of the cookie
            value (str): The cookie's value
            path (str): The path to which the cookie belongs
            secure (bool): Whether or not the cookie has the Secure attribute

        Returns:
        exploit_builder.ExploitBuilder"""

        self.driver.add_cookie({
            "name": name,
            "value": value,
            "path": path,
            "secure": secure
        })
        return self

    def type_entry(self, field: str, entry: str, by: SelectBy = SelectBy.XPATH):
        """type_entry
        Send text to a particular field.

        Parameters:
            field (str): The selector of the field
            entry (str): The text to enter into the field
            by (SelectBy): The selector type to use (default: SelectBy.XPATH)

        Returns:
        exploit_builder.ExploitBuilder"""
        self.wait_for_stealth(entry)
        element = self.driver.find_element(by, field)
        element.send_keys(entry)
        return self

    def type_by_class(self, field: str, entry: str):
        """type_by_class
        Send text to a particular field selected by CSS class name.

        Parameters:
            field (str): The CSS class name of the field
            entry (str): The text to enter into the field

        Returns:
        exploit_builder.ExploitBuilder"""

        self.wait_for_stealth(entry)
        element = self.driver.find_element_by_class_name(field)
        element.send_keys(entry)
        return self

    def type_by_id(self, field: str, entry: str):
        """type_by_id
        Send text to a particular field selected by HTML ID.

        Parameters:
            field (str): The HTML ID of the field
            entry (str): The text to enter into the field

        Returns:
        exploit_builder.ExploitBuilder"""

        self.wait_for_stealth(entry)
        element = self.driver.find_element_by_id(field)
        element.send_keys(entry)
        return self

    def type_by_xpath(self, field: str, entry: str):
        """type_by_xpath
        Send text to a particular field selected by XPath.

        Parameters:
            field (str): The XPath of the field
            entry (str): The text to enter into the field

        Returns:
        exploit_builder.ExploitBuilder"""

        self.wait_for_stealth(entry)
        element = self.driver.find_element(By.XPATH, field)
        element.send_keys(entry)
        return self

    def wait_for_stealth(self, entry: str = ""):
        """wait_for_stealth
        Waits for a period of time based on the length of the entry.

        Parameters:
            entry (str): The entry to scale by

        Returns:
        exploit_builder.ExploitBuilder"""

        if self.stealth:
            time.sleep(len(entry) / 5 + random.uniform(0.2, 1))
        return self

    def wait_for_alert(self, timeout: int = 3, marker: Optional[str] = None) -> bool:
        """wait_for_alert
        Waits for an alert to fire, looking for an optional marker.

        Parameters:
            timeout (int): The amount of time to wait for an alert
            marker (Optional[str]): A known marker, used to detect XSS attacks

        Returns:
        bool: whether or not the alert fired"""
        try:
            WebDriverWait(self.driver, timeout).until(
                EC.alert_is_present(), "Timeout waiting for alert.")
            alert = self.driver.switch_to.alert
            alert_text = alert.text
            alert.accept()
            if marker is not None and len(marker) > 0:
                return marker in alert_text
            return True
        except TimeoutException:
            return False

    def _format(self, endpoint: str, protocol: str = None, hostname: str = None) -> str:
        """_format
        Formats an endpoint into a usable URL with the exploit builder.

        Parameters:
            endpoint (str): The endpoint to format.
            protocol (str): The protocol to format (defaults to the protocol in the ExploitBuilder instance)
            hostname (str): The hostname to format (defaults to the hostname in the ExploitBuilder instance)

        Returns:
        str: The formatted URL"""
        if not protocol:
            protocol = self.protocol
        if not hostname:
            hostname = self.hostname
        return f"{protocol}://{hostname}{endpoint}"

    def _merge_cookies(self, **kwargs) -> Dict[str, str]:
        """_merge_cookies
        Helper function used to merge ExploitBuilder instance cookies with supplied cookies.

        Parameters:
            kwargs
        
        Returns:
        Dict[str, str]: The final list of cookies, ready for use in requests
        """
        tmp_cookies = self.get_cookies()
        arg_cookies = kwargs.get("cookies", {})
        cookies = {}
        for cookie in tmp_cookies:
            cookies[cookie["name"]] = cookie["value"]
        for cookie in arg_cookies:
            cookies[cookie] = arg_cookies[cookie]
        return cookies
