import collections
import itertools
import numbers
import warnings

import pandas as pd

from scmcallib.distributions import Distribution, Uniform
from scmcallib.utils import lhs


def _ensure_tuple(s):
    if isinstance(s, (str, numbers.Number)):
        return (s,)
    return tuple(s)


class ParameterSet(object):
    """Hold all the data used to describe how the emulation or calibration is constrained.

    Tune parameters
        These parameters are updated to find an optimal set/distribution of parameters

    Config parameters
        Other parameters which are passed to the SCM. Typically these parameters are used to set the models into a specific
        reproducible configuration.

    Another feature of the ParameterSet is that each run get's a globally unique run number (for the current python
    execution).

    Attributes
    ----------
    state: dict
        The current state. The key-value pairs of state is passed as arguments to any callable config parameters when they are evaluated

    """

    run_counter = itertools.count()

    def __init__(self):
        self.tune_parameters = collections.OrderedDict()
        self._config_parameters = collections.OrderedDict()
        self.state = {}

    def __repr__(self):  # pragma: no cover
        s = "Config Parameters:\n"
        for p in self._config_parameters:
            s += "{} = {}\n".format(p, self._config_parameters[p])
        s += "Tuning Parameters:\n"
        for p in self.tune_parameters:
            s += "{} = {}\n".format(p, self.tune_parameters[p])
        return s

    def set_tune(self, name, distribution, x0=None):
        """Define a tune parameter

        These are also know as priors, the prior knowledge of the parameter space. These parameters are allowed to change
        when calibrating or emulating.

        When a parameter set is used to emulate another model, the distribution used may not influence the find
        parameters found depending on how the model is setup.

        Parameters
        ----------
        name :
            Name of the parameter. This is the name which is passed to the SCM instance.
        distribution :
            An instance of a subclass of `scmcallib.distributions.Distribution`. This variable defines
            how the parameter can change which emulating or calibration.
        """
        # TODO: handle x0
        if not isinstance(distribution, Distribution):
            raise ValueError(
                "distributions must inherit scmcallib.distributions.Distribution"
            )

        if name in self.config_parameters:
            raise ValueError("{} is already defined as a config parameter")

        self.tune_parameters[name] = distribution

    def set_config(self, name, value):
        """Define a config parameter

        Config parameters are used to set the model into a specific setup for finding point estimates/distributions. They remain
        (mostly) fixed throughout the model runs. These are typically scalars or strings, but functions can also be passed. These
        functions are evaluated on calls to the `config_parameters` property and are passed the contents of `state`.

        Parameters
        ----------
        name :
            Name of the config parameter. This is the name which is used when the value is passed to the initialiser
            of a SCM instance.
        value :
            string or callable. The callables are passed the contents of `ParameterSet.state` as kwargs when they are
            evaluated in `config_parameters`.
        """
        if name in self.tune_parameters:
            raise ValueError("{} is already defined as a config parameter")
        self._config_parameters[name] = value

    def evaluate(self, count, include_config=False, method="random"):
        """Evaluate the parameters

        Parameters
        ----------
        count :
            Number of parameter sets to sample
        include_config :
            If True include the fixed parameters in the parameter dataframe (Default value = False)
        method: str
            Method used for generating samples.

        Returns
        -------
        pd.DataFrame
            DataFrame which contains a set of parameters generated by drawing from the parameter sets
            specified.

        """
        if method == "random":
            params = self._evaluate_random(count)
        elif method == "lhs":
            params = self._evaluate_lhs(count)
        else:
            raise ValueError("Unknown sampling method")

        if include_config:
            params.update(self.config_parameters)

        return pd.DataFrame(
            params, index=[next(self.run_counter) for _ in range(count)]
        )

    def _evaluate_random(self, count):
        return collections.OrderedDict(
            {k: self.tune_parameters[k].evaluate(count) for k in self.tune_parameters}
        )

    def _evaluate_lhs(self, count):
        lhs_samples = lhs(len(self.tune_parameters), n_samples=count)

        for i, k in enumerate(self.tune_parameters):
            dist = self.tune_parameters[k]

            if isinstance(dist, Uniform):
                lhs_samples[:, i] = dist.lower + lhs_samples[:, i] * (dist.upper - dist.lower)
            else:
                raise ValueError("Only currently implemented for Uniform distributions")
        return collections.OrderedDict(
            {k: self.tune_parameters[k].evaluate(count) for k in self.tune_parameters}
        )

    @property
    def names(self):
        return self.tune_parameters.keys()

    def _get_config_parameters(self, key=None, include_model_specific=True, include_common=True):
        res = collections.OrderedDict()

        key = key if key is not None else self.state

        for k in self._config_parameters:
            v = self._config_parameters[k]
            if callable(v) and include_model_specific:
                try:
                    v = v(**key)
                except TypeError:
                    warnings.warn(
                        "could not evaluate config parameter `{}`. Check `ParameterSet.state`".format(
                            k
                        )
                    )
                    continue
            else:
                if not include_common:
                    continue
            res[k] = v
        return res

    @property
    def config_parameters(self):
        """
        Evaluate all of the config parameters

        Returns
        -------
        collections.OrderedDict
            If the config parameters are functions they are also evaluated. Any functions that cannot be evaluated then
            they are excluded from the result and a warning is printed.
        """
        return self._get_config_parameters(include_common=True, include_model_specific=True)

    @property
    def model_specific_parameters(self):
        """
        Evaluate the parameters which are model specific

        Returns
        -------
        collections.OrderedDict
            If the config parameters are functions they are also evaluated. Any functions that cannot be evaulated then
            they are excluded from the result and a warning is printed.
        """
        return self._get_config_parameters(include_common=False, include_model_specific=True)


class KeyedParameterSet(ParameterSet):
    """
    ParameterSet where the items being iterated over are already known

    set_config can also handle dicts which are keyed by the values

    This parameter set should be used in preference to ParameterSet if state is changing. That functionality will be removed
    from ParameterSet in future.

    Parameters
    ----------
    items : dict of list
        List of values for each dimension of the `KeyedParameterSet`

    """

    def __init__(self, items):
        super().__init__()
        self._items = items
        self.state = {k: items[k][0] for k in items}
        self._item_keys = sorted(self._items.keys())
        self._iter = None

    def __repr__(self):  # pragma: no cover
        s = "{} state={}\nConfig Parameters:\n".format(self.__class__, self.state)
        df = pd.Series(self._get_config_parameters())
        s += str(df)
        s += "\nTuning Parameters:\n"
        for p in self.tune_parameters:
            s += "{} = {}\n".format(p, self.tune_parameters[p])
        return s

    def __iter__(self):
        self._iter = itertools.product(*[self._items[k] for k in self._item_keys])
        return self

    def __next__(self):
        k = next(self._iter)
        self.state = {k: v for k, v in zip(self._item_keys, k)}
        return self.config_parameters

    def set_config(self, name, value, dims=None):
        """Define a config parameter

        Config parameters are used to set the model into a specific setup for finding point estimates/distributions. They remain
        (mostly) fixed throughout the model runs. These are typically scalars or strings, but functions can also be passed. These
        functions are evaluated on calls to the `config_parameters` property and are passed the contents of `state`.

        Parameters
        ----------
        name :
            Name of the config parameter. This is the name which is used when the value is passed to the initialiser
            of a SCM instance.
        value : Scalar or dict of scalar
            The value of the config parameter.

            This can have multiple forms, a scalar, a callable function which takes the the current state or a dictionary.

        dims : tuple of str or None
            Dimensions over which the parameter changes

            If this is provided the value parameter should be a dict with tuple keys. These tuples are values for each
            dimension (in the same order as specified by `dims`).
        """
        if dims:
            # Ensure keys are always tuples
            value_tuples = {_ensure_tuple(k): value[k] for k in value}

            def _lookup_value(**kwargs):
                key = tuple(kwargs.get(k, None) for k in dims)
                try:
                    return value_tuples[key]
                except KeyError:
                    warnings.warn("No parameter for key: {}".format(key))
                    return None

            super().set_config(name, _lookup_value)
        else:
            super().set_config(name, value)

    def to_df(self):
        return pd.DataFrame(
            [k for k in self],
            index=pd.MultiIndex.from_product([self._items[k] for k in self._item_keys], names=self._item_keys)
        )


class SingleKeyedParameterSet(KeyedParameterSet):
    key = None

    def __init__(self, items):
        assert self.key is not None
        super().__init__({
            self.key: items
        })

    def set_config(self, name, value, dims=None):
        if isinstance(value, dict):
            dims = (self.key,)
        super().set_config(name, value, dims)


class ScenarioParameterSet(SingleKeyedParameterSet):
    key = "scenario"


class ModelParameterSet(SingleKeyedParameterSet):
    key = "model"
