"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateMachineCustomResourceProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
/**
 * A state machine custom resource provider
 */
class StateMachineCustomResourceProvider extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const cfnResponseSuccessFn = this.createCfnResponseFn('Success');
        const cfnResponseFailedFn = this.createCfnResponseFn('Failed');
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
        });
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [cfnResponseSuccessFn.functionArn, cfnResponseFailedFn.functionArn],
        }));
        // https://docs.aws.amazon.com/step-functions/latest/dg/stepfunctions-iam.html
        // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:StartExecution'],
            resources: [props.stateMachine.stateMachineArn],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:DescribeExecution', 'states:StopExecution'],
            resources: [aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'states',
                    resource: 'execution',
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                    resourceName: `${aws_cdk_lib_1.Stack.of(this).splitArn(props.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                })],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            resources: [aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'events',
                    resource: 'rule',
                    resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                })],
        }));
        const definition = aws_cdk_lib_1.Stack.of(this).toJsonString({
            StartAt: 'StartExecution',
            States: {
                StartExecution: {
                    Type: 'Task',
                    Resource: 'arn:aws:states:::states:startExecution.sync:2',
                    Parameters: {
                        'Input.$': '$',
                        'StateMachineArn': props.stateMachine.stateMachineArn,
                    },
                    TimeoutSeconds: (props.timeout ?? aws_cdk_lib_1.Duration.minutes(30)).toSeconds(),
                    Next: 'CfnResponseSuccess',
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'CfnResponseFailed',
                        }],
                },
                CfnResponseSuccess: {
                    Type: 'Task',
                    Resource: cfnResponseSuccessFn.functionArn,
                    End: true,
                },
                CfnResponseFailed: {
                    Type: 'Task',
                    Resource: cfnResponseFailedFn.functionArn,
                    End: true,
                },
            },
        });
        const stateMachine = new aws_cdk_lib_1.CfnResource(this, 'StateMachine', {
            type: 'AWS::StepFunctions::StateMachine',
            properties: {
                DefinitionString: definition,
                RoleArn: role.roleArn,
            },
        });
        stateMachine.node.addDependency(role);
        const startExecution = new lambda.Function(this, 'StartExecution', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            handler: 'index.startExecution',
            runtime: new lambda.Runtime('nodejs16.x', lambda.RuntimeFamily.NODEJS),
        });
        startExecution.addToRolePolicy(new iam.PolicyStatement({
            actions: ['states:StartExecution'],
            resources: [stateMachine.ref],
        }));
        startExecution.addEnvironment('STATE_MACHINE_ARN', stateMachine.ref);
        this.serviceToken = startExecution.functionArn;
    }
    createCfnResponseFn(status) {
        return new lambda.Function(this, `CfnResponse${status}`, {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            handler: `index.cfnResponse${status}`,
            runtime: new lambda.Runtime('nodejs16.x', lambda.RuntimeFamily.NODEJS),
        });
    }
}
exports.StateMachineCustomResourceProvider = StateMachineCustomResourceProvider;
_a = JSII_RTTI_SYMBOL_1;
StateMachineCustomResourceProvider[_a] = { fqn: "cloudstructs.StateMachineCustomResourceProvider", version: "0.6.27" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvc3RhdGUtbWFjaGluZS1jci1wcm92aWRlci9pbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZCQUE2QjtBQUM3Qiw2Q0FBc0U7QUFDdEUsMkNBQTJDO0FBQzNDLGlEQUFpRDtBQUNqRCwyQ0FBdUM7QUE2QnZDOztHQUVHO0FBQ0gsTUFBYSxrQ0FBbUMsU0FBUSxzQkFBUztJQU0vRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQThDO1FBQ3RGLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxvQkFBb0IsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDakUsTUFBTSxtQkFBbUIsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFL0QsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDdEMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1NBQzVELENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3ZDLE9BQU8sRUFBRSxDQUFDLHVCQUF1QixDQUFDO1lBQ2xDLFNBQVMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLFdBQVcsRUFBRSxtQkFBbUIsQ0FBQyxXQUFXLENBQUM7U0FDL0UsQ0FBQyxDQUFDLENBQUM7UUFDSiw4RUFBOEU7UUFDOUUsOEhBQThIO1FBQzlILElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3ZDLE9BQU8sRUFBRSxDQUFDLHVCQUF1QixDQUFDO1lBQ2xDLFNBQVMsRUFBRSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsZUFBZSxDQUFDO1NBQ2hELENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdkMsT0FBTyxFQUFFLENBQUMsMEJBQTBCLEVBQUUsc0JBQXNCLENBQUM7WUFDN0QsU0FBUyxFQUFFLENBQUMsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO29CQUNuQyxPQUFPLEVBQUUsUUFBUTtvQkFDakIsUUFBUSxFQUFFLFdBQVc7b0JBQ3JCLFNBQVMsRUFBRSx1QkFBUyxDQUFDLG1CQUFtQjtvQkFDeEMsWUFBWSxFQUFFLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsZUFBZSxFQUFFLHVCQUFTLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxZQUFZLEdBQUc7aUJBQzVILENBQUMsQ0FBQztTQUNKLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdkMsT0FBTyxFQUFFLENBQUMsbUJBQW1CLEVBQUUsZ0JBQWdCLEVBQUUscUJBQXFCLENBQUM7WUFDdkUsU0FBUyxFQUFFLENBQUMsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO29CQUNuQyxPQUFPLEVBQUUsUUFBUTtvQkFDakIsUUFBUSxFQUFFLE1BQU07b0JBQ2hCLFlBQVksRUFBRSxxREFBcUQ7aUJBQ3BFLENBQUMsQ0FBQztTQUNKLENBQUMsQ0FBQyxDQUFDO1FBRUosTUFBTSxVQUFVLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsWUFBWSxDQUFDO1lBQzdDLE9BQU8sRUFBRSxnQkFBZ0I7WUFDekIsTUFBTSxFQUFFO2dCQUNOLGNBQWMsRUFBRTtvQkFDZCxJQUFJLEVBQUUsTUFBTTtvQkFDWixRQUFRLEVBQUUsK0NBQStDO29CQUN6RCxVQUFVLEVBQUU7d0JBQ1YsU0FBUyxFQUFFLEdBQUc7d0JBQ2QsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxlQUFlO3FCQUN0RDtvQkFDRCxjQUFjLEVBQUUsQ0FBQyxLQUFLLENBQUMsT0FBTyxJQUFJLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxFQUFFO29CQUNuRSxJQUFJLEVBQUUsb0JBQW9CO29CQUMxQixLQUFLLEVBQUUsQ0FBQzs0QkFDTixXQUFXLEVBQUUsQ0FBQyxZQUFZLENBQUM7NEJBQzNCLElBQUksRUFBRSxtQkFBbUI7eUJBQzFCLENBQUM7aUJBQ0g7Z0JBQ0Qsa0JBQWtCLEVBQUU7b0JBQ2xCLElBQUksRUFBRSxNQUFNO29CQUNaLFFBQVEsRUFBRSxvQkFBb0IsQ0FBQyxXQUFXO29CQUMxQyxHQUFHLEVBQUUsSUFBSTtpQkFDVjtnQkFDRCxpQkFBaUIsRUFBRTtvQkFDakIsSUFBSSxFQUFFLE1BQU07b0JBQ1osUUFBUSxFQUFFLG1CQUFtQixDQUFDLFdBQVc7b0JBQ3pDLEdBQUcsRUFBRSxJQUFJO2lCQUNWO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLFlBQVksR0FBRyxJQUFJLHlCQUFXLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUN6RCxJQUFJLEVBQUUsa0NBQWtDO1lBQ3hDLFVBQVUsRUFBRTtnQkFDVixnQkFBZ0IsRUFBRSxVQUFVO2dCQUM1QixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87YUFDdEI7U0FDRixDQUFDLENBQUM7UUFDSCxZQUFZLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV0QyxNQUFNLGNBQWMsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO1lBQ2pFLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUMsQ0FBQztZQUM1RCxPQUFPLEVBQUUsc0JBQXNCO1lBQy9CLE9BQU8sRUFBRSxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE1BQU0sQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDO1NBQ3ZFLENBQUMsQ0FBQztRQUNILGNBQWMsQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3JELE9BQU8sRUFBRSxDQUFDLHVCQUF1QixDQUFDO1lBQ2xDLFNBQVMsRUFBRSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUM7U0FDOUIsQ0FBQyxDQUFDLENBQUM7UUFDSixjQUFjLENBQUMsY0FBYyxDQUFDLG1CQUFtQixFQUFFLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVyRSxJQUFJLENBQUMsWUFBWSxHQUFHLGNBQWMsQ0FBQyxXQUFXLENBQUM7SUFDakQsQ0FBQztJQUVPLG1CQUFtQixDQUFDLE1BQWM7UUFDeEMsT0FBTyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGNBQWMsTUFBTSxFQUFFLEVBQUU7WUFDdkQsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBQzVELE9BQU8sRUFBRSxvQkFBb0IsTUFBTSxFQUFFO1lBQ3JDLE9BQU8sRUFBRSxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE1BQU0sQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDO1NBQ3ZFLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBdEdILGdGQXVHQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBBcm5Gb3JtYXQsIENmblJlc291cmNlLCBEdXJhdGlvbiwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuLyoqXG4gKiBBIFN0YXRlIE1hY2hpbmVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJU3RhdGVNYWNoaW5lIHtcbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhlIHN0YXRlIG1hY2hpbmVcbiAgICovXG4gIHJlYWRvbmx5IHN0YXRlTWFjaGluZUFybjogc3RyaW5nO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgU3RhdGVNYWNoaW5lQ3VzdG9tUmVzb3VyY2VQcm92aWRlclxuICovXG5leHBvcnQgaW50ZXJmYWNlIFN0YXRlTWFjaGluZUN1c3RvbVJlc291cmNlUHJvdmlkZXJQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgc3RhdGUgbWFjaGluZVxuICAgKi9cbiAgcmVhZG9ubHkgc3RhdGVNYWNoaW5lOiBJU3RhdGVNYWNoaW5lO1xuXG4gIC8qKlxuICAgKiBUaW1lb3V0XG4gICAqXG4gICAqIEBkZWZhdWx0IER1cmF0aW9uLm1pbnV0ZXMoMzApXG4gICAqL1xuICByZWFkb25seSB0aW1lb3V0PzogRHVyYXRpb247XG59XG5cbi8qKlxuICogQSBzdGF0ZSBtYWNoaW5lIGN1c3RvbSByZXNvdXJjZSBwcm92aWRlclxuICovXG5leHBvcnQgY2xhc3MgU3RhdGVNYWNoaW5lQ3VzdG9tUmVzb3VyY2VQcm92aWRlciBleHRlbmRzIENvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBUaGUgc2VydmljZSB0b2tlblxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHNlcnZpY2VUb2tlbjogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTdGF0ZU1hY2hpbmVDdXN0b21SZXNvdXJjZVByb3ZpZGVyUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3QgY2ZuUmVzcG9uc2VTdWNjZXNzRm4gPSB0aGlzLmNyZWF0ZUNmblJlc3BvbnNlRm4oJ1N1Y2Nlc3MnKTtcbiAgICBjb25zdCBjZm5SZXNwb25zZUZhaWxlZEZuID0gdGhpcy5jcmVhdGVDZm5SZXNwb25zZUZuKCdGYWlsZWQnKTtcblxuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnc3RhdGVzLmFtYXpvbmF3cy5jb20nKSxcbiAgICB9KTtcbiAgICByb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnbGFtYmRhOkludm9rZUZ1bmN0aW9uJ10sXG4gICAgICByZXNvdXJjZXM6IFtjZm5SZXNwb25zZVN1Y2Nlc3NGbi5mdW5jdGlvbkFybiwgY2ZuUmVzcG9uc2VGYWlsZWRGbi5mdW5jdGlvbkFybl0sXG4gICAgfSkpO1xuICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdGVwLWZ1bmN0aW9ucy9sYXRlc3QvZGcvc3RlcGZ1bmN0aW9ucy1pYW0uaHRtbFxuICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdGVwLWZ1bmN0aW9ucy9sYXRlc3QvZGcvY29uY2VwdC1jcmVhdGUtaWFtLWFkdmFuY2VkLmh0bWwjY29uY2VwdC1jcmVhdGUtaWFtLWFkdmFuY2VkLWV4ZWN1dGlvblxuICAgIHJvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWydzdGF0ZXM6U3RhcnRFeGVjdXRpb24nXSxcbiAgICAgIHJlc291cmNlczogW3Byb3BzLnN0YXRlTWFjaGluZS5zdGF0ZU1hY2hpbmVBcm5dLFxuICAgIH0pKTtcbiAgICByb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnc3RhdGVzOkRlc2NyaWJlRXhlY3V0aW9uJywgJ3N0YXRlczpTdG9wRXhlY3V0aW9uJ10sXG4gICAgICByZXNvdXJjZXM6IFtTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICBzZXJ2aWNlOiAnc3RhdGVzJyxcbiAgICAgICAgcmVzb3VyY2U6ICdleGVjdXRpb24nLFxuICAgICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FLFxuICAgICAgICByZXNvdXJjZU5hbWU6IGAke1N0YWNrLm9mKHRoaXMpLnNwbGl0QXJuKHByb3BzLnN0YXRlTWFjaGluZS5zdGF0ZU1hY2hpbmVBcm4sIEFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FKS5yZXNvdXJjZU5hbWV9KmAsXG4gICAgICB9KV0sXG4gICAgfSkpO1xuICAgIHJvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWydldmVudHM6UHV0VGFyZ2V0cycsICdldmVudHM6UHV0UnVsZScsICdldmVudHM6RGVzY3JpYmVSdWxlJ10sXG4gICAgICByZXNvdXJjZXM6IFtTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgICBzZXJ2aWNlOiAnZXZlbnRzJyxcbiAgICAgICAgcmVzb3VyY2U6ICdydWxlJyxcbiAgICAgICAgcmVzb3VyY2VOYW1lOiAnU3RlcEZ1bmN0aW9uc0dldEV2ZW50c0ZvclN0ZXBGdW5jdGlvbnNFeGVjdXRpb25SdWxlJyxcbiAgICAgIH0pXSxcbiAgICB9KSk7XG5cbiAgICBjb25zdCBkZWZpbml0aW9uID0gU3RhY2sub2YodGhpcykudG9Kc29uU3RyaW5nKHtcbiAgICAgIFN0YXJ0QXQ6ICdTdGFydEV4ZWN1dGlvbicsXG4gICAgICBTdGF0ZXM6IHtcbiAgICAgICAgU3RhcnRFeGVjdXRpb246IHtcbiAgICAgICAgICBUeXBlOiAnVGFzaycsXG4gICAgICAgICAgUmVzb3VyY2U6ICdhcm46YXdzOnN0YXRlczo6OnN0YXRlczpzdGFydEV4ZWN1dGlvbi5zeW5jOjInLCAvLyB3aXRoIHN5bmM6MiB0aGUgT3V0cHV0IGlzIEpTT04gcGFyc2VkXG4gICAgICAgICAgUGFyYW1ldGVyczoge1xuICAgICAgICAgICAgJ0lucHV0LiQnOiAnJCcsXG4gICAgICAgICAgICAnU3RhdGVNYWNoaW5lQXJuJzogcHJvcHMuc3RhdGVNYWNoaW5lLnN0YXRlTWFjaGluZUFybixcbiAgICAgICAgICB9LFxuICAgICAgICAgIFRpbWVvdXRTZWNvbmRzOiAocHJvcHMudGltZW91dCA/PyBEdXJhdGlvbi5taW51dGVzKDMwKSkudG9TZWNvbmRzKCksXG4gICAgICAgICAgTmV4dDogJ0NmblJlc3BvbnNlU3VjY2VzcycsXG4gICAgICAgICAgQ2F0Y2g6IFt7XG4gICAgICAgICAgICBFcnJvckVxdWFsczogWydTdGF0ZXMuQUxMJ10sXG4gICAgICAgICAgICBOZXh0OiAnQ2ZuUmVzcG9uc2VGYWlsZWQnLFxuICAgICAgICAgIH1dLFxuICAgICAgICB9LFxuICAgICAgICBDZm5SZXNwb25zZVN1Y2Nlc3M6IHtcbiAgICAgICAgICBUeXBlOiAnVGFzaycsXG4gICAgICAgICAgUmVzb3VyY2U6IGNmblJlc3BvbnNlU3VjY2Vzc0ZuLmZ1bmN0aW9uQXJuLFxuICAgICAgICAgIEVuZDogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgICAgQ2ZuUmVzcG9uc2VGYWlsZWQ6IHtcbiAgICAgICAgICBUeXBlOiAnVGFzaycsXG4gICAgICAgICAgUmVzb3VyY2U6IGNmblJlc3BvbnNlRmFpbGVkRm4uZnVuY3Rpb25Bcm4sXG4gICAgICAgICAgRW5kOiB0cnVlLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHN0YXRlTWFjaGluZSA9IG5ldyBDZm5SZXNvdXJjZSh0aGlzLCAnU3RhdGVNYWNoaW5lJywge1xuICAgICAgdHlwZTogJ0FXUzo6U3RlcEZ1bmN0aW9uczo6U3RhdGVNYWNoaW5lJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgRGVmaW5pdGlvblN0cmluZzogZGVmaW5pdGlvbixcbiAgICAgICAgUm9sZUFybjogcm9sZS5yb2xlQXJuLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBzdGF0ZU1hY2hpbmUubm9kZS5hZGREZXBlbmRlbmN5KHJvbGUpO1xuXG4gICAgY29uc3Qgc3RhcnRFeGVjdXRpb24gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdTdGFydEV4ZWN1dGlvbicsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAncnVudGltZScpKSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5zdGFydEV4ZWN1dGlvbicsXG4gICAgICBydW50aW1lOiBuZXcgbGFtYmRhLlJ1bnRpbWUoJ25vZGVqczE2LngnLCBsYW1iZGEuUnVudGltZUZhbWlseS5OT0RFSlMpLFxuICAgIH0pO1xuICAgIHN0YXJ0RXhlY3V0aW9uLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbJ3N0YXRlczpTdGFydEV4ZWN1dGlvbiddLFxuICAgICAgcmVzb3VyY2VzOiBbc3RhdGVNYWNoaW5lLnJlZl0sXG4gICAgfSkpO1xuICAgIHN0YXJ0RXhlY3V0aW9uLmFkZEVudmlyb25tZW50KCdTVEFURV9NQUNISU5FX0FSTicsIHN0YXRlTWFjaGluZS5yZWYpO1xuXG4gICAgdGhpcy5zZXJ2aWNlVG9rZW4gPSBzdGFydEV4ZWN1dGlvbi5mdW5jdGlvbkFybjtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlQ2ZuUmVzcG9uc2VGbihzdGF0dXM6IHN0cmluZykge1xuICAgIHJldHVybiBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsIGBDZm5SZXNwb25zZSR7c3RhdHVzfWAsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAncnVudGltZScpKSxcbiAgICAgIGhhbmRsZXI6IGBpbmRleC5jZm5SZXNwb25zZSR7c3RhdHVzfWAsXG4gICAgICBydW50aW1lOiBuZXcgbGFtYmRhLlJ1bnRpbWUoJ25vZGVqczE2LngnLCBsYW1iZGEuUnVudGltZUZhbWlseS5OT0RFSlMpLFxuICAgIH0pO1xuICB9XG59XG4iXX0=