"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'When creating a VPC': {
        'with the default CIDR range': {
            'vpc.vpcId returns a token to the VPC ID'(test) {
                const stack = getTestStack();
                const vpc = new lib_1.Vpc(stack, 'TheVPC');
                test.deepEqual(stack.resolve(vpc.vpcId), { Ref: 'TheVPC92636AB0' });
                test.done();
            },
            'it uses the correct network range'(test) {
                const stack = getTestStack();
                new lib_1.Vpc(stack, 'TheVPC');
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                    CidrBlock: lib_1.Vpc.DEFAULT_CIDR_RANGE,
                    EnableDnsHostnames: true,
                    EnableDnsSupport: true,
                    InstanceTenancy: lib_1.DefaultInstanceTenancy.DEFAULT,
                }));
                test.done();
            },
            'the Name tag is defaulted to path'(test) {
                const stack = getTestStack();
                new lib_1.Vpc(stack, 'TheVPC');
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags([{ Key: 'Name', Value: 'TheVPC' }])));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::InternetGateway', hasTags([{ Key: 'Name', Value: 'TheVPC' }])));
                test.done();
            },
        },
        'with all of the properties set, it successfully sets the correct VPC properties'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '192.168.0.0/16',
                enableDnsHostnames: false,
                enableDnsSupport: false,
                defaultInstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                CidrBlock: '192.168.0.0/16',
                EnableDnsHostnames: false,
                EnableDnsSupport: false,
                InstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
            }));
            test.done();
        },
        'contains the correct number of subnets'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            const zones = stack.availabilityZones.length;
            test.equal(vpc.publicSubnets.length, zones);
            test.equal(vpc.privateSubnets.length, zones);
            test.deepEqual(stack.resolve(vpc.vpcId), { Ref: 'TheVPC92636AB0' });
            test.done();
        },
        'with only isolated subnets, the VPC should not contain an IGW or NAT Gateways'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'Isolated',
                    },
                ],
            });
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::InternetGateway'));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::NatGateway'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                MapPublicIpOnLaunch: false,
            }));
            test.done();
        },
        'with no private subnets, the VPC should have an IGW but no NAT Gateways'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'Isolated',
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::NatGateway'));
            test.done();
        },
        'with private subnets and custom networkAcl.'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        subnetType: lib_1.SubnetType.PRIVATE,
                        name: 'private',
                    },
                ],
            });
            const nacl1 = new lib_1.NetworkAcl(stack, 'myNACL1', {
                vpc,
                subnetSelection: { subnetType: lib_1.SubnetType.PRIVATE },
            });
            new lib_1.NetworkAclEntry(stack, 'AllowDNSEgress', {
                networkAcl: nacl1,
                ruleNumber: 100,
                traffic: lib_1.AclTraffic.udpPort(53),
                direction: lib_1.TrafficDirection.EGRESS,
                cidr: lib_1.AclCidr.ipv4('10.0.0.0/16'),
            });
            new lib_1.NetworkAclEntry(stack, 'AllowDNSIngress', {
                networkAcl: nacl1,
                ruleNumber: 100,
                traffic: lib_1.AclTraffic.udpPort(53),
                direction: lib_1.TrafficDirection.INGRESS,
                cidr: lib_1.AclCidr.anyIpv4(),
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NetworkAcl', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NetworkAclEntry', 2));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SubnetNetworkAclAssociation', 3));
            test.done();
        },
        'with no subnets defined, the VPC should have an IGW, and a NAT Gateway per AZ'(test) {
            const stack = getTestStack();
            const zones = stack.availabilityZones.length;
            new lib_1.Vpc(stack, 'TheVPC', {});
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', zones));
            test.done();
        },
        'with subnets and reserved subnets defined, VPC subnet count should not contain reserved subnets '(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        cidrMask: 24,
                        name: 'reserved',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    },
                ],
                maxAzs: 3,
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 6));
            test.done();
        },
        'with reserved subnets, any other subnets should not have cidrBlock from within reserved space'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'reserved',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                    {
                        cidrMask: 24,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                maxAzs: 3,
            });
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`,
                }));
            }
            for (let i = 3; i < 6; i++) {
                assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`,
                }));
            }
            for (let i = 6; i < 9; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`,
                }));
            }
            test.done();
        },
        'with custom subnets, the VPC should have the right number of subnets, an IGW, and a NAT Gateway per AZ'(test) {
            const stack = getTestStack();
            const zones = stack.availabilityZones.length;
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/21',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'application',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    },
                ],
                maxAzs: 3,
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', zones));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 9));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`,
                }));
            }
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.6.${i * 16}/28`,
                }));
            }
            test.done();
        },
        'with custom subents and natGateways = 2 there should be only two NATGW'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/21',
                natGateways: 2,
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'application',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    },
                ],
                maxAzs: 3,
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', 2));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 9));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`,
                }));
            }
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.6.${i * 16}/28`,
                }));
            }
            test.done();
        },
        'with enableDnsHostnames enabled but enableDnsSupport disabled, should throw an Error'(test) {
            const stack = getTestStack();
            test.throws(() => new lib_1.Vpc(stack, 'TheVPC', {
                enableDnsHostnames: true,
                enableDnsSupport: false,
            }));
            test.done();
        },
        'with public subnets MapPublicIpOnLaunch is true'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                maxAzs: 1,
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 1));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::NatGateway'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                MapPublicIpOnLaunch: true,
            }));
            test.done();
        },
        'maxAZs defaults to 3 if unset'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC');
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 6));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Route', 6));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i * 32}.0/19`,
                }));
            }
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        'with maxAZs set to 2'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', { maxAzs: 2 });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 4));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Route', 4));
            for (let i = 0; i < 4; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i * 64}.0/18`,
                }));
            }
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        'with natGateway set to 1'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                natGateways: 1,
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 6));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Route', 6));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', 1));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        'with natGateway subnets defined'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'egress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                natGatewaySubnets: {
                    subnetGroupName: 'egress',
                },
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', 3));
            for (let i = 1; i < 4; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', hasTags([{
                        Key: 'Name',
                        Value: `VPC/egressSubnet${i}`,
                    }, {
                        Key: 'aws-cdk:subnet-name',
                        Value: 'egress',
                    }])));
            }
            test.done();
        },
        'natGateways = 0 throws if no PRIVATE subnets configured'(test) {
            const stack = getTestStack();
            test.throws(() => {
                new lib_1.Vpc(stack, 'VPC', {
                    natGateways: 0,
                    subnetConfiguration: [
                        {
                            name: 'public',
                            subnetType: lib_1.SubnetType.PUBLIC,
                        },
                        {
                            name: 'private',
                            subnetType: lib_1.SubnetType.PRIVATE,
                        },
                    ],
                });
            }, /make sure you don't configure any PRIVATE subnets/);
            test.done();
        },
        'natGateway = 0 defaults with ISOLATED subnet'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                natGateways: 0,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', hasTags([{
                    Key: 'aws-cdk:subnet-type',
                    Value: 'Isolated',
                }])));
            test.done();
        },
        'unspecified natGateways constructs with PRIVATE subnet'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC');
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', hasTags([{
                    Key: 'aws-cdk:subnet-type',
                    Value: 'Private',
                }])));
            test.done();
        },
        'natGateways = 0 allows RESERVED PRIVATE subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                ],
                natGateways: 0,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', hasTags([{
                    Key: 'aws-cdk:subnet-name',
                    Value: 'ingress',
                }])));
            test.done();
        },
        'with mis-matched nat and subnet configs it throws'(test) {
            const stack = getTestStack();
            test.throws(() => new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                natGatewaySubnets: {
                    subnetGroupName: 'notthere',
                },
            }));
            test.done();
        },
        'with a vpn gateway'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                vpnGateway: true,
                vpnGatewayAsn: 65000,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGateway', {
                AmazonSideAsn: 65000,
                Type: 'ipsec.1',
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCGatewayAttachment', {
                VpcId: {
                    Ref: 'VPCB9E5F0B4',
                },
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPrivateSubnet1RouteTableBE8A6027',
                    },
                    {
                        Ref: 'VPCPrivateSubnet2RouteTable0A19E10E',
                    },
                    {
                        Ref: 'VPCPrivateSubnet3RouteTable192186F8',
                    },
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            test.done();
        },
        'with a vpn gateway and route propagation on isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
                vpnRoutePropagation: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA',
                    },
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            test.done();
        },
        'with a vpn gateway and route propagation on private and isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'Private' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
                vpnRoutePropagation: [
                    {
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPrivateSubnet1RouteTableBE8A6027',
                    },
                    {
                        Ref: 'VPCPrivateSubnet2RouteTable0A19E10E',
                    },
                    {
                        Ref: 'VPCPrivateSubnet3RouteTable192186F8',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA',
                    },
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            test.done();
        },
        'route propagation defaults to isolated subnets when there are no private subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC',
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA',
                    },
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            test.done();
        },
        'route propagation defaults to public subnets when there are no private/isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                ],
                vpnGateway: true,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPublicSubnet1RouteTableFEE4B781',
                    },
                    {
                        Ref: 'VPCPublicSubnet2RouteTable6F1A15F1',
                    },
                    {
                        Ref: 'VPCPublicSubnet3RouteTable98AE0E14',
                    },
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68',
                },
            }));
            test.done();
        },
        'fails when specifying vpnConnections with vpnGateway set to false'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
                vpnGateway: false,
                vpnConnections: {
                    VpnConnection: {
                        asn: 65000,
                        ip: '192.0.2.1',
                    },
                },
            }), /`vpnConnections`.+`vpnGateway`.+false/);
            test.done();
        },
        'fails when specifying vpnGatewayAsn with vpnGateway set to false'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
                vpnGateway: false,
                vpnGatewayAsn: 65000,
            }), /`vpnGatewayAsn`.+`vpnGateway`.+false/);
            test.done();
        },
        'Subnets have a defaultChild'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            test.ok(vpc.publicSubnets[0].node.defaultChild instanceof lib_1.CfnSubnet);
            test.done();
        },
        'CIDR cannot be a Token'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new lib_1.Vpc(stack, 'Vpc', {
                    cidr: core_1.Lazy.stringValue({ produce: () => 'abc' }),
                });
            }, /property must be a concrete CIDR string/);
            test.done();
        },
        'Default NAT gateway provider'(test) {
            const stack = new core_1.Stack();
            const natGatewayProvider = lib_1.NatProvider.gateway();
            new lib_1.Vpc(stack, 'VpcNetwork', { natGatewayProvider });
            test.ok(natGatewayProvider.configuredGateways.length > 0);
            test.done();
        },
        'Can add an IPv6 route'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'VPC');
            vpc.publicSubnets[0].addRoute('SomeRoute', {
                destinationIpv6CidrBlock: '2001:4860:4860::8888/32',
                routerId: 'router-1',
                routerType: lib_1.RouterType.NETWORK_INTERFACE,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationIpv6CidrBlock: '2001:4860:4860::8888/32',
                NetworkInterfaceId: 'router-1',
            }));
            test.done();
        },
        'Can add an IPv4 route'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'VPC');
            vpc.publicSubnets[0].addRoute('SomeRoute', {
                destinationCidrBlock: '0.0.0.0/0',
                routerId: 'router-1',
                routerType: lib_1.RouterType.NETWORK_INTERFACE,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NetworkInterfaceId: 'router-1',
            }));
            test.done();
        },
    },
    'NAT instances': {
        'Can configure NAT instances instead of NAT gateways'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const natGatewayProvider = lib_1.NatProvider.instance({
                instanceType: new lib_1.InstanceType('q86.mega'),
                machineImage: new lib_1.GenericLinuxImage({
                    'us-east-1': 'ami-1',
                }),
            });
            new lib_1.Vpc(stack, 'TheVPC', { natGatewayProvider });
            // THEN
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Instance', 3));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
                ImageId: 'ami-1',
                InstanceType: 'q86.mega',
                SourceDestCheck: false,
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Route', {
                RouteTableId: { Ref: 'TheVPCPrivateSubnet1RouteTableF6513BC2' },
                DestinationCidrBlock: '0.0.0.0/0',
                InstanceId: { Ref: 'TheVPCPublicSubnet1NatInstanceCC514192' },
            }));
            test.done();
        },
        'natGateways controls amount of NAT instances'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            new lib_1.Vpc(stack, 'TheVPC', {
                natGatewayProvider: lib_1.NatProvider.instance({
                    instanceType: new lib_1.InstanceType('q86.mega'),
                    machineImage: new lib_1.GenericLinuxImage({
                        'us-east-1': 'ami-1',
                    }),
                }),
                natGateways: 1,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Instance', 1));
            test.done();
        },
        'can configure Security Groups of NAT instances'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const provider = lib_1.NatProvider.instance({
                instanceType: new lib_1.InstanceType('q86.mega'),
                machineImage: new lib_1.GenericLinuxImage({
                    'us-east-1': 'ami-1',
                }),
                allowAllTraffic: false,
            });
            new lib_1.Vpc(stack, 'TheVPC', {
                natGatewayProvider: provider,
            });
            provider.connections.allowFrom(lib_1.Peer.ipv4('1.2.3.4/32'), lib_1.Port.tcp(86));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                SecurityGroupIngress: [
                    {
                        CidrIp: '1.2.3.4/32',
                        Description: 'from 1.2.3.4/32:86',
                        FromPort: 86,
                        IpProtocol: 'tcp',
                        ToPort: 86,
                    },
                ],
            }));
            test.done();
        },
    },
    'Network ACL association': {
        'by default uses default ACL reference'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            new core_1.CfnOutput(stack, 'Output', {
                value: vpc.publicSubnets[0].subnetNetworkAclAssociationId,
            });
            assert_1.expect(stack).toMatch({
                Outputs: {
                    Output: {
                        Value: { 'Fn::GetAtt': ['TheVPCPublicSubnet1Subnet770D4FF2', 'NetworkAclAssociationId'] },
                    },
                },
            }, assert_1.MatchStyle.SUPERSET);
            test.done();
        },
        'if ACL is replaced new ACL reference is returned'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            // WHEN
            new core_1.CfnOutput(stack, 'Output', {
                value: vpc.publicSubnets[0].subnetNetworkAclAssociationId,
            });
            new lib_1.NetworkAcl(stack, 'ACL', {
                vpc,
                subnetSelection: { subnetType: lib_1.SubnetType.PUBLIC },
            });
            assert_1.expect(stack).toMatch({
                Outputs: {
                    Output: {
                        Value: { Ref: 'ACLDBD1BB49' },
                    },
                },
            }, assert_1.MatchStyle.SUPERSET);
            test.done();
        },
    },
    'When creating a VPC with a custom CIDR range': {
        'vpc.vpcCidrBlock is the correct network range'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                CidrBlock: '192.168.0.0/16',
            }));
            test.done();
        },
    },
    'When tagging': {
        'VPC propagated tags will be on subnet, IGW, routetables, NATGW'(test) {
            const stack = getTestStack();
            const tags = {
                VpcType: 'Good',
            };
            const noPropTags = {
                BusinessUnit: 'Marketing',
            };
            const allTags = { ...tags, ...noPropTags };
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            // overwrite to set propagate
            vpc.node.applyAspect(new core_1.Tag('BusinessUnit', 'Marketing', { includeResourceTypes: [lib_1.CfnVPC.CFN_RESOURCE_TYPE_NAME] }));
            vpc.node.applyAspect(new core_1.Tag('VpcType', 'Good'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags(toCfnTags(allTags))));
            const taggables = ['Subnet', 'InternetGateway', 'NatGateway', 'RouteTable'];
            const propTags = toCfnTags(tags);
            const noProp = toCfnTags(noPropTags);
            for (const resource of taggables) {
                assert_1.expect(stack).to(assert_1.haveResource(`AWS::EC2::${resource}`, hasTags(propTags)));
                assert_1.expect(stack).notTo(assert_1.haveResource(`AWS::EC2::${resource}`, hasTags(noProp)));
            }
            test.done();
        },
        'Subnet Name will propagate to route tables and NATGW'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            for (const subnet of vpc.publicSubnets) {
                const tag = { Key: 'Name', Value: subnet.node.path };
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::NatGateway', hasTags([tag])));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::RouteTable', hasTags([tag])));
            }
            for (const subnet of vpc.privateSubnets) {
                const tag = { Key: 'Name', Value: subnet.node.path };
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::RouteTable', hasTags([tag])));
            }
            test.done();
        },
        'Tags can be added after the Vpc is created with `vpc.tags.setTag(...)`'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            const tag = { Key: 'Late', Value: 'Adder' };
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::VPC', hasTags([tag])));
            vpc.node.applyAspect(new core_1.Tag(tag.Key, tag.Value));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags([tag])));
            test.done();
        },
    },
    'subnet selection': {
        'selecting default subnets returns the private ones'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            const { subnetIds } = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select public subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetType: lib_1.SubnetType.PUBLIC });
            // THEN
            test.deepEqual(subnetIds, vpc.publicSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select isolated subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetType: lib_1.SubnetType.ISOLATED });
            // THEN
            test.deepEqual(subnetIds, vpc.isolatedSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select subnets by name'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'BlaBla' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'DontTalkToMe' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'DontTalkAtAll' },
                ],
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetGroupName: 'DontTalkToMe' });
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'subnetName is an alias for subnetGroupName (backwards compat)'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'BlaBla' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'DontTalkToMe' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'DontTalkAtAll' },
                ],
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetName: 'DontTalkToMe' });
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'selecting default subnets in a VPC with only isolated subnets returns the isolateds'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                isolatedSubnetIds: ['iso-1', 'iso-2', 'iso-3'],
                isolatedSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            // WHEN
            const subnets = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnets.subnetIds, ['iso-1', 'iso-2', 'iso-3']);
            test.done();
        },
        'selecting default subnets in a VPC with only public subnets returns the publics'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                publicSubnetIds: ['pub-1', 'pub-2', 'pub-3'],
                publicSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            // WHEN
            const subnets = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnets.subnetIds, ['pub-1', 'pub-2', 'pub-3']);
            test.done();
        },
        'selecting subnets by name fails if the name is unknown'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            test.throws(() => {
                vpc.selectSubnets({ subnetGroupName: 'Toot' });
            }, /There are no subnet groups with name 'Toot' in this VPC. Available names: Public,Private/);
            test.done();
        },
        'select subnets with az restriction'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork', {
                maxAzs: 1,
                subnetConfiguration: [
                    { name: 'lb', subnetType: lib_1.SubnetType.PUBLIC },
                    { name: 'app', subnetType: lib_1.SubnetType.PRIVATE },
                    { name: 'db', subnetType: lib_1.SubnetType.PRIVATE },
                ],
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ onePerAz: true });
            // THEN
            test.deepEqual(subnetIds.length, 1);
            test.deepEqual(subnetIds[0], vpc.privateSubnets[0].subnetId);
            test.done();
        },
        'select explicitly defined subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                publicSubnetIds: ['pub-1', 'pub-2', 'pub-3'],
                publicSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            const subnet = new lib_1.PrivateSubnet(stack, 'Subnet', {
                availabilityZone: vpc.availabilityZones[0],
                cidrBlock: '10.0.0.0/28',
                vpcId: vpc.vpcId,
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnets: [subnet] });
            // THEN
            test.deepEqual(subnetIds.length, 1);
            test.deepEqual(subnetIds[0], subnet.subnetId);
            test.done();
        },
        'subnet created from subnetId'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetId(stack, 'subnet1', 'pub-1');
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            test.done();
        },
        'Referencing AZ throws error when subnet created from subnetId'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetId(stack, 'subnet1', 'pub-1');
            // THEN
            // tslint:disable-next-line: max-line-length
            test.throws(() => subnet.availabilityZone, "You cannot reference a Subnet's availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()");
            test.done();
        },
        'Referencing AZ throws error when subnet created from attributes without az'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetAttributes(stack, 'subnet1', { subnetId: 'pub-1', availabilityZone: '' });
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            // tslint:disable-next-line: max-line-length
            test.throws(() => subnet.availabilityZone, "You cannot reference a Subnet's availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()");
            test.done();
        },
        'AZ have value when subnet created from attributes with az'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetAttributes(stack, 'subnet1', { subnetId: 'pub-1', availabilityZone: 'az-1234' });
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            test.deepEqual(subnet.availabilityZone, 'az-1234');
            test.done();
        },
        'Can select subnets by type and AZ'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                maxAzs: 3,
            });
            // WHEN
            new lib_1.InterfaceVpcEndpoint(stack, 'VPC Endpoint', {
                vpc,
                privateDnsEnabled: false,
                service: new lib_1.InterfaceVpcEndpointService('com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc', 443),
                subnets: {
                    subnetType: lib_1.SubnetType.PRIVATE,
                    availabilityZones: ['dummy1a', 'dummy1c'],
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: 'com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc',
                SubnetIds: [
                    {
                        Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                    },
                    {
                        Ref: 'VPCPrivateSubnet3Subnet3EDCD457',
                    },
                ],
            }));
            test.done();
        },
        'SubnetSelection filtered on az uses default subnetType when no subnet type specified'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                maxAzs: 3,
            });
            // WHEN
            new lib_1.InterfaceVpcEndpoint(stack, 'VPC Endpoint', {
                vpc,
                service: new lib_1.InterfaceVpcEndpointService('com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc', 443),
                subnets: {
                    availabilityZones: ['dummy1a', 'dummy1c'],
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: 'com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc',
                SubnetIds: [
                    {
                        Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                    },
                    {
                        Ref: 'VPCPrivateSubnet3Subnet3EDCD457',
                    },
                ],
            }));
            test.done();
        },
    },
});
function getTestStack() {
    return new core_1.Stack(undefined, 'TestStack', { env: { account: '123456789012', region: 'us-east-1' } });
}
function toCfnTags(tags) {
    return Object.keys(tags).map(key => {
        return { Key: key, Value: tags[key] };
    });
}
function hasTags(expectedTags) {
    return (props) => {
        try {
            const tags = props.Tags;
            const actualTags = tags.filter((tag) => {
                for (const expectedTag of expectedTags) {
                    if (assert_1.isSuperObject(expectedTag, tag)) {
                        return true;
                    }
                    else {
                        continue;
                    }
                }
                // no values in array so expecting empty
                return false;
            });
            return actualTags.length === expectedTags.length;
        }
        catch (e) {
            // tslint:disable:no-console
            console.error('Tags are incorrect');
            console.error('found tags ', props.Tags);
            console.error('expected tags ', expectedTags);
            // tslint:enable:no-console
            throw e;
        }
    };
}
//# sourceMappingURL=data:application/json;base64,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