"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RESERVED_TUNNEL_INSIDE_CIDR = exports.VpnConnection = exports.VpnGateway = exports.VpnConnectionType = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
const net = require("net");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The VPN connection type.
 */
var VpnConnectionType;
(function (VpnConnectionType) {
    /**
     * The IPsec 1 VPN connection type.
     */
    VpnConnectionType["IPSEC_1"] = "ipsec.1";
    /**
     * Dummy member
     * TODO: remove once https://github.com/aws/jsii/issues/231 is fixed
     */
    VpnConnectionType["DUMMY"] = "dummy";
})(VpnConnectionType = exports.VpnConnectionType || (exports.VpnConnectionType = {}));
/**
 * The VPN Gateway that shall be added to the VPC
 *
 * @resource AWS::EC2::VPNGateway
 */
class VpnGateway extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        // This is 'Default' instead of 'Resource', because using 'Default' will generate
        // a logical ID for a VpnGateway which is exactly the same as the logical ID that used
        // to be created for the CfnVPNGateway (and 'Resource' would not do that).
        const vpnGW = new ec2_generated_1.CfnVPNGateway(this, 'Default', props);
        this.gatewayId = vpnGW.ref;
    }
}
exports.VpnGateway = VpnGateway;
/**
 * Define a VPN Connection
 *
 * @resource AWS::EC2::VPNConnection
 */
class VpnConnection extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.vpc.vpnGatewayId) {
            props.vpc.enableVpnGateway({
                type: 'ipsec.1',
                amazonSideAsn: props.asn,
            });
        }
        if (!net.isIPv4(props.ip)) {
            throw new Error(`The \`ip\` ${props.ip} is not a valid IPv4 address.`);
        }
        const type = VpnConnectionType.IPSEC_1;
        const bgpAsn = props.asn || 65000;
        const customerGateway = new ec2_generated_1.CfnCustomerGateway(this, 'CustomerGateway', {
            bgpAsn,
            ipAddress: props.ip,
            type,
        });
        this.customerGatewayId = customerGateway.ref;
        this.customerGatewayAsn = bgpAsn;
        this.customerGatewayIp = props.ip;
        // Validate tunnel options
        if (props.tunnelOptions) {
            if (props.tunnelOptions.length > 2) {
                throw new Error('Cannot specify more than two `tunnelOptions`');
            }
            if (props.tunnelOptions.length === 2 && props.tunnelOptions[0].tunnelInsideCidr === props.tunnelOptions[1].tunnelInsideCidr) {
                throw new Error(`Same ${props.tunnelOptions[0].tunnelInsideCidr} \`tunnelInsideCidr\` cannot be used for both tunnels.`);
            }
            props.tunnelOptions.forEach((options, index) => {
                if (options.preSharedKey && !/^[a-zA-Z1-9._][a-zA-Z\d._]{7,63}$/.test(options.preSharedKey)) {
                    // tslint:disable:max-line-length
                    throw new Error(`The \`preSharedKey\` ${options.preSharedKey} for tunnel ${index + 1} is invalid. Allowed characters are alphanumeric characters and ._. Must be between 8 and 64 characters in length and cannot start with zero (0).`);
                    // tslint:enable:max-line-length
                }
                if (options.tunnelInsideCidr) {
                    if (exports.RESERVED_TUNNEL_INSIDE_CIDR.includes(options.tunnelInsideCidr)) {
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is a reserved inside CIDR.`);
                    }
                    if (!/^169\.254\.\d{1,3}\.\d{1,3}\/30$/.test(options.tunnelInsideCidr)) {
                        // tslint:disable:max-line-length
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is not a size /30 CIDR block from the 169.254.0.0/16 range.`);
                        // tslint:enable:max-line-length
                    }
                }
            });
        }
        const vpnConnection = new ec2_generated_1.CfnVPNConnection(this, 'Resource', {
            type,
            customerGatewayId: customerGateway.ref,
            staticRoutesOnly: props.staticRoutes ? true : false,
            vpnGatewayId: props.vpc.vpnGatewayId,
            vpnTunnelOptionsSpecifications: props.tunnelOptions,
        });
        this.vpnId = vpnConnection.ref;
        if (props.staticRoutes) {
            props.staticRoutes.forEach(route => {
                new ec2_generated_1.CfnVPNConnectionRoute(this, `Route${route.replace(/[^\d]/g, '')}`, {
                    destinationCidrBlock: route,
                    vpnConnectionId: this.vpnId,
                });
            });
        }
    }
    /**
     * Return the given named metric for all VPN connections in the account/region.
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/VPN',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the tunnel state of all VPN connections in the account/region.
     *
     * @default average over 5 minutes
     */
    static metricAllTunnelState(props) {
        return this.metricAll('TunnelState', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the tunnel data in of all VPN connections in the account/region.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataIn(props) {
        return this.metricAll('TunnelDataIn', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the tunnel data out of all VPN connections.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataOut(props) {
        return this.metricAll('TunnelDataOut', { statistic: 'sum', ...props });
    }
}
exports.VpnConnection = VpnConnection;
exports.RESERVED_TUNNEL_INSIDE_CIDR = [
    '169.254.0.0/30',
    '169.254.1.0/30',
    '169.254.2.0/30',
    '169.254.3.0/30',
    '169.254.4.0/30',
    '169.254.5.0/30',
    '169.254.169.252/30',
];
//# sourceMappingURL=data:application/json;base64,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