# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['flake8_type_checking']

package_data = \
{'': ['*']}

install_requires = \
['flake8']

entry_points = \
{'flake8.extension': ['TC = flake8_type_checking.plugin:Plugin']}

setup_kwargs = {
    'name': 'flake8-type-checking',
    'version': '1.4.0',
    'description': 'A flake8 plugin for managing type-checking imports & forward references',
    'long_description': '[![Package version](https://img.shields.io/pypi/v/flake8-type-checking.svg)](https://pypi.org/project/flake8-type-checking/)\n[![Code coverage](https://codecov.io/gh/sondrelg/flake8-type-checking/branch/main/graph/badge.svg)](https://codecov.io/gh/sondrelg/flake8-type-checking)\n[![Test status](https://github.com/sondrelg/flake8-type-checking/actions/workflows/testing.yml/badge.svg)](https://github.com/snok/flake8-type-checking/actions/workflows/testing.yml)\n[![Supported Python versions](https://img.shields.io/badge/python-3.8%2B-blue)](https://pypi.org/project/flake8-type-checking/)\n[![Checked with mypy](http://www.mypy-lang.org/static/mypy_badge.svg)](http://mypy-lang.org/)\n\n# flake8-type-checking\n\nLets you know which imports to move in or out of\n[type-checking](https://docs.python.org/3/library/typing.html#typing.TYPE_CHECKING) blocks.\n\nThe plugin assumes that the imports you only use for type hinting\n*are not* required at runtime. When imports aren\'t strictly required at runtime, it means we can guard them.\n\nThis provides 3 major benefits:\n\n- 🔧&nbsp;&nbsp;It reduces import circularity issues,\n- 🧹&nbsp;&nbsp;It organizes imports, and\n- 🚀&nbsp;&nbsp;It completely eliminates the overhead of type hint imports at runtime\n\n<br>\n\nEssentially, this code:\n\n```python\nimport pandas  # 15mb library\n\nx: pandas.DataFrame\n```\n\nbecomes this:\n\n```python\nfrom typing import TYPE_CHECKING\n\nif TYPE_CHECKING:\n    import pandas  # <-- no longer imported at runtime\n\nx: "pandas.DataFrame"\n```\n\nMore examples can be found in the [examples](#examples) section.\n\n<br>\n\nIf you\'re using [pydantic](https://pydantic-docs.helpmanual.io/) or [fastapi](https://fastapi.tiangolo.com/),\nsee the [configuration](#configuration) for how to enable support.\n\n## Primary features\n\nThe plugin will:\n\n- Tell you when an import should be moved into a type-checking block\n- Tell you when an import should be moved out again\n\nAnd depending on which error code range you\'ve opted into, it will tell you\n\n- Whether you need to add a `from __future__ import annotations` import\n- Whether you need to quote an annotation\n- Whether you can unquote a quoted annotation\n\n## Error codes\n\n| Code   | Description                                         |\n|--------|-----------------------------------------------------|\n| TC001 | Move import into a type-checking block  |\n| TC002 | Move third-party import into a type-checking block |\n| TC003 | Found multiple type checking blocks |\n| TC004 | Move import out of type-checking block. Import is used for more than type hinting. |\n| TC005 | Empty type-checking block |\n\n## Choosing how to handle forward references\n\nYou need to choose whether to opt-into using the\n`TC100`- or the `TC200`-range of error codes.\n\nThey represent two different ways of solving the same problem, so please only choose one.\n\n`TC100` and `TC101` manage forward references by taking advantage of\n[postponed evaluation of annotations](https://www.python.org/dev/peps/pep-0563/).\n\n| Code   | Description                                         |\n|--------|-----------------------------------------------------|\n| TC100 | Add \'from \\_\\_future\\_\\_ import annotations\' import |\n| TC101 | Annotation does not need to be a string literal |\n\n`TC200` and `TC201` manage forward references using [string literals](https://www.python.org/dev/peps/pep-0484/#forward-references).\n\n| Code   | Description                                         |\n|--------|-----------------------------------------------------|\n| TC200 | Annotation needs to be made into a string literal |\n| TC201 | Annotation does not need to be a string literal |\n\n## Enabling error ranges\n\nAdd `TC` and `TC1` or `TC2` to your flake8 config like this:\n\n```ini\n[flake8]\nmax-line-length = 80\nmax-complexity = 12\n...\nignore = E501\n# You can use \'select\':\nselect = C,E,F..., TC, TC2  # or TC1\n# OR \'enable-extensions\':\nenable-extensions = TC, TC2  # or TC1\n```\n\nIf you are unsure which `TC` range to pick, see the [rationale](#rationale) for more info.\n\n## Installation\n\n```shell\npip install flake8-type-checking\n```\n\n## Configuration\n\nThese options are configurable, and can be set in your flake8 config.\n\n### Exempt modules\n\nIf you wish to exempt certain modules from\nneeding to be moved into type-checking blocks, you can specify which\nmodules to ignore.\n\n- **setting name**: `type-checking-exempt-modules`\n- **type**: `list`\n\n```ini\n[flake8]\ntype-checking-exempt-modules = typing, typing_extensions  # default []\n```\n\n### Pydantic support\n\nIf you use Pydantic models in your code, you should enable Pydantic support.\nThis will treat any class variable annotation as being needed during runtime.\n\n- **name**: `type-checking-pydantic-enabled`\n- **type**: `bool`\n```ini\n[flake8]\ntype-checking-pydantic-enabled = true  # default false\n```\n### Pydantic support base-class passlist\n\nDisabling checks for all class annotations is a little aggressive.\n\nIf you feel comfortable that all base classes named, e.g., `NamedTuple` are *not* Pydantic models,\nthen you can pass the names of the base classes in this setting, to re-enable checking for classes\nwhich inherit from them.\n\n- **name**: `type-checking-pydantic-enabled-baseclass-passlist`\n- **type**: `list`\n```ini\n[flake8]\ntype-checking-pydantic-enabled-baseclass-passlist = NamedTuple, TypedDict  # default []\n```\n\n### FastAPI support\n\nIf you\'re using the plugin for a FastAPI project,\nyou should enable support. This will treat the annotations\nof any decorated function as needed at runtime.\n\nEnabling FastAPI support will also enable Pydantic support.\n\n- **name**: `type-checking-fastapi-enabled`\n- **type**: `bool`\n```ini\n[flake8]\ntype-checking-fastapi-enabled = true  # default false\n```\n\nOne more thing to note for FastAPI users is that dependencies\n(functions used in `Depends`) will produce false positives, unless\nyou enable dependency support as described below.\n\n### FastAPI dependency support\n\nIn addition to preventing false positives for decorators, we *can*\nprevent false positives for dependencies. We are making a pretty bad\ntrade-off however: by enabling this option we treat every annotation\nin every function definition across your entire project as a possible\ndependency annotation. In other words, we stop linting all function\nannotations completely, to avoid the possibility of false positives.\nIf you prefer to be on the safe side, you should enable this - otherwise\nit might be enough to be aware that false positives can happen for functions\nused as dependencies.\n\nEnabling dependency support will also enable FastAPI and Pydantic support.\n\n- **name**: `type-checking-fastapi-dependency-support-enabled`\n- **type**: `bool`\n```ini\n[flake8]\ntype-checking-fastapi-dependency-support-enabled: true  # default false\n```\n\n## Rationale\n\nWhy did we create this plugin?\n\nGood type hinting typically requires a lot of project imports, which can increase\nthe risk of [import cycles](https://mypy.readthedocs.io/en/stable/runtime_troubles.html?#import-cycles)\nin a project. The recommended way of preventing this problem is to use `typing.TYPE_CHECKING` blocks\nto guard these types of imports.\n\nBoth `TC001` and `TC002` help alleviate this problem; the reason there are two\ncodes instead of one, is because the import cycles rarely occur from\nlibrary/third-party imports, so this classification split provides a way to filter down\nthe total number of imports for users that want to guard against import cycles,\nbut don\'t want to manage every import in their projects *this* strictly.\n\nOnce imports are guarded, they will no longer be evaluated/imported during runtime. The\nconsequence of this is that these imports can no longer be treated as if they\nwere imported outside the block. Instead we need to use [forward references](https://www.python.org/dev/peps/pep-0484/#forward-references).\n\nFor Python version `>= 3.7`, there are actually two ways of solving this issue.\nYou can either make your annotations string literals, or you can use a `__futures__` import to enable [postponed evaluation of annotations](https://www.python.org/dev/peps/pep-0563/).\nSee [this](https://stackoverflow.com/a/55344418/8083459) excellent stackoverflow answer\nfor a great explanation of the differences.\n\n## Examples\n\n<details>\n<summary><b>Performance example</b></summary>\n\nImports for type hinting can have a performance impact.\n\n```python\nimport pandas\n\n\ndef dataframe_length(df: pandas.DataFrame) -> int:\n    return len(df)\n```\n\nIn this example, we import a 15mb library, for a single type hint.\n\nWe don\'t need to perform this operation at runtime, *at all*.\nIf we know that the import will not otherwise be needed by surrounding code,\nwe can simply guard it, like this:\n\n```python\nfrom typing import TYPE_CHECKING\n\n\nif TYPE_CHECKING:\n    import pandas  # <-- no longer imported at runtime\n\n\ndef dataframe_length(df: pandas.DataFrame) -> int:\n    return len(df)\n```\n\nNow the import is no longer made at runtime. If you\'re unsure about how this works, see the [mypy docs](https://mypy.readthedocs.io/en/stable/runtime_troubles.html?#typing-type-checking) for a basic introduction.\n</details>\n\n<details>\n<summary><b>Import circularity example</b></summary>\n\n**Bad code**\n\n`models/a.py`\n```python\nfrom models.b import B\n\nclass A(Model):\n    def foo(self, b: B): ...\n```\n\n`models/b.py`\n```python\nfrom models.a import A\n\nclass B(Model):\n    def bar(self, a: A): ...\n```\n\nWill result in these errors\n\n```shell\n>> a.py: TC002 Move third-party import \'models.b.B\' into a type-checking block\n>> b.py: TC002 Move third-party import \'models.a.A\' into a type-checking block\n```\n\nand consequently trigger these errors if imports are purely moved into type-checking block, without proper forward reference handling\n\n```shell\n>> a.py: TC100 Add \'from __future__ import annotations\' import\n>> b.py: TC100 Add \'from __future__ import annotations\' import\n```\n\nor\n\n```shell\n>> a.py: TC200 Annotation \'B\' needs to be made into a string literal\n>> b.py: TC200 Annotation \'A\' needs to be made into a string literal\n```\n\n**Good code**\n\n`models/a.py`\n```python\nfrom typing import TYPE_CHECKING\n\nif TYPE_CHECKING:\n    from models.b import B\n\nclass A(Model):\n    def foo(self, b: \'B\'): ...\n```\n`models/b.py`\n```python\n# TC1\nfrom __future__ import annotations\n\nfrom typing import TYPE_CHECKING\n\nif TYPE_CHECKING:\n    from models.a import A\n\nclass B(Model):\n    def bar(self, a: A): ...\n```\n\nor\n\n```python\n# TC2\nfrom typing import TYPE_CHECKING\n\nif TYPE_CHECKING:\n    from models.a import A\n\nclass B(Model):\n    def bar(self, a: \'A\'): ...\n```\n</details>\n\n## Running the plugin as a pre-commit hook\n\nYou can run this flake8 plugin as a [pre-commit](https://github.com/pre-commit/pre-commit) hook:\n\n```yaml\n- repo: https://github.com/pycqa/flake8\n  rev: 4.0.1\n  hooks:\n    - id: flake8\n      additional_dependencies:\n        - flake8-type-checking\n```\n\n## Contributing\n\nPlease feel free to open an issue or a PR 👏\n',
    'author': 'Sondre Lillebø Gundersen',
    'author_email': 'sondrelg@live.no',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/snok',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
