# Django Browsable Router

```
pip install django-browsable-router
```

A Django Restframework router that can show APIViews and include other routers as navigable urls in the root view.

```python
from browsable_router import APIRouter
from rest_framework.views import APIView
from rest_framework.viewsets import ViewSet

class TestView(APIView):
  ...
 
class TestViewSet(ViewSet):
  ...

router_1 = APIRouter()
router_1.format_root_view("other_routes", "These are under a different route.")
router_1.register(r"view-1", TestView.as_view(), "view_1")
router_1.register(r"view-2", TestViewSet.as_view(), "view_2")

router_2 = APIRouter()
router_2.register(r"view-3", TestView.as_view(), "view_3")
router_2.navigation_routes = {
    "route": router_1,
}

urlpatterns = [
    path("api/", include(router_2.urls))
]
```

Resulting browsable API:
```python
#   API Root:
#   """API root."""
# 
#   "route":    "/api/route/"
#   "view-3":   "/api/view-3/"
# 
#   Other Routes:
#   """These are under a different route."""
# 
#   "view-1":    "/api/route/view-1/"
#   "view-2":    "/api/route/view-2/"
```

---

# Additional stuff:

### BaseAPIView
A custom BaseAPIView class, that simplifies making a workflow of:
```python
input: "raw data" 
-> serializer = Serializer(data=...) 
-> serializers.is_valid(raise_exception=True) 
-> Response(data=serialziers.validated_data)
```
Simpy inherit from the class and any number of the included mixins. Then define the Serializers per method in the 'serializer_classes' attribute.
```python
from browsable_router import BaseAPIView, GetMixin, PostMixin
from rest_framework.serializers import Serializer


class SomeCustomView(GetMixin PostMixin, BaseAPIView):

  class GetSerializer(Serializer):
    ...
    
  class PostSerializer(Serializer):
    ...

  serializer_classes = {
    "GET": GetSerializer,
    "POST": PostSerializer,
  }
```

This allows different serializers for different HTTP methods. Incoming data can be handled in the HTTP method function itself, in serializer's `validate()` method, or in a `handle_<method>()` function added by the mixins:

```python
class SomeCustomView(GetMixin, BaseAPIView):
  
  class GetSerializer(Serializer):
    ...
    
    def validate(self, attrs):
      # (2) Second place to modify the data. Run during input validation.
      return attrs

  serializer_classes = {
    "GET": GetSerializer,
  }
  
  def get(request, *args, **kwargs):
    # (1) First place to modify the data. Run before input validation.
    return super().get(request, *args, **kwargs)
  
  def handle_get(request, data, *args, **kwargs):
    # (3) Last place to modify the data. Run after validation.
    return data
  
```

### BaseAPISerializer
A custom serializer, that maps serializer fields to a bound method call, and validates output through an OutputSerializer.
```python
from browsable_router import BaseAPISerializer
from rest_framework import serializers

class GetSerializer(BaseAPISerializer):
  class OutputSerializer(serializers.Serializer):
    output = serializers.CharField()
    
  input = serializers.CharField()
  
  def bound_method(*args, **kwargs):
    print(kwargs["input"])
    return {"output": "..."}
```
Using this base class will let you modify the output of the serializer inside it, perhaps with a service or selector method, and verify that output with the output serializer. Furthermore, you can set `many=True` attribute on the OutputSerializer if the output is a list. There is also a further benefit if you use the included metadata class, since it allows automatic schema documentation based on the OutputSerializer.

### ApiMetadata
This metadata class is enabled by default on BaseAPIView, and it will allow you to document your views automatically if you also use the BaseAPISerializer as the parent class for the method serializers. It essentially reads the method serializers fields as input for that view method, and the method serializers OutputSerializer as the output for that view method. You can then make a OPTIONS request to the view endpoint and get a JSON Schema-ish representation of the input and output for that endpoint.

### SerializerAsOutputMetadata
This metadata class works like APIMetadata class, but no OutpuMetadata is used inside the method serializer. Instead, it assumes that the endpoint takes no arguments and returns the method serializers fields as output. This is useful for list-view endpoints.

### BlockSchemaAccess
This is a permission class that can be used to block access to OPTIONS schema in production (DEBUG=False).
