# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['flowmancer',
 'flowmancer.checkpoint',
 'flowmancer.eventbus',
 'flowmancer.extensions',
 'flowmancer.extensions.notifications',
 'flowmancer.jobdefinition',
 'flowmancer.loggers']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML>=6.0.1,<7.0.0',
 'pyaml-env>=1.2.1,<2.0.0',
 'pydantic>=1.9.0,<2.0.0',
 'requests>=2.31.0,<3.0.0',
 'rich>=12.0.0,<13.0.0',
 'urllib3==1.26.15']

setup_kwargs = {
    'name': 'flowmancer',
    'version': '0.5.5',
    'description': 'The Python Thing-Doer',
    'long_description': '# Flowmancer\n\n[![pypi-version](https://img.shields.io/pypi/v/flowmancer?style=flat-square)](https://pypi.org/project/flowmancer)\n[![python-version](https://img.shields.io/badge/dynamic/json?query=info.requires_python&label=python&url=https%3A%2F%2Fpypi.org%2Fpypi%2Fflowmancer%2Fjson&style=flat-square)](https://pypi.org/project/flowmancer)\n[![license](https://img.shields.io/github/license/natsunlee/flowmancer?style=flat-square)](LICENSE)\n[![circle-ci](https://img.shields.io/circleci/build/github/natsunlee/flowmancer?style=flat-square)](https://app.circleci.com/pipelines/github/natsunlee/flowmancer)\n[![coveralls](https://img.shields.io/coveralls/github/natsunlee/flowmancer?style=flat-square)](https://coveralls.io/github/natsunlee/flowmancer?branch=main)\n[![pypi-downloads](https://img.shields.io/pypi/dm/flowmancer?style=flat-square)](https://pypistats.org/packages/flowmancer)\n[![Ko-Fi](https://img.shields.io/badge/Support%20Me%20On%20Ko--fi-F16061?style=flat-square&logo=ko-fi&logoColor=white)](https://ko-fi.com/natsunlee)\n\nFlowmancer aims to help you do *things* in a sequential or parallel manner. It enables you to write tasks in Python, describe their order, then execute them with as little effort as possible.\n\nBut why do I need this? Couldn\'t I just write my own Python code to do *stuff*?\n\nYou certainly could!\n\nThough Flowmancer provides gives you a head-start to building your custom processes with optional add-ons for logging, checkpoint/restarts in the event of failures, or even custom task observers to do...things while your things do things!\n\n## Installation\nSimply install the `flowmancer` package with:\n```bash\npip3 install flowmancer\n```\n\nNOTE: `flowmancer` supports only Python 3.7 and higher.\n\n## Basic Usage\nLet\'s assume you have a new project with a basic structure like so:\n```\nmy_project\n├─ job.yaml\n├─ main.py\n└─ tasks/\n   ├─ __init__.py\n   └─ mytasks.py\n```\n\nTo use `flowmancer`, you\'ll need to provide a few things:\n* `Task` implementations (`mytasks.py`)\n* A job YAML file (`job.yaml`)\n* Your main/driver code (`main.py`)\n\n### Tasks\nBy default, Flowmancer recursively searches in the `./tasks` directory (relative to where `Flowmancer()` is initialized - in this case, `main.py`) for `Task` implementations decorated with `@task`. See the Advanced Usage section for details on how to add other directories or packages that contain `Task` implementations.\n\nA `flowmancer` task is simply a class that extends the `Task` abstract class, which, at minimum requires that the `run` method be implemented:\n```python\nimport time\nfrom flowmancer.task import Task, task\n\n@task\nclass WaitAndSucceed(Task):\n    # All variables should be given type hints and optional vars should be given default values.\n    my_required_string_var: str\n    my_optional_int_var: int = 5\n\n    def run(self):\n        # Store string input var in the shared dictionary accessible by other tasks.\n        self.shared_dict["my_var"] = f"Hello from: {self.my_required_string_var}!"\n\n        # Sleep for seconds defined by input var (using default of 5).\n        print(f"Starting up and sleeping for {self.my_optional_int_var} seconds!")\n        time.sleep(self.my_optional_int_var)\n        print("Done!")\n\n@task\nclass ImmediatelySucceed(Task):\n    def run(self):\n        # Print statements will automatically be sent to configured loggers.\n        print("Success!")\n\n@task\nclass FailImmediately(Task):\n    def run(self):\n        print(f"Printing `my_var` value: {self.shared_dict[\'my_var\']}")\n        # Raise errors to cause tasks to fail and additionally block dependent tasks, if any.\n        raise RuntimeError("Let this be caught by Flowmancer")\n```\n\nAny `print()` or exceptions will write log messages to any configured loggers (zero or more loggers may be defined).\n\n### Job Definition YAML File\nThis file describes what code to run, in what order, as well as additional add-ons to supplement the job during execution:\n```yaml\nversion: 0.1\n\n# This entire config block is currently optional, however, it is recommended to at least provide a unique name for each\n# Job Definition YAML file, as this name is used for checkpointing jobs in the event of failures.\nconfig:\n  name: \'my-flowmancer-job\'\n\ntasks:\n  # No dependency - run right away\n  # Add `parameters` key-value pairs for any required and optional task variables.\n  succeed-task-a:\n    task: WaitAndSucceed\n    parameters:\n      my_required_string_var: "My First Task!"\n\n  # No dependency - run right away\n  succeed-task-b:\n    task: ImmediatelySucceed\n\n  # Only run if prior 2 tasks complete successfully\n  final-fail-task:\n    task: FailImmediately\n    dependencies:\n      - succeed-task-a\n      - succeed-task-b\n```\n\n### Driver\nThe driver is super simple and simply requires running an instance of `Flowmancer`\n```python\n# main.py\nimport sys\nfrom flowmancer import Flowmancer\n\nif __name__ == \'__main__\':\n    # The `start()` method will return a non-zero integer on failure, typically equal to the number of failed tasks.\n    ret = Flowmancer().start()\n\n    # Exceptions from tasks will be captured and logged, rather than being raised up to this level. To cause this\n    # driver program to fail, either explicitly raise your own error OR call sys.exit.\n    if ret:\n      raise RuntimeError(\'Flowmancer job has failed!\')\n\n    # Alternatively, instead of crashing w/ an exception, simply exit with a non-zero value.\n    # sys.exit(ret)\n```\n\n### Executing the Job\n```bash\npython3 main.py -j ./path/to/job.yaml\n```\n\nTo run from point-of-failure (if any):\n```bash\npython3 main.py -j ./path/to/job.yaml -r\n```\nIf no prior failure is detected, the job will start as if no `-r` flag were given.\n\nNote that the job definition must still be provided with the `-r` flag.\n\n## Advanced Usage\n\n### Optional Configurations\nIn the `config` block of the Job Definition, the following optional parameters may be given:\n|Parameter|Type|Default Value|Description|\n|---|---|---|---|\n|name|str|\'flowmancer\'|Name/identifier for Job Definition. Used for saving checkpoints used for job restarts in the event of a failure.|\n|max_concurrency|int|0|Maximum number tasks that can run in parallel. If 0 or less, then there is no limit.|\n|extension_directories|List[str]|[]|List of paths, either absolute or relative to driver `.py` file, that contain any `@task`, `@logger`, or `@extension` decorated classes to make accessible to Flowmancer. The `./task`, `./extensions`, and `./loggers` directories are ALWAYS checked by default.|\n|extension_packages|List[str]|[]|List of installed Python packages that contain `@task`, `@logger`, or `@extension` decorated classes to make accessible to Flowmancer.|\n\nFor example:\n```yaml\nconfig:\n  name: \'most-important-job\'\n  max_concurrency: 20\n  extension_directories:\n    - ./client_implementations\n    - /opt/flowmancer/tasks\n  extension_packages:\n    - internal_flowmancer_package\n```\n\n### Complex Parameters\nWhile this is mostly used for `Task` implementations, the details outlined here apply for any built-in and custom `Extension` and `Logger` implementations.\n\nFlowmancer makes heavy use of [Pydantic](https://docs.pydantic.dev/latest/) to validate parameters and ensure that values loaded from the Job Definition are of the appropriate type.\n\nThis means that a `Task` can have complex types (including custom models) like:\n```python\nfrom enum import Enum\nfrom flowmancer.task import Task, task\nfrom pydantic import BaseModel\nfrom typing import Dict, List\n\nclass Protocol(Enum):\n    HTTP: \'HTTP\'\n    HTTPS: \'HTTPS\'\n\nclass APIDetails(BaseModel):\n    protocol: Protocol = Protocol.HTTPS\n    base_url: str\n    endpoint: str\n\n@task\nclass DownloadDataFromRestApi(Task):\n    api_details: APIDetails\n    target_dir: str\n    target_filename: str = \'data.json\'\n\n    def run(self) -> None:\n        url = f\'{self.api_details.protocol}://{self.api_details.base_url}/{self.api_details.endpoint}\'\n        # Continued implementation...\n```\n\nAnd the Job Definition snippet for this task might be:\n```yaml\ntasks:\n  download-file-one:\n    task: DownloadDataFromRestApi\n    parameters:\n      api_details:\n        # We leave out `protocol` because we want to just use the default `HTTPS` value.\n        base_url: www.some_data_api.com\n        endpoint: /v1/data/weather/today\n      target_dir: /data/todays_weather\n      # Override the default `target_filename` value given in the class implementation.\n      target_filename: weather.json\n```\n\n### Task Lifecycle Methods\nIn addition to the required `run` method, an implementation of `Task` may optionally include the following methods:\n|Method|Required|Order|Description|\n|---|---|---|---|\n|on_create|No|1|First method executed when a task is released for execution. Note that a task is not considered "created" until it enters the `RUNNING` state.|\n|on_restart|No|2|Executed only if a task is running from the result of a recovery from `FAILED` state. If a task was failed in `DEFAULTED` state, this method will not be executed.|\n|run|Yes|3|Always required and always executed once task is in `RUNNING` state, unless prior lifecycle methods have failed.|\n|on_success|No|4|Executed only if `run` method ends in success.|\n|on_failure|No|5|Executed only if `run` method ends in failure/exception.|\n|on_destroy|No|6|Always executed after all other lifecycle methods.|\n|on_abort|No|-|Executed when `SIGINT` signal is sent to tasks/Flowmancer.|\n\nJust as with `run`, all lifecycle methods have access to `self.shared_dict` and any parameters.\n\n### Custom Loggers\nComing soon.\n\n### Custom Extensions\nComing soon.\n',
    'author': 'Nathan Lee',
    'author_email': 'lee.nathan.sh@outlook.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
