"use strict";
var _a, _b, _c, _d, _e, _f, _g;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetImageCode = exports.EcrImageCode = exports.CfnParametersCode = exports.AssetCode = exports.InlineCode = exports.S3Code = exports.Code = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ecr_assets = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
/**
 * Represents the Lambda Handler Code.
 */
class Code {
    /**
     * Lambda handler code as an S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * DEPRECATED
     * @deprecated use `fromBucket`
     */
    static bucket(bucket, key, objectVersion) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-lambda.Code#bucket", "use `fromBucket`");
        return this.fromBucket(bucket, key, objectVersion);
    }
    /**
     * Inline code for Lambda handler
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * DEPRECATED
     * @deprecated use `fromInline`
     */
    static inline(code) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-lambda.Code#inline", "use `fromInline`");
        return this.fromInline(code);
    }
    /**
     * Loads the function code from a local disk path.
     *
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static fromAsset(path, options) {
        return new AssetCode(path, options);
    }
    /**
     * Loads the function code from an asset created by a Docker build.
     *
     * By default, the asset is expected to be located at `/asset` in the
     * image.
     *
     * @param path The path to the directory containing the Docker file
     * @param options Docker build options
     */
    static fromDockerBuild(path, options = {}) {
        var _h;
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_DockerBuildAssetOptions(options);
        let imagePath = (_h = options.imagePath) !== null && _h !== void 0 ? _h : '/asset/.';
        // ensure imagePath ends with /. to copy the **content** at this path
        if (imagePath.endsWith('/')) {
            imagePath = `${imagePath}.`;
        }
        else if (!imagePath.endsWith('/.')) {
            imagePath = `${imagePath}/.`;
        }
        const assetPath = cdk.DockerImage
            .fromBuild(path, options)
            .cp(imagePath, options.outputPath);
        return new AssetCode(assetPath);
    }
    /**
     * DEPRECATED
     * @deprecated use `fromAsset`
     */
    static asset(path) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-lambda.Code#asset", "use `fromAsset`");
        return this.fromAsset(path);
    }
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @returns a new instance of `CfnParametersCode`
     * @param props optional construction properties of {@link CfnParametersCode}
     */
    static fromCfnParameters(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_CfnParametersCodeProps(props);
        return new CfnParametersCode(props);
    }
    /**
     * DEPRECATED
     * @deprecated use `fromCfnParameters`
     */
    static cfnParameters(props) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-lambda.Code#cfnParameters", "use `fromCfnParameters`");
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_CfnParametersCodeProps(props);
        return this.fromCfnParameters(props);
    }
    /**
     * Use an existing ECR image as the Lambda code.
     * @param repository the ECR repository that the image is in
     * @param props properties to further configure the selected image
     */
    static fromEcrImage(repository, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_EcrImageCodeProps(props);
        return new EcrImageCode(repository, props);
    }
    /**
     * Create an ECR image from the specified asset and bind it as the Lambda code.
     * @param directory the directory from which the asset must be created
     * @param props properties to further configure the selected image
     */
    static fromAssetImage(directory, props = {}) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_AssetImageCodeProps(props);
        return new AssetImageCode(directory, props);
    }
    /**
     * Called after the CFN function resource has been created to allow the code
     * class to bind to it. Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     */
    bindToResource(_resource, _options) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_ResourceBindOptions(_options);
        return;
    }
}
exports.Code = Code;
_a = JSII_RTTI_SYMBOL_1;
Code[_a] = { fqn: "@aws-cdk/aws-lambda.Code", version: "1.150.0" };
/**
 * Lambda code from an S3 archive.
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
_b = JSII_RTTI_SYMBOL_1;
S3Code[_b] = { fqn: "@aws-cdk/aws-lambda.S3Code", version: "1.150.0" };
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        this.isInline = true;
        if (code.length === 0) {
            throw new Error('Lambda inline code cannot be empty');
        }
        if (code.length > 4096) {
            throw new Error('Lambda source is too large, must be <= 4096 but is ' + code.length);
        }
    }
    bind(_scope) {
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
_c = JSII_RTTI_SYMBOL_1;
InlineCode[_c] = { fqn: "@aws-cdk/aws-lambda.InlineCode", version: "1.150.0" };
/**
 * Lambda code from a local directory.
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
        this.isInline = false;
    }
    bind(scope) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.path,
                ...this.options,
            });
        }
        else if (cdk.Stack.of(this.asset) !== cdk.Stack.of(scope)) {
            throw new Error(`Asset is already associated with another stack '${cdk.Stack.of(this.asset).stackName}'. ` +
                'Create a new Code instance for every stack.');
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    bindToResource(resource, options = {}) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_ResourceBindOptions(options);
        if (!this.asset) {
            throw new Error('bindToResource() must be called after bind()');
        }
        const resourceProperty = options.resourceProperty || 'Code';
        // https://github.com/aws/aws-cdk/issues/1432
        this.asset.addResourceMetadata(resource, resourceProperty);
    }
}
exports.AssetCode = AssetCode;
_d = JSII_RTTI_SYMBOL_1;
AssetCode[_d] = { fqn: "@aws-cdk/aws-lambda.AssetCode", version: "1.150.0" };
/**
 * Lambda code defined using 2 CloudFormation parameters.
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 */
class CfnParametersCode extends Code {
    constructor(props = {}) {
        super();
        this.isInline = false;
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_CfnParametersCodeProps(props);
        this._bucketNameParam = props.bucketNameParam;
        this._objectKeyParam = props.objectKeyParam;
    }
    bind(scope) {
        if (!this._bucketNameParam) {
            this._bucketNameParam = new cdk.CfnParameter(scope, 'LambdaSourceBucketNameParameter', {
                type: 'String',
            });
        }
        if (!this._objectKeyParam) {
            this._objectKeyParam = new cdk.CfnParameter(scope, 'LambdaSourceObjectKeyParameter', {
                type: 'String',
            });
        }
        return {
            s3Location: {
                bucketName: this._bucketNameParam.valueAsString,
                objectKey: this._objectKeyParam.valueAsString,
            },
        };
    }
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 location
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Location` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param location the location of the object in S3 that represents the Lambda code
     */
    assign(location) {
        const ret = {};
        ret[this.bucketNameParam] = location.bucketName;
        ret[this.objectKeyParam] = location.objectKey;
        return ret;
    }
    get bucketNameParam() {
        if (this._bucketNameParam) {
            return this._bucketNameParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the bucketNameParam property');
        }
    }
    get objectKeyParam() {
        if (this._objectKeyParam) {
            return this._objectKeyParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the objectKeyParam property');
        }
    }
}
exports.CfnParametersCode = CfnParametersCode;
_e = JSII_RTTI_SYMBOL_1;
CfnParametersCode[_e] = { fqn: "@aws-cdk/aws-lambda.CfnParametersCode", version: "1.150.0" };
/**
 * Represents a Docker image in ECR that can be bound as Lambda Code.
 */
class EcrImageCode extends Code {
    constructor(repository, props = {}) {
        super();
        this.repository = repository;
        this.props = props;
        this.isInline = false;
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_EcrImageCodeProps(props);
    }
    bind(_) {
        var _h, _j;
        this.repository.grantPull(new iam.ServicePrincipal('lambda.amazonaws.com'));
        return {
            image: {
                imageUri: this.repository.repositoryUriForTag((_j = (_h = this.props) === null || _h === void 0 ? void 0 : _h.tag) !== null && _j !== void 0 ? _j : 'latest'),
                cmd: this.props.cmd,
                entrypoint: this.props.entrypoint,
                workingDirectory: this.props.workingDirectory,
            },
        };
    }
}
exports.EcrImageCode = EcrImageCode;
_f = JSII_RTTI_SYMBOL_1;
EcrImageCode[_f] = { fqn: "@aws-cdk/aws-lambda.EcrImageCode", version: "1.150.0" };
/**
 * Represents an ECR image that will be constructed from the specified asset and can be bound as Lambda code.
 */
class AssetImageCode extends Code {
    constructor(directory, props) {
        super();
        this.directory = directory;
        this.props = props;
        this.isInline = false;
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_AssetImageCodeProps(props);
    }
    bind(scope) {
        // If the same AssetImageCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new ecr_assets.DockerImageAsset(scope, 'AssetImage', {
                directory: this.directory,
                ...this.props,
            });
            this.asset.repository.grantPull(new iam.ServicePrincipal('lambda.amazonaws.com'));
        }
        else if (cdk.Stack.of(this.asset) !== cdk.Stack.of(scope)) {
            throw new Error(`Asset is already associated with another stack '${cdk.Stack.of(this.asset).stackName}'. ` +
                'Create a new Code instance for every stack.');
        }
        return {
            image: {
                imageUri: this.asset.imageUri,
                entrypoint: this.props.entrypoint,
                cmd: this.props.cmd,
                workingDirectory: this.props.workingDirectory,
            },
        };
    }
    bindToResource(resource, options = {}) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_ResourceBindOptions(options);
        if (!this.asset) {
            throw new Error('bindToResource() must be called after bind()');
        }
        const resourceProperty = options.resourceProperty || 'Code.ImageUri';
        // https://github.com/aws/aws-cdk/issues/14593
        this.asset.addResourceMetadata(resource, resourceProperty);
    }
}
exports.AssetImageCode = AssetImageCode;
_g = JSII_RTTI_SYMBOL_1;
AssetImageCode[_g] = { fqn: "@aws-cdk/aws-lambda.AssetImageCode", version: "1.150.0" };
//# sourceMappingURL=data:application/json;base64,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