"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontWebDistribution = exports.ViewerCertificate = exports.CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedMethods = exports.OriginSslPolicy = exports.FailoverStatusCode = void 0;
const certificatemanager = require("@aws-cdk/aws-certificatemanager");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
const distribution_1 = require("./distribution");
/**
 * (experimental) HTTP status code to failover to second origin.
 *
 * @experimental
 */
var FailoverStatusCode;
(function (FailoverStatusCode) {
    FailoverStatusCode[FailoverStatusCode["FORBIDDEN"] = 403] = "FORBIDDEN";
    FailoverStatusCode[FailoverStatusCode["NOT_FOUND"] = 404] = "NOT_FOUND";
    FailoverStatusCode[FailoverStatusCode["INTERNAL_SERVER_ERROR"] = 500] = "INTERNAL_SERVER_ERROR";
    FailoverStatusCode[FailoverStatusCode["BAD_GATEWAY"] = 502] = "BAD_GATEWAY";
    FailoverStatusCode[FailoverStatusCode["SERVICE_UNAVAILABLE"] = 503] = "SERVICE_UNAVAILABLE";
    FailoverStatusCode[FailoverStatusCode["GATEWAY_TIMEOUT"] = 504] = "GATEWAY_TIMEOUT";
})(FailoverStatusCode = exports.FailoverStatusCode || (exports.FailoverStatusCode = {}));
/**
 * @experimental
 */
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
/**
 * (experimental) An enum for the supported methods to a CloudFront distribution.
 *
 * @experimental
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * (experimental) Enums for the methods CloudFront can cache.
 *
 * @experimental
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
/**
 * (experimental) Viewer certificate configuration class.
 *
 * @experimental
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * (experimental) Generate an AWS Certificate Manager (ACM) viewer certificate configuration.
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     * @param options certificate configuration options.
     * @experimental
     */
    static fromAcmCertificate(certificate, options = {}) {
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * (experimental) Generate an IAM viewer certificate configuration.
     *
     * @param iamCertificateId Identifier of the IAM certificate.
     * @param options certificate configuration options.
     * @experimental
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * (experimental) Generate a viewer certifcate configuration using the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net) and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases You also must create a CNAME record with your DNS service to route queries.
     * @experimental
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
/**
 * (experimental) Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos, applications, and APIs to your viewers with low latency and high transfer speeds.
 *
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFrontWebDistribution } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontWebDistribution(this, 'MyDistribution', {
 *   originConfigs: [
 *     {
 *       s3OriginSource: {
 *       s3BucketSource: sourceBucket
 *       },
 *       behaviors : [ {isDefaultBehavior: true}]
 *     }
 *   ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 * @experimental
 * @resource AWS::CloudFront::Distribution
 */
class CloudFrontWebDistribution extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [distribution_1.SSLMethod.SNI]: [
                distribution_1.SecurityPolicyProtocol.TLS_V1, distribution_1.SecurityPolicyProtocol.TLS_V1_1_2016,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2016, distribution_1.SecurityPolicyProtocol.TLS_V1_2_2018,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2_2019,
            ],
            [distribution_1.SSLMethod.VIP]: [distribution_1.SecurityPolicyProtocol.SSL_V3, distribution_1.SecurityPolicyProtocol.TLS_V1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : 'index.html',
            httpVersion: props.httpVersion || distribution_1.HttpVersion.HTTP2,
            priceClass: props.priceClass || distribution_1.PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // eslint-disable-next-line max-len
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        const originGroups = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            let originId = `origin${originIndex}`;
            const originProperty = this.toOriginProperty(originConfig, originId);
            if (originConfig.failoverCustomOriginSource || originConfig.failoverS3OriginSource) {
                const originSecondaryId = `originSecondary${originIndex}`;
                const originSecondaryProperty = this.toOriginProperty({
                    s3OriginSource: originConfig.failoverS3OriginSource,
                    customOriginSource: originConfig.failoverCustomOriginSource,
                    originPath: originConfig.originPath,
                    originHeaders: originConfig.originHeaders,
                }, originSecondaryId);
                const originGroupsId = `OriginGroup${originIndex}`;
                const failoverCodes = (_a = originConfig.failoverCriteriaStatusCodes) !== null && _a !== void 0 ? _a : [500, 502, 503, 504];
                originGroups.push({
                    id: originGroupsId,
                    members: {
                        items: [{ originId }, { originId: originSecondaryId }],
                        quantity: 2,
                    },
                    failoverCriteria: {
                        statusCodes: {
                            items: failoverCodes,
                            quantity: failoverCodes.length,
                        },
                    },
                });
                originId = originGroupsId;
                origins.push(originSecondaryProperty);
            }
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        const originGroupsDistConfig = originGroups.length > 0
            ? {
                items: originGroups,
                quantity: originGroups.length,
            }
            : undefined;
        distributionConfig = {
            ...distributionConfig,
            origins,
            originGroups: originGroupsDistConfig,
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // eslint-disable-next-line max-len
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = {
                ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket');
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        if (props.geoRestriction) {
            distributionConfig = {
                ...distributionConfig,
                restrictions: {
                    geoRestriction: {
                        restrictionType: props.geoRestriction.restrictionType,
                        locations: props.geoRestriction.locations,
                    },
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * (experimental) Creates a construct that represents an external (imported) distribution.
     *
     * @experimental
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends cdk.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || distribution_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            const includeBodyEventTypes = [distribution_1.LambdaEdgeEventType.ORIGIN_REQUEST, distribution_1.LambdaEdgeEventType.VIEWER_REQUEST];
            if (input.lambdaFunctionAssociations.some(fna => fna.includeBody && !includeBodyEventTypes.includes(fna.eventType))) {
                throw new Error('\'includeBody\' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types.');
            }
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.edgeArn,
                    includeBody: fna.includeBody,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
    toOriginProperty(originConfig, originId) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        if (!originConfig.s3OriginSource &&
            !originConfig.customOriginSource) {
            throw new Error('There must be at least one origin source - either an s3OriginSource, a customOriginSource');
        }
        if (originConfig.customOriginSource && originConfig.s3OriginSource) {
            throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
        }
        if ([
            originConfig.originHeaders,
            (_a = originConfig.s3OriginSource) === null || _a === void 0 ? void 0 : _a.originHeaders,
            (_b = originConfig.customOriginSource) === null || _b === void 0 ? void 0 : _b.originHeaders,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originHeaders field allowed across origin and failover origins');
        }
        if ([
            originConfig.originPath,
            (_c = originConfig.s3OriginSource) === null || _c === void 0 ? void 0 : _c.originPath,
            (_d = originConfig.customOriginSource) === null || _d === void 0 ? void 0 : _d.originPath,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originPath field allowed across origin and failover origins');
        }
        const headers = (_g = (_e = originConfig.originHeaders) !== null && _e !== void 0 ? _e : (_f = originConfig.s3OriginSource) === null || _f === void 0 ? void 0 : _f.originHeaders) !== null && _g !== void 0 ? _g : (_h = originConfig.customOriginSource) === null || _h === void 0 ? void 0 : _h.originHeaders;
        const originHeaders = [];
        if (headers) {
            Object.keys(headers).forEach((key) => {
                const oHeader = {
                    headerName: key,
                    headerValue: headers[key],
                };
                originHeaders.push(oHeader);
            });
        }
        let s3OriginConfig;
        if (originConfig.s3OriginSource) {
            // first case for backwards compatibility
            if (originConfig.s3OriginSource.originAccessIdentity) {
                // grant CloudFront OriginAccessIdentity read access to S3 bucket
                originConfig.s3OriginSource.s3BucketSource.grantRead(originConfig.s3OriginSource.originAccessIdentity);
                s3OriginConfig = {
                    originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`,
                };
            }
            else {
                s3OriginConfig = {};
            }
        }
        const connectionAttempts = (_j = originConfig.connectionAttempts) !== null && _j !== void 0 ? _j : 3;
        if (connectionAttempts < 1 || 3 < connectionAttempts || !Number.isInteger(connectionAttempts)) {
            throw new Error('connectionAttempts: You can specify 1, 2, or 3 as the number of attempts.');
        }
        const connectionTimeout = (originConfig.connectionTimeout || cdk.Duration.seconds(10)).toSeconds();
        if (connectionTimeout < 1 || 10 < connectionTimeout || !Number.isInteger(connectionTimeout)) {
            throw new Error('connectionTimeout: You can specify a number of seconds between 1 and 10 (inclusive).');
        }
        const originProperty = {
            id: originId,
            domainName: originConfig.s3OriginSource
                ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                : originConfig.customOriginSource.domainName,
            originPath: (_m = (_k = originConfig.originPath) !== null && _k !== void 0 ? _k : (_l = originConfig.customOriginSource) === null || _l === void 0 ? void 0 : _l.originPath) !== null && _m !== void 0 ? _m : (_o = originConfig.s3OriginSource) === null || _o === void 0 ? void 0 : _o.originPath,
            originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
            s3OriginConfig,
            customOriginConfig: originConfig.customOriginSource
                ? {
                    httpPort: originConfig.customOriginSource.httpPort || 80,
                    httpsPort: originConfig.customOriginSource.httpsPort || 443,
                    originKeepaliveTimeout: (originConfig.customOriginSource.originKeepaliveTimeout &&
                        originConfig.customOriginSource.originKeepaliveTimeout.toSeconds()) ||
                        5,
                    originReadTimeout: (originConfig.customOriginSource.originReadTimeout &&
                        originConfig.customOriginSource.originReadTimeout.toSeconds()) ||
                        30,
                    originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy ||
                        distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
                    originSslProtocols: originConfig.customOriginSource
                        .allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                }
                : undefined,
            connectionAttempts,
            connectionTimeout,
        };
        return originProperty;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,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