"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheBehavior = void 0;
const iam = require("@aws-cdk/aws-iam");
const distribution_1 = require("../distribution");
/**
 * Allows configuring a variety of CloudFront functionality for a given URL path pattern.
 *
 * Note: This really should simply by called 'Behavior', but this name is already taken by the legacy
 * CloudFrontWebDistribution implementation.
 */
class CacheBehavior {
    constructor(originId, props) {
        this.props = props;
        this.originId = originId;
        this.validateEdgeLambdas(props.edgeLambdas);
        this.grantEdgeLambdaFunctionExecutionRole(props.edgeLambdas);
    }
    /**
     * Creates and returns the CloudFormation representation of this behavior.
     * This renders as a "CacheBehaviorProperty" regardless of if this is a default
     * cache behavior or not, as the two are identical except that the pathPattern
     * is omitted for the default cache behavior.
     *
     * @internal
     */
    _renderBehavior() {
        var _a, _b, _c, _d;
        return {
            pathPattern: this.props.pathPattern,
            targetOriginId: this.originId,
            allowedMethods: (_a = this.props.allowedMethods) === null || _a === void 0 ? void 0 : _a.methods,
            cachedMethods: (_b = this.props.cachedMethods) === null || _b === void 0 ? void 0 : _b.methods,
            compress: this.props.compress,
            forwardedValues: {
                queryString: (_c = this.props.forwardQueryString) !== null && _c !== void 0 ? _c : false,
                queryStringCacheKeys: this.props.forwardQueryStringCacheKeys,
            },
            smoothStreaming: this.props.smoothStreaming,
            viewerProtocolPolicy: (_d = this.props.viewerProtocolPolicy) !== null && _d !== void 0 ? _d : distribution_1.ViewerProtocolPolicy.ALLOW_ALL,
            lambdaFunctionAssociations: this.props.edgeLambdas
                ? this.props.edgeLambdas.map(edgeLambda => ({
                    lambdaFunctionArn: edgeLambda.functionVersion.edgeArn,
                    eventType: edgeLambda.eventType.toString(),
                    includeBody: edgeLambda.includeBody,
                }))
                : undefined,
        };
    }
    validateEdgeLambdas(edgeLambdas) {
        const includeBodyEventTypes = [distribution_1.LambdaEdgeEventType.ORIGIN_REQUEST, distribution_1.LambdaEdgeEventType.VIEWER_REQUEST];
        if (edgeLambdas && edgeLambdas.some(lambda => lambda.includeBody && !includeBodyEventTypes.includes(lambda.eventType))) {
            throw new Error('\'includeBody\' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types.');
        }
    }
    grantEdgeLambdaFunctionExecutionRole(edgeLambdas) {
        if (!edgeLambdas || edgeLambdas.length === 0) {
            return;
        }
        edgeLambdas.forEach((edgeLambda) => {
            const role = edgeLambda.functionVersion.role;
            if (role && role instanceof iam.Role && role.assumeRolePolicy) {
                role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                    actions: ['sts:AssumeRole'],
                    principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                }));
            }
        });
    }
}
exports.CacheBehavior = CacheBehavior;
//# sourceMappingURL=data:application/json;base64,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