import argparse
import logging
import sys

from . import transient


def parse_arguments() -> argparse.Namespace:
    # If we use the default 'usage' generated by argparse, it doesn't show
    # the '--' (though it does work). To avoid confusion, just use our own.
    parser = argparse.ArgumentParser(prog='transient',
                                     description='A simple libvirt/vagrant alternative',
                                     usage="transient [OPTIONS] -- [ARG [ARG ...]]")

    parser.add_argument('-version', action='version', version='%(prog)s 0.1')

    parser.add_argument('-v', '--verbose', action='count', default=0,
                        help="Verbosity level (repeat to be more verbose)")

    parser.add_argument('-name', help='Set the vm name', required=True)

    parser.add_argument('-image', metavar='IMG', nargs='+', action='extend',
                        help='Disk image to use (this option can be repeated)')

    parser.add_argument('-ssh-console', '-s', action='store_const', const=True,
                        default=False, help='Use an ssh connection instead of the serial console')

    parser.add_argument('-ssh-user', '-u', default="vagrant", help='User to pass to SSH')

    parser.add_argument('-sync-before', '-b', nargs='+', action='extend',
                        help='Sync a host path to a guest path before starting the guest')

    parser.add_argument('-sync-after', '-a', nargs='+', action='extend',
                        help='Sync a guest path to a host path after stopping the guest')

    parser.add_argument('qemu_args', metavar='ARG', nargs='*',
                        help='Arguments passed directly to QEMU')
    return parser.parse_args()


def main() -> None:
    args = parse_arguments()

    log_level = logging.ERROR
    if args.verbose == 1:
        log_level = logging.WARNING
    elif args.verbose == 2:
        log_level = logging.INFO
    elif args.verbose >= 3:
        log_level = logging.DEBUG
    logging.basicConfig(level=log_level, format='%(asctime)s:%(levelname)s:%(message)s',
                        datefmt='%Y-%m-%d %H:%M:%S')

    logging.debug("Parsed arguments: {}".format(args))

    trans = transient.TransientVm(config=vars(args))
    returncode = trans.run()
    sys.exit(returncode)
