"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const core_1 = require("@aws-cdk/core");
const codecommit_generated_1 = require("./codecommit.generated");
/**
 * Represents a reference to a CodeCommit Repository.
 *
 * If you want to create a new Repository managed alongside your CDK code,
 * use the {@link Repository} class.
 *
 * If you want to reference an already existing Repository,
 * use the {@link Repository.import} method.
 */
class RepositoryBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for repository events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.codecommit'],
            resources: [this.repositoryArn]
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a "CodeCommit
     * Repository State Change" event occurs.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeCommit Repository State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * created (i.e. a new branch/tag is created) to the repository.
     */
    onReferenceCreated(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceCreated'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * updated (i.e. a commit is pushed to an existing or new branch) from the repository.
     */
    onReferenceUpdated(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceCreated', 'referenceUpdated'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is
     * delete (i.e. a branch/tag is deleted) from the repository.
     */
    onReferenceDeleted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceDeleted'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a pull request state is changed.
     */
    onPullRequestStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Pull Request State Change'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a pull request.
     */
    onCommentOnPullRequest(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Comment on Pull Request'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a commit.
     */
    onCommentOnCommit(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Comment on Commit'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a commit is pushed to a branch.
     */
    onCommit(id, options = {}) {
        const rule = this.onReferenceUpdated(id, options);
        if (options.branches) {
            rule.addEventPattern({ detail: { referenceName: options.branches } });
        }
        return rule;
    }
}
/**
 * Provides a CodeCommit Repository
 */
class Repository extends RepositoryBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.triggers = new Array();
        this.repository = new codecommit_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: props.repositoryName,
            repositoryDescription: props.description,
            triggers: core_1.Lazy.anyValue({ produce: () => this.triggers }, { omitEmptyArray: true }),
        });
        this.repositoryName = this.getResourceNameAttribute(this.repository.attrName);
        this.repositoryArn = this.getResourceArnAttribute(this.repository.attrArn, {
            service: 'codecommit',
            resource: this.physicalName,
        });
    }
    /**
     * Imports a codecommit repository.
     * @param repositoryArn (e.g. `arn:aws:codecommit:us-east-1:123456789012:MyDemoRepo`)
     */
    static fromRepositoryArn(scope, id, repositoryArn) {
        const stack = core_1.Stack.of(scope);
        const repositoryName = stack.parseArn(repositoryArn).resource;
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryArn = repositoryArn;
                this.repositoryName = repositoryName;
                this.repositoryCloneUrlHttp = Repository.makeCloneUrl(stack, repositoryName, 'https');
                this.repositoryCloneUrlSsh = Repository.makeCloneUrl(stack, repositoryName, 'ssh');
            }
        }
        return new Import(scope, id);
    }
    static fromRepositoryName(scope, id, repositoryName) {
        const stack = core_1.Stack.of(scope);
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = Repository.arnForLocalRepository(repositoryName, scope);
                this.repositoryCloneUrlHttp = Repository.makeCloneUrl(stack, repositoryName, 'https');
                this.repositoryCloneUrlSsh = Repository.makeCloneUrl(stack, repositoryName, 'ssh');
            }
        }
        return new Import(scope, id);
    }
    static makeCloneUrl(stack, repositoryName, protocol) {
        return `${protocol}://git-codecommit.${stack.region}.${stack.urlSuffix}/v1/repos/${repositoryName}`;
    }
    static arnForLocalRepository(repositoryName, scope) {
        return core_1.Stack.of(scope).formatArn({
            service: 'codecommit',
            resource: repositoryName,
        });
    }
    get repositoryCloneUrlHttp() {
        return this.repository.attrCloneUrlHttp;
    }
    get repositoryCloneUrlSsh() {
        return this.repository.attrCloneUrlSsh;
    }
    /**
     * Create a trigger to notify another service to run actions on repository events.
     * @param arn   Arn of the resource that repository events will notify
     * @param options Trigger options to run actions
     */
    notify(arn, options) {
        let evt = options && options.events;
        if (evt && evt.length > 1 && evt.indexOf(RepositoryEventTrigger.ALL) > -1) {
            evt = [RepositoryEventTrigger.ALL];
        }
        const customData = options && options.customData;
        const branches = options && options.branches;
        let name = options && options.name;
        if (!name) {
            name = this.node.path + '/' + arn;
        }
        if (this.triggers.find(prop => prop.name === name)) {
            throw new Error(`Unable to set repository trigger named ${name} because trigger names must be unique`);
        }
        this.triggers.push({
            destinationArn: arn,
            name,
            customData,
            branches,
            events: evt || [RepositoryEventTrigger.ALL],
        });
        return this;
    }
}
exports.Repository = Repository;
/**
 * Repository events that will cause the trigger to run actions in another service.
 */
var RepositoryEventTrigger;
(function (RepositoryEventTrigger) {
    RepositoryEventTrigger["ALL"] = "all";
    RepositoryEventTrigger["UPDATE_REF"] = "updateReference";
    RepositoryEventTrigger["CREATE_REF"] = "createReference";
    RepositoryEventTrigger["DELETE_REF"] = "deleteReference";
})(RepositoryEventTrigger = exports.RepositoryEventTrigger || (exports.RepositoryEventTrigger = {}));
//# sourceMappingURL=data:application/json;base64,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