#!/usr/bin/env python
"""
Titanium with 256 atoms and k-point sampling.
GS calculations with paralkgb==1 and wfoptalg in [default, 1]
"""
import sys
import operator
import numpy as np
import abipy.abilab as abilab
import abipy.flowtk as flowtk
import abipy.data as abidata

from functools import reduce
from itertools import product
from abipy.benchmarks import bench_main, BenchmarkFlow


def make_input(paw=True):
    """
    Titanium with 256 atoms.
    GS calculations with paralkgb==1
    """
    pseudos = abidata.pseudos("ti.paw") if paw else abidata.pseudos("Ti-sp.psp8")

    # Atomic Positions.
    xred = np.fromstring("""
0.00000000000000 0.00000000000000 0.00000000000000
0.06250000000000 0.12500000000000 0.12500000000000
0.00000000000000 0.00000000000000 0.25000000000000
0.06250000000000 0.12500000000000 0.37500000000000
0.00000000000000 0.00000000000000 0.50000000000000
0.06250000000000 0.12500000000000 0.62500000000000
0.00000000000000 0.00000000000000 0.75000000000000
0.06250000000000 0.12500000000000 0.87500000000000
0.00000000000000 0.25000000000000 0.00000000000000
0.06250000000000 0.37500000000000 0.12500000000000
0.00000000000000 0.25000000000000 0.25000000000000
0.06250000000000 0.37500000000000 0.37500000000000
0.00000000000000 0.25000000000000 0.50000000000000
0.06250000000000 0.37500000000000 0.62500000000000
0.00000000000000 0.25000000000000 0.75000000000000
0.06250000000000 0.37500000000000 0.87500000000000
0.00000000000000 0.50000000000000 0.00000000000000
0.06250000000000 0.62500000000000 0.12500000000000
0.00000000000000 0.50000000000000 0.25000000000000
0.06250000000000 0.62500000000000 0.37500000000000
0.00000000000000 0.50000000000000 0.50000000000000
0.06250000000000 0.62500000000000 0.62500000000000
0.00000000000000 0.50000000000000 0.75000000000000
0.06250000000000 0.62500000000000 0.87500000000000
0.00000000000000 0.75000000000000 0.00000000000000
0.06250000000000 0.87500000000000 0.12500000000000
0.00000000000000 0.75000000000000 0.25000000000000
0.06250000000000 0.87500000000000 0.37500000000000
0.00000000000000 0.75000000000000 0.50000000000000
0.06250000000000 0.87500000000000 0.62500000000000
0.00000000000000 0.75000000000000 0.75000000000000
0.06250000000000 0.87500000000000 0.87500000000000
0.12500000000000 0.00000000000000 0.00000000000000
0.18750000000000 0.12500000000000 0.12500000000000
0.12500000000000 0.00000000000000 0.25000000000000
0.18750000000000 0.12500000000000 0.37500000000000
0.12500000000000 0.00000000000000 0.50000000000000
0.18750000000000 0.12500000000000 0.62500000000000
0.12500000000000 0.00000000000000 0.75000000000000
0.18750000000000 0.12500000000000 0.87500000000000
0.12500000000000 0.25000000000000 0.00000000000000
0.18750000000000 0.37500000000000 0.12500000000000
0.12500000000000 0.25000000000000 0.25000000000000
0.18750000000000 0.37500000000000 0.37500000000000
0.12500000000000 0.25000000000000 0.50000000000000
0.18750000000000 0.37500000000000 0.62500000000000
0.12500000000000 0.25000000000000 0.75000000000000
0.18750000000000 0.37500000000000 0.87500000000000
0.12500000000000 0.50000000000000 0.00000000000000
0.18750000000000 0.62500000000000 0.12500000000000
0.12500000000000 0.50000000000000 0.25000000000000
0.18750000000000 0.62500000000000 0.37500000000000
0.12500000000000 0.50000000000000 0.50000000000000
0.18750000000000 0.62500000000000 0.62500000000000
0.12500000000000 0.50000000000000 0.75000000000000
0.18750000000000 0.62500000000000 0.87500000000000
0.12500000000000 0.75000000000000 0.00000000000000
0.18750000000000 0.87500000000000 0.12500000000000
0.12500000000000 0.75000000000000 0.25000000000000
0.18750000000000 0.87500000000000 0.37500000000000
0.12500000000000 0.75000000000000 0.50000000000000
0.18750000000000 0.87500000000000 0.62500000000000
0.12500000000000 0.75000000000000 0.75000000000000
0.18750000000000 0.87500000000000 0.87500000000000
0.25000000000000 0.00000000000000 0.00000000000000
0.31250000000000 0.12500000000000 0.12500000000000
0.25000000000000 0.00000000000000 0.25000000000000
0.31250000000000 0.12500000000000 0.37500000000000
0.25000000000000 0.00000000000000 0.50000000000000
0.31250000000000 0.12500000000000 0.62500000000000
0.25000000000000 0.00000000000000 0.75000000000000
0.31250000000000 0.12500000000000 0.87500000000000
0.25000000000000 0.25000000000000 0.00000000000000
0.31250000000000 0.37500000000000 0.12500000000000
0.25000000000000 0.25000000000000 0.25000000000000
0.31250000000000 0.37500000000000 0.37500000000000
0.25000000000000 0.25000000000000 0.50000000000000
0.31250000000000 0.37500000000000 0.62500000000000
0.25000000000000 0.25000000000000 0.75000000000000
0.31250000000000 0.37500000000000 0.87500000000000
0.25000000000000 0.50000000000000 0.00000000000000
0.31250000000000 0.62500000000000 0.12500000000000
0.25000000000000 0.50000000000000 0.25000000000000
0.31250000000000 0.62500000000000 0.37500000000000
0.25000000000000 0.50000000000000 0.50000000000000
0.31250000000000 0.62500000000000 0.62500000000000
0.25000000000000 0.50000000000000 0.75000000000000
0.31250000000000 0.62500000000000 0.87500000000000
0.25000000000000 0.75000000000000 0.00000000000000
0.31250000000000 0.87500000000000 0.12500000000000
0.25000000000000 0.75000000000000 0.25000000000000
0.31250000000000 0.87500000000000 0.37500000000000
0.25000000000000 0.75000000000000 0.50000000000000
0.31250000000000 0.87500000000000 0.62500000000000
0.25000000000000 0.75000000000000 0.75000000000000
0.31250000000000 0.87500000000000 0.87500000000000
0.37500000000000 0.00000000000000 0.00000000000000
0.43750000000000 0.12500000000000 0.12500000000000
0.37500000000000 0.00000000000000 0.25000000000000
0.43750000000000 0.12500000000000 0.37500000000000
0.37500000000000 0.00000000000000 0.50000000000000
0.43750000000000 0.12500000000000 0.62500000000000
0.37500000000000 0.00000000000000 0.75000000000000
0.43750000000000 0.12500000000000 0.87500000000000
0.37500000000000 0.25000000000000 0.00000000000000
0.43750000000000 0.37500000000000 0.12500000000000
0.37500000000000 0.25000000000000 0.25000000000000
0.43750000000000 0.37500000000000 0.37500000000000
0.37500000000000 0.25000000000000 0.50000000000000
0.43750000000000 0.37500000000000 0.62500000000000
0.37500000000000 0.25000000000000 0.75000000000000
0.43750000000000 0.37500000000000 0.87500000000000
0.37500000000000 0.50000000000000 0.00000000000000
0.43750000000000 0.62500000000000 0.12500000000000
0.37500000000000 0.50000000000000 0.25000000000000
0.43750000000000 0.62500000000000 0.37500000000000
0.37500000000000 0.50000000000000 0.50000000000000
0.43750000000000 0.62500000000000 0.62500000000000
0.37500000000000 0.50000000000000 0.75000000000000
0.43750000000000 0.62500000000000 0.87500000000000
0.37500000000000 0.75000000000000 0.00000000000000
0.43750000000000 0.87500000000000 0.12500000000000
0.37500000000000 0.75000000000000 0.25000000000000
0.43750000000000 0.87500000000000 0.37500000000000
0.37500000000000 0.75000000000000 0.50000000000000
0.43750000000000 0.87500000000000 0.62500000000000
0.37500000000000 0.75000000000000 0.75000000000000
0.43750000000000 0.87500000000000 0.87500000000000
0.50066333333333 0.43795366666666 0.20604666666667
0.58978833333333 0.17685666666666 0.24527666666667
0.59038833333333 0.04865666666666 0.46713866666667
0.57337833333333 0.36896766666666 0.44071566666667
0.49437333333333 0.29799666666666 0.62702666666667
0.58165333333333 0.34787366666666 0.95079666666667
0.54251333333333 0.10940666666666 0.84782666666667
0.67878919833333 0.40517666666666 0.06963666666667
0.59048833333333 0.45379666666666 0.10264666666667
0.60238833333333 0.47341066666666 0.31733066666667
0.50515833333333 0.27181666666666 0.25024666666667
0.60375833333333 0.54262666666666 0.50900866666667
0.59347833333333 0.44882266666666 0.67936666666667
0.67186083333333 0.62119666666666 0.73076666666667
0.63907933333333 0.58259666666666 0.93075666666667
0.67825691833333 0.39313466666666 0.28615666666667
0.55216833333333 0.64702666666666 0.29008666666667
0.65788433333333 0.66224666666666 0.34085666666667
0.54558333333333 0.70723666666666 0.50473966666667
0.59868333333333 0.84143666666666 0.65086666666667
0.56312333333333 0.65245666666666 0.66560666666667
0.59459833333333 0.74256666666666 0.86209666666667
0.54859333333333 0.64326666666666 0.02438666666667
0.61449833333333 0.84178666666666 0.03484666666667
0.50916333333333 0.81220666666666 0.08581666666667
0.96250333333333 0.00988666666666 0.16680666666667
0.61246833333333 0.84705666666666 0.43006566666667
0.58969333333333 0.24281666666666 0.62900666666667
0.52407833333333 0.87938666666666 0.92985666666667
0.63529633333333 0.04594666666666 0.68961666666667
0.55587333333333 0.04528666666666 0.08935666666667
0.66776783333333 0.01468666666666 0.17755666666667
0.70887233333333 0.20161666666666 0.21002666666667
0.92334333333333 0.16592666666666 0.29334666666667
0.67033283333333 0.01914666666666 0.36823266666667
0.65631933333333 0.21883666666666 0.36941666666667
0.62474833333333 0.22426666666666 0.80279666666667
0.66286883333333 0.41444856666666 0.82225666666667
0.62886383333333 0.18910666666666 0.05475666666667
0.71154233333333 0.27103666666666 0.73880666666667
0.73958333333333 0.15453666666666 0.01370666666667
0.75625833333333 0.64794666666666 0.24326666666667
0.76121333333333 0.28914666666666 0.37663266666667
0.72632283333333 0.51671666666666 0.43261966666667
0.67498088333333 0.39143366666666 0.52916666666667
0.76689333333333 0.74595666666666 0.77130666666667
0.74752333333333 0.55142666666666 0.86622666666667
0.74464333333333 0.55552666666666 0.08736666666667
0.64824733333333 0.59911666666666 0.13674666666667
0.83169333333333 0.97781666666666 0.30673666666667
0.66809783333333 0.72582666666666 0.55477666666667
0.76887333333333 0.70689666666666 0.49468866666667
0.74088833333333 0.56763666666666 0.64551666666667
0.79800333333333 0.77201666666666 0.97858666666667
0.69187333333333 0.74064666666666 0.93339666666667
0.71363633333333 0.95878666666666 0.99392666666667
0.65551333333333 0.83387666666666 0.24756666666667
0.75738333333333 0.99888666666666 0.18068666666667
0.74925833333333 0.84733666666666 0.35821766666667
0.68541433333333 0.90449666666666 0.52917666666667
0.68926733333333 0.85888666666666 0.73183666666667
0.75026333333333 0.06949666666666 0.64433666666667
0.60628833333333 0.97677666666666 0.86873666666667
0.69994983333333 0.07068666666666 0.84985666666667
0.87101333333333 0.05882666666666 0.16847666666667
0.84889333333333 0.41975886666666 0.18471666666667
0.93339833333333 0.98165666666666 0.38450366666667
0.67249083333333 0.17969666666666 0.57686666666667
0.85423833333333 0.10957666666666 0.47681066666667
0.82493333333333 0.49135366666666 0.55000666666667
0.78462333333333 0.29096666666666 0.57421666666667
0.76800333333333 0.36121566666666 0.92644666666667
0.81058333333333 0.16971666666666 0.22684666666667
0.90856333333333 0.57026666666666 0.23815666666667
0.76182833333333 0.37614666666666 0.18845666666667
0.88871833333333 0.53080666666666 0.42358146666667
0.85562833333333 0.32263166666666 0.36426866666667
0.83515833333333 0.59152666666666 0.77291666666667
0.78718833333333 0.38821366666666 0.74611666666667
0.83699333333333 0.68114666666666 0.18497666666667
0.82814333333333 0.73293666666666 0.36987966666667
0.85414333333333 0.87401666666666 0.48911766666667
0.81410333333333 0.50145866666666 0.34903766666667
0.85689833333333 0.67681666666666 0.58823666666667
0.86804333333333 0.78560666666666 0.84977666666667
0.93271333333333 0.57988666666666 0.81896666666667
0.71184083333333 0.77304666666666 0.11347666666667
0.90540833333333 0.74061666666666 0.03022666666667
0.81621333333333 0.85319666666666 0.13512666666667
0.52441333333333 0.08306666666666 0.34711166666667
0.75732333333333 0.03364666666666 0.44157466666667
0.94750833333333 0.01092666666666 0.60645666666667
0.77594833333333 0.83782666666666 0.62102666666667
0.86107833333333 0.04928666666666 0.67310666666667
0.79300333333333 0.97135666666666 0.79084666666667
0.82238833333333 0.01293666666666 0.97004666666667
0.83750833333333 0.23664666666666 0.02062666666667
0.95388333333333 0.34178766666666 0.44811166666667
0.88708833333333 0.27382666666666 0.58093666666667
0.53206333333333 0.20845666666666 0.48642566666667
0.94568833333333 0.16754666666666 0.74426666666667
0.49329333333333 0.31900866666666 0.82460666666667
0.79113333333333 0.15633666666666 0.84792666666667
0.51232833333333 0.25059666666666 0.04522666666667
0.91374333333333 0.47641466666666 0.07637666666667
0.48904333333333 0.52828666666666 0.38552966666667
0.85357333333333 0.27436666666666 0.76343666666667
0.51098333333333 0.49550966666666 0.56943666666667
0.90882833333333 0.41875446666666 0.67028666666667
0.55464333333333 0.52472666666666 0.82758666666667
0.88878333333333 0.39984166666666 0.89648666666667
0.49674833333333 0.46952266666666 0.97251666666667
0.50698333333333 0.81906666666666 0.32010066666667
0.56672333333333 0.96197666666666 0.24366666666667
0.93230333333333 0.73529666666666 0.43107266666667
0.50525833333333 0.82783666666666 0.66778666666667
0.94398833333333 0.65432666666666 0.66346666666667
0.48605333333333 0.74336666666666 0.85736666666667
0.83221333333333 0.55263666666666 0.00047666666667
0.47925833333333 0.63845666666666 0.16634666666667
0.91936333333333 0.84625666666666 0.22666666666667
0.91682333333333 0.28260666666666 0.13265666666667
0.51926833333333 0.92371666666666 0.50915766666667
0.53698833333333 0.07929666666666 0.66035666666667
0.89665333333333 0.84174666666666 0.67481666666667
0.93254833333333 0.97751666666666 0.81711666666667
0.91770833333333 0.93970666666666 0.99482666666667
0.92680333333333 0.14577666666666 0.95922666666667
""", sep=" ").reshape((-1,3))

    # Crystal structure.
    structure = abilab.Structure.from_abivars(
        acell=[50.4, 25.2, 25.2],
        rprim=np.eye(3),
        typat=256*[1],
        znucl=22,
        xred=xred,
    )

    inp = abilab.AbinitInput(structure, pseudos)
    inp.set_vars(
        # SCF algorithm
        paral_kgb=1,
        wfoptalg=1,
        fftalg=402,
        #fftalg-302,  # To use FFTW instead of ABINIT FFT

        # Basis set
        ecut=5,
        pawecutdg=10,

        # SCF cycle
        tolvrs=1.e-3,
        nstep=20,

        # K-Points and symmetries
        nkpt=2,
        kpt=[3*[0.5], [0, 0, 0.5]],
        kptopt=0,
        istwfk="*1",
        nsym=0,
        chksymbreak=0,
        chkprim=0,
        pawovlp=-1,

        # bands and occupation scheme.
        nband=2048,
        occopt=3,
        tsmear="1800. K",

        # IO
        optforces=2,
        optstress=1,
        prtwf=0,
        prtden=0,
        prteig=0,
        timopt=-1,
    )

    return inp


def build_flow(options):
    flow = BenchmarkFlow(workdir=options.get_workdir(__file__), remove=options.remove)

    template = make_input()

    # Processor distribution.
    pconfs = [
     dict(npkpt=2, npband=8 , npfft=8),   # 128
     dict(npkpt=2, npband=8 , npfft=16),  # 256
     dict(npkpt=2, npband=16, npfft=16),  # 512
     dict(npkpt=2, npband=16, npfft=32),  # 1024
     dict(npkpt=2, npband=32, npfft=32),  # 2048
    ]

    for wfoptalg in [None, 1]:
        work = flowtk.Work()
        for d, omp_threads in product(pconfs, options.omp_list):
            mpi_procs = reduce(operator.mul, d.values(), 1)
            if not options.accept_mpi_omp(mpi_procs, omp_threads): continue
            manager = options.manager.new_with_fixed_mpi_omp(mpi_procs, omp_threads)
            if options.verbose: print("wfoptalg:", wfoptalg, "done with MPI_PROCS:", mpi_procs, "and:", d)
            inp = template.new_with_vars(d, wfoptalg=wfoptalg, np_slk=32)
            #inp.abivalidate()
            work.register_scf_task(inp, manager=manager)

        flow.register_work(work)

    return flow.allocate()


@bench_main
def main(options):
    if options.info:
        # print doc string and exit.
        print(__doc__)
        return
    return build_flow(options)


if __name__ == "__main__":
    sys.exit(main())
