"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolEmail = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const punycode_1 = require("punycode/");
/**
 * Configure how Cognito sends emails
 */
class UserPoolEmail {
    /**
     * Send email using Cognito
     */
    static withCognito(replyTo) {
        return new CognitoEmail(replyTo);
    }
    /**
     * Send email using SES
     */
    static withSES(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cognito_UserPoolSESOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.withSES);
            }
            throw error;
        }
        return new SESEmail(options);
    }
}
exports.UserPoolEmail = UserPoolEmail;
_a = JSII_RTTI_SYMBOL_1;
UserPoolEmail[_a] = { fqn: "@aws-cdk/aws-cognito.UserPoolEmail", version: "1.172.0" };
class CognitoEmail extends UserPoolEmail {
    constructor(replyTo) {
        super();
        this.replyTo = replyTo;
    }
    _bind(_scope) {
        return {
            replyToEmailAddress: encodeAndTest(this.replyTo),
            emailSendingAccount: 'COGNITO_DEFAULT',
        };
    }
}
class SESEmail extends UserPoolEmail {
    constructor(options) {
        super();
        this.options = options;
    }
    _bind(scope) {
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region) && !this.options.sesRegion) {
            throw new Error('Your stack region cannot be determined so "sesRegion" is required in SESOptions');
        }
        let from = this.options.fromEmail;
        if (this.options.fromName) {
            from = `${this.options.fromName} <${this.options.fromEmail}>`;
        }
        if (this.options.sesVerifiedDomain) {
            const domainFromEmail = this.options.fromEmail.split('@').pop();
            if (domainFromEmail !== this.options.sesVerifiedDomain) {
                throw new Error('"fromEmail" contains a different domain than the "sesVerifiedDomain"');
            }
        }
        return {
            from: encodeAndTest(from),
            replyToEmailAddress: encodeAndTest(this.options.replyTo),
            configurationSet: this.options.configurationSetName,
            emailSendingAccount: 'DEVELOPER',
            sourceArn: core_1.Stack.of(scope).formatArn({
                service: 'ses',
                resource: 'identity',
                resourceName: encodeAndTest(this.options.sesVerifiedDomain ?? this.options.fromEmail),
                region: this.options.sesRegion ?? region,
            }),
        };
    }
}
function encodeAndTest(input) {
    if (input) {
        const local = input.split('@')[0];
        if (!/[\p{ASCII}]+/u.test(local)) {
            throw new Error('the local part of the email address must use ASCII characters only');
        }
        return punycode_1.toASCII(input);
    }
    else {
        return undefined;
    }
}
//# sourceMappingURL=data:application/json;base64,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