# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nlpiper', 'nlpiper.core', 'nlpiper.transformers']

package_data = \
{'': ['*']}

install_requires = \
['pydantic>=1.8.1,<2.0.0']

extras_require = \
{'all': ['bs4>=0.0.1,<0.0.2',
         'cyhunspell>=2.0.2,<3.0.0',
         'nltk>=3.5,<4.0',
         'sacremoses>=0.0.43,<0.0.44',
         'stanza>=1.3.0,<2.0.0',
         'gensim>=4.1.2,<5.0.0',
         'numpy>=1.22.2,<2.0.0'],
 'bs4': ['bs4>=0.0.1,<0.0.2'],
 'gensim': ['gensim>=4.1.2,<5.0.0'],
 'hunspell': ['cyhunspell>=2.0.2,<3.0.0'],
 'nltk': ['nltk>=3.5,<4.0'],
 'numpy': ['numpy>=1.22.2,<2.0.0'],
 'sacremoses': ['sacremoses>=0.0.43,<0.0.44'],
 'stanza': ['stanza>=1.3.0,<2.0.0']}

setup_kwargs = {
    'name': 'nlpiper',
    'version': '0.2.0',
    'description': 'NLPiper, a lightweight package integrated with a universe of frameworks to pre-process documents.',
    'long_description': '<p align="center">\n  <img src="https://raw.githubusercontent.com/tomassosorio/NLPiper/main/docs/imgs/nlpiper.png" />\n</p>\n\n[![Test](https://github.com/tomassosorio/NLPiper/actions/workflows/test.yml/badge.svg)](https://github.com/tomassosorio/NLPiper/actions/workflows/test.yml)\n[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)\n[![codecov](https://codecov.io/gh/tomassosorio/NLPiper/branch/main/graph/badge.svg?token=PK513BHBVC)](https://codecov.io/gh/tomassosorio/NLPiper)\n![Package Version](https://img.shields.io/pypi/v/NLPiper)\n![Python Version](https://img.shields.io/pypi/pyversions/NLPiper)\n\nNLPiper is a package that agglomerates different NLP tools and applies their transformations in the target document.\n\n## Goal\nLightweight package integrated with a universe of frameworks to pre-process documents.\n\n---\n## Install\n\nThe package can be installed using `pip`:\n\n`pip install nlpiper`\n\nFor all transforms be available:\n`pip install \'nlpiper[all]\'`, otherwise, just install the packages needed.\n\n## Usage\n\n### Define a Pipeline:\n\n```python\n>>> from nlpiper.core import Compose\n>>> from nlpiper.transformers import cleaners, normalizers, tokenizers\n>>> pipeline = Compose([\n...                    cleaners.CleanNumber(),\n...                    tokenizers.BasicTokenizer(),\n...                    normalizers.CaseTokens()\n... ])\n>>> pipeline\nCompose([CleanNumber(), BasicTokenizer(), CaseTokens(mode=\'lower\')])\n```\n\n### Generate a Document and Document structure:\n```python\n>>> from nlpiper.core import Document\n>>> doc = Document("The following character is a number: 1 and the next one is not a.")\n>>> doc\nDocument(\n    original=\'The following character is a number: 1 and the next one is not a.\', \n    cleaned=\'The following character is a number: 1 and the next one is not a.\', \n    tokens=None,\n    embedded=None,\n    steps=[]\n)\n```\n\n### Apply Pipeline to a Document:\n```python\n>>> doc = pipeline(doc)\n>>> doc\nDocument(\n    original=\'The following character is a number: 1 and the next one is not a.\', \n    cleaned=\'The following character is a number:  and the next one is not a.\', \n    tokens=[\n        Token(original=\'The\', cleaned=\'the\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'following\', cleaned=\'following\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'character\', cleaned=\'character\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'is\', cleaned=\'is\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'a\', cleaned=\'a\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'number:\', cleaned=\'number:\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'and\', cleaned=\'and\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'the\', cleaned=\'the\', lemma=None, stem=None, embedded=None), \n        Token(original=\'next\', cleaned=\'next\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'one\', cleaned=\'one\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'is\', cleaned=\'is\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'not\', cleaned=\'not\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'a.\', cleaned=\'a.\', lemma=None, stem=None, ner=None, embedded=None)\n    ],\n    embedded=None,\n    steps=[\'CleanNumber()\', \'BasicTokenizer()\', "CaseTokens(mode=\'lower\')"]\n)\n```\n\n### Available Transformers\n#### Cleaners\nClean document as a whole, e.g. remove HTML, remove accents, remove emails, etc.\n\n- `CleanURL`: remove URL from the text.\n- `CleanEmail`: remove email from the text.\n- `CleanNumber`: remove numbers from text.\n- `CleanPunctuation`: remove punctuation from text.\n- `CleanEOF`: remove end of file from text.\n- `CleanMarkup`: remove HTML or XML from text.\n- `CleanAccents`: remove accents from the text.\n\n#### Tokenizers\nTokenize a document after cleaning is done (Split document into tokens)\n\n- `BasicTokenizer`: Split tokens by spaces in the text.\n- `MosesTokenizer`: Split tokens using Moses tokenizer (https://github.com/alvations/sacremoses)\n- `StanzaTokenizer`: Split tokens using Stanza tokenizer (https://github.com/stanfordnlp/stanza)\n\n#### Normalizer\nApplies on the token level, e.g. remove stop-words, spell-check, etc.\n\n- `CaseTokens`: lower or upper case all tokens.\n- `RemovePunctuation`: Remove punctuation from resulting tokens.\n- `RemoveStopWords`: Remove stop-words as tokens.\n- `VocabularyFilter`: Only allow tokens from a pre-defined vocabulary.\n- `Stemmer`: Get the stem from the tokens.\n- `SpellCheck`: Spell check the token, if given max distance will calculate the Levenshtein distance from the token with\nthe suggested word and if lower the token is replaced by the suggestion else will keep the token. If no maximum distance is given if the\nword is not correctly spelt then will be replaced by an empty string.\n\n#### Embeddings\nApplies on the token level, converting words by embeddings\n\n- `GensimEmbeddings`: Use Gensim word embeddings.\n\n#### Document\n`Document` is a dataclass that contains all the information used during text preprocessing.\n\nDocument attributes:\n- `original`: original text to be processed.\n- `cleaned`: original text to be processed when document is initiated and then attribute which `Cleaners` and `Tokenizers` work.\n- `tokens`: list of tokens that where obtained using a `Tokenizer`.\n- `steps`: list of transforms applied on the document.\n- `embedded`: document embedding.\n\n`token`:\n- `original`: original token.\n- `cleaned`: original token at initiation, then modified according with `Normalizers`.\n- `lemma`: token lemma (need to use a normalizer or tokenizer to obtain).\n- `stem`: token stem (need to use a normalizer to obtain).\n- `ner`: token entity (need to use a normalizer or tokenizer to obtain).\n- `embedded`: token embedding.\n\n#### Compose\nCompose applies the chosen transformers into a given document.\nIt restricts the order that the transformers can be applied, first are the Cleaners, then the Tokenizers and lastly\nthe Normalizers and Embeddings.\n\nIt is possible to create a compose using the steps from a processed document:\n```python\n>>> doc.steps\n[\'CleanNumber()\', \'BasicTokenizer()\', "CaseTokens(mode=\'lower\')"]\n>>> new_pipeline = Compose.create_from_steps(doc.steps)\n>>> new_pipeline\nCompose([CleanNumber(), BasicTokenizer(), CaseTokens(mode=\'lower\')])\n```\nIt is also possible to rollback the steps applied to a document:\n```python\n>>> new_doc = Compose.rollback_document(doc, 2)\n>>> new_doc\nDocument(\n    original=\'The following character is a number: 1 and the next one is not a.\', \n    cleaned=\'The following character is a number:  and the next one is not a.\', \n    tokens=None,\n    embedded=None,\n    steps=[\'CleanNumber()\']\n)\n>>> doc\nDocument(\n    original=\'The following character is a number: 1 and the next one is not a.\', \n    cleaned=\'The following character is a number:  and the next one is not a.\', \n    tokens=[\n        Token(original=\'The\', cleaned=\'the\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'following\', cleaned=\'following\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'character\', cleaned=\'character\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'is\', cleaned=\'is\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'a\', cleaned=\'a\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'number:\', cleaned=\'number:\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'and\', cleaned=\'and\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'the\', cleaned=\'the\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'next\', cleaned=\'next\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'one\', cleaned=\'one\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'is\', cleaned=\'is\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'not\', cleaned=\'not\', lemma=None, stem=None, ner=None, embedded=None), \n        Token(original=\'a.\', cleaned=\'a.\', lemma=None, stem=None, ner=None, embedded=None)\n    ],\n    embedded=None,\n    steps=[\'CleanNumber()\', \'BasicTokenizer()\', "CaseTokens(mode=\'lower\')"]\n)\n```\n\n---\n\n## Development Installation\n\n```\ngit clone https://github.com/tomassosorio/NLPiper.git\ncd NLPiper\npoetry install\n```\n\n---\n\n## Contributions\n\nAll contributions, bug reports, bug fixes, documentation improvements,\nenhancements and ideas are welcome.\n\nA detailed overview on how to contribute can be found in the\n[contributing guide](CONTRIBUTING.md)\non GitHub.\n\n---\n\n## Issues\n\nGo [here](https://github.com/tomassosorio/NLPiper/issues) to submit feature\nrequests or bugfixes.\n\n---\n\n## License and Credits\n\n`NLPiper` is licensed under the [MIT license](LICENSE) and is written and\nmaintained by Tomás Osório ([@tomassosorio](https://github.com/tomassosorio)), Daniel Ferrari ([@FerrariDG](https://github.com/FerrariDG)) and Carlos Alves ([@cmalves](https://github.com/cmalves))\n',
    'author': 'Tomás Osório',
    'author_email': None,
    'maintainer': 'Carlos Alves, Daniel Ferrari, Tomás Osório',
    'maintainer_email': None,
    'url': 'https://github.com/tomassosorio/NLPiper',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
