"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyCodeConfig = exports.Function = exports.Tracing = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const aws_codeguruprofiler_1 = require("@aws-cdk/aws-codeguruprofiler");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const function_base_1 = require("./function-base");
const function_hash_1 = require("./function-hash");
const handler_1 = require("./handler");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html).
 *
 * @stability stable
 */
var Tracing;
(function (Tracing) {
    Tracing["ACTIVE"] = "Active";
    Tracing["PASS_THROUGH"] = "PassThrough";
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 *
 * @stability stable
 */
class Function extends function_base_1.FunctionBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id, {
            physicalName: props.functionName,
        });
        /**
         * The construct node where permissions are attached.
         *
         * @stability stable
         */
        this.permissionsNode = this.node;
        /**
         * Whether the addPermission() call adds any permissions.
         *
         * True for new Lambdas, false for version $LATEST and imported Lambdas
         * from different accounts.
         *
         * @stability stable
         */
        this.canCreatePermissions = true;
        this.layers = [];
        /**
         * Environment variables for this function
         */
        this.environment = {};
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        this.grantPrincipal = this.role;
        // add additional managed policies when necessary
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.policies) {
                config.policies.forEach(p => {
                    var _b;
                    (_b = this.role) === null || _b === void 0 ? void 0 : _b.addToPrincipalPolicy(p);
                });
            }
        }
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPrincipalPolicy(statement);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props);
        let profilingGroupEnvironmentVariables = {};
        if (props.profilingGroup && props.profiling !== false) {
            this.validateProfiling(props);
            props.profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: core_1.Stack.of(scope).formatArn({
                    service: 'codeguru-profiler',
                    resource: 'profilingGroup',
                    resourceName: props.profilingGroup.profilingGroupName,
                }),
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        else if (props.profiling) {
            this.validateProfiling(props);
            const profilingGroup = new aws_codeguruprofiler_1.ProfilingGroup(this, 'ProfilingGroup', {
                computePlatform: aws_codeguruprofiler_1.ComputePlatform.AWS_LAMBDA,
            });
            profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: profilingGroup.profilingGroupArn,
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        const env = { ...profilingGroupEnvironmentVariables, ...props.environment };
        for (const [key, value] of Object.entries(env)) {
            this.addEnvironment(key, value);
        }
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        let fileSystemConfigs = undefined;
        if (props.filesystem) {
            fileSystemConfigs = [{
                    arn: props.filesystem.config.arn,
                    localMountPath: props.filesystem.config.localMountPath,
                }];
        }
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode,
                imageUri: (_b = code.image) === null || _b === void 0 ? void 0 : _b.imageUri,
            },
            layers: core_1.Lazy.list({ produce: () => this.layers.map(layer => layer.layerVersionArn) }, { omitEmpty: true }),
            handler: props.handler === handler_1.Handler.FROM_IMAGE ? undefined : props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            packageType: props.runtime === runtime_1.Runtime.FROM_IMAGE ? 'Image' : undefined,
            runtime: props.runtime === runtime_1.Runtime.FROM_IMAGE ? undefined : props.runtime.name,
            role: this.role.roleArn,
            // Uncached because calling '_checkEdgeCompatibility', which gets called in the resolve of another
            // Token, actually *modifies* the 'environment' map.
            environment: core_1.Lazy.uncachedAny({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(this.deadLetterQueue),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions,
            imageConfig: undefinedIfNoKeys({
                command: (_c = code.image) === null || _c === void 0 ? void 0 : _c.cmd,
                entryPoint: (_d = code.image) === null || _d === void 0 ? void 0 : _d.entrypoint,
            }),
            kmsKeyArn: (_e = props.environmentEncryption) === null || _e === void 0 ? void 0 : _e.keyArn,
            fileSystemConfigs,
            codeSigningConfigArn: (_f = props.codeSigningConfig) === null || _f === void 0 ? void 0 : _f.codeSigningConfigArn,
        });
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.runtime = props.runtime;
        if (props.layers) {
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, 'LogGroup', logRetention.logGroupArn);
        }
        props.code.bindToResource(resource);
        // Event Invoke Config
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        this.currentVersionOptions = props.currentVersionOptions;
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.dependency) {
                this.node.addDependency(...config.dependency);
            }
        }
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this Lambda function. A new version will be created every time the function's configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     *
     * @stability stable
     */
    get currentVersion() {
        if (this._currentVersion) {
            return this._currentVersion;
        }
        this._currentVersion = new lambda_version_1.Version(this, 'CurrentVersion', {
            lambda: this,
            ...this.currentVersionOptions,
        });
        // override the version's logical ID with a lazy string which includes the
        // hash of the function itself, so a new version resource is created when
        // the function configuration changes.
        const cfn = this._currentVersion.node.defaultChild;
        const originalLogicalId = this.stack.resolve(cfn.logicalId);
        cfn.overrideLogicalId(core_1.Lazy.uncachedString({
            produce: () => {
                const hash = function_hash_1.calculateFunctionHash(this);
                const logicalId = function_hash_1.trimFromStart(originalLogicalId, 255 - 32);
                return `${logicalId}${hash}`;
            },
        }));
        return this._currentVersion;
    }
    /**
     * @stability stable
     */
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined within this stack.
     *
     * @param scope The parent construct.
     * @param id The name of the lambda construct.
     * @param attrs the attributes of the function to import.
     * @stability stable
     */
    static fromFunctionAttributes(scope, id, attrs) {
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                var _b;
                super(s, i);
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.canCreatePermissions = (_b = attrs.sameEnvironment) !== null && _b !== void 0 ? _b : this._isStackAccount();
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroup) {
                    this._connections = new ec2.Connections({
                        securityGroups: [attrs.securityGroup],
                    });
                }
                else if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)],
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda.
     *
     * @stability stable
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas.
     *
     * @default max over 5 minutes
     * @stability stable
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas.
     *
     * @default max over 5 minutes
     * @stability stable
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     *
     * If this is a ref to a Lambda function, this operation results in a no-op.
     *
     * @param key The environment variable key.
     * @param value The environment variable's value.
     * @param options Environment variable options.
     * @stability stable
     */
    addEnvironment(key, value, options) {
        this.environment[key] = { value, ...options };
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     * @stability stable
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        for (const layer of layers) {
            if (this.layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            this.layers.push(layer);
        }
    }
    /**
     * (deprecated) Add a new version for this Lambda.
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked asynchronously.
     * @returns A new Version object.
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name, codeSha256, description, provisionedExecutions, asyncInvokeConfig = {}) {
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
            provisionedConcurrentExecutions: provisionedExecutions,
            ...asyncInvokeConfig,
        });
    }
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     *
     * @stability stable
     */
    get logGroup() {
        if (!this._logGroup) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: logs.RetentionDays.INFINITE,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, `${this.node.id}-LogGroup`, logRetention.logGroupArn);
        }
        return this._logGroup;
    }
    /** @internal */
    _checkEdgeCompatibility() {
        // Check env vars
        const envEntries = Object.entries(this.environment);
        for (const [key, config] of envEntries) {
            if (config.removeInEdge) {
                delete this.environment[key];
                core_1.Annotations.of(this).addInfo(`Removed ${key} environment variable for Lambda@Edge compatibility`);
            }
        }
        const envKeys = Object.keys(this.environment);
        if (envKeys.length !== 0) {
            throw new Error(`The function ${this.node.path} contains environment variables [${envKeys}] and is not compatible with Lambda@Edge. \
Environment variables can be marked for removal when used in Lambda@Edge by setting the \'removeInEdge\' property in the \'addEnvironment()\' API.`);
        }
        return;
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        const variables = {};
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it). For backwards compatibility we do not sort environment variables in case
        // _currentVersion is not defined. Otherwise, this would have invalidated
        // the template, and for example, may cause unneeded updates for nested
        // stacks.
        const keys = this._currentVersion
            ? Object.keys(this.environment).sort()
            : Object.keys(this.environment);
        for (const key of keys) {
            variables[key] = this.environment[key].value;
        }
        return { variables };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        var _b;
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroup && props.securityGroups) {
            throw new Error('Only one of the function props, securityGroup or securityGroups, is allowed');
        }
        if (props.securityGroups) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Lambda Function ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        if (props.filesystem) {
            if (props.filesystem.config.connections) {
                props.filesystem.config.connections.allowDefaultPortFrom(this);
            }
        }
        const allowPublicSubnet = (_b = props.allowPublicSubnet) !== null && _b !== void 0 ? _b : false;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId) && !allowPublicSubnet) {
                throw new Error('Lambda Functions in a public subnet can NOT access the internet. ' +
                    'If you are aware of this limitation and would still like to place the function int a public subnet, set `allowPublicSubnet` to true');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        };
    }
    buildDeadLetterQueue(props) {
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
            retentionPeriod: core_1.Duration.days(14),
        });
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sqs:SendMessage'],
            resources: [deadLetterQueue.queueArn],
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                targetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            mode: props.tracing,
        };
    }
    validateProfiling(props) {
        if (!props.runtime.supportsCodeGuruProfiling) {
            throw new Error(`CodeGuru profiling is not supported by runtime ${props.runtime.name}`);
        }
        if (props.environment && (props.environment.AWS_CODEGURU_PROFILER_GROUP_ARN || props.environment.AWS_CODEGURU_PROFILER_ENABLED)) {
            throw new Error('AWS_CODEGURU_PROFILER_GROUP_ARN and AWS_CODEGURU_PROFILER_ENABLED must not be set when profiling options enabled');
        }
    }
}
exports.Function = Function;
_a = JSII_RTTI_SYMBOL_1;
Function[_a] = { fqn: "@aws-cdk/aws-lambda.Function", version: "1.100.0" };
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, props) {
    // mutually exclusive
    const codeType = [code.inlineCode, code.s3Location, code.image];
    if (codeType.filter(x => !!x).length !== 1) {
        throw new Error('lambda.Code must specify exactly one of: "inlineCode", "s3Location", or "image"');
    }
    if (!!code.image === (props.handler !== handler_1.Handler.FROM_IMAGE)) {
        throw new Error('handler must be `Handler.FROM_IMAGE` when using image asset for Lambda function');
    }
    if (!!code.image === (props.runtime !== runtime_1.Runtime.FROM_IMAGE)) {
        throw new Error('runtime must be `Runtime.FROM_IMAGE` when using image asset for Lambda function');
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !props.runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${props.runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).every(val => val === undefined);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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