"use strict";
var _a, _b, _c, _d, _e, _f, _g;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetImageCode = exports.EcrImageCode = exports.CfnParametersCode = exports.AssetCode = exports.InlineCode = exports.S3Code = exports.Code = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ecr_assets = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
/**
 * Represents the Lambda Handler Code.
 *
 * @stability stable
 */
class Code {
    /**
     * Lambda handler code as an S3 object.
     *
     * @param bucket The S3 bucket.
     * @param key The object key.
     * @param objectVersion Optional S3 object version.
     * @stability stable
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromBucket`
     */
    static bucket(bucket, key, objectVersion) {
        return this.fromBucket(bucket, key, objectVersion);
    }
    /**
     * Inline code for Lambda handler.
     *
     * @param code The actual handler code (limited to 4KiB).
     * @returns `LambdaInlineCode` with inline code.
     * @stability stable
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromInline`
     */
    static inline(code) {
        return this.fromInline(code);
    }
    /**
     * Loads the function code from a local disk path.
     *
     * @param path Either a directory with the Lambda code bundle or a .zip file.
     * @stability stable
     */
    static fromAsset(path, options) {
        return new AssetCode(path, options);
    }
    /**
     * Loads the function code from an asset created by a Docker build.
     *
     * By default, the asset is expected to be located at `/asset` in the
     * image.
     *
     * @param path The path to the directory containing the Docker file.
     * @param options Docker build options.
     * @stability stable
     */
    static fromDockerBuild(path, options = {}) {
        var _h;
        let imagePath = (_h = options.imagePath) !== null && _h !== void 0 ? _h : '/asset/.';
        // ensure imagePath ends with /. to copy the **content** at this path
        if (imagePath.endsWith('/')) {
            imagePath = `${imagePath}.`;
        }
        else if (!imagePath.endsWith('/.')) {
            imagePath = `${imagePath}/.`;
        }
        const assetPath = cdk.DockerImage
            .fromBuild(path, options)
            .cp(imagePath, options.outputPath);
        return new AssetCode(assetPath);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromAsset`
     */
    static asset(path) {
        return this.fromAsset(path);
    }
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @param props optional construction properties of {@link CfnParametersCode}.
     * @returns a new instance of `CfnParametersCode`
     * @stability stable
     */
    static fromCfnParameters(props) {
        return new CfnParametersCode(props);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromCfnParameters`
     */
    static cfnParameters(props) {
        return this.fromCfnParameters(props);
    }
    /**
     * Use an existing ECR image as the Lambda code.
     *
     * @param repository the ECR repository that the image is in.
     * @param props properties to further configure the selected image.
     * @stability stable
     */
    static fromEcrImage(repository, props) {
        return new EcrImageCode(repository, props);
    }
    /**
     * Create an ECR image from the specified asset and bind it as the Lambda code.
     *
     * @param directory the directory from which the asset must be created.
     * @param props properties to further configure the selected image.
     * @stability stable
     */
    static fromAssetImage(directory, props = {}) {
        return new AssetImageCode(directory, props);
    }
    /**
     * Called after the CFN function resource has been created to allow the code class to bind to it.
     *
     * Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     *
     * @stability stable
     */
    bindToResource(_resource, _options) {
        return;
    }
}
exports.Code = Code;
_a = JSII_RTTI_SYMBOL_1;
Code[_a] = { fqn: "@aws-cdk/aws-lambda.Code", version: "1.100.0" };
/**
 * Lambda code from an S3 archive.
 *
 * @stability stable
 */
class S3Code extends Code {
    /**
     * @stability stable
     */
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
_b = JSII_RTTI_SYMBOL_1;
S3Code[_b] = { fqn: "@aws-cdk/aws-lambda.S3Code", version: "1.100.0" };
/**
 * Lambda code from an inline string (limited to 4KiB).
 *
 * @stability stable
 */
class InlineCode extends Code {
    /**
     * @stability stable
     */
    constructor(code) {
        super();
        this.code = code;
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = true;
        if (code.length === 0) {
            throw new Error('Lambda inline code cannot be empty');
        }
        if (code.length > 4096) {
            throw new Error('Lambda source is too large, must be <= 4096 but is ' + code.length);
        }
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(_scope) {
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
_c = JSII_RTTI_SYMBOL_1;
InlineCode[_c] = { fqn: "@aws-cdk/aws-lambda.InlineCode", version: "1.100.0" };
/**
 * Lambda code from a local directory.
 *
 * @stability stable
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     * @stability stable
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = false;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(scope) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.path,
                ...this.options,
            });
        }
        else if (cdk.Stack.of(this.asset) !== cdk.Stack.of(scope)) {
            throw new Error(`Asset is already associated with another stack '${cdk.Stack.of(this.asset).stackName}'. ` +
                'Create a new Code instance for every stack.');
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Called after the CFN function resource has been created to allow the code class to bind to it.
     *
     * Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     *
     * @stability stable
     */
    bindToResource(resource, options = {}) {
        if (!this.asset) {
            throw new Error('bindToResource() must be called after bind()');
        }
        const resourceProperty = options.resourceProperty || 'Code';
        // https://github.com/aws/aws-cdk/issues/1432
        this.asset.addResourceMetadata(resource, resourceProperty);
    }
}
exports.AssetCode = AssetCode;
_d = JSII_RTTI_SYMBOL_1;
AssetCode[_d] = { fqn: "@aws-cdk/aws-lambda.AssetCode", version: "1.100.0" };
/**
 * Lambda code defined using 2 CloudFormation parameters.
 *
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 *
 * @stability stable
 */
class CfnParametersCode extends Code {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        super();
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = false;
        this._bucketNameParam = props.bucketNameParam;
        this._objectKeyParam = props.objectKeyParam;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(scope) {
        if (!this._bucketNameParam) {
            this._bucketNameParam = new cdk.CfnParameter(scope, 'LambdaSourceBucketNameParameter', {
                type: 'String',
            });
        }
        if (!this._objectKeyParam) {
            this._objectKeyParam = new cdk.CfnParameter(scope, 'LambdaSourceObjectKeyParameter', {
                type: 'String',
            });
        }
        return {
            s3Location: {
                bucketName: this._bucketNameParam.valueAsString,
                objectKey: this._objectKeyParam.valueAsString,
            },
        };
    }
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 location
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Location` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param location the location of the object in S3 that represents the Lambda code.
     * @stability stable
     */
    assign(location) {
        const ret = {};
        ret[this.bucketNameParam] = location.bucketName;
        ret[this.objectKeyParam] = location.objectKey;
        return ret;
    }
    /**
     * @stability stable
     */
    get bucketNameParam() {
        if (this._bucketNameParam) {
            return this._bucketNameParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the bucketNameParam property');
        }
    }
    /**
     * @stability stable
     */
    get objectKeyParam() {
        if (this._objectKeyParam) {
            return this._objectKeyParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the objectKeyParam property');
        }
    }
}
exports.CfnParametersCode = CfnParametersCode;
_e = JSII_RTTI_SYMBOL_1;
CfnParametersCode[_e] = { fqn: "@aws-cdk/aws-lambda.CfnParametersCode", version: "1.100.0" };
/**
 * Represents a Docker image in ECR that can be bound as Lambda Code.
 *
 * @stability stable
 */
class EcrImageCode extends Code {
    /**
     * @stability stable
     */
    constructor(repository, props = {}) {
        super();
        this.repository = repository;
        this.props = props;
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = false;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(_) {
        var _h, _j;
        this.repository.grantPull(new iam.ServicePrincipal('lambda.amazonaws.com'));
        return {
            image: {
                imageUri: this.repository.repositoryUriForTag((_j = (_h = this.props) === null || _h === void 0 ? void 0 : _h.tag) !== null && _j !== void 0 ? _j : 'latest'),
                cmd: this.props.cmd,
                entrypoint: this.props.entrypoint,
            },
        };
    }
}
exports.EcrImageCode = EcrImageCode;
_f = JSII_RTTI_SYMBOL_1;
EcrImageCode[_f] = { fqn: "@aws-cdk/aws-lambda.EcrImageCode", version: "1.100.0" };
/**
 * Represents an ECR image that will be constructed from the specified asset and can be bound as Lambda code.
 *
 * @stability stable
 */
class AssetImageCode extends Code {
    /**
     * @stability stable
     */
    constructor(directory, props) {
        super();
        this.directory = directory;
        this.props = props;
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = false;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(scope) {
        const asset = new ecr_assets.DockerImageAsset(scope, 'AssetImage', {
            directory: this.directory,
            ...this.props,
        });
        asset.repository.grantPull(new iam.ServicePrincipal('lambda.amazonaws.com'));
        return {
            image: {
                imageUri: asset.imageUri,
                entrypoint: this.props.entrypoint,
                cmd: this.props.cmd,
            },
        };
    }
}
exports.AssetImageCode = AssetImageCode;
_g = JSII_RTTI_SYMBOL_1;
AssetImageCode[_g] = { fqn: "@aws-cdk/aws-lambda.AssetImageCode", version: "1.100.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29kZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNvZGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFDQSxzREFBc0Q7QUFDdEQsd0NBQXdDO0FBRXhDLG9EQUFvRDtBQUNwRCxxQ0FBcUM7Ozs7OztBQVNyQyxNQUFzQixJQUFJOzs7Ozs7Ozs7SUFPakIsTUFBTSxDQUFDLFVBQVUsQ0FBQyxNQUFrQixFQUFFLEdBQVcsRUFBRSxhQUFzQjtRQUM5RSxPQUFPLElBQUksTUFBTSxDQUFDLE1BQU0sRUFBRSxHQUFHLEVBQUUsYUFBYSxDQUFDLENBQUM7SUFDaEQsQ0FBQzs7Ozs7O0lBTU0sTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFrQixFQUFFLEdBQVcsRUFBRSxhQUFzQjtRQUMxRSxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxFQUFFLEdBQUcsRUFBRSxhQUFhLENBQUMsQ0FBQztJQUNyRCxDQUFDOzs7Ozs7OztJQU9NLE1BQU0sQ0FBQyxVQUFVLENBQUMsSUFBWTtRQUNuQyxPQUFPLElBQUksVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzlCLENBQUM7Ozs7OztJQU1NLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBWTtRQUMvQixPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDL0IsQ0FBQzs7Ozs7OztJQU9NLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBWSxFQUFFLE9BQWdDO1FBQ3BFLE9BQU8sSUFBSSxTQUFTLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3RDLENBQUM7Ozs7Ozs7Ozs7O0lBV00sTUFBTSxDQUFDLGVBQWUsQ0FBQyxJQUFZLEVBQUUsVUFBbUMsRUFBRTs7UUFDL0UsSUFBSSxTQUFTLFNBQUcsT0FBTyxDQUFDLFNBQVMsbUNBQUksVUFBVSxDQUFDO1FBRWhELHFFQUFxRTtRQUNyRSxJQUFJLFNBQVMsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDM0IsU0FBUyxHQUFHLEdBQUcsU0FBUyxHQUFHLENBQUM7U0FDN0I7YUFBTSxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUNwQyxTQUFTLEdBQUcsR0FBRyxTQUFTLElBQUksQ0FBQztTQUM5QjtRQUVELE1BQU0sU0FBUyxHQUFHLEdBQUcsQ0FBQyxXQUFXO2FBQzlCLFNBQVMsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDO2FBQ3hCLEVBQUUsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRXJDLE9BQU8sSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDbEMsQ0FBQzs7Ozs7O0lBTU0sTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFZO1FBQzlCLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUM5QixDQUFDOzs7Ozs7OztJQVFNLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxLQUE4QjtRQUM1RCxPQUFPLElBQUksaUJBQWlCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDdEMsQ0FBQzs7Ozs7O0lBTU0sTUFBTSxDQUFDLGFBQWEsQ0FBQyxLQUE4QjtRQUN4RCxPQUFPLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN2QyxDQUFDOzs7Ozs7OztJQU9NLE1BQU0sQ0FBQyxZQUFZLENBQUMsVUFBMkIsRUFBRSxLQUF5QjtRQUMvRSxPQUFPLElBQUksWUFBWSxDQUFDLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM3QyxDQUFDOzs7Ozs7OztJQU9NLE1BQU0sQ0FBQyxjQUFjLENBQUMsU0FBaUIsRUFBRSxRQUE2QixFQUFFO1FBQzdFLE9BQU8sSUFBSSxjQUFjLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzlDLENBQUM7Ozs7Ozs7OztJQXdCTSxjQUFjLENBQUMsU0FBMEIsRUFBRSxRQUE4QjtRQUM5RSxPQUFPO0lBQ1QsQ0FBQzs7QUEzSUgsb0JBNElDOzs7Ozs7OztBQXVERCxNQUFhLE1BQU8sU0FBUSxJQUFJOzs7O0lBSTlCLFlBQVksTUFBa0IsRUFBVSxHQUFXLEVBQVUsYUFBc0I7UUFDakYsS0FBSyxFQUFFLENBQUM7UUFEOEIsUUFBRyxHQUFILEdBQUcsQ0FBUTtRQUFVLGtCQUFhLEdBQWIsYUFBYSxDQUFTOzs7Ozs7UUFIbkUsYUFBUSxHQUFHLEtBQUssQ0FBQztRQU0vQixJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRTtZQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLGlEQUFpRCxDQUFDLENBQUM7U0FDcEU7UUFFRCxJQUFJLENBQUMsVUFBVSxHQUFHLE1BQU0sQ0FBQyxVQUFVLENBQUM7SUFDdEMsQ0FBQzs7Ozs7O0lBRU0sSUFBSSxDQUFDLE1BQWlCO1FBQzNCLE9BQU87WUFDTCxVQUFVLEVBQUU7Z0JBQ1YsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO2dCQUMzQixTQUFTLEVBQUUsSUFBSSxDQUFDLEdBQUc7Z0JBQ25CLGFBQWEsRUFBRSxJQUFJLENBQUMsYUFBYTthQUNsQztTQUNGLENBQUM7SUFDSixDQUFDOztBQXRCSCx3QkF1QkM7Ozs7Ozs7O0FBS0QsTUFBYSxVQUFXLFNBQVEsSUFBSTs7OztJQUdsQyxZQUFvQixJQUFZO1FBQzlCLEtBQUssRUFBRSxDQUFDO1FBRFUsU0FBSSxHQUFKLElBQUksQ0FBUTs7Ozs7O1FBRmhCLGFBQVEsR0FBRyxJQUFJLENBQUM7UUFLOUIsSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNyQixNQUFNLElBQUksS0FBSyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7U0FDdkQ7UUFFRCxJQUFJLElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMscURBQXFELEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3RGO0lBQ0gsQ0FBQzs7Ozs7O0lBRU0sSUFBSSxDQUFDLE1BQWlCO1FBQzNCLE9BQU87WUFDTCxVQUFVLEVBQUUsSUFBSSxDQUFDLElBQUk7U0FDdEIsQ0FBQztJQUNKLENBQUM7O0FBbkJILGdDQW9CQzs7Ozs7Ozs7QUFLRCxNQUFhLFNBQVUsU0FBUSxJQUFJOzs7OztJQU9qQyxZQUE0QixJQUFZLEVBQW1CLFVBQWtDLEVBQUc7UUFDOUYsS0FBSyxFQUFFLENBQUM7UUFEa0IsU0FBSSxHQUFKLElBQUksQ0FBUTtRQUFtQixZQUFPLEdBQVAsT0FBTyxDQUE4Qjs7Ozs7O1FBTmhGLGFBQVEsR0FBRyxLQUFLLENBQUM7SUFRakMsQ0FBQzs7Ozs7O0lBRU0sSUFBSSxDQUFDLEtBQWdCO1FBQzFCLHFGQUFxRjtRQUNyRixJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRTtZQUNmLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxTQUFTLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQzlDLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtnQkFDZixHQUFHLElBQUksQ0FBQyxPQUFPO2FBQ2hCLENBQUMsQ0FBQztTQUNKO2FBQU0sSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDM0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxtREFBbUQsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsS0FBSztnQkFDeEcsNkNBQTZDLENBQUMsQ0FBQztTQUNsRDtRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksRUFBRTtZQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLDZDQUE2QyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQztTQUM1RTtRQUVELE9BQU87WUFDTCxVQUFVLEVBQUU7Z0JBQ1YsVUFBVSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWTtnQkFDbkMsU0FBUyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVzthQUNsQztTQUNGLENBQUM7SUFDSixDQUFDOzs7Ozs7Ozs7SUFFTSxjQUFjLENBQUMsUUFBeUIsRUFBRSxVQUErQixFQUFHO1FBQ2pGLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ2YsTUFBTSxJQUFJLEtBQUssQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO1NBQ2pFO1FBRUQsTUFBTSxnQkFBZ0IsR0FBRyxPQUFPLENBQUMsZ0JBQWdCLElBQUksTUFBTSxDQUFDO1FBRTVELDZDQUE2QztRQUM3QyxJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixDQUFDLFFBQVEsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO0lBQzdELENBQUM7O0FBNUNILDhCQTZDQzs7Ozs7Ozs7Ozs7O0FBd0NELE1BQWEsaUJBQWtCLFNBQVEsSUFBSTs7OztJQUt6QyxZQUFZLFFBQWdDLEVBQUU7UUFDNUMsS0FBSyxFQUFFLENBQUM7Ozs7OztRQUxNLGFBQVEsR0FBRyxLQUFLLENBQUM7UUFPL0IsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUM7UUFDOUMsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO0lBQzlDLENBQUM7Ozs7OztJQUVNLElBQUksQ0FBQyxLQUFnQjtRQUMxQixJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFO1lBQzFCLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLGlDQUFpQyxFQUFFO2dCQUNyRixJQUFJLEVBQUUsUUFBUTthQUNmLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDekIsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLGdDQUFnQyxFQUFFO2dCQUNuRixJQUFJLEVBQUUsUUFBUTthQUNmLENBQUMsQ0FBQztTQUNKO1FBRUQsT0FBTztZQUNMLFVBQVUsRUFBRTtnQkFDVixVQUFVLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLGFBQWE7Z0JBQy9DLFNBQVMsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLGFBQWE7YUFDOUM7U0FDRixDQUFDO0lBQ0osQ0FBQzs7Ozs7Ozs7Ozs7Ozs7SUFjTSxNQUFNLENBQUMsUUFBcUI7UUFDakMsTUFBTSxHQUFHLEdBQTRCLEVBQUUsQ0FBQztRQUN4QyxHQUFHLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxVQUFVLENBQUM7UUFDaEQsR0FBRyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRyxRQUFRLENBQUMsU0FBUyxDQUFDO1FBQzlDLE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQzs7OztJQUVELElBQVcsZUFBZTtRQUN4QixJQUFJLElBQUksQ0FBQyxnQkFBZ0IsRUFBRTtZQUN6QixPQUFPLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLENBQUM7U0FDeEM7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsMkZBQTJGLENBQUMsQ0FBQztTQUM5RztJQUNILENBQUM7Ozs7SUFFRCxJQUFXLGNBQWM7UUFDdkIsSUFBSSxJQUFJLENBQUMsZUFBZSxFQUFFO1lBQ3hCLE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQyxTQUFTLENBQUM7U0FDdkM7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsMEZBQTBGLENBQUMsQ0FBQztTQUM3RztJQUNILENBQUM7O0FBbEVILDhDQW1FQzs7Ozs7Ozs7QUFpQ0QsTUFBYSxZQUFhLFNBQVEsSUFBSTs7OztJQUdwQyxZQUE2QixVQUEyQixFQUFtQixRQUEyQixFQUFFO1FBQ3RHLEtBQUssRUFBRSxDQUFDO1FBRG1CLGVBQVUsR0FBVixVQUFVLENBQWlCO1FBQW1CLFVBQUssR0FBTCxLQUFLLENBQXdCOzs7Ozs7UUFGeEYsYUFBUSxHQUFZLEtBQUssQ0FBQztJQUkxQyxDQUFDOzs7Ozs7SUFFTSxJQUFJLENBQUMsQ0FBWTs7UUFDdEIsSUFBSSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxDQUFDO1FBRTVFLE9BQU87WUFDTCxLQUFLLEVBQUU7Z0JBQ0wsUUFBUSxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsbUJBQW1CLGFBQUMsSUFBSSxDQUFDLEtBQUssMENBQUUsR0FBRyxtQ0FBSSxRQUFRLENBQUM7Z0JBQzFFLEdBQUcsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUc7Z0JBQ25CLFVBQVUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVU7YUFDbEM7U0FDRixDQUFDO0lBQ0osQ0FBQzs7QUFqQkgsb0NBa0JDOzs7Ozs7OztBQTJCRCxNQUFhLGNBQWUsU0FBUSxJQUFJOzs7O0lBR3RDLFlBQTZCLFNBQWlCLEVBQW1CLEtBQTBCO1FBQ3pGLEtBQUssRUFBRSxDQUFDO1FBRG1CLGNBQVMsR0FBVCxTQUFTLENBQVE7UUFBbUIsVUFBSyxHQUFMLEtBQUssQ0FBcUI7Ozs7OztRQUYzRSxhQUFRLEdBQVksS0FBSyxDQUFDO0lBSTFDLENBQUM7Ozs7OztJQUVNLElBQUksQ0FBQyxLQUFnQjtRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ2pFLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztZQUN6QixHQUFHLElBQUksQ0FBQyxLQUFLO1NBQ2QsQ0FBQyxDQUFDO1FBRUgsS0FBSyxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxDQUFDO1FBRTdFLE9BQU87WUFDTCxLQUFLLEVBQUU7Z0JBQ0wsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO2dCQUN4QixVQUFVLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVO2dCQUNqQyxHQUFHLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHO2FBQ3BCO1NBQ0YsQ0FBQztJQUNKLENBQUM7O0FBdEJILHdDQXVCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjciBmcm9tICdAYXdzLWNkay9hd3MtZWNyJztcbmltcG9ydCAqIGFzIGVjcl9hc3NldHMgZnJvbSAnQGF3cy1jZGsvYXdzLWVjci1hc3NldHMnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCAqIGFzIHMzX2Fzc2V0cyBmcm9tICdAYXdzLWNkay9hd3MtczMtYXNzZXRzJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuLy8ga2VlcCB0aGlzIGltcG9ydCBzZXBhcmF0ZSBmcm9tIG90aGVyIGltcG9ydHMgdG8gcmVkdWNlIGNoYW5jZSBmb3IgbWVyZ2UgY29uZmxpY3RzIHdpdGggdjItbWFpblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWR1cGxpY2F0ZS1pbXBvcnRzLCBpbXBvcnQvb3JkZXJcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgQ29kZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21CdWNrZXQoYnVja2V0OiBzMy5JQnVja2V0LCBrZXk6IHN0cmluZywgb2JqZWN0VmVyc2lvbj86IHN0cmluZyk6IFMzQ29kZSB7XG4gICAgcmV0dXJuIG5ldyBTM0NvZGUoYnVja2V0LCBrZXksIG9iamVjdFZlcnNpb24pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBidWNrZXQoYnVja2V0OiBzMy5JQnVja2V0LCBrZXk6IHN0cmluZywgb2JqZWN0VmVyc2lvbj86IHN0cmluZyk6IFMzQ29kZSB7XG4gICAgcmV0dXJuIHRoaXMuZnJvbUJ1Y2tldChidWNrZXQsIGtleSwgb2JqZWN0VmVyc2lvbik7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21JbmxpbmUoY29kZTogc3RyaW5nKTogSW5saW5lQ29kZSB7XG4gICAgcmV0dXJuIG5ldyBJbmxpbmVDb2RlKGNvZGUpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBpbmxpbmUoY29kZTogc3RyaW5nKTogSW5saW5lQ29kZSB7XG4gICAgcmV0dXJuIHRoaXMuZnJvbUlubGluZShjb2RlKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUFzc2V0KHBhdGg6IHN0cmluZywgb3B0aW9ucz86IHMzX2Fzc2V0cy5Bc3NldE9wdGlvbnMpOiBBc3NldENvZGUge1xuICAgIHJldHVybiBuZXcgQXNzZXRDb2RlKHBhdGgsIG9wdGlvbnMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tRG9ja2VyQnVpbGQocGF0aDogc3RyaW5nLCBvcHRpb25zOiBEb2NrZXJCdWlsZEFzc2V0T3B0aW9ucyA9IHt9KTogQXNzZXRDb2RlIHtcbiAgICBsZXQgaW1hZ2VQYXRoID0gb3B0aW9ucy5pbWFnZVBhdGggPz8gJy9hc3NldC8uJztcblxuICAgIC8vIGVuc3VyZSBpbWFnZVBhdGggZW5kcyB3aXRoIC8uIHRvIGNvcHkgdGhlICoqY29udGVudCoqIGF0IHRoaXMgcGF0aFxuICAgIGlmIChpbWFnZVBhdGguZW5kc1dpdGgoJy8nKSkge1xuICAgICAgaW1hZ2VQYXRoID0gYCR7aW1hZ2VQYXRofS5gO1xuICAgIH0gZWxzZSBpZiAoIWltYWdlUGF0aC5lbmRzV2l0aCgnLy4nKSkge1xuICAgICAgaW1hZ2VQYXRoID0gYCR7aW1hZ2VQYXRofS8uYDtcbiAgICB9XG5cbiAgICBjb25zdCBhc3NldFBhdGggPSBjZGsuRG9ja2VySW1hZ2VcbiAgICAgIC5mcm9tQnVpbGQocGF0aCwgb3B0aW9ucylcbiAgICAgIC5jcChpbWFnZVBhdGgsIG9wdGlvbnMub3V0cHV0UGF0aCk7XG5cbiAgICByZXR1cm4gbmV3IEFzc2V0Q29kZShhc3NldFBhdGgpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFzc2V0KHBhdGg6IHN0cmluZyk6IEFzc2V0Q29kZSB7XG4gICAgcmV0dXJuIHRoaXMuZnJvbUFzc2V0KHBhdGgpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21DZm5QYXJhbWV0ZXJzKHByb3BzPzogQ2ZuUGFyYW1ldGVyc0NvZGVQcm9wcyk6IENmblBhcmFtZXRlcnNDb2RlIHtcbiAgICByZXR1cm4gbmV3IENmblBhcmFtZXRlcnNDb2RlKHByb3BzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNmblBhcmFtZXRlcnMocHJvcHM/OiBDZm5QYXJhbWV0ZXJzQ29kZVByb3BzKTogQ2ZuUGFyYW1ldGVyc0NvZGUge1xuICAgIHJldHVybiB0aGlzLmZyb21DZm5QYXJhbWV0ZXJzKHByb3BzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21FY3JJbWFnZShyZXBvc2l0b3J5OiBlY3IuSVJlcG9zaXRvcnksIHByb3BzPzogRWNySW1hZ2VDb2RlUHJvcHMpIHtcbiAgICByZXR1cm4gbmV3IEVjckltYWdlQ29kZShyZXBvc2l0b3J5LCBwcm9wcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21Bc3NldEltYWdlKGRpcmVjdG9yeTogc3RyaW5nLCBwcm9wczogQXNzZXRJbWFnZUNvZGVQcm9wcyA9IHt9KSB7XG4gICAgcmV0dXJuIG5ldyBBc3NldEltYWdlQ29kZShkaXJlY3RvcnksIHByb3BzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgaXNJbmxpbmU6IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWJzdHJhY3QgYmluZChzY29wZTogQ29uc3RydWN0KTogQ29kZUNvbmZpZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGJpbmRUb1Jlc291cmNlKF9yZXNvdXJjZTogY2RrLkNmblJlc291cmNlLCBfb3B0aW9ucz86IFJlc291cmNlQmluZE9wdGlvbnMpIHtcbiAgICByZXR1cm47XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIENvZGVDb25maWcge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzM0xvY2F0aW9uPzogczMuTG9jYXRpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbmxpbmVDb2RlPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGltYWdlPzogQ29kZUltYWdlQ29uZmlnO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBDb2RlSW1hZ2VDb25maWcge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW1hZ2VVcmk6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY21kPzogc3RyaW5nW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW50cnlwb2ludD86IHN0cmluZ1tdO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBTM0NvZGUgZXh0ZW5kcyBDb2RlIHtcbiAgcHVibGljIHJlYWRvbmx5IGlzSW5saW5lID0gZmFsc2U7XG4gIHByaXZhdGUgYnVja2V0TmFtZTogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKGJ1Y2tldDogczMuSUJ1Y2tldCwgcHJpdmF0ZSBrZXk6IHN0cmluZywgcHJpdmF0ZSBvYmplY3RWZXJzaW9uPzogc3RyaW5nKSB7XG4gICAgc3VwZXIoKTtcblxuICAgIGlmICghYnVja2V0LmJ1Y2tldE5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignYnVja2V0TmFtZSBpcyB1bmRlZmluZWQgZm9yIHRoZSBwcm92aWRlZCBidWNrZXQnKTtcbiAgICB9XG5cbiAgICB0aGlzLmJ1Y2tldE5hbWUgPSBidWNrZXQuYnVja2V0TmFtZTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kKF9zY29wZTogQ29uc3RydWN0KTogQ29kZUNvbmZpZyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHMzTG9jYXRpb246IHtcbiAgICAgICAgYnVja2V0TmFtZTogdGhpcy5idWNrZXROYW1lLFxuICAgICAgICBvYmplY3RLZXk6IHRoaXMua2V5LFxuICAgICAgICBvYmplY3RWZXJzaW9uOiB0aGlzLm9iamVjdFZlcnNpb24sXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgSW5saW5lQ29kZSBleHRlbmRzIENvZGUge1xuICBwdWJsaWMgcmVhZG9ubHkgaXNJbmxpbmUgPSB0cnVlO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgY29kZTogc3RyaW5nKSB7XG4gICAgc3VwZXIoKTtcblxuICAgIGlmIChjb2RlLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdMYW1iZGEgaW5saW5lIGNvZGUgY2Fubm90IGJlIGVtcHR5Jyk7XG4gICAgfVxuXG4gICAgaWYgKGNvZGUubGVuZ3RoID4gNDA5Nikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdMYW1iZGEgc291cmNlIGlzIHRvbyBsYXJnZSwgbXVzdCBiZSA8PSA0MDk2IGJ1dCBpcyAnICsgY29kZS5sZW5ndGgpO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBiaW5kKF9zY29wZTogQ29uc3RydWN0KTogQ29kZUNvbmZpZyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGlubGluZUNvZGU6IHRoaXMuY29kZSxcbiAgICB9O1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBBc3NldENvZGUgZXh0ZW5kcyBDb2RlIHtcbiAgcHVibGljIHJlYWRvbmx5IGlzSW5saW5lID0gZmFsc2U7XG4gIHByaXZhdGUgYXNzZXQ/OiBzM19hc3NldHMuQXNzZXQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgcGF0aDogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IG9wdGlvbnM6IHMzX2Fzc2V0cy5Bc3NldE9wdGlvbnMgPSB7IH0pIHtcbiAgICBzdXBlcigpO1xuICB9XG5cbiAgcHVibGljIGJpbmQoc2NvcGU6IENvbnN0cnVjdCk6IENvZGVDb25maWcge1xuICAgIC8vIElmIHRoZSBzYW1lIEFzc2V0Q29kZSBpcyB1c2VkIG11bHRpcGxlIHRpbWVzLCByZXRhaW4gb25seSB0aGUgZmlyc3QgaW5zdGFudGlhdGlvbi5cbiAgICBpZiAoIXRoaXMuYXNzZXQpIHtcbiAgICAgIHRoaXMuYXNzZXQgPSBuZXcgczNfYXNzZXRzLkFzc2V0KHNjb3BlLCAnQ29kZScsIHtcbiAgICAgICAgcGF0aDogdGhpcy5wYXRoLFxuICAgICAgICAuLi50aGlzLm9wdGlvbnMsXG4gICAgICB9KTtcbiAgICB9IGVsc2UgaWYgKGNkay5TdGFjay5vZih0aGlzLmFzc2V0KSAhPT0gY2RrLlN0YWNrLm9mKHNjb3BlKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBBc3NldCBpcyBhbHJlYWR5IGFzc29jaWF0ZWQgd2l0aCBhbm90aGVyIHN0YWNrICcke2Nkay5TdGFjay5vZih0aGlzLmFzc2V0KS5zdGFja05hbWV9Jy4gYCArXG4gICAgICAgICdDcmVhdGUgYSBuZXcgQ29kZSBpbnN0YW5jZSBmb3IgZXZlcnkgc3RhY2suJyk7XG4gICAgfVxuXG4gICAgaWYgKCF0aGlzLmFzc2V0LmlzWmlwQXJjaGl2ZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBBc3NldCBtdXN0IGJlIGEgLnppcCBmaWxlIG9yIGEgZGlyZWN0b3J5ICgke3RoaXMucGF0aH0pYCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHtcbiAgICAgIHMzTG9jYXRpb246IHtcbiAgICAgICAgYnVja2V0TmFtZTogdGhpcy5hc3NldC5zM0J1Y2tldE5hbWUsXG4gICAgICAgIG9iamVjdEtleTogdGhpcy5hc3NldC5zM09iamVjdEtleSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kVG9SZXNvdXJjZShyZXNvdXJjZTogY2RrLkNmblJlc291cmNlLCBvcHRpb25zOiBSZXNvdXJjZUJpbmRPcHRpb25zID0geyB9KSB7XG4gICAgaWYgKCF0aGlzLmFzc2V0KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2JpbmRUb1Jlc291cmNlKCkgbXVzdCBiZSBjYWxsZWQgYWZ0ZXIgYmluZCgpJyk7XG4gICAgfVxuXG4gICAgY29uc3QgcmVzb3VyY2VQcm9wZXJ0eSA9IG9wdGlvbnMucmVzb3VyY2VQcm9wZXJ0eSB8fCAnQ29kZSc7XG5cbiAgICAvLyBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1jZGsvaXNzdWVzLzE0MzJcbiAgICB0aGlzLmFzc2V0LmFkZFJlc291cmNlTWV0YWRhdGEocmVzb3VyY2UsIHJlc291cmNlUHJvcGVydHkpO1xuICB9XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgUmVzb3VyY2VCaW5kT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZXNvdXJjZVByb3BlcnR5Pzogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBDZm5QYXJhbWV0ZXJzQ29kZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBidWNrZXROYW1lUGFyYW0/OiBjZGsuQ2ZuUGFyYW1ldGVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG9iamVjdEtleVBhcmFtPzogY2RrLkNmblBhcmFtZXRlcjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBDZm5QYXJhbWV0ZXJzQ29kZSBleHRlbmRzIENvZGUge1xuICBwdWJsaWMgcmVhZG9ubHkgaXNJbmxpbmUgPSBmYWxzZTtcbiAgcHJpdmF0ZSBfYnVja2V0TmFtZVBhcmFtPzogY2RrLkNmblBhcmFtZXRlcjtcbiAgcHJpdmF0ZSBfb2JqZWN0S2V5UGFyYW0/OiBjZGsuQ2ZuUGFyYW1ldGVyO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBDZm5QYXJhbWV0ZXJzQ29kZVByb3BzID0ge30pIHtcbiAgICBzdXBlcigpO1xuXG4gICAgdGhpcy5fYnVja2V0TmFtZVBhcmFtID0gcHJvcHMuYnVja2V0TmFtZVBhcmFtO1xuICAgIHRoaXMuX29iamVjdEtleVBhcmFtID0gcHJvcHMub2JqZWN0S2V5UGFyYW07XG4gIH1cblxuICBwdWJsaWMgYmluZChzY29wZTogQ29uc3RydWN0KTogQ29kZUNvbmZpZyB7XG4gICAgaWYgKCF0aGlzLl9idWNrZXROYW1lUGFyYW0pIHtcbiAgICAgIHRoaXMuX2J1Y2tldE5hbWVQYXJhbSA9IG5ldyBjZGsuQ2ZuUGFyYW1ldGVyKHNjb3BlLCAnTGFtYmRhU291cmNlQnVja2V0TmFtZVBhcmFtZXRlcicsIHtcbiAgICAgICAgdHlwZTogJ1N0cmluZycsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBpZiAoIXRoaXMuX29iamVjdEtleVBhcmFtKSB7XG4gICAgICB0aGlzLl9vYmplY3RLZXlQYXJhbSA9IG5ldyBjZGsuQ2ZuUGFyYW1ldGVyKHNjb3BlLCAnTGFtYmRhU291cmNlT2JqZWN0S2V5UGFyYW1ldGVyJywge1xuICAgICAgICB0eXBlOiAnU3RyaW5nJyxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBzM0xvY2F0aW9uOiB7XG4gICAgICAgIGJ1Y2tldE5hbWU6IHRoaXMuX2J1Y2tldE5hbWVQYXJhbS52YWx1ZUFzU3RyaW5nLFxuICAgICAgICBvYmplY3RLZXk6IHRoaXMuX29iamVjdEtleVBhcmFtLnZhbHVlQXNTdHJpbmcsXG4gICAgICB9LFxuICAgIH07XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYXNzaWduKGxvY2F0aW9uOiBzMy5Mb2NhdGlvbik6IHsgW25hbWU6IHN0cmluZ106IGFueSB9IHtcbiAgICBjb25zdCByZXQ6IHsgW25hbWU6IHN0cmluZ106IGFueSB9ID0ge307XG4gICAgcmV0W3RoaXMuYnVja2V0TmFtZVBhcmFtXSA9IGxvY2F0aW9uLmJ1Y2tldE5hbWU7XG4gICAgcmV0W3RoaXMub2JqZWN0S2V5UGFyYW1dID0gbG9jYXRpb24ub2JqZWN0S2V5O1xuICAgIHJldHVybiByZXQ7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGJ1Y2tldE5hbWVQYXJhbSgpOiBzdHJpbmcge1xuICAgIGlmICh0aGlzLl9idWNrZXROYW1lUGFyYW0pIHtcbiAgICAgIHJldHVybiB0aGlzLl9idWNrZXROYW1lUGFyYW0ubG9naWNhbElkO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Bhc3MgQ2ZuUGFyYW1ldGVyc0NvZGUgdG8gYSBMYW1iZGEgRnVuY3Rpb24gYmVmb3JlIGFjY2Vzc2luZyB0aGUgYnVja2V0TmFtZVBhcmFtIHByb3BlcnR5Jyk7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIGdldCBvYmplY3RLZXlQYXJhbSgpOiBzdHJpbmcge1xuICAgIGlmICh0aGlzLl9vYmplY3RLZXlQYXJhbSkge1xuICAgICAgcmV0dXJuIHRoaXMuX29iamVjdEtleVBhcmFtLmxvZ2ljYWxJZDtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdQYXNzIENmblBhcmFtZXRlcnNDb2RlIHRvIGEgTGFtYmRhIEZ1bmN0aW9uIGJlZm9yZSBhY2Nlc3NpbmcgdGhlIG9iamVjdEtleVBhcmFtIHByb3BlcnR5Jyk7XG4gICAgfVxuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBFY3JJbWFnZUNvZGVQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjbWQ/OiBzdHJpbmdbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbnRyeXBvaW50Pzogc3RyaW5nW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0YWc/OiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgRWNySW1hZ2VDb2RlIGV4dGVuZHMgQ29kZSB7XG4gIHB1YmxpYyByZWFkb25seSBpc0lubGluZTogYm9vbGVhbiA9IGZhbHNlO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcmVwb3NpdG9yeTogZWNyLklSZXBvc2l0b3J5LCBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBFY3JJbWFnZUNvZGVQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoKTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kKF86IENvbnN0cnVjdCk6IENvZGVDb25maWcge1xuICAgIHRoaXMucmVwb3NpdG9yeS5ncmFudFB1bGwobmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpKTtcblxuICAgIHJldHVybiB7XG4gICAgICBpbWFnZToge1xuICAgICAgICBpbWFnZVVyaTogdGhpcy5yZXBvc2l0b3J5LnJlcG9zaXRvcnlVcmlGb3JUYWcodGhpcy5wcm9wcz8udGFnID8/ICdsYXRlc3QnKSxcbiAgICAgICAgY21kOiB0aGlzLnByb3BzLmNtZCxcbiAgICAgICAgZW50cnlwb2ludDogdGhpcy5wcm9wcy5lbnRyeXBvaW50LFxuICAgICAgfSxcbiAgICB9O1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQXNzZXRJbWFnZUNvZGVQcm9wcyBleHRlbmRzIGVjcl9hc3NldHMuRG9ja2VySW1hZ2VBc3NldE9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY21kPzogc3RyaW5nW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW50cnlwb2ludD86IHN0cmluZ1tdO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBBc3NldEltYWdlQ29kZSBleHRlbmRzIENvZGUge1xuICBwdWJsaWMgcmVhZG9ubHkgaXNJbmxpbmU6IGJvb2xlYW4gPSBmYWxzZTtcblxuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IGRpcmVjdG9yeTogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBBc3NldEltYWdlQ29kZVByb3BzKSB7XG4gICAgc3VwZXIoKTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kKHNjb3BlOiBDb25zdHJ1Y3QpOiBDb2RlQ29uZmlnIHtcbiAgICBjb25zdCBhc3NldCA9IG5ldyBlY3JfYXNzZXRzLkRvY2tlckltYWdlQXNzZXQoc2NvcGUsICdBc3NldEltYWdlJywge1xuICAgICAgZGlyZWN0b3J5OiB0aGlzLmRpcmVjdG9yeSxcbiAgICAgIC4uLnRoaXMucHJvcHMsXG4gICAgfSk7XG5cbiAgICBhc3NldC5yZXBvc2l0b3J5LmdyYW50UHVsbChuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJykpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGltYWdlOiB7XG4gICAgICAgIGltYWdlVXJpOiBhc3NldC5pbWFnZVVyaSxcbiAgICAgICAgZW50cnlwb2ludDogdGhpcy5wcm9wcy5lbnRyeXBvaW50LFxuICAgICAgICBjbWQ6IHRoaXMucHJvcHMuY21kLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIERvY2tlckJ1aWxkQXNzZXRPcHRpb25zIGV4dGVuZHMgY2RrLkRvY2tlckJ1aWxkT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGltYWdlUGF0aD86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG91dHB1dFBhdGg/OiBzdHJpbmc7XG59XG4iXX0=