"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const iam = require("@aws-cdk/aws-iam");
const function_base_1 = require("./function-base");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const scalable_function_attribute_1 = require("./private/scalable-function-attribute");
/**
 * A new alias to a particular version of a Lambda function.
 *
 * @stability stable
 */
class Alias extends function_base_1.QualifiedFunctionBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.aliasName,
        });
        /**
         * Whether the addPermission() call adds any permissions.
         *
         * True for new Lambdas, false for version $LATEST and imported Lambdas
         * from different accounts.
         *
         * @stability stable
         */
        this.canCreatePermissions = true;
        this.lambda = props.version.lambda;
        this.aliasName = this.physicalName;
        this.version = props.version;
        const alias = new lambda_generated_1.CfnAlias(this, 'Resource', {
            name: this.aliasName,
            description: props.description,
            functionName: this.version.lambda.functionName,
            functionVersion: props.version.version,
            routingConfig: this.determineRoutingConfig(props),
            provisionedConcurrencyConfig: this.determineProvisionedConcurrency(props),
        });
        // Use a Service Linked Role
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        this.scalingRole = iam.Role.fromRoleArn(this, 'ScalingRole', this.stack.formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/lambda.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_LambdaConcurrency',
        }));
        this.functionArn = this.getResourceArnAttribute(alias.ref, {
            service: 'lambda',
            resource: 'function',
            resourceName: `${this.lambda.functionName}:${this.physicalName}`,
            sep: ':',
        });
        this.qualifier = lambda_version_1.extractQualifierFromArn(alias.ref);
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        // ARN parsing splits on `:`, so we can only get the function's name from the ARN as resourceName...
        // And we're parsing it out (instead of using the underlying function directly) in order to have use of it incur
        // an implicit dependency on the resource.
        this.functionName = `${this.stack.parseArn(this.functionArn, ':').resourceName}:${this.aliasName}`;
    }
    /**
     * @stability stable
     */
    static fromAliasAttributes(scope, id, attrs) {
        class Imported extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.aliasName = attrs.aliasName;
                this.version = attrs.aliasVersion;
                this.lambda = attrs.aliasVersion.lambda;
                this.functionArn = `${attrs.aliasVersion.lambda.functionArn}:${attrs.aliasName}`;
                this.functionName = `${attrs.aliasVersion.lambda.functionName}:${attrs.aliasName}`;
                this.grantPrincipal = attrs.aliasVersion.grantPrincipal;
                this.role = attrs.aliasVersion.role;
                this.canCreatePermissions = this._isStackAccount();
                this.qualifier = attrs.aliasName;
            }
        }
        return new Imported(scope, id);
    }
    /**
     * The principal this Lambda Function is running as.
     *
     * @stability stable
     */
    get grantPrincipal() {
        return this.version.grantPrincipal;
    }
    /**
     * The IAM role associated with this function.
     *
     * Undefined if the function was imported without a role.
     *
     * @stability stable
     */
    get role() {
        return this.version.role;
    }
    /**
     * Return the given named metric for this Function.
     *
     * @stability stable
     */
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differs from the base behavior.
        return super.metric(metricName, {
            dimensions: {
                FunctionName: this.lambda.functionName,
                // construct the name from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/aws/aws-cdk/issues/2231
                Resource: `${this.lambda.functionName}:${this.aliasName}`,
            },
            ...props,
        });
    }
    /**
     * Configure provisioned concurrency autoscaling on a function alias.
     *
     * Returns a scalable attribute that can call
     * `scaleOnUtilization()` and `scaleOnSchedule()`.
     *
     * @param options Autoscaling options.
     * @stability stable
     */
    addAutoScaling(options) {
        var _b;
        if (this.scalableAlias) {
            throw new Error('AutoScaling already enabled for this alias');
        }
        return this.scalableAlias = new scalable_function_attribute_1.ScalableFunctionAttribute(this, 'AliasScaling', {
            minCapacity: (_b = options.minCapacity) !== null && _b !== void 0 ? _b : 1,
            maxCapacity: options.maxCapacity,
            resourceId: `function:${this.functionName}`,
            dimension: 'lambda:function:ProvisionedConcurrency',
            serviceNamespace: appscaling.ServiceNamespace.LAMBDA,
            role: this.scalingRole,
        });
    }
    /**
     * Calculate the routingConfig parameter from the input props
     */
    determineRoutingConfig(props) {
        if (!props.additionalVersions || props.additionalVersions.length === 0) {
            return undefined;
        }
        this.validateAdditionalWeights(props.additionalVersions);
        return {
            additionalVersionWeights: props.additionalVersions.map(vw => {
                return {
                    functionVersion: vw.version.version,
                    functionWeight: vw.weight,
                };
            }),
        };
    }
    /**
     * Validate that the additional version weights make sense
     *
     * We validate that they are positive and add up to something <= 1.
     */
    validateAdditionalWeights(weights) {
        const total = weights.map(w => {
            if (w.weight < 0 || w.weight > 1) {
                throw new Error(`Additional version weight must be between 0 and 1, got: ${w.weight}`);
            }
            return w.weight;
        }).reduce((a, x) => a + x);
        if (total > 1) {
            throw new Error(`Sum of additional version weights must not exceed 1, got: ${total}`);
        }
    }
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    determineProvisionedConcurrency(props) {
        if (!props.provisionedConcurrentExecutions) {
            return undefined;
        }
        if (props.provisionedConcurrentExecutions <= 0) {
            throw new Error('provisionedConcurrentExecutions must have value greater than or equal to 1');
        }
        return { provisionedConcurrentExecutions: props.provisionedConcurrentExecutions };
    }
}
exports.Alias = Alias;
_a = JSII_RTTI_SYMBOL_1;
Alias[_a] = { fqn: "@aws-cdk/aws-lambda.Alias", version: "1.100.0" };
//# sourceMappingURL=data:application/json;base64,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